import os
import glob
import fileinput

from arcgen.utils.time import current_time
from arcgen.utils.files import extension2type
from arcgen.utils.files import add_carriage_return
from arcgen.utils.files import get_skd_spacecrafts
from arcgen.utils.files import get_skd_targets
from arcgen.utils.files import get_spacecraft_text

class PDSLabel(object):

    def __init__(self, mission, product):

        self.root_dir = mission.root_dir
        self.product = product
        self.mission_accronym = mission.accronym
        self.mission = mission

        #
        # Fields from Mission
        #
        if mission.pds == '3':
            self.PDS3_DATASET           = mission.dataset
            self.PDS3_NAME_DATASET      = mission.dataset_name
        self.XML_MODEL                  = mission.xml_model
        self.SCHEMA_LOCATION            = mission.schema_location
        self.INFORMATION_MODEL_VERSION  = mission.information_model_version
        self.PDS4_MISSION_NAME          = mission.name
        self.PDS4_MISSION_LID           = mission.lid
        self.CURRENT_YEAR               = current_time().split('-')[0]
        self.BUNDLE_DESCRIPTION_LID     = \
            'urn:esa:psa:{}_spice:document:spiceds'.format(self.mission_accronym)
        self.PRODUCT_CREATION_TIME      = product.creation_time
        self.PDS3_PRODUCT_CREATION_TIME = product.creation_time.split('T')[0]
        self.FILE_SIZE                  = product.size
        self.FILE_CHECKSUM              = product.checksum

        self.PDS4_SPACECRAFT_NAME = mission.spacecraft
        self.PDS4_SPACECRAFT_TEXT = get_spacecraft_text(mission)

        # ONLY FOR PDS4
        if self.mission.pds == '4':
            #
            # These fields are obtained from the context products
            #
            for context_product in mission.context_products:
                if context_product['name'][0].upper() == mission.spacecraft.upper():

                    self.PDS4_SPACECRAFT_TYPE = context_product['type'][0].capitalize()
                    self.PDS4_SPACECRAFT_LID  = context_product['lidvid'].split('::')[0]

            #
            # For Bundles with multiple s/c
            #
            if mission.secondary_spacecrafts:
                #
                # We sort out how many S/C we have
                #
                sec_scs = mission.secondary_spacecrafts.split(',')

                for sc in sec_scs:
                    if sc:
                        sc = sc.strip()
                        if sc.lower() in product.name.lower():
                            self.PDS4_SPACECRAFT_NAME = sc

                            for context_product in mission.context_products:
                                if context_product['name'][0].upper() == sc.upper():
                                    self.PDS4_SPACECRAFT_TYPE = context_product['type'][0].capitalize()
                                    self.PDS4_SPACECRAFT_LID = context_product['lidvid'].split('::')[0]


            self.PDS4_TARGET_NAME = mission.target

            for context_product in mission.context_products:
                if context_product['name'][0].upper() ==  mission.target.upper():

                    self.PDS4_TARGET_TYPE = context_product['type'][0].capitalize()
                    self.PDS4_TARGET_LID  = context_product['lidvid'].split('::')[0]

            #
            # For Bundles with multiple targets
            #
            if mission.secondary_targets:
                #
                # We sort out how many S/C we have
                #
                sec_tars = mission.secondary_targets.split(',')

                for tar in sec_tars:
                    if tar:

                        if tar.lower() in product.name.lower():
                            self.PDS4_TARGET_NAME = tar
                            for context_product in mission.context_products:
                                if context_product['name'][0].upper() == tar.upper():
                                    self.PDS4_TARGET_TYPE = context_product['type'][0].capitalize()
                                    self.PDS4_TARGET_LID = context_product['lidvid'].split('::')[0]


            #
            # Fro labels that need to include all S/C and Targets of the mission
            #
            self.SKD_SPACECRAFTS = get_skd_spacecrafts(mission)
            self.SKD_TARGETS = get_skd_targets(mission, self.get_target_reference_type())

        return

    def get_target_reference_type(self):
        return "data_to_target"

    def write_label(self):

        label_dictionary = vars(self)

        if self.mission.pds == '3':
            label_extension = '.LBL'
        else:
            label_extension = '.xml'

        label_name = self.product.path.split('.')[0] + label_extension

        with open(label_name, "w+") as f:

            for line in fileinput.input(self.template):
                line = line.rstrip()
                for key, value in label_dictionary.items():
                    if isinstance(value, str) and key in line and '$' in line:
                        line = line.replace('$'+key, value)

                line = add_carriage_return(line)

                #
                # PDS3 specific lines
                #
                if 'END_OBJECT' in line:
                    f.write('\n' + line)
                elif '^SPICE_KERNEL' in line:
                    f.write(line)
                    f.write('\n')
                #
                # In general for PDS4 and most of PDS3 lines
                #
                else:
                    f.write(line)

        return


class KernelPDSLabel(PDSLabel):

    def __init__(self, mission, product):

        PDSLabel.__init__(self, mission, product)

        if mission.pds == '3':
            self.template = self.root_dir + '/etc/template_product_spice_kernel.LBL'
        else:
            self.template = self.root_dir + '/etc/template_product_spice_kernel.xml'

        #
        # Fields from Kernels
        #
        self.FILE_NAME = product.name
        self.PRODUCT_LID = self.product.lid
        self.FILE_FORMAT = product.file_format
        self.START_TIME = product.start_time
        self.STOP_TIME = product.stop_time
        self.KERNEL_TYPE_ID = product.type.upper()
        self.PRODUCT_VID = self.product.vid
        self.SPICE_KERNEL_DESCRIPTION = product.description

        #
        # PDS3 specific:
        #
        if mission.pds == '3':
            if product.mission_phases[-4:] != '\r\n':
                product.mission_phases = product.mission_phases + '\r\n'
            if 'Z' in product.start_time:
                self.START_TIME = product.start_time.split('Z')[0]
            if product.extension.lower() == 'tsc':
                self.START_TIME = mission.start
            if 'Z' in product.stop_time:
                self.STOP_TIME = product.stop_time.split('Z')[0]
            if product.extension.lower() in ['tf', 'ti', 'tsc', 'tls', 'tpc']:
                self.RECORD_BYTES = '"N/A"'
            else:
                self.RECORD_BYTES = "1024"
            self.MISSION_PHASES = product.mission_phases
            self.INTERCHANGE_FORMAT = product.file_format
            self.PDS3_KERNEL_TYPE = self.product.pds3type
            self.PRODUCER_ID = product.producer_id
            self.PRODUCT_VERSION_TYPE = product.product_version_type
            self.NAIF_INSTRUMENT_ID = product.naif_instrument_id
            self.RECORD_TYPE = product.record_type
            self.TARGET_NAME = product.target_name

            extension = '.LBL'
        else:
            extension = '.xml'
        #
        #    We try to extract it form the filename, otherwise it goes to the
        #    default one.
        #
        #try:
        #    spacecraft_name = product.name.split('_')[1].upper()
        #    if 'EDM' in spacecraft_name:
        #        spacecraft_name = 'edm'
        #except:
        #    spacecraft_name = mission.spacecraft


        self.name = product.name.split('.')[0] + extension

        self.write_label()

        return


class MetaKernelPDS4Label(PDSLabel):

    def __init__(self, mission, product):

        PDSLabel.__init__(self, mission, product)

        self.template = self.root_dir + '/etc/template_product_spice_kernel_mk.xml'

        #
        # Fields from Kernels
        #
        self.FILE_NAME = product.name
        self.PRODUCT_LID = self.product.lid
        self.FILE_FORMAT = 'Character'
        self.START_TIME = product.start_time
        self.STOP_TIME = product.stop_time
        self.KERNEL_TYPE_ID = product.type.upper()
        self.PRODUCT_VID = self.product.vid
        self.SPICE_KERNEL_DESCRIPTION = product.description

        self.KERNEL_INTERNAL_REFERENCES = self.get_kernel_internal_references()

        #
        #    We try to extract it form the filename, otherwise it goes to the
        #    default one.
        #
        # try:
        #    spacecraft_name = product.name.split('_')[1].upper()
        #    if not spacecraft_name in ['TGO', 'EDM']:
        #        spacecraft_name = mission.spacecraft
        # except:
        #    spacecraft_name = mission.spacecraft

        self.name = product.name.split('.')[0] + '.xml'

        self.write_label()

        return

    def get_kernel_internal_references(self):

        #
        # From the collection we only use kernels in the MK
        #
        kernel_list_for_label = ''
        for kernel in self.product.spice_kernels_metakernel:
            try:
                kernel_lid = kernel.lid
            except:
                #
                # When we generate the label from an already existing MK
                # or when we generate an increment of the archive.
                #
                kernel_lid = f"urn:esa:psa:" \
                             f"{self.mission.accronym}" \
                             f"_spice:spice_kernels:" \
                             f"{extension2type(kernel.split('.')[-1])}_" \
                             f"{kernel.split('/')[-1]}"

            kernel_list_for_label += \
            '        <Internal_Reference>\r\n' + \
            '            <lid_reference>{}\r\n'.format(kernel_lid) + \
            '            </lid_reference>\r\n' + \
            '            <reference_type>data_to_associate</reference_type>\r\n' +\
            '        </Internal_Reference>\r\n'

        kernel_list_for_label = kernel_list_for_label.rstrip() + '\r\n'

        return kernel_list_for_label


class InventoryPDS4Label(PDSLabel):

    def __init__(self, mission, collection, inventory):

        PDSLabel.__init__(self, mission, inventory)

        self.collection = collection
        self.template = self.root_dir + \
                        '/etc/template_collection_{}.xml'.format(collection.type)

        self.COLLECTION_LID = self.collection.lid
        self.COLLECTION_VID = self.collection.vid
        self.START_TIME = mission.start
        self.STOP_TIME = mission.increment_stop
        self.FILE_NAME = inventory.name
        self.DOI = mission.doi

        # Count number of lines in the inventory file
        self.N_RECORDS = str(len(open(self.product.path).readlines()))

        self.name = collection.name.split('.')[0] + '.xml'

        self.write_label()

        return

    def get_target_reference_type(self):
        return "collection_to_target"

    def write_label(self):

        label_dictionary = vars(self)
        label_name = self.product.path.split('_inventory')[0] + self.product.path.split('_inventory')[-1].split('.')[0] + '.xml'

        with open(label_name, "w+") as f:

            for line in fileinput.input(self.template):
                line = line.rstrip()
                for key, value in label_dictionary.items():
                    if isinstance(value, str) and key in line and '$' in line:
                        line = line.replace('$'+key, value)

                line = add_carriage_return(line)

                f.write(line)

        return


class InventoryPDS3Label(PDSLabel):

    def __init__(self, mission, collection, inventory):

        PDSLabel.__init__(self, mission, inventory)

        self.collection = collection
        self.template = self.root_dir + \
                        '/etc/template_collection_{}.LBL'.format(
                            collection.type)

        self.VOLUME = mission.volume
        self.PRODUCT_CREATION_TIME = current_time()
        self.DATASET = mission.dataset

        indexfile = open(mission.bundle_directory + '/INDEX/INDEX.TAB', 'r').readlines()
        self.N_RECORDS = str(len(indexfile))
        self.R_RECORDS = str(len(indexfile[0]) + 1)
        indexfields = indexfile[0].replace('"', '').split(',')
        self.START_BYTE1 = str(2)
        self.START_BYTE2 = str(2 + len(indexfields[0]) + 3)
        self.START_BYTE3 = str(2 + len(indexfields[0]) + 3 + len(indexfields[1]) + 2)
        self.START_BYTE4 = str(2 + len(indexfields[0]) + 3 + len(indexfields[1]) + 2 + len(indexfields[2]) + 2)
        self.BYTES1 = str(len(indexfields[0]))
        self.BYTES2 = str(len(indexfields[1]))
        self.BYTES3 = str(len(indexfields[2]))
        self.BYTES4 = str(len(indexfields[3].split('\n')[0]))

        self.write_label()

        return


    def write_label(self):

        label_dictionary = vars(self)


        with open(self.mission.bundle_directory + '/INDEX/INDEX.LBL', "w+") as f:

            for line in fileinput.input(self.template):
                line = line.rstrip()
                for key, value in label_dictionary.items():
                    if isinstance(value, str) and key in line and '$' in line:
                        line = line.replace('$' + key, value)

                line = add_carriage_return(line)

                if 'END_OBJECT' in line and line[:10] != 'END_OBJECT':
                    f.write(line.split('END_OBJECT')[0])
                    f.write('END_OBJECT               = COLUMN\n')
                else:
                    f.write(line)

        return


class DocumentPDS4Label(PDSLabel):

    def __init__(self, mission, collection, inventory):

        PDSLabel.__init__(self, mission, inventory)

        self.mission = mission
        self.collection = collection
        self.template = self.root_dir + \
                        '/etc/template_product_html_document.xml'

        self.PRODUCT_LID = inventory.lid
        self.PRODUCT_VID = inventory.vid
        self.START_TIME = mission.start
        self.STOP_TIME = mission.stop
        self.FILE_NAME = inventory.name
        self.CURRENT_TIME = current_time()
        self.CURRENT_DATE =self.CURRENT_TIME.split('T')[0]

        self.name = collection.name.split('.')[0] + '.xml'

        self.write_label()

        return


class BundlePDS4Label(PDSLabel):

    def __init__(self, mission, readme):

        PDSLabel.__init__(self, mission, readme)

        self.template = self.root_dir + \
                        '/etc/template_bundle.xml'

        self.BUNDLE_LID = self.product.bundle.lid
        self.BUNDLE_VID = self.product.bundle.vid

        self.START_TIME = mission.start
        self.STOP_TIME = mission.increment_stop
        self.FILE_NAME = readme.name
        self.CURRENT_TIME = current_time()
        self.CURRENT_DATE =self.CURRENT_TIME.split('T')[0]
        self.DOI =  mission.doi


        for collection in self.product.bundle.collections:
            if collection.name == 'spice_kernels':
                self.SPICE_COLL_LIDVID =  collection.lid + '::' + collection.vid
                self.SPICE_COLL_STATUS = 'Primary'
            if collection.name == 'document':
                self.DOC_COLL_LIDVID =  collection.lid + '::' + collection.vid
                self.DOC_COLL_STATUS = 'Primary'

        self.write_label()

        return

    def get_target_reference_type(self):
        return "bundle_to_target"
