C$Procedure MKCKSG ( Make CK segments )
 
      SUBROUTINE MKCKSG ( INPUT,   HANDLE,  SUBTYP,  
     .                    DEGREE,  INST,    FRAME, 
     .                    KBEG,    KEND,    TIMSYS,  RATE )
 
C$ Abstract
C
C     Write out one or more CK segments representing the pointing
C     defined by a MEX attitude file.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading 
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'ck05.inc'

      CHARACTER*(*)         INPUT
      INTEGER               HANDLE
      INTEGER               SUBTYP
      INTEGER               DEGREE
      INTEGER               INST
      CHARACTER*(*)         FRAME
      DOUBLE PRECISION      KBEG
      DOUBLE PRECISION      KEND
      CHARACTER*(*)         TIMSYS
      DOUBLE PRECISION      RATE
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     INPUT      I   EPM file to be read.
C     HANDLE     I   Handle of SPK file.
C     SUBTYP     I   CK subtype.
C     DEGREE     I   Interpolation degree.
C     INST       I   Instrument.
C     FRAME      I   Name of reference frame.
C     KBEG       I   Coverage start time of file.
C     KEND       I   Coverage stop time of file.
C     TIMSYS     I   Time system.
C     RATE       I   Nominal SCLK rate in seconds per tick.
C
C$ Detailed_Input
C
C     INPUT          is the name of an EPM file to be read.
C
C     HANDLE         is the handle of the CK file to write.  The
C                    file may be new or may be open for appending.
C
C     SUBTYP         is the subtype of the CK data type 
C                    corresponding to the input data representation.
C                    See the include file ck05.inc for CK type 05
C                    subtype codes.
C
C     DEGREE         is the polynomial degree associated with the
C                    interpolation method.
C
C     INST           is the NAIF integer ID of instrument or spacecraft
C                    component for which segments are to be written.
C
C     FRAME          is the name of the reference frame relative to
C                    which the output pointing is given.
C
C     KBEG           is the start time of the output file's coverage.
C                    By default KBEG is the start time of the input
C                    file, but the user may request a later start time.
C                    KBEG is expressed as seconds past J2000, TDB.
C
C     KEND           is the end time of the output file's coverage.
C                    By default KEND is the end time of the input
C                    file, but the user may request an earlier end time.
C                    KEND is expressed as seconds past J2000, TDB.
C
C     TIMSYS         is the name of the time system associated
C                    with time tags in the current metadata block
C                    or the states associated with this block.
C                     
C     RATE           is the nominal rate of the spacecraft clock
C                    associated with INST.  Units are seconds per
C                    tick.  RATE is used to scale angular velocity
C                    to radians/second.
C
C$ Detailed_Output
C
C     None.  See Particulars for a description of the action taken
C     by this routine.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If an unrecognized SPK or CK subtype is specified, the 
C        error SPICE(NOTSUPPORTED) will be signaled.
C
C     2) If an error occurs while writing a CK segment, the problem
C        will be diagnosed by routines called by this routine.
C
C     3) If an error occurs while parsing packet data from the input
C        EPM file, the problem will be diagnosed by routines called
C        by this routine.
C
C     4) If TIMSYS does not specify a supported time system, the 
C        error will be diagnosed by routines called by this routine.
C 
C$ Files
C
C     See the descriptions of the arguments INPUT and OUTPUT.
C
C$ Particulars
C
C     None.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     1)  This routine is intended for use only within the program 
C         MEX2KER.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C     J. Diaz del Rio (ESA)
C
C$ Version
C
C-    MEX2KER Version 2.0.0, 15-JUL-2014 (NJB) (JDR) (BVS)
C
C        Integrated updates by Jorge Diaz del Rio. These updates
C        enable inclusion of the input file creation time in the
C        comments created by this routine.
C
C-    MEX2KER Version 2.1.0, 21-JUN-2012 (NJB)
C
C        Bug fix: changed data type of local variables
C        B and E from d.p. to INTEGER. Made various
C        header updates.
C
C-    MEX2KER Version 2.0.0, 19-FEB-2004 (NJB)
C
C        Updated to support user selection of output file time
C        bounds.
C
C-    MEX2KER Version 1.0.0, 19-JUN-2003 (NJB)
C
C-&


C
C     SPICELIB functions
C      
      INTEGER               LSTLED
      INTEGER               LSTLTD

      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C
      CHARACTER*(*)         ATTTYP
      PARAMETER           ( ATTTYP = 'ATTITUDE FILE' )


      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 255 )

      INTEGER               MAXPK
      PARAMETER           ( MAXPK  = 10000 )

      INTEGER               SHORTL
      PARAMETER           ( SHORTL = 80 )

      INTEGER               SIDLEN
      PARAMETER           ( SIDLEN = 40 )

      INTEGER               MAXPS
      PARAMETER           ( MAXPS  = C05PS2 )
C
C     Local variables
C
      CHARACTER*(LNSIZE)    CRTTIM
      CHARACTER*(SHORTL)    FTYPE
      CHARACTER*(LNSIZE)    LINE
      CHARACTER*(SIDLEN)    SEGID
      CHARACTER*(SHORTL)    STYPE
      CHARACTER*(SHORTL)    REF
      CHARACTER*(SHORTL)    TSYS
      CHARACTER*(SHORTL)    TYPE

      DOUBLE PRECISION      BEG
      DOUBLE PRECISION      END
      DOUBLE PRECISION      ET
      DOUBLE PRECISION      FIRST
      DOUBLE PRECISION      LAST
      DOUBLE PRECISION      PACKET ( MAXPS )
      DOUBLE PRECISION      PKBUFF ( MAXPS *  MAXPK )
      DOUBLE PRECISION      SCLKDP ( MAXPK )
      DOUBLE PRECISION      SEGBEG
      DOUBLE PRECISION      SEGEND
      DOUBLE PRECISION      STARTS ( MAXPK )
      DOUBLE PRECISION      STCLKS ( MAXPK )
      DOUBLE PRECISION      TBUFF  ( MAXPK )
      DOUBLE PRECISION      TPACK  ( MAXPK )

      INTEGER               B
      INTEGER               CLKID
      INTEGER               CTR
      INTEGER               E
      INTEGER               FROM
      INTEGER               I
      INTEGER               INSTR
      INTEGER               LIDX
      INTEGER               LNUM
      INTEGER               LOC
      INTEGER               N
      INTEGER               NINTS
      INTEGER               NSEG
      INTEGER               NWR
      INTEGER               PACKSZ
      INTEGER               PAD
      INTEGER               PKB
      INTEGER               SHIFT
      INTEGER               STIDX
      INTEGER               TO
      INTEGER               WINSIZ

      LOGICAL               AVFLAG
      LOGICAL               EOF
      LOGICAL               FOUND

      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'MKCKSG' )

C
C     Use the first SIDLEN characters of the input file name as the
C     segment ID.
C
      SEGID = INPUT(:SIDLEN)

C
C     Set the window and packet sizes.
C
      IF (  SUBTYP .EQ. C05TP0 ) THEN

         WINSIZ = ( DEGREE + 1 ) / 2
         PACKSZ = C05PS0

      ELSE IF ( SUBTYP .EQ. C05TP1  ) THEN

         WINSIZ = DEGREE + 1
         PACKSZ = C05PS1

      ELSE IF (  SUBTYP .EQ. C05TP2 ) THEN

         WINSIZ = ( DEGREE + 1 ) / 2
         PACKSZ = C05PS2

      ELSE IF ( SUBTYP .EQ. C05TP3  ) THEN

         WINSIZ = DEGREE + 1
         PACKSZ = C05PS3

      ELSE

         CALL SETMSG ( 'CK type 5 SUBTYP <#> is not supported.' )
         CALL ERRINT ( '#', SUBTYP                              )
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                    )
         CALL CHKOUT ( 'MKCKSG'                                 )
         RETURN

      END IF

C
C     If the window size is even, the pad is one less than half the 
C     window size.  If the window size is odd, the pad is half of
C     one less than the window size.  In either case, the pad
C     can be computed using the same integer arithetic expression:
C           
      PAD = ( WINSIZ - 1 ) / 2

C
C     Initialize the index in the packet buffer of the first packet in
C     the segment's coverage interval.  Note that since segments may
C     contain pad packets, STIDX does not necessarily point to the
C     first packet to be written to a segment.
C
      STIDX = 1

C
C     Initialize the packet, interval, and segment counts.
C      
      N     = 0
      NINTS = 1
      NSEG  = 0
     
C
C     Get the SCLK ID associated with the instrument.  
C
      CALL CKMETA ( INST, 'SCLK', CLKID )

C
C
C     Indicate presence of angular velocity.
C
      AVFLAG = .TRUE.

C     
C     Read the first data line from the first block. 
C
      CALL PRSDR ( INPUT, ATTTYP, SUBTYP, TIMSYS, ET, PACKET, FOUND )

      STARTS(1) = ET


      DO WHILE (   FOUND  .AND.  ( .NOT. FAILED() )   ) 
C
C        Transform the packet from MEX-style to SPICE-style
C        quaternions, and if applicable, quaternion derivatives and
C        angular velocity.
C
         CALL PKTRAN ( SUBTYP, PACKET, TPACK  )
         CALL MOVED  ( TPACK,  PACKSZ, PACKET )

         N  =  N + 1

         TO = (N-1)*PACKSZ + 1

         CALL MOVED ( PACKET, PACKSZ, PKBUFF(TO) )

C
C        Capture the time tag.
C
         TBUFF(N) = ET


         IF ( N .EQ. MAXPK ) THEN
C
C           It's time to emit a segment.  We'll use the last PAD
C           packets as padding, so the last packet in the coverage
C           interval is at index N-PAD.  We use the variable LIDX
C           to refer to the index of the index in the packet buffer
C           of the last packet in the segment's coverage interval.
C
            LIDX  = N - PAD

            FIRST = MAX ( KBEG, TBUFF(STIDX) )
            LAST  = MIN ( KEND, TBUFF(LIDX)  )

            IF ( FIRST .LT. LAST ) THEN

C              Determine the range of packets that actually need to be
C              written. B and E are the indices of these packets. The
C              range allows for padding on each side of the segment
C              bounds.  We constrain B and E to lie in the range 1:N.
C
               B   = MAX ( 1,  LSTLED ( FIRST, N, TBUFF ) - PAD      )
               E   = MIN ( N,  LSTLTD ( LAST,  N, TBUFF ) + PAD + 1  )

C  
C              NWR is the number of packets to write.
C
               NWR = E - B + 1

C
C              PKB is the packet buffer start index.
C
               PKB = (B-1)*PACKSZ + 1

               
               IF ( B .GT. 1 ) THEN
C
C                 Modify the interpolation interval start time
C                 buffer as needed.  Any start times greater than
C                 TBUFF(B) remain unchanged.  If TBUFF(B) is not
C                 itself a start time, it becomes one.
C
                  LOC = LSTLED ( TBUFF(B), NINTS, STARTS )

                  IF ( LOC .GT. 0 ) THEN

                     STARTS(LOC) = TBUFF(B)
                  
                  ELSE

                     CALL SETMSG ( 'Case:  emitting segment because ' //
     .                             'packet buffer is full.  No '      //
     .                             'interval start time was '         //
     .                             'less than or equal to SCLK '      //
     .                             'epoch#.'                          )
                     CALL ERRDP  ( '#', TBUFF(B)                      )
                     CALL SIGERR ( 'SPICE(BUG)'                       )
                     CALL CHKOUT ( 'MKCKSG'                           )
                     RETURN

                  END IF

               ELSE
C
C                 The first epoch of a segment must always be a start
C                 time.
C
                  LOC = 1

               END IF

C
C              Convert time tags to SCLK.
C
               DO I = B, E
                  CALL SCE2C ( CLKID, TBUFF(I), SCLKDP(I-B+1) )
               END DO

               DO I = LOC, NINTS
                  CALL SCE2C ( CLKID, STARTS(I), STCLKS(I-LOC+1) )
               END DO

               NINTS = NINTS - LOC + 1

               CALL SCE2C ( CLKID, FIRST, SEGBEG )
               CALL SCE2C ( CLKID, LAST,  SEGEND )

C
C              Write out packets B : E.
C
               CALL CKW05 ( HANDLE,  SUBTYP,  DEGREE,  SEGBEG,  
     .                      SEGEND,  INST,    FRAME,   AVFLAG, 
     .                      SEGID,   NWR,     SCLKDP,  PKBUFF(PKB),
     .                      RATE,    NINTS,   STCLKS                ) 

C
C              Increment the count of segments written.
C
               NSEG = NSEG + 1

            END IF

C
C           The following actions are necessary whether or not we
C           actually had to write a segment.
C
C           Shift the last (2*PAD+1) packets into the beginning 
C           of the buffer.  Shift the corresponding epochs as well.
C           
            SHIFT = 2 * PAD  +  1

            DO I = 1, SHIFT

               FROM     =    ( N - SHIFT - 1 + I )*PACKSZ  +  1
               TO       =    ( I - 1             )*PACKSZ  +  1

               CALL MOVED( PKBUFF(FROM), PACKSZ, PKBUFF(TO) )

               TBUFF(I) = TBUFF(N-SHIFT+I)

            END DO

            N = SHIFT

C
C           We must adjust the interval start time array as well.
C           The first element of the array is just the first time
C           tag in the shifted time tag buffer.  Earlier start times
C           are discarded.
C
            LOC = LSTLED ( TBUFF(1), NINTS, STARTS )

            IF ( LOC .GE. 1 ) THEN
C
C              The first interpolation interval starts with the first
C              buffered time tag.
C
               TO        = 1
               STARTS(1) = TBUFF(1)

               DO I = LOC+1, NINTS

                  TO         = TO + 1
                  STARTS(TO) = STARTS(I)

               END DO

               NINTS = TO

            ELSE
C
C              All start times in the buffer are greater than TBUFF(1).
C
               DO I = NINTS, 1, -1

                  STARTS(I+1) = STARTS(I)

               END DO

               STARTS(1) = TBUFF(1)
               NINTS     = NINTS + 1

            END IF

C
C           Set the segment coverage start index.
C
            STIDX = PAD + 1


         END IF

C
C        Get the next packet and epoch.
C         
         CALL PRSDR ( INPUT, ATTTYP, SUBTYP, TIMSYS, ET, PACKET, FOUND )

         IF ( .NOT. FOUND ) THEN 
C
C           If we're out of data, we may be at the end of the current
C           data block, or we may be at the end of the file.  Try to
C           read a line from the file and categorize it.
C
            CALL RDLIN ( INPUT, LINE, LNUM, TYPE, EOF ) 

            IF ( .NOT. EOF ) THEN

               IF ( TYPE .EQ. 'METADATA_START_LINE' ) THEN
C
C                 Parse the following metadata block. 
C
                  CALL PRSMET ( INPUT, FTYPE, STYPE, INSTR, CTR, 
     .                          REF,   BEG,   END,   TSYS,  CRTTIM )

                  IF ( FAILED() ) THEN
                     CALL CHKOUT ( 'MKCKSG' )
                     RETURN
                  END IF

C
C                 Try to read the first data line of the new block.
C
                  CALL PRSDR ( INPUT, ATTTYP, SUBTYP, TIMSYS, 
     .                         ET,    PACKET, FOUND          )

C
C                 If we didn't find a data line, we have an input
C                 file format error.
C
                  IF ( .NOT. FOUND ) THEN
C
C                    We have an input file format error.
C
                     CALL SETMSG ( 'No data found following '    //
     .                             'metadata block having time ' //
     .                             'bounds #:#.'                 )
                     CALL ERRDP  ( '#', KBEG                     )
                     CALL ERRDP  ( '#', KEND                     )
                     CALL SIGERR ( 'SPICE(FORMATERROR)'          )
                     CALL CHKOUT ( 'MKCKSG'                      )
                     RETURN

                  END IF
C
C                 If the new block starts before the kernel end
C                 time KEND, we'll need to process data from the
C                 block.  If we're terminating the current segment,
C                 we must write out the currently buffered data.
C               
C                 If there's a gap between the coverage interval of
C                 the old block and the new one, we can start
C                 a new interpolation interval .  If the blocks are
C                 contiguous, we'll need to start a new segment.
C
                  IF (        ( BEG .GT. TBUFF(N) ) 
     .                 .AND.  ( BEG .LE. KEND     )  ) THEN
C
C                    There's a gap between the end time of the previous
C                    block and the start time of the new block.    Data
C                    from the new block simply become part of a new
C                    interpolation interval. We continue only since we
C                    need data coverage from the new block (BEG .LE.
C                    KEND).
C
C                    The metadata stop line will have been read at this
C                    point. The data lines following the metadata block
C                    may be processed.
C
C                    Set the new interval start time.
C
                     NINTS         = NINTS + 1
                     STARTS(NINTS) = ET


C                    We've skipped over a metadata block and read the
C                    first data line of the new block.  We've started a
C                    new interpolation interval.
C
                  ELSE 
C
C                    Either we don't need more data, or we have
C                    to start a new segment because the new block
C                    is contiguous with the old one.
C
C                    Remaining buffered data have to be written to the
C                    current segment.
C         
C                    If the new block is contiguous with the old one,
C                    we'll have to start a new segment.
C
C                    Don't try to write a segment with only one
C                    data point.
C
                     IF ( N .GT. 1 ) THEN
C
C                       We do need to write a segment, presuming
C                       the requested coverage interval is non-empty.
C                        
                        FIRST = MAX ( KBEG, TBUFF(STIDX) )
                        LAST  = MIN ( KEND, TBUFF(N)     )


                        IF ( FIRST .LT. LAST ) THEN
C
C                          Determine the range of packets that actually
C                          need to be written. B and E are the indices
C                          of these packets. The range allows for
C                          padding on each side of the segment bounds.
C                          We constrain B and E to lie in the range
C                          1:N.
C
                           B = MAX ( 1,   
     .                               LSTLED(FIRST, N, TBUFF) - PAD )

                           E = MIN ( N,   
     .                               LSTLTD(LAST,  N, TBUFF) + PAD + 1 )

C  
C                          NWR is the number of packets to write.
C
                           NWR = E - B + 1

C
C                          PKB is the packet buffer start index.
C
                           PKB = (B-1)*PACKSZ + 1

               
                           IF ( B .GT. 1 ) THEN
C
C                             Modify the interpolation interval start
C                             time buffer as needed.  Any start times
C                             greater than TBUFF(B) remain unchanged.
C                             If TBUFF(B) is not itself a start time, it
C                             becomes one.
C
                              LOC = LSTLED ( TBUFF(B), NINTS, STARTS )

                              IF ( LOC .GT. 0 ) THEN

                                 STARTS(LOC) = TBUFF(B)
                  
                              ELSE

                                 CALL SETMSG ( 'No interval start ' //
     .                                         'time was less than '//
     .                                         'or equal to SCLK '  //
     .                                         'epoch#.'            )
                                 CALL ERRDP  ( '#', TBUFF(B)        )
                                 CALL SIGERR ( 'SPICE(BUG)'         )
                                 CALL CHKOUT ( 'MKCKSG'             )
                                 RETURN

                              END IF

                           ELSE

                              LOC = 1

                           END IF
 
C
C                          Convert time tags to SCLK.
C
                           DO I = B, E

                              CALL SCE2C ( CLKID, 
     .                                     TBUFF(I), 
     .                                     SCLKDP(I-B+1) )
                           END DO


                           DO I = LOC, NINTS

                              CALL SCE2C ( CLKID, 
     .                                     STARTS(I), 
     .                                     STCLKS(I-LOC+1) )
                           END DO

                           NINTS = NINTS - LOC + 1
 
                           CALL SCE2C ( CLKID, FIRST, SEGBEG )
                           CALL SCE2C ( CLKID, LAST,  SEGEND )

C
C                          Write out packets B : E.
C
                           CALL CKW05 ( HANDLE,  SUBTYP,  DEGREE,     
     .                                  SEGBEG,  SEGEND,  INST,   
     .                                  FRAME,   AVFLAG,  SEGID, 
     .                                  NWR,     SCLKDP,  PKBUFF(PKB),
     .                                  RATE,    NINTS,   STCLKS      ) 
                           
                           NSEG = NSEG + 1

                        END IF

                     END IF

                     IF ( BEG .LE. KEND ) THEN
C
C                       Reset our counters; set the first interpolation
C                       interval start time.
C
                        N         = 0
                        STIDX     = 1
                        NINTS     = 1                        
                        STARTS(1) = ET

C
C                       We'll go back to the top of the loop to
C                       process the first data packet of the new
C                       segment.
C
                     END IF

                  END IF

C
C                 If the new block starts after the requested end time,
C                 we're done.
C
                  IF ( BEG .GT. KEND ) THEN
                     
                     CALL CKCLS ( HANDLE )

                     CALL CHKOUT ( 'MKCKSG' )
                     RETURN

                  END IF       
C
C                 We've decided how to handle the new data block.
C
               ELSE
C
C                 After running out of data lines, we found something
C                 other than a meta data block.  This was certainly
C                 unexpected.
C
                  CALL SETMSG ( 'Input file line was expected to be ' //
     .                          'metadata block start marker but '    //
     .                          'in fact was of type #.'              )
                  CALL ERRCH  ( '#', TYPE                             )
                  CALL SIGERR ( 'SPICE(FORMATERROR)'          )
                  CALL CHKOUT ( 'MKCKSG'                      )
                  RETURN


               END IF
C
C              We've handled the next line after the last data line
C              of the preceding block.
C
            END IF
C
C           We've processed the next input line if there was one.  If
C           we're at the end of the input file, we'll drop out of the
C           loop at the next termination test.
C
         END IF
C
C        FOUND has been set.  If true, we've fetched a data line.  
C        Otherwise, we're at the EOF.
C
      END DO

C
C     Don't write a segment with only one data point.
C
      IF ( N .LE. 1 ) THEN

         CALL CKCLS ( HANDLE )

         CALL CHKOUT ( 'MKCKSG' )
         RETURN

      END IF

C
C     At this point, we're going to prepare to write a segment.
C
C     Set the segment coverage bounds.
C
      FIRST = MAX ( KBEG, TBUFF(STIDX) )
      LAST  = MIN ( KEND, TBUFF(N)     )

      IF ( FIRST .LT. LAST ) THEN
C
C        Determine the range of packets that actually need to be
C        written. B and E are the indices of these packets. The range
C        allows for padding on each side of the segment bounds.  We
C        constrain B and E to lie in the range 1:N.
C
         B   = MAX ( 1,   LSTLED ( FIRST, N, TBUFF ) - PAD      )
         E   = MIN ( N,   LSTLTD ( LAST,  N, TBUFF ) + PAD + 1  )

C  
C        NWR is the number of records to write.
C
         NWR = E - B + 1

C
C        PKB is the packet buffer start index.
C
         PKB = (B-1)*PACKSZ + 1


         IF ( B .GT. 1 ) THEN
C
C           Modify the interpolation interval start time
C           buffer as needed.  Any start times greater than
C           TBUFF(B) remain unchanged.  If TBUFF(B) is not
C           itself a start time, it becomes one.
C
            LOC = LSTLED ( TBUFF(B), NINTS, STARTS )

            IF ( LOC .GT. 0 ) THEN

               STARTS(LOC) = TBUFF(B)
                  
            ELSE

               CALL SETMSG ( '(2) No interval start time was ' //
     .                       'less than or equal to SCLK '     //
     .                       'epoch#.'                         )
               CALL ERRDP  ( '#', TBUFF(B)                     )
               CALL SIGERR ( 'SPICE(BUG)'                      )
               CALL CHKOUT ( 'MKCKSG'                          )
               RETURN

            END IF

         ELSE

            LOC = 1 

         END IF

C
C        Convert time tags to SCLK.
C
         DO I = B, E
             CALL SCE2C ( CLKID, TBUFF(I), SCLKDP(I-B+1) )
         END DO

         DO I = LOC, NINTS
            CALL SCE2C ( CLKID, STARTS(I), STCLKS(I-LOC+1) )
         END DO

         NINTS = NINTS - LOC + 1

         CALL SCE2C ( CLKID, FIRST, SEGBEG )
         CALL SCE2C ( CLKID, LAST,  SEGEND )

C
C        Write out packets B : E.
C
         CALL CKW05 ( HANDLE,  SUBTYP,  DEGREE,   SEGBEG,  
     .                SEGEND,  INST,    FRAME,    AVFLAG, 
     .                SEGID,   NWR,     SCLKDP,   PKBUFF(PKB),
     .                RATE,    NINTS,   STCLKS                 ) 

      END IF
 
C
C     We've finished processing the attitude file.  Close the CK.
C
      CALL CKCLS ( HANDLE )

      CALL CHKOUT ( 'MKCKSG' )
      RETURN 
      END


