import os
import logging
from datetime import datetime, timedelta
from scipy.spatial.transform import Rotation as R


def fecs_processing(event_list, kernel):

    directions = ''
    orientations = ''
    timeline = ''
    first = True
    previous_event = False
    angles = []
    prev_end_angle = ''

    for event in event_list:

        #
        # We have two types of events:
        #    1) Normal Rolls      (ROLL)
        #    2) Calibration Rolls (CALIB_ROLL_MAG)
        #    3) Calibration Rolls (CALIB_ROLL_RPW)
        #
        # <ROLL time="2020-166T00:32:46Z" duration="1052" start_roll_deg="0" end_roll_deg="45"/><!-- SCI_ROLL in v1_2 -->
        # <CALIB_ROLL_MAG time="2025-335T12:34:00Z" duration="47851" n_rolls="12" comment="Constant 0.1 deg/s reached 1000 seconds after the event time and maintained during 12 rotations"/>
        # <CALIB_ROLL_RPW time="2025-351T19:00:00Z" duration="21600" n_rolls="6"/>
        #
        event_type = event.split()[0][1:]

        #
        # Note that for the events to be valid, start_roll_deg must be equal
        # to the value of end_roll_deg for the previous roll event.
        #
        if event_type == 'ROLL':
            srt_angle = event.split('start_roll_deg')[-1].split('"')[1]
            if not first and not previous_event:
                prev_end_angle = end_angle
                if not float(srt_angle) % 360 == float(prev_end_angle) % 360:
                    raise ValueError('Start Roll Angle different than previous End Roll Angle')
            end_angle = event.split('end_roll_deg')[-1].split('"')[1]
        else:
            if first:
                srt_angle = 0.0

        start_utc = event.split('"')[1]
        # 2020-092T04:34:10Z

        try:
            start_dt = datetime.strptime(start_utc, "%Y-%jT%H:%M:%SZ")
        except:
            start_dt = datetime.strptime(start_utc, "%Y-%jT%H:%M:%S.%fZ")

        duration = event.split('"')[3]

        if first:

            start_ck_dt = start_dt + timedelta(minutes=-1)
            start_ck_utc = start_ck_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")

            if float(srt_angle) == 0.0:

                timeline = timeline + \
                       "      CK-144000ORIENTATION  += 'ORB_NORM'\n" + \
                       "      CK-144000START        += {}\n".format(start_ck_utc)

            else:

                angles.append(srt_angle)
                roll_att = 'ROLL{}'.format(srt_angle)

                directions = directions + \
                     "         DIRECTION_SPECS +=  ('ROT{}      =  ROTATE CROSS     -')\n".format(srt_angle) + \
                     "         DIRECTION_SPECS +=  ('{} DEGREES                     -')\n".format(srt_angle) + \
                     "         DIRECTION_SPECS +=  ('ABOUT TOSUN'                     )\n\n"

                orientations = orientations + \
                       "      \\begindata\n\n" + \
                       "         ORIENTATION_NAME += 'ROLL{}'\n".format(srt_angle) + \
                       "         PRIMARY          += '+X = TOSUN'\n" + \
                       "         SECONDARY        += '+Z = ROT{}'\n".format(srt_angle) + \
                       "         BASE_FRAME       += 'J2000'\n\n" + \
                       "      \\begintext\n\n\n"

                timeline = timeline +\
                           "      CK-144000ORIENTATION  += '{}'\n".format(roll_att) + \
                           "      CK-144000START        += {}\n".format(start_ck_utc)

        if event_type == 'ROLL':

            if float(end_angle) != 0.0:

                if end_angle not in angles:
                    directions = directions + \
                        "         DIRECTION_SPECS +=  ('ROT{}      =  ROTATE CROSS     -')\n".format(end_angle) + \
                        "         DIRECTION_SPECS +=  ('{} DEGREES                     -')\n".format(end_angle) + \
                        "         DIRECTION_SPECS +=  ('ABOUT TOSUN'                     )\n\n"

                    orientations = orientations + \
                        "      \\begindata\n\n" + \
                        "         ORIENTATION_NAME += 'ROLL{}'\n".format(end_angle) + \
                        "         PRIMARY          += '+X = TOSUN'\n" + \
                        "         SECONDARY        += '+Z = ROT{}'\n".format(end_angle) + \
                        "         BASE_FRAME       += 'J2000'\n\n" + \
                        "      \\begintext\n\n\n"

                start_cal_angle = 'ROLL{}'.format(end_angle)
            else:
                start_cal_angle = 'ORB_NORM'


            start_roll_dt = start_dt
            start_roll_utc = start_roll_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")
            timeline = timeline + \
                       "      CK-144000STOP         += {}\n\n".format(start_roll_utc)

            finish_roll_dt = start_dt + timedelta(seconds=float(duration))
            start_roll_utc = finish_roll_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")
            timeline = timeline + \
                   "      CK-144000ORIENTATION  += '{}'\n".format(start_cal_angle) + \
                   "      CK-144000START        += {}\n".format(start_roll_utc)

            first = False
            angles.append(end_angle)
            prev_end_angle = end_angle

        #
        # Here we implement the Calibration Roll
        #
        else:
            start_cal_angle = 0.0
            if prev_end_angle:
                start_cal_angle += float(prev_end_angle)

            n_rotations = int(kernel.config['calibration_rotations'])
            if kernel.input_type == 'EVENTS' and 'EFECS' in kernel.source:
                n_rotations = int(event.split('"')[5])
            rotation_duration = float(duration) / float(n_rotations)

            for number in range(n_rotations):

                if number == 0:

                    if start_cal_angle != 0.0 and start_cal_angle not in angles:


                        angles.append(start_cal_angle)
                        start_roll_att = 'ROLL{}'.format(srt_angle)

                        directions = directions + \
                                 "         DIRECTION_SPECS +=  ('ROT{}      =  ROTATE CROSS     -')\n".format(srt_angle) + \
                                 "         DIRECTION_SPECS +=  ('{} DEGREES                     -')\n".format(srt_angle) + \
                                 "         DIRECTION_SPECS +=  ('ABOUT TOSUN'                     )\n\n"

                        orientations = orientations + \
                                   "      \\begindata\n\n" + \
                                   "         ORIENTATION_NAME += '{}'\n".format(start_roll_att) + \
                                   "         PRIMARY          += '+X = TOSUN'\n" + \
                                   "         SECONDARY        += '+Z = ROT{}'\n".format(srt_angle) + \
                                   "         BASE_FRAME       += 'J2000'\n\n" + \
                                   "      \\begintext\n\n\n"
                    else:
                        start_roll_att = 'ORB_NORM'

                    cal_angle = start_cal_angle
                    cal_dt = start_dt
                    start_roll_dt = start_dt
                    start_roll_utc = start_roll_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")
                    cal_utc = start_roll_utc
                    timeline = timeline + \
                               "      CK-144000STOP         += {}\n\n".format(start_roll_utc)

                cal_angle += 90
                if cal_angle > 360:
                    cal_angle -= 360

                if cal_angle not in angles:
                    directions = directions + \
                         "         DIRECTION_SPECS +=  ('ROT{}      =  ROTATE CROSS     -')\n".format(str(cal_angle).split('.')[0]) + \
                         "         DIRECTION_SPECS +=  ('{} DEGREES                     -')\n".format(cal_angle) + \
                         "         DIRECTION_SPECS +=  ('ABOUT TOSUN'                     )\n\n"

                    orientations = orientations + \
                           "      \\begindata\n\n" + \
                           "         ORIENTATION_NAME += 'ROLL{}'\n".format(str(cal_angle).split('.')[0]) + \
                           "         PRIMARY          += '+X = TOSUN'\n" + \
                           "         SECONDARY        += '+Z = ROT{}'\n".format(str(cal_angle).split('.')[0]) + \
                           "         BASE_FRAME       += 'J2000'\n\n" + \
                           "      \\begintext\n\n\n"

                    angles.append(cal_angle)

                previous_cal_utc = cal_utc
                cal_dt += timedelta(seconds=0.001)
                cal_utc = cal_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")
                timeline = timeline + \
                       "      CK-144000ORIENTATION  += 'ROLL{}'\n".format(str(cal_angle).split('.')[0]) + \
                       "      CK-144000START        += {}\n".format(previous_cal_utc) + \
                       "      CK-144000STOP         += {}\n\n".format(cal_utc)
                cal_dt += timedelta(seconds=rotation_duration / 4.)
                cal_utc = cal_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")

                cal_angle += 90
                if cal_angle > 360:
                    cal_angle -= 360

                if cal_angle not in angles:
                    directions = directions + \
                         "         DIRECTION_SPECS +=  ('ROT{}      =  ROTATE CROSS     -')\n".format(str(cal_angle).split('.')[0]) + \
                         "         DIRECTION_SPECS +=  ('{} DEGREES                     -')\n".format(cal_angle) + \
                         "         DIRECTION_SPECS +=  ('ABOUT TOSUN'                     )\n\n"

                    orientations = orientations + \
                           "      \\begindata\n\n" + \
                           "         ORIENTATION_NAME += 'ROLL{}'\n".format(str(cal_angle).split('.')[0]) + \
                           "         PRIMARY          += '+X = TOSUN'\n" + \
                           "         SECONDARY        += '+Z = ROT{}'\n".format(str(cal_angle).split('.')[0]) + \
                           "         BASE_FRAME       += 'J2000'\n\n" + \
                           "      \\begintext\n\n\n"

                    angles.append(cal_angle)

                previous_cal_utc = cal_utc
                cal_dt += timedelta(seconds=0.001)
                cal_utc = cal_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")
                timeline = timeline + \
                       "      CK-144000ORIENTATION  += 'ROLL{}'\n".format(str(cal_angle).split('.')[0]) + \
                       "      CK-144000START        += {}\n".format(previous_cal_utc) + \
                       "      CK-144000STOP         += {}\n\n".format(cal_utc)
                cal_dt += timedelta(seconds=rotation_duration / 4.)
                cal_utc = cal_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")

                cal_angle += 90
                if cal_angle > 360:
                    cal_angle -= 360

                if cal_angle not in angles:
                    directions = directions + \
                         "         DIRECTION_SPECS +=  ('ROT{}      =  ROTATE CROSS     -')\n".format(str(cal_angle).split('.')[0]) + \
                         "         DIRECTION_SPECS +=  ('{} DEGREES                     -')\n".format(cal_angle) + \
                         "         DIRECTION_SPECS +=  ('ABOUT TOSUN'                     )\n\n"

                    orientations = orientations + \
                           "      \\begindata\n\n" + \
                           "         ORIENTATION_NAME += 'ROLL{}'\n".format(str(cal_angle).split('.')[0]) + \
                           "         PRIMARY          += '+X = TOSUN'\n" + \
                           "         SECONDARY        += '+Z = ROT{}'\n".format(str(cal_angle).split('.')[0]) + \
                           "         BASE_FRAME       += 'J2000'\n\n" + \
                           "      \\begintext\n\n\n"

                    angles.append(cal_angle)

                previous_cal_utc = cal_utc
                cal_dt += timedelta(seconds=0.001)
                cal_utc = cal_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")
                timeline = timeline + \
                       "      CK-144000ORIENTATION  += 'ROLL{}'\n".format(str(cal_angle).split('.')[0]) + \
                       "      CK-144000START        += {}\n".format(previous_cal_utc) + \
                       "      CK-144000STOP         += {}\n\n".format(cal_utc)
                cal_dt += timedelta(seconds=rotation_duration / 4.)
                cal_utc = cal_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")

                cal_angle += 90
                if cal_angle > 360:
                    cal_angle -= 360

                if cal_angle not in angles:
                    directions = directions + \
                         "         DIRECTION_SPECS +=  ('ROT{}      =  ROTATE CROSS     -')\n".format(str(cal_angle).split('.')[0]) + \
                         "         DIRECTION_SPECS +=  ('{} DEGREES                     -')\n".format(cal_angle) + \
                         "         DIRECTION_SPECS +=  ('ABOUT TOSUN'                     )\n\n"

                    orientations = orientations + \
                           "      \\begindata\n\n" + \
                           "         ORIENTATION_NAME += 'ROLL{}'\n".format(str(cal_angle).split('.')[0]) + \
                           "         PRIMARY          += '+X = TOSUN'\n" + \
                           "         SECONDARY        += '+Z = ROT{}'\n".format(str(cal_angle).split('.')[0]) + \
                           "         BASE_FRAME       += 'J2000'\n\n" + \
                           "      \\begintext\n\n\n"

                    angles.append(cal_angle)

                previous_cal_utc = cal_utc
                cal_dt += timedelta(seconds=0.001)
                cal_utc = cal_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")
                timeline = timeline + \
                       "      CK-144000ORIENTATION  += 'ROLL{}'\n".format(str(cal_angle).split('.')[0]) + \
                       "      CK-144000START        += {}\n".format(previous_cal_utc) + \
                       "      CK-144000STOP         += {}\n\n".format(cal_utc)
                cal_dt += timedelta(seconds=rotation_duration / 4.)
                cal_utc = cal_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")


            cal_dt_end = cal_dt + timedelta(seconds=1)
            cal_utc_end = cal_dt_end.strftime("@%Y-%m-%dT%H:%M:%S.%f")
            timeline = timeline + \
                       "      CK-144000ORIENTATION  += '{}'\n".format(start_roll_att) + \
                       "      CK-144000START        += {}\n".format(cal_utc_end)

            first = False
            previous_event = True

            #
            # Just in case this is the last event we need to update the
            # start_roll_utc for the last entry in the timeline string
            # at the end of the loop.
            #
            start_roll_dt = cal_dt_end + timedelta(seconds=60*60)
            start_roll_utc = start_roll_dt.strftime("@%Y-%m-%dT%H:%M:%S.%f")

            #
            # Just in case the calib roll is the first event
            #
            end_angle = start_cal_angle


    timeline = timeline + \
                   "      CK-144000STOP         += {}\n\n".format(start_roll_utc)

    kernel.config['directions'] = directions
    kernel.config['orientations'] = orientations
    kernel.config['timeline'] = timeline

    return


def fecs_extending(kernel, orbit_end_date):
    timeline = kernel.config['timeline']
    previous_period = timeline.split('\n')[:-5]
    extended_period = timeline.split('\n')[-5:]
    #
    # in the case last period had duration 0
    #
    if extended_period[1].split('@')[1] == extended_period[2].split('@')[1]:
        extended_period[2] = extended_period[2].split('@')[0] + '@' + orbit_end_date
        timeline = '\n'.join(previous_period) + '\n' + '\n'.join(extended_period)
    #
    # otherwise add an extra period extending last roll to orbit end date
    #
    else:
        extended_period[1] = extended_period[1].split('@')[0] + '@' + extended_period[2].split('@')[1]
        extended_period[2] = extended_period[2].split('@')[0] + '@' + orbit_end_date
        timeline = timeline + '\n'.join(extended_period)
    kernel.config['timeline'] = timeline
    return


def events_filter(path, sources):

    new_filenames = []

    for source in sources:
        if source[1]['input_type'] == 'TM_QUATERNIONS' and not 'source_list' in source[1].keys():

            source[1]['source_list'] = list()
            input_file =  path + '/' + source[0]

            dataFiles = {}
            fileHandler = open(input_file, "r", encoding='utf-8')
            listOfLines = fileHandler.readlines()
            fileHandler.close()
            #
            # Homogeneize & remove duplicate lines
            #
            listOfLines = list(map(str.strip, listOfLines))
            listOfLines = list(dict.fromkeys(listOfLines))

            first_line = True
            index = 0
            file_number = 0
            for line in listOfLines:
                if first_line:
                    file = path + '/' + source[0].split('.')[0] + '_' + '{0:03}'.format(file_number) + '__adcsng_temp.txt'
                    dataFiles[file] = []
                    first_line = False
                    if len(dataFiles) != 0 and 'EVENT' not in line:
                        dataFiles[file].append(line + '\n')
                        previous_line = line
                    elif 'EVENT' in line:
                        pass

                elif len(dataFiles[file]) == 1 and 'EVENT' in line: pass
                elif 'EVENT' not in line:
                    dataFiles[file].append(line+'\n')
                    previous_line = line
                elif 'EVENT' in line:
                    previous_time = line.split(',')[0]
                    dataFiles[file].append('{},{},{},{},{}\n'.format(previous_time,
                                                    previous_line.split(',')[1],
                                                    previous_line.split(',')[2],
                                                    previous_line.split(',')[3],
                                                    previous_line.split(',')[4]))
                    try:
                        next_line_number = index
                        next_line = listOfLines[next_line_number]
                        while 'EVENT' in next_line:
                            next_line_number += 1
                            next_line = listOfLines[next_line_number]
                        next_time = next_line.split(',')[0]

                        delta_time = int(next_time.split('/')[-1].split(':')[0]) - int(previous_time.split('/')[-1].split(':')[0])
                        if delta_time < 60:
                            dataFiles[file].append('{},{},{},{},{}\n'.format(next_time,
                                                          previous_line.split(',')[1],
                                                          previous_line.split(',')[2],
                                                          previous_line.split(',')[3],
                                                          previous_line.split(',')[4]))
                    except:
                        pass
                    first_line = True
                    file_number += 1
                index += 1

    try:
        for file in dataFiles:
            with open(file, 'w') as f:
                for line in dataFiles[file]:
                    f.write(line)
            #
            # We need to add the new source files.
            #
            source[1]['source_list'].append(file.split('/')[-1])
            new_filenames.append(file.split('/')[-1])
    except:
        return new_filenames


    return new_filenames


def boresights_filter(path, sources):

    new_filenames = []
    for source in sources:
        if source[1]['input_type'] == 'BSUP' and \
                            'adcsng_temp' not in source[0]:
            #
            # We will create a dictionary inside a dictionary per OBJECT_NAME of the
            # AEM, the dictionary will have a 'ref_frame_a' key a ref_frame_b key and
            # a object_id key. Finally there will be a data key with a list of strings.
            #
            boresights = {}

            input_file =  path + '/' + source[0]

            with open(input_file, 'r') as t:
                detector_flag = False
                optical_flag = False
                for line in t:
                    #
                    # read Detector angles
                    #
                    if '<detector>' in line:
                        detector_flag = True
                    if 'detectorID' in line and detector_flag:
                        object_name = line.split('>', 1)[1].split('<')[0]
                        if object_name not in boresights.keys():
                            boresights[object_name] = {'object_name':object_name}
                            boresights[object_name]['DATA'] = []
                            boresights[object_name]['OBJECT_ID'] = source[1]['detectors'][0][object_name][0][
                                'instrument_id']
                            boresights[object_name]['OBJECT_FRAME'] = source[1]['detectors'][0][object_name][0][
                                'instrument_frame']
                            boresights[object_name]['REF_FRAME_A'] = source[1]['detectors'][0][object_name][0][
                                'frame_id']
                            boresights[object_name]['REF_FRAME_B'] = source[1]['detectors'][0][object_name][0][
                                'instrument_frame']
                    if 'calibrationDataTime' in line and detector_flag:
                        utc = line.split('>', 1)[1].split('Z<')[0]
                    if 'yaw' in line and detector_flag:
                        yaw = line.split('>', 1)[1].split('<')[0]
                        yaw = float(yaw)/3600   # arcseconds units
                    if 'pitch' in line and detector_flag:
                        pitch = line.split('>', 1)[1].split('<')[0]
                        pitch = float(pitch) / 3600  # arcseconds units
                    if 'roll' in line and detector_flag:
                        roll = line.split('>', 1)[1].split('<')[0]
                        roll = float(roll) / 3600  # arcseconds units
                    if '</detector>' in line and detector_flag:
                        data = [utc, yaw, pitch, roll]
                        boresights[object_name]['DETECTOR_ANGLES'] = data
                        if not 'OPTICAL_ANGLES' in boresights[object_name]:
                            boresights[object_name]['OPTICAL_ANGLES'] = [utc, 0, 0]
                        detector_flag = False

                    #
                    # read Optical angles
                    #
                    if '<optical>' in line:
                        optical_flag = True
                    if 'detectorID' in line and optical_flag:
                        object_name = line.split('>', 1)[1].split('<')[0]
                        if object_name not in boresights.keys():
                            boresights[object_name] = {'object_name': object_name}
                            boresights[object_name]['DATA'] = []
                            boresights[object_name]['OBJECT_ID'] = source[1]['detectors'][0][object_name][0][
                                'instrument_id']
                            boresights[object_name]['OBJECT_FRAME'] = source[1]['detectors'][0][object_name][0][
                                'instrument_frame']
                            boresights[object_name]['REF_FRAME_A'] = source[1]['detectors'][0][object_name][0][
                                'frame_id']
                            boresights[object_name]['REF_FRAME_B'] = source[1]['detectors'][0][object_name][0][
                                'instrument_frame']
                    if 'calibrationDataTime' in line and optical_flag:
                        utc = line.split('>', 1)[1].split('Z<')[0]
                    if 'yaw' in line and optical_flag:
                        yaw = line.split('>', 1)[1].split('<')[0]
                        yaw = float(yaw) / 3600  # arcseconds units
                    if 'pitch' in line and optical_flag:
                        pitch = line.split('>', 1)[1].split('<')[0]
                        pitch = float(pitch) / 3600  # arcseconds units
                    if '</optical>' in line and optical_flag:
                        data = [utc, yaw, pitch]
                        boresights[object_name]['OPTICAL_ANGLES'] = data
                        if not 'DETECTOR_ANGLES' in boresights[object_name]:
                            boresights[object_name]['DETECTOR_ANGLES'] = [utc, 0, 0, 0]
                        optical_flag = False

            #
            # concatenate 5 rotations (3 detectors + 5 optical) and prepare
            # msopck input as 3 rotations z-y-x
            #
            for object_name in boresights:
                r1 = R.from_euler('z', boresights[object_name]['DETECTOR_ANGLES'][1], degrees=True)
                r2 = R.from_euler('y', boresights[object_name]['DETECTOR_ANGLES'][2], degrees=True)
                r3 = R.from_euler('x', boresights[object_name]['DETECTOR_ANGLES'][3], degrees=True)
                r4 = R.from_euler('z', boresights[object_name]['OPTICAL_ANGLES'][1], degrees=True)
                r5 = R.from_euler('y', boresights[object_name]['OPTICAL_ANGLES'][2], degrees=True)
                M = r5 * r4 * r3 * r2 * r1
                angles = M.as_euler('zyx', degrees=True)
                yaw = str(angles[0])
                pitch = str(angles[1])
                roll = str(angles[2])
                data = boresights[object_name]['DETECTOR_ANGLES'][0] + ' ' + yaw + ' ' + pitch + ' ' + roll + '\n'
                boresights[object_name]['DATA'].append(data)

            #
            # extend last segment to date specified in config
            #
            if 'extend_date' in source[1]:
                for object_name in boresights.keys():
                    data = source[1]['extend_date'] + ' ' + boresights[object_name]['DATA'][-1].split(' ', 1)[1]
                    boresights[object_name]['DATA'].append(data)


            #
            # Now we need to generate the new inputs, one per mechanism.
            # This new inputs, because they have different structures
            # need to be specified in the exomarsrsp.json configuration
            # file.
            #
            for key in boresights.keys():
                new_filename = source[0].split('.')[0]+f'_adcsng_temp_{key.lower()}.bsup'
                input_file = path + '/' + new_filename
                new_filenames.append(new_filename)

                #
                # We also provide the MSPOCK_QUAT configuration per source
                # (In the case of Solar Orbiter is not necessary because they
                # are all the same).
                #
                source[1][new_filename] = {}
                source[1][new_filename]["target_frame"]  = boresights[key]['REF_FRAME_A']
                source[1][new_filename]["object_id"]    = boresights[key]['OBJECT_ID']
                source[1][new_filename]["object_frame"]    = boresights[key]['OBJECT_FRAME']
                source[1][new_filename]["mission_accr"] = boresights[key]['REF_FRAME_B']

                with open(input_file, 'w') as t:
                    for line in boresights[key]['DATA']:
                        t.write(line)

                #
                # We need to add the new source files.
                #
                if not 'source_list' in source[1].keys():
                    source[1]['source_list'] = list()
                if not new_filename in source[1]['source_list']:
                    source[1]['source_list'].append(new_filename)

    return new_filenames

