# import the required modules
import glob
import os
import json
import re
import copy
import logging
import subprocess
from datetime import datetime

import adcsng
import adcsng.classes.kernel as Kernel
from adcsng.classes.exceptions import KernelNotFound
from adcsng.classes.missiondir import MissionDir
from adcsng.classes.comments import Comments
from adcsng.core.solo import events_filter
from adcsng.core.solo import boresights_filter
from adcsng.core.emrsp import mechanism_filter
from adcsng.core.readers import sclk_reader
from adcsng.utils.files import get_exe_dir
from adcsng.utils.files import kernel_type2extension
from adcsng.utils.files import get_date_from_filename
from adcsng.utils.files import commnt_read
from adcsng.utils.files import run_ckbrief


def generic_kernel_config(config, kernel, source, directories,
                          commnt_template='', fk='', lsk='', sclk='', mk=''):
    #
    # Add the kernels to be loaded. Note that if we are dealing with
    # an SPK, we do not have to load the SCLK.
    #
    kernel.set('ker2load', fk)
    kernel.set('ker2load', lsk)
    kernel.set('ker2load', sclk)
    kernel.set('ker2load', mk)
    #
    # Specify the kernel type and the input type and the originator
    #
    kernel.set('source_config', source)
    kernel.set('source_type', source[1]['kernel_type'])
    kernel.set('input_type', source[1]['input_type'])
    kernel.set('input_processing', source[1]['input_processing'])
    kernel.set('originator', source[1]['originator'])
    if 'orbnum' in source[1]:
        kernel.set('orbnum', source[1]['orbnum'])
    else:
        kernel.set('orbnum', 'False')

    #
    # Create the part of the output file name that we can produced
    # at this stage.
    #
    kernel.name.set('kernel_name', source[1]['kernel_name'])

    #
    # Some kernels don't have description in their filenames:
    #
    if 'description' in source[1]:
        description = '_' + source[1]['description']

    else:
        description = get_description(source[0],
                                      source[1]['input_description_index'])

    kernel.name.set('description', description)
    kernel.config['description'] = description
    kernel.config['adcsng_version'] = config['generic'][0]['adcsng_version']

    #
    # Setup the Comments object if present
    #
    if commnt_template:
        comments = Comments(kernel.config,
                            path=directories.templates,
                            template=commnt_template)

        kernel.set('comments', comments)
        kernel.comments.load_kernels(kernel.ker2load)

    return


def analyze_input_data(path, setup, sclk=False):
    """
    This function provides a list of sources.

    :param path: Directory where the source files are -incoming directory-
    :type path: str
    :param setup: Mission specific adcsng configuration from json file
    :type setup: dict
    :return: List of sources where each element of the list contains a source
       filename and the corresponding input
    :rtype: list
    """
    sources = []
    source_files = []

    input_types = setup["input_types"][0]

    for key, value in input_types.items():

        files = glob.glob(path + '/' + key)
        for filename in files:
            filename = filename.split('/')[-1]

            if sclk:
                if key == setup[value[0]['spacecraft']][0]["tcp_schema"]:
                    sources.append([filename, value[0].copy()])
                    source_files.append(filename)
            elif filename not in source_files:
                if key != setup[value[0]['spacecraft']][0]["tcp_schema"]:
                    sources.append([filename, value[0].copy()])
                    source_files.append(filename)

                #
                # We need to split the sources for Solar Orbiter and ExoMarsRSP
                #
                if 'solo' in setup.keys():
                    #
                    # filter events
                    #
                    new_filenames = events_filter(path, sources)
                    if new_filenames:
                        for name in new_filenames:
                            if name not in source_files:
                                source_files.append(name)
                    #
                    # filter boresight updates
                    #
                    new_filenames = boresights_filter(path, sources)
                    if new_filenames:
                        for name in new_filenames:
                            if name not in source_files:
                                source_files.append(name)

                if 'rm' in setup.keys():
                    new_filenames = mechanism_filter(path, sources)
                    for name in new_filenames:
                        if name not in source_files:
                            source_files.append(name)

    return sources, source_files


def configure_execution(configuration):
    """
    This function configures the adcsng execution by providing a list of
    kernel objects from the sources -and that need to be generated-, a
    directories object, that contains all the relevant directories to
    be used and a dictionary containing all the configuration information
    from the given mission from the original {mission}.json file.

    In order to provide this information this function scans the incoming
    directory and generates a kernel for each source file. Please note
    that if a given SPK kernel has a default orientation, this file
    is also generated.

    :param mission: Mission for which adcsng is executed
    :type mission: str
    :return: A List of Kernel Objects to be converted by adcsng, A MissionDir
        object that contains all the relevant directories for the execution and
        a dictionary containing the mission specific adcsng JSON configuration
    :rtype: (list, Object, dict)
    """

    #
    # We determine the platform system and machine for the executables
    #
    executables_dir = get_exe_dir()

    root_dir = os.path.dirname(adcsng.command_line.__file__)

    #
    # We check that the fields of the configuration are present
    #
    with open(root_dir + '/config/config.json', 'r') as f:
        config_template = json.load(f)

    for key in config_template.keys():
        if key not in configuration.keys():
            error_message = "Error: The ADCSng JSON configuration file has missing keys."
            raise NameError(error_message)

    configuration['root_dir'] = root_dir
    configuration['executables_dir'] = root_dir + executables_dir

    #
    # We read the mission configuration
    #
    config = get_config(configuration)

    #
    # To avoid repetition we create the setup variable for the generic part of
    # the configuration dictionary
    #
    setup = config['generic'][0]
    adcsng_config = configuration

    #
    # We implement the possibility of providing relative paths in the
    # configuration (in order to implement test)
    #
    if not os.path.isabs(adcsng_config['source_dir']):
        source = os.path.join(os.getcwd(), adcsng_config['source_dir'])
    else:
        source = adcsng_config['source_dir']
    if not os.path.isabs(adcsng_config['output_dir']):
        output = os.path.join(os.getcwd(), adcsng_config['output_dir'])
    else:
        output = adcsng_config['output_dir']
    if adcsng_config['processed_dir']:
        if not os.path.isabs(adcsng_config['processed_dir']):
            processed = os.path.join(os.getcwd(), adcsng_config['processed_dir'])
        else:
            processed = adcsng_config['processed_dir']
    else:
        processed = ''
    if not os.path.isabs(adcsng_config['kernels_dir']):
        kernels = os.path.join(os.getcwd(), adcsng_config['kernels_dir'])
    else:
        kernels = adcsng_config['kernels_dir']
    if not os.path.isabs(adcsng_config['temp_dir']):
        temp = os.path.join(os.getcwd(), adcsng_config['temp_dir'])
    else:
        temp = adcsng_config['temp_dir']
    if not os.path.isabs(adcsng_config['log_dir']):
        log = os.path.join(os.getcwd(), adcsng_config['log_dir'])
    else:
        log = adcsng_config['log_dir']
    try:
        if not os.path.isabs(adcsng_config['exec_log']):
            exec_log = os.path.join(os.getcwd(), adcsng_config['exec_log'])
        else:
            exec_log = adcsng_config['exec_log']
    except:
        exec_log = ''
    try:
        if not os.path.isabs(adcsng_config['reports_dir']):
            reports = os.path.join(os.getcwd(), adcsng_config['reports_dir'])
        else:
            reports = adcsng_config['reports_dir']
    except:
        reports = ''

    #
    # For convenience we generate an object that contains all the relevant
    # directories and the existance of the directories is checked.
    #
    directories = MissionDir(config,
                             mission=configuration['mission'],
                             root_dir=adcsng_config['root_dir'],
                             source=source,
                             output=output,
                             processed=processed,
                             kernels=kernels,
                             temp=temp,
                             templates='/$root_dir/etc',
                             executables=adcsng_config['executables_dir'],
                             log=log,
                             exec_log=exec_log,
                             reports=reports
                             )

    return setup, config, directories


def sources2kernels(setup, config, directories, sclk=False, original_sources=''):

    if not original_sources:
        original_sources = []
    #
    # We obtain the latest LSK present in the mission directories.
    #
    lsk = os.path.join('lsk', get_latest_kernel('lsk', directories.kernels, setup['lsk_schema']))

    #
    # The input data is analyzed to know which type of data it is and what kind
    # of kernel is to be generated. Note that if sclk is set to False,
    # TCP will not be searched for.
    #
    (input_data, source_files) = analyze_input_data(directories.source, setup, sclk=sclk)
    original_sources += source_files

    kernels = []
    source_tcp = []
    sources_mapps_quat = []
    sources_tm_map = {}
    sources_tm_quat_map = {}
    sources_tm_daf_map = {}
    sources_oem = []
    sources_aem_quat = []
    sources_bsup_ang = []
    sources_csmithed = []
    source_types_aem_quat = []
    source_types_bsup_ang = []
    source_types_mapps_quat = []
    input_data_elements = len(input_data)

    index = 1
    for source in input_data:

        # We discard empty files
        if not os.path.getsize(os.path.join(directories.source, source[0])):
            original_sources.remove(source[0])
            input_data_elements -= 1
            logging.warning(f'Source file: {source[0]} will not be processed because is an empty file.')
            logging.info('')
            continue

        #
        # We discard the sources that do not come from the appropriate sources
        # directory
        #
        if 'input_directory' in source[1]:
            if source[1]['input_directory'] != directories.source.split(os.sep)[-1]:
                original_sources.remove(source[0])
                logging.warning(f'Source file: {source[0]} will not be processed.')
                logging.warning(f'The source is only accepted from: {source[1]["input_directory"]}')
                logging.info('')
                continue

        if source[1]['input_type'] == 'TCP' or source[1]['input_type'] == 'TCP_MEX':
            source_tcp.append(source)

        if source_tcp and index == input_data_elements:

            #
            # We determine if several sources need to be merged
            #
            merged_sources = []
            if config['tcp2sclk'][0]['merge_sources'] == 'True':
                merged_sources = source_tcp

            sources = []

            for src in merged_sources:
                sources.append(src[0])

            kernel = Kernel.SCLK(sources, directories,
                                 Kernel.KernelName(source[1]['kernel_schema']))

            if source[1]['input_type'] == 'TCP_MEX':
                merged_sources = sorted(merged_sources)
                source = merged_sources[-1]

            kernel.set('config', source[1])

            #
            # We need to load the previous version of the SCLK, it's an input
            #
            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                     setup[source[1]['spacecraft']][0]['sclk_step_schema'])
            if source[1]['input_type'] == 'TCP_MEX' and sclk[:3] != 'sclk':
                sclk = 'sclk/' + sclk
                sclkid = re.findall('\d+', sclk)[-1]
                kernel.config['sclkid'] = sclkid

            if source[1]['input_type'] == 'TCP_MEX':
                commnt_template = 'makclk_mex.commnt'
                dt_creation = datetime.now()
                kernel.config['gen_utc'] = dt_creation.strftime("%Y-%m-%dT%H:%M:%S")
                kernel.config['mission_name'] = kernel.config['main_sc_name'].upper()
                kernel.config['filename'] = str(kernel.name)
                kernel.config['sclk_kernel'] = sclk
                kernel.config['lsk_kernel'] = lsk
                kernel.config['bad_tcps'] = config['tcp2scet'][0]['bad_tcps']
            else:
                commnt_template = 'tcp2sclk.commnt'

                if "sclk_partition_end" in source[1]:

                    # Prepare sclk_partition_starts and sclk_partition_ends token values to be replaced
                    # in the tcp2sclk.commnt file
                    sclk_partition_starts = "0.0000000000000E+00"
                    sclk_partition_ends = ""

                    if "sclk_partitions" in config['tcp2sclk'][0]:
                        # In case of partition start times defined just generate appropriate strings
                        indent = " " * 36
                        for sclk_part_start in config['tcp2sclk'][0]['sclk_partitions']:
                            sclk_partition_starts += "\n" + indent + sclk_part_start
                            sclk_partition_ends += sclk_part_start + "\n" + indent

                    sclk_partition_ends += source[1]["sclk_partition_end"]

                    kernel.config['sclk_partition_starts'] = sclk_partition_starts
                    kernel.config['sclk_partition_ends'] = sclk_partition_ends

                else:
                    raise Exception('"sclk_partition_end" not specified in Configuration file for : ' + source[0])

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  lsk=lsk, sclk=sclk)

            if source[1]['input_type'] != 'TCP_MEX':

                (sclk_partition, sclk_delimiter) = sclk_reader(kernel)

                #
                # We now check if this coincides with configuration if not the
                # message is shown later to be in the run.
                #
                kernel.config['previous_sclk_separator'] = sclk_delimiter

                if sclk_delimiter == ':':
                    kernel.config['sclk_separator_id'] = 2
                elif sclk_delimiter == '.':
                    kernel.config['sclk_separator_id'] = 1
                else:
                    raise Exception('SCLK separator not specified in Configuration file.')

            kernels.append(kernel)

        #
        # We create the corresponding Kernel object
        #
        if source[1]['input_type'] == 'AEM':

            kernel = Kernel.CK(source[0], directories,
                               Kernel.KernelName(source[1]['kernel_schema']))

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            #
            # We introduce the possibility of having a planning CK
            #
            if source[1]['planning'] == 'True':
                sclk_schema = 'sclk_fict_schema'
                commnt_template = 'aem2ck_plan.commnt'
            else:
                sclk_schema = 'sclk_step_schema'
                commnt_template = 'aem2ck.commnt'

            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                     setup[source[1]['spacecraft']][0][sclk_schema])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid

            kernel.name.set('sclkid', sclkid)
            kernel.set('config', config['aem2ck'][0])

            sclk = os.path.join('sclk', sclk)

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk, sclk=sclk)

            kernels.append(kernel)

        #
        # ExoMars RSP is a special case
        #
        if source[1]['input_type'] == 'OEM':

            kernel = Kernel.SPK(source[0], directories,
                                Kernel.KernelName(source[1]['kernel_schema']))

            kernel.set('config', config['oem2spk'][0])

            commnt_template = 'oem2spk.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            try:
                kernel.config['extended_days'] = setup['rm'][0]['extended_days']
                kernel.config['fake_segments'] = setup['rm'][0]['fake_segments']
            except:
                pass

            kernels.append(kernel)

        if source[1]['input_type'] == 'COG':
            kernel = Kernel.SPK(source[0], directories,
                                Kernel.KernelName(source[1]['kernel_schema']))

            commnt_template = 'mkspk.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            kernel.set('config', config['mkspk'][0])

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernel.set('config', source[1])

            kernels.append(kernel)

        if source[1]['input_type'] == 'SEP':
            kernel = Kernel.CK(source[0], directories,
                               Kernel.KernelName(source[1]['kernel_schema']))

            commnt_template = 'msopck.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            sclk_schema = 'sclk_fict_schema'
            sclk = os.path.join('sclk', get_latest_kernel('sclk', directories.kernels,
                                                          setup[source[1]['spacecraft']][0][sclk_schema]))

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.name.set('sclkid', sclkid)

            kernel.config['sclkid'] = sclkid

            kernel.config['ck_type'] = '2'
            kernel.set('config', config['msopck_sep'][0])

            generic_kernel_config(config, kernel, source,
                                  directories, sclk=sclk, lsk=lsk,
                                  fk=fk, commnt_template=commnt_template)

            kernel.set('config', source[1])

            kernels.append(kernel)

        if source[1]['input_type'] == 'OASW_ORBIT':
            kernel = Kernel.SPK(source[0], directories,
                                Kernel.KernelName(source[1]['kernel_schema']))

            kernel.set('config', config['mex2ker_orbit'][0])

            commnt_template = 'mex2ker_orbit.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernels.append(kernel)

        if source[1]['input_type'] == 'OASW_ATTITUDE':

            kernel = Kernel.CK(source[0], directories,
                               Kernel.KernelName(source[1]['kernel_schema']))

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            #
            # We introduce the possibility of having a planning CK
            #
            if source[1]['planning'] == 'True':
                sclk_schema = 'sclk_fict_schema'
                commnt_template = 'mex2ker_attitude_plan.commnt'
            else:
                sclk_schema = 'sclk_step_schema'
                commnt_template = 'mex2ker_attitude.commnt'

            kernel.set('config', config['mex2ker_attitude'][0])

            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                     setup[source[1]['spacecraft']][0][sclk_schema])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid

            kernel.name.set('sclkid', sclkid)

            sclk = os.path.join('sclk', sclk)

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk, sclk=sclk)

            #
            # do sgs triming
            #
            if 'sgs_trim' in source[1] and source[1]['sgs_trim'] == 'True':
                kernelsgs = copy.deepcopy(kernel)
                kernelsgs.name = str(kernel.name).split('_', 1)[0] + '_SGS_' + str(kernel.name).split('_', 1)[1]
                kernelsgs.source = kernel.source.split('_', 1)[0] + '_SGS_' + kernel.source.split('_', 1)[1]
                kernelsgs.source_config[1]['type6_conversion'] = 'True'
                kernelsgs.source_config[1]['type6_token'] = ''
                kernelsgs.source_config[1]['type5_keep'] = 'False'
                kernels.append(kernelsgs)

            kernel.source_config[1]['sgs_trim'] = 'False'
            kernels.append(kernel)

        if source[1]['input_type'] == 'SPK13':
            kernel = Kernel.SPK(source[0], directories,
                                Kernel.KernelName(source[1]['kernel_schema']))

            kernel.set('config', config['oem2spk'][0])

            commnt_template = 'oem2spk.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernels.append(kernel)

        if source[1]['input_type'] == 'SPK18':
            kernel = Kernel.SPK(source[0], directories,
                                Kernel.KernelName(source[1]['kernel_schema']))

            kernel.set('config', config['mex2ker_orbit'][0])

            commnt_template = 'mex2ker_orbit.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernels.append(kernel)

        if source[1]['input_type'] == 'CK5':

            kernel = Kernel.CK(source[0], directories,
                               Kernel.KernelName(source[1]['kernel_schema']))

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            #
            # We introduce the possibility of having a planning CK
            #
            if source[1]['planning'] == 'True':
                sclk_schema = 'sclk_fict_schema'
                commnt_template = 'mex2ker_attitude_plan.commnt'
            else:
                sclk_schema = 'sclk_step_schema'
                commnt_template = 'mex2ker_attitude.commnt'

            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                     setup[source[1]['spacecraft']][0][sclk_schema])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid

            kernel.name.set('sclkid', sclkid)

            kernel.config['spacecraft'] = source[1]['spacecraft']

            sclk = os.path.join('sclk', sclk)

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk, sclk=sclk)

            kernels.append(kernel)

        if source[1]['input_type'] == 'CK':
            kernel = Kernel.CK(source[0], directories,
                               Kernel.KernelName(source[1]['kernel_schema']))

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1]['spacecraft']][
                                                    0]['fk_schema']))

            sclk = get_from_kernel('sclk', kernel)

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid

            kernel.name.set('sclkid', sclkid)

            kernel.config['spacecraft'] = source[1]['spacecraft']

            sclk = os.path.join('sclk', sclk)

            commnt_template = 'aem2ck.commnt'

            generic_kernel_config(config, kernel, source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk, sclk=sclk)

            kernels.append(kernel)

        if source[1]['input_type'] == 'C-SMITHED':
            sources_csmithed.append(source)

        if source[1]['input_type'] == 'OEM_MERGED':
            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            sources_oem.append(source)

            #
            # We also generate the individual SPK!
            # For it we need to change the source configuration
            #
            for input_type in setup['input_types'][0]:
                if input_type.split('*')[0] in source[0] and \
                        input_type.split('*')[0] != 'emrsp_rm_mes' and \
                        setup['input_types'][0][input_type][0]['kernel_type'] != 'ck':
                    new_source = [source[0], setup['input_types'][0][input_type][0]]

            kernel = Kernel.SPK(new_source[0], directories,
                                Kernel.KernelName(new_source[1]['kernel_schema']))

            kernel.set('config', config['oem2spk'][0])
            kernel.config['append_kernel'] = new_source[1]['append_kernel']
            kernel.config['kernel_name'] = new_source[1]['kernel_name']
            kernel.config['kernel_schema'] = new_source[1]['kernel_schema']

            try:
                kernel.config['extended_days'] = setup['rm'][0]['extended_days']
                kernel.config['fake_segments'] = setup['rm'][0]['fake_segments']
            except: pass

            commnt_template = 'oem2spk.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[new_source[1]['spacecraft']][
                                                    0]['fk_schema']))

            generic_kernel_config(config, kernel, new_source, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernels.append(kernel)

        # When all the source files have been processed, we obtain one kernel
        # object out of the tm. Later on, this can result into
        # multiple kernels.
        if sources_oem and index == input_data_elements:

            sources = []
            for source_oem in sources_oem:
                sources.append(source_oem[0])

            source_cfg = sources_oem[0]

            kernel = Kernel.SPK(sources, directories, Kernel.KernelName(source_cfg[1]['kernel_schema']))

            kernel.set('config', config['oem2spk'][0])
            kernel.config['append_kernel'] = source_cfg[1]['append_kernel']
            kernel.config['kernel_name'] = source_cfg[1]['kernel_name']
            kernel.config['kernel_schema'] = source_cfg[1]['kernel_schema']

            kernel.config['extended_days'] = setup['rm'][0]['extended_days']
            kernel.config['fake_segments'] = setup['rm'][0]['fake_segments']

            commnt_template = 'oem2spk.commnt'

            fk = os.path.join('fk',
                              get_latest_kernel('fk',
                                                directories.kernels,
                                                setup[source[1]['spacecraft']][0]['fk_schema']))

            generic_kernel_config(config, kernel, source_cfg, directories,
                                  commnt_template=commnt_template,
                                  fk=fk, lsk=lsk)

            kernels.append(kernel)

        if source[1]['input_type'] == 'MAPPS_QUATERNIONS':
            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            sources_mapps_quat.append(source)
            source_types_mapps_quat.append(source[1]['kernel_name'])

        if source[1]['input_type'] == 'AEM_QUAT_EXTENDED':

            kernel = Kernel.CK(source[0], directories, Kernel.KernelName(
                source[1]['kernel_schema']))

            fk = os.path.join('fk', get_latest_kernel('fk', directories.kernels,
                                                      setup[source[1][
                                                          'spacecraft']][0][
                                                          'fk_schema']))

            sclk = get_latest_kernel('sclk',
                                     [directories.kernels, directories.output],
                                     setup[source[1]['spacecraft']][0][
                                         'sclk_step_schema'])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid
            if source[1]['ck_type'] == '2':
                kernel.config['angular_rate_present'] = 'YES'
            else:
                kernel.config[
                    'angular_rate_present'] = 'MAKE UP/NO AVERAGING'

            kernel.name.set('sclkid', sclkid)

            kernel.set('config', config['msopck_quat'][0])
            kernel.set('config', source[1])
            kernel.config['extended_days'] = setup['rm'][0]['extended_days']
            kernel.config['fake_segments'] = setup['rm'][0]['fake_segments']

            sclk = os.path.join('sclk', sclk)

            commnt_template = 'msopck_quat.commnt'

            generic_kernel_config(config, kernel, source,
                                  directories,
                                  fk=fk, lsk=lsk, sclk=sclk,
                                  commnt_template=commnt_template)

            kernels.append(kernel)

        if source[1]['input_type'] == 'AEM_QUAT':
            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            sources_aem_quat.append(source)
            source_types_aem_quat.append(source[1]['kernel_name'])

        if source[1]['input_type'] == 'BSUP':
            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            sources_bsup_ang.append(source)
            source_types_bsup_ang.append(source[1]['kernel_name'])

        # When all the source files have been processed, we obtain one kernel
        # object out of the tm_quaternions. Later on, this can result into
        # multiple kernels.
        if sources_aem_quat and index == input_data_elements:

            for source in sources_aem_quat:

                for source_file in source[1]['source_list']:

                    kernel = Kernel.CK(source_file, directories, Kernel.KernelName(
                        source[1]['kernel_schema']))

                    fk = os.path.join('fk', get_latest_kernel('fk', directories.kernels,
                                                              setup[source[1]['spacecraft']][0]['fk_schema']))

                    sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                             setup[source[1]['spacecraft']][0][
                                                 'sclk_step_schema'])

                    sclkid = re.findall('\d+', sclk)[-1]

                    kernel.config['sclkid'] = sclkid
                    if source[1]['ck_type'] == '2':
                        kernel.config['angular_rate_present'] = 'YES'
                    else:
                        kernel.config[
                            'angular_rate_present'] = 'MAKE UP/NO AVERAGING'

                    kernel.name.set('sclkid', sclkid)

                    #
                    # The name needs to be updated.
                    #
                    if len(source[1]['source_list']) > 1:
                        kernel.name.xkma = kernel.source.split('.')[0] + '.bc'

                    kernel.set('config', config['msopck_quat'][0])
                    kernel.set('config', source[1])
                    kernel.config['extended_days'] = setup['rm'][0]['extended_days']
                    kernel.config['fake_segments'] = setup['rm'][0]['fake_segments']

                    sclk = os.path.join('sclk', sclk)

                    commnt_template = 'msopck_quat.commnt'

                    generic_kernel_config(config, kernel, source,
                                          directories,
                                          fk=fk, lsk=lsk, sclk=sclk,
                                          commnt_template=commnt_template)

                    kernels.append(kernel)

                    #
                    # We need to generate the interpolation duplicate
                    #
                    kernel_interp = copy.deepcopy(kernel)
                    kernel_interp.config['input_type'] = 'AEM_QUAT_MERGED'
                    kernel_interp.set('interp_config', setup['input_types'][0]['emrsp_rm_mes_interp*.aem'])
                    generic_kernel_config(config, kernel_interp, source,
                                          directories,
                                          fk=fk, lsk=lsk, sclk=sclk,
                                          commnt_template=commnt_template)
                    kernels.append(kernel_interp)


        if source[1]['input_type'] == 'TM_QUATERNIONS':

            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            source_type = source[1]['kernel_name']
            if source_type not in sources_tm_quat_map:
                sources_tm_quat_map[source_type] = []
            sources_tm_quat_map[source_type].append(source)

        # When all the source files have been processed, we obtain one kernel
        # object out of the boresight angles. Later on, this can result into
        # multiple kernels.
        if sources_bsup_ang and index == input_data_elements:

            for source in sources_bsup_ang:

                for source_file in source[1]['source_list']:

                    kernel = Kernel.CK(source_file, directories, Kernel.KernelName(
                        source[1]['kernel_schema']))

                    fk = os.path.join('fk', get_latest_kernel('fk', directories.kernels,
                                                              setup[source[1]['spacecraft']][0]['fk_schema']))

                    sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                             setup[source[1]['spacecraft']][0][
                                                 'sclk_step_schema'])

                    sclkid = re.findall('\d+', sclk)[-1]

                    kernel.config['sclkid'] = sclkid
                    if source[1]['ck_type'] == '2':
                        kernel.config['angular_rate_present'] = 'YES'
                    else:
                        kernel.config[
                            'angular_rate_present'] = 'MAKE UP/NO AVERAGING'

                    kernel.name.set('sclkid', sclkid)

                    #
                    # The name needs to be updated.
                    #
                    if len(source[1]['source_list']) > 1:
                        kernel.name.xkma = kernel.source.split('.')[0] + '.bc'

                    kernel.set('config', source[1])
                    kernel.name.tokens['detector_name'] = source[0].split('_')[1].lower()
                    kernel.config['detector_name'] = source[0].split('_')[1].lower()
                    kernel.config['extended_days'] = setup['solo'][0]['extended_days']
                    kernel.config['fake_segments'] = setup['solo'][0]['fake_segments']
                    kernel.config['target_frame'] = kernel.config[kernel.source]['target_frame']
                    kernel.config['object_frame'] = kernel.config[kernel.source]['object_frame']
                    kernel.config['object_id'] = kernel.config[kernel.source]['object_id']
                    kernel.config['object_name'] = kernel.config['detector_name'].upper()

                    sclk = os.path.join('sclk', sclk)

                    commnt_template = 'msopck_bsup.commnt'

                    generic_kernel_config(config, kernel, source,
                                          directories,
                                          fk=fk, lsk=lsk, sclk=sclk,
                                          commnt_template=commnt_template)

                    kernels.append(kernel)


        # When all the source files have been processed, we obtain one kernel
        # object out of the tm_quaternions. Later on, this can result into
        # multiple kernels.
        if len(sources_tm_quat_map) and index == input_data_elements:

            #
            # We need to distinguish in between source types, we now that for any added source_type there is
            # at least one source.
            #
            for source_type in sources_tm_quat_map:

                merge_flag = False
                sources_tm_quat = sorted(sources_tm_quat_map[source_type])
                merged_sources = []

                # Get start date from latest kernel
                kernel_type = sources_tm_quat[0][1]["kernel_type"]
                sc_setup = setup[sources_tm_quat[0][1]['spacecraft']][0]
                last_rot_date = get_latest_kernel_startdate(kernel_type, source_type, directories, setup, sc_setup)

                for source_tm_quat in sources_tm_quat:
                    if not source_tm_quat[0] in merged_sources and not merge_flag:

                        sources = []

                        # We determine if several sources need to be merged
                        if source_tm_quat[1]['merge_sources'] == 'True':

                            # Sources to merge are all the sources not already merged
                            sources_to_merge = [source for source in sources_tm_quat if source[0] not in merged_sources]

                            # Get sources to merge, the source configuration, date of latest rotation
                            # if they exist, the flag indicating if sources has been merged, and the flag indicating if
                            # any rotation has been done in this execution for this source type
                            sources, source_cfg, last_rot_date, merge_flag = get_sources_to_merge(sources_to_merge,
                                                                                                  last_rot_date)

                            # Add sources to merge to the list of merged sources
                            merged_sources.extend(sources)

                            if len(sources) == 0:
                                # In case of the first file matches a kernel rotation, avoid kernel creations,
                                # and process next source.
                                continue

                        #
                        # We determine if we have to concatenate kernels
                        #
                        generate_single_kernel = False
                        if 'merge_kernel' in source_tm_quat[1].keys():
                            if source_tm_quat[1]['merge_kernel'] == 'True':

                                #
                                # We obtain the sources
                                #
                                sources = source_tm_quat[1]['source_list']

                                index = 0
                                for source_tm_quat_daf in sources:

                                    source_cfg = source_tm_quat

                                    #
                                    # Get FK, SCLK, SCLK Id and kernel schema
                                    #
                                    fk, sclkid, sclk = get_fk_and_sclk(sc_setup, directories)
                                    kernel_schema = source_tm_quat[1]['kernel_schema'].split('.')[
                                                        0] + '_{0:03}.bc'.format(index)

                                    #
                                    # We generate concatenated kernel i
                                    #
                                    kernel = Kernel.CK(source_tm_quat_daf, directories,
                                                       Kernel.KernelName(kernel_schema))

                                    kernel.config['sclkid'] = sclkid
                                    if source[1]['ck_type'] == '2':
                                        kernel.config['angular_rate_present'] = 'YES'
                                    else:
                                        kernel.config['angular_rate_present'] = 'MAKE UP/NO AVERAGING'

                                    kernel.name.set('sclkid', sclkid)
                                    kernel.set('config', config['msopck_quat'][0])
                                    kernel.set('config', source_tm_quat[1])

                                    commnt_template = 'msopck_quat.commnt'

                                    generic_kernel_config(config, kernel, source_cfg, directories,
                                                          fk=fk, lsk=lsk, sclk=sclk, commnt_template=commnt_template)

                                    kernels.append(kernel)
                                    index += 1
                            #
                            # We need to distinguish the special case of Solar Orbiter
                            # were we need to split the generation for the interpolation
                            # intervals
                            #
                            elif 'source_list' in source_tm_quat[1]:
                                if len(source_tm_quat[1]['source_list']) >= 1:  # or "merge_intervals":"True"
                                    #
                                    # We obtain the sources
                                    #
                                    sources = source_tm_quat[1]['source_list']

                                    index = 0
                                    for source_tm_quat_daf in sources:

                                        source_cfg = source_tm_quat

                                        #
                                        # Get FK, SCLK, SCLK Id and kernel schema
                                        #
                                        fk, sclkid, sclk = get_fk_and_sclk(sc_setup, directories)
                                        kernel_schema = source_tm_quat[1]['kernel_schema'].split('.')[
                                                            0] + '_{0:03}.bc'.format(index)

                                        #
                                        # We generate concatenated kernel i
                                        #
                                        kernel = Kernel.CK(source_tm_quat_daf, directories,
                                                           Kernel.KernelName(kernel_schema))

                                        kernel.config['sclkid'] = sclkid
                                        if source[1]['ck_type'] == '2':
                                            kernel.config['angular_rate_present'] = 'YES'
                                        else:
                                            kernel.config['angular_rate_present'] = 'MAKE UP/NO AVERAGING'

                                        kernel.name.set('sclkid', sclkid)
                                        kernel.set('config', config['msopck_quat'][0])
                                        kernel.set('config', source_tm_quat[1])

                                        commnt_template = 'msopck_quat.commnt'

                                        generic_kernel_config(config, kernel, source_cfg, directories,
                                                              fk=fk, lsk=lsk, sclk=sclk,
                                                              commnt_template=commnt_template)
                                        kernels.append(kernel)
                                        index += 1
                                else:
                                    source_cfg = source_tm_quat
                                    generate_single_kernel = True
                            else:
                                source_cfg = source_tm_quat
                                generate_single_kernel = True

                        else:
                            source_cfg = source_tm_quat
                            generate_single_kernel = True

                        if generate_single_kernel:
                            #
                            # Get FK, SCLK, SCLK Id and kernel schema
                            #
                            fk, sclkid, sclk = get_fk_and_sclk(sc_setup, directories)
                            kernel_schema = source_tm_quat[1]['kernel_schema']

                            #
                            # We generate kernel
                            #
                            kernel = create_ck_kernel_with_tm(sources, directories, kernel_schema, sclkid,
                                                              source_cfg, config['msopck_quat'][0])

                            # IMPORTANT NOTE!!! In this case the order that we follow to set the config
                            # is relevant for a proper execution.
                            # TODO: Investigate why this config order is soo crucial
                            kernel.set('config', config['msopck_quat'][0])
                            kernel.set('config', source_cfg[1])

                            commnt_template = 'msopck_quat.commnt'

                            generic_kernel_config(config, kernel, source_cfg, directories,
                                                  fk=fk, lsk=lsk, sclk=sclk,
                                                  commnt_template=commnt_template)

                            kernels.append(kernel)

        if sources_csmithed and index == input_data_elements:

            csmithed = []
            for source_csmithed in sources_csmithed:
                csmithed.append(source_csmithed[0])

            csmithed.sort()

            source_cfg = sources_csmithed[0]
            kernel = Kernel.CK(csmithed, directories, Kernel.KernelName(source_cfg[1]['kernel_schema']))

            fk = os.path.join('fk', get_latest_kernel('fk', directories.kernels,
                                                      setup[source_cfg[1]['spacecraft']][0]['fk_schema']))

            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                     setup[source_cfg[1]['spacecraft']][0]['sclk_step_schema'])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.config['sclkid'] = sclkid
            kernel.config['ck_type'] = '3'

            kernel.name.set('sclkid', sclkid)
            kernel.set('config', config['msopck_quat'][0])
            kernel.set('config', source_cfg[1])

            sclk = os.path.join('sclk', sclk)

            commnt_template = 'msopck_quat.commnt'

            generic_kernel_config(config, kernel, source_cfg,
                                  directories,
                                  fk=fk, lsk=lsk, sclk=sclk,
                                  commnt_template=commnt_template)

            kernels.append(kernel)

        if sources_mapps_quat and index == input_data_elements:

            #
            # We need to distinguish in between source types.
            #
            source_types_mapps_quat = list(set(source_types_mapps_quat))

            for source_type in source_types_mapps_quat:

                merge_flag = False

                for source_mapps_quat in sources_mapps_quat:
                    if source_type == source_mapps_quat[1]['kernel_name'] and not merge_flag:

                        sources = []

                        #
                        # We determine if several sources need to be merged
                        #
                        if source_mapps_quat[1]['merge_sources'] == 'True':
                            merged_sources = sources_mapps_quat
                            for source_mapps_quat in merged_sources:
                                if source_type == source_mapps_quat[1]['kernel_name']:
                                    sources.append(source_mapps_quat[0])
                                    merge_flag = True
                                    source_cfg = source_mapps_quat
                        else:
                            sources = source_mapps_quat[0]
                            source_cfg = source_mapps_quat

                        kernel = Kernel.CK(sources, directories,
                                           Kernel.KernelName(source_mapps_quat[1]['kernel_schema']))

                        fk = os.path.join('fk',
                                          get_latest_kernel('fk',
                                                            directories.kernels,
                                                            setup[source[1][
                                                                'spacecraft']][
                                                                0][
                                                                'fk_schema']))

                        sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                                 setup[source[1]['spacecraft']][0]['sclk_step_schema'])

                        sclkid = re.findall('\d+', sclk)[-1]

                        kernel.config['sclkid'] = sclkid
                        kernel.config['ck_type'] = '3'

                        kernel.name.set('sclkid', sclkid)
                        kernel.set('config', config['msopck_quat'][0])
                        kernel.set('config', source_cfg[1])

                        sclk = os.path.join('sclk', sclk)

                        commnt_template = 'msopck_quat.commnt'

                        generic_kernel_config(config, kernel, source_cfg, directories,
                                              fk=fk, lsk=lsk, sclk=sclk,
                                              commnt_template=commnt_template)

                        kernels.append(kernel)

        if source[1]['input_type'] == 'TM':

            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            source_type = source[1]['kernel_name']
            if source_type not in sources_tm_map:
                sources_tm_map[source_type] = []
            sources_tm_map[source_type].append(source)

        # When all the source files have been processed, we obtain one kernel
        # object out of the tm. Later on, this can result into multiple kernels.
        if len(sources_tm_map) and index == input_data_elements:

            #
            # We need to distinguish in between source types, we now that for any added source_type there is
            # at least one source.
            #
            for source_type in sources_tm_map:

                merge_flag = False
                sources_tm = sorted(sources_tm_map[source_type])
                merged_sources = []

                # Get start date from latest kernel
                kernel_type = sources_tm[0][1]["kernel_type"]
                sc_setup = setup[sources_tm[0][1]['spacecraft']][0]
                last_rot_date = get_latest_kernel_startdate(kernel_type, source_type, directories, setup, sc_setup)

                for source_tm in sources_tm:
                    if not source_tm[0] in merged_sources and not merge_flag:

                        sources = []

                        # We determine if several sources need to be merged
                        if source_tm[1]['merge_sources'] == 'True':

                            # Sources to merge are all the sources not already merged
                            sources_to_merge = [source for source in sources_tm if source[0] not in merged_sources]

                            # Get sources to merge, the source configuration, date of latest rotation
                            # if they exist, the flag indicating if sources has been merged, and the flag indicating if
                            # any rotation has been done in this execution for this source type
                            sources, source_cfg, last_rot_date, merge_flag = get_sources_to_merge(sources_to_merge,
                                                                                                  last_rot_date)
                            # Add sources to merge to the list of merged sources
                            merged_sources.extend(sources)

                            if len(sources) == 0:
                                # In case of the first file matches a kernel rotation, avoid kernel creations,
                                # and process next source.
                                continue

                        else:
                            sources = source_tm[0]
                            source_cfg = source_tm

                        #
                        # Get FK, SCLK, SCLK Id and kernel schema
                        #
                        fk, sclkid, sclk = get_fk_and_sclk(sc_setup, directories)
                        kernel_schema = source_tm[1]['kernel_schema']

                        kernel = create_ck_kernel_with_tm(sources, directories, kernel_schema,
                                                          sclkid, source_cfg, config['msopck'][0])

                        commnt_template = 'msopck.commnt'

                        generic_kernel_config(config, kernel, source_cfg, directories,
                                              fk=fk, lsk=lsk, sclk=sclk, commnt_template=commnt_template)
                        kernels.append(kernel)

        if source[1]['input_type'] == 'TM_DAFCAT':

            # We keep adding sources to the source list until we have reviewed
            # all the inputs (when the index is equal to the number of kernels
            # minus one)
            #
            source_type = source[1]['kernel_name']
            if source_type not in sources_tm_daf_map:
                sources_tm_daf_map[source_type] = []
            sources_tm_daf_map[source_type].append(source)

        # When all the source files have been processed, we obtain one kernel
        # object out of the tm. Later on, this can result into multiple kernels.
        if len(sources_tm_daf_map) and index == input_data_elements:

            #
            # We need to distinguish in between source types, we now that for any added source_type there is
            # at least one source.
            #
            for source_type in sources_tm_daf_map:

                merge_flag = False
                sources_tm_daf = sorted(sources_tm_daf_map[source_type])
                merged_sources = []

                # Get start date from latest kernel
                kernel_type = sources_tm_daf[0][1]["kernel_type"]
                sc_setup = setup[sources_tm_daf[0][1]['spacecraft']][0]
                last_rot_date = get_latest_kernel_startdate(kernel_type, source_type, directories, setup, sc_setup)

                for source_tm_daf in sources_tm_daf:
                    if not source_tm_daf[0] in merged_sources and not merge_flag:

                        sources = []

                        # We determine if several sources need to be merged
                        if source_tm_daf[1]['merge_sources'] == 'True':

                            # Sources to merge are all the sources not already merged
                            sources_to_merge = [source for source in sources_tm_daf if source[0] not in merged_sources]

                            # Get sources to merge, the source configuration, date of latest rotation
                            # if they exist, the flag indicating if sources has been merged, and the flag indicating if
                            # any rotation has been done in this execution for this source type
                            sources, source_cfg, last_rot_date, merge_flag = get_sources_to_merge(sources_to_merge,
                                                                                                  last_rot_date)
                            # Add sources to merge to the list of merged sources
                            merged_sources.extend(sources)

                            if len(sources) == 0:
                                # In case of the first file matches a kernel rotation, avoid kernel creations,
                                # and process next source.
                                continue

                        else:
                            sources = source_tm_daf[0]
                            source_cfg = source_tm_daf

                        #
                        # Get FK, SCLK, SCLK Id and kernel schema
                        #
                        fk, sclkid, sclk = get_fk_and_sclk(sc_setup, directories)
                        kernel_schema = source_tm_daf[1]['kernel_schema']

                        #
                        # We generate DAFCAT kernel 1
                        #
                        kernel = create_ck_kernel_with_tm(sources, directories, kernel_schema,
                                                          sclkid, source_cfg, config['msopck'][0])

                        kernel.set('dafcat', 1)
                        kernel.name.set('dafcat', "1")
                        commnt_template = 'msopck_dafcat_1.commnt'

                        generic_kernel_config(config, kernel, source_cfg, directories,
                                              fk=fk, lsk=lsk, sclk=sclk, commnt_template=commnt_template)
                        kernels.append(kernel)

                        #
                        # We generate DAFCAT kernel 2
                        #
                        kernel = create_ck_kernel_with_tm(sources, directories, kernel_schema,
                                                          sclkid, source_cfg, config['msopck'][0])

                        kernel.set('dafcat', 2)
                        kernel.name.set('dafcat', "2")
                        commnt_template = 'msopck_dafcat_2.commnt'

                        generic_kernel_config(config, kernel, source_cfg, directories,
                                              fk=fk, lsk=lsk, sclk=sclk, commnt_template=commnt_template)
                        kernels.append(kernel)

        index += 1

        if source[1]['input_type'] == 'EVENTS':

            specs2ck_config = config['specs2ck'][0]['attitude_types']
            attitude_type = source[1]['default_attitude']

            kernel = Kernel.CK(source[0], directories,
                               Kernel.KernelName(
                                   source[1]['kernel_schema']))

            #
            # We introduce the possibility of having a planning CK
            #
            if source[1]['planning'] == 'False':
                sclk_schema = 'sclk_step_schema'
            else:
                sclk_schema = 'sclk_fict_schema'

            fk = os.path.join('fk',
                              get_latest_kernel('fk', directories.kernels,
                                                setup[source[1][
                                                    'spacecraft']][0][
                                                    'fk_schema']))

            sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                     setup[source[1]['spacecraft']][0][
                                         sclk_schema])

            sclkid = re.findall('\d+', sclk)[-1]

            kernel.name.set('sclkid', sclkid)

            kernel.set('config', specs2ck_config[0][attitude_type][0])

            kernel.config['sclkid'] = sclkid

            sclk = os.path.join('sclk', sclk)

            mk = get_latest_kernel('mk', directories.kernels,
                                   setup['mk_schema'])

            generic_kernel_config(config, kernel, source,
                                  directories, mk=mk, sclk=sclk, lsk=lsk,
                                  fk=fk)

            #
            # We need to load the mk and the future SPK kernel
            # that the CK will be generated from
            #
            kernel.mk = mk

            kernels.append(kernel)

        #
        # Configuration for the default attitude (SPECS)
        #
        if source[1]['kernel_type'] == "spk":
            if source[1]['default_attitude'] != '':

                specs2ck_config = config['specs2ck'][0]
                attitude_types = specs2ck_config['attitude_types'][0]
                default_attitude = source[1]['default_attitude']
                attitude_type = attitude_types[default_attitude]

                source = [source[0], attitude_type[0]]

                kernel = Kernel.CK(source[0], directories,
                                   Kernel.KernelName(
                                       attitude_type[0]['kernel_schema']))

                #
                # We introduce the possibility of having a planning CK
                #
                if source[1]['planning'] == 'False':
                    sclk_schema = 'sclk_step_schema'
                else:
                    sclk_schema = 'sclk_fict_schema'

                fk = os.path.join('fk',
                                  get_latest_kernel('fk', directories.kernels,
                                                    setup[source[1]['spacecraft']][0]['fk_schema']))

                sclk = get_latest_kernel('sclk', [directories.kernels, directories.output],
                                         setup[source[1]['spacecraft']][0][sclk_schema])

                sclkid = re.findall('\d+', sclk)[-1]

                kernel.name.set('sclkid', sclkid)

                kernel.config['sclkid'] = sclkid

                sclk = os.path.join('sclk', sclk)

                kernel.set('config', attitude_types[default_attitude][0])

                mk = get_latest_kernel('mk', directories.kernels, setup['mk_schema'])

                generic_kernel_config(config, kernel, source,
                                      directories, mk=mk, sclk=sclk, lsk=lsk, fk=fk)

                #
                # We need to load the mk and the future SPK kernel
                # that the CK will be generated from
                #
                kernel.mk = mk

                for k in kernels:
                    if k.source == source[0]:
                        kernel.spk = k

                kernels.append(kernel)

    #
    # Look for extra sources to be moved to temp and then to processed folders
    #
    for kernel in kernels:
        if "extra_sources" in kernel.config:
            for extra_source in kernel.config["extra_sources"]:

                sources = kernel.source if isinstance(kernel.source, list) else [kernel.source]

                for source in sources:
                    source_name, source_extension = os.path.splitext(source)
                    extra_source_file = extra_source.replace("{filename}", source_name)
                    extra_source_path = os.path.join(directories.source, extra_source_file)
                    if os.path.exists(extra_source_path):
                        kernel.extra_sources.append(extra_source_file)
                    else:
                        logging.warning("Extra source not found: " + extra_source_file)

    #
    # We also need to return the list of original sources in order to
    # clean-up the source directory if there is any error or if we are
    # testing ADCSng
    #
    return kernels, original_sources


def get_sources_to_merge(sources_to_merge, last_rot_date):
    sources = []
    source_cfg = None
    merge_flag = False

    rotation_period = ""
    source_date_format = ""
    if "kernel_rotation" in sources_to_merge[0][1]:
        # If kernel_rotation is specified (Year, Monthly, Weekly, Daily) then we
        # need to check that "source_date_format" config key has been set.

        rotation_period = sources_to_merge[0][1]["kernel_rotation"].lower()
        if rotation_period not in ["yearly", "monthly", "weekly", "daily"]:
            raise Exception("Invalid value '" + rotation_period + "' for key 'kernel_rotation', supported values "
                                                                  "(Yearly, Monthly, Weekly, Daily) for file: " +
                            sources_to_merge[0][0])

        if "source_date_format" in sources_to_merge[0][1]:
            source_date_format = sources_to_merge[0][1]["source_date_format"]
        else:
            raise Exception("Configuration key 'source_date_format': 'start_index|date_length|strptime_format' "
                            + " is required when using 'kernel_rotation', missing for file: " + sources_to_merge[0][0])

        if not last_rot_date:
            logging.warning(f'KERNEL ROTATION: Cannot obtain previous kernel time coverage for: '
                            + sources_to_merge[0][0] + ', try run with log=debug option for details.')

    for source_to_merge in sources_to_merge:

        # Check if we need to rotate with the given source file name, if need rotate
        # stop here and generate kernel with appended sources,
        # else we will continue appending sources to the sources list.
        must_rotate = False
        source_date = None
        if rotation_period and source_date_format:
            source_date = get_date_from_filename(source_to_merge[0], source_date_format)

            if source_date and not last_rot_date:
                last_rot_date = source_date
            else:
                must_rotate = is_different_rotation_period(rotation_period, source_date, last_rot_date)

        merge_flag = not must_rotate
        if must_rotate:
            last_rot_date = source_date

            # Only break if there are more sources to process
            if len(sources_to_merge) > 1:
                break

        sources.append(source_to_merge[0])
        source_cfg = source_to_merge

    return sources, source_cfg, last_rot_date, merge_flag


def is_different_rotation_period(rotation_period, source_date, last_date):
    if isinstance(source_date, datetime) and isinstance(last_date, datetime):

        if rotation_period == "yearly":
            return source_date.year != last_date.year

        if rotation_period == "monthly":
            return source_date.year != last_date.year \
                   or source_date.month != last_date.month

        if rotation_period == "weekly":
            return source_date.year != last_date.year \
                   or source_date.strftime("%W") != last_date.strftime("%W")

        if rotation_period == "daily":
            return source_date.year != last_date.year \
                   or source_date.strftime("%j") != last_date.strftime("%j")

    return False


def cks_are_in_different_rotation_period(ck1, ck2, rotation_period, directories):

    kernel_cov1 = get_ck_kernel_coverage_from_path(ck1, directories)
    kernel_cov2 = get_ck_kernel_coverage_from_path(ck2, directories)

    if len(kernel_cov1) and len(kernel_cov2):
        avg_date1 = kernel_cov1[0][0] + (kernel_cov1[-1][1] - kernel_cov1[0][0]) / 2
        avg_date2 = kernel_cov2[0][0] + (kernel_cov2[-1][1] - kernel_cov2[0][0]) / 2

        return is_different_rotation_period(rotation_period, avg_date1, avg_date2)

    return True


def get_fk_and_sclk(sc_setup, directories):
    # Get FK, SCLK, SCLK Id
    fk = os.path.join('fk', get_latest_kernel('fk', directories.kernels, sc_setup['fk_schema']))

    sclk = get_latest_kernel('sclk', [directories.kernels, directories.output], sc_setup['sclk_step_schema'])
    sclkid = re.findall('\d+', sclk)[-1]
    sclk = os.path.join('sclk', sclk)

    return fk, sclkid, sclk


def create_ck_kernel_with_tm(sources, directories, kernel_schema, sclkid, source_cfg, msopck_config):
    kernel = Kernel.CK(sources, directories, Kernel.KernelName(kernel_schema))

    kernel.name.set('sclkid', sclkid)
    kernel.config['sclkid'] = sclkid
    kernel.set('config', source_cfg[1])
    kernel.set('config', msopck_config)
    kernel.config['ck_type'] = '3'
    kernel.config['angular_rate_present'] = 'MAKE UP/NO AVERAGING'

    return kernel


def get_config(config):
    """
    Return a dictionary containing the configuration provided for the mission
    in the mission specific *.jSOn configuration file.

    :param mission: Mission name
    :type mission: str
    :return: Mission specific adcsng configuration from json file.
    :rtype: dict
    """
    configuration = {}
    mission = config['mission']

    filename = open(config['root_dir'] + '/config/' + mission.lower() + '.json', 'r')

    configuration = json.load(filename)
    filename.close()

    #
    # We add the ADCSng version
    #
    with open(config['root_dir'] + '/config/version', 'r') as f:
        for line in f:
            version = line
        configuration['generic'][0]['adcsng_version'] = version

    return configuration


def get_description(filename, index):
    """
    This function gets the description value to populate the kernel in creation
    name from the input file. If no index is provided the function returns
    no description.

    :param filename: Source file name
    :type filename: str
    :param index: Position of the description in the source file.
       Format is "x-y" where x is the location of the first letter in the
       string and y the last one
    :type index:
    :return: Description value for the Kernel filename
    :rtype: str
    :raises
    """
    if index == "":
        return ""

    idx = []
    idx.append(int(index.split('-')[0]))
    idx.append(int(index.split('-')[1]))
    if len(idx) != 2 or idx[0] > idx[1]:
        raise ValueError("invalid indexing sequence: {}".format(idx))

    description = filename

    #
    # Get the description field from the input filename.
    #
    if len(description) > idx[1] + 1:
        description = description[idx[0]: idx[1] + 1]
    elif idx[1] + 1 > len(description):
        description = description[idx[0]: len(description)]

    #
    # Remove leading and training underscores.
    #
    description = description.strip('_')

    return description


def get_latest_kernel(kernel_type, paths, pattern, dates=False,
                      include_latest=0,
                      excluded_kernels=False, mkgen=False, rotation_period='', directories=None):
    """
    Returns the name of the latest MK, LSK, FK or SCLK present in the path

    :param kernel_type: Kernel type (lsk, sclk, fk) which also defines the subdirectory name.
    :type kernel_type: str
    :param path: Path to the root of the SPICE directory where the kernels are store in a directory named ``type``.
    :type path: str
    :param patterns: Patterns to search for that defines the kernel ``type`` file naming scheme.
    :type patterns: list
    :return: Name of the latest kernel of ``type`` that matches the naming scheme defined in ``token`` present in the ``path`` directory.
    :rtype: strÐ
    :raises:
       KernelNotFound if no kernel of ``type`` matching the naming scheme
       defined in ``token is present in the ``path`` directory
    """
    kernels = []
    if not isinstance(paths, list):
        paths = [paths]

    paths = list(dict.fromkeys(paths))

    #
    # Get the kernels of type ``type`` from the ``path``/``type`` directory.
    #
    kernels_with_path = []
    kernels_paths = {}
    kernel_path = ""
    for path in paths:
        kernel_path = os.path.join(path, kernel_type)
        kernels_with_path += glob.glob(kernel_path + '/' + pattern)

        #
        # Include kernels in former_versions if the directory exists except for
        # meta-kernel generation
        #
        if os.path.isdir(kernel_path + '/former_versions') and not mkgen:
            kernels_with_path += glob.glob(kernel_path + '/former_versions/' + pattern)

        for kernel in kernels_with_path:
            kernel_name = kernel.split('/')[-1]
            kernels.append(kernel_name)
            kernels_paths[kernel_name] = kernel

    if not kernels:
        raise KernelNotFound(kernel_type, kernel_path)

    #
    # We remove possible duplicates
    #
    kernels = list(dict.fromkeys(kernels))

    #
    # Put the kernels in order
    #
    kernels.sort()

    #
    # We remove the kernel if it is included in the excluded kernels list
    #
    if excluded_kernels:
        for kernel in excluded_kernels:
            if kernel in kernels:
                kernels.remove(kernel)

    if not dates:
        #
        # Return the latest kernel
        #
        if include_latest > 0:
            return kernels[-include_latest:]
        if include_latest > len(kernels):
            return kernels
        try:
            latest_kernel = kernels.pop()
        except:
            latest_kernel = []

        return latest_kernel
    else:
        #
        # Return all the kernels with a given date
        #
        previous_kernel = ''
        kernels_date = []
        for kernel in kernels:

            # Remove previous version of the kernel if found a newest one
            if previous_kernel:
                if re.split('_V\d\d', previous_kernel.upper())[0] == re.split('_V\d\d', kernel.upper())[0] \
                        or re.split('_V\d\d\d', previous_kernel.upper())[0] == re.split('_V\d\d\d', kernel.upper())[0]:
                    kernels_date.remove(previous_kernel)

                # Remove previous CK kernel if found the same CK but newest of same rotation period
                elif kernel_type == "ck":
                    if len(rotation_period):

                        if not cks_are_in_different_rotation_period(kernels_paths[previous_kernel],
                                                                    kernels_paths[kernel],
                                                                    rotation_period, directories):
                            # Remove previous CK kernel if both have same rotation period
                            kernels_date.remove(previous_kernel)

            previous_kernel = kernel
            kernels_date.append(kernel)

        return kernels_date


def get_from_kernel(kernel_type, kernel):
    """
    Returns the name of the latest LSK, FK or SCLK present in the kernel (wtih which the kernel has been generated)
    this has

    """
    return get_from_kernel_path(kernel_type,
                                kernel.directories.source + '/' + kernel.source,
                                kernel.directories)[0]


def get_from_kernel_path(kernel_type, kernel_path, directories):
    """
    Returns the name of the latest LSK, FK or SCLK present in the kernel (wtih which the kernel has been generated)
    this has

    """

    process_output = commnt_read(kernel_path, directories)
    commnt = process_output.split('\n')

    kernel_type_kernel = ""
    kernel_path = ""

    kernel_type_keyword = kernel_type.upper() + '_FILE_NAME'
    if kernel_type.upper() == "FK":
        kernel_type_keyword = 'FRAMES_FILE_NAME'

    for line in commnt:
        if kernel_type_keyword in line:
            kernel_path = line.split("'")[1]

            # Remove folders before the kernel type folder
            if len(kernel_path.split(os.sep)) > 2:
                kernel_path = os.path.join(kernel_path.split(os.sep)[-2], kernel_path.split(os.sep)[-1])

            # Get kernel filename
            kernel_type_kernel = kernel_path.split("\\")[-1]
            if '.' + kernel_type2extension(kernel_type, case='lower')[0] not in kernel_type_kernel:
                kernel_type_kernel = kernel_path.split(os.sep)[-1]

            break

    # Check if kernel_path doesn't exists try to find it in former_versions or other directories
    if len(kernel_path):
        rel_kernel_path = kernel_path

        # Looks if in default kernel type folder
        kernel_path = os.path.join(directories.kernels, kernel_path)
        if not os.path.exists(kernel_path):
            # Looks if in former_versions
            kernel_path = kernel_path.replace(kernel_type_kernel, "former_versions" + os.sep + kernel_type_kernel)
            if not os.path.exists(kernel_path):
                # Looks if is a just created kernel
                kernel_path = os.path.join(directories.output, rel_kernel_path)
                if not os.path.exists(kernel_path):
                    kernel_path = ""

    return kernel_type_kernel, kernel_path


def get_version_number(k_type, path, filename):
    #
    # So far ExoMarsRSP is the only mission with 3 zeros in the version
    # number, that is why it is not generalised
    #
    if 'emrsp' in filename:
        version_format = '{:03d}'
    else:
        version_format = '{:02d}'

    if k_type == 'mk':
        version_format = '{:03d}'

    #
    # Create the pattern to look for and find if there's a kernel in the
    # given path that matches the criteria.
    #
    pattern = filename.replace('{version}', '*')

    #
    # DAFCAT CK kernels need to be treated separately otherwise we do not
    # get the appropiate filename
    #
    if '_1.' in pattern:
        pattern = pattern.split('_1.')[0]
    elif '_2.' in pattern:
        pattern = pattern.split('_2.')[0]

    try:
        kernel = get_latest_kernel(k_type, path, pattern)
        version = version_format.format(int(kernel.split('.')[0][-2:]) + 1)
    except:
        version = version_format.format(1)

    return version


def get_latest_kernel_startdate(kernel_type, kernel_name, directories, setup, sc_setup):
    try:
        # Get latest CK kernel, mkgen=True to avoid search in former_versions
        latest_kernel_path = os.path.join(directories.kernels, os.path.join(kernel_type,
                                                                            get_latest_kernel(kernel_type,
                                                                                              directories.kernels,
                                                                                              kernel_name + "*",
                                                                                              mkgen=True)))
    except KernelNotFound:
        logging.warning(f'Cannot found the latest kernel for: ' + kernel_name + ", kernel type: " + kernel_type)
        return None

    coverage = None
    if kernel_type == "ck":
        coverage = get_ck_kernel_coverage_with_latest(latest_kernel_path, directories, setup, sc_setup)

    if len(coverage):
        # Return the start date of the first segment
        return coverage[0][0]

    return None


def get_ck_kernel_coverage_with_latest(kernel_path, directories, setup, sc_setup):
    # Get SCLK used for creating the CK kernel, mkgen=True to avoid search in former_versions
    try:
        sclk_path = os.path.join(directories.kernels, os.path.join("sclk",
                                                                   get_latest_kernel("sclk", directories.kernels,
                                                                                     sc_setup['sclk_step_schema'],
                                                                                     mkgen=True)))
    except KernelNotFound:
        logging.warning(f'Cannot find the SCLK kernel for kernel_path: ' + kernel_path
                        + ', sclk_step_schema: ' + sc_setup['sclk_step_schema'])
        return []

    # Get LSK used for creating the CK kernel, mkgen=True to avoid search in former_versions
    try:
        lsk_path = os.path.join(directories.kernels, os.path.join("lsk",
                                                                  get_latest_kernel("lsk", directories.kernels,
                                                                                    setup['lsk_schema'],
                                                                                    mkgen=True)))
    except KernelNotFound:
        logging.warning(f'Cannot find the LSK kernel for kernel_path: ' + kernel_path
                        + ', lsk_schema: ' + setup['lsk_schema'])
        return []

    # Get FK used for creating the CK kernel, mkgen=True to avoid search in former_versions
    try:
        fk_path = os.path.join(directories.kernels, os.path.join("fk",
                                                                 get_latest_kernel("fk", directories.kernels,
                                                                                   sc_setup['fk_schema'],
                                                                                   mkgen=True)))
    except KernelNotFound:
        logging.warning(f'Cannot find the FK kernel for kernel_path: ' + kernel_path
                        + ', fk_schema: ' + setup['fk_schema'])
        return []

    return get_ck_kernel_coverage_with_paths(kernel_path, sclk_path, lsk_path, fk_path, directories)


def get_ck_kernel_coverage_from_path(kernel_path, directories):
    sclk_path = get_from_kernel_path("sclk", kernel_path, directories)[1]
    if not len(sclk_path):
        logging.warning(f'Cannot find the SCLK kernel for kernel_path: ' + kernel_path)

    lsk_path = get_from_kernel_path("lsk", kernel_path, directories)[1]
    if not len(lsk_path):
        logging.warning(f'Cannot find the LSK kernel for kernel_path: ' + kernel_path)

    fk_path = ""
    try:
        fk_file = get_from_kernel_path("fk", kernel_path, directories)[1]
        if len(fk_file):
            fk_path = os.path.join(directories.kernels, fk_file)
    except:
        fk_path = ""

    return get_ck_kernel_coverage_with_paths(kernel_path, sclk_path, lsk_path, fk_path, directories)


def get_ck_kernel_coverage_with_paths(kernel_path, sclk_path, lsk_path, fk_path, directories):
    # Run ckbrief with given ck, lsk, sclk kernels
    support_kernels = sclk_path + ' ' + lsk_path + ' ' + fk_path
    coverage_text = run_ckbrief(kernel_path, support_kernels, directories)

    cov_list = coverage_text.split('\n')
    coverage = []

    try:
        for line in cov_list:

            # Search for:
            # 'Begin UTC: 2018-APR-14 00:03:19.636  End UTC: 2018-APR-14 23:59:59.642'
            if 'Begin UTC:' in line:
                line_parts = line.replace("Begin UTC:", "").split("End UTC:")
                begin_date = datetime.strptime(line_parts[0].strip(), '%Y-%b-%d %H:%M:%S.%f')
                end_date = datetime.strptime(line_parts[1].strip(), '%Y-%b-%d %H:%M:%S.%f')
                coverage.append([begin_date, end_date])

    except:
        logging.debug(f'Cannot obtain the time CK coverage from: ' + kernel_path +
                      '\n, sclk_path: ' + sclk_path + ',\n lsk_path: ' + lsk_path + ',\n result: ' + coverage_text)
        return []

    return coverage

