#!/bin/tcsh
#
# This script generates SPICE bundle PDS4 label.
#
# Pre-requisites/requirements:
#
# -- PDS4 values file, bundle XML template must exist
#
# -- PDS3->PDS4 tool, md5sum must be executables
#
# -- PDS4 labels and inventories must exists for document and SPICE 
#    kernel collections.
#
# -- PDS4 values file must include required keywords (see code for list)
#
# Version 1.0.0 -- May 19, 2015
#
#   Initial version.
#
# Version 1.1.0 -- November 9, 2015
#
#   BUG FIX: in the "if"s that check for presence of a LIDVID in existing
#   bundle labels changed "grep -c" to "grep ... | wc -l" to deal with 
#   different grep output formats for single and multiple files.
#
# Version 1.2.0 -- February 3, 2016
#
#   Changed readme.html to readme.txt..
#
# Version 1.3.0 -- November 15, 2017
#
#   Updated Unix date format string (%X->%T).
#
# Script name and version:
#
set SCRIPTNAME = "make_pds4_bundle_label.csh"
set SCRIPTVER  = "1.3.0 -- Novemver 15, 2017"

#
# Get command line stuff.
#
if ( $#argv != 2 ) then
    echo " "
    echo "$SCRIPTNAME, version $SCRIPTVER "
    echo " "
    echo "Usage: "
    echo
    echo "   % $SCRIPTNAME <valuefile> <templatefile>"
    echo " "
    echo "where:"
    echo " "
    echo "   valuefile    -- fixed PDS4 values KWD=VAL file"
    echo "   templatefile -- bundle label XML template file"
    echo
    exit 1
endif

set PDS4VALUES = $1
set XMLTEMPL   = $2

#
# Tools: PDS3-to-PDS4 converter, checksum utility.
#
set PDS4GEN    = "/home/bsemenov/maven/pds/make_pds4_xml_from_lbl.csh"
set MD5SUM     = "/usr/bin/md5sum"

#
# Collection directory and inventory and label name prefixes.
#
set SCOLDIR    = "spice_kernels"
set DCOLDIR    = "document"
set SCOLXPAT   = "$SCOLDIR/collection_spice_kernels_v???.xml"
set SCOLIPAT   = "$SCOLDIR/collection_spice_kernels_inventory_v???.csv"
set DCOLXPAT   = "$DCOLDIR/collection_document_v???.xml"
set DCOLIPAT   = "$DCOLDIR/collection_document_inventory_v???.csv"
set XMLEXT     = "xml"
set README     = "readme.txt"

#
# Check input files and programs.
#
foreach FF ( $PDS4VALUES $XMLTEMPL $README )
    if ( ! -e $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF does not exit."
	echo
	exit 1
    endif
end

foreach FF ( $PDS4GEN $MD5SUM )
    if ( ! -x $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF is not an excutable."
	echo
	exit 1
    endif
end

#
# Check that PDS4 values file has all required keywords. Extract
# BUNDLE_LID.
#
foreach KWD ( PDS4_MISSION_NAME \
              PDS4_MISSION_LID \
              PDS4_SPACECRAFT_NAME  \
              PDS4_SPACECRAFT_LID  \
              PDS4_TARGET_NAME  \
              PDS4_TARGET_TYPE  \
              PDS4_TARGET_LID  \
              BUNDLE_LID  \
              BUNDLE_DESCRIPTION_LID )

    set COUNT = `grep -c "^ *$KWD *=" $PDS4VALUES` 
    if ( $COUNT == 0 ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $PDS4VALUES does not include $KWD."
	echo
	exit 1
    endif
    if ( $COUNT > 1 ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $PDS4VALUES includes 2 or more $KWD."
	echo
	exit 1
    endif
end

set BUNDLID  = `grep BUNDLE_LID $PDS4VALUES         | sed 's/=/ /' | awk '{print $NF}'`

#
# Set bundle label file name prefix based on LID. 
#
set XMLPREF = "bundle_"`echo $BUNDLID | sed 's/:/ /g' | awk '{print $NF}' | sed 's/\./_/g' | sed 's/-/_/g' | sed 's/__*/_/g'`"_v"

#
# Check SPICE kernels collections. Set names, versions, *IDs.
#
foreach FF ( $SCOLDIR )
    if ( ! -d $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF is not a directory."
	echo
	exit 1
    endif
end

ls -1 $SCOLXPAT >& /dev/null
if ( $status ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot find files matching pattern(s) $SCOLXPAT."
    echo
    exit 1
endif

ls -1 $SCOLIPAT >& /dev/null
if ( $status ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot find files matching pattern(s) $SCOLIPAT."
    echo
    exit 1
endif

set SCOLXML  = `ls -1 $SCOLXPAT | tail -1`
set SCOLINV  = `ls -1 $SCOLIPAT | tail -1`
set SCOLXVER = `echo $SCOLXML | sed 's/^.*_v\([0-9][0-9][0-9]\)\.xml$/\1/'`
set SCOLIVER = `echo $SCOLINV | sed 's/^.*_v\([0-9][0-9][0-9]\)\.csv$/\1/'`

if ( "$SCOLXVER" != "$SCOLIVER" ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: latest SPICE kernel collection label and inventory version mismatch: $SCOLXML vs. $SCOLINV."
    echo
    exit 1
endif

set SCOLLID  = "${BUNDLID}":"$SCOLDIR"
set SCOLVID  = `echo $SCOLXVER | sed 's/^0*//' | awk '{printf "%d.0", $1}'`
set SCOLLVD  = "${SCOLLID}::${SCOLVID}"

if ( `grep -c "<logical_identifier> *$SCOLLID *</logical_identifier>" $SCOLXML` != 1 ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: zero or more than one LIDs $SCOLLID in $SCOLXML."
    echo
    exit 1
endif

if ( `grep -c "<version_id> *$SCOLVID *</version_id>" $SCOLXML` != 1 ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: zero or more than one VIDs $SCOLVID in $SCOLXML."
    echo
    exit 1
endif

set SCOLSTA  = "Primary"
ls ./$XMLPREF???.$XMLEXT >& /dev/null
if ( $status == 0 ) then
    if ( `grep "<lidvid_reference> *$SCOLLVD *</lidvid_reference>" ./$XMLPREF???.$XMLEXT | wc -l` != 0 ) then
	set SCOLSTA  = "Secondary"
    endif
endif

#
# Check documents collections. Set names, versions, *IDs.
#
foreach FF ( $DCOLDIR )
    if ( ! -d $FF ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: $FF is not a directory."
	echo
	exit 1
    endif
end

ls -1 $DCOLXPAT >& /dev/null
if ( $status ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot find files matching pattern(s) $DCOLXPAT."
    echo
    exit 1
endif

ls -1 $DCOLIPAT >& /dev/null
if ( $status ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot find files matching pattern(s) $DCOLIPAT."
    echo
    exit 1
endif

set DCOLXML  = `ls -1 $DCOLXPAT | tail -1`
set DCOLINV  = `ls -1 $DCOLIPAT | tail -1`
set DCOLXVER = `echo $DCOLXML | sed 's/^.*_v\([0-9][0-9][0-9]\)\.xml$/\1/'`
set DCOLIVER = `echo $DCOLINV | sed 's/^.*_v\([0-9][0-9][0-9]\)\.csv$/\1/'`

if ( "$DCOLXVER" != "$DCOLIVER" ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: latest document collection label and inventory version mismatch: $DCOLXML vs. $DCOLINV."
    echo
    exit 1
endif

set DCOLLID  = "${BUNDLID}":"$DCOLDIR"
set DCOLVID  = `echo $DCOLXVER | sed 's/^0*//' | awk '{printf "%d.0", $1}'`
set DCOLLVD  = "${DCOLLID}::${DCOLVID}"

if ( `grep -c "<logical_identifier> *$DCOLLID *</logical_identifier>" $DCOLXML` != 1 ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: zero or more than one LIDs $DCOLLID in $DCOLXML."
    echo
    exit 1
endif

if ( `grep -c "<version_id> *$DCOLVID *</version_id>" $DCOLXML` != 1 ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: zero or more than one VIDs $DCOLVID in $DCOLXML."
    echo
    exit 1
endif

set DCOLSTA  = "Primary"
ls ./$XMLPREF???.$XMLEXT >& /dev/null
if ( $status == 0 ) then
    if ( `grep "<lidvid_reference> *$DCOLLVD *</lidvid_reference>" ./$XMLPREF???.$XMLEXT | wc -l` != 0 ) then
	set DCOLSTA  = "Secondary"
    endif
endif

#
# Check that readme has CRs.
#
if ( `cat $README | wc -l` != `sed 's/\r/ CR/' $README | grep ' CR$' | wc -l` ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: the count of lines and CRs in $README don't agree."
    echo
    exit 1
endif

#
# Figure out the next bundle version.
#
set BUNDVER = "001"
while ( -e ./$XMLPREF$BUNDVER.$XMLEXT ) 
    set BUNDVER = `echo $BUNDVER | sed 's/^0*//' | awk '{printf "%03d", $1+1}'`
end

set BUNDXML = "./$XMLPREF$BUNDVER.$XMLEXT"

#
# Make bundle label.
#
set TMPLIST = "" 

#
# Determine START_TIME and STOP_TIME time. Grep MK PDS labels
# and pick the earliest START and the latest STOP.
#
\ls $SCOLDIR/mk/*.xml >& /dev/null

if ( $status ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot find any MK labels $SCOLDIR/mk/*.xml."
    echo
    exit 1
endif

set START = `grep 'start_date_time.*[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]' $SCOLDIR/mk/*.xml | sed 's/^.*\([0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]\).*$/\1/i' | sort | head -1`
set STOP  = `grep 'stop_date_time.*[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]'  $SCOLDIR/mk/*.xml | sed 's/^.*\([0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]\).*$/\1/i' | sort | tail -1`

if ( "$START" == "" ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot determine start time from MK labels $SCOLDIR/mk/*.xml."
    echo
    exit 1
endif

if ( "$STOP" == "" ) then
    echo 
    echo "ERROR: ${SCRIPTNAME}: cannot determine stop time from MK labels $SCOLDIR/mk/*.xml."
    echo
    exit 1
endif

#
# Set PRODUCT_CREATION_TIME and year to CPU clock.
#
set CURRTIME = `date +"%FT%T"`
set CURRYEAR = `date +"%Y"`

#
# Get file size from the file system.
#
set FILESIZE = `ls -l $README | awk '{print $5}'`

#
# Get MD5 checksum.
#
set CHECKSUM = `$MD5SUM $README | awk '{print $1}'`

#
# Set bundle VID: version.0
#
set BUNDVID = `echo $BUNDVER | sed 's/^0*//' | awk '{printf "%d.0", $1}'`

#
# Make complete temporary KWD=VAL PDS3-like label
#
set TMPLBL = "$BUNDXML:r.tmp.lbl"

\rm $TMPLBL >& /dev/null

echo                                        >>  $TMPLBL
echo "BUNDLE_VID            = "$BUNDVID     >>  $TMPLBL
echo "FILE_NAME             = "$README:t    >>  $TMPLBL
echo "CURRENT_YEAR          = "$CURRYEAR    >>  $TMPLBL
echo "FILE_SIZE             = "$FILESIZE    >>  $TMPLBL
echo "FILE_CHECKSUM         = "$CHECKSUM    >>  $TMPLBL
echo "SPICE_COLL_LIDVID     = "$SCOLLVD     >>  $TMPLBL
echo "SPICE_COLL_STATUS     = "$SCOLSTA     >>  $TMPLBL
echo "DOC_COLL_LIDVID       = "$DCOLLVD     >>  $TMPLBL
echo "DOC_COLL_STATUS       = "$DCOLSTA     >>  $TMPLBL
echo                                        >>  $TMPLBL
cat $PDS4VALUES                             >>  $TMPLBL
echo                                        >>  $TMPLBL
echo "START_TIME            = "$START       >>  $TMPLBL
echo "STOP_TIME             = "$STOP        >>  $TMPLBL
echo "PRODUCT_CREATION_TIME = "$CURRTIME    >>  $TMPLBL
echo                                        >>  $TMPLBL

set TMPLIST = "$TMPLIST $TMPLBL"

#
# Make temporary PDS4 label.
#
set TMPXML    = "$TMPLBL.xml"
set TMPXMLLOG = "$TMPXML.log"

\rm $TMPXML $TMPXMLLOG >& /dev/null

$PDS4GEN $TMPLBL $XMLTEMPL >& $TMPXMLLOG

if ( $status ) then

    #
    # Show error output.
    #
    cat $TMPXMLLOG
    echo
    echo "Values   file was $TMPLBL."
    echo "Template file was $XMLTEMPL."
    echo 

else 

    #
    # Success. Rename temporary XML label.
    #
    mv $TMPXML $BUNDXML
    if ( $status ) then
	echo 
	echo "ERROR: ${SCRIPTNAME}: failed to move $TMPXML to $BUNDXML."
	echo
	exit 1
    endif

    set TMPLIST = "$TMPLIST $TMPXMLLOG"

    echo 
    echo "Created $BUNDXML"

endif

#
# Cleanup.
#
\rm $TMPLIST >& /dev/null

#
# All done.
#
echo

exit
