import os
import select
import subprocess

import spiceypy
from spiceypy.utils.support_types import SpiceyError
from arcgen.utils.files import is_empty_file
from arcgen.utils.files import has_badchars
from arcgen.utils.files import exceeds_line_lengths

KERNEL_TEXT_EXTENSIONS = [".tf", ".ti", ".tls", ".tm", ".tpc", ".tsc"]
KERNEL_BINARY_EXTENSIONS = [".bc", ".bds", ".bpc", ".bsp"]
KERNEL_TEXT_HEADERS = {".tf": "KPL/FK",
                       ".ti": "KPL/IK",
                       ".tls": "KPL/LSK",
                       ".tm": "KPL/MK",
                       ".tpc": "KPL/PCK",
                       ".tsc": "KPL/SCLK"}
KERNEL_EXTENSIONS = KERNEL_TEXT_EXTENSIONS + KERNEL_BINARY_EXTENSIONS


# Modification of:
# https://spiceypy.readthedocs.io/en/main/other_stuff.html#lesson-1-kernel-management-with-the-kernel-subsystem
def write_mk_kernels_report(mk_path, out_report_file):

    # Prepare report file for writing
    out_file = open(out_report_file, 'w+')

    try:
        # Load the meta kernel then use KTOTAL to interrogate the SPICE
        # kernel subsystem.
        out_file.write('Loading MK: {0}\n'.format(mk_path))
        spiceypy.furnsh(mk_path)

        count = spiceypy.ktotal('ALL')
        out_file.write('Kernel count after load: {0}\n'.format(count))

        # Loop over the number of files; interrogate the SPICE system
        # with spiceypy.kdata for the kernel names and the type.
        # 'found' returns a boolean indicating whether any kernel files
        # of the specified type were loaded by the kernel subsystem.
        # This example ignores checking 'found' as kernels are known
        # to be loaded.
        for i in range(0, count):
            [file, ktype, source, handle] = spiceypy.kdata(i, 'ALL')
            out_file.write('File   {0}\n'.format(file))
            out_file.write('Type   {0}\n'.format(ktype))
            out_file.write('Source {0}\n\n'.format(source))

        # Lets write the first 1000 variables and their values
        cvals = spiceypy.gnpool("*", 0, 1000)
        for cval in cvals:

            #
            # Use spiceypy.dtpool to return the dimension and type,
            # C (character) or N (numeric), of each pool
            # variable name in the cvals array. We know the
            # kernel data exists.
            #
            [dim, type] = spiceypy.dtpool(cval)

            out_file.write('\n' + cval)
            out_file.write(' Number items: {0}   Of type: {1}\n'.format(dim, type))

            #
            # Test character equality, 'N' or 'C'.
            #
            if type == 'N':

                #
                # If 'type' equals 'N', we found a numeric array.
                # In this case any numeric array will be an array
                # of double precision numbers ('doubles').
                # spiceypy.gdpool retrieves doubles from the
                # kernel pool.
                #
                dvars = spiceypy.gdpool(cval, 0, 10)
                for dvar in dvars:
                    out_file.write('  Numeric value: {0:20.6f}\n'.format(dvar))

            elif type == 'C':

                #
                # If 'type' equals 'C', we found a string array.
                # spiceypy.gcpool retrieves string values from the
                # kernel pool.
                #
                cvars = spiceypy.gcpool(cval, 0, 10)

                for cvar in cvars:
                    out_file.write('  String value: {0}\n'.format(cvar))

            else:

                #
                # This block should never execute.
                #
                out_file.write('Unknown type. Code error.\n')

        # Now unload the meta kernel. This action unloads all
        # files listed in the meta kernel.
        spiceypy.unload(mk_path)

    except SpiceyError as err:
        out_file.write(str(err))

    # Done. Unload the kernels.
    spiceypy.kclear()

    return


def is_valid_kernel(kernel_file, check_line_length=True):

    extension = str(os.path.splitext(kernel_file)[1]).lower()
    if extension in KERNEL_EXTENSIONS:

        if extension in KERNEL_TEXT_EXTENSIONS:
            if not is_valid_text_kernel(kernel_file, check_line_length):
                print("ERROR!! - NOT A VALID TEXT KERNEL: " + kernel_file)
                return False

        else:
            # Is a binary kernel file, extract comments:
            if not is_valid_binary_kernel(kernel_file, check_line_length):
                print("ERROR!! - NOT A VALID BINARY KERNEL: " + kernel_file)
                return False

        if extension == ".tm":
            if not is_valid_metakernel(kernel_file):
                print("ERROR!! - WRONG META-KERNEL: " + kernel_file)
                return False

    else:
        print("ERROR!! - NOT A KERNEL EXTENSION: " + extension + " for file " + kernel_file)
        return False

    return True


def is_valid_text_kernel(filename, check_line_length=True):

    if not has_valid_text_kernel_header(filename):
        return False

    if not is_valid_comment_file(filename, check_line_length):
        return False

    # TODO: Continue implementation

    return True


def is_valid_comment_file(filename, check_line_length=True):

    if is_empty_file(filename):
        print("ERROR!! - EMPTY FILE: " + filename)
        return False

    if has_badchars(filename):
        print("ERROR!! - HAS BAD CHARS: " + filename)
        return False

    if check_line_length:
        if exceeds_line_lengths(filename):
            print("ERROR!! - EXCEEDS LINE LENGTH: " + filename)
            return False

    # TODO: Continue implementation

    return True


def is_valid_binary_kernel(filename, check_line_length=True):
    extension = str(os.path.splitext(filename)[1]).lower()

    # Write comments to a fie
    comment_filename = filename.replace(extension, ".commnt")
    with open(comment_filename, 'w') as commnt_file:
        commnt_file.write(str(commnt_read(filename)))

    is_valid = is_valid_comment_file(comment_filename, check_line_length)
    if not is_valid:
        print("ERROR!! - WRONG COMMENTS: " + filename)
    os.remove(comment_filename)

    return is_valid


# Copied from spice/adcsng/adcsng/utils/files.py
def commnt_read(kernel_path):

    command_line_process = subprocess.Popen(['commnt', '-r', kernel_path],
                                            bufsize=8192, shell=False,
                                            stdout=subprocess.PIPE,
                                            stderr=subprocess.PIPE)
    process_output = ""
    dataend = False
    while (command_line_process.returncode is None) or (not dataend):
        command_line_process.poll()
        dataend = False

        ready = select.select([command_line_process.stdout, command_line_process.stderr], [], [], 1.0)

        if command_line_process.stderr in ready[0]:
            data = command_line_process.stderr.read(1024)
            if len(data) > 0:
                raise Exception(data)

        if command_line_process.stdout in ready[0]:
            data = command_line_process.stdout.read(1024)
            if len(data) == 0:  # Read of zero bytes means EOF
                dataend = True
            else:
                process_output += data.decode('utf-8')

    return process_output


def is_valid_metakernel(mk_path):

    is_valid_mk = True

    try:
        # Load the meta kernel then use KTOTAL to interrogate the SPICE
        # kernel subsystem.
        abspath = os.path.abspath(mk_path)
        dname = os.path.dirname(abspath)
        os.chdir(dname)
        spiceypy.furnsh(mk_path)

        count = spiceypy.ktotal('ALL')

        # Loop over the number of files; interrogate the SPICE system
        # with spiceypy.kdata for the kernel names and the type.
        # 'found' returns a boolean indicating whether any kernel files
        # of the specified type were loaded by the kernel subsystem.
        # This example ignores checking 'found' as kernels are known
        # to be loaded.
        for i in range(0, count):
            [file, ktype, source, handle] = spiceypy.kdata(i, 'ALL')

        # Lets write the first 1000 variables and their values
        cvals = spiceypy.gnpool("*", 0, 1000)
        for cval in cvals:

            #
            # Use spiceypy.dtpool to return the dimension and type,
            # C (character) or N (numeric), of each pool
            # variable name in the cvals array. We know the
            # kernel data exists.
            #
            [dim, type] = spiceypy.dtpool(cval)

            #
            # Test character equality, 'N' or 'C'.
            #
            if type == 'N':

                #
                # If 'type' equals 'N', we found a numeric array.
                # In this case any numeric array will be an array
                # of double precision numbers ('doubles').
                # spiceypy.gdpool retrieves doubles from the
                # kernel pool.
                #
                dvars = spiceypy.gdpool(cval, 0, 10)

            elif type == 'C':

                #
                # If 'type' equals 'C', we found a string array.
                # spiceypy.gcpool retrieves string values from the
                # kernel pool.
                #
                cvars = spiceypy.gcpool(cval, 0, 10)

            else:

                #
                # This block should never execute.
                #
                print("ERROR!! - WRONG META-KERNEL: Unknonw var type: " + str(type) +
                      " at variable " + str(cval) + " in mk: " + mk_path)
                is_valid_mk = False

        # Now unload the meta kernel. This action unloads all
        # files listed in the meta kernel.
        spiceypy.unload(mk_path)

    except SpiceyError as err:
        print("ERROR!! - WRONG META-KERNEL: Raised exception: " + str(err) + " in mk: " + mk_path)
        is_valid_mk = False

    # Done. Unload the kernels.
    spiceypy.kclear()

    return is_valid_mk


def has_valid_text_kernel_header(kernel_file):
    f = open(kernel_file).readlines()
    extension = str(os.path.splitext(kernel_file)[1]).lower()
    if not KERNEL_TEXT_HEADERS[extension] in f[0]:
        print('ERROR!! - WRONG KERNEL HEADER: ' + f[0] + " instead of " +
              KERNEL_TEXT_HEADERS[extension] + " for text kernel: " + kernel_file)
        return False

    return True
