#!/bin/csh
#
#  This script performs a set of superficial checks on the
#  meta-information files and data files staged for a SPICE archive
#  prepared following the steps and using the tools from the SPICE 
#  Archive Preparation Guide.
#
#  This script uses 'badchar' program (source provided with SPICE
#  Archive Preparation Guide), 'bff' program (executable available from
#  Toolkit/Utilities page of the NAIF server, and 'awk' and 'grep'
#  UNIX utilities.
#
#  MCS/ESAC, Jan 3, 2017
#
#     Updated script with particulars for Rosetta. 
#
#  BVS/NAIF, Jan 8, 2009
#
#     Removed '-e' option from awk commands. Replaced 'strings' with
#     'bff' as the tool used to determine binary kernel architecture.
#     Made ./catalog/spice_hsk.cat exempt from long line check in all
#     places.
# 
#  BVS/NAIF, Jul 30, 2007
#
#     Added checks for text and binary kernels.
# 
#  BVS/NAIF, Feb 27, 2007
# 
#     Added extras to the checked lists.
#
#  LSE/NAIF, Sept 9, 2005
#
#     Made modifications so that the housekeeping file, spice_hsk.cat,
#     is exempt from line length checking.
#
#  LSE/NAIF, Sept 7, 2005
#
#     Added several *.cat files
#
#  LSE/NAIF, June 20, 2002
#
#     Made modifications, including lower case, for PDS-D. NOTE: Unlike 
#     other scripts, we had to make a (UPPER/lower) case choice.
#     This script will not work with the old PDS files.
#  
#  BVS/NAIF, April 25, 2000.
#
#     Added second pass capturing all text files (*.TXT, *.LBL, *.CAT ...)
#
#  BVS/NAIF, January 10, 2000.
#
#     Updated line length check to be for 78 char.
#
#  BVS/NAIF, January 5, 2000.
#

#
#  badchar and bff executables
#
set BADCHAR = 'badchar'
set BFF     = 'bff'

echo " "
echo "Problems are marked with (E) in at the beginning of report lines."
echo " "
#
#  Checking files.
#
echo " "
echo "   Verifying files -- REQUIRED meta-info files:"
echo " "

foreach FF ( readme.txt bundle_em16_spice_v002.xml document/collection_document_inventory_v002.csv document/collection_document_v002.xml document/spiceds_v002.html document/spiceds_v002.xml )

   #
   #  Does it exist?
   #
   if ( -e $FF ) then
   
      #
      #  Does it have non-zero size?
      #
      if ( ! ( -z $FF ) ) then
      
         #
         #  Does it have any bad characters?
         #
         $BADCHAR -s $FF
         
         if ( $status != 1 ) then
            
            #
            #  Does it have lines longer than 78 characters?
            #
            awk '{if(length>78){print $0}}' < $FF | grep '.' > /dev/null
            
            if ( ( $status != 0 ) || $FF == "aareadme.htm" || $FF == "index/INDEX.TAB" || $FF == "CATALOG/SPICE_HSK.CAT" ) then
               
               echo "      $FF is OK."
               
            else
               #
               #  Yes it does. Run awk again to dump these lines.
               #
               echo "(E)   $FF has lines longer than 78 char. They are:"
               echo " "
               awk '{if(length>78){print $0}}' < $FF
               echo " "
            endif
            
         else
            #
            #  Yes, it does have bad characters. Run badchar again
            #  to generate report.
            #
            echo "(E)   $FF has bad characters. badchar report is:"
            echo " "
            $BADCHAR -v $FF
            echo " "
         endif

      else
         echo "(E)   $FF has zero size."
      endif
      
   else
      echo "(E)   $FF doesn't exist."     
   endif

end

#
#  Checking files.
#
echo " "
echo "   Verifying files -- ALL meta-info files:"
echo " "

foreach FF ( `find . -name '*.csv' -print` -print` `find . -name '*.txt' -print`)

   #
   #  Does it exist?
   #
   if ( -e $FF ) then

      #
      #  Does it have non-zero size?
      #
      if ( ! ( -z $FF ) ) then

         #
         #  Does it have any bad characters?
         #
         $BADCHAR -s $FF

         if ( $status != 1 ) then

            #
            #  Does it have lines longer than 78 characters?
            #
            awk '{if(length>78){print $0}}' < $FF | grep '.' > /dev/null

            if ( ( $status != 0 ) || $FF:e == "html" || $FF:e == "csv" ) then

               echo "      $FF is OK."

            else
               #
               #  Yes it does. Run awk again to dump these lines.
               #
               echo "(E)   $FF has lines longer than 78 char. They are:"
               echo " "
               awk '{if(length>78){print $0}}' < $FF
               echo " "
            endif

         else
            #
            #  Yes, it does have bad characters. Run badchar again
            #  to generate report.
            #
            echo "(E)   $FF has bad characters. badchar report is:"
            echo " "
            $BADCHAR -v $FF
            echo " "
         endif

      else
         echo "(E)   $FF has zero size."
      endif

   else
      echo "(E)   $FF doesn't exist."
   endif

end


#
#  Checking text kernels.
#
echo " "
echo "   Verifying files -- text kernels, meta-kernels, and ORBNUM files:"
echo " "

foreach FF ( `find . -name '*.ti' -print` `find . -name '*.tf' -print` `find . -name '*.tls' -print` `find . -name '*.tpc' -print` `find . -name '*.tsc' -print` `find . -name '*.ten' -print` `find . -name '*.tm' -print` `find . -name '*.orb' -print` `find . -name '*.nrb' -print` )

   #
   #  Does it exist?
   #
   if ( -e $FF ) then

      #
      #  Does it have non-zero size?
      #
      if ( ! ( -z $FF ) ) then

         #
         #  Does it have any bad characters?
         #
         $BADCHAR -s $FF

         if ( $status != 1 ) then

            echo "      $FF is OK."

         else
            #
            #  Yes, it does have bad characters. Run badchar again
            #  to generate report.
            #
            echo "(E)   $FF has bad characters. badchar report is:"
            echo " "
            $BADCHAR -v $FF
            echo " "
         endif

      else
         echo "(E)   $FF has zero size."
      endif

   else
      echo "(E)   $FF doesn't exist."
   endif

end


#
#  Checking binary kernels.
#
echo " "
echo "   Verifying files -- binary kernels:"
echo " "

foreach FF ( `find . -name '*.bsp' -print` `find . -name '*.bc' -print` `find . -name '*.bpc' -print` `find . -name '*.bes' -print` `find . -name '*.bep' -print` `find . -name '*.bds' -print`)

   #
   #  Does it exist?
   #
   if ( -e $FF ) then

      #
      #  Does it have non-zero size?
      #
      if ( ! ( -z $FF ) ) then

         #
         #  Is this binary kernel in LTL-IEEE form?
         #
         if ( `$BFF $FF | grep -c 'LTL-IEEE'`  == 1 ) then

            echo "      $FF is OK."

         else
            #
            #  No, it is not. Report this fact:
            #
            echo "(E)   $FF is not in LTL-IEEE format. Its format is:"
            echo " "
            if ( `$BFF $FF | grep -c 'BIG-IEEE'` == 1 ) then
               echo "         BIG-IEEE"
            else 
               echo "         UNKNOWN"
            endif
            echo " "
         endif

      else
         echo "(E)   $FF has zero size."
      endif
      
   else
      echo "(E)   $FF doesn't exist."     
   endif

end

echo " "
echo "All done."
echo " "

exit
