C$Procedure PRCSET ( Process setup file for MEX2KER---umbrella routine )

      SUBROUTINE PRCSET ( SETUP,   INPUT,   OUTPUT,  APPEND, 
     .                    CMTFIL,  METHOD,  DEGREE,  SPCLIM,
     .                    QUOLIM,  KERBEG,  KEREND          )
  
C$ Abstract
C
C     Umbrella routine for MEX2KER setup file processing.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'mex2ker.inc'

      CHARACTER*(*)         SETUP
      CHARACTER*(*)         INPUT
      CHARACTER*(*)         OUTPUT
      LOGICAL               APPEND
      CHARACTER*(*)         CMTFIL
      CHARACTER*(*)         METHOD
      INTEGER               DEGREE
      DOUBLE PRECISION      SPCLIM
      DOUBLE PRECISION      QUOLIM
      DOUBLE PRECISION      KERBEG
      DOUBLE PRECISION      KEREND

C$ Brief_I/O
C
C     VARIABLE  I/O  Entry points
C     --------  ---  --------------------------------------------------
C     SETUP      I   GETSET
C     INPUT     I/O  GETSET
C     OUTPUT    I/O  GETSET
C     APPEND    I/O  GETSET
C     CMTFIL     O   GETINT
C     METHOD     O   GETINT
C     DEGREE     O   GETINT
C     SPCLIM     O   GETINT
C     QUOLIM     O   GETINT
C     KERBEG     O   GETINT
C     KEREND     O   GETINT
C
C$ Detailed_Input
C
C     See the entry points for a description of their inputs.
C
C$ Detailed_Output
C
C     See the entry points for a description of their outputs.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     If this routine is called directly, the error SPICE(BOGUSENTRY)
C     is signaled.  See the entry points for descriptions of 
C     exceptions specific to those routines.
C
C$ Files
C
C     This suite of routines reads and returns information from
C     an MEX2KER setup file.  This file is a SPICE text kernel
C     that specifies 
C
C        - A leapseconds kernel to be loaded.  The associated
C          keyword is
C
C             LEAPSECONDS_FILE
C
C        - The name of an interpolation method to use.  Currently
C          the supported values are 
C
C             'HERMITE'
C             'LAGRANGE'
C
C          The associated keyword is
C
C             INPUT_DATA_FILE
C
C        - The polynomial degree to use with the interpolation 
C          method.  (This item may become optional if other 
C          interpolation methods are added.)
C
C          The associated keyword is
C
C             INTERPOLATION_DEGREE
C
C         Optionally, the names of the input OASW orbit/attitude file
C         and the output SPK file may be specified in the setup file.
C         
C          The associated keywords are
C
C             INPUT_DATA_FILE
C             OUTPUT_SPK_FILE
C
C
C$ Particulars
C
C     The entry points in this package are
C
C        GETSET  {Get setup file information for MEX2KER}
C        GETINT  {Get interpolation information for MEX2KER}
C
C     GETSET must be called before GETINT may be called.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     1) Currently, parsing code for the following items is
C        disabled:
C
C           Interpolation method
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    MEX2KER Version 2.0.1, 21-JUN-2012 (NJB)
C
C        Updated header format.
C
C-    MEX2KER Version 2.0.0, 19-JUN-2003 (NJB)
C
C        Added processing of START_TIME and STOP_TIME keywords.
C
C-    MEX2KER Version 1.1.0, 09-JAN-2003 (NJB)
C
C        Added RETURN() check.  Added capability to process 
C        comment file specification.
C
C-    MEX2KER Version 1.0.0, 17-APR-2002 (NJB)
C
C-&

C
C     SPICELIB functions
C
      DOUBLE PRECISION      DPMAX
      DOUBLE PRECISION      DPMIN

      LOGICAL               EQCHR
      LOGICAL               EXISTS
      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C
      CHARACTER*(*)         APPNAM
      PARAMETER           ( APPNAM = 'APPEND_TO_OUTPUT'     )

      CHARACTER*(*)         BEGNAM
      PARAMETER           ( BEGNAM = 'START_TIME'           )

      CHARACTER*(*)         DEGNAM
      PARAMETER           ( DEGNAM = 'INTERPOLATION_DEGREE' )

      CHARACTER*(*)         INPNAM
      PARAMETER           ( INPNAM = 'INPUT_DATA_FILE'      )

      CHARACTER*(*)         INTNAM
      PARAMETER           ( INTNAM = 'INTERPOLATION_METHOD' )

      CHARACTER*(*)         LSKNAM
      PARAMETER           ( LSKNAM = 'LEAPSECONDS_FILE'     )

      CHARACTER*(*)         CLKNAM
      PARAMETER           ( CLKNAM = 'SCLK_KERNEL'          )

      CHARACTER*(*)         OUTCK
      PARAMETER           ( OUTCK  = 'OUTPUT_CK_FILE'       )

      CHARACTER*(*)         OUTSPK
      PARAMETER           ( OUTSPK = 'OUTPUT_SPK_FILE'      )

      CHARACTER*(*)         CMTNAM
      PARAMETER           ( CMTNAM = 'COMMENT_FILE'         )

      CHARACTER*(*)         NMPNAM
      PARAMETER           ( NMPNAM = 'NUMERIC_MAP'          )
      
      CHARACTER*(*)         CMPNAM
      PARAMETER           ( CMPNAM = 'CHARACTER_MAP'        )

      CHARACTER*(*)         QUONAM
      PARAMETER           ( QUONAM = 'MAXIMUM_SEP_QUOTIENT' )

      CHARACTER*(*)         SPCNAM
      PARAMETER           ( SPCNAM = 'MINIMUM_SPACING'      )

      CHARACTER*(*)         STPNAM
      PARAMETER           ( STPNAM = 'STOP_TIME'            )


      INTEGER               SHORT
      PARAMETER           ( SHORT  = 15 )

      INTEGER               TIMLEN
      PARAMETER           ( TIMLEN = 50 )

C
C     Local variables
C
      CHARACTER*(SHORT)     APPVAL
      CHARACTER*(1)         DTYPE
      CHARACTER*(FILSIZ)    LSK
      CHARACTER*(LNSIZE)    SAVMTH
      CHARACTER*(FILSIZ)    SCLK
      CHARACTER*(TIMLEN)    TIMSTR

      DOUBLE PRECISION      SAVBEG
      DOUBLE PRECISION      SAVEND
      DOUBLE PRECISION      SAVQUO
      DOUBLE PRECISION      SAVSPC

      INTEGER               SAVDEG
      INTEGER               N

      LOGICAL               INIT
      LOGICAL               FOUND

C
C     Saved variables
C     
      SAVE                 INIT
      SAVE                 SAVBEG
      SAVE                 SAVEND
      SAVE                 SAVMTH
      SAVE                 SAVDEG
      SAVE                 SAVQUO
      SAVE                 SAVSPC

C
C     Initial values
C
      DATA                 INIT  / .FALSE. /


      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'PRCSET' )

      CALL SIGERR ( 'SPICE(BOGUSENTRY)' )

      CALL CHKOUT ( 'PRCSET' )
      RETURN

       





C$Procedure GETSET ( Get setup file information for MEX2KER )
 
      ENTRY GETSET ( SETUP, INPUT, OUTPUT, APPEND, CMTFIL )
 
C$ Abstract
C
C     Get the names of the input OASW orbit/attitude file, the output
C     SPK file, and the append flag from an MEX2KER setup file.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
C
C     CHARACTER*(*)         SETUP
C     CHARACTER*(*)         INPUT
C     CHARACTER*(*)         OUTPUT
C     LOGICAL               APPEND
C     CHARACTER*(*)         CMTFIL
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     SETUP      I   Name of setup file.
C     INPUT     I/O  Name of input OASW orbit/attitude file.
C     OUTPUT    I/O  Name of output SPK file.
C     APPEND    I/O  Append flag.
C     CMTFIL     O   Name of comment file.
C
C$ Detailed_Input
C
C     SETUP          is the name of an MEX2KER setup file.
C
C     INPUT          is the name of an OASW orbit/attitude file to be
C                    converted to SPK format.

C     OUTPUT         is the name of an SPK file to be written.
C                    OUTPUT may be a new or existing file.
C                    OUTPUT may contain type 13 or type 9 segments.
C
C     APPEND         is a logical flag indicating whether to 
C                    append data to an existing output file.
C                    If APPEND is .TRUE., the file designated 
C                    by OUTPUT must exist.  If APPEND is .FALSE.,
C                    the file designated by OUTPUT must not exist.
C
C$ Detailed_Output
C
C     CMTFIL         is the name of a comment file whose contents
C                    are to be added to the comment area of
C                    the SPK file created by this program.  The
C                    comment file contents precede the default
C                    comments added by MEX2KER.
C
C                    If no comment file is specified, CMTFIL is
C                    returned blank.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1)  If the setup file name is blank, the error
C         SPICE(BLANKFILENAME) is signaled.
C
C     2)  If the setup file is not found, the error
C         SPICE(FILENOTFOUND) is signaled.
C
C     3)  If the input file name argument INPUT is blank, and the
C         name of the input file is not specified in the setup file,
C         the error SPICE(NOFILESPEC) is signaled.
C
C     4)  If the input file is not found, the error
C         SPICE(FILENOTFOUND) is signaled.
C
C     5)  If the output file name argument INPUT is blank, and the
C         name of the output file is not specified in the setup file,
C         the error SPICE(NOFILESPEC) is signaled.
C
C     6)  if the APPEND argument is true and the output file is not
C         found, the error SPICE(FILENOTFOUND) is signaled.
C
C     7)  if the APPEND argument is false and the output file already
C         exists, the error SPICE(FILEEXISTS) is signaled.
C
C     8)  If the name of the leapseconds kernel is not specified in the
C         setup file, the error SPICE(NOFILESPEC) is signaled.
C
C     9)  If no interpolation degree was found in the setup file, the
C         error SPICE(NODEGREE) is signaled.
C
C     10)  If a comment file is specified via a kernel variable of
C          numeric type, the error SPICE(TYPEMISMATCH) is signaled.
C
C     11)  If a minimum state spacing is specified via a kernel
C          variable of character type, the error SPICE(TYPEMISMATCH) is
C          signaled.
C
C     12)  If a maximum state spacing ratio is specified via a kernel
C          variable of character type, the error SPICE(TYPEMISMATCH) is
C          signaled.
C
C     13)  If a kernel start time is specified via a kernel
C          variable of character type, the error SPICE(TYPEMISMATCH) is
C          signaled.
C
C     14)  If a kernel stop time is specified via a kernel
C          variable of character type, the error SPICE(TYPEMISMATCH) is
C          signaled.
C
C$ Files
C
C     See the descriptions of INPUT and OUTPUT above.
C
C$ Particulars
C
C     None.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Red Book, revision Acton 502-H.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    MEX2KER Version 1.1.1, 21-JUN-2012 (NJB)
C
C        Updated header format.
C
C-    MEX2KER Version 1.1.0, 13-JAN-2003 (NJB)
C
C        Added RETURN() check.  Added capability to process 
C        comment file, minimum spacing, and maximum spacing
C        quotient specifications.
C
C-    MEX2KER Version 1.0.0, 02-APR-2002 (NJB)
C
C-&

      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'GETSET' )

C
C     Check the setup file name.
C
      IF ( SETUP .EQ. ' ' ) THEN

         CALL SETMSG ( 'Setup file name may not be blank.' )
         CALL SIGERR ( 'SPICE(BLANKFILENAME)'              )
         CALL CHKOUT ( 'GETSET'                            )
         RETURN

      END IF

      IF ( .NOT. EXISTS(SETUP) ) THEN

         CALL SETMSG ( 'Setup file <#> was not found.' )
         CALL ERRCH  ( '#', SETUP                      )
         CALL SIGERR ( 'SPICE(FILENOTFOUND)'           )
         CALL CHKOUT ( 'GETSET'                        )
         RETURN
         
      END IF

C
C     Load the setup file.
C      
      CALL FURNSH ( SETUP )

      IF ( FAILED() ) THEN
      
         CALL CHKOUT ( 'GETSET' )
         RETURN

      END IF

C
C     Check the append flag.  If APPEND is .FALSE. on entry, see
C     whether appending is specified in the setup file.  Otherwise
C     leave APPEND untouched.
C
      IF ( .NOT. APPEND ) THEN

         CALL GCPOOL ( APPNAM, 1, 1, N, APPVAL, FOUND )

         IF ( FOUND ) THEN
C
C           Use the first non-blank letter of the value to 
C           determine the RHS value.
C
            CALL LJUST ( APPVAL, APPVAL )

            APPEND = EQCHR( APPVAL(1:1), 'Y' ) 

         END IF

      END IF
      
C
C     Check the input file name.  If the name is available, it 
C     supersedes an input file name supplied in the setup file.
C     If the name is blank, the setup file must supply the name.
C
      IF ( INPUT .EQ. ' ' ) THEN         
C
C        Extract the input file name from the kernel pool.
C        Set the INPUT argument to the specified file name.
C
         CALL GCPOOL ( INPNAM, 1, 1, N, INPUT, FOUND )

         IF ( .NOT. FOUND ) THEN

            CALL SETMSG ( 'Input file was not specified on the ' //
     .                    'command line or in the setup file.'    )
            CALL SIGERR ( 'SPICE(NOFILESPEC)'                     )
            CALL CHKOUT ( 'GETSET'                                )
            RETURN

         END IF

      END IF

      IF ( .NOT. EXISTS(INPUT) ) THEN

         CALL SETMSG ( 'Input file <#> was not found.' )
         CALL ERRCH  ( '#', INPUT                      )
         CALL SIGERR ( 'SPICE(FILENOTFOUND)'           )
         CALL CHKOUT ( 'GETSET'                        )
         RETURN

      END IF
      
C
C     Check the output file name.  If the name is available, it 
C     supersedes an output file name supplied in the setup file.
C     If the name is blank, the setup file must supply the name.
C
      IF ( OUTPUT .EQ. ' ' ) THEN
C
C        Extract the output file name from the kernel pool.
C        Set the INPUT argument to the specified file name.
C
         CALL GCPOOL ( OUTSPK, 1, 1, N, OUTPUT, FOUND )

         IF ( .NOT. FOUND ) THEN

            CALL GCPOOL ( OUTCK, 1, 1, N, OUTPUT, FOUND )
           
            IF ( .NOT. FOUND ) THEN

               CALL SETMSG ( 'Output file was not specified on '      //
     .                       'the command line or in the setup file.' )
               CALL SIGERR ( 'SPICE(NOFILESPEC)'                      )
               CALL CHKOUT ( 'GETSET'                                 )
               RETURN

            END IF

         END IF

      END IF


      IF ( APPEND ) THEN

         IF ( .NOT. EXISTS(OUTPUT)  ) THEN

            CALL SETMSG ( 'Output file <#> was not found.  File ' //
     .                    'must exist for appending.'              )
            CALL ERRCH  ( '#', OUTPUT                              )
            CALL SIGERR ( 'SPICE(FILENOTFOUND)'                    )
            CALL CHKOUT ( 'GETSET'                                 )
            RETURN

         END IF

      ELSE

         IF ( EXISTS(OUTPUT)  ) THEN

            CALL SETMSG ( 'Output file <#> already exists.  File ' //
     .                    'must not exist unless appending '       //
     .                    'is requested.'                           )
            CALL ERRCH  ( '#', OUTPUT                               )
            CALL SIGERR ( 'SPICE(FILEEXISTS)'                       )
            CALL CHKOUT ( 'GETSET'                                  )
            RETURN

         END IF

      END IF

C
C     Obtain the name of the leapseconds kernel and load the kernel.
C
      CALL GCPOOL ( LSKNAM, 1, 1, N, LSK, FOUND )

      IF ( .NOT. FOUND ) THEN

         CALL SETMSG ( 'Leapseconds kernel was not specified  ' //
     .                 'in the setup file.'                      )
         CALL SIGERR ( 'SPICE(NOFILESPEC)'                       )
         CALL CHKOUT ( 'GETSET'                                  )
         RETURN

      END IF

      CALL FURNSH ( LSK )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'GETSET' )
         RETURN
      END IF

C
C     Obtain the name of the SCLK kernel, if any.  If an SCLK
C     kernel is specfied, load the kernel.
C
      CALL GCPOOL ( CLKNAM, 1, 1, N, SCLK, FOUND )

      IF ( FOUND ) THEN

         CALL FURNSH ( SCLK )

         IF ( FAILED() ) THEN
            CALL CHKOUT ( 'GETSET' )
            RETURN
         END IF

      END IF


C
C     Look up the interpolation method and interpolation degree 
C     (we're assuming a polynomial method is used).
C
      CALL GCPOOL ( INTNAM, 1, 1, N, SAVMTH, FOUND )

      IF ( .NOT. FOUND ) THEN

         SAVMTH = ' '
C
C        CALL SETMSG ( 'Interpolation method was not specified  ' //...
C                      'in the setup file.'                        )
C        CALL SIGERR ( 'SPICE(NOINTERP)'                           )
C        CALL CHKOUT ( 'GETSET'                                    )
C        RETURN
C
      END IF


      CALL GIPOOL ( DEGNAM, 1, 1, N, SAVDEG, FOUND )

      IF ( .NOT. FOUND ) THEN

         CALL SETMSG ( 'Interpolation degree was not specified  ' //
     .                 'in the setup file.'                        )
         CALL SIGERR ( 'SPICE(NODEGREE)'                           )
         CALL CHKOUT ( 'GETSET'                                    )
         RETURN

      END IF

C
C     See whether a comment file specification was given.
C     Capture the file name if so.
C
      CALL DTPOOL ( CMTNAM, FOUND, N, DTYPE )

      IF ( FOUND ) THEN

         IF ( DTYPE .NE. 'C' ) THEN

            CALL SETMSG ( 'Comment file name was not given a '       //
     .                    'character string value in the setup file.' )
            CALL SIGERR ( 'SPICE(TYPEMISMATCH)'                       )
            CALL CHKOUT ( 'GETSET'                                    )
            RETURN

         END IF

         CALL GCPOOL ( CMTNAM, 1, 1, N, CMTFIL, FOUND ) 
         
      END IF

      IF ( .NOT. FOUND ) THEN
         CMTFIL =  ' '
      END IF

C
C     See whether a minimum state spacing was given.
C     Capture the value if so.  Otherwise use the default.
C
      CALL DTPOOL ( SPCNAM, FOUND, N, DTYPE )

      IF ( FOUND ) THEN

         IF ( DTYPE .NE. 'N' ) THEN

            CALL SETMSG ( 'Minimum state spacing was not given a '   //
     .                    'double precision value in the setup file.' )
            CALL SIGERR ( 'SPICE(TYPEMISMATCH)'                       )
            CALL CHKOUT ( 'GETSET'                                    )
            RETURN

         END IF

         CALL GDPOOL ( SPCNAM, 1, 1, N, SAVSPC, FOUND ) 
         
      END IF

      IF ( .NOT. FOUND ) THEN
         SAVSPC = MINSPC
      END IF


C
C     See whether a maximum state spacing quotient was given.
C     Capture the value if so.  Otherwise use the default.
C
      CALL DTPOOL ( QUONAM, FOUND, N, DTYPE )

      IF ( FOUND ) THEN

         IF ( DTYPE .NE. 'N' ) THEN

            CALL SETMSG ( 'Maximum state spacing quotient was not '  //
     .                    'given a double precision value in the '   //
     .                    'setup file.'                               )
            CALL SIGERR ( 'SPICE(TYPEMISMATCH)'                       )
            CALL CHKOUT ( 'GETSET'                                    )
            RETURN

         END IF

         CALL GDPOOL ( QUONAM, 1, 1, N, SAVQUO, FOUND ) 
         
      END IF

      IF ( .NOT. FOUND ) THEN
         SAVQUO = MAXQUO
      END IF

C
C     See whether an output kernel start time was given.
C     Capture the value if so.  Otherwise use the default.
C
      CALL DTPOOL ( BEGNAM, FOUND, N, DTYPE )

      IF ( FOUND ) THEN

         IF ( DTYPE .NE. 'C' ) THEN

            CALL SETMSG ( 'Kernel start time was not '               //
     .                    'given a character string value in the '   //
     .                    'setup file.'                               )
            CALL SIGERR ( 'SPICE(TYPEMISMATCH)'                       )
            CALL CHKOUT ( 'GETSET'                                    )
            RETURN

         END IF

         CALL GCPOOL ( BEGNAM, 1, 1, N, TIMSTR, FOUND ) 
         
         IF ( FOUND ) THEN

            CALL STR2ET ( TIMSTR, SAVBEG )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'GETSET' )
               RETURN
            END IF

         END IF

      END IF

      IF ( .NOT. FOUND ) THEN
         SAVBEG = DPMIN()
      END IF

C
C     See whether an output kernel stop time was given.
C     Capture the value if so.  Otherwise use the default.
C
      CALL DTPOOL ( STPNAM, FOUND, N, DTYPE )

      IF ( FOUND ) THEN

         IF ( DTYPE .NE. 'C' ) THEN

            CALL SETMSG ( 'Kernel stop time was not '                //
     .                    'given a character string value in the '   //
     .                    'setup file.'                               )
            CALL SIGERR ( 'SPICE(TYPEMISMATCH)'                       )
            CALL CHKOUT ( 'GETSET'                                    )
            RETURN

         END IF

         CALL GCPOOL ( STPNAM, 1, 1, N, TIMSTR, FOUND ) 
         
         IF ( FOUND ) THEN

            CALL STR2ET ( TIMSTR, SAVEND )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'GETSET' )
               RETURN
            END IF

         END IF

      END IF

      IF ( .NOT. FOUND ) THEN
         SAVEND = DPMAX()
      END IF


      INIT = .TRUE.

      CALL CHKOUT ( 'GETSET' )
      RETURN






C$Procedure GETINT ( Get interpolation parameters for MEX2KER )
 
      ENTRY GETINT ( METHOD, DEGREE, SPCLIM, QUOLIM, KERBEG, KEREND )
 
C$ Abstract
C
C     Return interpolation parameters specified
C     in an MEX2KER setup file.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
C
C     CHARACTER*(*)         METHOD
C     INTEGER               DEGREE
C     DOUBLE PRECISION      SPCLIM
C     DOUBLE PRECISION      QUOLIM
C     DOUBLE PRECISION      KERBEG
C     DOUBLE PRECISION      KEREND
C
C$ Brief_I/O
C
C     VARIABLE  I/O  Description
C     --------  ---  --------------------------------------------------
C     METHOD     O   Interpolation method.
C     DEGREE     O   Degree of interpolating polynomials.
C     SPCLIM     O   Minimum state spacing.
C     QUOLIM     O   Maximum quotient of successive state separations.
C     KERBEG     O   TDB begin time of ouput kernel coverage. 
C     KEREND     O   TDB end time of ouput kernel coverage.
C
C$ Detailed_Input
C
C     None.
C
C$ Detailed_Output
C
C     METHOD         is the interpolation method:  'HERMITE' or
C                    'LAGRANGE'.
C
C     DEGREE         is the associated polynomial degree. 
C
C     SPCLIM         is the minimum spacing, measured in ephemeris
C                    seconds, between successive states in one
C                    input data block (the set of states following
C                    a metadata block and preceding the next
C                    metadata block or EOF).  If not specified in the
C                    setup file, SPCLIM defaults to MINSPC (declared in
C                    epm2spk.inc).
C     
C     QUOLIM         is the maximum quotient of successive separations
C                    between states (considered in either order).  If
C                    not specified in the setup file, QUOLIM defaults
C                    to MAXQUO (declared in epm2spk.inc).
C 
C     KERBEG,
C     KEREND         are time bounds for the output kernel file.  These
C                    bounds are optionally specified in the setup file
C                    by the keywords
C
C                       START_TIME
C                       STOP_TIME
C                   
C                    These bounds are expressed as seconds past J2000,
C                    TDB for both SPK and CK files.
C 
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     If this routine is called before the setup file has been
C     read by GETSET, the error SPICE(NOTINITIALIZED) is signaled.
C
C$ Files
C
C     The interpolation method and degree are specified in an
C     MEX2KER setup file.  See the header of the umbrella routine
C     PRCSET for further information.
C
C$ Particulars
C
C     None.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Red Book, revision Acton 502-H.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    MEX2KER Version 3.0.1, 21-JUN-2012 (NJB)
C
C        Updated header format.
C
C-    MEX2KER Version 3.0.0, 19-JUN-2003 (NJB)
C
C        Interface was updated to return KERBEG and KEREND.
C
C-    MEX2KER Version 2.0.0, 10-JAN-2003 (NJB)
C
C        Interface was updated to return SPCLIM and QUOLIM.
C
C-    MEX2KER Version 1.0.0, 17-APR-2002 (NJB)
C
C-&
      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'GETINT' )

      IF ( .NOT. INIT ) THEN

         CALL SETMSG ( 'GETINT may not be called before setup ' //
     .                 'initialization has been completed.'      )
         CALL SIGERR ( 'SPICE(NOTINITIALIZED)'                   )
         CALL CHKOUT ( 'GETINT'                                  )
         RETURN

      END IF

      METHOD = SAVMTH
      DEGREE = SAVDEG
      SPCLIM = SAVSPC
      QUOLIM = SAVQUO
      KERBEG = SAVBEG
      KEREND = SAVEND

      CALL CHKOUT ( 'GETINT' )
      RETURN
      END

