import os
import re
import glob
import shutil
import tarfile
import subprocess
import numpy.testing as npt

from adcsng.utils.files import print_html
from adcsng.utils.files import get_exe_dir
from adcsng.command_line import main as adcsng


def runADCSng(version, debug=False, log=False, untar=True):

    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    if untar:
        tar = tarfile.open('incoming.tar.gz')
        tar.extractall()
        tar.close()

    print_html(f"Executing ADCSng v{version} for ExoMarsRSP")
    adcsng('exomarsrsp.json', debug=debug, log=log)


def cleanup():
    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    try:
        os.remove('emrsp_test.lock')
    except:
        pass

    try:
        shutil.rmtree('incoming')
        os.remove('._incoming')
    except:
        pass

    for CleanUp in glob.glob('ker_gen/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('ancdr/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('log/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('temp/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt') or not CleanUp.endswith('temp.log'):
            try:
                shutil.rmtree(CleanUp)
            except:
                if 'aareadme.txt' not in CleanUp:
                    os.remove(CleanUp)


def SPKcompare():

    spks = glob.glob('ker_gen/spk/*.bsp')
    for spk in spks:

        exe_dir = get_exe_dir()
        spk_1 = spk
        spk_2 = f'ker_val/spk/{spk.split(os.sep)[-1]}'
        fk = 'ker_dir/fk/emrsp_rm_v006.tf'

        if not os.path.exists(spk_1): raise NameError(f"SPK Kernel {spk_1} does not exist")
        if not os.path.exists(spk_2): raise NameError(f"SPK Kernel {spk_2} does not exist")

        command = f'../..{exe_dir}/spkdiff -k {fk} {spk_1}  {spk_2}'
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                           stdout=subprocess.PIPE,stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)
        if 'Toolkit version:' in text:
            raise NameError(f"SPKDIFF did not work with SPK Kernels {spk}")

        try:
            text = text.split('\n')
            for line in text:
                if 'Position:' in line or 'Velocity:' in line:
                    npt.assert_almost_equal(float(line.split()[1]), 0.0, decimal=6)
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=6)
                if 'Position (km)' in line or 'Velocity (km/s)' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=6)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=6)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"SPK Kernels {spk} are different")


def CKcompare():

    kernels =  'ker_dir/fk/emrsp_rm_v006.tf ' + \
               'ker_dir/sclk/emrsp_rm_test_ret_20191128_v001.tsc ' + \
               'ker_dir/lsk/naif0012.tls '

    cks = {'1':{'ck': 'emrsp_rm_surf_gnc-tlm_0000_v001.bc',
              'fframe': 'RM_ROVER',
              'tframe': 'RM_MLG',
              'start':  '2020-NOV-01 00:00:00.000 TDB',
              'finish': '2020-NOV-01 00:00:30.000 TDB'},
           '2': {
               'ck':     'emrsp_rm_surf_interp-tlm_0007_v001.bc',
               'fframe': 'RM_ROVER',
               'tframe': 'RM_MLG',
               'start':  '2020-NOV-01 00:00:00.000 TDB',
               'finish': '2020-NOV-08 04:37:36.000 TDB'},
           '3': {
               'ck':     'emrsp_rm_surf_interp-tlm_0007_v001.bc',
               'fframe': 'RM_MAST',
               'tframe': 'RM_MAST_ZERO',
               'start':  '2020-NOV-01 00:00:00.000 TDB',
               'finish': '2020-NOV-08 04:37:36.000 TDB'},
           '4': {
               'ck':     'emrsp_rm_surf_interp-tlm_0007_v001.bc',
               'fframe': 'RM_PTR',
               'tframe': 'RM_PTR_ZERO',
               'start':  '2020-NOV-01 00:00:00.000 TDB',
               'finish': '2020-NOV-08 04:37:36.000 TDB'},
           '5': {
               'ck':     'emrsp_rm_surf_mech-tlm_0000_v001.bc',
               'fframe': 'RM_DRILL_POS_MOV',
               'tframe': 'RM_DRILL_POS_FIX',
               'start':  '2020-NOV-01 00:00:00.000 TDB',
               'finish': '2020-NOV-01 00:00:30.000 TDB'},
           '6': {'ck':     'emrsp_rm_surf_rec_0007_v001.bc',
                 'fframe': 'RM_ROVER',
                 'tframe': 'RM_MLG',
                 'start':  '2020-NOV-02 00:00:00.000 TDB',
                 'finish': '2020-NOV-09 04:37:36.000 TDB'},
           }

    if len(cks) < len(os.listdir("ker_gen/ck/")):
        raise NameError("WARNING: some kernels in ker_gen are not being tested")


    for ck in cks.items():
        ck = ck[1]
        exe_dir = get_exe_dir()
        ck_1 = f"ker_val/ck/{ck['ck']}"
        ck_2 = f"ker_gen/ck/{ck['ck']}"

        if not os.path.exists(ck_1): raise NameError(f"CK Kernel {ck_1} does not exist")
        if not os.path.exists(ck_2): raise NameError(f"CK Kernel {ck_2} does not exist")

        command = f"../..{exe_dir}/frmdiff -k {kernels} -f1 {ck['fframe']} " \
                  f"-t1 {ck['tframe']} -b {ck['start']} -e {ck['finish']} " \
                  f" {ck_1} {ck_2}"
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                                stdout=subprocess.PIPE,
                                                stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)
        if 'version: N0066' in text:
            raise NameError(f"FRMDIFF did not work with CK Kernels {ck['ck']}: {text}")

        #
        # Whilst with the SPKs and SCLKs it is enough to compare the DIFF
        # files, for PREDICKT the mantissa and the precision differ from
        # Linux to MacOS:
        #
        #   OS:    Rotation (rad):             +4.8013978590743E-16      +1.4316667159571E-16
        #   LINUX: Rotation (rad):             +0.0000000000000E+00      +0.0000000000000E+00
        #
        # Therefore we will use an assertion instead of a file comparison
        #
        try:
            text = text.split('\n')
            for line in text:
                if 'Rotation (rad):' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=10)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=10)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"CK Kernels {ck['ck']} are different")

    return


def SCLKcompare():

    begin = '2019-04-30'
    finish = '2020-02-15'
    id =  '-174'

    lsk = 'ker_dir/lsk/naif0012.tls'
    sclks = glob.glob('ker_gen/sclk/*.tsc')
    for sclk in sclks:

        sclk = sclk.split(os.sep)[-1]
        exe_dir = get_exe_dir()
        command = f"../..{exe_dir}/diffsclk {lsk} {begin} {finish} " \
                  f"864000 {id} ker_val/sclk/{sclk} ker_gen/sclk/{sclk}"
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                           stdout=subprocess.PIPE,
                                           stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)

        try:
            text = text.split('\n')
            for line in text:
                if line.strip():
                        npt.assert_almost_equal(float(line.split()[1]), 0.0,
                                            decimal=5)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"SCLK Kernels {sclk} are different")

    return


def MKcompare():

    mks = [f for f in os.listdir('ker_gen/mk/') if
           re.search(r'.*.tm', f)]

    if not mks:
        raise NameError("No MK Kernels available")

    for mk in mks:
        data_gen = []
        record = False
        with open(f'ker_gen/mk/{mk}', 'r') as m:
            for line in m:
                if record: data_gen.append(line)
                if '\\begindata' in line: record = True
                if '\\begintext' in line: break

        print_html(''.join(data_gen))

        with open(f'ker_val/mk/{mk}', 'r') as m: data_val = m.readlines()

        if not set(data_gen).issubset(data_val):
            raise NameError("MK Kernels are different")

    return


if __name__ == '__main__':
    cleanup()
    runADCSng('1.1.0', debug=True, log=True, untar=True)
#    os.chdir(os.getcwd()+'/..')
#    SCLKcompare()
#    SPKcompare()
#    CKcompare()
#    MKcompare()