C$Procedure MKCK06 ( Make CK type 6 segments )
 
      SUBROUTINE MKCK06 ( INPUT,  HANDLE, DEFMTH, DEFDEG, SPCLIM,  
     .                    QUOLIM, CKBEG,  CKEND,  RATE,   SELLST )
 
C$ Abstract
C
C     Write out one or more CK type 6 segments representing the
C     attitude provided by an AEM file.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     AEM2CK
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'aem2ck.inc'
      INCLUDE 'ck06.inc'

      CHARACTER*(*)         INPUT
      INTEGER               HANDLE
      CHARACTER*(*)         DEFMTH
      INTEGER               DEFDEG
      DOUBLE PRECISION      SPCLIM
      DOUBLE PRECISION      QUOLIM
      DOUBLE PRECISION      CKBEG
      DOUBLE PRECISION      CKEND
      DOUBLE PRECISION      RATE
      LOGICAL               SELLST
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     INPUT      I   AEM file to be read.
C     HANDLE     I   Handle of CK file.
C     DEFMTH     I   Default interpolation method.
C     DEFDEG     I   Default interpolation degree.
C     SPCLIM     I   Minimum attitude record time tag spacing.
C     QUOLIM     I   Maximum quotient of successive time tag
C                    separations.
C     CKBEG      I   CK start time bound.
C     CKEND      I   CK end time bound.
C     RATE       I   Nominal SCLK rate.
C     SELLST     I   Segment selection flag.
C
C$ Detailed_Input
C
C     INPUT          is the name of an AEM file to be read.
C
C     HANDLE         is the handle of the CK file to write.  The
C                    file may be new or may be open for appending.
C
C     DEFMTH         is a string specifying the default interpolation
C                    method to use. This method is used only for input
C                    AEM blocks that don't have an interpolation method
C                    specified in their preceding metadata block.
C                    Allowed values are
C
C                       'HERMITE'
C                       'LAGRANGE'
C                       
C                    Case and white space are insignificant.
C
C     DEFDEG         is the polynomial degree associated with the
C                    interpolation method. This method is used only for
C                    input AEM blocks that don't have an interpolation
C                    method specified in their preceding metadata
C                    block.
C
C                    Note that DEFDEG is not used unless a default
C                    method is provided.
C
C
C     SPCLIM         is the minimum spacing, measured in ephemeris
C                    seconds, between time tags of successive attitude
C                    records in one input data block (the set of
C                    records following a metadata block and preceding
C                    the next metadata block or EOF). If not specified
C                    in the setup file, SPCLIM defaults to MINSPC
C                    (declared in
C                    aem2ck.inc).
C     
C     QUOLIM         is the maximum quotient of successive separations
C                    between attitude records (considered in either
C                    order). If not specified in the setup file, QUOLIM
C                    defaults to MAXQUO (declared in aem2ck.inc).
C
C     CKBEG,
C     CKEND          are, respectively, output CK file begin and end
C                    times specified in the setup file.  The coverage
C                    of the output file is that of the input AEM file
C                    intersected with the interval [CKBEG, CKEND]. The
C                    bounds are expressed as seconds past J2000 TDB.
C
C     RATE           is the nominal SCLK rate in seconds/tick. A 
C                    single rate is used for all segments in the
C                    output CK. The rate is used to convert angular
C                    velocity from radians/tick to radians/second.
C
C     SELLST         is a logical flag indicating to the CK type 6
C                    segment reader CKR06 how to select the
C                    mini-segment interval when a request time
C                    coincides with a time boundary shared by two
C                    mini-segment intervals. When SELLST ("select
C                    last") is .TRUE., the later interval is selected;
C                    otherwise the earlier interval is selected.
C                                         
C$ Detailed_Output
C
C     None.  See Particulars for a description of the action taken
C     by this routine.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1)  If an unrecognized interpolation method is requested, the
C         error SPICE(NOTSUPPORTED) will be signaled.
C
C     2)  If an error occurs while writing a CK segment, the problem
C         will be diagnosed by routines called by this routine.
C
C     3)  If an error occurs while parsing attitude data from the input
C         AEM file, the problem will be diagnosed by routines called by
C         this routine.
C
C     4)  If TIMSYS does not specify a supported time system, the error
C         will be diagnosed by routines called by this routine.
C
C     5)  If the time tags of any two consecutive attitude records in
C         the AEM data block have separation less than SPCLIM, the
C         error SPICE(TIMETAGERROR) will be signaled.
C
C     6)  If the time tags T1, T2, T3 of any three consecutive attitude
C         records in the AEM data block have spacing ratio R greater
C         than QUOLIM, where R is defined by
C
C            R  =  MAX ( (T3-T2)/(T2-T1), (T2-T1)/(T3-T2) )
C
C         the error SPICE(TIMETAGERROR) will be signaled.
C 
C$ Files
C
C     See the descriptions of the arguments INPUT and OUTPUT.
C
C$ Particulars
C
C     This routine creates one or more type 6 CK segments in the
C     specified output CK file, using data from the specified AEM file.
C     Data blocks in the input file normally map to mini-segments in
C     the output file. Blocks can be split up into multiple segments if
C     necessary due to buffer size limits. If blocks are split, padding
C     is used to preserve interpolation behavior.
C
C     Consecutive input data blocks with intervening time gaps can
C     are stored in consecutive mini-segments, provided that:
C
C        - There is room to buffer the input data
C
C        - The data blocks have compatible CK segment attributes:
C          specifically the same base and CK reference frames and
C          correspondence to a subtype of CK type 6.
C
C        - Data blocks preceding a gap don't have padding on the
C          right hand side. 
C
C     When these conditions are not met, the block on the right hand
C     side of the gap is mapped to a new CK segment.
C
C     This routine trims data that precede the input CK start time
C     or exceed the input CK stop time. Trimming is done so that
C     available padding is retained.
C
C     The degree associated with each output mini-segment is based on
C     the data type and the number of attitude records available in the
C     input block; for short blocks, the mini-segment degree will be
C     reduced as necessary.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     1)  Input data blocks are assumed to be in increasing time order
C         and to have at most singleton overlap of usable time ranges.
C
C     2)  This routine currently supports just the "quaternion only"
C         attitude representation.
C
C     3)  This routine creates only type 6 output segments.
C
C     4)  A single SCLK rate is assumed to be applicable to the 
C         entire output CK.
C  
C$ Literature_References
C
C     [1]  CCSDS Attitude Data Messages Blue Book, version CCSDS 
C          504.0-B-1, May, 2008.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    AEM2CK Version 2.0.0, 01-MAR-2017 (NJB)
C
C        Now stores in output mini-segments the average SCLK rate 
C        over each interpolation interval, based on SCLK and TDB
C        deltas over those intervals. Formerly a nominal SCLK rate,
C        provided in the setup file, was stored in the mini-segments.
C
C-    AEM2CK Version 1.0.0, 10-SEP-2015 (NJB)
C
C-&

C$ Revisions
C
C     None.
C
C-&


C
C     SPICELIB functions
C      

      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C

C
C     Maximum number of records held by the data buffers:
C
C     Use 35 for testing, 100K for operations.
C
      INTEGER               MAXN
      PARAMETER           ( MAXN   = 100000 )

C
C     Maximum number of interpolation intervals/mini-segments.
C
      INTEGER               MAXIVL
      PARAMETER           ( MAXIVL = MAXN )

C
C     Maximum number of records outside the CK time range
C     that can be buffered. This number should accommodate
C     the largest possible pad size.
C
      INTEGER               MAXOVR
      PARAMETER           ( MAXOVR = MAXDEG/2 )

C
C     Minimum mini-segment non-pad record count. 
C
C     This is the minimum number of records, not counting padding, that
C     will be included in a mini-segment whose size is set due to
C     splitting of an input data block. Data from a partial input block
C     will not be written out if the number of non-pad records is less
C     than this limit.
C
      INTEGER               MINMIN
      PARAMETER           ( MINMIN = MAXDEG + 1 )


      INTEGER               NSUBTP
      PARAMETER           ( NSUBTP = 4 )

      INTEGER               SIDLEN
      PARAMETER           ( SIDLEN = 40 )

C
C     Local variables
C
      CHARACTER*(ATRLEN)    ATTREP
      CHARACTER*(FRNMLN)    FRAMEA
      CHARACTER*(FRNMLN)    FRAMEB
      CHARACTER*(LNSIZE)    LINE
      CHARACTER*(FRNMLN)    PRVREF
      CHARACTER*(MTHLEN)    METHOD
      CHARACTER*(FRNMLN)    REF
      CHARACTER*(SIDLEN)    SEGID
      CHARACTER*(TSYLEN)    TIMSYS
      CHARACTER*(TYPLEN)    TYPE

      DOUBLE PRECISION      DET
      DOUBLE PRECISION      DSCLK
      DOUBLE PRECISION      ET
      DOUBLE PRECISION      ET0
      DOUBLE PRECISION      ET1
      DOUBLE PRECISION      FIRST
      DOUBLE PRECISION      IVLBDS ( MAXIVL + 1 )
      DOUBLE PRECISION      IVLBEG
      DOUBLE PRECISION      IVLEND
      DOUBLE PRECISION      LAST
      DOUBLE PRECISION      RATES  ( MAXIVL )
      DOUBLE PRECISION      PACKET ( C06MXZ )
      DOUBLE PRECISION      PACKTS ( 4 * MAXN )
      DOUBLE PRECISION      PRVEND
      DOUBLE PRECISION      SCLKDP ( MAXN )
      DOUBLE PRECISION      TBEG
      DOUBLE PRECISION      TEND
      DOUBLE PRECISION      UBEG
      DOUBLE PRECISION      UEND

      INTEGER               BEGPAD
      INTEGER               DEGREE
      INTEGER               DEGRES ( MAXIVL )
      INTEGER               ENDPAD
      INTEGER               I
      INTEGER               INSTID
      INTEGER               J
      INTEGER               K
      INTEGER               LNUM
      INTEGER               LSTIDX
      INTEGER               MINSIZ
      INTEGER               NMINI
      INTEGER               NPKTS  ( MAXIVL )
      INTEGER               NREC
      INTEGER               NSMINI
      INTEGER               OBJECT
      INTEGER               PADSIZ
      INTEGER               PKTPTR
      INTEGER               PKTSIZ
      INTEGER               PRVINS
      INTEGER               PSIZES ( 0 : NSUBTP-1 )
      INTEGER               RETAIN
      INTEGER               SCLKID
      INTEGER               SEGNO 
      INTEGER               SUBTPS ( MAXIVL )
      INTEGER               W

      LOGICAL               A2B
      LOGICAL               AVFLAG
      LOGICAL               CNSIST
      LOGICAL               DONE
      LOGICAL               FOUND
      LOGICAL               GAPOK
      LOGICAL               INDATA
      LOGICAL               QLAST
      LOGICAL               PRVPAD
      
C
C     Saved variables
C
C     Save everything to avoid stack problems on some platforms.
C
      SAVE

C
C     Initial values
C
      DATA                  PSIZES  /  C06PS0, C06PS1, C06PS2, C06PS3 /


      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'MKCK06' )


C
C     Note to programmers: in this routine, the variables
C
C        CKBEG
C        CKEND
C        UBEG
C        UEND
C        TBEG
C        TEND
C
C     all contain TDB times. The variables
C
C        FIRST
C        LAST
C        IVLBDS
C        IVLBEG
C        IVLEND
C
C     all contain encoded SCLK times.
C


C
C     Use the first SIDLEN characters of the input file name as the
C     segment ID.
C
      SEGNO  = 0
      SEGID  = INPUT(:SIDLEN)
      
      PRVINS = 0
      PRVREF = ' '


C
C     Locate the first metadata block in the input AEM.
C
      CALL RDLIN ( INPUT, LINE, LNUM, TYPE, DONE ) 

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'MKCK06' )
         RETURN
      END IF

      DO WHILE ( .NOT. DONE )

         IF ( TYPE .EQ. METBEG ) THEN
C
C           Parse the following metadata block.
C
            CALL PRSMET ( INPUT,  OBJECT, FRAMEA, FRAMEB, 
     .                    TBEG,   TEND,   UBEG,   UEND,
     .                    TIMSYS, ATTREP, QLAST,  A2B,   
     .                    METHOD, DEGREE                 )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'MKCK06' )
               RETURN
            END IF

            IF ( UEND .GT. CKBEG ) THEN
C
C              We've found a block having coverage that exceeds the CK
C              start time. We'll check after loop exit that the block
C              coverage intersects the input CK time bounds.
C
               DONE = .TRUE.

            END IF

         END IF


         IF ( .NOT. DONE ) THEN
C
C           Read the next input line.
C
            CALL RDLIN ( INPUT, LINE, LNUM, TYPE, DONE ) 

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'MKCK06' )
               RETURN
            END IF

         END IF

      END DO

C
C     If we didn't find a data block in range, that's an error.
C
      IF (  ( UEND .LE. CKBEG ) .OR. ( UBEG .GT. CKEND )  ) THEN

         CALL SETMSG ( 'No coverage for the requested CK time range '
     .   //            'was found. CK time bounds are # TDB : # TDB; '
     .   //            'usable bounds of first input block having '
     .   //            'an end time greater than CKBEG are # TDB : '
     .   //            '# TDB.'                                        )
         CALL ERRDP  ( '#', CKBEG                                      )
         CALL ERRDP  ( '#', CKEND                                      )
         CALL ERRDP  ( '#', UBEG                                       )
         CALL ERRDP  ( '#', UEND                                       )
         CALL SIGERR ( 'SPICE(NOCOVERAGE)'                             )
         CALL CHKOUT ( 'MKCK06'                                        )
         RETURN

      END IF

C
C     Get the name of the base frame and the class ID of the
C     instrument frame.
C
      CALL MAPFRM ( A2B, FRAMEA, FRAMEB, REF, INSTID )
 
C
C     Map the CK frame class ID to a SCLK ID.
C
      CALL CKMETA ( INSTID, 'SCLK', SCLKID )
  
      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'MKCK06' )
         RETURN
      END IF
 
      IF ( METHOD .EQ. ' ' ) THEN
C
C        No method was provided in the meta-data block. Use
C        the default method AND degree.
C
C        We consider the degree in a meta-data block to be
C        unreliable unless an interpolation method is provided.
C
         METHOD = DEFMTH
         DEGREE = DEFDEG

      END IF

C
C     Adjust the block bounds using the CK time bounds.
C
      TBEG = MAX( TBEG, CKBEG )
      TEND = MIN( TEND, CKEND )

      UBEG = MAX( UBEG, TBEG  )
      UEND = MIN( UEND, TEND  )
      
C
C     Set parameters for the first segment. The segment coverage bounds
C     FIRST and LAST are expressed in ticks. The value of LAST will be
C     set just before writing the segment.
C
      AVFLAG = .TRUE.
      SEGNO  = 1

      CALL SCE2C ( SCLKID, UBEG, FIRST )
      
C
C     Set parameters for the first mini-segment.
C
      NMINI     = 1

      CALL SCE2C ( SCLKID, UBEG, IVLBEG )
      CALL SCE2C ( SCLKID, UEND, IVLEND )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'MKCK06' )
         RETURN
      END IF

C
C     In the set of assignments below
C
C        - The minimum mini-segment size includes block-dependent
C          padding; that's why it's set for each mini-segment.
C
C        - The type 6 subtype is a function of the attitude 
C          representation. This code must be updated to generalize
C          the set of supported representations.
C     
      IVLBDS(1) = IVLBEG
      IVLBDS(2) = IVLEND
      NPKTS (1) = 0
      SUBTPS(1) = C06TP1
      PKTSIZ    = PSIZES ( SUBTPS(1) )
C
C     Note: RATES(1) is no longer assigned its final value here; it's
C     assigned before calling CKW06. The assignment here is made 
C     just to suppress compiler warnings.
C
      RATES(1)  = RATE
      DEGRES(1) = DEGREE
      PADSIZ    = DEGREE / 2
      MINSIZ    = MAX ( 2*PADSIZ + 2,  MINMIN )
      NREC      = 0

C
C     Check our parameter declarations for compatibility.
C
C        - The maximum record count must be greater than MINSIZ.
C
      IF ( MINSIZ .GE. MAXN ) THEN

         CALL SETMSG ( 'Assertion failed: maximum record count MAXN, '
     .   //            'which is #, must be larger than the minimum ' 
     .   //            'size MINSIZ of mini-segment large enough '
     .   //            'to be broken up, which is #.' 
     .   //            'This is a bug.'                               )
         CALL ERRINT ( '#', MAXN                                      )
         CALL ERRINT ( '#', MINSIZ                                    )
         CALL SIGERR ( 'SPICE(BUG)'                                   )
         CALL CHKOUT ( 'MKCK06' )
         RETURN
         
      END IF
      
C
C     Get the next line.
C
      CALL RDLIN ( INPUT, LINE, LNUM, TYPE, DONE ) 

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'MKCK06' )
         RETURN
      END IF

C
C     Initialize pad counts.
C
      BEGPAD = 0
      ENDPAD = 0

C
C     Set segment and mini-segment parameters.
C
C
C     Main loop: process inputs while we have data and the 
C     current block starts before the CK end time.
C
      DONE   = .FALSE.
      INDATA = .FALSE.


      DO WHILE (  ( .NOT. DONE ) .AND. ( UBEG .LT. CKEND )  )

         IF ( TYPE .EQ. DATBEG ) THEN

            INDATA = .TRUE.

         ELSE IF ( TYPE .EQ. DATEND ) THEN

            INDATA = .FALSE.

         ELSE IF ( TYPE .EQ. DATLIN ) THEN
C
C           Process data line.
C
C           At this point, ET and PACKET contain the latest
C           values read from the input data block.
C
C           Update the count of buffered records (packets).
C   
            NREC = NREC + 1

C
C           Update the current mini-segment packet count.
C
            NPKTS( NMINI ) = NPKTS( NMINI ) + 1

C
C           If the current ET epoch precedes the CK start time,
C           increment the initial pad count.
C
            IF ( ET .LT. CKBEG ) THEN

               BEGPAD = BEGPAD + 1

            END IF
C
C           If the current ET epoch exceeds the CK stop time,
C           increment the final pad count.
C
            IF ( ET .GT. CKEND ) THEN
               ENDPAD = ENDPAD + 1
            END IF

C
C           By design, there's room in the buffer.
C
C              - Convert attitude data packet to SPICE CK type 6
C                packet.
C                              
C              - Convert time tag to SCLK
C
C              - Buffer attitude and epoch
C
            PKTPTR = ( ( NREC - 1 ) * PKTSIZ )  +  1

            CALL PKTRAN ( SUBTPS(NMINI), PACKET, PACKTS(PKTPTR) )

            CALL SCE2C ( SCLKID, ET,     SCLKDP(NREC)   )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'MKCK06' )
               RETURN
            END IF

C
C           If the buffer is full, see if we can clean it up by
C           deleting data that precede the CK start time (taking
C           care to preserve any padding).
C           
            IF ( NREC .EQ. MAXN ) THEN

               IF ( BEGPAD .GT. MAXOVR ) THEN
C
C                 Shift the buffered data so that we retain only the
C                 entries having time tags greater than or equal to
C                 CKBEG, plus any initial padding. PADSIZ has been
C                 derived from the last metadata block we encountered.
C
C                 Let J be the number of entries we can delete. 
C
                  J      = MAX( 0, BEGPAD - PADSIZ )
                  RETAIN = NREC - J

                  DO I = 1, RETAIN
C
C                    Shift the current time tag by J.
C
                     SCLKDP(I) = SCLKDP(J+I)
                     
C
C                    Shift the current data block by J*PKTSIZ.
C
                     K         = (  I-1)*PKTSIZ + 1
                     W         = (J+I-1)*PKTSIZ + 1

                     CALL MOVED ( PACKTS(W), PKTSIZ, PACKTS(K) )

                  END DO 

C
C                 Adjust our record counts. Note that we must be
C                 processing the first data block that overlaps the
C                 specified coverage interval, so we're collecting data
C                 for the first mini-segment.
C
                  NREC     = RETAIN
                  NPKTS(1) = RETAIN

C
C                 We won't revisit this branch of the code.
C                 
                  BEGPAD = 0

               END IF

            END IF

C
C           Cleaning up the buffers by removing data having time tags
C           preceding CKBEG only works once per program run (at most).
C           After that, if the buffers are full, we'll have to write a
C           segment.
C
            IF ( NREC .EQ. MAXN ) THEN
C
C              We just filled up the buffers. We'll write a segment.
C
C              We must have enough data in the current mini-segment
C              so that we can break the mini-segment up into multiple
C              parts. If the mini-segment is too small, we'll exclude
C              it from the output segment and keep it buffered until
C              we write the next segment.
C               
               IF ( NPKTS(NMINI) .LT. 2*MINSIZ ) THEN
C
C                 We don't have enough packets in the current
C                 mini-segment to break it up and guarantee that
C                 both parts have at least MINSIZ packets.
C
C                 We'll end the segment at the end of the previous
C                 mini-segment. Note that a previous mini-segment must
C                 exist, since the total number of packets preceding
C                 the current mini-segment is at least MAXN-MINSIZ.
C
C                 We'll retain all buffered data from the current
C                 mini-segment (indicated by NPKTS(MINI)); these data
C                 will be shifted to the top of the epoch and packet
C                 buffers.
C
C                 The segment end time will be the interval stop time
C                 of the previous mini-segment.
C
C                 Let NSMINI be the number of mini-segments in the 
C                 segment to be written.
C
                  RETAIN = NPKTS ( NMINI ) 
                  LAST   = IVLBDS( NMINI )
                  NSMINI = NMINI - 1


               ELSE
C
C                 The current mini-segment is large enough so that we
C                 can break it up. We'll write out at least MINSIZ
C                 records of the mini-segment; the rest will become
C                 part of the first mini-segment of the next segment.
C                 The part we write will contain a right hand pad.
C
C                 We'll retain MINSIZ records of the current
C                 mini-segment to ensure that the next mini-segment has
C                 valid size, and that the next mini-segment starts
C                 with a left hand pad.
C
C                 Let LSTIDX be the record index of the last non-pad
C                 record of the last mini-segment of the segment we're
C                 about to write. The record at this index will be the
C                 first non-pad record of the first mini-segment of the
C                 next segment.
C
C                 The last mini-segment will contain PADSIZ records
C                 past index LSTIDX. The next mini-segment will contain
C                 PADSIZ records preceding LSTIDX.
C
C                 Below is a graphical representation of the buffer 
C                 subsets that are involved. Sizes are not to scale.
C
C
C              |<-----------------------NREC-------------------------->|
C
C                               |<------------NPKTS(NMINI)------------>|
C
C                               |<--.GE. MINSIZ--->||<----MINSIZ------>|
C
C                                                   |<PAD>| |<PAD>|
C                                                          |
C                                                        LSTIDX
C
C                               |<----NPKTS(NMINI) (written)----->|
C
C                                                   |<-----RETAIN----->|
C
C

                  
                  RETAIN          = MINSIZ       
                  LSTIDX          = NREC         - MINSIZ + 1 +   PADSIZ
                  NPKTS(NMINI)    = NPKTS(NMINI) - MINSIZ + 1 + 2*PADSIZ
                  NSMINI          = NMINI

                  LAST            = MIN( IVLEND, SCLKDP(LSTIDX) )
                  IVLBDS(NMINI+1) = SCLKDP(LSTIDX) 

               END IF

C
C              We'll write the segment only if LAST > FIRST. This
C              normally will be true, but could be false if the data
C              buffers are set to a very small size (as could be done
C              for testing).
C
               IF ( LAST .GT. FIRST ) THEN
C
C                 Check time tag spacing.
C               
                  CALL CHKTAG ( SCLKID, SPCLIM, QUOLIM, 
     .                          NMINI,  NPKTS,  SCLKDP )

C
C                 Compute SCLK rates for the interpolation intervals.
C
                  DO I = 1, NSMINI

                     DSCLK = IVLBDS(I+1) - IVLBDS(I)

                     IF ( DSCLK .EQ. 0.D0 ) THEN

                        CALL SETMSG ( 'Mini-segment interval # start '
     .                  //            'time (ticks) is #; equals ' 
     .                  //            'interval end time.'            )
                        CALL ERRINT ( '#', I                          )
                        CALL ERRDP  ( '#', IVLBDS(I)                  )
                        CALL SIGERR ( 'SPICE(BUG)'                    )
                        CALL CHKOUT ( 'MKCK06'                        )
                        RETURN

                     END IF

                     CALL SCT2E ( SCLKID, IVLBDS(I  ), ET0 )
                     CALL SCT2E ( SCLKID, IVLBDS(I+1), ET1 )

                     IF ( FAILED() ) THEN
                        CALL CHKOUT ( 'MKCK06' )
                        RETURN
                     END IF

                     DET = ET1 - ET0
C
C                    Rates have units of seconds/tick.
C                     
                     RATES(I) = DET / DSCLK

                  END DO

C
C                 Write segment.
C
                  CALL CKW06 ( HANDLE,  INSTID,  REF,     AVFLAG,
     .                         FIRST,   LAST,    SEGID,   NSMINI,
     .                         NPKTS,   SUBTPS,  DEGRES,  PACKTS, 
     .                         RATES,   SCLKDP,  IVLBDS,  SELLST )

                  IF ( FAILED() ) THEN
                     CALL CHKOUT ( 'MKCK06' )
                     RETURN
                  END IF

               END IF
               
C
C              Save segment parameters.
C
               PRVREF = REF
               PRVINS = INSTID

C
C              The data buffers are always non-empty at this point.
C
C              Prepare the buffers for continued processing. We're
C              working on the first mini-segment of a new segment.
C
C              We need to reset the interpolation interval bounds. The
C              new interval corresponds to the record at index LSTIDX;
C              this is also the interval end time of the portion of the
C              mini-segment we just wrote. We overwrote the interval
C              end time in the array IVLBDS; restore this time from the
C              value parsed from the current input metadata block.
C
               FIRST     = LAST
               IVLBDS(1) = FIRST
               IVLBDS(2) = IVLEND

C
C              Shift the buffers so the first retained values are in
C              the respective first buffer elements.
C                 
               J = NREC - RETAIN 

               DO I = 1, RETAIN

                  SCLKDP(I) = SCLKDP(J+I)

                  K         = (  I-1)*PKTSIZ + 1
                  W         = (J+I-1)*PKTSIZ + 1

                  CALL MOVED ( PACKTS(W), PKTSIZ, PACKTS(K) )

               END DO

               NREC      = RETAIN
               NPKTS(1)  = RETAIN
               DEGRES(1) = DEGRES(NMINI)
               SUBTPS(1) = SUBTPS(NMINI)
C
C              Note: RATE(1) is no longer assigned here; it's
C              assigned before calling CKW06.
C
               NMINI     = 1
               SEGNO     = SEGNO + 1

            END IF


         ELSE IF ( TYPE .EQ. METBEG ) THEN
C
C           The next line is the start of a meta-data block.
C
C           The data buffers are always non-empty at this point, and
C           they contain at least part of the data of the previous
C           mini-segment.
C
C           Before processing the metadata block, save some information
C           about the data block that was just read. We need to know if
C           the block ended with padding. This is equivalent to the
C           last epoch exceeding the usable stop time.
C
            PRVEND = IVLBDS(NMINI+1)
            PRVPAD = SCLKDP(NREC)  .GT.  PRVEND

C
C           Update the previous frame and instrument ID as well.
C
            PRVREF = REF
            PRVINS = INSTID
C
C           Parse the following metadata block.
C
            CALL PRSMET ( INPUT,  OBJECT, FRAMEA, FRAMEB, 
     .                    TBEG,   TEND,   UBEG,   UEND,
     .                    TIMSYS, ATTREP, QLAST,  A2B,   
     .                    METHOD, DEGREE                 )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'MKCK06' )
               RETURN
            END IF

C
C           Get the name of the base frame and the class ID of the
C           instrument frame.
C 
            CALL MAPFRM ( A2B, FRAMEA, FRAMEB, REF, INSTID )
            
            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'MKCK06' )
               RETURN
            END IF
  
            IF ( METHOD .EQ. ' ' ) THEN
C
C              No method was provided in the meta-data block. Use
C              the default method AND degree.
C
               METHOD = DEFMTH
               DEGREE = DEFDEG

            END IF

C
C           Adjust the block bounds using the CK time bounds.
C
            TBEG = MAX( TBEG, CKBEG )
            TEND = MIN( TEND, CKEND )

            UBEG = MAX( UBEG, TBEG  )
            UEND = MIN( UEND, TEND  )

            CALL SCE2C ( SCLKID, UBEG, IVLBEG )
            CALL SCE2C ( SCLKID, UEND, IVLEND )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'MKCK06' )
               RETURN
            END IF

C
C           Decide whether to start a new mini-segment in the current
C           segment or to start a new segment.
C
C           Gaps are allowed between the last record of one mini-segment
C           and the first interval start time of the next. However, if
C           the previous input block has right hand side padding and 
C           has a usable stop time preceding the usable start time of
C           the current block, a new segment is needed. 
C
C           If the current block has a different base or instrument
C           frames, a new segment is needed.
C
            CNSIST = ( REF .EQ. PRVREF ) .AND. ( INSTID .EQ. PRVINS )

C
C           The flag GAPOK is true if there is no gap or if there's
C           a gap not preceded by padding.
C
            GAPOK  =   ( IVLBEG .EQ. PRVEND ) .OR.  
     .              (  ( IVLBEG .GT. PRVEND ) .AND. ( .NOT. PRVPAD )  )


            IF ( CNSIST .AND. GAPOK ) THEN
C
C              We can handle the current block by starting a new
C              mini-segment within the current segment.
C
C              We'll overwrite the previous interval end time with
C              the start time of this interval. If there's no gap,
C              this is a no-op; otherwise this places the gap at
C              the end of the previous interpolation interval.
C
               NMINI           = NMINI + 1
               IVLBDS(NMINI  ) = IVLBEG
               IVLBDS(NMINI+1) = IVLEND

C
C              Set the other mini-segment parameters.
C
               NPKTS (NMINI  ) = 0
               SUBTPS(NMINI  ) = C06TP1
               DEGRES(NMINI  ) = DEGREE
C
C              Note: RATES(NMINI) is no longer assigned here; it's
C              assigned before calling CKW06.
C
            ELSE
C
C              We'll start a new mini-segment in a new segment.
C              Before that, we'll write the buffered data to 
C              a segment.
C
               LAST = IVLBDS( NMINI+1 )

C
C              Note that in this case we're writing data collected
C              before the latest metadata block was encountered, so
C              we use the previous values of INSTID and REF.
C
               IF ( LAST .GT. FIRST ) THEN
C
C                 Check time tag spacing.
C               
                  CALL CHKTAG ( SCLKID, SPCLIM, QUOLIM, 
     .                          NMINI,  NPKTS,  SCLKDP )

C
C                 Compute SCLK rates for the interpolation intervals.
C
                  DO I = 1, NMINI

                     DSCLK = IVLBDS(I+1) - IVLBDS(I)

                     IF ( DSCLK .EQ. 0.D0 ) THEN

                        CALL SETMSG ( 'Mini-segment interval # start '
     .                  //            'time (ticks) is #; equals ' 
     .                  //            'interval end time.'            )
                        CALL ERRINT ( '#', I                          )
                        CALL ERRDP  ( '#', IVLBDS(I)                  )
                        CALL SIGERR ( 'SPICE(BUG)'                    )
                        CALL CHKOUT ( 'MKCK06'                        )
                        RETURN

                     END IF

                     CALL SCT2E ( SCLKID, IVLBDS(I  ), ET0 )
                     CALL SCT2E ( SCLKID, IVLBDS(I+1), ET1 )

                     IF ( FAILED() ) THEN
                        CALL CHKOUT ( 'MKCK06' )
                        RETURN
                     END IF

                     DET = ET1 - ET0
C
C                    Rates have units of seconds/tick.
C                     
                     RATES(I) = DET / DSCLK

                  END DO


                  CALL CKW06 ( HANDLE,  PRVINS,  PRVREF,  AVFLAG,
     .                         FIRST,   LAST,    SEGID,   NMINI,
     .                         NPKTS,   SUBTPS,  DEGRES,  PACKTS, 
     .                         RATES,   SCLKDP,  IVLBDS,  SELLST )

                  IF ( FAILED() ) THEN
                     CALL CHKOUT ( 'MKCK06' )
                     RETURN
                  END IF
           
               END IF
    
C
C              Save segment parameters.
C
               PRVREF = REF
               PRVINS = INSTID

C
C              We empty the buffers.
C
               NREC      = 0
C
C              Get ready to start a new segment. 
C
               SEGNO     = SEGNO + 1
C
C              This is the first mini-segment of the current segment.
C
               NMINI     = 1
               NPKTS(1)  = 0
               IVLBDS(1) = IVLBEG
               IVLBDS(2) = IVLEND
               SUBTPS(1) = C06TP1
               PKTSIZ    = PSIZES ( SUBTPS(1) )
               DEGRES(1) = DEGREE
C
C              Note: RATES(1) is no longer assigned here; it's
C              assigned before calling CKW06.
C
               FIRST     = IVLBEG
             
            END IF

         ELSE  

            CALL SETMSG ( 'Unrecognized line type: #. This '
     .      //            'indicates a bug in AEM2CK.'       )
            CALL ERRCH  ( '#',  TYPE                         )
            CALL SIGERR ( 'SPICE(BUG)'                       )
            CALL CHKOUT ( 'MKCK06'                           )
            RETURN
                        
         END IF

C
C        Decide how to read the next record.
C        
         IF ( ENDPAD .GT. MAXOVR ) THEN

            DONE = .TRUE.

         ELSE IF ( INDATA ) THEN
C
C           Read a data record. Note that a record may span
C           multiple lines.
C
            CALL PRSDR ( INPUT, TIMSYS, ATTREP, ET, PACKET, FOUND )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'MKCK06' )
               RETURN
            END IF

            IF ( FOUND ) THEN

               TYPE = DATLIN

            ELSE
C
C              PRSDR will have pushed back the line it just read.
C              Fetch this line.
C
               CALL RDLIN ( INPUT, LINE, LNUM, TYPE, DONE ) 

               IF ( FAILED() ) THEN
                  CALL CHKOUT ( 'MKCK06' )
                  RETURN
               END IF

            END IF

         ELSE
 
C           Read a line.
C         
            CALL RDLIN ( INPUT, LINE, LNUM, TYPE, DONE ) 

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'MKCK06' )
               RETURN
            END IF

         END IF


      END DO
C
C     We're done reading the input AEM file.
C

      IF ( NREC .GT. 0 ) THEN
C
C        Set the segment end time and end time of 
C        the last mini-segment.
C
         IVLBDS(NMINI+1) = IVLEND
         LAST            = IVLEND

         IF ( LAST .GT. FIRST ) THEN
C
C           The buffered data are ready to be written.
C
C           Check time tag spacing.
C               
            CALL CHKTAG ( SCLKID, SPCLIM, QUOLIM, 
     .                    NMINI,  NPKTS,  SCLKDP )
C
C           Compute SCLK rates for the interpolation intervals.
C
            DO I = 1, NMINI

               DSCLK = IVLBDS(I+1) - IVLBDS(I)

               IF ( DSCLK .EQ. 0.D0 ) THEN

                  CALL SETMSG ( 'Mini-segment interval # start '
     .            //            'time (ticks) is #; equals ' 
     .            //            'interval end time.'            )
                  CALL ERRINT ( '#', I                          )
                  CALL ERRDP  ( '#', IVLBDS(I)                  )
                  CALL SIGERR ( 'SPICE(BUG)'                    )
                  CALL CHKOUT ( 'MKCK06'                        )
                  RETURN

               END IF

               CALL SCT2E ( SCLKID, IVLBDS(I  ), ET0 )
               CALL SCT2E ( SCLKID, IVLBDS(I+1), ET1 )

               IF ( FAILED() ) THEN
                  CALL CHKOUT ( 'MKCK06' )
                  RETURN
               END IF

               DET = ET1 - ET0
C
C              Rates have units of seconds/tick.
C                     
               RATES(I) = DET / DSCLK

            END DO

            CALL CKW06 ( HANDLE,  INSTID,  REF,     AVFLAG,
     .                   FIRST,   LAST,    SEGID,   NMINI,
     .                   NPKTS,   SUBTPS,  DEGRES,  PACKTS, 
     .                   RATES,   SCLKDP,  IVLBDS,  SELLST )
         END IF

      END IF

      CALL CHKOUT ( 'MKCK06' )
      RETURN 
      END


