C$Procedure MAPFRM ( Map frames )
 
      SUBROUTINE MAPFRM ( A2B, FRAMEA, FRAMEB, REF, INSTID )
 
C$ Abstract
C
C     Convert frame names and mapping direction in an AEM metadata
C     block to SPICE base frame name and a CK frame ID.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     AEM2CK
C
C$ Declarations
 
      IMPLICIT NONE

      LOGICAL               A2B
      CHARACTER*(*)         FRAMEA
      CHARACTER*(*)         FRAMEB
      CHARACTER*(*)         REF
      INTEGER               INSTID
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     A2B        I   Frame transformation direction.
C     FRAMEA     I   Name of first frame.
C     FRAMEB     I   Name of second frame.
C     REF        O   Name of base frame.
C     INSTID     O   Class ID of CK frame.
C
C$ Detailed_Input
C              
C     A2B            is a logical flag that is .TRUE. if and only if
C                    attitude data associated with a metadata block
C                    describes a transformation from frame A to frame
C                    B. Otherwise the transformation is from frame
C                    B to frame A.
C                   
C     FRAMEA,
C     FRAMEB         are the names of two reference frames whose
C                    relative orientation is described by an AEM data
C                    block.
C
C$ Detailed_Output
C
C     REF            is the SPICE name of the base reference frame.
C                    This is the "from" frame of the transformation.
C
C     INSTID         is the SPICE CK frame class ID (aka "instrument
C                    ID") of the "to" frame of the transformation.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C 
C     1)  If the NAIF frame ID of either input frame cannot be
C         found, the error SPICE(FRAMEIDNOTFOUND) is signaled.
C
C     2)  If the frame class ID of the "to" frame identified by the
C         input flag A2B cannot be found, the error
C         SPICE(FRAMEINFONOTFOUND) is signaled.
C
C$ Files
C
C     The inputs are normally parsed from an AEM metadata block.
C     See [1] for details.
C
C$ Particulars
C
C     This routine interprets the assignments of an AEM metadata block
C     that describe a frame transformation. The AEM assignments specify
C     two frame names and a transformation direction. These inputs
C     are mapped to a CK base frame name and a CK frame class ID. 
C     These results are inputs required by CK writer routines.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     [1]  CCSDS Attitude Data Messages Blue Book, version CCSDS 
C          504.0-B-1, May, 2008.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    AEM2CK Version 1.0.0, 19-AUG-2015 (NJB)
C
C-&


C
C     SPICELIB functions
C
      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C
      INTEGER               FRNMLN
      PARAMETER           ( FRNMLN = 32 )
      
C
C     Local variables
C
      CHARACTER*(FRNMLN)    INSFRM

      INTEGER               BASEID
      INTEGER               CENTER
      INTEGER               CLASS
      INTEGER               FRAMID

      LOGICAL               FOUND



      IF ( RETURN() ) THEN         
         RETURN
      END IF

      CALL CHKIN ( 'MAPFRM' )

      IF ( A2B ) THEN
C
C        The base frame is "A."
C
         REF    = FRAMEA
         INSFRM = FRAMEB
      ELSE
         REF    = FRAMEB
         INSFRM = FRAMEA
      END IF


C
C     Get the frame ID of the base frame. This is done
C     solely as an error check; we won't use the ID.
C
      CALL NAMFRM ( REF, BASEID )

      IF ( BASEID .EQ. 0 ) THEN

         CALL SETMSG ( 'Could not find ID code of frame #.' )
         CALL ERRCH  ( '#', REF                             )
         CALL SIGERR ( 'SPICE(FRAMEIDNOTFOUND)'             )
         CALL CHKOUT ( 'MAPFRM'                             )
         RETURN

      END IF

C
C     Get the frame ID of the CK ("instrument") frame.
C
      CALL NAMFRM ( INSFRM, FRAMID )

      IF ( FRAMID .EQ. 0 ) THEN

         CALL SETMSG ( 'Could not find ID code of frame #.' )
         CALL ERRCH  ( '#', INSFRM                          )
         CALL SIGERR ( 'SPICE(FRAMEIDNOTFOUND)'             )
         CALL CHKOUT ( 'MAPFRM'                             )
         RETURN

      END IF

C
C     Get the frame class ID of the CK frame.
C
      CALL FRINFO ( FRAMID, CENTER, CLASS, INSTID, FOUND )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'MAPFRM' )
         RETURN
      END IF

      IF ( .NOT. FOUND ) THEN

         CALL SETMSG ( 'Could not frame info for frame #.' )
         CALL ERRCH  ( '#', INSFRM                         )
         CALL SIGERR ( 'SPICE(FRAMEINFONOTFOUND)'          )
         CALL CHKOUT ( 'MAPFRM'                            )
         RETURN

      END IF

      CALL CHKOUT ( 'MAPFRM' )
      RETURN
      END
