from adcsng.utils.files import fill_template

import datetime
import logging

from collections import OrderedDict

def get_rm_site(setup_file, kernel):


    sites = []
    with open(kernel.source, 'r') as f:
        for line in f:
            if 'RM_SITE' in line:
                sites.append(line.split('RM_SITE_')[-1].strip())

    sites_string = ''
    for site in sites:
        add_to_string = f"'RM SITE {int(site):03}','RM_SITE_{int(site):03}',\n"
        if add_to_string not in sites_string:
            sites_string += add_to_string

    #
    # 'RM SITE 001','RM_SITE_001',\n
    #
    replacements = {'site_numbers': sites_string}
    fill_template(setup_file, setup_file, replacements)

    return


def oem_extension(oem_list, extension):


    #
    # we are extending all the mechanisms, hence we need to retrieve an
    # extension segment for each.
    #
    object_list = []
    for line in oem_list:
        if 'OBJECT_NAME' in line:
            object_list.append(line.split('=')[-1].strip())
    list(OrderedDict.fromkeys(object_list))


    #
    # This kernel will include the gnc and mech position extrapolated with a 0
    # order interpolation for 7 sols beyond the latest telemetry received
    # to allow for rover-orbiter link budget calculation.
    #
    oem_list_extension = []
    for object in object_list:

        (object_id, ref_frame, time_system) = '', '', ''
        (times, sites, centers, data) = [], [], [], []
        (data_bool, general_bool) =  False, False

        for line in oem_list:
            if 'META_START' in line or 'CCSDS_OEM_VERS' in line:
                data_bool = False
                general_bool = False
            if data_bool:
                if line.strip() != '' or not line.strip().isprintable():
                    data.append(line)
            if 'OBJECT_NAME' in line:
                if object in line:
                    general_bool = True
            if 'OBJECT_ID' in line and general_bool:
                object_id = line.split('=')[-1].strip()
            if 'REF_FRAME' in line and general_bool:
                ref_frame = line.split('=')[-1].strip()
            if 'TIME_SYSTEM' in line and general_bool:
                time_system = line.split('=')[-1].strip()
            if general_bool and 'STOP_TIME' in line:
                times.append(line.split('=')[-1].strip())
            if general_bool and 'CENTER_NAME' in line:
                if object == 'RM_ROVER':
                    sites.append(line.split('RM_SITE_')[-1].strip())
                else:
                    centers.append(line.split('=')[-1].strip())
            if general_bool and 'META_STOP' in line:
                data_bool = True

        if sites:
            site = sites[-1]
            center_name = f'RM_SITE_{site}'
        else:
            center = centers[-1]
            center_name = center

        data_line = data[-1]
        start_time = data_line.split()[0].strip() #times[-1]
        if '.' in start_time:
            start_date = datetime.datetime.strptime(start_time, "%Y-%m-%dT%H:%M:%S.%f")
        else:
            start_date = datetime.datetime.strptime(start_time, "%Y-%m-%dT%H:%M:%S")
        start_time = start_date.strftime("%Y-%m-%dT%H:%M:%S.%f")

        #
        # In average 1 Mars SOL is equivalent to: 1.02749125 Earth Days
        #
        stop_date = datetime.datetime.strptime(start_time,"%Y-%m-%dT%H:%M:%S.%f") + extension
        stop_time = stop_date.strftime("%Y-%m-%dT%H:%M:%S.%f")

        data = data_line.split()
        (x, y, z, vx, vy, vz) = data[1], data[2], data[3], data[4], data[5], data[6]


        oem_list_extension.append('\n'                                        )
        oem_list_extension.append('META_START\n'                              )
        oem_list_extension.append(f'OBJECT_NAME          = {object}\n'        )
        oem_list_extension.append(f'OBJECT_ID            = {object_id}\n'     )
        oem_list_extension.append(f'CENTER_NAME          = {center_name}\n'   )
        oem_list_extension.append(f'REF_FRAME            = {ref_frame}\n'     )
        oem_list_extension.append(f'TIME_SYSTEM          = {time_system}\n'   )
        oem_list_extension.append(f'START_TIME           = {start_time}\n'    )
        oem_list_extension.append(f'STOP_TIME            = {stop_time}\n'     )
        oem_list_extension.append( 'INTERPOLATION        = LAGRANGE\n'        )
        oem_list_extension.append( 'INTERPOLATION_DEGREE = 1\n'               )
        oem_list_extension.append( 'META_STOP\n'                              )
        oem_list_extension.append( '\n'                                       )
        oem_list_extension.append(f'{start_time} {x} {y} {z} {vx} {vy} {vz}\n')
        oem_list_extension.append(f'{stop_time} {x} {y} {z} {vx} {vy} {vz}'   )

    return oem_list_extension


def quat_extension(quat_list, extension):

    #
    # This kernel will include the gnc position extrapolated with a zero
    # order interpolation for 7 sols beyond the latest telemetry received
    # to allow for rover-orbiter link budget calculation.
    #
    last_quat = quat_list[-1]
    last_time = last_quat.split()[1].strip()
    data = [last_quat.split()[2].strip(), last_quat.split()[3].strip(),
            last_quat.split()[4].strip(), last_quat.split()[5].strip()]
    #
    # In average 1 Mars SOL is equivalent to: 1.02749125 Earth Days
    #
    try:
        stop_date = datetime.datetime.strptime(last_time.split('TDB')[0],"%Y-%m-%d-%H:%M:%S") + extension
    except:
        stop_date = datetime.datetime.strptime(last_time.split('TDB')[0], "%Y-%m-%d-%H:%M:%S.%f") + extension
    stop_time = stop_date.strftime("%Y-%m-%d-%H:%M:%S.%f")

    quat_list.append(f'{last_time} {stop_time}TDB {data[0]} {data[1]} {data[2]} {data[3]} 0.0 0.0 0.0 0.0\n')

    return quat_list


def mechanism_filter(path, sources):

    new_filenames = []
    for source in sources:
        if source[1]['input_type'] == 'AEM_QUAT' and \
                            'adcsng_temp' not in source[0]:
            #
            # We will create a dictionary inside a dictionary per OBJECT_NAME of the
            # AEM, the dictionary will have a 'ref_frame_a' key a ref_frame_b key and
            # a object_id key. Finally there will be a data key with a list of strings.
            #
            mechanisms = {}
            record_interval = False
            record_object = False

            input_file =  path + '/' + source[0]

            with open(input_file, 'r') as t:
                for line in t:
                    if 'DATA_STOP' in line:
                        record_interval = False
                        record_object = False
                    if record_interval:
                        mechanisms[object_name]['DATA'].append(line)

                    if 'OBJECT_NAME' in line:
                        object_name = line.split('=')[-1].strip().replace(' ', '_')
                        record_object = True
                        if object_name not in mechanisms.keys():
                            mechanisms[object_name] = {'object_name':object_name}
                            mechanisms[object_name]['DATA'] = []
                    if record_object and 'OBJECT_ID' in line:
                            mechanisms[object_name]['OBJECT_ID'] = line.split('=')[-1].strip()
                    if record_object and 'REF_FRAME_A' in line:
                            mechanisms[object_name]['REF_FRAME_A'] = line.split('=')[-1].strip()
                    if record_object and 'REF_FRAME_B' in line:
                            mechanisms[object_name]['REF_FRAME_B'] = line.split('=')[-1].strip()
                    if 'DATA_START' in line:
                            record_interval = True

            #
            # Now we need to generate the new inputs, one per mechanism.
            # This new inputs, because they have different structures
            # need to be specified in the exomarsrsp.json configuration
            # file.
            #
            for key in mechanisms.keys():
                new_filename = source[0].split('.')[0] + '_{sol_number}' + f'_adcsng_temp_{key.lower()}.aem'
                input_file = path + '/' + new_filename
                new_filenames.append(new_filename)

                #
                # We also provide the MSPOCK_QUAT configuration per source
                # (In the case of Solar Orbiter is not necessary because they
                # are all the same).
                #
                source[1][new_filename] = {}
                source[1][new_filename]["target_frame"]  = mechanisms[key]['REF_FRAME_A']
                source[1][new_filename]["object_id"]    = mechanisms[key]['OBJECT_ID']
                source[1][new_filename]["mission_accr"] = mechanisms[key]['REF_FRAME_B']

                with open(input_file, 'w') as t:
                    t.write('DATA_START\n')
                    for line in mechanisms[key]['DATA']:
                        t.write(line)
                    t.write('DATA_STOP')

                #
                # We need to add the new source files.
                #
                if not 'source_list' in source[1].keys():
                    source[1]['source_list'] = list()
                if not new_filename in source[1]['source_list']:
                    source[1]['source_list'].append(new_filename)

    return new_filenames


def aem_to_quat(source):

    with open(source.split('.')[0]+'_input.quat', 'w') as i:
        with open(source, 'r') as t:
            data_record = False
            for line in t:
                if 'DATA_STOP' in line:
                    data_record = False
                if data_record:
                    line_list = line.split()
                    i.write(f'{line_list[0]} {line_list[4]} {line_list[1]} {line_list[2]} {line_list[3]}\n')
                if 'DATA_START' in line:
                    data_record = True

    return source.split('.')[0]+'_input.quat'
