#!/bin/csh
#
#  This script performs a set of superficial checks on the
#  meta-information files and data files staged for a SPICE archive
#  prepared following the steps and using the tools from the SPICE 
#  Archive Preparation Guide.
#
#  This script uses 'badchar' program (source provided with SPICE
#  Archive Preparation Guide), 'bff' program (executable available from
#  Toolkit/Utilities page of the NAIF server, and 'awk' and 'grep'
#  UNIX utilities.
#
#  MCS/ESAC, Jan 3, 2017
#
#     Updated script with particulars for Rosetta. 
#
#  BVS/NAIF, Jan 8, 2009
#
#     Removed '-e' option from awk commands. Replaced 'strings' with
#     'bff' as the tool used to determine binary kernel architecture.
#     Made ./catalog/spice_hsk.cat exempt from long line check in all
#     places.
# 
#  BVS/NAIF, Jul 30, 2007
#
#     Added checks for text and binary kernels.
# 
#  BVS/NAIF, Feb 27, 2007
# 
#     Added extras to the checked lists.
#
#  LSE/NAIF, Sept 9, 2005
#
#     Made modifications so that the housekeeping file, spice_hsk.cat,
#     is exempt from line length checking.
#
#  LSE/NAIF, Sept 7, 2005
#
#     Added several *.cat files
#
#  LSE/NAIF, June 20, 2002
#
#     Made modifications, including lower case, for PDS-D. NOTE: Unlike 
#     other scripts, we had to make a (UPPER/lower) case choice.
#     This script will not work with the old PDS files.
#  
#  BVS/NAIF, April 25, 2000.
#
#     Added second pass capturing all text files (*.TXT, *.LBL, *.CAT ...)
#
#  BVS/NAIF, January 10, 2000.
#
#     Updated line length check to be for 78 char.
#
#  BVS/NAIF, January 5, 2000.
#

#
#  badchar and bff executables
#
set BADCHAR = 'badchar'
set BFF     = 'bff'

echo " "
echo "Problems are marked with (E) in at the beginning of report lines."
echo " "
#
#  Checking directories.
#
echo "   Verifying directories:"
echo " "

foreach DD ( CATALOG DATA DATA/CK DATA/EK DATA/FK DATA/IK DATA/LSK DATA/PCK DATA/SCLK DATA/SPK DOCUMENT EXTRAS EXTRAS/ORBNUM EXTRAS/MK INDEX SOFTWARE )
   
   #
   #  Does it exist?
   #
   if ( -e $DD ) then
   
      #
      #  Is it a directory?
      #
      if ( -d $DD ) then
         echo "      $DD exists and is a directory."
      else
         echo "(E)   $DD is not a directory."
      endif
      
   else
      echo "(E)   $DD doesn't exist."     
   endif

end

#
#  Checking files.
#
echo " "
echo "   Verifying files -- REQUIRED meta-info files:"
echo " "

foreach FF ( AAREADME.HTM AAREADME.LBL AAREADME.TXT CATALOG/CATINFO.TXT CATALOG/INSTHOST.CAT CATALOG/MISSION.CAT CATALOG/PERSON.CAT CATALOG/REF.CAT CATALOG/RELEASE.CAT CATALOG/SPICE_HSK.CAT CATALOG/SPICE_INST.CAT CATALOG/SPICEDS.CAT DOCUMENT/CKINFO.TXT DOCUMENT/EKINFO.TXT DOCUMENT/FKINFO.TXT DOCUMENT/IKINFO.TXT DOCUMENT/LSKINFO.TXT DOCUMENT/PCKINFO.TXT DOCUMENT/SCLKINFO.TXT DOCUMENT/SPKINFO.TXT DOCUMENT/DOCINFO.TXT DOCUMENT/LBLINFO.TXT DOCUMENT/ONLABELS.TXT EXTRAS/EXTRINFO.TXT EXTRAS/MK/MKINFO.TXT EXTRAS/ORBNUM/ORBINFO.TXT INDEX/INDEX.LBL INDEX/INDEX.TAB INDEX/INDXINFO.TXT SOFTWARE/SOFTINFO.TXT VOLDESC.CAT )

   #
   #  Does it exist?
   #
   if ( -e $FF ) then
   
      #
      #  Does it have non-zero size?
      #
      if ( ! ( -z $FF ) ) then
      
         #
         #  Does it have any bad characters?
         #
         $BADCHAR -s $FF
         
         if ( $status != 1 ) then
            
            #
            #  Does it have lines longer than 78 characters?
            #
            awk '{if(length>78){print $0}}' < $FF | grep '.' > /dev/null
            
            if ( ( $status != 0 ) || $FF == "aareadme.htm" || $FF == "index/INDEX.TAB" || $FF == "CATALOG/SPICE_HSK.CAT" ) then
               
               echo "      $FF is OK."
               
            else
               #
               #  Yes it does. Run awk again to dump these lines.
               #
               echo "(E)   $FF has lines longer than 78 char. They are:"
               echo " "
               awk '{if(length>78){print $0}}' < $FF
               echo " "
            endif
            
         else
            #
            #  Yes, it does have bad characters. Run badchar again
            #  to generate report.
            #
            echo "(E)   $FF has bad characters. badchar report is:"
            echo " "
            $BADCHAR -v $FF
            echo " "
         endif

      else
         echo "(E)   $FF has zero size."
      endif
      
   else
      echo "(E)   $FF doesn't exist."     
   endif

end

#
#  Checking files.
#
echo " "
echo "   Verifying files -- ALL meta-info files:"
echo " "

foreach FF ( `find . -name '*.LBL' -print` `find . -name '*.TXT' -print` `find . -name '*.CAT' -print` `find . -name '*.HTM' -print` `find . -name '*.TAB' -print`)

   #
   #  Does it exist?
   #
   if ( -e $FF ) then
   
      #
      #  Does it have non-zero size?
      #
      if ( ! ( -z $FF ) ) then
      
         #
         #  Does it have any bad characters?
         #
         $BADCHAR -s $FF
         
         if ( $status != 1 ) then
            
            #
            #  Does it have lines longer than 78 characters?
            #
            awk '{if(length>78){print $0}}' < $FF | grep '.' > /dev/null
            
            if ( ( $status != 0 ) || $FF:e == "HTM" || $FF:e == "TAB" || $FF == "./CATALOG/SPICE_HSK.CAT" ) then
               
               echo "      $FF is OK."
               
            else
               #
               #  Yes it does. Run awk again to dump these lines.
               #
               echo "(E)   $FF has lines longer than 78 char. They are:"
               echo " "
               awk '{if(length>78){print $0}}' < $FF
               echo " "
            endif
            
         else
            #
            #  Yes, it does have bad characters. Run badchar again
            #  to generate report.
            #
            echo "(E)   $FF has bad characters. badchar report is:"
            echo " "
            $BADCHAR -v $FF
            echo " "
         endif

      else
         echo "(E)   $FF has zero size."
      endif
      
   else
      echo "(E)   $FF doesn't exist."     
   endif

end


#
#  Checking text kernels.
#
echo " "
echo "   Verifying files -- text kernels, meta-kernels, and ORBNUM files:"
echo " "

foreach FF ( `find . -name '*.TI' -print` `find . -name '*.TF' -print` `find . -name '*.TLS' -print` `find . -name '*.TPC' -print` `find . -name '*.TSC' -print` `find . -name '*.TEN' -print` `find . -name '*.TM' -print` `find . -name '*.ORB' -print` `find . -name '*.NRB' -print` )

   #
   #  Does it exist?
   #
   if ( -e $FF ) then
   
      #
      #  Does it have non-zero size?
      #
      if ( ! ( -z $FF ) ) then
      
         #
         #  Does it have any bad characters?
         #
         $BADCHAR -s $FF
         
         if ( $status != 1 ) then
            
            echo "      $FF is OK."

         else
            #
            #  Yes, it does have bad characters. Run badchar again
            #  to generate report.
            #
            echo "(E)   $FF has bad characters. badchar report is:"
            echo " "
            $BADCHAR -v $FF
            echo " "
         endif

      else
         echo "(E)   $FF has zero size."
      endif
      
   else
      echo "(E)   $FF doesn't exist."     
   endif

end


#
#  Checking binary kernels.
#
echo " "
echo "   Verifying files -- binary kernels:"
echo " "

foreach FF ( `find . -name '*.BSP' -print` `find . -name '*.BC' -print` `find . -name '*.BPC' -print` `find . -name '*.BES' -print` `find . -name '*.BEP' -print` `find . -name '*.BDB' -print` `find . -name '*.BDS' -print`)

   #
   #  Does it exist?
   #
   if ( -e $FF ) then
   
      #
      #  Does it have non-zero size?
      #
      if ( ! ( -z $FF ) ) then
      
         #
         #  Is this binary kernel in LTL-IEEE form?
         #
         if ( `$BFF $FF | grep -c 'LTL-IEEE'`  == 1 ) then
            
            echo "      $FF is OK."

         else
            #
            #  No, it is not. Report this fact:
            #
            echo "(E)   $FF is not in LTL-IEEE format. Its format is:"
            echo " "
            if ( `$BFF $FF | grep -c 'BIG-IEEE'` == 1 ) then
               echo "         BIG-IEEE"
            else 
               echo "         UNKNOWN"
            endif
            echo " "
         endif

      else
         echo "(E)   $FF has zero size."
      endif
      
   else
      echo "(E)   $FF doesn't exist."     
   endif

end


echo " "
echo "All done."
echo " "

exit
