#!/bin/tcsh
#
# This script generates PDS4 labels based on PDS3 labels.
#
# Pre-requisites/requirements:
#
# -- kernel list, PDS4 values file, kernel XML template must exist
# -- PDS3->PDS4 tool, md5sum, SPKDIFF, FRMDIFF must be executables
#
# -- file names in the list must be in <dir>/<dir>/<name> form
#
# -- PDS3 labels (*.lbl) must exists for all files in the list.
# -- PDS4 labels (*.xml) must not exists for any files in the list.
#
# -- PDS4 values file must include required keywords (see code for list)
# -- PDS3 labels must include required keywords (see code for list)
#
# -- listed MKs must have extension .tm or .TM
# -- listed MKs must have coverage iether for SPK or CK ID
# -- listed MK names must end with version, "vNN"
# -- listed MKs must have kernels at ../*/ relative paths
# -- all kernels listed in an MK must be present
#
# Version 1.0.0 -- May 19, 2015
#
#   Initial version.
#
# Version 1.1.0 -- November 15, 2017
#
#   Updated Unix date format string (%X->%T).
#
# Script name and version:
#
set SCRIPTNAME = "make_pds4_prod_labels.csh"
set SCRIPTVER  = "1.1.0 -- November 15, 2017"

#
# Get command line stuff.
#
if ( $#argv != 5 ) then
    echo " "
    echo "$SCRIPTNAME, version $SCRIPTVER "
    echo " "
    echo "Usage: "
    echo
    echo "   % $SCRIPTNAME <listfile> <valuefile> <templatefile> <ckid> <spkid>"
    echo " "
    echo "where:"
    echo " "
    echo "   listfile     -- kernel list for this release"
    echo "   valuefile    -- fixed PDS4 values KWD=VAL file"
    echo "   templatefile -- SPICE kernel label XML template file"
    echo "   ckid         -- CK ID (e.g. -202000)" 
    echo "   spkid        -- SPK ID (e.g. -202)"
    echo
    exit 1
endif

set LISTFILE   = $1
set PDS4VALUES = $2
set XMLTEMPL   = $3
set CKID       = $4
set SPKID      = $5

#
# Tools: PDS3-to-PDS4 converter, checksum utility, SPICE tools.
#
set PDS4GEN    = "/home/bsemenov/maven/pds/make_pds4_xml_from_lbl.csh"
set MD5SUM     = "/usr/bin/md5sum"
set SPKDIFF    = "/naif/cspice/exe/spkdiff"
set FRMDIFF    = "/naif/cspice/exe/frmdiff"
set CHKLOADED  = "/home/bsemenov/bin/check_loaded"

#
# Check input files and programs.
#
foreach FF ( $LISTFILE $PDS4VALUES $XMLTEMPL )
    if ( ! -e $FF ) then
        echo 
        echo "ERROR: ${SCRIPTNAME}: $FF does not exit."
        echo
        exit 1
    endif
end

foreach FF ( $PDS4GEN $MD5SUM $SPKDIFF $FRMDIFF $CHKLOADED )
    if ( ! -x $FF ) then
        echo 
        echo "ERROR: ${SCRIPTNAME}: $FF is not an excutable."
        echo
        exit 1
    endif
end

#
# Check that PDS4 values files has all required keywords. Extract
# BUNDLE_LID.
#
foreach KWD ( PDS4_MISSION_NAME \
              PDS4_MISSION_LID \
              PDS4_SPACECRAFT_NAME  \
              PDS4_SPACECRAFT_LID  \
              PDS4_TARGET_NAME  \
              PDS4_TARGET_TYPE  \
              PDS4_TARGET_LID  \
              BUNDLE_LID  \
              BUNDLE_DESCRIPTION_LID )

    set COUNT = `grep -c "^ *$KWD *=" $PDS4VALUES` 
    if ( $COUNT == 0 ) then
        echo 
        echo "ERROR: ${SCRIPTNAME}: $PDS4VALUES does not include $KWD."
        echo
        exit 1
    endif
    if ( $COUNT > 1 ) then
        echo 
        echo "ERROR: ${SCRIPTNAME}: $PDS4VALUES includes 2 or more $KWD."
        echo
        exit 1
    endif
end

set BUNDLID  = `grep BUNDLE_LID $PDS4VALUES         | sed 's/=/ /' | awk '{print $NF}'`

#
# Loops through all files in the list.
#
set MKLIST  = ""

foreach FF ( `grep '^ *FILE *=' $LISTFILE  | sed 's/=/ /' | awk '{print $NF}'` ) 

    set TMPLIST = "" 

    #
    # Check if the file exists.
    #
    if ( ! -e $FF ) then
        echo 
        echo "ERROR: ${SCRIPTNAME}: $FF listed in $LISTFILE does not exist."
        echo
        exit 1
    endif

    #
    # Check if the file's path is exactly three levels (2 dirs, 1 file name)
    #
    if ( `echo $FF | sed 's/\// /g' | wc -w` != 3 ) then
        echo 
        echo "ERROR: ${SCRIPTNAME}: $FF name is not in <dir>/<dir>/<name> form."
        echo
        exit 1
    endif

    #
    # Skip MKs and collect their names in a list.
    #
    if ( "$FF:e" == "tm" || "$FF:e" == "TM" ) then 

        set MKLIST = "$MKLIST $FF"

    else

        #
        # Check if there is a PDS3 label for this file. Assume that
        # the label has the same name with lowecase extension .lbl.
        # Check that the label has required keywords.
        #
        set LBL = "$FF:r.lbl"
        
        if ( ! -e $LBL ) then
            echo 
            echo "ERROR: ${SCRIPTNAME}: $LBL does not exist."
            echo
            exit 1
        endif

        foreach KWD ( DESCRIPTION \
                      START_TIME \
                      STOP_TIME \
                      KERNEL_TYPE_ID \
                      INTERCHANGE_FORMAT \
                      PRODUCT_CREATION_TIME )
            set COUNT = `grep -c "^ *$KWD *=" $LBL` 
            if ( $COUNT == 0 ) then
                echo 
                echo "ERROR: ${SCRIPTNAME}: $LBL does not include $KWD."
                echo
                exit 1
            endif
            if ( $COUNT > 1 ) then
                echo 
                echo "ERROR: ${SCRIPTNAME}: $LBL includes 2 or more of $KWD."
                echo
                exit 1
            endif
        end

        #
        # Check that there is no PDS4 label for this file. Assume that
        # the PDS4 label has the same name with lowecase extension .xml.
        #
        set XML = "$FF:r.xml"
        
        if ( -e $XML ) then
            echo 
            echo "ERROR: ${SCRIPTNAME}: $XML already exists."
            echo
            exit 1
        endif

        #
        # Get more values that we need. 
        #
        # Get file size from the file system.
        #
        set FILESIZE = `ls -l $FF | awk '{print $5}'`

        #
        # Get MD5 checksum.
        #
        set CHECKSUM = `$MD5SUM $FF | awk '{print $1}'`

        #
        # Extract interchange format from the PDS3 label.
        #
        set FORMAT = `grep INTERCHANGE_FORMAT $LBL | sed 's/=/ /' | awk '{print $NF}' | sed 's/ASCII/Character/' | sed 's/BINARY/Binary/'`

        #
        # Extract creation date/time from the PDS3 label. Set creaton year.
        #
        set CURRYEAR = `grep PRODUCT_CREATION_TIME $LBL | sed 's/=/ /' | awk '{print $NF}' | sed 's/-/ /' | awk '{print $1}'`

        #
        # Set product LID: <bundle_lid>:<colldir>:<subdir>_<filename>
        #
        set PRODLID = "${BUNDLID}":`echo $FF | sed 's/\//:/' | sed 's/\//_/'`

        #
        # Set product VID: for all non-MKs it is 1.0.
        #
        set PRODVID = "1.0"

        #
        # Make complete temporary KWD=VAL PDS3-like label
        #
        set TMPLBL = "$FF:r.tmp.lbl"

        \rm $TMPLBL >& /dev/null

        echo                                  >>  $TMPLBL
        echo "PRODUCT_LID     = "$PRODLID     >>  $TMPLBL
        echo "PRODUCT_VID     = "$PRODVID     >>  $TMPLBL
        echo "FILE_NAME       = "$FF:t        >>  $TMPLBL
        echo "CURRENT_YEAR    = "$CURRYEAR    >>  $TMPLBL
        echo "FILE_SIZE       = "$FILESIZE    >>  $TMPLBL
        echo "FILE_CHECKSUM   = "$CHECKSUM    >>  $TMPLBL
        echo "FILE_FORMAT     = "$FORMAT      >>  $TMPLBL
        echo                                  >>  $TMPLBL
        cat $PDS4VALUES                       >>  $TMPLBL
        echo                                  >>  $TMPLBL
        cat $LBL                              >>  $TMPLBL
        echo                                  >>  $TMPLBL

        set TMPLIST = "$TMPLIST $TMPLBL"

        #
        # Make temporary PDS4 label.
        #
        set TMPXML    = "$FF:r.tmp.lbl.xml"
        set TMPXMLLOG = "$TMPXML.log"

        \rm $TMPXML >& /dev/null

        $PDS4GEN $TMPLBL $XMLTEMPL >& $TMPXMLLOG

        if ( $status ) then

            #
            # Show error output.
            #
            cat $TMPXMLLOG
            echo
            echo "Values   file was $TMPLBL."
            echo "Template file was $XMLTEMPL."
            echo 

        else 

            #
            # Success. Rename temporary XML label.
            #
            mv $TMPXML $XML
            if ( $status ) then
                echo 
                echo "ERROR: ${SCRIPTNAME}: failed to move $TMPXML to $XML."
                echo
                exit 1
            endif

            set TMPLIST = "$TMPLIST $TMPXMLLOG"

            echo 
            echo "Created $XML"

        endif

    endif

    #
    # Cleanup.
    #
    \rm $TMPLIST >& /dev/null

end

#
# Label MKs.
#
foreach FF ( $MKLIST ) 

    set TMPLIST = "" 

    #
    # Check that there is no PDS4 label for this file. Assume that
    # the PDS4 label has the same name with lowecase extension .xml.
    #
    set XML = "$FF:r.xml"
        
    if ( -e $XML ) then
        echo 
        echo "ERROR: ${SCRIPTNAME}: $XML already exists."
        echo
        exit 1
    endif

    #
    # Extract DESCRIPTION for this MK from the list.
    #
    set DESC  = `grep -A 10 $FF $LISTFILE | grep '^ *DESCRIPTION *=' | head -1 | sed 's/^ *DESCRIPTION *= *//'`

    #
    # Determine START_TIME and STOP_TIME time. Use FRMDIFF and SPKDIFF.
    # Pick latest start date and earliest stop date. Recall that MK must be
    # loadable from their directory, so change there, then back.
    #
    set HERE = `pwd`
    cd $FF:h

    set FRMCMD = "$FRMDIFF -k $FF:t -t dumpc -f YYYY-MM-DDTHR:MN:SC ::RND -f1 J2000 -t1 $CKID" 
    set SPKCMD = "$SPKDIFF -k $FF:t -t dumpc -f YYYY-MM-DDTHR:MN:SC ::RND -c1 0 -r1 J2000 -b1 $SPKID" 

    if ( `$FRMCMD | grep -c '^....-..-..T..:..:..'` != 0 ) then
        set STARTF  = `$FRMCMD | grep '^....-..-..T..:..:..' | head -1 | awk '{print $1}'`
        set STOPF   = `$FRMCMD | grep '^....-..-..T..:..:..' | tail -1 | awk '{print $2}'`
        set STARTFD = `echo $STARTF | sed 's/-//g' | sed 's/T.*$//g'`
        set STOPFD  = `echo $STOPF  | sed 's/-//g' | sed 's/T.*$//g'`
    else
        set STARTFD = "0"
    endif
    
    if ( `$SPKCMD | grep -c '^....-..-..T..:..:..'` != 0 ) then
        set STARTS = `$SPKCMD | grep '^....-..-..T..:..:..' | head -1 | awk '{print $1}'`
        set STOPS  = `$SPKCMD | grep '^....-..-..T..:..:..' | tail -1 | awk '{print $2}'`
        set STARTSD = `echo $STARTS | sed 's/-//g' | sed 's/T.*$//g'`
        set STOPSD  = `echo $STOPS  | sed 's/-//g' | sed 's/T.*$//g'`
    else
        set STARTSD = "0"
    endif
    
    if      ( $STARTFD != 0 && $STARTSD != 0 ) then
        if ( $STARTFD > $STARTSD ) then
            set START = "$STARTF"
        else
            set START = "$STARTS"
        endif
        if ( $STOPFD > $STOPSD ) then
            set STOP = "$STOPS"
        else
            set STOP = "$STOPF"
        endif
    else if ( $STARTFD != 0 ) then
        set START = "$STARTF"
        set STOP  = "$STOPF"
    else if ( $STARTSD != 0 ) then
        set START = "$STARTS"
        set STOP  = "$STOPS"
    else 
        echo 
        echo "ERROR: ${SCRIPTNAME}: cannot get coverage for $FF with '$FRMCMD' or '$SPKCMD'."
        echo
        exit 1
    endif 

    cd $HERE

    #
    # Set PRODUCT_CREATION_TIME and year to CPU clock.
    #
    set CURRTIME = `date +"%FT%T"`
    set CURRYEAR = `date +"%Y"`

    #
    # Set file format.
    #
    set FORMAT = "Character"

    #
    # Set kernel type format.
    #
    set KERTYP = "MK"

    #
    # Get file size from the file system.
    #
    set FILESIZE = `ls -l $FF | awk '{print $5}'`

    #
    # Get MD5 checksum.
    #
    set CHECKSUM = `$MD5SUM $FF | awk '{print $1}'`

    #
    # Set product LID: <bundle_lid>:<colldir>:<subdir>_<filename_upto_version>
    #
    set PRODLID = "${BUNDLID}":`echo $FF | sed 's/\//:/' | sed 's/\//_/' | sed 's/_v[0-9][0-9]*.tm//i'`

    #
    # Set product VID: version from files name
    #
    set PRODVID = `echo $FF | sed 's/^.*_v\([0-9][0-9]*\)\.tm/\1/i' | awk '{printf "%d", $1}'`".0"

    #
    # Make complete temporary KWD=VAL PDS3-like label
    #
    set TMPLBL = "$FF:r.tmp.lbl"

    \rm $TMPLBL >& /dev/null
 
    echo                                        >>  $TMPLBL
    echo "PRODUCT_LID           = "$PRODLID     >>  $TMPLBL
    echo "PRODUCT_VID           = "$PRODVID     >>  $TMPLBL
    echo "FILE_NAME             = "$FF:t        >>  $TMPLBL
    echo "CURRENT_YEAR          = "$CURRYEAR    >>  $TMPLBL
    echo "FILE_SIZE             = "$FILESIZE    >>  $TMPLBL
    echo "FILE_CHECKSUM         = "$CHECKSUM    >>  $TMPLBL
    echo "FILE_FORMAT           = "$FORMAT      >>  $TMPLBL
    echo                                        >>  $TMPLBL
    cat $PDS4VALUES                             >>  $TMPLBL
    echo                                        >>  $TMPLBL
    echo "START_TIME            = "$START       >>  $TMPLBL
    echo "STOP_TIME             = "$STOP        >>  $TMPLBL
    echo "KERNEL_TYPE_ID        = "$KERTYP      >>  $TMPLBL
    echo "PRODUCT_CREATION_TIME = "$CURRTIME    >>  $TMPLBL
    echo "OBJECT                = SPICE_KERNEL" >>  $TMPLBL
    echo 'DESCRIPTION           = "'$DESC'"'    >>  $TMPLBL
    echo "END_OBJECT            = SPICE_KERNEL" >>  $TMPLBL
    echo                                        >>  $TMPLBL

    set TMPLIST = "$TMPLIST $TMPLBL"

    #
    # Make temporary PDS4 label.
    #
    set TMPXML    = "$FF:r.tmp.lbl.xml"
    set TMPXMLLOG = "$TMPXML.log"

    \rm $TMPXML >& /dev/null

    $PDS4GEN $TMPLBL $XMLTEMPL >& $TMPXMLLOG

    if ( $status ) then

        #
        # Show error output.
        #
        cat $TMPXMLLOG
        echo
        echo "Values   file was $TMPLBL."
        echo "Template file was $XMLTEMPL."
        echo 

    else 

        #
        # Success. For MKs, instead of just renaming temporary XML label
        # we will insert references to kernels that MK lists. First check
        # PDS4 label has Reference_List object.
        #
        if ( `grep -c '^ *</Reference_List> *$' $TMPXML` != 1 ) then
            echo 
            echo "ERROR: ${SCRIPTNAME}: $TMPXML does not have Reference_List."
            echo
            exit 1
        endif

        #
        # Print the top part of XML label up to closing Reference List tag.
        #
        grep -B 1000000 '^ *</Reference_List> *$' $TMPXML | head -n -1        > $XML

        #
        # Knowing that file names are in <dir>/<dir>/<name>
        # form, save the top directory name for use in URN.
        # Also for use in URN extract bundle LID from the PDS4
        # values file.
        #
        set TOPDIR    = $FF:h:h
        set BUNDLELID = `grep "^ *BUNDLE_LID *=" $PDS4VALUES | head -1 | sed 's/=/ /' | awk '{print $NF}'` 

        #
        # Change to MK directory as MKs are supposed to be loadable 
        # relative to it. Knowing that file names are in <dir>/<dir>/<name>
        # form, save the top directory name for use in URN.
        #
        set HERE = `pwd`
        cd $FF:h

        #
        # Use check_loaded to get the list of files .
        #
        $CHKLOADED $FF:t >& /dev/null
        if ( $status ) then
            echo 
            echo "ERROR: ${SCRIPTNAME}:$CHKLOADED failed to load $FF:t from "`pwd`
            echo "       The errors were:"
            $CHKLOADED $FF:t
            echo
            exit 1
        endif

        foreach KERNEL ( `$CHKLOADED $FF:t | grep '^ *File *: *' | grep -v $FF:t | awk '{print $NF}' ` )

            #
            # Check that this kernel is in the same subdirectory tree
            # and at the same level as the MK.
            #
            if ( `echo $KERNEL | sed 's/\// /g' | wc -w` != 3  || \
                 `echo $KERNEL | grep -c '^\.\.\/'`      != 1       ) then
                echo 
                echo "ERROR: ${SCRIPTNAME}: $KERNEL listed in $FF is not in ../<dir>/<name> form."
                echo
                exit 1
            endif
            
            #
            # Set kernel's URN. Check that the kernel has a PDS4 label
            # and the URN is present in it.
            #
            set URN = "${BUNDLELID}:${TOPDIR}:"`echo $KERNEL | sed 's/^\.\.\///g' | sed 's/\//_/'`

            if ( ! -e $KERNEL:r.xml ) then
                echo 
                echo "ERROR: ${SCRIPTNAME}: XML label for $KERNEL doe not exist."
                echo
                exit 1
            endif

            if ( `grep -c $URN $KERNEL:r.xml` != 1 ) then
                echo 
                echo "ERROR: ${SCRIPTNAME}: URN $URN doesn't appear in $KERNEL:r.xml or it appears more than once."
                echo
                exit 1
            endif

            #
            # Print reference block for this kernel.
            #
            echo "    <Internal_Reference>"                                  >> $HERE/$XML
            echo "      <lid_reference>$URN</lid_reference>"                 >> $HERE/$XML
            echo "      <reference_type>data_to_associate</reference_type>"  >> $HERE/$XML
            echo "    </Internal_Reference>"                                 >> $HERE/$XML

        end

        cd $HERE

        #
        # Print the bottom part of XML label from the closing Reference List tag.
        #
        grep -A 1000000 '^ *</Reference_List> *$' $TMPXML                    >> $XML

        set TMPLIST = "$TMPLIST $TMPXML $TMPXMLLOG"

        echo 
        echo "Created $XML"

    endif

    #
    # Cleanup.
    #
    \rm $TMPLIST >& /dev/null

end

#
# All done.
#
echo 

exit
