#!/bin/csh -f
#
# This script merges daily M01 s/c CKs into tri-monthly files. 
#
# BVS/NAIF, 07/04/02
#
if ( $#argv != 6 ) then
   echo " "
   echo "This script merges daily M01 s/c CKs into monthly files."
   echo " "
   echo " Usage:    "
   echo " "
   echo "   % merge_sc_trimonthly.csh SCLKFILE PHASE SUBPHASEID YY-MM YY-MM YY-MM"
   echo " "
   echo " Examples: "
   echo " "
   echo "   % merge_sc_trimonthly.csh ORB1_SCLKSCET.00033.tsc map 1 02-03 02-04 02-05"
   echo " "
   exit 1 
else 
   set SCLK   = $argv[1]
   set PHASE  = $argv[2]
   set SPHASE = $argv[3]
   set YYMM1  = $argv[4]
   set YYMM2  = $argv[5]
   set YYMM3  = $argv[6]
   set PHASEU = `echo $PHASE | tr a-z A-Z`
   if      ( $PHASEU == "MAP" ) then
      set PHASEL = "Mapping"
   else if ( $PHASEU == "AB" ) then
      set PHASEL = "Aerobraking"
   else if ( $PHASEU == "EXT" ) then
      set PHASEL = "Extended"
   else 
      echo "PHASE argument can be only map or ab."
      exit(1)
   endif
endif

#
# Current date/time.
#
set NOWDAY  = `date '+20%y-%m-%d'`
set NOWDATE = `date`

#
# Executables
#
set DAFCAT  = "/naif/toolkit/exe/dafcat"
set CKBRIEF = "/naif/toolkit/exe/ckbrief"
set CKSMRG  = "/naif/toolkit/exe/cksmrg"
set COMMNT  = "/naif/toolkit/exe/commnt"

if ( ! -x $DAFCAT ) then
   echo "$DAFCAT does not exist or is not an executable."
   exit(1)
endif
if ( ! -x $CKBRIEF ) then
   echo "$CKBRIEF does not exist or is not an executable."
   exit(1)
endif
if ( ! -x $CKSMRG ) then
   echo "$CKSMRG does not exist or is not an executable."
   exit(1)
endif
if ( ! -x $COMMNT ) then
   echo "$COMMNT does not exist or is not an executable."
   exit(1)
endif

#
# Fixed items
#
set CMTFILE = "m01_sc_trimonthly.ascii"
set LSK     = "naif0008.tls"
set FK      = "m01_v23.tf"
set SEGID   = "Merged CHTLM Qs - Made-up AVs"
set FILEID  = "M01 SC CK; $PHASEU (20$YYMM1...20$YYMM3); NAIF/JPL; Ver 1.0; $NOWDAY"
set BODY    = "-53000"
set REF     = "MARSIAU"
set TOL     = "60 seconds"

if ( ! -e $CMTFILE ) then
   echo "$CMTFILE does not exist."
   exit(1)
endif
if ( ! -e "./"$LSK ) then
   echo "$LSK does not in current directory."
   exit(1)
endif
if ( ! -e "./"$SCLK ) then
   echo "$SCLK does not in current directory."
   exit(1)
endif


#
# File names
#
set CURDIR   = `pwd`
set FTPDIR   = "/ftp/pub/naif/M01/kernels/ck"

set BASENAME = "m01_sc_"$PHASE$SPHASE

set ORIGLIST = $BASENAME".list"

set ORIGCK   = $BASENAME".orig.bc"
set ORIGCMT  = $BASENAME".orig.cmt"

set FINALCK  = $BASENAME".bc"
set FINALCMT = $BASENAME".bc.lbl"

#
# Change to FTP directory.
#
cd $FTPDIR

#
# Generate CK file list for specified month.
#
echo " "
echo "o-o-o  Listing daily CKs to be processed to $ORIGLIST ..."
echo " "
ls -1 m01_sc_20$YYMM1-*.bc m01_sc_20$YYMM2-*.bc m01_sc_20$YYMM3-*.bc > $CURDIR/$ORIGLIST

#
# DAFCAT listed CKs
#
echo " "
echo "o-o-o  Merging daily CKs using DAFCAT to $ORIGCK ..."
echo " "
$DAFCAT $CURDIR/$ORIGCK < $CURDIR/$ORIGLIST

#
# Extract comments from daily CKs
#
echo " "
echo "o-o-o  Extracting comments from daily CKs to $ORIGCMT ..."
echo " "
touch $CURDIR/$ORIGCMT
foreach FF ( `cat $CURDIR/$ORIGLIST` )
   $COMMNT -r $FF >> $CURDIR/$ORIGCMT
end

#
# Go back to staring directory
#
cd $CURDIR

#
# Adding original comment to merged file.
#
echo " "
echo "o-o-o  Adding extracted comments to $ORIGCK ..."
echo " "
$COMMNT -a $ORIGCK $ORIGCMT
rm $ORIGCMT

#
# Running CKSMRG to produce final output file.
#
echo " "
echo "o-o-o  Running CKSMRG to make $FINALCK from $ORIGCK ..."
echo " "
$CKSMRG -k $LSK $SCLK $FK -i $ORIGCK -o $FINALCK -s $SEGID -f $FILEID -b $BODY -r $REF -A KEEP -T $TOL

#
# Generating comments for final CK.
#
echo " "
echo "o-o-o  Making comments $FINALCMT for $FINALCK ..."
echo " "

set COVERAGE = `$CKBRIEF $FINALCK $SCLK $LSK -utc | grep '^  200' | sed 's/ Y//g'`
cat $CMTFILE  | sed "s/REPPHASE/$PHASEL/g"   \
              | sed "s/REPMONTHF/20$YYMM1/g" \
              | sed "s/REPMONTHL/20$YYMM3/g" \
              | sed "s/REPDATE/$NOWDATE/g"   \
              | sed "s/REPLSK/$LSK/g"        \
              | sed "s/REPSCLK/$SCLK/g"      \
              | sed "s/REPTOLERANCE/$TOL/g"  \
              | sed "s/REPCOVER/$COVERAGE/g"                        > $FINALCMT

cat $ORIGLIST                                                      >> $FINALCMT

echo " "                                                           >> $FINALCMT
echo " "                                                           >> $FINALCMT
echo "Appendix 2: Interpolation Intervals in the File"             >> $FINALCMT
echo "--------------------------------------------------------"    >> $FINALCMT
echo " "                                                           >> $FINALCMT
echo "     The information in this section has been generated by running CKBRIEF" \
                                                                   >> $FINALCMT
echo "     utility program using -dump -utc command line options." >> $FINALCMT
echo " "                                                           >> $FINALCMT

$CKBRIEF -utc -dump $FINALCK $SCLK $LSK                            >> $FINALCMT

#
# Adding comments to final file.
#
echo " "
echo "o-o-o  Adding comments to $FINALCK ..."
echo " "
$COMMNT -a $FINALCK $FINALCMT

#
# All done.
#
echo " "
echo "All done."
echo " "
exit
