#!/bin/csh -f
#
# This script merges daily MRO CRISM CKs into monthly files. 
#
# BVS/NAIF, 06/01/07
#
if ( $#argv != 4 ) then
   echo " "
   echo "This script merges daily MRO CRISM CKs into monthly files."
   echo " "
   echo " Usage:    "
   echo " "
   echo "   % merge_crism_monthly.csh SCLKFILE PHASE YY-MM CKLISTFILE"
   echo " "
   echo " Examples: "
   echo " "
   echo "   % merge_crism_monthly.csh MRO_SCLKSCET.00021.65536.tsc PSP 06-09 mro_crm_psp_060925_060930.list"
   echo " "
   exit 1 
else 
   set SCLK     = $argv[1]
   set PHASE    = $argv[2]
   set YYMM     = $argv[3]
   set ORIGLIST = $argv[4]

   set PHASEU = `echo $PHASE | tr a-z A-Z`
   if      ( $PHASEU == "PSP" ) then
      set PHASEL = "Primary Science"
   else if ( $PHASEU == "CRU" ) then
      set PHASEL = "Cruise"
   else 
      echo "PHASE argument can be only psp."
      exit(1)
   endif

   if ( ! -e "./"$SCLK ) then
      echo "$SCLK does not exist in current directory."
      exit(1)
   endif

   if ( ! -e "./"$ORIGLIST ) then
      echo "$ORIGLIST does not exist in current directory."
      exit(1)
   endif

endif

#
# Current date/time.
#
set NOWDAY  = `date '+20%y-%m-%d'`
set NOWDATE = `date`

#
# Executables
#
set DAFCAT  = "/naif/toolkit/exe/dafcat"
set CKBRIEF = "/naif/toolkit/exe/ckbrief"
set COMMNT  = "/naif/toolkit/exe/commnt"

if ( ! -x $DAFCAT ) then
   echo "$DAFCAT does not exist or is not an executable."
   exit(1)
endif
if ( ! -x $CKBRIEF ) then
   echo "$CKBRIEF does not exist or is not an executable."
   exit(1)
endif
if ( ! -x $COMMNT ) then
   echo "$COMMNT does not exist or is not an executable."
   exit(1)
endif

#
# Fixed items
#
set CMTFILE = "mro_crm_monthly.ascii"
set LSK     = "naif0008.tls"
set FK      = "mro_v10.tf"
set CKID    = "-74012"

if ( ! -e $CMTFILE ) then
   echo "$CMTFILE does not exist."
   exit(1)
endif
if ( ! -e "./"$LSK ) then
   echo "$LSK does not exist in current directory."
   exit(1)
endif
if ( ! -e "./"$FK ) then
   echo "$FK does not exist in current directory."
   exit(1)
endif



#
# File names
#
set CURDIR   = `pwd`

set BASENAME = "$ORIGLIST:r"

set ORIGCK   = $BASENAME".orig.bc"
set ORIGCMT  = $BASENAME".orig.cmt"

set FINALCK  = $BASENAME".bc"
set FINALCMT = $BASENAME".bc.lbl"

#
# DAFCAT listed CKs
#
echo " "
echo "o-o-o  Merging daily CKs using DAFCAT to $ORIGCK ..."
echo " "
$DAFCAT $CURDIR/$ORIGCK < $CURDIR/$ORIGLIST

#
# Extract comments from daily CKs
#
echo " "
echo "o-o-o  Extracting comments from daily CKs to $ORIGCMT ..."
echo " "
touch $CURDIR/$ORIGCMT
foreach FF ( `cat $CURDIR/$ORIGLIST` )
   $COMMNT -r $FF >> $CURDIR/$ORIGCMT
end

#
# Adding original comment to merged file.
#
echo " "
echo "o-o-o  Adding extracted comments to $ORIGCK ..."
echo " "
$COMMNT -a $ORIGCK $ORIGCMT
rm $ORIGCMT

#
# Copy original CK to final CK.
#
echo " "
echo "o-o-o  Copying $ORIGCK to $FINALCK ..."
echo " "
cp $ORIGCK $FINALCK

#
# Generating comments for final CK.
#
echo " "
echo "o-o-o  Making comments $FINALCMT for $FINALCK ..."
echo " "

set COVERAGE = `$CKBRIEF $FINALCK $SCLK $LSK $FK -utc $CKID | grep '^  20' | sed 's/ Y//g'`
set COVERAGE = `echo $COVERAGE | awk '{print "          "$1" "$2" "$(NF-1)" "$NF}'`
cat $CMTFILE  | sed "s/REPPHASE/$PHASEL/g"   \
              | sed "s/REPMONTH/20$YYMM/g"   \
              | sed "s/REPDATE/$NOWDATE/g"   \
              | sed "s/REPCOVER/$COVERAGE/g"                        > $FINALCMT

cat $ORIGLIST                                                      >> $FINALCMT
echo                                                               >> $FINALCMT

#
# Adding comments to final file.
#
echo " "
echo "o-o-o  Adding comments to $FINALCK ..."
echo " "
$COMMNT -d $FINALCK
$COMMNT -a $FINALCK $FINALCMT

#
# All done.
#
echo " "
echo "All done."
echo " "
exit
