import json
import spiceypy
import os
import glob
import logging

import arcgen

from arcgen.utils.files import copy
from arcgen.utils.files import get_context_products


class Mission(object):

    def __init__(self, config, cwd):

        logging.info("Loading configuration file: " + config)
        with open(os.path.join(cwd, config), 'r') as f:
            config = json.load(f)

        #
        # Moved MK loading to top because for Rosetta is required to
        # be loaded for calling bod2n for obtaining the target type.
        #
        self.metakernel                 = config['metakernel']
        if self.metakernel:
            logging.info("Loading metakernel: " + self.metakernel)
            spiceypy.furnsh(self.metakernel)

        #
        #  The first thing we do now is to get the PDS version:
        #
        self.pds                        = config['pds']

        self.root_dir                   = os.path.dirname(arcgen.__file__)
        self.name                       = config['mission_name']
        self.lid = \
            'urn:esa:psa:context:investigation:mission.{}'.format(config['mission_accronym'])
        self.target                     = config['target']
        self.spacecraft                 = config['spacecraft']
        self.accronym                   = config['mission_accronym']
        self.target_type                = self.set_target_type()
        self.kernels_directory          = config['kernels_directory']
        self.bundle_directory           = config['bundle_directory']
        self.start                      = config['mission_start']
        self.stop                       = config['mission_stop']
        self.increment_stop             = config['increment_stop']
        self.increment                  = config['increment']
        self.secondary_spacecrafts      = config['secondary_spacecrafts']
        self.secondary_targets          = config['secondary_targets']
        self.xml_model                  = config['xml_model']
        self.schema_location            = config['schema_location']
        self.information_model_version  = config['information_model_version']

        self.mk_grammar                 = f"{self.root_dir}/config/{self.accronym.lower()}.grammar"
        if "mk_grammar" in config:
            # Overwrite grammar in case of specified by config, at the moment only used in tests
            self.mk_grammar = config['mk_grammar']
            logging.info("Using specific grammar from configuration: " + self.mk_grammar)

        self.author                     = config['author']

        #
        # Implemented for ExoMarRSP
        #
        if not self.metakernel:

            logging.info("'metakernel' not specified in configuration, "
                         "loading specific kernels following 'fk_pattern', 'sclk_pattern' and 'lsk_pattern'")

            fks = glob.glob(config['fk_pattern'])
            sclks = glob.glob(config['sclk_pattern'])
            lsks = glob.glob(config['lsk_pattern'])

            fks.sort(), sclks.sort(), lsks.sort()

            self.fk = fks[-1]
            self.sclk = sclks[-1]
            self.lsk = lsks[-1]

            spiceypy.furnsh(self.fk)
            spiceypy.furnsh(self.sclk)
            spiceypy.furnsh(self.lsk)

        self.version = '01'

        if int(self.pds) == 4:
            # PDS4 specific fields

            self.errata                 = f"{self.root_dir}/config/{self.accronym }.errata"
            if "errata" in config:
                # Overwrite errata in case of specified by config, at the moment only used in tests
                self.errata = config['errata']
                logging.info("Using specific errata from configuration: " + self.errata)

            self.context_products       = get_context_products(self)
            self.producer_name          = config['producer_name']
            self.producer_email         = config['producer_email']
            self.producer_phone         = config['producer_phone']
            self.doi                    = config['doi']

            #
            # PDS4 version increment (implies inventory and meta-kernel)
            #
            if self.increment:

                logging.info("Copying " + self.increment + " into " + self.bundle_directory)
                copy(self.increment + os.sep, self.bundle_directory + os.sep)

                versions = glob.glob(self.increment + os.sep +
                                 f'bundle_{self.accronym}_spice_v*')
                versions.sort()

                current_version_int = int(versions[-1].split('_spice_v')[-1].split('.')[0])
                version_int = current_version_int + 1

                current_version = '{:03}'.format(current_version_int)
                version = '{:03}'.format(version_int)

            else:
                version = '001'
                current_version = ''

            self.version = version
            self.current_version = current_version
            logging.info("Bundle version: " + self.version + " , previous: " + self.current_version)

        else:
            # PDS3 specific fields
            self.dataset = config['dataset']
            self.dataset_name = config['dataset_name']
            self.volume = config['volume_id']

            logging.info("Dataset: " + self.dataset + " , Volume: " + self.volume)

        return

    def set_target_type(self):

        code = spiceypy.bodn2c(self.target)

        if str(code)[1:3] == '99' and 1 <= int(str(code)[0]) <= 9:
            target_type = 'Planet'
        elif int(str(code)[0:7]) >= 1000001:
            target_type = 'Comet'

        return target_type


def get_mission_name(config, cwd):

    with open(os.path.join(cwd, config), 'r') as f:
        config = json.load(f)
        return config['mission_name']

    return "Unknown"
