import os
import glob
import logging
import platform
import subprocess, select

from tempfile import mkstemp
from shutil import move
from datetime import datetime


def mk2list(mk):

    path_symbol = ''
    for_next_line = ''
    ker_mk_list = []
    with open(mk, 'r') as f:
        for line in f:

            if '+' in line:
                for_next_line = line
                continue

            if for_next_line:
                line = for_next_line.split('+')[0]+line.split("'")[1]+"'"
                for_next_line = ''

            if path_symbol and '+' not in line:
                if path_symbol in line:

                    kernel = line.split(path_symbol)[1]
                    kernel = kernel.strip()
                    kernel = kernel[:-1]

                    ker_mk_list.append(kernel)

            if 'PATH_SYMBOLS' in line.upper():
                path_symbol = '$' + line.split("'")[1]

            if path_symbol and '+' in line:
                for_next_line = line
                continue

            if '\\begintext' in line:
                break

    return ker_mk_list


def replace(file_path, pattern, subst):

    replaced = False

    #Create temp file
    fh, abs_path = mkstemp()
    with os.fdopen(fh,'w') as new_file:
        with open(file_path) as old_file:
            for line in old_file:

                updated_line = line.replace(pattern, subst)
                if '$KERNELS/' in line and len(updated_line.strip()) > 82:
                    updated_line = os.sep.join(updated_line.split(os.sep)[0:-1]) + \
                                  os.sep + "+'\n" + 27 * " " + "'" + \
                                  updated_line.split(os.sep)[-1]

                new_file.write(updated_line)
                #flag for replacing having happened
                if updated_line != line:
                    replaced = True
    #Remove original file

    if replaced:

        os.remove(file_path)
        # Update the permissions
        os.chmod(abs_path, 0o644)
        #Move new file
        move(abs_path, file_path)

        return True

    return False


def format_str(string, dictionary):
    """
    Return a character string in which all the {KEYWORD} tokens existing in
    the input ``string`` have been replaced by the corresponding value
    contained in the input ``dictionary`` should these KEYWORD=value pairs
    exist in the ``dictionary`` and remain untouched otherwise.

    :param string: str
       Character string containing {KEYWORD} tokens to be replaced by the
       corresponding values provided in the ``dictionary``.
    :param dictionary: dict
       Dictionary that provides several or all the values for the {KEYWORD}s
       contained in the input character ``string``
    :return: str
       Character string in which all the {KEYWORD} tokens in the input
       ``string`` for which a value exists in the input ``dictionary`` have
       been replaced by that value.
    """
    tokens = string.split('}')
    output = []
    for token in tokens[:-1]:
        if token.split('{')[-1] in dictionary.keys():
            output.append((token + '}').format(**dictionary))
        else:
            output.append(token + '}')

    return ''.join(output) + tokens[-1]


def fill_template(template,
                  file,
                  replacements,
                  cleanup=False):


   #
   # If the temp   late file is equal to the output file then we need to create a temporary template - which will be
   # a duplicate - in order to write in the file. A situation where we would like to have them be the same is
   # for example if we call this function several times in a row, replacing keywords in the template in steps
   #
   if template == file:
       with open(file, 'r') as f:
           with open('fill_template.temp', "w+") as t:
               for line in f:
                   t.write(line)

       template = 'fill_template.temp'

   with open(file, "w+") as f:
       #
       # Items are replaced as per correspondance in between the replacements dictionary
       #
       with open(template, 'r') as t:
           for line in t:
               if '{' in line:
                   for k, v in replacements.items():
                       if '{' + k + '}' in line: line = line.replace('{' + k + '}', v)
               #
               # We need to manage the lenght of the line and slpit it if need be.
               #
               #if 'LSK_FILE_NAME' in line or 'FK_FILE_NAME' in line or 'SCLK_FILE_NAME' in line:
               #    if len(line.split("'")[1]) > 80:
               #        line = "'\n      SCLK_FILE_NAME         += '/".join(line.rsplit('/',1))

               f.write(line)

               #
               # If the option cleanup is set as true, we remove the keyword assignments in the filled templated which are
               # unfilled (they should be optional)
               #
   if cleanup:

       with open(file, 'r') as f:
           with open('fill_template.temp', 'w+') as t:
               for line in f:
                   t.write(line)

       template = 'fill_template.temp'

       with open(file, 'w+') as f:
           with open('fill_template.temp', 'r') as t:
               for line in t:
                   if '{' not in line:
                       f.write(line)

                       #
                       # The temporary files are removed
                       #
   if os.path.isfile('fill_template.temp'):
       os.remove('fill_template.temp')


#
# kernel extension two type conversion
#

def kernel_type2extension(kernel_type, case='lower'):

    kernel_type = kernel_type.upper()


    kernel_type_map = {
        "IK": ["TI"],
        "FK": ["TF"],
        "MK": ["TM"],
        "SCLK": ["TSC"],
        "LSK": ["TLS"],
        "PCK": ["TPC","BPC"],
        "CK": ["BC"],
        "SPK": ["BSP"],
        "DSK": ["BDS"]
    }
    kernel_extension = kernel_type_map[kernel_type]

    if case == 'lower':
        kernel_extension_temp = kernel_extension
        kernel_extension = []
        for kernel in kernel_extension_temp:
            kernel_extension.append(kernel.lower())

    return kernel_extension


#
# kernel extension two type conversion
#

def kernel_extension2type(kernel_extension):
    kernel_type_map = {
        "TI": "IK",
        "TF": "FK",
        "TM": "MK",
        "TSC": "SCLK",
        "TLS": "LSK",
        "TPC": "PCK",
        "BC": "CK",
        "BSP": "SPK",
        "BPC": "PCK",
        "BDS": "DSK"
    }

    kernel_type = kernel_type_map[kernel_extension]

    return kernel_type


def kernel_type(kernel):
    kernel_extension = kernel.split('.')[1]

    kernel_type = kernel_extension2type(kernel_extension.upper()).lower()

    return kernel_type


def set_path(root_dir, path):

    if '$root_dir' in path:
        path = root_dir + path.split('$root_dir')[1]

    return(path)


def compare_mk(mk1, mk2):

    mk1_list = []
    mk2_list = []

    with open(mk1, 'r') as f1:
        for line in f1:
            if '$KERNELS/' in line:
                mk1_list.append(line.split('/')[-1][:-2])

    with open(mk2, 'r') as f2:
        for line in f2:
            if '$KERNELS/' in line:
                mk2_list.append(line.split('/')[-1][:-2])

    if sorted(mk1_list) == sorted(mk2_list):
        return False
    else:
        return True


def dirlist(d, dir_list=[], upper_dir='processed'):

    try:
        for k, v in d.items():
            if isinstance(v, dict) or isinstance(v, list):
                dirlist(v, dir_list)
            else:
                if k == upper_dir:
                    dir_list.append(v)
    except:
        for element in d:
            if isinstance(element, dict) or isinstance(element, list):
                dirlist(element, dir_list)
            else:
                if k == upper_dir:
                    dir_list.append(v)

    dir_list = list(dict.fromkeys(dir_list))
    dir_list = list(filter(None, dir_list))

    return dir_list


def clean_temp_input(dir, debug=False):
    #
    # We need to clean the incoming directory from unprocessed adcsng-generated
    # inout files for generation of multiple kernels from single input
    #
    os.chdir(dir)
    try:
        cwd = os.getcwd()
        dir = os.path.join(cwd,dir)
        os.chdir(dir)
    except:
        dir = os.getcwd()
        pass

    files = glob.glob('*_adcsng_temp*')
    files.sort()

    if files:
        logging.info(
                '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -')
        logging.info(
                'The following ADCSng generated temporary files were present in the sources directory:')
        logging.info('')
        for filename in files:
            if 'adcsng_temp' in filename:
                logging.error(f"   {filename.split(os.sep)[-1]}")
                os.remove(os.path.join(dir,filename))
        logging.error('\n')

    try:
        os.chdir(cwd)
    except:
        pass

    return


def update_former_versions(mk_path, kernels_path, updated_mk=False):

    if not updated_mk:
        updated_mk = []
    #
    # Check the meta-kernels in the former versions directory. And
    # update them if needed.
    #
    if os.path.isdir(os.path.join(mk_path, 'former_versions')):

        os.chdir(os.path.join(mk_path, 'former_versions'))
        mks_in_dir = glob.glob('*.TM')
        mks_in_dir += glob.glob('*.tm')

        for mk_in_dir in mks_in_dir:
            updated_mk_flag = replace(mk_in_dir, "'..'", "'../..'")
            kernels_list = mk2list(mk_in_dir)

            for kernel in kernels_list:
                former_deprecated_dir = 'ERROR'
                kernel_as_list = ['ERROR','ERROR']

                #
                # First we consider that non-present kernels
                # are in former versions. Then we look into
                # deprecated kernels.
                #
                ker_path = os.path.join(kernels_path, kernel[1:])
                if not os.path.exists(ker_path) and 'former_versions' not in ker_path and 'deprecated_kernels' not in ker_path:
                    try:
                        kernel_as_list = kernel.split('/')
                        kernel_former = os.path.join('/' + kernel_as_list[1] + '/former_versions',kernel_as_list[2])
                        kernel_former_path = os.path.join(kernels_path + '/' + kernel_as_list[1] + '/former_versions',kernel_as_list[2])
                        former_deprecated_dir = 'former_versions'
                        if not os.path.exists(kernel_former_path):
                            kernel_former = os.path.join('/../misc/deprecated_kernels/' +  kernel_as_list[1] + '/' ,kernel_as_list[2])
                            former_deprecated_dir = 'misc/deprecated_kernels'
                        updated_mk_flag = replace(mk_in_dir, kernel, kernel_former)
                    except:
                        logging.error(f"No '{former_deprecated_dir}' directory available for {kernel_as_list[1]}")

            if updated_mk_flag:
                updated_mk.append(mk_in_dir)

    return updated_mk


def get_exe_dir():

    if platform.system() == 'Darwin':
        if platform.machine() == 'x86_64':
            executables_dir = '/exe/macintel_osx_64bit'
    else:
        executables_dir = '/exe/pc_linux_64bit'

    return executables_dir


def print_html(string):
    try:
        print(string.replace("\n", "<br />"))
    except:
        print(string)


# Given a filename with a given format, returns a datetime with the date expressed in the filename
# Eg: filename => MEX_SC_201022_SADE_M.TAB ,  date_format => 7|6|%y%m%d or ["7|6|%y%m%d", "7|6|yyy%m%d"]
#     Returns =>  datetime with value: 2020-10-22 00:00:00.000
# Note: Each date_format element shall have the following format=> start_index|date_length|strptime_format
#       date_format can be a string or a list of date format strings, in case of list all the formats will be
#       tried. The first valid date parsed will be returned.
def get_date_from_filename(filename, date_format):
    date_str = ""
    exception_str = ""

    # If date_format is not of type list convert it to list.
    if not isinstance(date_format, list):
        date_format = [date_format]

    for date_format_elem in date_format:
        date_format_parts = date_format_elem.split("|")
        try:

            if len(date_format_parts) != 3:
                raise Exception("Wrong 'source_date_format': 'start_index|date_length|strptime_format' "
                                + "in config for file: " + filename)

            start_idx = int(date_format_parts[0])
            end_idx = start_idx + int(date_format_parts[1])
            date_str = filename[start_idx:end_idx]
            return datetime.strptime(date_str, date_format_parts[2])

        except:
            exception_str += "Wrong 'source_date_format': 'start_index|date_length|strptime_format' in config for file: " \
                             + filename + " , format: " + date_format_elem + " , extracted date: " + date_str + "\n"

    # In case of any date returned, raise exception with all tries done.
    raise Exception(exception_str)


def commnt_read(kernel_path, directories):

    command_line_process = subprocess.Popen([directories.executables + '/' + 'commnt', '-r', kernel_path],
                                            bufsize=8192, shell=False,
                                            stdout=subprocess.PIPE,
                                            stderr=subprocess.PIPE)
    process_output = ""
    dataend = False
    while (command_line_process.returncode is None) or (not dataend):
        command_line_process.poll()
        dataend = False

        ready = select.select([command_line_process.stdout, command_line_process.stderr], [], [], 1.0)

        if command_line_process.stderr in ready[0]:
            data = command_line_process.stderr.read(1024)
            if len(data) > 0:
                raise Exception(data)

        if command_line_process.stdout in ready[0]:
            data = command_line_process.stdout.read(1024)
            if len(data) == 0:  # Read of zero bytes means EOF
                dataend = True
            else:
                process_output += data.decode('utf-8')

    return process_output


def commnt_add(kernel_path, commnt_path, directories):
    command_line_process = subprocess.Popen(
            [directories.executables + '/' + 'commnt', '-a', kernel_path, commnt_path],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output, _ = command_line_process.communicate()
    return


def commnt_delete(kernel_path, directories):
    command_line_process = subprocess.Popen(
            [directories.executables + '/' + 'commnt', '-d', kernel_path],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output, _ = command_line_process.communicate()
    return


def run_ckbrief(kernel_path, support_kernels, directories):
    command_line_process = subprocess.Popen([directories.executables + '/' +
                                             'ckbrief', kernel_path, support_kernels, '-utc', '-g'],
                                            stdout=subprocess.PIPE,
                                            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output, _ = command_line_process.communicate()
    coverage_text = process_output.decode("utf-8")
    return coverage_text


def find(name, path):
    for root, dirs, files in os.walk(path):
        if name in files:
            return os.path.join(root, name)

