import os
import re
import glob
import shutil
import tarfile
import subprocess
import numpy.testing as npt

from adcsng.utils.files import print_html
from adcsng.utils.files import get_exe_dir
from adcsng.command_line import main as adcsng


def runADCSng(version: object, debug: object = False, log: object = False, untar: object = True) -> object:

    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    if untar:
        tar = tarfile.open('incoming.tar.gz')
        tar.extractall()
        tar.close()

    print_html(f"Executing ADCSng v{version} for ExoMars2016")
    adcsng('exomars2016.json', debug=debug, log=log)


def runMkADCSng(version, debug=False, log=False):

    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    print_html(f"Executing ADCSng v{version} for ExoMars2016")
    adcsng('exomars2016mk.json', debug=debug, log=log)


def cleanup():

    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    try:
        os.remove('em16_test.lock')
    except:
        pass

    try:
        shutil.rmtree('incoming')
        os.remove('._incoming')
    except:
        pass

    for CleanUp in glob.glob('ker_gen/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('ancdr/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('log/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('temp/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt') or not CleanUp.endswith('temp.log'):
            try:
                shutil.rmtree(CleanUp)
            except:
                if 'aareadme.txt' not in CleanUp:
                    os.remove(CleanUp)


def SPKcompare():

    spks = glob.glob('ker_gen/spk/*.bsp')
    for spk in spks:

        exe_dir = get_exe_dir()
        spk_1 = spk
        spk_2 = f'ker_val/spk/{spk.split(os.sep)[-1]}'
        fk = 'ker_dir/fk/em16_tgo_v18.tf'

        if not os.path.exists(spk_1): raise NameError(f"SPK Kernel {spk_1} does not exist")
        if not os.path.exists(spk_2): raise NameError(f"SPK Kernel {spk_2} does not exist")

        command = f'../..{exe_dir}/spkdiff  -k1 {fk} -k2 {fk} {spk_1}  {spk_2}'
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                           stdout=subprocess.PIPE,stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)
        if 'Toolkit version:' in text:
            print_html(text)
            raise NameError(f"SPKDIFF did not work with SPK Kernels {spk}")

        try:
            text = text.split('\n')
            for line in text:
                if 'Position:' in line or 'Velocity:' in line:
                    npt.assert_almost_equal(float(line.split()[1]), 0.0, decimal=5)
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=5)
                if 'Position (km)' in line or 'Velocity (km/s)' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=5)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=5)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"SPK Kernels {spk} are different")


def CKcompare():

    kernels = 'ker_dir/fk/em16_tgo_v18.tf ' + \
              'ker_gen/sclk/em16_tgo_step_20200824.tsc ' + \
              'ker_dir/lsk/naif0012.tls ' + \
              'ker_dir/sclk/em16_tgo_fict_20180215.tsc'

    cks = {'1':{'ck': 'em16_tgo_sc_fmp_007_01_20181006_20181006_f20180215_v02.bc',
              'fframe': 'TGO_SPACECRAFT',
              'tframe': 'J2000',
              'start':  '2018-OCT-06 13:05:15 TDB',
              'finish': '2018-OCT-06 13:07:10 TDB'},
           '2': {
               'ck':     'em16_tgo_sc_fcp_001_01_20160314_20160318_s20200824_v01.bc',
               'fframe': 'TGO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2016-MAR-14 20:14:11 TDB',
               'finish': '2016-MAR-18 20:14:08 TDB'},
           '3': {
               'ck':     'em16_tgo_sc_fsp_001_01_20180222_20180222_s20200824_v01.bc',
               'fframe': 'TGO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2018-FEB-22 12:01:10 TDB',
               'finish': '2018-FEB-22 13:31:09 TDB'},
           '4': {
               'ck':     'em16_tgo_sc_fsp_002_01_20180222_20180222_s20200824_v01.bc',
               'fframe': 'TGO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2018-FEB-22 12:01:10 TDB',
               'finish': '2018-FEB-22 13:31:09 TDB'},
           '5': {
               'ck':     'em16_tgo_sc_slt_npo_005_01_20160314_20160314_f20180215_v02.bc',
               'fframe': 'TGO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2016-MAR-14 22:02:00 TDB',
               'finish': '2016-MAR-14 22:47:00 TDB'},
           '6': {
               'ck':     'em16_tgo_sc_ssm_20160520_20181003_s20200824_v01.bc',
               'fframe': 'TGO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2018-OCT-03 00:01:05 TDB',
               'finish': '2018-OCT-03 00:01:24 TDB'},
           '7': {
               'ck':     'em16_tgo_sc_ssm_20160520_20181003_s20200824_v01.bc',
               'fframe': 'TGO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2016-MAY-20 00:05:00 TDB',
               'finish': '2016-MAY-20 23:05:00 TDB'},
           '8': {
               'ck':     'em16_tgo_sa_ssm_20160520_20181003_s20200824_v01.bc',
               'fframe': 'TGO_SA+Z_ZERO',
               'tframe': 'TGO_SA+Z',
               'start':  '2018-OCT-02 00:00:54 TDB',
               'finish': '2018-OCT-03 00:01:24 TDB'},
           '9': {
               'ck':     'em16_tgo_sa_ssm_20160520_20181003_s20200824_v01.bc',
               'fframe': 'TGO_SA+Z_ZERO',
               'tframe': 'TGO_SA+Z',
               'start':  '2016-MAY-20 00:05:00 TDB',
               'finish': '2016-MAY-20 23:05:00 TDB'},
           '10': {
               'ck':     'em16_tgo_hga_ssm_20181001_20181003_s20200824_v01.bc',
               'fframe': 'TGO_HGA_EL',
               'tframe': 'TGO_HGA_AZ',
               'start':  '2018-OCT-02 00:00:54 TDB',
               'finish': '2018-OCT-03 00:01:24 TDB'},
           '11': {
               'ck':     'em16_tgo_cassis_tel_20160407_20181231_s20171007_v01.bc',
               'fframe': 'TGO_CASSIS_CRU',
               'tframe': 'TGO_CASSIS_TEL',
               'start':  '2016-APR-07 07:21:34 TDB',
               'finish': '2018-DEC-31 22:59:58 TDB'},
           '12': {
               'ck':     'em16_tgo_sc_isr_cassis_20180820_20181222_s20200824_v01.bc',
               'fframe': 'TGO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2018-DEC-22 16:40:28.985 TDB',
               'finish': '2018-DEC-22 16:40:28.987 TDB'},
           '13': {
               'ck':     'em16_tgo_sc_smp_037_00_01_20210123_20210125_f20180215_v01.bc',
               'fframe': 'TGO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2021-JAN-23 12:09:08.190 TDB',
               'finish': '2021-JAN-25 23:22:51.900 TDB'},
           '14': {
                'ck': 'em16_tgo_nomad_ssm_20210507_20210507_s20200824_v01.bc',
                'fframe': 'TGO_NOMAD_LNO_FMM',
                'tframe': 'TGO_NOMAD_LNO_BASE',
                'start': '2021-MAY-07 09:07:30.155 TDB',
                'finish': '2021-MAY-07 09:11:17.230 TDB'}
           }

    if len(cks) < len(os.listdir("ker_gen/ck/")):
        raise NameError("WARNING: some kernels in ker_gen are not being tested")

    for ck in cks.items():
        ck = ck[1]
        exe_dir = get_exe_dir()
        ck_1 = f"ker_val/ck/{ck['ck']}"
        ck_2 = f"ker_gen/ck/{ck['ck']}"

        if not os.path.exists(ck_1): raise NameError(f"CK Kernel {ck_1} does not exist")
        if not os.path.exists(ck_2): raise NameError(f"CK Kernel {ck_2} does not exist")

        command = f"../..{exe_dir}/frmdiff -k {kernels} -f1 {ck['fframe']} " \
                  f"-t1 {ck['tframe']} -b {ck['start']} -e {ck['finish']} " \
                  f" {ck_1} {ck_2}"
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                                stdout=subprocess.PIPE,
                                                stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)
        if 'version: N0066' in text:
            raise NameError(f"FRMDIFF did not work with CK Kernels {ck['ck']}: {text}")

        #
        # Whilst with the SPKs and SCLKs it is enough to compare the DIFF
        # files, for PREDICKT the mantissa and the precision differ from
        # Linux to MacOS:
        #
        #   OS:    Rotation (rad):             +4.8013978590743E-16      +1.4316667159571E-16
        #   LINUX: Rotation (rad):             +0.0000000000000E+00      +0.0000000000000E+00
        #
        # Therefore we will use an assertion instead of a file comparison
        #
        try:
            text = text.split('\n')
            for line in text:
                if 'Rotation (rad):' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=7)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=7)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"CK Kernels {ck['ck']} are different")

    #
    # Test extra-sources, check that extra source are moved with the TM file to the processed folder.
    #
    extra_sources = ["ancdr/hkt/nmd_raw_sc_lno_20210507T090552-20210507T093937-25-15407-1_2.0.xml"]
    for extra_source in extra_sources:
        if not os.path.exists(extra_source):
            raise NameError(f"Extra source {extra_source} does not exist")

    return


def SCLKcompare():

    begin = '2016-04-30'
    finish = '2018-02-15'
    id =  '-143'

    lsk = 'ker_dir/lsk/naif0012.tls'
    sclks = glob.glob('ker_gen/sclk/*.tsc')
    for sclk in sclks:

        sclk = sclk.split(os.sep)[-1]
        exe_dir = get_exe_dir()
        command = f"../..{exe_dir}/diffsclk {lsk} {begin} {finish} " \
                  f"864000 {id} ker_val/sclk/{sclk} ker_gen/sclk/{sclk}"
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                           stdout=subprocess.PIPE,
                                           stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)

        try:
            text = text.split('\n')
            for line in text:
                if line.strip():
                        npt.assert_almost_equal(float(line.split()[1]), 0.0,
                                            decimal=5)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"SCLK Kernels {sclk} are different")

    return


def MKcompare():

    mks = [f for f in os.listdir('ker_gen/mk/') if
           re.search(r'.*[^0-9].tm', f)]

    if not mks:
        raise NameError("No MK Kernels available")

    for mk in mks:
        data_gen = []
        record = False
        with open(f'ker_gen/mk/{mk}', 'r') as m:
            for line in m:
                if record: data_gen.append(line)
                if '\\begindata' in line: record = True
                if '\\begintext' in line: break

        print_html(''.join(data_gen))

        with open(f'ker_val/mk/{mk}', 'r') as m: data_val = m.readlines()

        if not set(data_gen).issubset(data_val):
            raise NameError("MK Kernels are different")

    return


def GAPcompare():

    gaps = [f for f in os.listdir('ker_gen/reports/') if
           re.search(r'.*gap', f)]

    if not gaps:
        raise NameError("No GAP files available")

    for gap in gaps:
        with open(f'ker_gen/reports/{gap}', 'r') as m: gap_gen = m.readlines()[2:]
        with open(f'misc/reports/{gap}', 'r') as m: gap_val = m.readlines()[2:]
        for line in gap_gen: print_html(line)

        if not gap_gen == gap_val:
            raise NameError("GAP files are different")

    return




if __name__ == '__main__':
    #runMkADCSng('1.0.0', debug=True, log=True)
    #cleanup()
    runADCSng('1.0.0', debug=True, log=True, untar=True)
    #os.chdir(os.getcwd()+'/..')
    #GAPcompare()
    #CKcompare()
    #SPKcompare()
    #MKcompare()
    #cleanup()

