C$Procedure      CHWCML ( Extract arguments from SPKDIFF command line )

      SUBROUTINE CHWCML ( LINE, SPK, BODNAM, BODID, CENNAM, CENID,  
     .                    FRAME, CMPWIN, NITR, STEP, DIFTYP, TIMFMT, 
     .                    KERNLS, SAMPLE, SIGDIG )


C$ Abstract
C
C     Extract arguments from SPKDIFF command line and return them
C     and/or default values via individual variables. If input command
C     line is incomplete or if specific command line keys requesting
C     help are specified, this routine displays usage and stops the
C     program.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     SPKDIFF User's Guide.
C
C$ Keywords
C
C     TBD.
C
C$ Declarations

      IMPLICIT NONE

      INCLUDE              'spkdiff.inc'

      CHARACTER*(*)         LINE
      CHARACTER*(*)         SPK ( 2 )
      CHARACTER*(*)         BODNAM ( 2 )
      INTEGER               BODID  ( 2 )
      CHARACTER*(*)         CENNAM ( 2 )
      INTEGER               CENID  ( 2 )
      CHARACTER*(*)         FRAME  ( 2 )
      DOUBLE PRECISION      CMPWIN ( LBCELL : * )
      INTEGER               NITR
      DOUBLE PRECISION      STEP
      CHARACTER*(*)         DIFTYP
      CHARACTER*(*)         TIMFMT
      CHARACTER*(*)         KERNLS ( 3 )
      LOGICAL               SAMPLE
      INTEGER               SIGDIG

C$ Brief_I/O
C
C     Variable  I/O  Description
C     --------  ---  --------------------------------------------------
C     LINE       I   SPKDIFF command line
C     SPK        O   1st and 2nd SPK file names
C     BODNAM     O   1st and 2nd body names
C     BODID      O   1st and 2nd body IDs
C     CENNAM     O   1st and 2nd center names
C     CENID      O   1st and 2nd center IDs
C     FRAME      O   1st and 2nd frame names
C     CMPWIN     O   Time window for comparison
C     NITR       O   Number of points to be used in comparison.
C     STEP       O   Time step in seconds.
C     DIFTYP     O   Type of summary to be generated by SPKDIFF.
C     TIMFMT     O   Output time format string "dump" summaries.
C     KERNLS     O   Lists of additional kernels.
C     SAMPLE     O   Flag indicating a sampling run.
C     SIGDIG     O   Number of significant digits
C
C$ Detailed_Input
C
C     LINE        is the command line provided to SPKDIFF. See SPKDIFF
C                 User's Guide for the command line syntax and detailed
C                 specification of allowed arguments.
C
C$ Detailed_Output
C
C     SPK         is a two element array containing the names of the
C                 first and second SPK files.
C
C     BODNAM      is a two element array containing the names of the
C                 first and second bodies, position of which is to be
C                 computed.
C
C     BODID       is a two element array containing the IDs of the
C                 first and second bodies, position of which is to be
C                 computed.
C     
C     CENNAM      is a two element array containing the names of the
C                 first and second bodies, position with respect to
C                 which is to be computed.
C     
C     CENID       is a two element array containing the IDs of the first
C                 and second bodies, position with respect to which is
C                 to be computed.
C     
C     FRAME       is a two element array containing the names of the
C                 first and second frames, in which positions are to be
C                 computed.
C     
C     CMPWIN      is a time window for comparison.
C
C     NITR        is the number of points to be used in comparison.
C
C     STEP        is the time step in seconds. Depending on the number
C                 of intervals in CMPWIN, STEP may be returned as zero
C                 if it was not provided on the command line.
C
C     DIFTYP      is a string indicating the type of summary to be
C                 generated by SPKDIFF.
C
C     TIMFMT      is a string containing output time format picture for
C                 "dump"-type summaries.
C
C     KERNLS      is an array of three strings containing the lists of
C                 additional kernels provided on the command line.
C                 KERNLS(1) contains the list applicable to the first
C                 SPK to be examined. KERNLS(2) contains the list
C                 applicable to the second SPK to be examined.
C                 KERNLS(3) contains the list applicable both SPKs
C                 to be examined.
C
C     SAMPLE      is a logical flag indicating whether it is a sampling
C                 run.
C
C     SIGDIG      is the number of significant digits in the output
C                 numbers in scientific format included in the dump
C                 reports.
C
C$ Parameters
C
C     See include file.
C
C$ Exceptions
C
C     TBD.
C
C$ Files
C
C     TBD.
C
C$ Particulars
C
C     TBD.
C
C$ Examples
C
C     TBD.
C
C$ Restrictions
C
C     TBD.
C
C$ Literature_References
C
C     TBD.
C
C$ Author_and_Institution
C
C     B.V. Semenov   (JPL)
C
C$ Version
C
C-    Version 2.1.0, 04-JUL-2014 (BVS)
C
C        Changed short error messages SPICE(NOTANINTEGERNUMBER) and
C        SPICE(NOTANINTEGERNUMBER2) to avoid truncation.
C
C-    Version 2.0.0, 21-OCT-2011 (BVS)
C
C        Updated for majors functionality additions (sampling, window
C        with gaps, etc).
C
C-    Version 1.0.0, 18-APR-2006 (BVS)
C
C-&

C
C     SPICELIB functions
C
      DOUBLE PRECISION      DPMAX
      DOUBLE PRECISION      DPMIN

      INTEGER               INTMAX
      INTEGER               ISRCHC
      INTEGER               RTRIM
      INTEGER               WDCNT
      INTEGER               WNCARD

      LOGICAL               ELEMI
      LOGICAL               EQSTR
      LOGICAL               EXISTS

C
C     Local variables.
C
      CHARACTER*(3)         ARCH
      CHARACTER*(3)         TYPE
      CHARACTER*(5*LINSIZ)  HLLINE
      CHARACTER*(DSPSIZ)    HLPMSG ( HLPCNT )
      CHARACTER*(DSPSIZ)    USGMSG ( USGCNT )
      CHARACTER*(DSPSIZ)    VERMSG ( VERCNT )
      CHARACTER*(LINSIZ)    CLVALS ( MAXKEY )
      CHARACTER*(LINSIZ)    COVDSC ( 2 )
      CHARACTER*(LINSIZ)    ERROR
      CHARACTER*(LINSIZ)    HKRNAM
      CHARACTER*(LINSIZ)    HLINE
      CHARACTER*(LINSIZ)    HLINE2
      CHARACTER*(LINSIZ)    TIMDSC
      CHARACTER*(LINSIZ)    UNPRSD
      CHARACTER*(LINSIZ)    TIME   ( 2 )
      CHARACTER*(WRDSIZ)    CLKEYS ( MAXKEY )
      CHARACTER*(WRDSIZ)    CLKEYU ( MAXKEY )
      CHARACTER*(WRDSIZ)    HWORD

      DOUBLE PRECISION      COVER1 ( LBCELL : WINSIZ )
      DOUBLE PRECISION      COVER2 ( LBCELL : WINSIZ )
      DOUBLE PRECISION      COVERC ( LBCELL : 2 )
      DOUBLE PRECISION      DC     ( ND )
      DOUBLE PRECISION      DESCR  ( ND + NI/2 )
      DOUBLE PRECISION      ET     ( 2 )
      DOUBLE PRECISION      ETT    ( 2 )
      DOUBLE PRECISION      HDP1
      DOUBLE PRECISION      HDP2

      INTEGER               COUNT
      INTEGER               FLIDS1 ( LBCELL : MAXIDS )
      INTEGER               FLIDS2 ( LBCELL : MAXIDS )
      INTEGER               SPIDS1 ( LBCELL : MAXIDS )
      INTEGER               SPIDS2 ( LBCELL : MAXIDS )
      INTEGER               SP1CNT
      INTEGER               SP2CNT
      INTEGER               FRAMID
      INTEGER               HANDLE
      INTEGER               I
      INTEGER               IC     ( NI )
      INTEGER               ICLSTB ( NI )
      INTEGER               ICLSTN ( NI )
      INTEGER               ICLSTS ( NI )
      INTEGER               NARGS
      INTEGER               PTR

      LOGICAL               CLFLAG ( MAXKEY )
      LOGICAL               FOUND

C
C     Save everything to prevent potential memory problems in f2c'ed
C     version.
C
      SAVE

C
C     Initialize command line keys.
C
      DATA ( CLKEYS(I), I =  1, MAXKEY )
     .     /
     .      SB1KEY,
     .      SC1KEY,
     .      FR1KEY,
     .      KE1KEY,
     .      SB2KEY,
     .      SC2KEY,
     .      FR2KEY,
     .      KE2KEY,
     .      KERKEY,
     .      BEGKEY,
     .      ENDKEY,
     .      NSTKEY,
     .      STPKEY,
     .      FMTKEY,
     .      TYPKEY,
     .      DIGKEY,
     .      USGKEY,
     .      UKEY,
     .      HLPKEY,
     .      HKEY,
     .      VKEY
     .     /

C
C     Check in.
C
      CALL CHKIN ( 'CHWCML' )

C
C     Generate uppercase version of command lines keys.
C
      DO I = 1, MAXKEY
         CALL UCASE( CLKEYS(I), CLKEYU(I) )
      END DO

C
C     Initialize version display.
C
      CALL TKVRSN( 'TOOLKIT', HWORD )
      VERMSG(  1 ) = ' '
      VERMSG(  2 ) = PGMNAM // ' -- Version ' // PGMVER //
     .               ' -- Toolkit Version ' // HWORD(:RTRIM(HWORD))
      VERMSG(  3 ) = ' '

C
C     Initialize help display.
C
      HLPMSG(  1 ) = '   # provides means for comparing the t'
     .//             'rajectories of two bodies or'
      HLPMSG(  2 ) = '   sampling the trajectory of a single '
     .//             'body known to SPICE and'
      HLPMSG(  3 ) = '   supported by data from SPICE kernels'
     .//             '.'
      HLPMSG(  4 ) = ' '
      HLPMSG(  5 ) = '   To compare trajectories the program '
     .//             'computes a set of geometric'
      HLPMSG(  6 ) = '   states of a body as seen from a cent'
     .//             'er in a reference frame over an'
      HLPMSG(  7 ) = '   interval of time with a fixed or var'
     .//             'iable time step using one SPK'
      HLPMSG(  8 ) = '   file, computes a set of geometric st'
     .//             'ates for the same or different'
      HLPMSG(  9 ) = '   body-center-frame combination at the'
     .//             ' same times using the other SPK'
      HLPMSG( 10 ) = '   file, and then subtracts the corresp'
     .//             'onding states from each other.'
      HLPMSG( 11 ) = '   Depending on the type of output repo'
     .//             'rt requested the program prints'
      HLPMSG( 12 ) = '   to the screen only the maximum diffe'
     .//             'rences in position and velocity,'
      HLPMSG( 13 ) = '   or a complete table of position and '
     .//             'velocity differences, or a'
      HLPMSG( 14 ) = '   complete table of differences expres'
     .//             'sed in ``view frame'''''
      HLPMSG( 15 ) = '   coordinates, or results of a simple '
     .//             'statistical analysis of the'
      HLPMSG( 16 ) = '   differences expressed in ``view fram'
     .//             'e'''' coordinates.'
      HLPMSG( 17 ) = ' '
      HLPMSG( 18 ) = '   To sample trajectory of a body the p'
     .//             'rogram computes a set of'
      HLPMSG( 19 ) = '   geometric states of a body as seen f'
     .//             'rom a center in a reference'
      HLPMSG( 20 ) = '   frame over an interval of time with '
     .//             'fixed or variable time step,'
      HLPMSG( 21 ) = '   using a given set of kernel files, a'
     .//             'nd prints to the screen a table'
      HLPMSG( 22 ) = '   containing these states.'
      HLPMSG( 23 ) = ' '

      CALL REPMC ( HLPMSG(  1 ), '#', PGMNAM, HLPMSG(  1 ) )

C
C     Initialize usage display.
C
      USGMSG(  1 ) = '   # provides means for comparing'
     .//             ' the trajectories of two bodies'
      USGMSG(  2 ) = '   or sampling the trajectory of a sing'
     .//             'le body using data from SPICE'
      USGMSG(  3 ) = '   kernels (see the User''''s Guide for'
     .//             ' more details.). The program usage'
      USGMSG(  4 ) = '   is:'
      USGMSG(  5 ) = ' '
      USGMSG(  6 ) = '      % # [options] <first SPK name> <s'
     .//             'econd SPK name>'
      USGMSG(  7 ) = '      % # [options] <SPK name>'
      USGMSG(  8 ) = '      % # [options]'
      USGMSG(  9 ) = ' '
      USGMSG( 10 ) = '   Options are shown below. Order and c'
     .//             'ase of keys are not significant.'
      USGMSG( 11 ) = '   Values must be space-separated from '
     .//             'keys, i.e. ''# 10'', not ''#10''.'
      USGMSG( 12 ) = ' '
      USGMSG( 13 ) = '      #  <supporting kernel(s) name(s)>'
      USGMSG( 14 ) = '      # <first body name or ID>'
      USGMSG( 15 ) = '      # <first center name or ID>'
      USGMSG( 16 ) = '      # <first reference frame name>'
      USGMSG( 17 ) = '      # <additional supporting kernel(s'
     .//             ') for first SPK>'
      USGMSG( 18 ) = '      # <second body name or ID>'
      USGMSG( 19 ) = '      # <second center name or ID>'
      USGMSG( 20 ) = '      # <second reference frame name>'
      USGMSG( 21 ) = '      # <additional supporting kernel(s'
     .//             ') for second SPK>'
      USGMSG( 22 ) = '      #  <interval start time>'
      USGMSG( 23 ) = '      #  <interval stop time>'
      USGMSG( 24 ) = '      #  <time step in seconds>'
      USGMSG( 25 ) = '      #  <number of states: # to # (def'
     .//             'ault: #)>'
      USGMSG( 26 ) = '      #  <output time format (default: '
     .//             'TDB seconds past J2000)>'
      USGMSG( 27 ) = '      #  <number of significant digits: '
     .//             '# to # (default: #)>'
      USGMSG( 28 ) = '      #  <report type: #|#|#|#|#|# (def.'
     .//             ': #|#)>'
      USGMSG( 29 ) = ' '

      CALL REPMC ( USGMSG(  1 ), '#', PGMNAM, USGMSG(  1 ) )
      CALL REPMC ( USGMSG(  6 ), '#', PGMNAM, USGMSG(  6 ) )
      CALL REPMC ( USGMSG(  7 ), '#', PGMNAM, USGMSG(  7 ) )
      CALL REPMC ( USGMSG(  8 ), '#', PGMNAM, USGMSG(  8 ) )

      CALL REPMC ( USGMSG( 11 ), '#', NSTKEY, USGMSG( 11 ) )
      CALL REPMC ( USGMSG( 11 ), '#', NSTKEY, USGMSG( 11 ) )

      CALL REPMC ( USGMSG( 13 ), '#', KERKEY, USGMSG( 13 ) )
      CALL REPMC ( USGMSG( 14 ), '#', SB1KEY, USGMSG( 14 ) )
      CALL REPMC ( USGMSG( 15 ), '#', SC1KEY, USGMSG( 15 ) )
      CALL REPMC ( USGMSG( 16 ), '#', FR1KEY, USGMSG( 16 ) )
      CALL REPMC ( USGMSG( 17 ), '#', KE1KEY, USGMSG( 17 ) )
      CALL REPMC ( USGMSG( 18 ), '#', SB2KEY, USGMSG( 18 ) )
      CALL REPMC ( USGMSG( 19 ), '#', SC2KEY, USGMSG( 19 ) )
      CALL REPMC ( USGMSG( 20 ), '#', FR2KEY, USGMSG( 20 ) )
      CALL REPMC ( USGMSG( 21 ), '#', KE2KEY, USGMSG( 21 ) )
      CALL REPMC ( USGMSG( 22 ), '#', BEGKEY, USGMSG( 22 ) )
      CALL REPMC ( USGMSG( 23 ), '#', ENDKEY, USGMSG( 23 ) )
      CALL REPMC ( USGMSG( 24 ), '#', STPKEY, USGMSG( 24 ) )
      CALL REPMC ( USGMSG( 25 ), '#', NSTKEY, USGMSG( 25 ) )
      CALL REPMI ( USGMSG( 25 ), '#', MINITR, USGMSG( 25 ) )
      CALL REPMI ( USGMSG( 25 ), '#', MAXITR, USGMSG( 25 ) )
      CALL REPMI ( USGMSG( 25 ), '#', DEFITR, USGMSG( 25 ) )
      CALL REPMC ( USGMSG( 26 ), '#', FMTKEY, USGMSG( 26 ) )
      CALL REPMC ( USGMSG( 27 ), '#', DIGKEY, USGMSG( 27 ) )
      CALL REPMI ( USGMSG( 27 ), '#', MINSDG, USGMSG( 27 ) )
      CALL REPMI ( USGMSG( 27 ), '#', MAXSDG, USGMSG( 27 ) )
      CALL REPMI ( USGMSG( 27 ), '#', DEFSDG, USGMSG( 27 ) )
      CALL REPMC ( USGMSG( 28 ), '#', TYPKEY, USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', BSCVAL, USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', STSVAL, USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', DMPVAL, USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', DVFVAL, USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', DCVAL,  USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', DGVAL,  USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', BSCVAL, USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', DMPVAL, USGMSG( 28 ) )

C
C     Get command line and do first attempt at parsing. All we need to
C     find out in this try is if one of the help/usage/version key
C     variations was present.
C
      HLINE = LINE
      CALL PARCML( HLINE, MAXKEY, CLKEYU, CLFLAG, CLVALS, FOUND, 
     .             UNPRSD )

C
C     Was command line blank? Is one of the version, usage, or help
C     keys present? Display usage, help, or version and stop.
C
      NARGS = WDCNT( LINE )
      IF ( NARGS .EQ. 0 .OR.
     .     CLFLAG( ISRCHC( VKEY,   MAXKEY, CLKEYS ) ) .OR.
     .     CLFLAG( ISRCHC( HLPKEY, MAXKEY, CLKEYS ) ) .OR.
     .     CLFLAG( ISRCHC( HKEY,   MAXKEY, CLKEYS ) ) .OR.
     .     CLFLAG( ISRCHC( USGKEY, MAXKEY, CLKEYS ) ) .OR.
     .     CLFLAG( ISRCHC( UKEY,   MAXKEY, CLKEYS ) )       ) THEN

C
C        Display version in all cases.
C
         DO I = 1, VERCNT
            CALL TOSTDO( VERMSG(I) )
         END DO

C
C        Depending on what other information was requested, display
C        usage, help, or nothing and stop.
C
         IF ( NARGS .EQ. 0 .OR.
     .        CLFLAG( ISRCHC( USGKEY, MAXKEY, CLKEYS ) ) .OR.
     .        CLFLAG( ISRCHC( UKEY,   MAXKEY, CLKEYS ) )       ) THEN

            DO I = 1, USGCNT
               CALL TOSTDO ( USGMSG(I) )
            END DO

         ELSE IF (
     .        CLFLAG( ISRCHC( HLPKEY, MAXKEY, CLKEYS ) ) .OR.
     .        CLFLAG( ISRCHC( HKEY,   MAXKEY, CLKEYS ) )       ) THEN

            DO I = 1, HLPCNT
               CALL TOSTDO ( HLPMSG(I) )
            END DO

         END IF

         STOP

      END IF

C
C     Pull the last word from the command line and see if it's the name
C     of a file. If it is, check if it's an SPK file.
C
      CALL NTHWD ( LINE, NARGS, SPK(2), PTR )

      IF ( EXISTS( SPK(2) ) ) THEN

         CALL GETFAT ( SPK(2), ARCH, TYPE )

         IF ( ARCH .EQ. 'DAF' .AND. TYPE .EQ. 'SPK' ) THEN

C
C           OK, it looks like the last item on the command line was an
C           SPK file. Let's chop it off and check the second to last
C           item.
C
            LINE(PTR:) = ' '
            NARGS = WDCNT( LINE )

            IF ( NARGS .NE. 0 ) THEN

               CALL NTHWD ( LINE, NARGS, SPK(1), PTR )

               IF ( EXISTS( SPK(1) ) ) THEN

                  CALL GETFAT ( SPK(1), ARCH, TYPE )

                  IF ( ARCH .EQ. 'DAF' .AND. TYPE .EQ. 'SPK' ) THEN

C
C                    Aren't we lucky?! The second to last item on the
C                    command line is also an SPK file. Wipe it off the
C                    tail of the line before moving on.
C
                     LINE(PTR:) = ' '
                     NARGS = WDCNT( LINE )

                  ELSE

C
C                    The second to last item on the command line was a
C                    file but not an SPK file. Set the name of the
C                    first SPK to blank.
C
                     SPK(1)    = ' '

                  END IF

               ELSE

C
C                 The second to last item on the command line is not a
C                 file name. Set the name of the first SPK to blank.
C
                  SPK(1)    = ' '

               END IF

            ELSE

C
C              We have only one argument on the command line -- the
C              second SPK. Obviously the name the first SPK should be
C              set to blank.
C
               SPK(1)    = ' '

            END IF

         ELSE

C
C           The last item on the command line was a file but not an SPK
C           file. Set both SPK names to blanks.
C
            SPK(1)    = ' '
            SPK(2)    = ' '

         END IF

      ELSE

C
C        The last items on the command line was not a file. Set both
C        SPK names to blanks.
C
         SPK(1)    = ' '
         SPK(2)    = ' '

      END IF


C
C     If the files were provided and are SPKs, fetch IDs of all boides
C     covered by them for later use.
C
      CALL SSIZEI ( MAXIDS, FLIDS1 )
      CALL SSIZEI ( MAXIDS, FLIDS2 )
      CALL SCARDI ( 0,      FLIDS1 )
      CALL SCARDI ( 0,      FLIDS2 )

      IF ( SPK(1) .NE. ' ' ) THEN
         CALL SPKOBJ  ( SPK(1), FLIDS1 )
      END IF

      IF ( SPK(2) .NE. ' ' ) THEN
         CALL SPKOBJ  ( SPK(2), FLIDS2 )
      END IF

C
C     Parse the command line again because one or two last items may
C     have been removed from it.
C
      HLINE = LINE
      CALL PARCML( HLINE, MAXKEY, CLKEYU, CLFLAG, CLVALS, FOUND, 
     .             UNPRSD )

C
C     Was there any un-recognized stuff at the beginning of the command
C     line? If yes, complain and exit.
C
      IF ( UNPRSD .NE. ' ' ) THEN
         CALL SETMSG ( '''#'' specified at the beginning of the '    //
     .                 'command line is not recognized as a valid '  //
     .                 'command line key or option or as the name '  //
     .                 'of an existing SPK (for runs when only one ' //
     .                 'SPK name is to be provided on the command '  //
     .                 'line) or as the names of two existing '      //
     .                 'SPKs (for runs when two SPK names are to be '//
     .                 'provided on the command line.) See '         //
     .                 'the program usage for the proper command '   //
     .                 'line syntax and the full list of '           //
     .                 'recognized options.'                         )
         CALL ERRCH  ( '#', UNPRSD                                   )
         CALL SIGERR ( 'SPICE(INCORRECTUSAGE)'                       )
      END IF

C
C     Go on processing the rest of the command line. All other
C     arguments are optional and, if not present, will have to be set
C     to some default values.
C

C
C     First, get additional kernels provided on the command line
C     applicable to both files and/or each specific file. We need to
C     deal with them first because some of them may define frames or
C     name-ID mappings that are needed to process the rest of command
C     line arguments. Start with looking for kernels applicable to the
C     first trajectory set.
C
      I = ISRCHC( KE1KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         IF ( CLVALS(I) .NE. ' ' ) THEN

            KERNLS(1) = CLVALS(I)

            HLINE = KERNLS(1)

            DO WHILE ( HLINE .NE. ' ' )

               CALL NEXTWD( HLINE, HLINE2, HLINE )

               IF ( .NOT. EXISTS( HLINE2 ) ) THEN
                  CALL SETMSG ( 'File ''#'' listed after ''#'' key ' //
     .                          'does not exist.'                    )
                  CALL ERRCH  ( '#', HLINE2                          )
                  CALL ERRCH  ( '#', KE1KEY                          )
                  CALL SIGERR ( 'SPICE(FILEDOESNTEXIST1)'            )
               END IF

            END DO

         ELSE
            CALL SETMSG ( 'Although ''#'' key was provided on the '  //
     .                    'command line no kernel file names were '  //
     .                    'following it.'                            )
            CALL ERRCH  ( '#', KE1KEY                                )
            CALL SIGERR ( 'SPICE(MISSINGFILENAMES1)'                 )
         END IF

      ELSE

         KERNLS(1) = ' '

      END IF

C
C     Second, look for kernels applicable to the second trajectory set.
C
      I = ISRCHC( KE2KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         IF ( CLVALS(I) .NE. ' ' ) THEN

            KERNLS(2) = CLVALS(I)

            HLINE = KERNLS(2)

            DO WHILE ( HLINE .NE. ' ' )

               CALL NEXTWD( HLINE, HLINE2, HLINE )

               IF ( .NOT. EXISTS( HLINE2 ) ) THEN
                  CALL SETMSG ( 'File ''#'' listed after ''#'' key ' //
     .                          'does not exist.'                    )
                  CALL ERRCH  ( '#', HLINE2                          )
                  CALL ERRCH  ( '#', KE2KEY                          )
                  CALL SIGERR ( 'SPICE(FILEDOESNTEXIST2)'            )
               END IF

            END DO

         ELSE
            CALL SETMSG ( 'Although ''#'' key was provided on the '  //
     .                    'command line no kernel file names were '  //
     .                    'following it.'                            )
            CALL ERRCH  ( '#', KE2KEY                                )
            CALL SIGERR ( 'SPICE(MISSINGFILENAMES2)'                 )
         END IF

      ELSE

         KERNLS(2) = ' '

      END IF

C
C     Last, look for kernels applicable to both trajectory sets.
C
      I = ISRCHC( KERKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         IF ( CLVALS(I) .NE. ' ' ) THEN

            KERNLS(3) = CLVALS(I)

            HLINE = KERNLS(3)

            DO WHILE ( HLINE .NE. ' ' )

               CALL NEXTWD( HLINE, HLINE2, HLINE )

               IF ( .NOT. EXISTS( HLINE2 ) ) THEN
                  CALL SETMSG ( 'File ''#'' listed after ''#'' key ' //
     .                          'does not exist.'                    )
                  CALL ERRCH  ( '#', HLINE2                          )
                  CALL ERRCH  ( '#', KERKEY                          )
                  CALL SIGERR ( 'SPICE(FILEDOESNTEXIST3)'            )
               END IF

            END DO

         ELSE
            CALL SETMSG ( 'Although ''#'' key was provided on the '  //
     .                    'command line no kernel file names were '  //
     .                    'following it.'                            )
            CALL ERRCH  ( '#', KERKEY                                )
            CALL SIGERR ( 'SPICE(MISSINGFILENAMES3)'                 )
         END IF

      ELSE

         KERNLS(3) = ' '

      END IF


C
C     Get total count of SPKs and all all IDs for both complete
C     supporting kernel sets (each set is common kernels + specific
C     kernels but not the corresponding SPK). These will be needed to
C     get default coverage windows if no SPKs were provided but
C     body(ies) were explicitly given on the command line.
C
      CALL SSIZEI ( MAXIDS, SPIDS1 )
      CALL SSIZEI ( MAXIDS, SPIDS2 )
      CALL SCARDI ( 0,      SPIDS1 )
      CALL SCARDI ( 0,      SPIDS2 )

      HLLINE = KERNLS(3)//' '//KERNLS(1)

      CALL KCLEAR
      CALL LDKLST ( HLLINE )
      CALL KTOTAL ( 'SPK', SP1CNT )

      DO I = 1, SP1CNT
         CALL KDATA ( I, 'SPK', HKRNAM, HLINE, HLINE2, HANDLE, FOUND )
         IF ( FOUND ) THEN
            CALL SPKOBJ  ( HKRNAM, SPIDS1 )
         ELSE
            CALL SETMSG ( 'There is a bug in the program. Please, '  //
     .                    'contact NAIF.'                            )
            CALL SIGERR ( 'SPICE(SPKDIFFBUG5)'                       )
         END IF
      END DO

      CALL KCLEAR
      HLLINE = KERNLS(3)//' '//KERNLS(2)
      CALL LDKLST( HLLINE )
      CALL KTOTAL ( 'SPK', SP2CNT )

      DO I = 1, SP2CNT
         CALL KDATA ( I, 'SPK', HKRNAM, HLINE, HLINE2, HANDLE, FOUND )
         IF ( FOUND ) THEN
            CALL SPKOBJ  ( HKRNAM, SPIDS2 )
         ELSE
            CALL SETMSG ( 'There is a bug in the program. Please, '  //
     .                    'contact NAIF.'                            )
            CALL SIGERR ( 'SPICE(SPKDIFFBUG6)'                       )
         END IF
      END DO

C
C     Load additional kernels supplied for the first SPK and kernels
C     applicable for both SPKs.
C
      CALL KCLEAR
      HLLINE = KERNLS(3)//' '//KERNLS(1)
      CALL LDKLST( HLLINE )

C
C     After loading all kernels provided for computing the first
C     trajectory set we can move on to getting command line inputs
C     defining it.
C
C
C     Was the center name or ID for the first SPK provided on the
C     command line?
C
      CENNAM(1) = ' '

      I = ISRCHC( SC1KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         CENNAM(1) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), CENID(1), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN
            
            CALL BODN2C ( CLVALS(I), CENID(1), FOUND )

            IF ( .NOT. FOUND ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate NAIF ID '    //
     .                       'nor an object name recognized in '     //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', SC1KEY                             )
               CALL SIGERR ( 'SPICE(BADCENTER1SPEC)'                 )
            END IF

         ELSE

            CALL BODC2N ( CENID(1), HWORD, FOUND )
            IF ( FOUND ) THEN
               CENNAM(1) = HWORD
            END IF

         END IF

      END IF

C
C     Was the body name or ID for the first SPK provided on the
C     command line?
C
      BODNAM(1) = ' '

      I = ISRCHC( SB1KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         BODNAM(1) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), BODID(1), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

            CALL BODN2C ( CLVALS(I), BODID(1), FOUND )

            IF ( .NOT. FOUND ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate NAIF ID '    //
     .                       'nor an object name recognized in '     //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', SB1KEY                             )
               CALL SIGERR ( 'SPICE(BADBODY1SPEC)'                   )
            END IF

         ELSE

            CALL BODC2N ( BODID(1), HWORD, FOUND )
            IF ( FOUND ) THEN
               BODNAM(1) = HWORD
            END IF

         END IF

      END IF

C
C     Was the first frame name provided on the command line?
C
      FRAME(1) = ' '

      I = ISRCHC( FR1KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         FRAME(1) = CLVALS(I)
         CALL NAMFRM( FRAME(1), FRAMID )

         IF ( FRAMID .EQ. 0 ) THEN
            CALL SETMSG ( 'Cannot recognize frame ''#'' provided '   //
     .                    'on the command line after ''#'' key.'     )
            CALL ERRCH  ( '#', FRAME(1)                              )
            CALL ERRCH  ( '#', FR1KEY                                )
            CALL SIGERR ( 'SPICE(BADFRAME1NAME)'                     )
         END IF

      END IF

C
C     We are done with command line keys defining the first trajectory
C     set. We will unload all kernels, load kernels applicable to the
C     second attitude set and move on to processing keys for it.
C
      CALL KCLEAR

C
C     Load additional kernels supplied for the second kernel and
C     kernels applicable for both kernels.
C
      HLLINE = KERNLS(3)//' '//KERNLS(2)
      CALL LDKLST( HLLINE )

C
C     After loading all kernels provided for computing the second
C     trajectory set we can move on to getting command line inputs
C     defining it.
C
C
C     Was the center name or ID for the second SPK provided on the
C     command line?
C
      CENNAM(2) = ' '

      I = ISRCHC( SC2KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         CENNAM(2) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), CENID(2), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

            CALL BODN2C ( CLVALS(I), CENID(2), FOUND )

            IF ( .NOT. FOUND ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing  a legitimate NAIF ID '   //
     .                       'nor an object name recognized in '     //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', SC2KEY                             )
               CALL SIGERR ( 'SPICE(BADCENTER2SPEC)'                 )
            END IF

         ELSE

            CALL BODC2N ( CENID(2), HWORD, FOUND )
            IF ( FOUND ) THEN
               CENNAM(2) = HWORD
            END IF

         END IF

      END IF
 
C
C     Was the body name or ID for the second SPK provided on the
C     command line?
C
      BODNAM(2) = ' '

      I = ISRCHC( SB2KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         BODNAM(2) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), BODID(2), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

            CALL BODN2C ( CLVALS(I), BODID(2), FOUND )

            IF ( .NOT. FOUND ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate NAIF ID '    //
     .                       'nor an object name recognized in '     //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', SB2KEY                             )
               CALL SIGERR ( 'SPICE(BADBODY2SPEC)'                   )
            END IF

         ELSE

            CALL BODC2N ( BODID(2), HWORD, FOUND )
            IF ( FOUND ) THEN
               BODNAM(2) = HWORD
            END IF

         END IF

      END IF

C
C     Was the second frame name provided on the command line?
C
      FRAME(2) = ' '

      I = ISRCHC( FR2KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         FRAME(2) = CLVALS(I)
         CALL NAMFRM( FRAME(2), FRAMID )

         IF ( FRAMID .EQ. 0 ) THEN
            CALL SETMSG ( 'Cannot recognize frame ''#'' provided '   //
     .                    'on the command line after ''#'' key.'     )
            CALL ERRCH  ( '#', FRAME(2)                              )
            CALL ERRCH  ( '#', FR2KEY                                )
            CALL SIGERR ( 'SPICE(BADFRAME2NAME)'                     )
         END IF

      END IF

C
C     Process begin and end time arguments. As these need to be
C     converted to ET, we will need LSK data, which can come in any of
C     the three file sets on the command line -- kernels specific for
C     first of second file or kernels applicable to both files. Note
C     that kernels applicable for the second file and kernels
C     applicable for both file are already loaded. We will only kernels
C     specific to the first file to make sure that we have loaded all
C     kernels that we could.
C
      HLLINE = KERNLS(1)
      CALL LDKLST( HLLINE )

      TIME(1) = ' '
      ET(1)   = DPMIN()

      I = ISRCHC( BEGKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN
         TIME(1) = CLVALS(I)
         CALL STR2ET( CLVALS(I), ET(1) )
      END IF

      TIME(2) = ' '
      ET(2)   = DPMAX()

      I = ISRCHC( ENDKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN
         TIME(2) = CLVALS(I)
         CALL STR2ET( CLVALS(I), ET(2) )
      END IF

C
C     Check that begin time is less than the end time.
C
      IF ( TIME(1) .NE. ' ' .AND. TIME(2) .NE. ' ' ) THEN

         IF ( ET(2) .LT. ET(1) ) THEN
            CALL SETMSG ( 'Specified start time ''#'' is greater '   //
     .                    'than specified stop time ''#''.'          )
            CALL ERRCH  ( '#', TIME(1)                               )
            CALL ERRCH  ( '#', TIME(2)                               )
            CALL SIGERR ( 'SPICE(INCONSISTENTTIMES)'                 )
         END IF

      END IF

C
C     Make a string describing time bounds provided on the command
C     line to be used later in error messages.
C
      IF      ( TIME(1) .NE. ' ' .AND. TIME(2) .NE. ' ' ) THEN

         TIMDSC = 'Time window specified on the command line ' //
     .            'was from ''#'' to ''#''.'
         CALL REPMC ( TIMDSC, '#', TIME(1), TIMDSC )
         CALL REPMC ( TIMDSC, '#', TIME(2), TIMDSC )

      ELSE IF ( TIME(1) .NE. ' ' ) THEN

         TIMDSC = 'Time window specified on the command line ' //
     .            'was all times after ''#''.'
         CALL REPMC ( TIMDSC, '#', TIME(1), TIMDSC )

      ELSE IF ( TIME(2) .NE. ' ' ) THEN

         TIMDSC = 'Time window specified on the command line ' //
     .            'was all times before ''#''.'
         CALL REPMC ( TIMDSC, '#', TIME(2), TIMDSC )

      ELSE

         TIMDSC = ' '

      END IF

C
C     Sanity check: body should be distinct from center for all possible
C     input combinations (-b1/-c1, -b2/-c2, -b1/-c2 and -b2/-c1).
C
      IF      ( BODNAM(1) .NE. ' ' .AND. CENNAM(1) .NE. ' ' ) THEN
         IF   ( BODID(1) .EQ. CENID(1) ) THEN 
            CALL SETMSG ( 'Body and center specified on the '        //
     .                    'command line line must be distinct. '     //
     .                    'They were ''#''(#) and ''#''(#).'         )
            CALL ERRCH  ( '#', BODNAM(1)                             )
            CALL ERRINT ( '#', BODID(1)                              )
            CALL ERRCH  ( '#', CENNAM(1)                             )
            CALL ERRINT ( '#', CENID(1)                              )
            CALL SIGERR ( 'SPICE(SAMEBODY1CENTER1)'                  )
         END IF
      ELSE IF ( BODNAM(2) .NE. ' ' .AND. CENNAM(2) .NE. ' ' ) THEN
         IF   ( BODID(2) .EQ. CENID(2) ) THEN
            CALL SETMSG ( 'Body and center specified on the '        //
     .                    'command line line must be distinct. '     //
     .                    ' They were ''#''(#) and ''#''(#).'        )
            CALL ERRCH  ( '#', BODNAM(2)                             )
            CALL ERRINT ( '#', BODID(2)                              )
            CALL ERRCH  ( '#', CENNAM(2)                             )
            CALL ERRINT ( '#', CENID(2)                              )
            CALL SIGERR ( 'SPICE(SAMEBODY2CENTER2)'                  )
         END IF
      ELSE IF ( BODNAM(1) .NE. ' ' .AND. CENNAM(2) .NE. ' ' 
     .                           .AND. CENNAM(1) .EQ. ' ' ) THEN
         IF   ( BODID(1) .EQ. CENID(2) ) THEN
            CALL SETMSG ( 'Body and center specified on the '        //
     .                    'command line line must be distinct. '     //
     .                    ' They were ''#''(#) and ''#''(#).'        )
            CALL ERRCH  ( '#', BODNAM(1)                             )
            CALL ERRINT ( '#', BODID(1)                              )
            CALL ERRCH  ( '#', CENNAM(2)                             )
            CALL ERRINT ( '#', CENID(2)                              )
            CALL SIGERR ( 'SPICE(SAMEBODY1CENTER2)'                  )
         END IF
      ELSE IF ( BODNAM(2) .NE. ' ' .AND. CENNAM(1) .NE. ' ' 
     .                           .AND. CENNAM(2) .EQ. ' ' ) THEN
         IF   ( BODID(2) .EQ. CENID(1) ) THEN
            CALL SETMSG ( 'Body and center specified on the '        //
     .                    'command line line must be distinct. '     //
     .                    ' They were ''#''(#) and ''#''(#).'        )
            CALL ERRCH  ( '#', BODNAM(2)                             )
            CALL ERRINT ( '#', BODID(2)                              )
            CALL ERRCH  ( '#', CENNAM(1)                             )
            CALL ERRINT ( '#', CENID(1)                              )
            CALL SIGERR ( 'SPICE(SAMEBODY2CENTER1)'                  )
         END IF
      END IF

C
C     At this point we processed all command line keys defining
C     attributes of the two trajectory sets as well as those
C     restricting time boundaries. It is likely that some of the items
C     we need were not provided (``body'', ``center'', ``frame'') or
C     could not be provided (the final comparison window). Let go ahead
C     and fill in these ``blanks''.
C

C
C     If only one body name/ID was provided, set the other one to be
C     the same.
C
      IF      ( BODNAM(1) .EQ. ' ' .AND. BODNAM(2) .NE. ' ' ) THEN

         BODID(1)  = BODID(2)
         BODNAM(1) = BODNAM(2)

      ELSE IF ( BODNAM(2) .EQ. ' ' .AND. BODNAM(1) .NE. ' ' ) THEN

         BODID(2)  = BODID(1)
         BODNAM(2) = BODNAM(1)

      END IF

C
C     If only one center name/ID was provided, set the other one to be
C     the same.
C
      IF      ( CENNAM(1) .EQ. ' ' .AND. CENNAM(2) .NE. ' ' ) THEN

         CENID(1)  = CENID(2)
         CENNAM(1) = CENNAM(2)

      ELSE IF ( CENNAM(2) .EQ. ' ' .AND. CENNAM(1) .NE. ' ' ) THEN

         CENID(2)  = CENID(1)
         CENNAM(2) = CENNAM(1)

      END IF

C
C     If only one frame name was provided, set the other one to be the
C     same.
C
      IF      ( FRAME(1) .EQ. ' ' .AND. FRAME(2) .NE. ' ' ) THEN

         FRAME(1) = FRAME(2)

      ELSE IF ( FRAME(2) .EQ. ' ' .AND. FRAME(1) .NE. ' ' ) THEN

         FRAME(2)  = FRAME(1)

      END IF

C
C     Check if at least one body, center, and frame were provided on
C     the command line. If not, obtain default values by looking at one
C     of the SPK files.
C
      IF      ( ( BODNAM(1) .EQ. ' ' .AND. BODNAM(2) .EQ. ' ' ) .OR.
     .          ( CENNAM(1) .EQ. ' ' .AND. CENNAM(2) .EQ. ' ' ) .OR.
     .          ( FRAME(1)  .EQ. ' ' .AND. FRAME(2) .EQ. ' '  )   ) THEN

C
C        We don't have complete body-center-frame triplets on the
C        command line and need to get some defaults from the SPKs
C        provided at the end of the command line.
C
C        First check if we have at least one SPK provided on the
C        command line to fetch the missing defaults from. If not,
C        complain and stop. If yes, set the SPK to be examined to the
C        first file if two files were provided or to the second file if
C        only one SPK was provided.
C
         IF      ( SPK(1) .NE. ' ' ) THEN

            HKRNAM = SPK(1)

         ELSE IF ( SPK(2) .NE. ' ' ) THEN

            HKRNAM = SPK(2)

         ELSE

            CALL SETMSG ( 'Cannot pick default values for bodies, '  //
     .                    'centers, or frames because insufficient ' //
     .                    'information was given using the relevant '//
     .                    'command line keys and the last item (or ' //
     .                    'the last two items) on the command line ' //
     .                    'are not the names of SPK files.'          )
            CALL SIGERR ( 'SPICE(CANNOTGETDEFAULTS1)'                )

         END IF

C
C        If first, second, or both body IDs were specified on the
C        command line, then center and frame from the segment for this
C        body closest to the end of the file will be picked as default
C        values.
C        
C        If neither first nor second body ID was specified on the
C        command line and SPK contains segments for one or more
C        spacecraft, the body, center, and frame from the spacecraft
C        segment closest to the end of the file will be picked as
C        default values.
C
C        If neither first nor second body ID was specified on the
C        command line and SPK contains no spacecraft segments, the
C        body, center, and frame from the very last segment of the file
C        will be picked as default values.
C

C
C        Zero out descriptor buffers for last segment, last s/c segment
C        and last segment for specified body. Note that SPK type
C        element (ICXXXX(4)) cannot be 0 for any real SPK segment; this
C        property will be relied upon in checks in the loop below to
C        determine if any of these descriptors have already been set.
C
         CALL FILLI ( 0, NI, ICLSTN )
         CALL FILLI ( 0, NI, ICLSTS )
         CALL FILLI ( 0, NI, ICLSTB )

C
C        Open the SPK file that we picked and search it in backward
C        order.
C
         CALL DAFOPR ( HKRNAM, HANDLE )
         CALL DAFBBS ( HANDLE )
         CALL DAFFPA ( FOUND )

         DO WHILE ( FOUND )

C
C           Fetch and unpack the segment descriptor.
C
            CALL DAFGS ( DESCR )
            CALL DAFUS ( DESCR, ND, NI, DC, IC )

C
C           Save integer components of the last descriptor.
C
            IF ( ICLSTN(4) .EQ. 0 ) THEN
               CALL MOVEI( IC, NI, ICLSTN )
            END IF

C
C           Save integer components of the last descriptor for a
C           spacecraft.
C
            IF ( ICLSTS(4) .EQ. 0 .AND. IC(1) .LT. 0 ) THEN
               CALL MOVEI( IC, NI, ICLSTS )
            END IF

C
C           Save integer components of the data descriptor for
C           the specified body.
C
            IF ( ICLSTB(4) .EQ. 0 .AND. BODNAM(1) .NE. ' ' .AND. 
     .                                  BODID(1)  .EQ. IC(1)     ) THEN
               CALL MOVEI( IC, NI, ICLSTB )
            END IF

C
C           Find next segment.
C
            CALL DAFFPA ( FOUND )
            
         END DO
 
C
C        Release the file.
C        
         CALL DAFCLS ( HANDLE )

C
C        Set default values based on priorities described above and the
C        descriptor data collected in the loop.
C
         IF ( BODNAM(1) .NE. ' ' ) THEN

C
C           Check if any segments for specified body were found. If
C           yes, set defaults. If no, complain and stop.
C
            IF ( ICLSTB(4) .NE. 0 ) THEN
               CALL MOVEI( ICLSTB, NI, IC )
            ELSE
               CALL SETMSG ( 'SPK file ''#'' does not contain any '  //
     .                       'data for body ''#''(#) specified '     //
     .                       'on the command line,'                  )
               CALL ERRCH  ( '#', HKRNAM                             )
               CALL ERRCH  ( '#', BODNAM(1)                          )
               CALL ERRINT ( '#', BODID(1)                           )
               CALL SIGERR ( 'SPICE(1NODATAFORBODY)'                 )
            END IF

         ELSE IF ( ICLSTS(4) .NE. 0 ) THEN

C
C           Set defaults to the values from the last segment for a
C           spacecraft.
C
            CALL MOVEI( ICLSTS, NI, IC )

         ELSE

C
C           Set defaults to the values from the last segment
C
            CALL MOVEI( ICLSTN, NI, IC )

         END IF

C
C        Do a sanity check. At this point descriptor containing defaults
C        (IC) should have been set to something meaningful therefore
C        IC(4) should be non-zero.
C
         IF ( IC(4) .EQ. 0 ) THEN
            CALL SETMSG ( 'Cannot retrieve default values from SPK ' //
     .                    'file ''#''. It either is damaged or '     //
     .                    'contains no data segments.'               )
            CALL ERRCH  ( '#', HKRNAM                                )
            CALL SIGERR ( 'SPICE(CANNOTGETDEFAULTS2)'                )
         END IF

C
C        Set default body.
C
         IF ( BODNAM(1) .EQ. ' ' ) THEN
            BODID(1)  = IC(1)
            BODID(2)  = IC(1)
            CALL BODC2N ( BODID(1), BODNAM(1), FOUND )
            IF ( .NOT. FOUND ) THEN
               CALL INTSTR ( IC(1), BODNAM(1) )
            END IF
            BODNAM(2) = BODNAM(1)
         END IF

C
C        Set default center.
C
         IF ( CENNAM(1) .EQ. ' ' ) THEN
            CENID(1) = IC(2)
            CENID(2) = IC(2)
            CALL BODC2N ( CENID(1), CENNAM(1), FOUND )
            IF ( .NOT. FOUND ) THEN
               CALL INTSTR ( IC(2), CENNAM(1) )
            END IF
            CENNAM(2) = CENNAM(1)
         END IF

C
C        Set default frame.
C
         IF ( FRAME(1) .EQ. ' ' ) THEN
            CALL FRMNAM( IC(3), FRAME(1) )
            IF ( FRAME(1) .EQ. ' ' ) THEN
               CALL SETMSG ( 'Cannot pick default frame for this '   //
     .                       'run becasue no frame names were '      //
     .                       'specified on the command line and '    //
     .                       'the frame ID # picked from '           //
     .                       'the SPK ''#'' could not be mapped '    //
     .                       'to a frame name. Please, specify '     //
     .                       'the frame relative to which states '   //
     .                       'should be computed using ''#'' or '    //
     .                       '''#'' command line keys or provide '   //
     .                       'a text kernel defining the frame '     //
     .                       'with the above ID.'                    )
               CALL ERRINT ( '#', IC(3)                              )
               CALL ERRCH  ( '#', HKRNAM                             )
               CALL ERRCH  ( '#', FR1KEY                             )
               CALL ERRCH  ( '#', FR2KEY                             )
               CALL SIGERR ( 'SPICE(CANNOTPICKFRAME)'                )
            END IF
            FRAME(2) = FRAME(1)
         END IF

      END IF

C
C     Now that we have body and center for each pair we can repeat the 
C     same sanity check: body should be distinct from center.
C
      IF      ( BODID(1) .EQ. CENID(1) ) THEN 
 
         CALL SETMSG ( 'Body and center picked from SPK and/or '     //
     .                 'command line line must be distinct. They '   //
     .                 'were ''#''(#) and ''#''(#).'                 )
         CALL ERRCH  ( '#', BODNAM(1)                                )
         CALL ERRINT ( '#', BODID(1)                                 )
         CALL ERRCH  ( '#', CENNAM(1)                                ) 
         CALL ERRINT ( '#', CENID(1)                                 )
         CALL SIGERR ( 'SPICE(SAMEBODYANDCENTER3)'                   )

      ELSE IF ( BODID(2) .EQ. CENID(2) ) THEN

         CALL SETMSG ( 'Body and center picked from SPK and/or '     //
     .                 'command line line must be distinct. They '   //
     .                 'were ''#''(#) and ''#''(#).'                 )
         CALL ERRCH  ( '#', BODNAM(2)                                )
         CALL ERRINT ( '#', BODID(2)                                 )
         CALL ERRCH  ( '#', CENNAM(2)                                )
         CALL ERRINT ( '#', CENID(2)                                 )
         CALL SIGERR ( 'SPICE(SAMEBODYANDCENTER4)'                   )

      END IF

C
C     Now we need to determine the comparison window.
C
C     If we got no SPK file(s) on the command line, the user must have
C     provided start and stop times on the command line. If not, we
C     stop.
C
C     If one SPK file was provided, we will try to get coverage window
C     from it and further restrict this window using start and stop
C     times if they were provided.
C
C     If both SPK files were provided, we will try to get coverage
C     window from each of them, intersect these windows and them and
C     further restrict the resulting window using start and stop times
C     if they were provided.
C

C
C     First check all cases when we should have had start and stop
C     times given on the command line. First check is for no files and
C     missing start or stop.
C     
C     This check was commented out because the program was updated
C     to attempt to get coverage from supporting kernels if no SPKs
C     were provided.
C
C      IF (   SPK(1) .EQ. ' '       .AND.
C     .       SPK(2) .EQ. ' '       .AND.
C     .     ( TIME(1)   .EQ. ' ' .OR.
C     .       TIME(2)   .EQ. ' '      )      ) THEN
C
C         CALL SETMSG ( 'Cannot determine time range for comparison. '//
C     .                 'Both start time and stop time must be '      //
C     .                 'provided using ''#'' and ''#'' keys when '   //
C     .                 'no SPK files are given as the second to '    //
C     .                 'and the last items on the command line.'     )
C         CALL ERRCH  ( '#', BEGKEY                                   )
C         CALL ERRCH  ( '#', ENDKEY                                   )
C         CALL SIGERR ( 'SPICE(NOTIMEBOUNDS1)'                        )
C
C      END IF

C
C     We are done checking the obvious error cases. Since we know our
C     bodies, let's try to get coverage windows for each of them and do
C     more checks after that. For the first body we may get coverage
C     either from the first or the only file. For the second body we
C     will try to get coverage from the second or only file. But first,
C     let's initialize our coverage windows.
C
      CALL SSIZED ( WINSIZ, COVER1 )
      CALL SCARDD ( 0,      COVER1 )

      CALL SSIZED ( WINSIZ, COVER2 )
      CALL SCARDD ( 0,      COVER2 )

C
C     Try to get coverage for the first body from first or second SPK
C     if
C
C              the first SPK is present
C           AND
C              data for the first body is present in the first SPK
C        OR if
C              the second SPK is present
C           AND
C              data for the first body is present in the second SPK.
C
C     OR try to get coverage from the common kernel set + the first
C     supporting kernel set if
C
C              no SPKs were provided, 
C           AND
C              data for the first body is present in the first complete
C              set of supporting kernels
C
      COVDSC(1) = ' '

      IF ( ( SPK(1) .NE. ' '           .AND.
     .       ELEMI( BODID(1), FLIDS1 )       ) .OR.
     .     ( SPK(2) .NE. ' '           .AND.
     .       ELEMI( BODID(1), FLIDS2 )       )      ) THEN

C
C        We have to look up coverage from one of the SPKs specifically
C        given on the command line. Set the file we will work with.
C
         IF ( SPK(1) .NE. ' '           .AND.
     .        ELEMI( BODID(1), FLIDS1 )       ) THEN
            HKRNAM = SPK(1)
         ELSE
            HKRNAM = SPK(2)
         END IF

C
C        Get ET coverage window for the first body from the picked 
C        SPK.
C
         CALL SPKCOV( HKRNAM, BODID(1), COVER1 )

C
C        If we got an empty window back, it's a bug because the ID was
C        in the list of IDs for which SPK provides coverage. If not,
C        make an information string that we may have to use in the
C        error message(s) later.
C
         IF ( WNCARD(COVER1) .EQ. 0 ) THEN

            CALL SETMSG ( 'There is a bug in the program. Please, '//
     .                    'contact NAIF.'                          )
            CALL SIGERR ( 'SPICE(SPKDIFFBUG7)'                     )

         ELSE

            COVDSC(1) = 'Coverage for body with ID '//
     .                  '''#'' was determined from SPK file ''#''.'
            CALL REPMI ( COVDSC(1), '#', BODID(1), COVDSC(1) )
            CALL REPMC ( COVDSC(1), '#', HKRNAM,   COVDSC(1) )

         END IF

C
C        Done determining coverage from one of the SPKs specifically
C        given on the command line.
C


      ELSE IF ( ( SPK(1) .EQ. ' '           .AND.
     .            SPK(2) .EQ. ' '           .AND.
     .            ELEMI( BODID(1), SPIDS1 )       ) ) THEN

C
C        We have to look up coverage from the first set of supporting
C        kernels given on the command line.
C
         CALL KCLEAR
         HLLINE = KERNLS(3)//' '//KERNLS(1)
         CALL LDKLST( HLLINE )

         DO I = 1, SP1CNT
            CALL KDATA ( I, 'SPK', HKRNAM, HLINE, HLINE2, HANDLE, FOUND)
            IF ( FOUND ) THEN
               CALL SPKCOV ( HKRNAM, BODID(1), COVER1 )
            ELSE
               CALL SETMSG ( 'There is a bug in the program. Please, '//
     .                       'contact NAIF.'                          )
               CALL SIGERR ( 'SPICE(SPKDIFFBUG8)'                     )
            END IF
         END DO

C
C        If we got an empty window back, it's a bug because the ID was
C        in the list of IDs for which supporting kernels provide
C        coverage. If not, make an information string that we may have
C        to use in the error message(s) later.
C
         IF ( WNCARD(COVER1) .EQ. 0 ) THEN

            CALL SETMSG ( 'There is a bug in the program. Please, '//
     .                    'contact NAIF.'                          )
            CALL SIGERR ( 'SPICE(SPKDIFFBUG9)'                     )

         ELSE

            COVDSC(1) = 'Coverage for body with ID '//
     .                  '''#'' was determined from supporting ' //
     .                  'kernels ''#'' and ''#''.'
            CALL REPMI ( COVDSC(1), '#', BODID(1),  COVDSC(1) )
            CALL REPMC ( COVDSC(1), '#', KERNLS(3), COVDSC(1) )
            CALL REPMC ( COVDSC(1), '#', KERNLS(1), COVDSC(1) )

         END IF

C
C        Done determining coverage from supporting kernels.
C

      END IF

C
C     Get coverage for the second body if
C
C           the second SPK is present
C        AND 
C           data for the second body is present in the second SPK.
C
C     OR try to get coverage from the common kernel set + the first
C     supporting kernel set if
C
C           no SPKs were provided
C        AND
C           data for the second body is present in the supporting
C           kernels
C
      COVDSC(2) = ' '

      IF ( ( SPK(2) .NE. ' '           .AND.
     .       ELEMI( BODID(2), FLIDS2 )       ) ) THEN

C
C        Get ET coverage window for the second body from the second
C        SPK.
C
         CALL SPKCOV( SPK(2), BODID(2), COVER2 )

C
C        If we got an empty window back, report an error. If not,
C        make an information string that we may have to use in the
C        error message(s) later.
C
         IF ( WNCARD(COVER2) .EQ. 0 ) THEN

            CALL SETMSG ( 'There is a bug in the program. Please, '//
     .                    'contact NAIF.'                          )
            CALL SIGERR ( 'SPICE(SPKDIFFBUG10)'                    )

         ELSE

            COVDSC(2) = 'Coverage for body with ID '//
     .                  '''#'' was determined from SPK file ''#''.'
            CALL REPMI ( COVDSC(2), '#', BODID(2), COVDSC(2) )
            CALL REPMC ( COVDSC(2), '#', SPK(2),   COVDSC(2) )

         END IF

C
C        Done determining coverage from one of the SPKs specifically
C        given on the command line.
C


      ELSE IF ( ( SPK(1) .EQ. ' '           .AND.
     .            SPK(2) .EQ. ' '           .AND.
     .            ELEMI( BODID(2), SPIDS2 )       ) ) THEN

C
C        We have to look up coverage from the second set of supporting
C        kernels given on the command line.
C
         CALL KCLEAR
         HLLINE = KERNLS(3)//' '//KERNLS(2)
         CALL LDKLST( HLLINE )

         DO I = 1, SP2CNT
            CALL KDATA ( I, 'SPK', HKRNAM, HLINE, HLINE2, HANDLE, FOUND)
            IF ( FOUND ) THEN
               CALL SPKCOV ( HKRNAM, BODID(2), COVER2 )
            ELSE
               CALL SETMSG ( 'There is a bug in the program. '//
     .                       'Please, contact NAIF.'          )
               CALL SIGERR ( 'SPICE(SPKDIFFBUG11)'            )
            END IF
         END DO

C
C        If we got an empty window back, it's a bug because the ID was
C        in the list of IDs for which supporting kernels provide
C        coverage. If not, make an information string that we may have
C        to use in the error message(s) later.
C
         IF ( WNCARD(COVER2) .EQ. 0 ) THEN

            CALL SETMSG ( 'There is a bug in the program. Please, '//
     .                    'contact NAIF.'                          )
            CALL SIGERR ( 'SPICE(SPKDIFFBUG12)'                    )

         ELSE

            COVDSC(2) = 'Coverage for body with ID '//
     .                  '''#'' was determined from supporting ' //
     .                  'kernels ''#'' and ''#''.'
            CALL REPMI ( COVDSC(2), '#', BODID(1),  COVDSC(2) )
            CALL REPMC ( COVDSC(2), '#', KERNLS(3), COVDSC(2) )
            CALL REPMC ( COVDSC(2), '#', KERNLS(2), COVDSC(2) )

         END IF

C
C        Done determining coverage from supporting kernels.
C

      END IF

C
C     Check that either both begin and end time are specified or that
C     at least one of the coverage windows is not empty. If not, report
C     a bug because at this point we should have had enough information
C     to determine final comparison window. (At the moment I can think
C     only these cases when this can happen: either or both ``to''
C     frames were were given and are non-CK/PCK frames or are CK/PCK
C     frames but with no data in the files.)
C
      IF (   WNCARD(COVER1) .EQ. 0           .AND.
     .       WNCARD(COVER2) .EQ. 0           .AND.
     .     ( TIME(1)   .EQ. ' '      .OR.
     .       TIME(2)   .EQ. ' '            )       ) THEN

         CALL SETMSG ( 'Insufficient information was provided '      //
     .                 'on the command line to determine time '      //
     .                 'window of interest. The window can be '      //
     .                 'determined by examining the SPK files '      //
     .                 'specified as the second to last and '        //
     .                 'the last items on the command line or '      //
     .                 'by examining supporting kernels if no '      //
     .                 'SPKs were provided and/or given '            //
     .                 'explicitly using the ''#'' and ''#'' '       //
     .                 'keys specifying the start and stop times. '  //
     .                 'When no SPKs are provided at the end of '    //
     .                 'the command line, supporting kernels must '  //
     .                 'provide some coverage for the specified '    //
     .                 'bodies. When one or two SPKs are provided '  //
     .                 'but do not contain coverage for the '        //
     .                 'specified bodies, both the ''#'' and'        //
     .                 '''#'' keys must be used to define the '      //
     .                 'time window of interest. '                   )
         CALL ERRCH  ( '#', BEGKEY                                   )
         CALL ERRCH  ( '#', ENDKEY                                   )
         CALL ERRCH  ( '#', BEGKEY                                   )
         CALL ERRCH  ( '#', ENDKEY                                   )
         CALL SIGERR ( 'SPICE(NOTIMEBOUNDS8)'                        )

      END IF

C
C     Wrap up determining the final comparison window by intersecting
C     the two coverage windows and the window made using begin and end
C     time given on the command line. We will do it in a rather long
C     "if" block instead of a simple way -- setting undefined windows
C     to (DPMIN,DPMAX), intersecting all three of them, and reporting
C     error if there is no intersection -- because we want report more
C     meaningful errors. First, make a constraint window using start
C     and stop time provided on the command line. Note that ET(1) is
C     either a real time provided on the command line or DPMIN while
C     ET(2) is either a real time provided on the command line or
C     DPMAX.
C
      CALL SSIZED ( 2, COVERC )
      CALL SCARDD ( 0, COVERC )

      CALL WNINSD ( ET(1), ET(2), COVERC )

C
C     Now check if we have determined both, one, or no coverage windows
C     from the file(s).
C
      IF      ( WNCARD(COVER1) .NE. 0  .AND.
     .          WNCARD(COVER2) .NE. 0        ) THEN

C
C        Both coverage windows are not empty. Intersect them and report
C        an error if intersection is empty.
C
         CALL WNINTD( COVER1, COVER2, CMPWIN )

         IF ( WNCARD(CMPWIN) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because coverage windows '     //
     .                    'obtained by examining # provided on the ' //
     .                    'command line do not overlap. # #'         )
            IF (  SPK(1) .NE. ' ' ) THEN
               CALL ERRCH  ( '#', 'the SPKs'                         )
            ELSE
               CALL ERRCH  ( '#', 'the SPK'                          )
            END IF
            CALL ERRCH  ( '#', COVDSC(1)                             )
            IF ( COVDSC(1) .NE. COVDSC(2) ) THEN
               CALL ERRCH  ( '#', COVDSC(2)                          )
            ELSE
               CALL ERRCH  ( '#', ' '                                )
            END IF
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS9)'                     )

         END IF

C
C        Intersect result window with the start/stop window and place
C        it in a temporary window. Report an error if intersection is
C        empty.
C
         CALL WNINTD( CMPWIN, COVERC, COVER1 )

         IF ( WNCARD(COVER1) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because the intersection of '  //
     .                    'the coverage windows obtained by '        //
     .                    'examining # provided on the command '     //
     .                    'line does not overlap the time range '    //
     .                    'specified in the command line using '     //
     .                    '''#'' and/or ''#'' keys. # # # '          )
            IF (  SPK(1) .NE. ' ' ) THEN
               CALL ERRCH  ( '#', 'the SPKs'                         )
            ELSE
               CALL ERRCH  ( '#', 'the SPK'                          )
            END IF
            CALL ERRCH  ( '#', BEGKEY                                )
            CALL ERRCH  ( '#', ENDKEY                                )
            CALL ERRCH  ( '#', COVDSC(1)                             )
            IF ( COVDSC(1) .NE. COVDSC(2) ) THEN
               CALL ERRCH  ( '#', COVDSC(2)                          )
            ELSE
               CALL ERRCH  ( '#', ' '                                )
            END IF
            CALL ERRCH  ( '#', TIMDSC                                )
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS10)'                    )

         END IF

C
C        Copy temporary window to final window.
C
         CALL COPYD  ( COVER1, CMPWIN )

      ELSE IF ( WNCARD(COVER1) .NE. 0 ) THEN

C
C        Only first coverage window is not empty. Intersect it with the
C        start/stop window. Report an error if intersection is empty.
C
         CALL WNINTD( COVER1, COVERC, CMPWIN )

         IF ( WNCARD(CMPWIN) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because the coverage window '  //
     .                    'obtained by examining the SPK '           //
     .                    'provided on the command line does not '   //
     .                    'overlap the time range specified in '     //
     .                    'the command line using ''#'' and/or '     //
     .                    '''#'' keys. # #'                          )
            CALL ERRCH  ( '#', BEGKEY                                )
            CALL ERRCH  ( '#', ENDKEY                                )
            CALL ERRCH  ( '#', COVDSC(1)                             )
            CALL ERRCH  ( '#', TIMDSC                                )
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS11)'                    )

         END IF

      ELSE IF ( WNCARD(COVER2) .NE. 0 ) THEN

C
C        Only second coverage window is not empty. Intersect it with
C        the start/stop window. Report an error if intersection is
C        empty.
C
         CALL WNINTD( COVER2, COVERC, CMPWIN )

         IF ( WNCARD(CMPWIN) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because the coverage window '  //
     .                    'obtained by examining the SPK '           //
     .                    'provided on the command line does not '   //
     .                    'overlap the time range specified in '     //
     .                    'the command line using ''#'' and/or '     //
     .                    '''#'' keys. # #'                          )
            CALL ERRCH  ( '#', BEGKEY                                )
            CALL ERRCH  ( '#', ENDKEY                                )
            CALL ERRCH  ( '#', COVDSC(2)                             )
            CALL ERRCH  ( '#', TIMDSC                                )
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS12)'                    )

         END IF

      ELSE

C
C        Both coverage windows are empty. Copy the window set using
C        start and stop time to final comparison window.
C
         CALL COPYD  ( COVERC, CMPWIN )

C
C        Sanity check. If either start or stop time is blank, something
C        is wrong with the code determining coverages. Report a bug.
C
         IF ( TIME(1) .EQ. ' ' .OR. TIME(2) .EQ. ' ' ) THEN
            CALL SETMSG ( 'There is a bug in the program. Please, '  //
     .                    'contact NAIF.'                            )
            CALL SIGERR ( 'SPICE(SPKDIFFBUG1)'                       )
         END IF

      END IF

C
C     Reset ET(1) and ET(2) to be ends of the actual comparison window.
C
      CALL WNFETD( CMPWIN, 1,              ET(1), HDP1  )
      CALL WNFETD( CMPWIN, WNCARD(CMPWIN), HDP1,  ET(2) )

      CALL ETCAL( ET(1), TIME(1) )
      CALL ETCAL( ET(2), TIME(2) )
      TIME(1) = TIME(1)(:RTRIM(TIME(1))) // ' TDB'
      TIME(2) = TIME(2)(:RTRIM(TIME(2))) // ' TDB'


C
C     Set flag indicating whether we are running in comparison mode or
C     in sampling mode. When two SPKs were provided, we will do
C     sampling if centers are the same, bodies are the same, frames are
C     the same, SPK-specific kernels are the same, and files are the
C     same. When one or no SPKs were provided, we will do sampling if
C     centers are the same, bodies are the same, frames are the same,
C     and SPK-specific kernels are the same.
C
      IF ( SPK(1) .NE. ' ' .AND. SPK(2) .NE. ' ' ) THEN

C
C        For backward compatibility with verison 1.0.0 we will always 
C        run in comparison mode when two SPKs were provided.
C
C         SAMPLE = EQSTR( CENNAM(1), CENNAM(2) ) .AND.
C     .            EQSTR( BODNAM(1), BODNAM(2) ) .AND.
C     .            EQSTR( FRAME(1),  FRAME(2)  ) .AND.
C     .            KERNLS(1) .EQ. KERNLS(2)      .AND.
C     .            SPK(1) .EQ. SPK(2)
C
         SAMPLE = .FALSE.


      ELSE

         SAMPLE = EQSTR( CENNAM(1), CENNAM(2) ) .AND.
     .            EQSTR( BODNAM(1), BODNAM(2) ) .AND.
     .            EQSTR( FRAME(1),  FRAME(2)  ) .AND.
     .            KERNLS(1) .EQ. KERNLS(2)

      END IF

C
C     Check command line for the rest of arguments. Unlike the
C     attributes of the first and second trajectory sets and comparison
C     window, the rest of arguments are truly optional.
C
C     Start with the type of output. Default type is basic.
C
      I = ISRCHC( TYPKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         DIFTYP = CLVALS(I)

         IF ( .NOT. ( EQSTR( DIFTYP, STSVAL ) .OR. 
     .                EQSTR( DIFTYP, DMPVAL ) .OR. 
     .                EQSTR( DIFTYP, DVFVAL ) .OR. 
     .                EQSTR( DIFTYP, DCVAL  ) .OR. 
     .                EQSTR( DIFTYP, DGVAL  ) .OR. 
     .                EQSTR( DIFTYP, BSCVAL )      ) ) THEN

            CALL SETMSG ( 'Output type ''#'' specified after '       //
     .                    '''#'' key is not recognized. Recognized ' //
     .                    'output types are ''#'', ''#'', ''#'', '   //
     .                    '''#'', ''#'', and ''#''.'                 )
            CALL ERRCH  ( '#', CLVALS(I)                             )
            CALL ERRCH  ( '#', TYPKEY                                )
            CALL ERRCH  ( '#', BSCVAL                                )
            CALL ERRCH  ( '#', STSVAL                                )
            CALL ERRCH  ( '#', DMPVAL                                )
            CALL ERRCH  ( '#', DVFVAL                                )
            CALL ERRCH  ( '#', DCVAL                                 )
            CALL ERRCH  ( '#', DGVAL                                 )
            CALL SIGERR ( 'SPICE(BADOUTPUTTYPE)'                     )

         END IF

         IF (   SAMPLE                        .AND.
     .        ( EQSTR( DIFTYP, STSVAL ) .OR. 
     .          EQSTR( DIFTYP, DVFVAL ) .OR. 
     .          EQSTR( DIFTYP, BSCVAL )      )     ) THEN

            CALL SETMSG ( 'Output type ''#'' specified after '       //
     .                    '''#'' key is not applicable for '         //
     .                    'sampling runs. Only '                     //
     .                    'output types ''#'', ''#'', and ''#'' '    //
     .                    'are applicable.'                          )
            CALL ERRCH  ( '#', CLVALS(I)                             )
            CALL ERRCH  ( '#', TYPKEY                                )
            CALL ERRCH  ( '#', DMPVAL                                )
            CALL ERRCH  ( '#', DCVAL                                 )
            CALL ERRCH  ( '#', DGVAL                                 )
            CALL SIGERR ( 'SPICE(BADOUTPUTTYPE)'                     )

         END IF

      ELSE

C
C        If output type was not specified, set it based on whether this
C        is a sampling or comparison run.
C
         IF ( SAMPLE ) THEN
            DIFTYP = DMPVAL
         ELSE
            DIFTYP = BSCVAL
         END IF

      END IF

C
C     Next, get time step or number of steps. If both are specified,
C     time step has higher priority and, for this reason, should be
C     processed first. Default step is zero (meaning
C     "not set") and default number of points is a parameter set
C     in the include file.
C
      STEP = 0.D0
      NITR = DEFITR

      I = ISRCHC( STPKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

C
C        Is the step a DP number?
C
         CALL NPARSD ( CLVALS(I), STEP, ERROR, PTR )

         IF ( PTR .EQ. 0 ) THEN

C
C           Check that step is a positive number and is greater that
C           the smallest step we can allow.
C
            IF ( STEP .LT. STPTOL ) THEN
               CALL SETMSG ( 'Time step ''#'' specified after '   //
     .                       '''#'' key is smaller than # '       //
     .                       'seconds.'                           )
               CALL ERRCH  ( '#', CLVALS(I)                       )
               CALL ERRCH  ( '#', STPKEY                          )
               CALL ERRDP  ( '#', STPTOL                          )
               CALL SIGERR ( 'SPICE(STEPTOOSMALL1)'               )
            END IF

C
C           Fork for backwards compatibility with the first version of
C           SPKDIFF that could work only on continuous intervals if the
C           comparison window is a single interval.
C
            IF ( WNCARD(CMPWIN) .EQ. 1 .AND. .NOT. SAMPLE ) THEN

C
C              Compute the number of steps that will be required to
C              step over the time interval with this step.
C
               CALL RMAIND ( ET(2) - ET(1), STEP, HDP1, HDP2 )

C
C              If step is greater than time range, we will have only
C              two points. If step overflows maximum integer we reset
C              it to INTMAX-2 and error out later. If remainder is
C              closer than tolerance to zero, we will not introduce
C              "extra" step for the end of the interval. If otherwise
C              we will add an extra step for end of the interval.
C
               IF      ( HDP1 .EQ. 0.D0   ) THEN
                  NITR = 2
               ELSE IF ( HDP1 .GT. INTMAX() - 2 ) THEN
                  NITR = INTMAX() - 2
               ELSE IF ( HDP2 .LE. STPTOL ) THEN
                  NITR = NINT( HDP1 ) + 1
                  HDP1 = HDP1 + 1
               ELSE
                  NITR = NINT( HDP1 ) + 2
                  HDP1 = HDP1 + 2
               END IF

C
C              Check that this number of states will fit into the
C              buffer.
C           
               IF ( NITR .GT. MAXITR ) THEN
                  CALL SETMSG ( 'The number of states, #, computed '  //
     .                          'using step of # seconds within time '//
     .                          'interval from ''#''(# TDB seconds) ' //
     .                          'to ''#''(# TDB seconds) is greater ' //
     .                          'than can fit into program''s '       //
     .                          'buffers (# states maximum.) '        //
     .                          'Increase step or '                   //
     .                          'make the time window smaller in '    //
     .                          'order to run the program.'           )
                  CALL ERRDP  ( '#', HDP1                             )
                  CALL ERRDP  ( '#', STEP                             )
                  CALL ERRCH  ( '#', TIME(1)                          )
                  CALL ERRDP  ( '#', ET(1)                            )
                  CALL ERRCH  ( '#', TIME(2)                          )
                  CALL ERRDP  ( '#', ET(2)                            )
                  CALL ERRINT ( '#', MAXITR                           )
                  CALL SIGERR ( 'SPICE(STEPTOOSMALL2)'                )
               END IF

            ELSE

C
C              Compute the number of steps that will be required to
C              step over the comparison window with this step. To do
C              that, we loop over the window and increment the total
C              step count by 2 (one interval's start and one for stop)
C              plus whatever number of steps fits within the interval.
C
               NITR = 0
               DO I = 1, WNCARD(CMPWIN)

C
C                 Fetch endpoints of the next interval.
C
                  CALL WNFETD ( CMPWIN, I, ETT(1), ETT(2) )

C
C                 Add one step for start of the interval.
C
                  NITR = NITR + 1

C
C                 Add one step for each point between endpoints
C                 up to the end of the interval minus padding.
C
                  HDP2 = ETT(2) - STEP * STPPAD

                  COUNT = 1
                  HDP1  = ETT(1) + STEP * COUNT

                  DO WHILE ( HDP1 .LT. HDP2 .AND. NITR .LT. MAXITR )

                     NITR = NITR + 1

                     COUNT = COUNT + 1 
                     HDP1  = ETT(1) + STEP * COUNT

                  END DO

C
C                 If interval begin time is not equal to interval end
C                 time add one step for the end of the interval. If we
C                 terminated the loop with NITR equal to MAXITR this
C                 will take us over the allowed maximum and will
C                 trigger the error message that follows.
C
                  IF ( ETT(1) .NE. ETT(2) ) THEN
                     NITR = NITR + 1
                  END IF

               END DO

C
C              Check if this number of points fits into the buffer.
C
               IF ( NITR .GT. MAXITR ) THEN
                  CALL SETMSG ( 'The number of points within '       //
     .                          'the # window determined using '     //
     .                          'step of ''#'' seconds specified '   //
     .                          'after the command line key ''#'' '  //
     .                          'is greater than can fit into '      //
     .                          'program''s buffers (# epochs '      //
     .                          'maximum.) Increase the step or use '//
     .                          'command line keys ''#'' and ''#'' ' //
     .                          'to make the time window smaller in '//
     .                          'order to run the program. The '     //
     .                          'comparison window was #. # # #'     )
                  IF ( SAMPLE ) THEN
                     CALL ERRCH  ( '#', 'sampling'                   )
                  ELSE
                     CALL ERRCH  ( '#', 'comparison'                 )
                  END IF
                  CALL ERRDP  ( '#', STEP                            )
                  CALL ERRCH  ( '#', STPKEY                          )
                  CALL ERRINT ( '#', MAXITR                          )
                  CALL ERRCH  ( '#', BEGKEY                          )
                  CALL ERRCH  ( '#', ENDKEY                          )
                  IF ( TIMDSC .NE. ' ' ) THEN
                     IF      ( COVDSC(1) .NE. ' ' .AND.
     .                         COVDSC(2) .NE. ' '        ) THEN
                        CALL ERRCH  ( '#', 'determined by applying ' //
     .                                'constraints specified on the '//
     .                                'command line to the '         //
     .                                'intersection of coverages '   //
     .                                'obtained from the files'      )
                     ELSE IF ( COVDSC(1) .NE. ' ' .OR.
     .                         COVDSC(2) .NE. ' '        ) THEN
                        CALL ERRCH  ( '#', 'determined by applying ' //
     .                                'constraints specified on the '//
     .                                'command line to the coverage '//
     .                                'obtained from the file'       )
                     ELSE
                        CALL ERRCH  ( '#', 'specified on the '       //
     .                                'command line'                 )
                     END IF
                  ELSE
                     IF      ( COVDSC(1) .NE. ' ' .AND.
     .                         COVDSC(2) .NE. ' '        ) THEN
                        CALL ERRCH  ( '#', 'determined by '          //
     .                                'intersecting coverages '      //
     .                                'obtained from the files'      )
                     ELSE IF ( COVDSC(1) .NE. ' ' .OR.
     .                         COVDSC(2) .NE. ' '        ) THEN
                        CALL ERRCH  ( '#', 'determined from the file')
                     ELSE
C
C                       We can never hit this branch. Set replacement
C                       string to indicate a bug.
C
                        CALL ERRCH  ( '#', 'not determined properly '//
     .                                'due to a bug in the program. '//
     .                                'Please, contact NAIF.'        )
                     END IF
                  END IF
                  CALL ERRCH  ( '#', COVDSC(1)                       )
                  IF ( COVDSC(1) .NE.  COVDSC(2) ) THEN
                     CALL ERRCH  ( '#', COVDSC(2)                    )
                  ELSE
                     CALL ERRCH  ( '#', ' '                          )
                  END IF
                  CALL ERRCH  ( '#', TIMDSC                          )
                  CALL SIGERR ( 'SPICE(STEPTOOSMALL2)'               )
               END IF

            END IF

         ELSE
            CALL SETMSG ( 'Time step ''#'' specified after ''#'' '//
     .                    'key is not a DP number.'               )
            CALL ERRCH  ( '#', CLVALS(I)                          )
            CALL ERRCH  ( '#', STPKEY                             )
            CALL SIGERR ( 'SPICE(NOTANDPNUMBER)'                  )
         END IF

      ELSE

C
C        Step was not provided on the command line. What about the
C        number of steps?
C
         I = ISRCHC( NSTKEY, MAXKEY, CLKEYS )

         IF ( CLFLAG(I) ) THEN

C
C           Is the number of steps an integer number?
C
            CALL NPARSI ( CLVALS(I), NITR, ERROR, PTR )

            IF ( PTR .EQ. 0 ) THEN

               IF ( NITR .LT. MINITR .OR. NITR .GT. MAXITR ) THEN
                  CALL SETMSG ( 'Number of states must be an '       //
     .                          'integer number between # and '      //
     .                          '#. It was #.'                       )
                  CALL ERRINT ( '#', MINITR                          )
                  CALL ERRINT ( '#', MAXITR                          )
                  CALL ERRINT ( '#', NITR                            )
                  CALL SIGERR ( 'SPICE(BADNOFSTATES)'                )
               END IF

            ELSE 
               CALL SETMSG ( 'Number of states ''#'' specified '     //
     .                       'after ''#''  key is not an integer '   //
     .                       'number.'                               )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', NSTKEY                             )
               CALL SIGERR ( 'SPICE(NOTANINTNUMBER)'                 )
            END IF

         ELSE

C
C           For compatibility with the version 1, set number of states
C           to the default number or the number of minimum iterations
C           as was done in the verison 1 if the comparison window is a
C           single interval.
C           
            IF ( WNCARD(CMPWIN) .EQ. 1 ) THEN

               IF ( ET(2) .EQ. ET(1) ) THEN
                  NITR = MINITR
               ELSE
                  NITR = DEFITR
               END IF

            END IF

         END IF

C
C        For compatibility with the version 1, calculate step for this
C        number of steps as was done in the version 1 if the comparison
C        window is a single interval.
C
         IF ( WNCARD(CMPWIN) .EQ. 1 .AND. .NOT. SAMPLE ) THEN

            STEP = ( ET(2) - ET(1) ) / DBLE ( NITR - 1 )
               
         END IF

      END IF

C
C     If any of the dump reports was requested we need to check whether
C     output time format was provided on the command line. Default
C     format is ET seconds.
C
      TIMFMT = ' '

      IF ( EQSTR( DIFTYP, DMPVAL ) .OR. 
     .     EQSTR( DIFTYP, DVFVAL ) .OR. 
     .     EQSTR( DIFTYP, DCVAL  ) .OR. 
     .     EQSTR( DIFTYP, DGVAL  )      ) THEN

         I = ISRCHC( FMTKEY, MAXKEY, CLKEYS )
         
         IF ( CLFLAG(I) ) THEN

            TIMFMT = CLVALS(I)

C
C           In this incarnation of the program we are not going to try
C           to verify that the format is OK at the time when we process
C           command line. We will let TIMOUT fail when it's called for
C           the first time down stream from here. The only thing that
C           makes sense to check for is if the format string is non
C           blank.
C
            IF ( TIMFMT .EQ. ' ' ) THEN
               CALL SETMSG ( 'Output time format specified after '   //
     .                       '''#'' key is blank.'                   )
               CALL ERRCH  ( '#', FMTKEY                             )
               CALL SIGERR ( 'SPICE(BLANKTIMEFORMAT)'                )
            END IF
            
         END IF

      END IF

C
C     If any of the dump reports was requested we need to check whether
C     the number significant digits for output was provided on the
C     command line.
C
      SIGDIG = DEFSDG

      IF ( EQSTR( DIFTYP, DMPVAL ) .OR. 
     .     EQSTR( DIFTYP, DVFVAL ) .OR. 
     .     EQSTR( DIFTYP, DCVAL  ) .OR. 
     .     EQSTR( DIFTYP, DGVAL  )      ) THEN

         I = ISRCHC( DIGKEY, MAXKEY, CLKEYS )

         IF ( CLFLAG(I) ) THEN

C
C           Is the number of digits an integer number?
C
            CALL NPARSI ( CLVALS(I), SIGDIG, ERROR, PTR )

            IF ( PTR .EQ. 0 ) THEN

               IF ( SIGDIG .LT. MINSDG .OR. SIGDIG .GT. MAXSDG ) THEN
                  CALL SETMSG ( 'Number of digits must be an '       //
     .                          'integer number between # and '      //
     .                          '#. It was #.'                       )
                  CALL ERRINT ( '#', MINSDG                          )
                  CALL ERRINT ( '#', MAXSDG                          )
                  CALL ERRINT ( '#', SIGDIG                          )
                  CALL SIGERR ( 'SPICE(BADNOFDIGITS)'                )
               END IF

            ELSE 
               CALL SETMSG ( 'Number of digits ''#'' specified '     //
     .                       'after ''#''  key is not an integer '   //
     .                       'number.'                               )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', DIGKEY                             )
               CALL SIGERR ( 'SPICE(NOTANINTNUMBER2)'                )
            END IF

         END IF

      END IF

C
C     Check out.
C
      CALL CHKOUT( 'CHWCML' )

      END
