C$Procedure      ZZPRSDIR ( Parse a direction specification )
 
      SUBROUTINE ZZPRSDIR ( PUT, DESCRP,
     .                      CORR,  FRAME,  OBS,    TARG,
     .                      A,     B,      C,      XYZ,
     .                      PVN,  METHOD, REFDIR,  BNRYDP )
 
 
C$ Abstract
C
C     SPICE Private routine intended solely for the support of SPICE
C     routines.  Users should not call this routine directly due
C     to the volatile nature of this routine.
C
C     This is a utility program that parses a direction description
C     and places the parsed quantities in the specified
C     variables.  It is useful only in the routine DYNMCK.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     UTILITY
C
C$ Declarations
 
      IMPLICIT NONE
      INCLUDE              'zzdynenm.inc'
      INTEGER               PUT
      CHARACTER*(*)         DESCRP
 
      CHARACTER*(*)         CORR(*)
      CHARACTER*(*)         FRAME(*)
      INTEGER               OBS(*)
      INTEGER               TARG(*)
 
      DOUBLE PRECISION      A(*)
      DOUBLE PRECISION      B(*)
      DOUBLE PRECISION      C(*)
 
      DOUBLE PRECISION      XYZ(3,*)
 
      INTEGER               PVN(*)
      INTEGER               METHOD(*)
      CHARACTER*(*)         REFDIR(2,*)
      DOUBLE PRECISION      BNRYDP(*)
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     PUT        I   Index to use when putting values.
C     AXIS       I   String indicating which axis is being described
C     DESCRP     I   Description of the direction of the specified axis
C     CORR       O   Correction to apply in the position/velocity case
C     FRAME      O   Frame associated with the specified description
C     OBS        O   Observer
C     TARG       O   Target/reference body
C     A          O   Coordinate on surface point on x-axis of target
C     B          O   Coordinate on surface point on y-axis of target
C     C          O   Coordinate on surface point on z-axis of target
C     XYZ(3,2)   O   Rectangular coordinates of a point
C     PVN       O   Component: Position or Velocity
C     METHOD     O   Method to apply when computing position
C     REFDIR     O   Reference directions for binary methods
C     BNRYDP     O   The numeric component associated with binarys
C
C
C$ Detailed_Input
C
C     PUT        indicates whether to fill in the buffers used
C                to hold the parsed description
C
C     AXIS       String specifying the axis the specified direction
C                corresponds to: +X, -X, +Y, -Y, +Z, or -Z are
C                acceptable values.  The sign is required.  The
C                case of the characters does not matter ( +x and +X
C                mean the same thing.
C
C     DESCRP     is the description of the pointing of the specified
C                axis.  DESCRP should match one of the following
C                templates:
C
C                STATE:  POSITION OF @body FROM @body CORRECTION @word
C
C                        VELOCITY OF @body FROM @body CORRECTION @word
C
C                        ORBIT NORMAL OF @body FROM @body
C                        CORRECTION @word
C
C                    These cases describe a direction obtained from
C                    ephemeris information.
C
C
C                FIXED:  FIXED @name XYZ @number @number @number'
C
C                        FIXED @name RADEC RA @number DEC @number
C
C                   These cases describe a direction fixed to some
C                   reference frame.
C
C                SUB:    SUBOBSERVER ON @body FROM @body
C                        CORRECTION @word
C
C                   This case describes a direction determined by the
C                   subobserver location on a reference spheroid.
C
C                NORMAL: NORMAL TO @body AT (2:2){ LATITUDE @number |
C                        LONGITUDE @number }
C
C                        NORMAL TO @body AT (2:2){ LATITUDE @number |
C                        CARTOGRAPHIC LONGITUDE @number }
C
C                        NORMAL TO @body AT (3:3){ X @number | Y @number
C                        | Z @number }
C
C                    These case describe a direction determined by the
C                    normal to a reference ellipsoid.
C
C                COORD: INCREASING (1:1){ LATITUDE
C                                       | LONGTIDUTE
C                                       | CARTOGRAPHIC LONGITUDE
C                                       }
C                               AT (2:2){ LATITUDE @number
C                                       | @option
C                                       | LONGITUDE @number
C                                       | CARTOGRAPHIC LONGITUDE @number
C                                       }  ON @body
C
C                       INCREASING (1:1){ LATITUDE
C                                       | LONGTIDUTE
C                                       | CARTOGRAPHIC LONGITUDE
C                                       }
C                               AT (3:3){ X @number
C                                       | Y @number
C                                       | Z @number
C                                       }  ON @body
C
C                       DECREASING (1:1){ LATITUDE
C                                       | LONGTIDUTE
C                                       | CARTOGRAPHIC LONGITUDE
C                                       }
C                               AT (2:2){ LATITUDE @number
C                                       | @option
C                                       | LONGITUDE @number
C                                       | CARTOGRAPHIC LONGITUDE @number
C                                       }  ON @body
C
C                       DECREASING (1:1){ LATITUDE
C                                       | LONGTIDUTE
C                                       | CARTOGRAPHIC LONGITUDE
C                                       }
C                               AT (3:3){ X @number
C                                       | Y @number
C                                       | Z @number
C                                       }  ON @body
C
C
C                    These case describe a direction determined by the
C                    normal to a reference ellipsoid.
C
C$ Detailed_Output
C
C     CORR       Correction to apply when computing states. The return
C                value is meaningful only if DESCRP matches one of the
C                STATE or SUB templates.  The value should be one of
C                the values supported by SPKEZ.  However, no checks
C                are performed by this routine to make sure this is
C                the case.
C
C     FRAME      Frame associated with the specified description. The
C                return value is meaningful only if DESCP matches one
C                of the FIXED, SUB, NORMAL or COORD templates.  If
C                DESCRP matches one of the FIXED templates FRAME is the
C                frame of the fixed frame.  If DESCRP matches a SUB,
C                NORMAL or COORD template, FRAME is the bodyfixed frame
C                of the ON or TO object specified.
C
C     OBS        Observer. The return value is meaningful only if
C                DESCRP is matches one of the STATE or SUB templates.
C                In these cases OBS corresponds to the body specified
C                by FROM
C
C     TARG       Target or reference body. The return value is
C                meaningful only if DESCRP matches one of the
C                STATE, SUB, NORMAL or COORD templates.  In these cases
C                TARG corresponds to the body specified by OF, ON or
C                TO respectively.
C
C     A          The length of the semi-axis of an ellipsoid along
C                the x-axis of the ellipsoid.  The return value
C                is meaningful only if DESCRP matches the SUB template.
C                In this case A is the semi-axis length of the body
C                specified by ON.
C
C     B          The length of the semi-axis of an ellipsoid along
C                the y-axis of the ellipsoid.  The return value
C                is meaningful only if DESCRP matches the SUB template.
C                In this case A is the semi-axis length of the body
C                specified by ON.
C
C     C          The length of the semi-axis of an ellipsoid along
C                the z-axis of the ellipsoid.  The return value
C                is meaningful only if DESCRP matches the SUB template.
C                In this case A is the semi-axis length of the body
C                specified by ON.
C
C     XYZ        Rectangular coordinates of a direction.  The return
C                value is meaningful only if DESCRP matches  one of the
C                FIXED, NORMAL, or COORD templates.  If DESCRP matches
C                one of the FIXED templates, XYZ will have the frame
C                fixed coordinates of the specified direction.  If
C                DESCRP matches one of the NORMAL templates, XYZ will
C                contain the normal vector. If DESCRP matches one of
C                the COORD templates XYZ will contain the vector tangent
C                to the reference surface that points in the increasing
C                or decreasing direction of the specified coordinate.
C
C     PVN        Component: Position, Velocity or normal to the
C                instantaneous orbital plane. The return value is
C                meaningful only if DESCRP matches one of the STATE
C                templates. If DESCRP matches the POSITION template
C                PVN will have the value specified by the parameter
C                POS.  If DESCRP matches the VELOCITY template, PVN
C                will have the value specified by the parameter VEL.
C                If DESCRP matches the NORMAL template, PVN will have
C                the value specified by the parameter NORM.
C                (See zzckenum.inc for these values.)
C
C     METHOD     Method to apply when computing position.  This depends
C                upon the template matched by DESCRP if DESRP matches
C                The table below illustrates the values returned
C
C                 DESCRP matches     Value returned in
C                 template of        METHOD (See zzckenum.inc)
C                 --------------------------------------------
C                   STATE            SPK
C                   FIXED            FIXED
C                   SUB              SUB
C                   NORMAL           FIXED
C                   COORD            FIXED
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If AXIS is not recognized the error 'SPICE(BADAXISSPEC)'
C        will be signaled.
C
C     2) If DESCRP does not match one of the templates given
C        above, the error 'SPICE(BADDIRECTIONSPEC)' will be signaled.
C
C     3) If DESCRP matches one of the non-STATE templates, a frame
C        must be determined and returned in the variable FRAME.  If
C        the frame cannot be determined (for example the bodyfixed
C        frame of a body is unknown) the error 'SPICE(CANFINDFRAME)'
C        will be signaled.
C
C     4) If DESCRP matches the cartographic longitude template, and the
C        sense of rotation of the specified body cannot be determined,
C        the error 'SPICE(NOROTATIONSENSE)' will be signaled.
C
C     5) The SUB and NORMAL templates refer to objects whose axes
C        must be determined.  The routine BODVAR is used for this
C        determination.  If the axes cannot be located, BODVAR
C        will signal an error indicating the problem.
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This is a utility routine for parsing the description of a
C     direction.  It is only an intermediary routine and not intended
C     for public consumption.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C     B.V. Semenov    (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.2.0, 18-DEC-2013 (BVS)
C
C        BUG FIX: added buffering of the body-fixed frame name for the 
C        increasing/descresing lon/lat cases.
C
C-    SPICELIB Version 1.1.0, 29-APR-2009 (BVS)
C
C        Updated to remove non-standard use of duplicate arguments in
C        CONVRT_3 and VSCL.
C
C-    SPICELIB Version 1.0.2, 26-APR-2006 (BVS)
C
C        Deleted unused variables THISAX and FLIPIT and parameter
C        NAXSPC.
C
C-    SPICELIB Version 1.0.1, 19-JUN-2002 (BVS)
C
C        Fixed the order of inputs in the RADREC call.
C
C-    SPICELIB Version 1.0.0, 15-MAR-2000 (WLT)
C
C        Beware the ides of March.
C
C-&
 
C
C     SPICELib and other functions
C
 
      INTEGER               PLNSNS
      LOGICAL               HAVE
      LOGICAL               M2XIST
      LOGICAL               RETURN
 
      INTEGER               WDSIZE
      PARAMETER           ( WDSIZE = 32 )
 
      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 320 )
 
      INTEGER               ERRSIZ
      PARAMETER           ( ERRSIZ = 1200 )
 
      INTEGER               MAXSYN
      PARAMETER           ( MAXSYN = 17 )
 
      INTEGER               LBCELL
      PARAMETER           ( LBCELL = -5 )
  
      CHARACTER*(WDSIZE)    SYNKEY ( LBCELL: MAXSYN )
      INTEGER               SYNPTR ( LBCELL: MAXSYN )
      CHARACTER*(LNSIZE)    SYNVAL ( LBCELL: MAXSYN )
      INTEGER               NSYN
 
 
      CHARACTER*(ERRSIZ)    ERROR ( 2 )
      CHARACTER*(80)        ULAT
      CHARACTER*(80)        ULON
 
 
      DOUBLE PRECISION      ABC ( 3 )
      DOUBLE PRECISION      CHANGE
      DOUBLE PRECISION      DEC
      DOUBLE PRECISION      DLAT  ( 3 )
      DOUBLE PRECISION      DLONG ( 3 )
      DOUBLE PRECISION      F
      DOUBLE PRECISION      LAT
      DOUBLE PRECISION      LONG
      DOUBLE PRECISION      NORML ( 3 )
      DOUBLE PRECISION      RA
      DOUBLE PRECISION      RE
      DOUBLE PRECISION      RP
      DOUBLE PRECISION      VECTOR ( 3 )
      DOUBLE PRECISION      ZAXIS  ( 3 )
 
      DOUBLE PRECISION      TMPDP
      DOUBLE PRECISION      TVEC3  ( 3 )

      INTEGER               DIM
      INTEGER               FRCODE
      INTEGER               STATUS
 
      LOGICAL               FIRST
      LOGICAL               FOUND
 
      SAVE                  FIRST
      SAVE                  NSYN
      SAVE                  SYNKEY
      SAVE                  SYNPTR
      SAVE                  SYNVAL
      SAVE                  ZAXIS
 
 
      DATA                  FIRST /.TRUE./
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'ZZPRSDIR')
 
      IF ( FIRST ) THEN
 
         FIRST = .FALSE.
 
         SYNVAL ( 1) = 'POSITION[position] OF[spk] @body[target] '
     .   //            'FROM @body[observer] CORRECTION '
     .   //            '@word[corr] '
 
         SYNVAL ( 2) = 'VELOCITY[velocity] OF[spk] @body[target] '
     .   //            'FROM @body[observer] CORRECTION '
     .   //            '@word[corr] '
 
         SYNVAL ( 3) = 'FIXED[fixed] @name[frame] XYZ[xyz] '
     .   //            '@number[x] @number[y] @number[z] '
 
         SYNVAL ( 4) = 'FIXED[fixed] @name[frame]  '
     .   //            'RA @number[ra] (0:1)@unit(degree)[ulon] '
     .   //            'DEC @number[dec] '
     .   //            '(0:1)@unit(degree)[ulat] '
 
         SYNVAL ( 5) = 'SUBOBSERVER[subobserver] ON @body[target] '
     .   //            'FROM @body[observer] CORRECTION @word[corr]'
 
         SYNVAL ( 6) = 'NORMAL[normal] TO @body[body] AT (2:2){ '
     .   //            'LATITUDE @number[lat] '
     .   //            '(0:1)@unit(degree)[ulat] | LONGITUDE '
     .   //            '@number[long] (0:1)@unit(degree)[ulon] } '
 
 
         SYNVAL ( 7) = 'NORMAL[cnormal] TO @body[body] AT (2:2){ '
     .   //            'LATITUDE @number[lat] '
     .   //            '(0:1)@unit(degree)[ulat] | CARTOGRAPHIC '
     .   //            'LONGITUDE @number[long] '
     .   //            '(0:1)@unit(degree)[ulon] } '
 
         SYNVAL ( 8) = 'NORMAL[xyznormal] TO @body[body] AT (3:3){ '
     .   //            'X @number[x] | Y @number[y] | Z @number[z] } '
 
         SYNVAL ( 9) = 'INCREASING[increasing] (1:1){ '
     .   //            'LATITUDE[dlat] | LONGITUDE[dlong] | '
     .   //            'CARTOGRAPHIC LONGITUDE[dclong] } AT '
     .   //            '(2:2){ LATITUDE @number[lat] '
     .   //            '(0:1)@unit(degree)[ulat] | @option | '
     .   //            'LONGITUDE @number[long] '
     .   //            '(0:1)@unit(degree)[ulon] | CARTOGRAPHIC '
     .   //            'LONGITUDE @number[clong] '
     .   //            '(0:1)@unit(degree)[ulon] } ON '
     .   //            '@body[body] '
 
 
         SYNVAL (10) = 'INCREASING[increasing] (1:1){ '
     .   //            'LATITUDE[dlat] | LONGITUDE[dlong] | '
     .   //            'CARTOGRAPHIC LONGITUDE[dclong] } AT '
     .   //            '(3:3){ X @number[x] | Y @number[y] | Z '
     .   //            '@number[z] } ON @body[body] '
 
         SYNVAL (11) = 'DECREASING[decreasing] (1:1){ '
     .   //            'LATITUDE[dlat] | LONGITUDE[dlong] | '
     .   //            'CARTOGRAPHIC LONGITUDE[dclong] } AT '
     .   //            '(2:2){ LATITUDE @number[lat] '
     .   //            '(0:1)@unit(degree)[ulat] | @option | '
     .   //            'LONGITUDE @number[long] '
     .   //            '(0:1)@unit(degree)[ulon] | CARTOGRAPHIC '
     .   //            'LONGITUDE @number[clong] '
     .   //            '(0:1)@unit(degree)[ulon] } ON '
     .   //            '@body[body] '
 
 
 
         SYNVAL (12) = 'DECREASING[decreasing] (1:1){ '
     .   //            'LATITUDE[dlat] | LONGITUDE[dlong] | '
     .   //            'CARTOGRAPHIC LONGITUDE[dclong] } AT '
     .   //            '(3:3){ X @number[x] | Y @number[y] | Z '
     .   //            '@number[z] } ON @body[body] '
 
 
         SYNVAL (13) = 'ORBIT NORMAL[orbnorm] OF[spk] @body[target] '
     .   //            'FROM @body[observer] CORRECTION '
     .   //            '@word[corr] '
 
C============================================================
C        Binary definitions appear below.
C
         SYNVAL(14) =  'ROTATE[rotate] @name[primary] @number[angle] '
     .   //            '(0:1)@unit(degrees)[units] '
     .   //            '(1:1){ TOWARDS[towards] '
     .   //                 '| AWAY[away] FROM } '
     .   //            '@name[secondary] '
 
         SYNVAL(15) = 'CROSS PRODUCT[cross] @name[primary] '
     .   //           '@name[secondary] '
 
         SYNVAL(16) = 'ROTATE[rotate_about] @name[primary] '
     .   //           '@number[angle] (0:1)@unit(degrees)[units] '
     .   //           'ABOUT @name[secondary] '
 
         SYNVAL(17) = 'ROTATE[rotate_fraction] @name[primary] '
     .   //           '@number(0:1)[fraction] OF THE WAY TOWARDS '
     .   //           '@name[secondary] '
 
 
         NSYN = 17
 
 
         ZAXIS(1)  = 0.0D0
         ZAXIS(2)  = 0.0D0
         ZAXIS(3)  = 1.0D0
 
         CALL M2INTS ( NSYN, SYNKEY, SYNPTR, SYNVAL )
 
      END IF
 
      REFDIR(1,PUT) = ' '
      REFDIR(2,PUT) = ' '
 
      CALL M2CHCK ( DESCRP, SYNKEY, SYNPTR, SYNVAL, ERROR )
 
      IF ( HAVE(ERROR) ) THEN
         CALL SETMSG ( ERROR(1) )
         CALL SIGERR ( 'SPICE(BADDIRECTIONSPEC)' )
         CALL CHKOUT ( 'ZZPRSDIR' )
         RETURN
      END IF
 
      ULAT = 'degrees'
      ULON = 'degrees'
 
      CALL M2GETC ( 'ulat', DESCRP, FOUND, ULAT )
      CALL M2GETC ( 'ulon', DESCRP, FOUND, ULON )
 
      IF ( M2XIST( 'spk' ) ) THEN
 
 
         METHOD(PUT) = SPK
 
         IF ( M2XIST('position') ) THEN
            PVN(PUT) = POS
         ELSE IF ( M2XIST('velocity') ) THEN
            PVN(PUT) = VEL
         ELSE IF ( M2XIST('orbnorm') ) THEN
            PVN(PUT) = NORM
         END IF
 
         CALL M2GETB( 'target',   DESCRP, FOUND, TARG(PUT) )
         CALL M2GETB( 'observer', DESCRP, FOUND, OBS(PUT)  )
         CALL M2GETC( 'corr',     DESCRP, FOUND, CORR(PUT) )
 
      ELSE IF ( M2XIST( 'fixed' ) ) THEN
 
         METHOD(PUT) = FIXED
 
         IF ( M2XIST('xyz') ) THEN
            CALL M2GETD ( 'x', DESCRP, FOUND, XYZ(X,PUT) )
            CALL M2GETD ( 'y', DESCRP, FOUND, XYZ(Y,PUT) )
            CALL M2GETD ( 'z', DESCRP, FOUND, XYZ(Z,PUT) )
         ELSE
            CALL M2GETD ( 'ra',  DESCRP, FOUND, RA  )
            CALL M2GETD ( 'dec', DESCRP, FOUND, DEC )
 
            CALL CONVRT_3 ( DEC, ULAT, 'RADIANS', TMPDP, STATUS )
            DEC = TMPDP
            CALL CONVRT_3 ( RA,  ULON, 'RADIANS', TMPDP,  STATUS )
            RA = TMPDP
 
            CALL RADREC ( 1.0D0, RA, DEC, XYZ(1,PUT) )
         END IF
 
         CALL M2GETC ( 'frame', DESCRP, FOUND, FRAME(PUT) )
 
      ELSE IF ( M2XIST( 'subobserver' ) ) THEN
 
         METHOD(PUT) = SUBOBS
 
         CALL M2GETB( 'target',   DESCRP, FOUND, TARG(PUT) )
         CALL M2GETB( 'observer', DESCRP, FOUND, OBS(PUT)  )
         CALL M2GETC( 'corr',     DESCRP, FOUND, CORR(PUT) )
 
         CALL CIDFRM ( TARG(PUT), FRCODE, FRAME(PUT), FOUND )
 
         IF ( .NOT. FOUND ) THEN
            CALL SETMSG ( 'The bodyfixed frame for the object '
     .      //            'with NAIF-ID # could not be '
     .      //            'determined. ' )
            CALL ERRINT ( '#', TARG(PUT) )
            CALL SIGERR ( 'SPICE(CANFINDFRAME)'  )
            CALL CHKOUT ( 'ZZPRSDIR' )
            RETURN
         END IF
 
         CALL BODVAR ( TARG(PUT), 'RADII',DIM, ABC )
 
         A(PUT) = ABC(1)
         B(PUT) = ABC(2)
         C(PUT) = ABC(3)
 
      ELSE IF ( M2XIST( 'normal' ) ) THEN
 
         METHOD(PUT) = FIXED
 
         CALL M2GETB ( 'body', DESCRP, FOUND, TARG(PUT) )
         CALL M2GETD ( 'lat',  DESCRP, FOUND, LAT         )
         CALL M2GETD ( 'long', DESCRP, FOUND, LONG        )
 
         CALL CONVRT_3 ( LAT,   ULAT, 'RADIANS', TMPDP,  STATUS )
         LAT = TMPDP
         CALL CONVRT_3 ( LONG,  ULON, 'RADIANS', TMPDP, STATUS )
         LONG = TMPDP
 
         CALL CIDFRM ( TARG(PUT), FRCODE, FRAME(PUT), FOUND )
 
         IF ( .NOT. FOUND ) THEN
            CALL SETMSG ( 'The bodyfixed frame for the object '
     .      //            'with NAIF-ID # could not be '
     .      //            'determined. ' )
            CALL ERRINT ( '#', TARG(PUT) )
            CALL SIGERR ( 'SPICE(CANFINDFRAME)'  )
            CALL CHKOUT ( 'ZZPRSDIR' )
            RETURN
         END IF
 
         CALL BODVAR ( TARG(PUT), 'RADII',DIM, ABC )
 
         A(PUT) = ABC(1)
         B(PUT) = ABC(2)
         C(PUT) = ABC(3)
 
         RE = ABC(1)
         RP = ABC(3)
         F  = (RE-RP) / RE
 
         CALL GEOREC ( LONG, LAT, 0.0D0,  RE, F, VECTOR )
 
         CALL SURFNM ( ABC(1), ABC(2), ABC(3),  VECTOR, XYZ(1,PUT) )
 
 
      ELSE IF ( M2XIST( 'cnormal' ) ) THEN
 
         METHOD(PUT) = FIXED
 
         CALL M2GETB ( 'body', DESCRP, FOUND, TARG(PUT) )
         CALL M2GETD ( 'lat',  DESCRP, FOUND, LAT         )
         CALL M2GETD ( 'long', DESCRP, FOUND, LONG        )
 
         CALL CONVRT_3 ( LAT,   ULAT, 'RADIANS', TMPDP,  STATUS )
         LAT = TMPDP
         CALL CONVRT_3 ( LONG,  ULON, 'RADIANS', TMPDP, STATUS )
         LONG = TMPDP
 
         CALL CIDFRM ( TARG(PUT), FRCODE, FRAME(PUT), FOUND )
 
         IF ( .NOT. FOUND ) THEN
            CALL SETMSG ( 'The bodyfixed frame for the object '
     .      //            'with NAIF-ID # could not be '
     .      //            'determined. ' )
            CALL ERRINT ( '#', TARG(PUT) )
            CALL SIGERR ( 'SPICE(CANFINDFRAME)'  )
            CALL CHKOUT ( 'ZZPRSDIR' )
            RETURN
         END IF
 
         CALL BODVAR ( TARG(PUT), 'RADII',DIM, ABC )
 
         A(PUT) = ABC(1)
         B(PUT) = ABC(2)
         C(PUT) = ABC(3)
 
         RE = ABC(1)
         RP = ABC(3)
         F  = (RE-RP) / RE
 
         CHANGE = PLNSNS ( TARG(PUT) )
 
         IF ( CHANGE .EQ. 0.0D0 ) THEN
            CALL SETMSG ( 'The sense of rotation of body # could '
     .      //            'not be determined. Since the mapping '
     .      //            'between planetocentric coordinates '
     .      //            'and rectangular coordinates is '
     .      //            'unknown, the supplied direction '
     .      //            'specification for the normal vector '
     .      //            'is ambiguous. ' )
            CALL ERRINT ( '#', TARG(PUT) )
            CALL SIGERR ( 'SPICE(NOROTATIONSENSE)'  )
            CALL CHKOUT ( 'ZZPRSDIR' )
            RETURN
         END IF
 
         LONG = CHANGE*LONG
 
         CALL GEOREC ( LONG, LAT, 0.0D0,  RE, F,  VECTOR )
         CALL SURFNM (  ABC(1), ABC(2), ABC(3), VECTOR, XYZ(1,PUT) )
 
      ELSE IF ( M2XIST( 'xyznormal' ) ) THEN
 
         METHOD(PUT) = FIXED
 
         CALL M2GETB ( 'body', DESCRP, FOUND, TARG(PUT) )
         CALL M2GETD ( 'x',    DESCRP, FOUND, VECTOR(X) )
         CALL M2GETD ( 'y',    DESCRP, FOUND, VECTOR(Y) )
         CALL M2GETD ( 'z',    DESCRP, FOUND, VECTOR(Z) )
 
         CALL CIDFRM ( TARG(PUT), FRCODE, FRAME(PUT), FOUND )
 
         IF ( .NOT. FOUND ) THEN
            CALL SETMSG ( 'The bodyfixed frame for the object '
     .      //            'with NAIF-ID # could not be '
     .      //            'determined. ' )
            CALL ERRINT ( '#', TARG(PUT) )
            CALL SIGERR ( 'SPICE(CANFINDFRAME)'  )
            CALL CHKOUT ( 'ZZPRSDIR' )
            RETURN
         END IF
 
         CALL BODVAR ( TARG(PUT), 'RADII',DIM, ABC )
 
         A(PUT) = ABC(1)
         B(PUT) = ABC(2)
         C(PUT) = ABC(3)
 
         CALL SURFNM ( ABC(1), ABC(2), ABC(3), VECTOR, XYZ(1,PUT) )
 
      ELSE IF (     M2XIST( 'increasing' )
     .         .OR. M2XIST( 'decreasing' ) ) THEN
 
         METHOD(PUT) = FIXED
         CALL M2GETB ( 'body', DESCRP, FOUND, TARG(PUT) )
 
         CALL CIDFRM ( TARG(PUT), FRCODE, FRAME(PUT), FOUND )
 
         IF ( .NOT. FOUND ) THEN
            CALL SETMSG ( 'The bodyfixed frame for the object '
     .      //            'with NAIF-ID # could not be '
     .      //            'determined. ' )
            CALL ERRINT ( '#', TARG(PUT) )
            CALL SIGERR ( 'SPICE(CANFINDFRAME)'  )
            CALL CHKOUT ( 'ZZPRSDIR' )
            RETURN
         END IF
 
         CALL BODVAR ( TARG(PUT), 'RADII',DIM, ABC )
 
         A(PUT) = ABC(1)
         B(PUT) = ABC(2)
         C(PUT) = ABC(3)
 
         RE = ABC(1)
         RP = ABC(3)
         F  = (RE-RP) / RE
C
C        If we need it, get the sense of rotation for the body
C        now.
C
         IF ( M2XIST ( 'dclong' ) .OR. M2XIST( 'clong' ) ) THEN
            CHANGE = PLNSNS( TARG(PUT) )
 
            IF ( CHANGE .EQ. 0.0D0 ) THEN
               CALL SETMSG ( 'The sense of rotation of body # '
     .         //            'could not be determined. Since '
     .         //            'the mapping between '
     .         //            'planetocentric coordinates and '
     .         //            'rectangular coordinates is '
     .         //            'unknown, the supplied direction '
     .         //            'specification for the normal '
     .         //            'vector is ambiguous.' )
               CALL ERRINT ( '#', TARG(PUT) )
               CALL SIGERR ( 'SPICE(NOROTATIONSENSE)'  )
               CALL CHKOUT ( 'ZZPRSDIR' )
               RETURN
            END IF
         ELSE
            CHANGE = 1.0D0
         END IF
 
         IF ( M2XIST( 'x' ) ) THEN
 
            CALL M2GETD ( 'x',    DESCRP, FOUND, VECTOR(X) )
            CALL M2GETD ( 'y',    DESCRP, FOUND, VECTOR(Y) )
            CALL M2GETD ( 'z',    DESCRP, FOUND, VECTOR(Z) )
 
         ELSE
 
            CALL M2GETD ( 'lat',   DESCRP, FOUND, LAT  )
            CALL M2GETD ( 'long',  DESCRP, FOUND, LONG )
            CALL M2GETD ( 'clong', DESCRP, FOUND, LONG )
 
            CALL CONVRT_3 ( LAT,   ULAT, 'RADIANS', TMPDP, STATUS )
            LAT = TMPDP
            CALL CONVRT_3 ( LONG,  ULON, 'RADIANS', TMPDP, STATUS )
            LONG = TMPDP
 
            LONG = LONG*CHANGE
            CALL GEOREC ( LONG, LAT, 0.0D0,  RE, F,  VECTOR )
 
         END IF
 
         CALL SURFNM (  ABC(1), ABC(2), ABC(3), VECTOR, NORML )
         CALL UCRSS  ( ZAXIS,  NORML, DLONG )
         CALL UCRSS  ( NORML,  DLONG, DLAT  )
         CALL VSCL   ( CHANGE, DLONG, TVEC3 )
         CALL VEQU   ( TVEC3, DLONG )
 
         IF ( M2XIST ( 'decreasing' ) ) THEN
            CALL VSCL ( -1.0D0, DLAT,  TVEC3 )
            CALL VEQU ( TVEC3, DLAT )
            CALL VSCL ( -1.0D0, DLONG, TVEC3 )
            CALL VEQU ( TVEC3, DLONG )
         END IF
 
         IF ( M2XIST( 'dlat' ) )THEN
            CALL VEQU ( DLAT,  XYZ(1,PUT) )
         ELSE
            CALL VEQU ( DLONG, XYZ(1,PUT) )
         END IF
 
      ELSE IF ( M2XIST('rotate') ) THEN
 
         METHOD(PUT) = ROTTWD
 
         CALL M2GETC ( 'primary',   DESCRP, FOUND, REFDIR(1,PUT) )
         CALL M2GETC ( 'secondary', DESCRP, FOUND, REFDIR(2,PUT) )
 
         CALL M2GETD ( 'angle',     DESCRP, FOUND, LONG )
         CALL M2GETC ( 'units',     DESCRP, FOUND, ULON )
         CALL CONVRT_3 ( LONG,  ULON, 'RADIANS', BNRYDP(PUT), STATUS )
 
         IF ( M2XIST( 'away' ) ) THEN
            BNRYDP(PUT) = -BNRYDP(PUT)
         END IF
 
      ELSE IF ( M2XIST('cross')  ) THEN
 
         METHOD(PUT) = XPRDCT
 
         CALL M2GETC ( 'primary',   DESCRP, FOUND, REFDIR(1,PUT) )
         CALL M2GETC ( 'secondary', DESCRP, FOUND, REFDIR(2,PUT) )
         BNRYDP(PUT) = 0.0D0
 
      ELSE IF ( M2XIST('rotate_about') ) THEN
 
         METHOD(PUT) = ROTABT
 
         CALL M2GETC ( 'primary',   DESCRP, FOUND, REFDIR(1,PUT) )
         CALL M2GETC ( 'secondary', DESCRP, FOUND, REFDIR(2,PUT) )
 
         CALL M2GETD ( 'angle',     DESCRP, FOUND, LONG  )
         CALL M2GETC ( 'units',     DESCRP, FOUND, ULON )
         CALL CONVRT_3 ( LONG,  ULON, 'RADIANS', BNRYDP(PUT), STATUS )
 
       ELSE IF ( M2XIST('rotate_fraction') ) THEN
 
         METHOD(PUT) = FSECT
 
         CALL M2GETC ( 'primary',   DESCRP, FOUND, REFDIR(1,PUT) )
         CALL M2GETC ( 'secondary', DESCRP, FOUND, REFDIR(2,PUT) )
 
         CALL M2GETD ( 'fraction',  DESCRP, FOUND, BNRYDP(PUT)   )

      END IF
 
      CALL CHKOUT ( 'ZZPRSDIR' )
      RETURN
      END
