C$Procedure      ZZGETCAV ( Get C-matrix and angular velocity )
 
      SUBROUTINE ZZGETCAV ( CHKNAM, SPICE, CAVSRC, ET, CMATRX, AV, BAD )
 
C$ Abstract
C
C     SPICE Private routine intended solely for the support of SPICE
C     routines.  Users should not call this routine directly due
C     to the volatile nature of this routine.
C
C     Look up the c-matrix and angular velocity using a supplied
C     routine (CAVSRC) and handle all associated exception handling.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     UTILITY
C
C$ Declarations
 
      IMPLICIT NONE
 
      CHARACTER*(*)         CHKNAM
      LOGICAL               SPICE
      EXTERNAL              CAVSRC
      DOUBLE PRECISION      ET
      DOUBLE PRECISION      CMATRX ( 3, 3 )
      DOUBLE PRECISION      AV     ( 3 )
      LOGICAL               BAD
  
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     CHKNAM     I   Name to use for check in and check out
C     SPICE      I   Tells us the source of the routine CAVSRC
C     CAVSRC     I   A routine that computes C-matrix and AV.
C     ET         I   Epoch of interest
C     CMATRX     O   C-matrix at ET
C     AV         O   Angular Velocity at ET.
C     BAD        O   True if an exception is signalled.
C
C$ Detailed_Input
C
C     CHKNAM     is a name to use for check in/out
C
C     SPICE      is a logical that tells us the source or the routine
C                CAVSRC.  If we are confident that CAVSRC participates
C                correctly in SPICE exception handling, SPICE should
C                be set to TRUE (it saves some time).  If on the other
C                hand CAVSRC has been contributed by an untried user,
C                SPICE should be set to .FALSE. so as to isolate the
C                routine CAVSRC inside special exception handling
C                code.
C
C     CAVSRC     is a routine that given an ET will produce Q and AV.
C                The calling sequence should be:
C
C                   SUBROUTINE CAVSRC ( ET, CMATRX, AV )
C
C                the only input is ET.  The outputs are the CMATRX
C                and AV (C-matrix and Angular Velocity)
C                to associate with the structure specified by PLTFRM
C                relative to the reference frame specified by REF.
C
C                It is probably a good design practice to create an
C                umbrella routine that contains an initialization
C                entry point and the entry point that performs the
C                function of CAVSRC.  In this way, you can set up any
C                particular parameters needed by CAVSRC prior to calling
C                this routine.  When CAVSRC is called by FITCK3, it
C                can simply look up these parameters from the variables
C                shared by the initialization entry point and the
C                entry point corresponding to CAVSRC.
C
C     ET         is a standard Epoch in seconds past J2000
C
C$ Detailed_Output
C
C     CMATRX    is the C-matrix at ET
C
C     AV        is the angular velocity at ET
C
C     BAD       is a flag that inidicates that exceptions were
C               signalled.  (It's just a bit faster than calling
C               FAILED() a second time in FITCK3.)
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     1)
C
C$ Particulars
C
C     This routine is essentially a macro routine used by fitck3
C     so that the code for that routine will be more readable.  It's
C     not intended for general use.
C
C$ Examples
C
C     See fitck3.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 02-AUG-2000 (WLT)
C
C
C-&
 
      LOGICAL               FAILED
 
 
      INTEGER               LONGSZ
      PARAMETER           ( LONGSZ = 1760 )
 
      INTEGER               WDSIZE
      PARAMETER           ( WDSIZE = 32 )
 
      CHARACTER*(WDSIZE)    ACTION
      CHARACTER*(WDSIZE)    DEVICE
      CHARACTER*(WDSIZE)    SSTR
      CHARACTER*(WDSIZE)    SHORT
      CHARACTER*(LONGSZ)    MESSG
      
      DOUBLE PRECISION      COPY
      DOUBLE PRECISION      DIFF
      
      LOGICAL               FIRST
      SAVE                  FIRST
 
      DATA                  FIRST / .TRUE. /
 

C
C     Copy the input ET so that we can be sure this routine does not
C     corrupt it.
C
      COPY = ET
      BAD  = .FALSE.
 
 
      CALL CHKIN  ( CHKNAM )
 
      IF ( SPICE ) THEN
 
         CALL CAVSRC ( COPY, CMATRX, AV )
         BAD =  FAILED()
         CALL CHKOUT ( CHKNAM )
         RETURN
 
      END IF
C
C     Set up the error handling to give this routine local
C     control of message before aborting.
C
      IF ( FIRST ) THEN
         FIRST = .FALSE.
         CALL ERRACT ( 'GET',  ACTION  )
         CALL ERRDEV ( 'GET',  DEVICE  )
      END IF
 
      
C      
C     The only way to get here is through an untested C-matrix
C     evaluation routine. Take control of exception handling so that we
C     can diagnose problems with the supplied routine CAVSRC.
C
      CALL ERRACT ( 'SET', 'RETURN' )
      CALL ERRDEV ( 'SET', 'NULL'   )
C
C     Evaluate the C-matrix.
C
      CALL CAVSRC ( COPY, CMATRX, AV )
 
      IF ( FAILED() ) THEN
 
         BAD = .TRUE.
 
         CALL GETLMS ( MESSG )
         CALL GETSMS ( SHORT  )
 
         CALL RESET
 
         CALL ERRACT ( 'SET', ACTION )
         CALL ERRDEV ( 'SET', DEVICE )
 
         CALL SUFFIX ( 'The input value of ET to CAVSRC was # '
     .   //            '(#).', 8, MESSG )
 
         CALL ETCAL ( ET, SSTR )
 
         CALL RESET
         CALL SETMSG ( MESSG )
         CALL ERRDP  ( '#', ET )
         CALL ERRCH  ( '#', SSTR   )
         CALL SIGERR ( SHORT       )
         CALL CHKOUT ( CHKNAM      )
         RETURN
 
      END IF
 
      CALL ERRACT ( 'SET', ACTION )
      CALL ERRDEV ( 'SET', DEVICE )
 
      IF ( ET .NE. COPY ) THEN
 
         BAD = .TRUE.
 
         CALL ERRACT ( 'SET', ACTION )
         CALL ERRDEV ( 'SET', DEVICE )
 
         DIFF = ET - COPY
 
         CALL SETMSG ( 'The external routine "CAVSRC" passed '
     .   //            'to FITCK3 modified the input value of '
     .   //            'ET.  The original value of ET '
     .   //            'was #. It was modified to #. The '
     .   //            'difference in these values is #. ' )
         CALL ERRDP  ( '#', ET     )
         CALL ERRDP  ( '#', COPY   )
         CALL ERRDP  ( '#', DIFF   )
         CALL SIGERR ( 'SPICE(BADPASSEDROUTINE)'  )
         CALL CHKOUT ( CHKNAM )
         RETURN
 
      END IF
 
      CALL CHKOUT ( CHKNAM )
      RETURN
 
C$Procedure      ZZRESET ( Reset the "FIRST" flag )
 
      ENTRY ZZRESET ()
 
C$ Abstract
C
C     SPICE Private routine intended solely for the support of SPICE
C     routines.  Users should not call this routine directly due
C     to the volatile nature of this routine.
C
C     This entry point allows FITCK3 to re-initialize the FIRST
C     flag used by the main entry point, so that the routine will
C     re-fetcth the current exception handling status.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     UTILITY
C
C$ Declarations
C
C
C$ Brief_I/O
C
C     None.
C
C$ Detailed_Input
C
C     None.
C
C     
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C     1)
C
C$ Particulars
C
C     This routine gives control to FITCK3 so that it may force
C     the main entry point (ZZGETCAV) to look up the current
C     exception handling mode/status.
C
C$ Examples
C
C     See FITCK3
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 02-AUG-2000 (WLT)
C
C
C-&
 
 
 
      FIRST = .TRUE.
      RETURN

      END




