C$Procedure      QAVGEN ( Quaternion and Angular velocity generator )
 
      SUBROUTINE QAVGEN ( ET, ROT, AV, RELTO, DYNMID, CKID )
 
C$ Abstract
C
C     This is an umbrella routine that wrap calls to the dynamic
C     frame transformation routine.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME
C
C$ Declarations
 
      IMPLICIT NONE
      DOUBLE PRECISION      ET
      DOUBLE PRECISION      ROT( 3, 3 )
      DOUBLE PRECISION      AV( 3)
      CHARACTER*(*)         RELTO
      INTEGER               DYNMID
      INTEGER               CKID
 
 
C$ Brief_I/O
C
C     VARIABLE  I/O  ENTRY POINT / DESCRIPTION
C     --------  ---  --------------------------------------------------
C     ET         I   DYNMCK / Epoch of interest
C     ROT        O   DYNMCK / Rotation at epoch
C     AV         O   DYNMCK / Angular Velocity at epoch
C     DYNMID     I   SOLV4  / Id-code for dynamic frame when "solving"
C     CKID       I   SOLV4  / Id-code for CK frame when "solving"
C     RELTO      I   DRELTO / Reference frame Q is relative to
C
C     The function returns
C
C$ Detailed_Input
C
C     See Entry Points
C
C$ Detailed_Output
C
C     See Entry Points.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     1) If this umbrella routine is called directly the error
C        'SPICE(BOGUSENTRY)' will be signalled.
C
C$ Particulars
C
C     This is an umbrella routine for the two entry points DYNMCK
C     and DRELTO which manage calls to the dynamic frames generation
C     routine DYNMFR.
C
C     The entry point DRELTO is used to set the frame that quaternions
C     are with respect to.
C
C     The entry point DYNMCK returns Quaternions and angular velocities
C     associated with the dynamic frame at user specified epochs (ET).
C
C$ Examples
C
C     See the program prediCkt.pgm.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 21-MAR-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Manage predict quaternion production
C
C-&
      CHARACTER*(64)        FROM
      SAVE                  FROM
 
      CHARACTER*(16)        TO

      DOUBLE PRECISION      AT
      DOUBLE PRECISION      XFORM ( 6, 6 )
      DOUBLE PRECISION      XFORM2( 6, 6 )
      DOUBLE PRECISION      FACTOR( 6, 6 )
      DOUBLE PRECISION      EPOCH
 
      LOGICAL               FIXED
 
      INTEGER               TOID
      INTEGER               SLV4ID
      LOGICAL               SOLVE
 
      SAVE                  TOID
      SAVE                  SLV4ID
      SAVE                  SOLVE
 
      SAVE                  EPOCH
      SAVE                  FIXED
 
      DATA                  FIXED / .FALSE. /
      DATA                  SOLVE / .FALSE. /
 
      CALL CHKIN  ( 'QAVGEN' )
      CALL SETMSG ( 'A call has been made to the umbrella '
     .//            'routine QAVGEN.  This indicates a '
     .//            'programming bug.  You should probably '
     .//            'abandon use of the program that issued this '
     .//            'message. ')
      CALL SIGERR ( 'SPICE(BOGUSENTRY)'  )
      CALL CHKOUT ( 'QAVGEN' )
      RETURN
 
 
C$Procedure DYNMCK ( Dynamic CK )
 
      ENTRY DYNMCK ( ET, ROT, AV )
 
C$ Abstract
C
C     Given ET, return the C-matrix and angular velocity associated
C     with a dynamic frame used to specify a C-kernel
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     CK, FRAME
C
C$ Declarations
C
C
C     DOUBLE PRECISION      ET
C     DOUBLE PRECISION      ROT(3,3)
C     DOUBLE PRECISION      AV ( 3 )
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     ET         I   Epoch of interest
C     ROT        O   Rotation corresponding to state transformation
C     AV         O   Angular velocity corresponding to state transform
C
C$ Detailed_Input
C
C     ET         is the epoch for which one wants to retrieve a
C                quaternion and angular velocity associated with
C                some structure.
C
C$ Detailed_Output
C
C     Q          is the rotation associated with the transformation
C                that maps from the reference frame to the instrument
C                frame.
C
C     AV         is the angular velocity associated with ROT.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     1) Errors are diagnosed by the routine DYNMXF.
C
C$ Particulars
C
C     This routine returns rotations and angular velocities for
C     some platform whose orientation can be predicted through
C     use of the routine DYNMFR.  It is intended only for use
C     by the program prediCkt.
C
C$ Examples
C
C     See the program prediCkt.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 21-MAR-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Generate Quaternion and Angular velocity
C
C-&
 
 
      TO   = 'DYNAMIC'
 
      IF ( FIXED ) THEN
         AT = EPOCH
      ELSE
         AT = ET
      END IF
 
      CALL DYNMXF ( AT, FROM, TO,  XFORM )
 
      IF ( SOLVE ) THEN
C
C        Recall XFORM transfrom the frame "FROM" to the frame "TO".
C        We want the transformation that goes from "FROM" to the
C        frame associated the structure whose CK id is SLV4ID.
C
C        To get there we get the transformation from TO to SOLV4
C        and then complete the tranformation by combining XFORM
C        with the aditional factor TO-to-SOLV4
C
         CALL FRMCHG ( TOID,   SLV4ID,  AT,    FACTOR )
         CALL MXMG   ( FACTOR, XFORM, 6, 6, 6, XFORM2 )
         CALL XF2RAV ( XFORM2, ROT,   AV )
      ELSE
         CALL XF2RAV( XFORM, ROT, AV )
      END IF
 
 
      IF ( FIXED ) THEN
         AV(1) = 0.0D0
         AV(2) = 0.0D0
         AV(3) = 0.0D0
      END IF
 
 
      RETURN
 
 
C$Procedure DRELTO ( Set the frame for quaternions )
 
      ENTRY DRELTO ( RELTO )
 
C$ Abstract
C
C     Set the frame to which quaternions and angular velocities are
C     relative.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME, CK
C
C$ Declarations
C
C     RELTO
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     RELTO      I   the name of a reference frame recognized by SPICE
C
C$ Detailed_Input
C
C     RELTO      is the name of the frame whose ID-code will be stored
C                in the frame variable of a C-kernel segment
C                descriptor.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C$ Particulars
C
C     This entry point is used to set the reference frame of the
C     Quaternion and angular velocity returned by DYNMCK.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 21-MAR-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Set the Quaternion frame
C
C-&
 
 
      FROM = 'FRAME ' // RELTO
      RETURN
 
C$Procedure      USEVAR ( Use a time varying frame interpretation )
 
      ENTRY USEVAR ( )
 
C$ Abstract
C
C     Set the interpretation of the frame to be time varying
C     with respect to its "relative" frame.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME
C
C$ Declarations
C
C
C$ Brief_I/O
C
C     None
C
C     The function returns
C
C$ Detailed_Input
C
C     None
C
C$ Detailed_Output
C
C     None
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C$ Particulars
C
C     This entry point allows you to set the interpretation of the
C     frame to be one that varies with time with respect to
C     its base frame.  This is the default behavior of the
C     routine DYNMCK.  This routine should be called when you
C     want to make sure that the frame is interpreted as changing
C     w.r.t time.
C
C$ Examples
C
C     See SETFRM
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 05-APR-2000 (WLT)
C
C
C-&
 
      FIXED = .FALSE.
      RETURN
 
 
C$Procedure USEFXD ( Use a fixed epoch in the definition of a frame )
 
      ENTRY USEFXD ( ET )
 
C$ Abstract
C
C     Modify the behavior of DYNMCK so that the frame is regarded
C     as fixed with respect to its base over time.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME
C
C$ Declarations
C
C     DOUBLE PRECISION      ET
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     ET         I   is the epoch at which to fix the frame
C
C$ Detailed_Input
C
C     ET         is the epoch at which the relationship between
C                the dynamic frame and its underlying base (RELTO)
C                frame is fixed.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C$ Particulars
C
C     This routine allows you to specify that the relationship between
C     a dynamic frame and its base (RELTO) frame is constant for
C     all time and that the rotation between them is the same
C     as the time varying relationship at the epoch ET.
C     The angular velocity of the dynamic frame w.r.t its base
C     will be regarded as zero.
C
C$ Examples
C
C     See SETFRM.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 5-APR-2000 (WLT)
C
C
C-&
 
 
      FIXED = .TRUE.
      EPOCH = ET
      RETURN
 
C$Procedure      SOLV4 ( Solve for the Rotation and Angular Velocity )
 
      ENTRY SOLV4 ( DYNMID, CKID )
 
C$ Abstract
C
C     Set up the information so that the rotation and angular velocity
C     returned are the ones that solve the equation
C
C     Xform(base-to-dynamic) = Xform(ck-to-todynamic)*Xform(base-to-ck)
C
C     Where DYNMID is the id-code some frame whose relationship to the
C     Ck frame is known and available as a function of time through the
C     SPICE frames system.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME
C
C$ Declarations
C
C     IMPLICIT NONE
C     INTEGER               DYNMID
C     INTEGER               CKID
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     DYNMID     I   Id-code of some frame aligned with dynamic frame
C     CKID       I   Is the Id-code that DYNMID is offset from.
C
C     The function returns
C
C$ Detailed_Input
C
C     DYNMID     is the Id-code of some frame that should be aligned
C                with the "dynamic" frame, but whose relationship
C                in time with respect to some other frame (specified
C                by CKID) is known.
C
C     CKID       is the Id-code of a frame whose orientation is to
C                be solved for by specifyign that the frame with
C                Id-code DYNMID should always be aligned with the
C                "dynamic frame."
C
C$ Detailed_Output
C
C     None.      All output is through the companion entry point
C                DYNMCK.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C$ Particulars
C
C     This entry point allows you to tell the companion entry point
C     DYNMCK that it should solve for the implicit orientation of the
C     C-kernel associated with CKID by first finding the orientation
C     of the dynamic frame (that is specified by DYNMID) and then
C     using the known relationship between the dynamic frame and
C     the C-kernel frame to construct the associated CK orientation.
C
C$ Examples
C
C     See the routine SETFRM in the prediCkt package.
C
C$ Restrictions
C
C     Don't even start down this path unless you know what you're doing.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 14-AUG-2000 (WLT)
C
C
C-&
 
      TOID   = DYNMID
      SLV4ID = CKID
      SOLVE  = .TRUE.
 
      RETURN
 
C$Procedure      NOSOLV ( No solution for ROT and AV is required )
 
      ENTRY NOSOLV ()
 
C$ Abstract
C
C     This entry point tells the entry point DYNMCK that the
C     Rotation and AV to return are the explicit rotation and AV
C     not something to be determined indirectly through a
C     known relationship between two frames.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     KEYWORD
C
C$ Declarations
C
C     None.
C
C
C$ Brief_I/O
C
C     None
C
C$ Detailed_Input
C
C     None.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C$ Particulars
C
C     This routine makes it explicit that no solution should be
C     attempted when computing ROT and AV returned from DYNMCK.
C
C$ Examples
C
C     See SETFRM
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 14-AUG-2000 (WLT)
C
C
C-&
 
      SOLVE = .FALSE.
 
      END
