C$ PROCEDURE     EVALDR ( Evaluate Direction )
C
      SUBROUTINE EVALDR ( DIRCTN, AT,  RELTO, ZEROV, STATE )
C
C$ Abstract
C
C     Evaluate the names direction at the specified epoch relative to
C     the specified reference frame.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     DIRECTION
C
C$ Declarations
 
      IMPLICIT NONE
      CHARACTER*(*)         DIRCTN
      DOUBLE PRECISION      AT
      CHARACTER*(*)         RELTO
      LOGICAL               ZEROV
      DOUBLE PRECISION      STATE ( 6 )
 
 
      INTEGER               LIMIT
      PARAMETER           ( LIMIT = 40 )
 
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     DIRCTN     I   Name of the direction to evaluate
C     AT         I   Epoch at which to evaluate the direction
C     RELTO      I   Name of the reference frame for returned state
C     KIND       I   Kind of evaluation: 1 for position, 2 for state
C     STATE      O   Requested direction (and derivative if requested)
C     LIMIT      P   is the maximum "stack depth" available
C
C$ Detailed_Input
C
C     DIRCT     is the name of some direction for which a 3(or 6) vector
C               is desired.
C
C     AT        is the epoch at which to evaluate the direction in TDB
C               seconds past the epoch of J2000.
C
C     RELTO     is the reference frame in which the returned direction
C               (or direction and derivative) should be returned.
C
C     ZEROV     is a logical which if true, causes the EVALDR to return
C               zeros for the velocity component of STATE.  If FALSE
C               STATE(4...6) will be the derivative of STATE(1...3).
C
C$ Detailed_Output
C
C     STATE     is a unit direction vector ( STATE(1...3) ) and its
C               derivative ( STATE(4...6) ) for the specified direction
C               at the specified epoch, relative to the frame RELTO.
C               Note that if ZEROV is set to TRUE, STATE(4...6) will
C               be the zero vector.
C
C$ Parameters
C
C     LIMIT     is the maximum depth allowed in the stack used to
C               manage the computation of STATE.  In practice
C               it is not expected that a direction will require
C               references to more than 2 other directions. In some
C               rare cases, it is conceivable that as many as
C               6 primitive directions may be required.  If anyone
C               ever hits the current limit, they should be awarded
C               some kind of prize.
C
C$ Files
C
C     None.
C
C$ Exceptions
C
C           (1) If the stack becomes full, this routine will attempt
C               to determine if a recursive direction definition
C               has been supplied.  If so the routine will signal
C               the error 'SPICE(RECURSIVEDEFINITION)'
C
C           (2) If the stack becomes full, and a recursive definition
C               is not the cause, the routine will signal the
C               error 'SPICE(STACKEXHAUSTED)'  and the user will
C               be issued a dubious achievement award.
C
C           (3) All other exceptions (such as not enough ephemeris
C               data available, no PCK data available, etc. ) are
C               diagnosed by routines called by this one.
C
C$ Particulars
C
C     This routine manages the task of computing direction vectors from
C     the various descriptions supplied to the direction manager.
C     Computations are handled through a "direction stack" that
C     resolves directions defined in terms of other directions.
C
C     Because the stack is implemented in this routine, we can avoid
C     recursion.  However, there is a small price to pay.  It is
C     assumed that a stack of size LIMIT is sufficient to handle
C     all reasonably defined directions.  If you find the error
C     'SPICE(STACKEXHAUSTED)' showing up, you can fix the problem
C     by increasing LIMIT (defined above) and rebuilding this routine
C     and the calling application.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Literature_References
C
C     None.
C
C$ Version
C
C-    SPICELIB Version 1.1.0, 6-JUL-2001 (WLT)
C
C-       Added mispelling diagnostics in the event a direction name
C        is not recognized.
C
C-    SPICELIB Version 1.0.0, 20-JUL-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Compute a direction vector.
C
C-&
 
C
C     Spicelib Functions
C
      LOGICAL               RETURN
C
C     The following set of parameters ( NULL, ... COMBIN ) are used to
C     control the actions of the IF THEN block inside the evaluation
C     loop.
C
      INTEGER               NULL
      PARAMETER           ( NULL = 0 )
 
      INTEGER               FETCH1
      PARAMETER           ( FETCH1 =  1 )
 
      INTEGER               FETCH2
      PARAMETER           ( FETCH2 = 2 )
 
      INTEGER               COMBIN
      PARAMETER           ( COMBIN = 3 )
 
C
C     The parameter INTLIM is  the internal limit used to declare the
C     direction stack.  By making it one more than LIMIT we gain
C     some ease in coding the various checks while avoiding overflow
C     of the various arrays to implement the stack.
C
      INTEGER               INTLIM
      PARAMETER           ( INTLIM = LIMIT + 1 )
 
 
 
 
      INTEGER               WDSIZE
      PARAMETER           ( WDSIZE = 32 )
C
C     The next three arrays together with the integer DEPTH make up the
C     direction stack and current location within the stack.
C
      CHARACTER*(WDSIZE)    NAME  (    INTLIM )
      CHARACTER*(WDSIZE)    GUESS
      DOUBLE PRECISION      DDIR  ( 6, INTLIM )
      INTEGER               DONEXT(    INTLIM )
      INTEGER               DEPTH
      INTEGER               HANDLE
 
C
C     Utilify Variables
C
      DOUBLE PRECISION      DIR ( 6 )
      INTEGER               K
      LOGICAL               PRMTIV
      LOGICAL               FOUND
      LOGICAL               FNDG
 
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'EVALDR')
 
C
C     Seed the stack with the first direction name.
C
      DEPTH         = 1
      NAME  (DEPTH) = DIRCTN
      DONEXT(DEPTH) = FETCH1
C
C     Zeroing out the first column of DDIR is NOT necessary, we do
C     it for the sake of making the changes in this buffer easy
C     to see during debuggin (should that ever be necessary)
C     Once this routine is believed to be flawless, (or if you just
C     need the speed) feel free to comment out the DO LOOP below.
C
      DO K = 1, 6
         DDIR ( K, 1 ) = 0.0D0
      END DO
 
      DO WHILE ( DEPTH .GT. 0 )
 
C
C        Determine how this direction is supposed to be evaluated.
C
C
         CALL GETDRM ( NAME(DEPTH), PRMTIV, HANDLE, FOUND )
C
C        Make sure the requested direction is a known direction.
C
         IF ( .NOT. FOUND ) THEN
 
            IF ( DEPTH .GT. 1 ) THEN

               CALL DRDIAG ( NAME(DEPTH), GUESS, FNDG ) 
               CALL SETMSG ( 'The direction ''#'' is defined '
     .         //            'implicitely in terms of a '
     .         //            'direction named ''#''.  However, '
     .         //            'the direction ''#'' is not on the '
     .         //            'list of known directions. #' )
               CALL ERRCH  ( '#', DIRCTN )
               CALL ERRCH  ( '#', NAME(DEPTH) )
               CALL ERRCH  ( '#', NAME(DEPTH) )
               
            ELSE

               CALL DRDIAG ( NAME(DEPTH), GUESS, FNDG ) 
          
               CALL SETMSG ( 'The direction ''#'' has not been '
     .         //            'defined. #' )
               CALL ERRCH  ( '#', DIRCTN )

            END IF
 
            IF ( FNDG ) THEN
               CALL ERRCH ( '#', 'Maybe you meant "#".' )
               CALL ERRCH ( '#', GUESS )
            ELSE
               CALL ERRCH ( '#', ' ' )
            END IF
            
            CALL SIGERR ( 'SPICE(UNDEFINEDDIRECTION)'  )
            CALL CHKOUT ( 'EVALDR' )
            RETURN
 
C
C        If the current method is a "primitive" method, we can use the
C        handle just retrieved and immediately evaluate this direction.
C
         ELSE IF ( PRMTIV ) THEN
C
C           After evaluating this "primitive" there's nothing left
C           to do at the current depth.
C
            CALL DYNMDR(  HANDLE, AT, RELTO, ZEROV, DDIR(1,DEPTH) )
 
            DONEXT(DEPTH) = NULL
            DEPTH         = DEPTH - 1
 
         ELSE IF ( DONEXT(DEPTH) .EQ. FETCH1 ) THEN
 
C
C          The only way to get to this point is if the direction passed
C          to GETDRM above was a non-primitive direction (i.e. a binary
C          direction)  We need to FETCH1 that is fetch the name of the
C          first direction associated with NAME(DEPTH) this should go
C          into the next location on the direction stack.
C
            CALL DYNMCP ( HANDLE,  1, NAME(DEPTH+1) )
 
C
C           The next operation to perform at the current depth
C           is to fetch the second component for this direction.
C           The next thing to do at the next depth, is fetch the
C           first component of its definition (unless we find out
C           in the first IF test above that the next direction
C           is a primitive.
C
            DONEXT(DEPTH) = FETCH2
            DEPTH         = DEPTH+1
            DONEXT(DEPTH) = FETCH1
 
 
         ELSE IF ( DONEXT(DEPTH) .EQ. FETCH2 ) THEN
C
C           We have fully resolved the first direction fetched, get the
C           computed direction and put it into the direction stack at
C           this DEPTH.
C
            CALL VEQUG ( DDIR(1,DEPTH+1), 6, DDIR(1,DEPTH) )
C
C           This next block is, again, just for the sake of being
C           able to observe changes in the state of this routine
C           during use of a "display capable" debugger.  Feel free
C           to comment out the code if/when you find it useful
C           for the sake of improving performance.
C
            DO K = 1, 6
               DDIR(K,DEPTH+1) = 0.0D0
            END DO
 
C
C           Fetch the name of the second subcomponent of this direction
C           This should go into the next slot down in the name stack.
C
            CALL DYNMCP( HANDLE, 2, NAME(DEPTH+1) )
C
C           The next operation at this depth will be to combine the
C           "popped" direction above with the one we are about to
C           compute.
C
            DONEXT(DEPTH)   =  COMBIN
            DEPTH           =  DEPTH + 1
C
C           Until we know better, the thing to do with the next
C           direction is fetch the first component.
C
            DONEXT(DEPTH) =  FETCH1
 
 
         ELSE IF ( DONEXT(DEPTH) .EQ. COMBIN ) THEN
C
C           We can only get to this point if we had a binary operation
C           and the second of the two components has been evaluated.
C           Look up the method and then combine the components.
C
            CALL BNRYDR ( HANDLE,
     .                    DDIR(1,DEPTH), DDIR(1,DEPTH+1), DIR )
            CALL VEQUG  ( DIR, 6, DDIR(1,DEPTH) )
C
C           Same story here.  The DO LOOP below just makes using
C           the dugger a bit easier.  Feel free to comment out this
C           block if/when you need to improve performance.
C
            DO K = 1, 6
               DDIR(K,DEPTH+1) = 0.0D0
            END DO
 
            DEPTH = DEPTH - 1
 
 
         ELSE
C
C           We are never supposed to be able to get to this point.
C           If we do, it indicates that some of the stack manipulation
C           above went awry.  Signal an error so that we
C           can debug the problem
C
            CALL SETMSG ( 'An "unreachable" branch of an IF-THEN '
     .      //            'block has been reached.  Something is '
     .      //            'terribly wrong.  Here''s the current '
     .      //            'stack information:  Depth: #, Name #, '
     .      //            'Operation ## ')
 
            K = DEPTH
            DO WHILE ( K .GT. 0 )
               CALL ERRINT ( '#', K )
               CALL ERRCH  ( '#', NAME(K) )
               IF      ( DONEXT(K) .EQ. NULL   ) THEN
                  CALL ERRCH  ( '#', 'NULL' )
               ELSE IF ( DONEXT(K) .EQ. FETCH1 ) THEN
                  CALL ERRCH  ( '#', 'FETCH1' )
               ELSE IF ( DONEXT(K) .EQ. FETCH2 ) THEN
                  CALL ERRCH  ( '#', 'FETCH2' )
               ELSE IF ( DONEXT(K) .EQ. COMBIN ) THEN
                  CALL ERRCH  ( '#', 'COMBIN' )
               ELSE
                  CALL ERRINT ( '#', DONEXT(K) )
               END IF
               K = K - 1
               IF ( K .EQ. 0 ) THEN
                  CALL ERRCH  ( '#', '.' )
               ELSE
                  CALL ERRCH  ( '#','Depth: #, Name #, '
     .            //                'Operation ##' )
               END IF
            END DO
 
         END IF
 
C
C        Perform a check to make sure that DEPTH has not gone past
C        the prescribed limit of our stack.
C
         IF ( DEPTH .GT. LIMIT ) THEN
C
C           Some kind of error has occurred.   Let the private routine
C           handle the diagnostics.
C
            CALL ZZEVALDG ( LIMIT, NAME )
            CALL CHKOUT ( 'EVALDR' )
            RETURN
 
         END IF
 
 
      END DO
C
C     Move the top value in the stack of directions into the output
C     variable
C
      CALL VEQUG ( DDIR(1,1), 6, STATE )
      CALL CHKOUT ( 'EVALDR' )
 
      RETURN
      END
