C$Procedure      DYNDIR ( DynamicFrame )
 
      SUBROUTINE DYNDIR( NAME,   DESCRP,
     .                   ET,     RELTO,  ZEROV,
     .                   UNARY, HANDLE, FOUND, WHICH, COMPNM,
     .                   DDIR1,  DDIR2,  DDIR )
 
C$ Abstract
C
C     This is an umbrella routine that provides the capability
C     of generating frame transformations based upon the dynamics
C     of bodies and other reference frames.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAMES
C
C$ Declarations
 
      IMPLICIT NONE
 
      INCLUDE              'zzdynenm.inc'
      CHARACTER*(*)         NAME
      CHARACTER*(*)         DESCRP
      DOUBLE PRECISION      ET
      CHARACTER*(*)         RELTO
      LOGICAL               ZEROV
      LOGICAL               UNARY
      INTEGER               HANDLE
      LOGICAL               FOUND
      INTEGER               WHICH
      CHARACTER*(*)         COMPNM
 
      DOUBLE PRECISION      DDIR1 ( 6 )
      DOUBLE PRECISION      DDIR2 ( 6 )
      DOUBLE PRECISION      DDIR  ( 6 )
 
      INTEGER               MAXDIR
      PARAMETER           ( MAXDIR = 100 )
 
C$ Brief_I/O
C
C     VARIABLE  I/O  Entry Point
C     --------  ---  --------------------------------------------------
C     AXIS       I
C     DESCRP     I
C     ET         I
C     FROM       I
C     TO         I
C     XFORM      O
C     ROTATE     O
C
C$ Detailed_Input
C
C     AXIS       Specifies which axis of the frame a direction
C                is tied to.  This is either the +X, +Y, +Z or
C                -X, -Y, or -Z axis of the dynamic frame
C
C     DESCRP     Describes the direction that is aligned with AXIS.
C
C     ET         Epoch at which to return a state transformation
C                matrix
C
C     FROM       The frame that the state transformation matrix
C                transforms from.
C
C
C     TO         The frame that the state transformation matrix
C                transforms to.
C
C                FROM and TO may be the word 'DYNAMIC' (to indicate
C                the frame defined by calls to DYNMPR and DYNMSC)
C                or may have the form 'FRAME <frame>' where <frame>
C                is the string for any frame recognized by the SPICE
C                frame system.
C
C$ Detailed_Output
C
C     XFORM      A state transformation matrix that converts states
C                relative to the dynamic frame to states in the RELTO
C                frame.
C
C     ROTATE     A rotation matrix that converts positions relative to
C                the dynamic frame to positions relative to the RELTO
C                frame.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1)
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This is an umbrella routine for three entry points that
C     allow you to define a special frame called 'DYNAMIC' that
C     is oriented relative to the directions associated with various
C     physical directions.
C
C     The DYNAMIC frame is defined using calls to DYNMPR (defines the
C     primary axis of the frame) and DYNMSC (defines the secondary
C     axis of the frame).
C
C     Once the axes of the frames have been defined, the entry points
C     DYNMXF and DYNMRT are available to return state and position
C     transformation matrices, respectively.
C
C     More details as they become available.
C
C$ Examples
C
C     Suppose you wanted to construct a frame whose X-axis is pointed
C     at the sun (as seen from the spacecraft PHOENIX) and whose Y-axis
C     is pointed in the direction of the earth.
C
C     The following set calls will define the frame.
C
C     First set the primary axis.
C
C     CALL DYNMPR ('+X',
C                  'POSITION OF SUN FROM PHOENIX CORRECTION LT+S' )
C
C     Next set the secondary axis:
C
C     CALL DYNMSC ( '+Y',
C                   'POSITION OF EARTH FROM PHOENIX CORRECTION LT+S' )
C
C
C     To get the state transformation from the dynamic frame to
C     J2000, make the following call:
C
C     CALL DYNMXF ( ET, 'FRAME J2000', 'DYNAMIC', XFORM )
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C     B.V. Semenov    (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.3.0, 29-APR-2009 (BVS)
C
C        Updated entry points DYNMDR and BNRYDR to remove non-standard
C        use of duplicate arguments.
C
C-    SPICELIB Version 1.2.0, 5-DEC-2002  (BVS)
C
C        Fixed a bug in SUBOBS branch of entry DYNMDR. STATE 
C        computed by SPKEZ is now negated before being passed to 
C        DNEARP and then subtracted from the near point state
C        to indeed compute direction from the observer toward 
C        the nearest point.
C
C-    SPICELIB Version 1.1.0, 1-SEP-2000  (WLT)
C
C        Fixed a bug in the call to VLCOMG.  The dimension must
C        be the first argument; it was the next to last argument
C        in the buggy version 1.0.0.
C
C-    SPICELIB Version 1.0.0, 15-MAR-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Construct a dynamic frame
C
C-&
C
C     SPICELIB Functions
C
      LOGICAL               RETURN
      LOGICAL               FAILED
      INTEGER               CARDC
      INTEGER               CARDI
      INTEGER               ISRCHC
      DOUBLE PRECISION      VDOT
      DOUBLE PRECISION      DVDOT
      LOGICAL               VZERO
C
C     Local Parameters
C
      INTEGER               WDSIZE
      PARAMETER           ( WDSIZE = 32 )
 
      INTEGER               LBCELL
      PARAMETER           ( LBCELL = -5 )
 
      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 256 )
C
C     Initialization Flag
C
      LOGICAL               FIRST
      SAVE                  FIRST
C
C     Directory Buffer
C
      CHARACTER*(8)         CORR   (    MAXDIR )
      CHARACTER*(WDSIZE)    FRAME  (    MAXDIR )
      CHARACTER*(WDSIZE)    NAMES  (    MAXDIR )
      CHARACTER*(WDSIZE)    REFDRS ( 2, MAXDIR )
      DOUBLE PRECISION      A      (    MAXDIR )
      DOUBLE PRECISION      B      (    MAXDIR )
      DOUBLE PRECISION      C      (    MAXDIR )
      DOUBLE PRECISION      XYZ    ( 3, MAXDIR )
      DOUBLE PRECISION      BNRYDP (    MAXDIR )
      INTEGER               PVN    (    MAXDIR )
      INTEGER               METHOD (    MAXDIR )
      INTEGER               OBS    (    MAXDIR )
      INTEGER               TARG   (    MAXDIR )
      LOGICAL               PRIMTV (    MAXDIR )
      LOGICAL               INACTV (    MAXDIR )
 
 
      SAVE                  A
      SAVE                  B
      SAVE                  BNRYDP
      SAVE                  C
      SAVE                  CORR
      SAVE                  FRAME
      SAVE                  INACTV
      SAVE                  METHOD
      SAVE                  NAMES
      SAVE                  OBS
      SAVE                  PRIMTV
      SAVE                  PVN
      SAVE                  REFDRS
      SAVE                  TARG
      SAVE                  XYZ
 
C
C     Utility Variables
C
      DOUBLE PRECISION      AXIS   ( 6 )
      DOUBLE PRECISION      COSANG
      DOUBLE PRECISION      CTERM  ( 6 )
      DOUBLE PRECISION      CTHETA
      DOUBLE PRECISION      DALT   ( 2 )
      DOUBLE PRECISION      DSTATE ( 6 )
      DOUBLE PRECISION      DT
      DOUBLE PRECISION      DTHETA
      DOUBLE PRECISION      FDTHETA
      DOUBLE PRECISION      FRACTN
      DOUBLE PRECISION      FTHETA
      DOUBLE PRECISION      MCTHETA
      DOUBLE PRECISION      MFRAC
      DOUBLE PRECISION      MSTHETA
      DOUBLE PRECISION      MYXFRM ( 6, 6 )
      DOUBLE PRECISION      PART   ( 6 )
      DOUBLE PRECISION      SINANG
      DOUBLE PRECISION      STATE  ( 6 )
      DOUBLE PRECISION      STATE2 ( 6 )
      DOUBLE PRECISION      STERM  ( 6 )
      DOUBLE PRECISION      STHETA
      DOUBLE PRECISION      THETA
      DOUBLE PRECISION      TOSS   ( 2 )
      DOUBLE PRECISION      TSTATE ( 6 )
      DOUBLE PRECISION      ZERO   ( 3 )
      SAVE                  ZERO

      DOUBLE PRECISION      TVEC3  ( 3 )
      DOUBLE PRECISION      TVEC6  ( 6 )
      
 
      CHARACTER*(LNSIZE)    MSSG
      CHARACTER*(WDSIZE)    KNOWN  ( LBCELL: MAXDIR )
      INTEGER               BEST   ( LBCELL: MAXDIR )
      INTEGER               SCORE  ( LBCELL: MAXDIR )
 
      INTEGER               GET
      INTEGER               I
      INTEGER               HIGH
      INTEGER               K
      INTEGER               PUT
 
      LOGICAL               GOTIT
 
      DATA                  FIRST  / .TRUE. /
      DATA                  ZERO   / 0.0D0, 0.0D0, 0.0D0 /
 
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'DYNDIR')
      CALL SETMSG ( 'The umbrella routine DYNDIR has been called '
     .//            'directly. This routine exists only to '
     .//            'include its entry points. Calling it '
     .//            'directly is a programming error. ' )
      CALL SIGERR ( 'SPICE(BOGUSENTRY)'  )
      CALL CHKOUT ( 'DYNDIR' )
      RETURN
 
 
C$Procedure DYNMPR ( Dynamic Direction Parse )
 
      ENTRY DYNMPR ( NAME, DESCRP )
 
C$ Abstract
C
C     Define the primary axis of a dynamic frame.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     DIRECTION
C
C$ Declarations
C
C     CHARACTER*(*)         AXIS
C     CHARACTER*(*)         DESCRP
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     AXIS       I   Name of the primary axis
C     DESCRP     I   Description of the direction to attach to axix.
C
C$ Detailed_Input
C
C     NAME       is the name of the direction that is to be defined.
C
C     DESCRP     is a description of the the direction to associate
C                with the input name.  Templates for valid
C                descriptions are:
C
C                Primitive Directions: These are directly computable
C                (provided sufficient SPICE kernels have been loaded).
C                Use the entry point DYNMDR to evaluate these
C                directions.
C
C                POSITION OF @body FROM @body CORRECTION @word
C
C                VELOCITY OF @body FROM @body CORRECTION @word
C
C                   where @body is the name or Id-code of a body, and
C                   @word is the one of the corrections allowed by
C                   SPKEZ.
C
C                FIXED @name XYZ @number @number @number'
C
C                FIXED @name RADEC RA @number DEC @number
C
C                    where @name is the name of some reference frame
C                    and @number represents the "obvious" coordinate.
C
C                SUBOBSERVER ON @body FROM @body
C                CORRECTION @word
C
C                   where @body is the name or Id-code of a body, and
C                   @word is the one of the corrections allowed by
C                   SPKEZ.
C
C                NORMAL TO @body AT (2:2){ LATITUDE @number
C                                        | LONGITUDE @number }
C
C                NORMAL TO @body AT
C                (2:2){ LATITUDE @number
C                     | CARTOGRAPHIC LONGITUDE @number }
C
C                NORMAL TO @body AT (3:3){ X @number
C                                        | Y @number
C                                        | Z @number }
C
C                   where @body is the name of some object and @number
C                   is the obvious coordinate.
C
C
C                INCREASING (1:1){ LATITUDE
C                                | LONGTIDUTE
C                                | CARTOGRAPHIC LONGITUDE
C                                }
C                        AT (2:2){ LATITUDE @number
C                                | @option
C                                | LONGITUDE @number
C                                | CARTOGRAPHIC LONGITUDE @number
C                                } ON @body
C
C
C
C                INCREASING (1:1){ LATITUDE
C                                | LONGTIDUTE
C                                | CARTOGRAPHIC LONGITUDE
C                                }
C                        AT (3:3){ X @number
C                                | Y @number
C                                | Z @number
C                                } ON @body
C
C
C                DECREASING (1:1){ LATITUDE
C                                | LONGTIDUTE
C                                | CARTOGRAPHIC LONGITUDE
C                                }
C                        AT (2:2){ LATITUDE @number
C                                | @option
C                                | LONGITUDE @number
C                                | CARTOGRAPHIC LONGITUDE @number
C                                } ON @body
C
C                DECREASING (1:1){ LATITUDE
C                                | LONGTIDUTE
C                                | CARTOGRAPHIC LONGITUDE
C                                }
C                        AT (3:3){ X @number
C                                | Y @number
C                                | Z @number
C                                } ON @body
C
C
C   Binary definitions: In all cases below @name refers to another
C   direction.  Use the entry point BNRYDR to compute these
C
C                ROTATE @name ABOUT @name BY @number (0:1)@unit(degrees)
C
C                ROTATE @name @number (0:1)@unit(degrees)
C                (1:1){ TOWARDS @name
C                     | AWAY FROM @name }
C
C
C                AT @number FROM @name TO @name
C
C                CROSS @name WITH @name
C
C                   where @body is the name of some object and @number
C                   is the obvious coordinate.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C      1) Errors are diagnosed by the private routine ZZPRSDIR.  See
C         that routine for a complete descriptions.  Potential problems
C         include improperly specified descriptions, improperly named
C         Axis, inadequate date in the kernel pool to determine
C         particulars of a direction
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This entry point and the entry point DYNMSC are used to describe
C     the directions associated with the axes of the dynamic frame.
C
C     By using the DYNMPR entry point, you specify exact direction the
C     axis specified by AXIS.  The companion entry point DYNMSC defines
C     the direction of its axis to be as close to the specified
C     direction description as is possible given the constraint imposed
C     by the primary axis definition.
C
C$ Examples
C
C     See the main umbrella.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 15-MAY-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Define a direction
C
C-&
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'DYNMPR')
 
      IF ( FIRST ) THEN
 
         CALL SETMSG ( 'The initialization entry point (INITDR) '
     .   //            'has not been called.  This entry point '
     .   //            'must be called prior to calling any of '
     .   //            'the other entry points to DYNDIR. ' )
         CALL SIGERR ( 'SPICE(IMPROPERUSAGE)'  )
         CALL CHKOUT ( 'DYNMPR' )
         RETURN
 
      END IF
 
 
C
C     Blank names are not allowed.
C
      IF ( NAME .EQ. ' ' ) THEN
         CALL SETMSG ( 'An attempt has been made to create a '
     .   //            'direction with an empty name.  Blank names '
     .   //            'are not allowed. ' )
         CALL SIGERR ( 'SPICE(INVALIDNAME)'  )
         CALL CHKOUT ( 'DYNMPR' )
         RETURN
      END IF
 
C
C     The utility ZZPRSDIR does all of the work of parsing the
C     description.
C
 
      PUT = ISRCHC ( NAME, MAXDIR, NAMES )
 
      IF ( PUT .EQ. 0 ) THEN
C
C        Find a free slot in the names array.
C
         PUT = 1
 
         DO WHILE ( .NOT. INACTV(PUT) )
 
            PUT = PUT + 1
 
            IF ( PUT .GT. MAXDIR ) THEN
               CALL SETMSG ( 'The direction buffer has been '
     .         //            'exhausted.  You need to remove a '
     .         //            'direction before trying to add '
     .         //            'this one: ''#'' ')
               CALL ERRCH  ( '#', NAME )
               CALL SIGERR ( 'SPICE(BUFFEREXHAUSTED)'  )
               CALL CHKOUT ( 'DYNMPR' )
               RETURN
            END IF
 
         END DO
 
      END IF
C
C     We've located someplace to store this name.  Parse the direction
C     specification.
C
      CALL ZZPRSDIR ( PUT,   DESCRP,
     .                CORR,  FRAME,  OBS,     TARG,
     .                A,     B,      C,       XYZ,
     .                PVN,   METHOD, REFDRS,  BNRYDP )
 
 
C
C     Don't put the name in the list of names unless we know we've
C     successfully parsed the direction specification.
C
      IF ( .NOT. FAILED() ) THEN
         NAMES (PUT) = NAME
         INACTV(PUT) = .FALSE.
 
         PRIMTV(PUT) =      METHOD(PUT) .EQ. SPK
     .                 .OR. METHOD(PUT) .EQ. FIXED
     .                 .OR. METHOD(PUT) .EQ. SUBOBS
      END IF
 
      CALL CHKOUT ( 'DYNMPR' )
      RETURN
 
 
 
 
C$Procedure DYNMDR ( Dynamic Direction )
 
      ENTRY DYNMDR ( HANDLE, ET, RELTO, ZEROV, DDIR )
 
C$ Abstract
C
C     Evaluate a dynamically defined direction at a specified epoch
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     DIRECTION
C
C$ Declarations
C
C     IMPLICIT NONE
C     DOUBLE PRECISION      ET
C     CHARACTER*(*)         NAME
C     CHARACTER*(*)         RELTO
C     LOGICAL               ZEROV
C     DOUBLE PRECISION      DDIR ( 6 )
C
C$ Brief_I/O
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     ET         I   Epoch at which to evaluate a direction
C     NAME       I   Name of the direction to evaluate
C     RELTO      I   Frame to return the direction in.
C     ZEROV      I   Zero out derivative info if ZEROV is .TRUE.
C     DDIR       O   Unit vector and its "derivative" for the direction
C
C$ Detailed_Input
C
C     ET         is the epoch at which the direction should be evaluated
C                given in ephemeris seconds past the J2000 epoch.
C
C     NAME       is the name of the direction to evaluate
C
C     RELTO      is the name of a reference frame that the returned
C                direction (and derivative) should be relative to.
C
C     ZEROV      is a logical flag that allows you to force the
C                entry point to zero out the derivative components
C                of the direction vector.
C
C$ Detailed_Output
C
C     DDIR       A six vector giving the specified direction and its
C                derivative.  The first 3 components make up the
C                direction and is a unit vector.  The second three
C                components give the derivative (or zero if forced by
C                the input ZEROV) of the direction.
C
C                (  DDIR(4...6) = d DDIR(1...3) / dt )
C
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C
C     1)
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This entry point performs the basic work of returning the 6
C     vectors associated with a "unary" dynamically defined direction.
C
C$ Examples
C
C
C     This routine is used principally by EVALDR.  See that routine
C     for its use.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C     B.V. Semenov    (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.1.0, 29-APR-2009 (BVS)
C
C        Updated to remove non-standard use of duplicate arguments.
C
C-    SPICELIB Version 1.0.0, 24-JUL-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Evaluate a unary dynamic direction
C
C-&
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'DYNMDR')
 
 
      IF ( FIRST ) THEN
 
         CALL SETMSG ( 'The initialization entry point (INITDR) '
     .   //            'has not been called.  This entry point '
     .   //            'must be called prior to calling any of '
     .   //            'the other entry points to DYNDIR. ' )
         CALL SIGERR ( 'SPICE(IMPROPERUSAGE)'  )
         CALL CHKOUT ( 'DYNMDR' )
         RETURN
 
      END IF
 
 
 
C
C     First look up this name in the collection of known names.
C
      GET = HANDLE
 
C
C     If this is not a recognized name, say so and quit.
C
      IF ( GET .LE. 0 .OR. GET .GT. MAXDIR ) THEN
 
         CALL SETMSG ( 'The handle used to specify a direction '
     .   //            'is not between the values 1 and # '
     .   //            'inclusive. ' )
         CALL ERRINT ( '#', GET )
         CALL SIGERR ( 'SPICE(INVALIDDIRHANDLE)'  )
         CALL CHKOUT ( 'DYNMDR' )
         RETURN
 
      ELSE IF ( INACTV(GET) ) THEN
 
         CALL SETMSG ( 'The handle (#) used to specify a '
     .   //            'direction, does not point to an active '
     .   //            'direction. ' )
         CALL ERRINT ( '#', GET )
         CALL SIGERR ( 'SPICE(INVALIDDIRHANDLE)' )
         CALL CHKOUT ( 'DYNMDR' )
         RETURN
 
      ELSE IF ( .NOT. PRIMTV(GET) ) THEN
 
         CALL SETMSG ( 'The specified direction "#" (associated '
     .   //            'with handle #) is not a "unary" '
     .   //            'direction.  The entry point BNRYDR needs '
     .   //            'to be called to evaluate this direction. ' )
         CALL ERRCH  ( '#', NAMES(HANDLE) )
         CALL ERRINT ( '#', HANDLE )
         CALL SIGERR ( 'SPICE(NOTUNARY)'  )
         CALL CHKOUT ( 'DYNMDR' )
         RETURN
 
      END IF
 
 
      IF ( METHOD(GET) .EQ. SPK ) THEN
 
C
C        At the end of this block, DDIR will contain the direction
C        and derivative of the direction of interest for this pass
C        of the loop.
C
C        If we are using the velocity to define the frame, we
C        need to compute the acceleration.  We do this numerically.
C
         IF (      PVN(GET) .EQ. VEL
     .        .OR. PVN(GET) .EQ. NORM ) THEN
 
            CALL SPKEZ( TARG(GET), ET, RELTO, CORR(GET), OBS(GET),
     .                  STATE,   TOSS )
C
C           If we need the derivative of the direction, we need
C           to get it numerically.
C
            DT = 1.0D0
 
            CALL SPKEZ( TARG(GET), ET+DT,    RELTO, CORR(GET),
     .                  OBS(GET),  STATE2,  TOSS )
 
            CALL VSUBG( STATE2, STATE, 6, DSTATE )
 
C
C           Now compute the unnormalized direction vector.
C
            IF ( PVN(GET) .EQ. VEL ) THEN
               DDIR(1) = STATE(4)
               DDIR(2) = STATE(5)
               DDIR(3) = STATE(6)
               CALL VEQU ( DSTATE(4), DDIR(4) )
            ELSE
               DSTATE(1) = STATE(4)
               DSTATE(2) = STATE(5)
               DSTATE(3) = STATE(6)
 
               CALL DVCRSS ( STATE, DSTATE, DDIR )
            END IF
 
         ELSE
 
            CALL SPKEZ ( TARG(GET), ET, RELTO, CORR(GET), OBS(GET),
     .                   DDIR,    TOSS )
 
         END IF
 
      ELSE IF ( METHOD(GET) .EQ. FIXED ) THEN
 
C
C        We need the state of some object that is fixed
C        relative to a definition frame converted to the
C        specified frame.
C
         CALL VEQU   ( XYZ(1,GET), STATE(1) )
         CALL VEQU   ( ZERO,       STATE(4) )
         CALL SXFORM ( FRAME(GET), RELTO,   ET,   MYXFRM )
         CALL MXVG   ( MYXFRM,     STATE,   6, 6, DDIR   )
 
      ELSE IF ( METHOD(GET) .EQ. SUBOBS ) THEN
 
C
C        Get the state of the observer with respect to the
C        body of interest.
C
         CALL SPKEZ( TARG(GET), ET, FRAME(GET), CORR(GET), OBS(GET),
     .               STATE,   TOSS )
         CALL VMINUG( STATE, 6, TVEC6 )
         CALL VEQUG ( TVEC6, 6, STATE )
 
C
C        Compute the state of the near point ...
C
         CALL DNEARP( STATE, A(GET), B(GET), C(GET),
     .                TSTATE, DALT,  GOTIT )

C
C        Compute state of the near point wrt to observer.
C        
         CALL VSUBG ( TSTATE, STATE, 6, TVEC6 )
         CALL VEQUG ( TVEC6, 6, TSTATE )
 
C
C        Convert the state of the near point to the specified
C        frame.
C
         CALL SXFORM( FRAME(GET), RELTO,   ET,   MYXFRM )
         CALL MXVG  ( MYXFRM,     TSTATE,  6, 6, DDIR   )
 
      ELSE
 
         CALL SETMSG ( 'The specified direction, ''#'', is '
     .   //            'not one of the recognized forms.' )
         CALL ERRCH  ( '#', NAMES(GET) )
         CALL SIGERR ( 'SPICE(NOTKNOWN)'  )
         CALL CHKOUT ( 'DYNMDR' )
         RETURN
 
      END IF
 
      IF ( ZEROV ) THEN
         CALL VHAT ( DDIR, TVEC3 )
         CALL VEQU ( TVEC3, DDIR )
         CALL VEQU ( ZERO, DDIR(4) )
      ELSE
         CALL DVHAT( DDIR, TVEC6 )
         CALL VEQUG( TVEC6, 6, DDIR )
      END IF
 
 
      CALL CHKOUT ( 'DYNMDR' )
      RETURN
 
 
C$Procedure      GETDRM ( Get direction method )
 
      ENTRY GETDRM ( NAME, UNARY, HANDLE, FOUND )
 
C$ Abstract
C
C     Retrieve the handle associated with a direction and determine
C     whether or not it is a unary direction.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     DIRECTION
C
C$ Declarations
C
C     CHARACTER*(*)         NAME
C     LOGICAL               UNARY
C     INTEGER               HANDLE
C     LOGICAL               FOUND
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     NAME       I   Name of the direction to locate.
C     UNARY      O   Indicates whether the direction is a unary
C     HANDLE     O   Handle to use when evaluating this direction
C     FOUND      O   Indicates whether the input NAME is recognized
C
C$ Detailed_Input
C
C     NAME       is the name of a direction to look up.
C
C$ Detailed_Output
C
C     UNARY      is TRUE if the specified direction can be evaluated
C                using DYNMDR
C
C     HANDLE     is the identifier to use when attempting to evaluate
C                a direction.
C
C     FOUND      TRUE if the specified direction is recognized.
C                Otherwise it is returned FALSE.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     Look up the handle and evaluation information associated with
C     a direction and determine whether or not the direction is
C     a unary direction.
C
C
C$ Examples
C
C     See EVALDR.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 25-JUL-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Get the handle and computation method for a direction.
C
C-&
 
 
      IF ( FIRST ) THEN
 
         CALL CHKIN  ( 'GETDRM' )
         CALL SETMSG ( 'The initialization entry point (INITDR) '
     .   //            'has not been called.  This entry point '
     .   //            'must be called prior to calling any of '
     .   //            'the other entry points to DYNDIR. ' )
         CALL SIGERR ( 'SPICE(IMPROPERUSAGE)'  )
         CALL CHKOUT ( 'GETDRM' )
         RETURN
 
      END IF
 
 
 
      HANDLE = ISRCHC( NAME, MAXDIR, NAMES )
 
      IF ( HANDLE .GT. 0 ) THEN
         UNARY  =       PRIMTV( HANDLE )
         FOUND  = .NOT. INACTV( HANDLE )
      ELSE
         FOUND = .FALSE.
      END IF
 
      RETURN
 
 
C=========================================================
 
C$Procedure      DRDIAG ( Direction name diagnostic )
 
      ENTRY DRDIAG ( NAME, COMPNM, FOUND )
 
C$ Abstract
C
C     Find the name in the set of know directions that is "closest" to
C     the name supplied.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     DIRECTION
C
C$ Declarations
C
C     CHARACTER*(*)         NAME
C     CHARACTER*(*)         COMPNM
C     LOGICAL               FOUND
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     NAME       I   Name of the direction to locate.
C     COMPNM     O   Name close to NAME in spelling.
C     FOUND      O   Indicates whether the input NAME is recognized
C
C$ Detailed_Input
C
C     NAME       string that might be a misspelling of some direction
C                name.
C
C$ Detailed_Output
C
C     COMPNM     Direction name that most closely resembles NAME.
C
C     FOUND      TRUE if the specified misspelling is recognized.
C                Otherwise it is returned FALSE.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This entry point is for error detection only.  It provides a
C     friendly diagnostic of what might have been meant by a misspelled
C     direction name.
C
C$ Examples
C
C     See EVALDR.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 25-JUN-2001 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Get the handle and computation method for a direction.
C
C-&
 
 
      IF ( FIRST ) THEN
 
         CALL CHKIN  ( 'DRDIAG' )
         CALL SETMSG ( 'The initialization entry point (INITDR) '
     .   //            'has not been called.  This entry point '
     .   //            'must be called prior to calling any of '
     .   //            'the other entry points to DYNDIR. ' )
         CALL SIGERR ( 'SPICE(IMPROPERUSAGE)'  )
         CALL CHKOUT ( 'DRDIAG' )
         RETURN
 
      END IF
 
      FOUND = .FALSE.
 
      CALL SSIZEC ( MAXDIR, KNOWN  )
      CALL SSIZEI ( MAXDIR, BEST   )
      CALL SSIZEI ( MAXDIR, SCORE  )
C
C     Insert all of the active names into the cell of known direction
C     names.
C
      DO I = 1, MAXDIR
         IF ( .NOT. INACTV(I) ) THEN
            CALL APPNDC ( NAMES(I), KNOWN )
         END IF
      END DO
 
      IF ( CARDC( KNOWN ) .GT. 0 ) THEN
         CALL BESTWD ( NAME, KNOWN, 70, BEST, SCORE, MSSG )
C
C        Locate the highest score in the list.
C
         HIGH = 1
         DO I = 1, CARDI(BEST)
            IF ( SCORE(I) .GT. SCORE(HIGH) ) THEN
               HIGH = I
            END IF
         END DO
 
         IF ( CARDI( BEST ) .GT. 0 ) THEN
            FOUND = .TRUE.
            COMPNM = KNOWN( BEST(HIGH) )
         END IF
 
      END IF
      RETURN
 
 
 
 
C=========================================================
C$Procedure      DYNMCP ( Get dynamic direction component )
 
      ENTRY DYNMCP ( HANDLE, WHICH, COMPNM )
 
C$ Abstract
C
C     Fetch the name of a direction associated with a dynamic
C     direction.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME
C
C$ Declarations
C
C     IMPLICIT NONE
C     INTEGER               HANDLE
C     INTEGER               WHICH
C     CHARACTER*(*)         COMPNM
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     HANDLE     I   is the handle associated with some direction
C     WHICH      I   is the index of the component to retrieve (1 or 2)
C     COMPNM     O   is the name of the requested component direction
C
C$ Detailed_Input
C
C     HANDLE     is the handle of some direction.  This is determined
C                by calling the entry point GETDRM.
C
C     WHICH      is the index of the component direction associated
C                with handle that should be retrieved
C
C$ Detailed_Output
C
C     COMPNM     is the name of the WHICH'th component direction
C                associated with HANDLE.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If the handle specified is "stale" or out of range, the error
C        'SPICE(INVALIDDIRHANDLE)' will be signaled.
C
C     2) If the component requested is not within range for a direction
C        (1 or 2) the error 'SPICE(BADCOMPONENTSPEC)' will be signaled.
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This routine allows you to retrieve the name of either of the
C     components associated with a binary direction.
C
C$ Examples
C
C     See EVALDR
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 26-JUL-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     None.
C
C-&
 
 
      IF ( FIRST ) THEN
 
         CALL CHKIN  ( 'DYNMCP' )
         CALL SETMSG ( 'The initialization entry point (INITDR) '
     .   //            'has not been called.  This entry point '
     .   //            'must be called prior to calling any of '
     .   //            'the other entry points to DYNDIR. ' )
         CALL SIGERR ( 'SPICE(IMPROPERUSAGE)'  )
         CALL CHKOUT ( 'DYNMCP' )
         RETURN
 
      END IF
 
 
 
      IF (       HANDLE .GT. 0
     .     .AND. HANDLE .LE. MAXDIR ) THEN
 
         IF ( INACTV(HANDLE) ) THEN
 
            CALL CHKIN  ( 'DYNMCP' )
            CALL SETMSG ( 'A request has been made to retrieve '
     .      //            'the components for a direction handle '
     .      //            '(#) that is not associated with an '
     .      //            'active direction. ' )
            CALL ERRINT ( '#', HANDLE )
            CALL SIGERR ( 'SPICE(INVALIDDIRHANDLE)'  )
            CALL CHKOUT ( 'DYNMCP' )
            RETURN
 
         ELSE IF (      WHICH .EQ. 1
     .             .OR. WHICH .EQ. 2 ) THEN
 
            COMPNM = REFDRS(WHICH,HANDLE)
 
         ELSE
 
            CALL CHKIN  ( 'DYNMCP' )
            CALL SETMSG ( 'A request has been made for the '
     .      //            '#''th components of the direction '
     .      //            '#.  Only components 1 and 2 are '
     .      //            'available. ' )
 
            CALL ERRINT ( '#', WHICH )
            CALL SIGERR ( 'SPICE(BADCOMPONENTSPEC)'  )
            CALL CHKOUT ( 'DYNMCP' )
            RETURN
 
 
         END IF
 
      ELSE
 
         CALL CHKIN  ( 'DYNMCP' )
         CALL SETMSG ( 'A request has been made to retrieve a '
     .   //            'component for the #''th direction. '
     .   //            'There is no such direction.  This means '
     .   //            'that either the handle bookkeeper has '
     .   //            'gotten out of sync (a bug in the '
     .   //            'direction manager) or that you''ve '
     .   //            'passed in stale or invalid handle from '
     .   //            'some higher level routine (a bud in that '
     .   //            'routine). ' )
         CALL ERRINT ( '#', HANDLE )
         CALL SIGERR ( 'SPICE(INVALIDDIRHANDLE)' )
         CALL CHKOUT ( 'DYNMCP' )
         RETURN
 
      END IF
 
      RETURN
 
 
C$Procedure      BNRYDR ( Binary direction evaluation )
 
      ENTRY BNRYDR ( HANDLE, DDIR1, DDIR2, DDIR )
 
C$ Abstract
C
C     Evaluate a "binary" direction specification.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     DIRECTION
C
C$ Declarations
C
C     IMPLICIT NONE
C     INTEGER               HANDLE
C     DOUBLE PRECISION      DDIR1 ( 6 )
C     DOUBLE PRECISION      DDIR2 ( 6 )
C     DOUBLE PRECISION      DDIR  ( 6 )
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     HANDLE     I   is the handle associated with some binary direction
C     DDIR1      I   a direction state vector
C     DDIR2      I   a direction state vector
C     DDIR       O   the direction obtained by "combining" the inputs
C
C$ Detailed_Input
C
C     HANDLE     is the handle of some direction. Call the entry point
C                GETDRM to retrieve the handle for a direction name
C
C     DDIR1      is a direction state vector.  Normally this will be
C                obtained by calls to DYNMDR or previous calls to
C                this entry point.
C
C     DDIR2      is the second direction state to be used in the
C                computation of the output direction state.
C
C$ Detailed_Output
C
C     DDIR       is the direction state created by combining DDIR1
C                and DDIR2 according to the method associated with
C                HANDLE.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If the supplied handle is not associated with a buffered
C        direction, the error 'SPICE(INVALIDDIRHANDLE)' will be
C        signaled.
C
C     2) If the direction associated with HANDLE is not a "binary"
C        direction, the error  'SPICE(NOTABINARY)' will be signaled.
C
C     3) If the two input directions are linearly dependent making
C        the combined direction ambiguous, the error
C        'SPICE(DEGENERATECASE)' will be signaled.
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This routine combines two direction state vectors to produce
C     yet another production state vector.  The two input direction
C     states will usually be be computed by prior calls to this
C     routine or the routine DYNMDR.
C
C$ Examples
C
C     See EVALDR
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C     B.V. Semenov    (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.1.0, 29-APR-2009 (BVS)
C
C        Updated to remove non-standard use of duplicate arguments.
C
C-    SPICELIB Version 1.0.0, JUL-26-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Binary direction combination
C
C-&
 
C
C     Standard SPICE error handling.
C
      IF ( RETURN() ) THEN
         RETURN
      END IF
 
      CALL CHKIN ( 'BNRYDR' )
 
 
      IF ( FIRST ) THEN
 
         CALL SETMSG ( 'The initialization entry point (INITDR) '
     .   //            'has not been called.  This entry point '
     .   //            'must be called prior to calling any of '
     .   //            'the other entry points to DYNDIR. ' )
         CALL SIGERR ( 'SPICE(IMPROPERUSAGE)'  )
         CALL CHKOUT ( 'BNRYDR' )
         RETURN
 
      END IF
 
 
 
      IF ( HANDLE .LT. 0 .OR. HANDLE .GT. MAXDIR ) THEN
 
         CALL SETMSG ( 'A request has been made to evaluate '
     .   //            'the #''th direction. '
     .   //            'There is no such direction.  This means '
     .   //            'that either the handle bookkeeper has '
     .   //            'gotten out of sync (a bug in the '
     .   //            'direction manager) or that you''ve '
     .   //            'passed in stale or invalid handle from '
     .   //            'some higher level routine (a bug in that '
     .   //            'routine). ' )
 
         CALL ERRINT ( '#', HANDLE )
         CALL SIGERR ( 'SPICE(INVALIDDIRHANDLE)'  )
         CALL CHKOUT ( 'BNRYDR' )
         RETURN
 
      ELSE IF ( PRIMTV(HANDLE) ) THEN
 
         CALL SETMSG ( 'A request has been made to perform a '
     .   //            'binary evaluation on the unary '
     .   //            'direction "#".' )
         CALL ERRCH  ( '#', NAMES(HANDLE) )
         CALL SIGERR ( 'SPICE(NOTABINARY)' )
         CALL CHKOUT ( 'BNRYDR' )
         RETURN
 
      END IF
 
      IF      ( METHOD(HANDLE) .EQ. ROTABT ) THEN
C
C        Rotate DDIR1 about DDIR2 by an angle.
C
C        Recall that to rotate a vector about another vector we
C        simply perform the following operation:
C
C          V_rotated = V + (1-cos(THETA)) AXIS x ( AXIS x V)
C                        +    sin(THETA)  AXIS x V
C
         MCTHETA = 1.0D0 - DCOS( BNRYDP(HANDLE) )
         STHETA  =         DSIN( BNRYDP(HANDLE) )
 
         CALL DVHAT  ( DDIR2,   AXIS )
         CALL DVCRSS ( AXIS,    DDIR1, STERM )
         CALL DVCRSS ( AXIS,    STERM, CTERM )
         CALL VLCOMG ( 6, MCTHETA, CTERM, STHETA, STERM, PART )
         CALL VADDG  ( DDIR1,   PART,  6,   DDIR )
C
C        By construction we should have a unit vector in the first 3
C        components of DDIR, but it doesn't hurt to polish things
C        up a bit.
C
         CALL DVHAT  ( DDIR, TVEC6 )
         CALL VEQUG  ( TVEC6, 6, DDIR )
 
      ELSE IF ( METHOD(HANDLE) .EQ. ROTTWD ) THEN
C
C        This is just like the previous case except that we
C        have to rotate about the cross product of our two
C        directions.
C
C        Rotate DDIR1 "towards" DDIR2 through an angle.
C
         MCTHETA = 1.0D0 - DCOS( BNRYDP(HANDLE) )
         STHETA  =         DSIN( BNRYDP(HANDLE) )
 
         CALL DVCRSS ( DDIR1, DDIR2, AXIS )
C
C        Make sure that we haven't crossed a pair of parallel
C        or anti-parallel directions.
C
         IF ( VZERO ( AXIS ) ) THEN
            CALL SETMSG ( 'The reference direction is '
     .      //            'parallel to the direction that was '
     .      //            'to be rotated.  It''s not possible '
     .      //            'to construct a direction using the '
     .      //            '"rotate towards" method. ' )
            CALL SIGERR ( 'SPICE(DEGENERATECASE)'  )
            CALL CHKOUT ( 'BNRYDR' )
            RETURN
         END IF
 
         CALL DVHAT  ( AXIS, TVEC6 )
         CALL VEQUG  ( TVEC6, 6, AXIS )

         CALL DVCRSS ( AXIS,    DDIR1, STERM )
         CALL DVCRSS ( AXIS,    STERM, CTERM )
         CALL VLCOMG ( 6, MCTHETA, CTERM, STHETA, STERM, PART )
         CALL VADDG  ( DDIR1,   PART,  6, DDIR )
C
C        Polish up the direction vector.
C
         CALL DVHAT  ( DDIR, TVEC6 )
         CALL VEQUG  ( TVEC6, 6, DDIR )
 
 
      ELSE IF ( METHOD(HANDLE) .EQ. FSECT  ) THEN
C
C        Rotate DDIR1 some fraction of the way to DDIR2.
C
         FRACTN = BNRYDP(HANDLE)
 
         COSANG = VDOT( DDIR1, DDIR2 )
         SINANG = SQRT( 1.0D0 - COSANG*COSANG )
C
C        If the angle between DDIR1 and DDIR2 is non-zero, we
C        can easily compute the derivative of the angle between
C        the two directions, and from there extend the previous
C        two cases to deal with
C
         IF ( SINANG .NE. 0.0D0 ) THEN
 
 
            THETA   = DATAN2 ( SINANG, COSANG )
            FTHETA  = FRACTN * THETA
 
            CTHETA  = DCOS( FTHETA )
            STHETA  = DSIN( FTHETA )
            MCTHETA = 1.0D0 - CTHETA
            MSTHETA = - STHETA
 
            DTHETA  = DVDOT ( DDIR1, DDIR2 )
     .              / SQRT( 1.0D0 - COSANG*COSANG )
 
            FDTHETA = FRACTN * DTHETA
C
C           First compute the portion direction:
C
C           V_rotated = V + (1-cos(FRACTION*THETA)) AXIS x ( AXIS x V)
C                           +  sin(FRACTION*THETA)  AXIS x V
C
C           that does not depend upon the derivative of THETA
C
 
            CALL DVCRSS ( DDIR1, DDIR2, AXIS )
            CALL DVHAT  ( AXIS, TVEC6 )
            CALL VEQUG  ( TVEC6, 6, AXIS )
 
            CALL DVCRSS ( AXIS,  DDIR1,   STERM )
            CALL DVCRSS ( AXIS,  STERM,   CTERM )
            CALL VLCOMG ( 6, MCTHETA, CTERM, STHETA, STERM, PART )
            CALL VADDG  ( DDIR1,   PART,  6,  DDIR )
C
C           Now we need the component of the derivative that depends
C           upon the derivative of THETA:
C
C            { -sin(FRACTION*THETA)) AXIS x ( AXIS x V)
C            +  cos(FRACTION*THETA)  AXIS x V } FRACTION*dTHETA/dt
C
             CALL VLCOM3 ( MSTHETA, CTERM,
     .                     CTHETA,  STERM,
     .                     1.0D0,   DDIR(4), TVEC3 )
             CALL VEQU   ( TVEC3, DDIR(4) )
 
 
             CALL DVHAT  ( DDIR, TVEC6 )
             CALL VEQUG  ( TVEC6, 6, DDIR )
 
 
 
         ELSE IF ( COSANG .GT. 0 ) THEN
C
C           The angle between the two directions happens to be
C           zero.  The direction that is some fraction of the
C           way from DDIR1 to DDIR2  is just either direction.
C           And through a bit of calculus necromancy we can
C           compute the derivative.  First get the direction
C           component.
C
            CALL VEQU ( DDIR1, DDIR )
 
C
C           Now recall that the direction that is the fraction f of the
C           way between DDIR1 and DDIR2 is given by
C
C                    sin( theta*(1-f) ) DDIR1 + sin( theta*f ) DDIR2
C           FDIR =   -----------------------------------------------
C                               sin(theta)
C
C           where theta is the angle between DDIR1 and DDIR2. The
C           derivative of this expression w.r.t. theta is:
C
C           [ sin(theta)*
C           { (1-f)*cos(theta*(1-f))*DDIR1  + sin(theta(1-f)) dDDIR1/dt
C            +   f *cos(theta*f)    *DDIR2  + sin(theta*f   ) dDDIR2/dt
C           } / sin(theta)
C
C           - [cos(theta)/sin(theta)]*FDIR
C
C           by doing a lot of rearranging of terms and using
C           L'Hopital's rule, it is possible to show that the limit of
C           dFDIR/dt as theta goes to zero is;
C
C           (1-f)* dDDIR1/dt + f * dDDIR2/dt
C
 
            MFRAC = 1.0D0 - FRACTN
 
            CALL VLCOM ( MFRAC, DDIR1(4), FRACTN, DDIR2(4), DDIR(4) )
 
         ELSE
C
C           This case is impossible, we want a vector some fraction
C           of the way between two antipodal directions.  Which
C           plane would you like to use?  A limiting plane might
C           exist (if derivatives of the two directions are non-zero)
C           but there are still two great circle arcs
C           that pass through the antipodal points.  Again, the
C           choice of which arc to use is not clear.  So instead
C           of making a potentially wrong choice, we make no choice
C           and call this case an error.
C
            CALL SETMSG ( 'The two directions specified are '
     .      //            'antipodal.  No canonical direction '
     .      //            'that is a fraction of the way between '
     .      //            'the two directions exists. ' )
            CALL SIGERR ( 'SPICE(DEGENERATECASE)'  )
            CALL CHKOUT ( 'BNRYDR' )
            RETURN
 
 
         END IF
 
 
 
      ELSE IF ( METHOD(HANDLE) .EQ. XPRDCT ) THEN
C
C        Form the cross product direction  DDIR1 x DDIR2
C
         CALL DVCRSS ( DDIR1, DDIR2, PART )
 
         IF ( VZERO ( PART ) ) THEN
            CALL SETMSG ( 'The two vectors used to define the '
     .      //            'cross product direction are parallel. '
     .      //            'No direction vector ' )
            CALL SIGERR ( 'SPICE(DEGENERATECASE)'  )
            CALL CHKOUT ( 'BNRYDR' )
            RETURN
         END IF
 
         CALL DVHAT  ( PART,  DDIR )
 
      ELSE
 
         CALL SETMSG ( 'The method specified ( method code # ) '
     .   //            'is not a recognized binary method.  This '
     .   //            'indicates either a corruption of the '
     .   //            'internal direction database or an error '
     .   //            'in the parsing of a direction '
     .   //            'specification.  In either case it''s a '
     .   //            'bug. in the routine dyndir.f ' )
         CALL SIGERR ( 'SPICE(BUG)' )
         CALL CHKOUT ( 'BNRYDR' )
         RETURN
 
      END IF
 
      CALL CHKOUT ( 'BNRYDR' )
      RETURN
 
 
C$Procedure      INITDR ( Initialize the direction tables )
 
      ENTRY INITDR ( )
 
C$ Abstract
C
C     Initialize the direction tables.  This should be done only
C     once per program.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     FRAME
C
C$ Declarations
C
C     None.
C
C$ Brief_I/O
C
C    None.
C
C$ Detailed_Input
C
C     None.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     Error free.
C
C     1)
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This routine initializes the direction tables.  It should be
C     called only once per program execution. (Subsequent calls
C     have no effect anyway.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 26-JUL-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Initialize the direction table
C
C-&
 
      IF ( FIRST ) THEN
 
         FIRST = .FALSE.
 
         DO K = 1, MAXDIR
            NAMES(K)  = ' '
            INACTV(K) = .TRUE.
         END DO
 
      END IF
 
      RETURN
 
C$Procedure      DROPDR ( Drop a direction from the table )
 
      ENTRY DROPDR ( HANDLE )
 
C$ Abstract
C
C     Remove a direction from the direction table.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     DIRECTION
C
C$ Declarations
C
C     IMPLICIT NONE
C     INTEGER               HANDLE
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     HANDLE     I   handle of the direction to remove.
C
C$ Detailed_Input
C
C     HANDLE     is the integer handle associated with some direction
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1)  If the handle specified is not within the range of valid
C         handles the error 'SPICE(INVALIDDIRHANDLE)' will be
C         signaled.
C
C$ Files
C
C     None.
C
C$ Particulars
C
C     This entry point allows you to free up space in the direction
C     table.  It is a rope.  Use it with caution.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     W.L. Taber      (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.0.0, 26-JUL-2000 (WLT)
C
C
C-&
 
C$ Index_Entries
C
C     Free up space in the direction table
C
C-&
 
      IF ( FIRST ) THEN
 
         CALL CHKIN  ( 'DROPDR' )
         CALL SETMSG ( 'The initialization entry point (INITDR) '
     .   //            'has not been called.  This entry point '
     .   //            'must be called prior to calling any of '
     .   //            'the other entry points to DYNDIR. ' )
         CALL SIGERR ( 'SPICE(IMPROPERUSAGE)'  )
         CALL CHKOUT ( 'DROPDR' )
         RETURN
 
      END IF
 
 
      IF (       HANDLE .GT. 0
     .     .AND. HANDLE .LE. MAXDIR ) THEN
 
         NAMES (HANDLE) = ' '
         INACTV(HANDLE) = .TRUE.
 
      ELSE
 
         CALL CHKIN  ( 'DROPDR' )
         CALL SETMSG ( 'A request has been made to "drop" the '
     .   //            'direction associated with the handle #. '
     .   //            'This is outside the range of valid '
     .   //            'handles (1 to #). ' )
         CALL ERRINT ( '#', HANDLE )
         CALL ERRINT ( '#', MAXDIR )
 
 
         CALL ERRINT ( '#', HANDLE )
         CALL SIGERR ( 'SPICE(INVALIDDIRHANDLE)' )
         CALL CHKOUT ( 'DROPDR' )
         RETURN
 
      END IF
 
      RETURN
 
 
      END
