C$Procedure KWPC ( Return a character string from the command line )

      FUNCTION KWPC( CID, POSNT, DFAULT )


C$ Abstract
C
C     Function to return a character string associated with a key word
C     parameter and a positional designator on the command line.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     KEY WORD PARAMETER
C     COMMAND LINE ARGUMENT
C
C$ Declarations

      CHARACTER*(*)         KWPC
      CHARACTER*(*)         CID
      CHARACTER*(*)         DFAULT

      INTEGER               POSNT

C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     KWPC       O   Character string returned by the routine
C     CID        I   Character key word parameter identifier
C     POSNT      I   Position in the key word parameter list to return
C     DFAULT     I   Default value for parameter.
C
C$ Detailed_Input
C
C     CID        The character string which defines a key word
C                parameter.  This string must start with a letter.
C
C     POSNT      The integer location after the key word parameter to
C                parse for a value.
C
C     DFAULT     The default value for a key word parameter position.
C                This value is returned if there is a parse error or
C                if the designated parameter position was not defined
C                on the command line.
C
C$ Detailed_Output
C
C     KWPC       The character string corresponding to the command line
C                key word parameter at the designated position or the
C                default value corresponding to the same position.
C
C$ Parameters
C
C      None.
C
C$ Exceptions
C
C      Error free (probably).
C
C$ Files
C
C      None.
C
C$ Particulars
C
C      1).  The routine GETCLM is required for linking.
C
C      2).  A key word parameter may be any combination of letters.
C           Indicating a parameter on the command line is
C           accomplished by using the designated character '-' as
C           a prefix.
C
C      3).  Key word parameters are not case sensitive.
C
C$ Examples
C
C      Given a command line of
C
C      prog_name -c my dog has fleas
C
C      CHAR  = KWPC( 'c', 0, 'glub' )
C      CHAR1 = KWPC( 'c', 1, 'So'   )
C      CHAR2 = KWPC( 'c', 2, 'he'   )
C      CHAR3 = KWPC( 'c', 5, 'died' )
C
C      will result in
C
C      CHAR  = 'my dog has fleas'
C      CHAR1 = 'my'
C      CHAR2 = 'dog'
C      CHAR3 = 'died'.
C
C      There is no 5th positional parameter of the command line so
C      CHAR3 was set to the default entry 'died'.  Positional parameters
C      3 and 4 were not referenced, and so the corresponding inputs on
C      the command line were ignored.
C
C      Using 0 as a positional parameter is a request for all entries
C      after the key word parameter to be returned.  This list ends
C      when another key word parameter ID is encountered or at the end
C      of the command argument list. Both
C
C      prog_name -c my dog has fleas
C
C                         and
C
C      prog_name -c my dog has fleas -l so he died
C
C      for the call
C
C      CHAR  = KWPC( 'c', 0, 'glub' )
C
C      will result in
C
C      CHAR  = 'my dog has fleas'
C
C$ Restrictions
C
C      1).  A key word parameter may NOT(!) have a number as the first
C           character.  The first character should ALWAYS(!!) be a
C           letter.
C
C$ Literature_References
C
C      None.
C
C$ Author_and_Institution
C
C      Ed Wright (JPL)
C
C$ Version
C
C-    SPICELIB Version 1.0.1,  APR-08-1998 (EDW)
C
C       Added additional comments.
C
C-    SPICELIB Version 1.0.0,  AUG-28-1996 (EDW)
C
C-&

C$ Index_Entries
C
C     command line arguments
C
C-&


C
C     SPICELIB routines
C
      LOGICAL               EQSTR
      INTEGER               LASTNB

C
C     Locals.
C
      CHARACTER*(255)       LINE

      INTEGER               NITEMS
      INTEGER               NOBLNK
      INTEGER               I
      INTEGER               J

C
C     Maximum number of space delimited items on the command line.
C
      INTEGER               NUMITM
      PARAMETER           ( NUMITM = 90 )


      CHARACTER*(255)       ITEMS  ( NUMITM )
      CHARACTER*(255)       CHTEMP

      LOGICAL               KWFND


C
C     Set the function value to the default value.
C
      KWPC  = DFAULT
      KWFND = .TRUE.

C
C     Get the command line arguments.
C
      CALL GETCML ( LINE )

C
C     Parse for the character identifier.
C
      CALL LPARSE ( LINE, ' ', NUMITM, NITEMS, ITEMS )

C
C     Make the identified with the delimiter character.
C
      NOBLNK = LASTNB( CID )
      CHTEMP = '-'//CID( 1 : NOBLNK )



C
C     Loop over non zero number of items on the command line.
C
      IF ( NITEMS. NE. 0 ) THEN

         DO I = 1, NITEMS

C
C           Is the current item the key word parameter identifier?
C
            IF ( EQSTR( ITEMS( I ), CHTEMP ) ) THEN


C           We have the identifier.  Do we want a particular entry
C           or the full string?

               IF ( POSNT. EQ. 0 ) THEN

C
C                 Want the full string.  Right now.
C

                  DO J = I + 1, NITEMS

                     CHTEMP = ITEMS( J )


C
C                    Check for a keyword marker in the ITEMS( J )
C                    entry.  If it is a keyword marker the first
C                    character will be a '-' and the second will not be
C                    a number.
C
                     IF ( CHTEMP(1:1). EQ. '-' .AND.
     .                    CHTEMP(2:2). NE. '0' .AND.
     .                    CHTEMP(2:2). NE. '1' .AND.
     .                    CHTEMP(2:2). NE. '2' .AND.
     .                    CHTEMP(2:2). NE. '3' .AND.
     .                    CHTEMP(2:2). NE. '4' .AND.
     .                    CHTEMP(2:2). NE. '5' .AND.
     .                    CHTEMP(2:2). NE. '6' .AND.
     .                    CHTEMP(2:2). NE. '7' .AND.
     .                    CHTEMP(2:2). NE. '8' .AND.
     .                    CHTEMP(2:2). NE. '9'      ) THEN

C
C                       Found what looked like a key word marker,
C                       so return the parsed string now.
C

                        RETURN

                     ELSE

C
C                       Collect all the command line arguments after the
C                       key word parameter.  Concatenate the correct
C                       ITEMS value onto the KWPC string.
C
                        IF ( KWPC. EQ. DFAULT ) THEN
                           KWPC   = ITEMS( J )

                        ELSE
                           NOBLNK = LASTNB( KWPC )
                           KWPC   = KWPC( 1: NOBLNK )//' '//ITEMS( J )

                        END IF

C
C                       If this is the last element of the command
C                       argument, return now.
C
                        IF ( J. EQ. NITEMS ) THEN
                           RETURN
                        END IF


                     END IF

                  END DO

               ELSE

C
C              The request is for a particular positional entry after
C              the key word parameter.  Is there a value at the
C              I + POSNT entry?  If not do nothing.
C
                  IF ( ( I + POSNT ) . LE. NITEMS ) THEN

C
C                    Is any item in the ITEMS list from I to I + POSNT
C                    another keyword?  Use the same test as before.
C
                     DO J = 1, POSNT

                        CHTEMP = ITEMS( I + J )

C
C                       Test for the key word marker.  If its there,
C                       flag a false so the parse will stop.
C
                        IF ( CHTEMP(1:1). EQ. '-' .AND.
     .                       CHTEMP(2:2). NE. '0' .AND.
     .                       CHTEMP(2:2). NE. '1' .AND.
     .                       CHTEMP(2:2). NE. '2' .AND.
     .                       CHTEMP(2:2). NE. '3' .AND.
     .                       CHTEMP(2:2). NE. '4' .AND.
     .                       CHTEMP(2:2). NE. '5' .AND.
     .                       CHTEMP(2:2). NE. '6' .AND.
     .                       CHTEMP(2:2). NE. '7' .AND.
     .                       CHTEMP(2:2). NE. '8' .AND.
     .                       CHTEMP(2:2). NE. '9'      ) THEN

                           KWFND = .FALSE.

                        END IF

                     END DO


C
C                    Is it okay to parse the line for the key word
C                    position parameter?
C
                     IF ( KWFND ) THEN

                        KWPC = ITEMS ( I + POSNT )

                     END IF

C
C                    We either have a value or an error, so return
C                    now.
C
                     RETURN

                  END IF



               END IF

            END IF


         END DO


      END IF


      RETURN
      END
