C$Procedure TIMCVT ( Time conversion )
 
      SUBROUTINE TIMCVT ( FILE, LINE, LNUM, TIMSYS, TIMSTR, ET )
 
C$ Abstract
C
C     Convert an OEM metadata or state time tag to TDB.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     OEM2SPK
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'errhnd.inc'

      CHARACTER*(*)         FILE
      CHARACTER*(*)         LINE
      INTEGER               LNUM
      CHARACTER*(*)         TIMSYS
      CHARACTER*(*)         TIMSTR
      DOUBLE PRECISION      ET
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     FILE       I   Input OEM file.
C     LINE       I   Line on which time tag was found.
C     LNUM       I   Line number.
C     TIMSYS     I   Time system.
C     TIMSTR     I   Time string.
C     ET         O   TDB value corresponding to TIMSTR.
C
C$ Detailed_Input
C                     
C     FILE           is the name of the OEM file from which a time
C                    tag of interest was read.  FILE is provided
C                    to support construction of error messages.
C                     
C     LINE           is an OEM line containing a time tag.  LINE is 
C                    provided to support construction of error messages.
C
C     LNUM           is the ordinal position of LINE in the file.
C
C     TIMSYS         is the name of the time system associated with
C                    the input time tag.
C
C     TIMSTR         is a time tag read from an OEM file.  The
C                    tag must be formatted so that it can be parsed
C                    by the SPICELIB routine TPARTV.  
C
C$ Detailed_Output
C
C     ET             is the TDB epoch, expressed as seconds past J2000,
C                    defined by TIMSYS and TIMSTR.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     See the description of the argument FILE.
C
C$ Exceptions
C
C     1) If the input string TIMSTR cannot be parsed by TPARTV, the 
C        error SPICE(INVALIDTIME) will be signaled.
C
C$ Particulars
C
C     The OEM specification calls for time strings conforming to
C     ISO formats but representing times in a variety of systems.
C     Since SPICELIB assumes ISO-formatted strings represent UTC
C     epochs, this routine first converts the input string to 
C     calendar format.  Next, a time system label is appended to
C     the string, and the the string is converted to TDB.
C     A small round-off error is introduced by these conversion
C     steps.
C     
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-1,
C          September, 2004.
C
C$ Version
C
C-    OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&

C
C     SPICELIB functions
C
      LOGICAL               FAILED
      LOGICAL               RETURN


C
C     Local parameters
C
      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 80 )

      INTEGER               MAXVEC
      PARAMETER           ( MAXVEC = 6 )

      INTEGER               TIMLEN
      PARAMETER           ( TIMLEN = 50 )

      INTEGER               TYPLEN
      PARAMETER           ( TYPLEN = 10 )

      INTEGER               MAXMOD
      PARAMETER           ( MAXMOD = 20 )


      CHARACTER*(LMSGLN)    ERRMSG
      CHARACTER*(LNSIZE)    MODIFY ( MAXMOD )
      CHARACTER*(LNSIZE)    PICTUR
      CHARACTER*(TIMLEN)    TMPTIM
      CHARACTER*(TYPLEN)    TTYPE

      DOUBLE PRECISION      T
      DOUBLE PRECISION      TVEC   ( MAXVEC )

      INTEGER               NTVEC

      LOGICAL               MODS
      LOGICAL               SUCCES
      LOGICAL               YABBRV
     


      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'TIMCVT' )

C
C     See wether TPARTV understands the time string.  TPARTV has the
C     convenient property of not signaling an error if it doesn't
C     recognize the time string.
C
      CALL TPARTV ( TIMSTR, TVEC,   NTVEC,  TTYPE,  MODIFY,
     .              MODS,   YABBRV, SUCCES, PICTUR, ERRMSG )

      IF ( .NOT. SUCCES ) THEN

         CALL SETMSG ( 'Time string found at line #* in file  '    //
     .                 '<*> could not be parsed. Line was <*> * '   )
         CALL ERRINT ( '*',  LNUM                                   )
         CALL ERRCH  ( '*',  FILE                                   ) 
         CALL ERRCH  ( '*',  LINE                                   ) 
         CALL ERRCH  ( '*',  ERRMSG                                 ) 
         CALL SIGERR ( 'SPICE(INVALIDTIME)'                         )
         CALL CHKOUT ( 'TIMCVT'                                     )
         RETURN

      END IF

C
C     Append the appropriate time system flag to the input time string.
C
      TMPTIM = TIMSTR

C
C     Convert the input string to calendar format, so we can
C     append a time system label.
C
      CALL STR2ET ( TIMSTR, T )

      CALL TIMOUT ( T,
     .              'YYYY MON DD HR:MN:SC.#########',
     .              TMPTIM                                  )
      CALL SUFFIX ( TIMSYS, 1, TMPTIM )
         
C
C     We're ready for STR2ET to take a shot at translating this 
C     string.
C
      CALL STR2ET ( TMPTIM, ET )

      IF ( FAILED() ) THEN

         CALL RESET

         CALL SETMSG ( 'Time string found at line #* in file  '    //
     .                 '<*> could not be parsed after time system '//
     .                 'flag * was appended.  Input line was <*>'   )
         CALL ERRINT ( '*',  LNUM                                   )
         CALL ERRCH  ( '*',  FILE                                   ) 
         CALL ERRCH  ( '*',  TIMSYS                                 ) 
         CALL ERRCH  ( '*',  LINE                                   ) 
         CALL SIGERR ( 'SPICE(INVALIDTIME)'                         )
         CALL CHKOUT ( 'TIMCVT'                                     )
         RETURN
 
      END IF
         
      CALL CHKOUT ( 'TIMCVT' )
      RETURN
      END


      
      
