C$Procedure PRSMET ( Parse metadata block )
 
      SUBROUTINE PRSMET ( INPUT, OBJECT, CENTER, FRAME, EPOCH,
     .                    TBEG,  TEND,   TIMSYS, METHOD, DEGREE  )
 
C$ Abstract
C
C     Parse an OEM metadata block.  Return segment descriptor
C     information, plus the name of the time system associated
C     with the metadata block.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     OEM2SPK
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'oem2spk.inc'

      CHARACTER*(*)         INPUT
      INTEGER               OBJECT
      INTEGER               CENTER
      CHARACTER*(*)         FRAME
      CHARACTER*(*)         EPOCH
      DOUBLE PRECISION      TBEG
      DOUBLE PRECISION      TEND
      CHARACTER*(*)         TIMSYS
      CHARACTER*(*)         METHOD
      INTEGER               DEGREE
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     INPUT      I   Name of OEM file.
C     OBJECT     O   Ephemeris object.
C     CENTER     O   Center of motion of object.
C     FRAME      O   Reference frame name.
C     EPOCH      O   Reference frame epoch.
C     TBEG       O   Segment begin time.
C     TEND       O   Segment end time.
C     TIMSYS     O   Time system name.
C     METHOD     O   Interpolation method.
C     DEGREE     O   Interpolation degree.
C
C$ Detailed_Input
C           
C     INPUT          is the name of an OEM file from which a 
C                    metadata block to be read.
C          
C$ Detailed_Output
C
C     OBJECT         is the NAIF integer ID code of the ephemeris
C                    object to which the data following the 
C                    current metadata block correspond.  
C
C     CENTER         is the NAIF integer ID code of the center of
C                    motion of the ephemeris object.
C
C     FRAME          is the name of the reference frame of the data.
C
C     EPOCH          is the Epoch of the reference frame of the data,
C                    if not intrinsic to the definition of the
C                    reference frame.
C
C     TBEG           is the ephemeris time, expressed as seconds
C                    past J2000 TDB, of the start of coverage.
C
C                    TBEG is greater than or equal to the epoch 
C                    of the first state of the contiguous set of 
C                    states following the metadata block.
C
C     TEND           is the ephemeris time, expressed as seconds
C                    past J2000 TDB, of the end of coverage.
C
C                    TEND is less than or equal to the epoch 
C                    of the last state of the contiguous set of
C                    states following the metadata block.
C
C     TIMSYS         is the name of the time system associated with
C                    the metadata block.
C
C     METHOD         is the name of the interpolation method, if
C                    one was supplied in the metadata block.  
C                    Otherwise METHOD is returned blank.
C
C     DEGREE         is the degree of the interpolation method, if
C                    a method was supplied in the metadata block.  
C                    Otherwise DEGREE is set to zero.
C             
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If a required metadata keyword is not found, the error
C        SPICE(MISSINGITEM) is signaled.
C
C     2) If either the object or center names cannot be mapped to
C        a NAIF integer ID code, the error SPICE(IDCODENOTFOUND)
C        is signaled.
C
C     3) If any metadata line does not follow the expected 
C        "keyword = value" syntax, the error will be diagnosed
C        by routines called by this routine.
C
C     4) If any time bounds given by the metadata block (either
C        start and stop times, or usable start and stop times)
C        cannot be parsed and converted to TDB, the error will
C        be diagnosed by routines called by this routine.
C
C     5) If the reference frame name cannot be mapped to a NAIF
C        integer code, the error SPICE(INVALIDREFFRAME) will be 
C        signaled.
C
C$ Files
C
C     See the description of INPUT.
C
C$ Particulars
C
C     This routine assumes that the first non-blank, non-comment
C     line it reads belongs to a metadata block.  The last line
C     this routine will normally read is a metadata block end 
C     marker.
C
C     If both the USEABLE_START_TIME and USEABLE_STOP_TIME keywords
C     are found in the metadata block, the corresponding epochs
C     will be used as the output segment descriptor's time bounds.
C     If these keywords are not present, the time bounds will be
C     derived from the values associated with the START_TIME and
C     STOP_TIME keywords.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     1) This routine assumes that any time tag used in an OEM file
C        can be parsed by the SPICE routine TPARTV.
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-1,
C          September, 2004.
C     [2]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-2,
C          November, 2009.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C     M.C. Kim        (JPL)
C
C$ Version
C
C-    OEM2SPK Version 2.0.0, 30-AUG-2013 (MCK)
C
C        Enhanced to read OEM 2; the program reads the new keyword,
C        REF_FRAME_EPOCH, but ignores it.
C
C-    OEM2SPK Version 1.2.0, 05-OCT-2011 (NJB)
C
C        Bug fix: replacement of underscores by blanks in metadata
C        string values is now done BEFORE the string mapping, so
C        the string mapping can be used to override this action.
C
C-    OEM2SPK Version 1.1.0, 06-OCT-2006 (NJB)
C
C        Enhanced long error message for unrecognized frame.
C
C-    OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&


C
C     SPICELIB functions
C
      INTEGER               CARDC
      INTEGER               ISRCHC

      LOGICAL               ELEMC
      LOGICAL               FAILED
      LOGICAL               RETURN
      
C
C     Local parameters
C
      INTEGER               NREQ
      PARAMETER           ( NREQ   = 6 )

      INTEGER               NKEYS
      PARAMETER           ( NKEYS  = 11 )

C
C     Indices of saved items
C
      INTEGER               CTRIDX 
      PARAMETER           ( CTRIDX = 1 )

      INTEGER               INTIDX
      PARAMETER           ( INTIDX = CTRIDX + 1 )

      INTEGER               DEGIDX
      PARAMETER           ( DEGIDX = INTIDX + 1 )

      INTEGER               OBJIDX
      PARAMETER           ( OBJIDX = DEGIDX + 1 )

      INTEGER               REFIDX
      PARAMETER           ( REFIDX = OBJIDX + 1 )
      
      INTEGER               EPKIDX
      PARAMETER           ( EPKIDX = REFIDX + 1 )
      
      INTEGER               TBIDX
      PARAMETER           ( TBIDX  = EPKIDX + 1 )

      INTEGER               TEIDX
      PARAMETER           ( TEIDX  = TBIDX  + 1 )

      INTEGER               SYSIDX
      PARAMETER           ( SYSIDX = TEIDX + 1 )
      
      INTEGER               UBIDX
      PARAMETER           ( UBIDX  = SYSIDX + 1 )

      INTEGER               UEIDX
      PARAMETER           ( UEIDX  = UBIDX  + 1 )

C
C     Other parameters
C
      INTEGER               TVSIZ
      PARAMETER           ( TVSIZ  = 6 )

C
C     Local variables
C
      CHARACTER*(LNSIZE)    ITEMS   ( LBCELL : NKEYS )
      CHARACTER*(LNSIZE)    KEYWRD
      CHARACTER*(LNSIZE)    KEYS    ( NKEYS )
      CHARACTER*(LNSIZE)    KLINES  ( NKEYS )
      CHARACTER*(LNSIZE)    LINE
      CHARACTER*(LNSIZE)    RITEMS  ( LBCELL : NREQ )
      CHARACTER*(LNSIZE)    RKEYS   ( NREQ )
      CHARACTER*(LNSIZE)    TYPE
      CHARACTER*(LNSIZE)    VALUE
      CHARACTER*(LNSIZE)    VSTRS   ( NKEYS )
      
      INTEGER               BIDX
      INTEGER               EIDX
      INTEGER               KNUMS   ( NKEYS )
      INTEGER               LOC
      INTEGER               LOC2
      INTEGER               LNUM
      INTEGER               REFCOD
      

      LOGICAL               EOF
      LOGICAL               FOUND

C
C     Saved variables
C
      SAVE                  KEYS
      SAVE                  KLINES
      SAVE                  KNUMS
      SAVE                  RKEYS
      SAVE                  VSTRS

C
C     Initial values
C
      DATA                  KEYS     /  'CENTER_NAME',
     .                                  'INTERPOLATION',
     .                                  'INTERPOLATION_DEGREE',
     .                                  'OBJECT_NAME',
     .                                  'REF_FRAME', 
     .                                  'REF_FRAME_EPOCH', 
     .                                  'START_TIME',
     .                                  'STOP_TIME',
     .                                  'TIME_SYSTEM',
     .                                  'USEABLE_START_TIME',
     .                                  'USEABLE_STOP_TIME'   /

      DATA                  RKEYS    /  'CENTER_NAME',
     .                                  'OBJECT_NAME',
     .                                  'REF_FRAME', 
     .                                  'START_TIME',
     .                                  'STOP_TIME',
     .                                  'TIME_SYSTEM'        /
     

 

      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'PRSMET' )

C
C     Initialize the sets of metadata items seen so far in
C     the current metadata block.  The set ITEMS contains all
C     keywords seen; the set RITEMS contains the required 
C     keywords seen.
C
      CALL SSIZEC ( NKEYS, ITEMS  )
      CALL SSIZEC ( NREQ,  RITEMS )

C
C     Read non-comment lines until we hit the end
C     of file or the end of the metadata block.
C
      CALL GETNCL ( INPUT, LINE, LNUM, TYPE, EOF )

      DO WHILE (       ( .NOT. EOF                 )
     .           .AND. ( .NOT. FAILED()            ) 
     .           .AND. ( TYPE .EQ. 'METADATA_LINE' ) )
C
C        We expect the current line to conform to the "keyword = value"
C        syntax.
C
         CALL KEYVLC ( INPUT, LINE, LNUM, KEYWRD, VALUE )
         
         IF ( FAILED() ) THEN

            CALL CHKOUT ( 'PRSMET' )
            RETURN

         END IF

C
C        Map both keyword and value to their standardized forms.
C
         CALL CMAP  ( KEYWRD, KEYWRD )
         CALL LJUST ( KEYWRD, KEYWRD )
         CALL UCASE ( KEYWRD, KEYWRD )

C
C        Replace with blanks any underscores found in the value string.
C
         CALL REPLCH ( VALUE, '_', ' ', VALUE )

         CALL CMAP  ( VALUE,  VALUE  )
         CALL LJUST ( VALUE,  VALUE  )
         CALL UCASE ( VALUE,  VALUE  )

C
C        Save the value, line, and line number associated with 
C        the current keyword, if the keyword is recognized.  
C
         LOC = ISRCHC ( KEYWRD, NKEYS, KEYS )

         IF ( LOC .GT. 0 ) THEN
C
C           Store the RHS value.
C
            VSTRS(LOC) = VALUE

C
C           Mark the current keyword as seen.           
C
            CALL INSRTC ( KEYWRD, ITEMS )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'PRSMET' )
               RETURN
            END IF

C
C           If the keyword is required, add it to the set
C           of required keywords seen.
C
            LOC2 = ISRCHC ( KEYWRD, NREQ, RKEYS )

            IF (  LOC2 .GT. 0 ) THEN

               CALL INSRTC ( KEYWRD, RITEMS )

               IF ( FAILED() ) THEN
                  CALL CHKOUT ( 'PRSMET' )
                  RETURN
               END IF

            END IF

C
C           Save the source line and line number for this keyword.
C           These will be needed if a semantic error is detected
C           later.
C
            KLINES(LOC) = LINE
            KNUMS(LOC)  = LNUM
    
         END IF

         CALL GETNCL ( INPUT, LINE, LNUM, TYPE, EOF )

      END DO

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'PRSMET' )
         RETURN
      END IF

C
C     Is the set of required items full?
C
      IF ( CARDC(RITEMS) .LT. NREQ ) THEN

         CALL SETMSG ( 'Not all required metadata items were seen ' //
     .                 'in the metadata block ending at line #* in '//
     .                 'file <*>.  Required items are:  '           //
     .                 'OBJECT_NAME, CENTER_NAME, REF_FRAME, '      //
     .                 'START_TIME, STOP_TIME, TIME_SYSTEM.'        )
         CALL ERRINT ( '*', LNUM                                    )
         CALL ERRCH  ( '*', INPUT                                   ) 
         CALL SIGERR ( 'SPICE(MISSINGITEM)'                         )
         CALL CHKOUT ( 'PRSMET'                                     )
         RETURN

      END IF

C
C     The interpolation degree keyword is required only if the
C     interpolation keyword was present.
C
      IF (  ELEMC( KEYS(INTIDX), ITEMS )  ) THEN
C
C        Set the interpolation method and check for presence of
C        the degree.
C
         METHOD = VSTRS(INTIDX)

         IF (  ELEMC( KEYS(DEGIDX), ITEMS )  ) THEN
C
C           Parse the interpolation degree.
C
            CALL PRSINT (  VSTRS(DEGIDX),  DEGREE  )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'PRSMET' )
               RETURN
            END IF

         ELSE
C
C           The required degree is not present.
C
            CALL SETMSG ( 'No interpolation degree was seen '   //
     .                    'in the metadata block ending at '    //
     .                    'line #* in file <*>.  The degree '   //
     .                    'is required when the interpolation ' //
     .                    'method is specified, as it is here.' )
            CALL ERRINT ( '*', LNUM                             )
            CALL ERRCH  ( '*', INPUT                            ) 
            CALL SIGERR ( 'SPICE(MISSINGITEM)'                  )
            CALL CHKOUT ( 'PRSMET'                              )
            RETURN

         END IF

      ELSE
C
C        The interpolation method was not specified.
C
         METHOD = ' '
         DEGREE = 0

      END IF

C
C     Map object name to NAIF ID code.
C
      CALL BODN2C ( VSTRS(OBJIDX), OBJECT, FOUND )

      IF ( .NOT. FOUND ) THEN

         CALL SETMSG ( 'Object * specified in metadata block '     //
     .                 'at line #* in file <*> could not be '      //
     .                 'mapped to a NAIF ID code.  Line was *'      )
         CALL ERRCH  ( '*',  VSTRS(OBJIDX)                          ) 
         CALL ERRINT ( '*',  KNUMS(OBJIDX)                          )
         CALL ERRCH  ( '*',  INPUT                                  ) 
         CALL ERRCH  ( '*',  KLINES(OBJIDX)                         ) 
         CALL SIGERR ( 'SPICE(IDCODENOTFOUND)'                      )
         CALL CHKOUT ( 'PRSMET'                                     )
         RETURN

      END IF

C
C     Map center name to NAIF ID code.
C
      CALL BODN2C ( VSTRS(CTRIDX), CENTER, FOUND )

      IF ( .NOT. FOUND ) THEN

         CALL SETMSG ( 'Center * specified in metadata block '     //
     .                 'at line #* in file <*> could not be '      //
     .                 'mapped to a NAIF ID code.  Line was *'      )
         CALL ERRCH  ( '*',  VSTRS(CTRIDX)                          ) 
         CALL ERRINT ( '*',  KNUMS(OBJIDX)                          )
         CALL ERRCH  ( '*',  INPUT                                  ) 
         CALL ERRCH  ( '*',  KLINES(CTRIDX)                         ) 
         CALL SIGERR ( 'SPICE(IDCODENOTFOUND)'                      )
         CALL CHKOUT ( 'PRSMET'                                     )
         RETURN

      END IF

C
C     Convert time bounds to ET.   If "usable" bounds have been 
C     supplied, use these bounds.  Otherwise use the default
C     start and stop times.
C     
      IF (        ELEMC( KEYS(UBIDX), ITEMS )
     .     .AND.  ELEMC( KEYS(UEIDX), ITEMS )  ) THEN

         BIDX = UBIDX
         EIDX = UEIDX

      ELSE

         BIDX = TBIDX
         EIDX = TEIDX
         
      END IF


      TIMSYS  =  VSTRS(SYSIDX)

      CALL TIMCVT ( INPUT,   KLINES(BIDX),  KNUMS(BIDX), 
     .              TIMSYS,  VSTRS(BIDX),   TBEG          )


      CALL TIMCVT ( INPUT,   KLINES(EIDX),  KNUMS(EIDX), 
     .              TIMSYS,  VSTRS(EIDX),   TEND          )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'PRSMET' )
         RETURN
      END IF

C
C     Set the frame output argument.  Check the frame.
C
      FRAME = VSTRS(REFIDX) 

      CALL NAMFRM ( FRAME, REFCOD )
 
      IF ( REFCOD .EQ. 0 ) THEN
 
         CALL SETMSG ( 'The reference frame # is not supported. '  //
     .                 'This problem may be due to the lack of a ' //
     .                 'STRING_MAPPING assignment in your setup '  //
     .                 'file; such an assignment may be used to '  //
     .                 'map a frame name not recognized by SPICE ' //
     .                 'to an equivalent name that is recognized. '//
     .                 'For example, the frame name EME2000 '      //
     .                 'may be mapped to the frame name J2000. '   //
     .                 'See the OEM2SPK User''s Guide '            //
     .                 'for more information.  Also note that '    //
     .                 'the setup file must end with a newline '   //
     .                 'character in order for the final line '    //
     .                 'of the file to be readable.'               )
         CALL ERRCH  ( '#', FRAME                                  )
         CALL SIGERR ( 'SPICE(INVALIDREFFRAME)'                    )
         CALL CHKOUT ( 'PRSMET'                                    )
         RETURN
 
      END IF

C
C     All of the outputs are set.
C
      CALL CHKOUT ( 'PRSMET' )
      RETURN
      END
