C$Procedure KEYVLC ( Keyword and value parser, character )

      SUBROUTINE KEYVLC ( INPUT, LINE, LNUM, KEYWRD, VALUE )

C$ Abstract
C
C     Parse a "keyword = value" statement in an OEM file and
C     return the keyword and the value as a character string.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     OEM2SPK
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'errhnd.inc'

      CHARACTER*(*)         INPUT
      CHARACTER*(*)         LINE
      INTEGER               LNUM
      CHARACTER*(*)         KEYWRD
      CHARACTER*(*)         VALUE         
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     INPUT      I   Input OEM file.
C     LINE       I   Line on which keyword was found.
C     LNUM       I   Line number.
C     KEYWRD     O   Keyword.
C     VALUE      O   Character string value associated with keyword.
C
C$ Detailed_Input
C
C     INPUT          is the name of the OEM file from which a time
C                    tag of interest was read.  FILE is provided
C                    to support construction of error messages.
C                     
C     LINE           is an OEM line containing a time tag.  LINE is 
C                    provided to support construction of error messages.
C
C     LNUM           is the ordinal position of LINE in the file.
C                     
C$ Detailed_Output
C
C     KEYWRD         is the keyword found on the left hand side of
C                    a "keyword = value" expression.
C
C     VALUE          is the string value found on the right hand 
C                    side of a "keyword = value" expression.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     See the description of the argument INPUT.
C
C$ Exceptions
C
C     1) If the input line does not contain an equal sign, the error
C        SPICE(INVALIDFORMAT) is signaled.
C
C     2) If the input line contains no characters to the left 
C        of the equal sign, the error SPICE(INVALIDFORMAT) is signaled.
C
C     3) If the input line contains no non-blank text to the left 
C        of the equal sign, the error SPICE(INVALIDFORMAT) is signaled.
C
C     4) If the input line contains no non-blank text to the right 
C        of the equal sign, the error SPICE(INVALIDFORMAT) is signaled.
C
C$ Particulars
C
C     This routine centralizes algorithms required for parsing
C     metadata lines for the program OEM2SPK.  
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-1,
C          September, 2004.
C
C$ Version
C
C-    OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&


C
C     SPICELIB functions
C
      INTEGER               RTRIM
      LOGICAL               RETURN

C
C     Local parameters
C

C
C     Local variables
C
      CHARACTER*(LMSGLN)    TEMPLT ( 3 )

      INTEGER               L
      INTEGER               LOC
      
      LOGICAL               FIRST

C
C     Saved variables
C
      SAVE                  FIRST
      SAVE                  TEMPLT

C
C     Initial values
C
      DATA                  FIRST / .TRUE. /


      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'KEYVLC' )


      IF ( FIRST ) THEN

         TEMPLT(1) =  'Equals sign "=" not found in expected '    //
     .                '"keyword = value" expression '             //
     .                'in line #* of file <*>.  Initial portion ' //
     .                'of line is: *'                           

         TEMPLT(2) =  'Keyword not found in expected '            //
     .                '"keyword = value" expression '             //
     .                'in line #* of file <*>.  Initial portion ' //
     .                'of line is: *'                           

         TEMPLT(3) =  'RHS value not found in expected '          //
     .                '"keyword = value" expression '             //
     .                'in line #* of file <*>.  Initial portion ' //
     .                'of line is: *'                           

         FIRST = .FALSE.

      END IF


C
C     Get the non-blank length of the input line.  Locate the
C     equals sign.
C
      L     =  RTRIM ( LINE )
      LOC   =  INDEX ( LINE,  '=' )      

C
C     Check for missing equals sign.
C
      IF ( LOC .EQ. 0 ) THEN
         
         CALL SETMSG (  TEMPLT(1)             )
         CALL ERRINT ( '*',  LNUM             )
         CALL ERRCH  ( '*',  INPUT            )
         CALL ERRCH  ( '*',  LINE             )
         CALL SIGERR ( 'SPICE(INVALIDFORMAT)' )
         CALL CHKOUT ( 'KEYVLC'               )
         RETURN

      END IF

C
C     Check for missing LHS of "keyword = value" expression.
C
      IF ( LOC .EQ. 1 ) THEN
         
         CALL SETMSG (  TEMPLT(2)             )
         CALL ERRINT ( '*',  LNUM             )
         CALL ERRCH  ( '*',  INPUT            )
         CALL ERRCH  ( '*',  LINE             )
         CALL SIGERR ( 'SPICE(INVALIDFORMAT)' )
         CALL CHKOUT ( 'KEYVLC'               )
         RETURN

      END IF

C
C     Check for blank LHS.  LOC is at least 2 here.
C
      IF ( LINE(:LOC-1) .EQ. ' ' ) THEN
         
         CALL SETMSG (  TEMPLT(2)             )
         CALL ERRINT ( '*',  LNUM             )
         CALL ERRCH  ( '*',  INPUT            )
         CALL ERRCH  ( '*',  LINE             )
         CALL SIGERR ( 'SPICE(INVALIDFORMAT)' )
         CALL CHKOUT ( 'KEYVLC'               )
         RETURN

      END IF

C
C     Check for blank RHS.  LOC is at least 2 here.
C
      IF ( L .EQ. LOC ) THEN
         
         CALL SETMSG (  TEMPLT(3)             )
         CALL ERRINT ( '*',  LNUM             )
         CALL ERRCH  ( '*',  INPUT            )
         CALL ERRCH  ( '*',  LINE             )
         CALL SIGERR ( 'SPICE(INVALIDFORMAT)' )
         CALL CHKOUT ( 'KEYVLC'               )
         RETURN

      END IF

C
C     LOC is at least 2, and there's at least one non-blank 
C     character past the equal sign.
C
      KEYWRD = LINE (       : LOC-1 )
      VALUE  = LINE ( LOC+1 :       )

      CALL CHKOUT ( 'KEYVLC' )
      RETURN
      END

