C$Procedure CVTOEM ( Convert OEM )
 
      SUBROUTINE CVTOEM ( SETUP, INPUT, OUTPUT, APPEND, CMTFIL )
 
C$ Abstract
C
C     Convert OEM file to SPK format.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     OEM2SPK
C
C$ Declarations
 
      IMPLICIT NONE
 
      CHARACTER*(*)         SETUP
      CHARACTER*(*)         INPUT
      CHARACTER*(*)         OUTPUT
      LOGICAL               APPEND
      CHARACTER*(*)         CMTFIL
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     SETUP      I   Name of setup file.
C     INPUT      I   Name of input OEM file.
C     OUTPUT     I   Name of output SPK file.
C     APPEND     I   Append flag.
C     CMTFIL     I   Comment file name.
C
C$ Detailed_Input
C
C     SETUP          is the name of an OEM2SPK setup file.
C
C     INPUT          is the name of an OEM file to be converted
C                    to SPK format.  This file conforms to any
C                    format specification given by a current 
C                    or past version of the CCSDS Orbit Data Messages 
C                    Red Book.
C
C     OUTPUT         is the name of an SPK file to be written.
C                    OUTPUT may be a new or existing file.
C                    OUTPUT may contain type 13 or type 9 segments.
C
C     APPEND         is a logical flag indicating whether to 
C                    append data to an existing output file.
C                    If APPEND is .TRUE., the file designated 
C                    by OUTPUT must exist.  If APPEND is .FALSE.,
C                    the file designated by OUTPUT must not exist.
C
C     CMTFIL         is the name of a comment file whose contents
C                    are to be added to the comment area of
C                    the SPK file created by this program.  The
C                    comment file contents precede the default
C                    comments added by OEM2SPK.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If the version ID line is not found in the OEM file,
C        the error SPICE(INVALIDFORMAT) will be signaled.
C
C     2) If the format version is not supported,
C        the error SPICE(NOTSUPPORTED) will be signaled.
C
C     3) Errors detected while reading the OEM file will be 
C        diagnosed by routines called by this routine.
C
C     4) Errors detected while parsing the OEM file will be
C        diagnosed by routines called by this routine.
C
C     5) Errors detected while opening the output SPK file will be
C        diagnosed by routines called by this routine.
C
C     6) Errors detected while writing the output SPK file will be
C        diagnosed by routines called by this routine.
C
C$ Files
C
C     See the descriptions of INPUT and OUTPUT above.
C
C$ Particulars
C
C     This routine is intended to enable OEM2SPK to accommodate
C     updates to the OEM format specification.  CVTOEM reads the
C     OEM's version ID line, then calls the appropriate subroutine
C     to convert the OEM file to SPK format.
C
C     Currently, there are two OEM format versions: version 1.0 and
C     version 2.0. The expected version ID line should be, up to
C     variations of white space:
C
C        CCSDS_OEM_VERS = 1.0
C
C     or
C
C        CCSDS_OEM_VERS = 2.0
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-1,
C          September, 2004.
C
C     [2]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-2,
C          November, 2009.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C     M.C. Kim    (JPL)
C
C$ Version
C
C-    OEM2SPK Version 2.0.0, 30-AUG-2013 (MCK)
C
C        Upgraded to read OEM 2.0 as well as OEM 1.0.
C
C-    OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&


C
C     SPICELIB functions
C
      LOGICAL               EQSTR
      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C
      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 80 )



C
C     Local variables
C
      CHARACTER*(LNSIZE)    KEYWRD
      CHARACTER*(LNSIZE)    LINE
      CHARACTER*(LNSIZE)    TYPE
      CHARACTER*(LNSIZE)    VERSTR


      INTEGER               LNUM

      LOGICAL               EOF



      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'CVTOEM' )
C

C     Find the version line in the input file.  Hand off the
C     conversion problem to the appropriate routine.
C

      CALL RDLIN ( INPUT, LINE, LNUM, TYPE, EOF ) 

      DO WHILE (        ( TYPE .NE. 'VERSION_LINE' ) 
     .            .AND. ( .NOT. EOF                )
     .            .AND. ( .NOT. FAILED()           )  )
         
         CALL RDLIN ( INPUT, LINE, LNUM, TYPE, EOF )

      END DO

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'CVTOEM' )
         RETURN
      END IF

      IF ( EOF ) THEN

         CALL SETMSG ( 'Format error in OEM file <#>: version line ' //
     .                 'not found.'                                   )
         CALL ERRCH  ( '#', INPUT                                     )
         CALL SIGERR ( 'SPICE(INVALIDFORMAT)'                         )
         CALL CHKOUT ( 'CVTOEM'                                       )
         RETURN

      END IF

C
C     Pick off the format version from the version line.
C

      CALL KEYVLC ( INPUT, LINE, LNUM, KEYWRD, VERSTR ) 

      IF (  EQSTR( VERSTR, '1.0' )  ) THEN
C
C        Call the converter for this format type.
C
         CALL CVT01 ( SETUP, INPUT, OUTPUT, APPEND, CMTFIL )

      ELSE IF (  EQSTR( VERSTR, '2.0' )  ) THEN
C
C        Call the converter for this format type ( version 2.0 ).
C        This includes the covariance or accelerations.
C
         CALL CVT02 ( SETUP, INPUT, OUTPUT, APPEND, CMTFIL )

      ELSE

         CALL SETMSG ( 'Format version <#> of OEM file <#> '      //
     .                 'is not supported by this program. '       //
     .                 'Check whether a later version of OEM2SPK '//
     .                 'is available.'                             )
         CALL ERRCH  ( '#', VERSTR                                 )
         CALL ERRCH  ( '#', INPUT                                  )
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                       )
         CALL CHKOUT ( 'CVTOEM'                                    )
         RETURN

      END IF

      CALL CHKOUT ( 'CVTOEM' )
      RETURN
      END

      
