C$Procedure TIMCVT ( Time conversion )
 
      SUBROUTINE TIMCVT ( FILE, LINE, LNUM, TIMSYS, TIMSTR, ET )
 
C$ Abstract
C
C     Convert an OASW orbit/attitude metadata or state time tag to TDB.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'errhnd.inc'

      CHARACTER*(*)         FILE
      CHARACTER*(*)         LINE
      INTEGER               LNUM
      CHARACTER*(*)         TIMSYS
      CHARACTER*(*)         TIMSTR
      DOUBLE PRECISION      ET
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     FILE       I   Input OASW orbit/attitude file.
C     LINE       I   Line on which time tag was found.
C     LNUM       I   Line number.
C     TIMSYS     I   Time system.
C     TIMSTR     I   Time string.
C     ET         O   TDB value corresponding to TIMSTR.
C
C$ Detailed_Input
C                     
C     FILE           is the name of the OASW orbit/attitude file from
C                    which a time tag of interest was read. FILE is
C                    provided to support construction of error
C                    messages.
C                     
C     LINE           is an OASW orbit/attitude file line containing a
C                    time tag. LINE is provided to support construction
C                    of error messages.
C
C     LNUM           is the ordinal position of LINE in the file.
C
C     TIMSYS         is the name of the time system associated with
C                    the input time tag.
C 
C     TIMSTR         is a time tag read from an OASW orbit/attitude
C                    file. The tag must be formatted so that it can be
C                    parsed by the SPICELIB routine TPARTV.
C
C$ Detailed_Output
C
C     ET             is the TDB epoch, expressed as seconds past J2000,
C                    defined by TIMSYS and TIMSTR.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If the input string TIMSTR cannot be parsed by TPARTV, the 
C        error SPICE(INVALIDTIME) will be signaled.
C
C$ Files
C
C     See the description of the argument FILE.
C
C$ Particulars
C
C     The OASW orbit/attitude specification calls for time strings
C     conforming to ISO formats but representing times in a variety of
C     systems. Since SPICELIB assumes ISO-formatted strings represent
C     UTC epochs, this routine first converts the input string to
C     calendar format.  Next, a time system label is appended to the
C     string, and the string is converted to TDB. A small round-off
C     error is introduced by these conversion steps.
C     
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    MEX2KER Version 1.0.1, 21-JUN-2012 (NJB)
C
C-&

      LOGICAL               RETURN



      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 80 )

      INTEGER               MAXVEC
      PARAMETER           ( MAXVEC = 6 )

      INTEGER               TIMLEN
      PARAMETER           ( TIMLEN = 50 )

      INTEGER               TYPLEN
      PARAMETER           ( TYPLEN = 10 )

      INTEGER               MAXMOD
      PARAMETER           ( MAXMOD = 20 )


      CHARACTER*(LMSGLN)    ERRMSG
      CHARACTER*(LNSIZE)    MODIFY ( MAXMOD )
      CHARACTER*(LNSIZE)    PICTUR
      CHARACTER*(TIMLEN)    TMPTIM
      CHARACTER*(TYPLEN)    TTYPE

      DOUBLE PRECISION      T
      DOUBLE PRECISION      TVEC   ( MAXVEC )

      INTEGER               NTVEC

      LOGICAL               MODS
      LOGICAL               SUCCES
      LOGICAL               YABBRV
     


      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'TIMCVT' )

C
C     See wether TPARTV understands the time string.  TPARTV has the
C     convenient property of not signaling an error if it doesn't
C     recognize the time string.
C
      CALL TPARTV ( TIMSTR, TVEC,   NTVEC,  TTYPE,  MODIFY,
     .              MODS,   YABBRV, SUCCES, PICTUR, ERRMSG )

      IF ( .NOT. SUCCES ) THEN

         CALL SETMSG ( 'Time string found at line #* in file  '    //
     .                 '<*> could not be parsed. Line was <*> * '   )
         CALL ERRINT ( '*',  LNUM                                   )
         CALL ERRCH  ( '*',  FILE                                   ) 
         CALL ERRCH  ( '*',  LINE                                   ) 
         CALL ERRCH  ( '*',  ERRMSG                                 ) 
         CALL SIGERR ( 'SPICE(INVALIDTIME)'                         )
         CALL CHKOUT ( 'TIMCVT'                                     )
         RETURN

      END IF

C
C     Append the appropriate time system flag to the input time string.
C
      TMPTIM = TIMSTR

C
C     Convert the input string to calendar format, so we can
C     append a time system label.
C
      CALL STR2ET ( TIMSTR, T )
      CALL TIMOUT ( T,
     .              'YYYY MON DD HR:MN:SC.#########',
     .              TMPTIM                                  )
      CALL SUFFIX ( TIMSYS, 1, TMPTIM )
         
C
C     We're ready for STR2ET to take a shot at translating this 
C     string.
C
      CALL STR2ET ( TMPTIM, ET )

      CALL CHKOUT ( 'TIMCVT' )
      RETURN
      END


      
      
