C$Procedure RDMEX ( Read MEX2KER input file --- umbrella routine )
 
      SUBROUTINE RDMEX ( FILE, LINE, LNUM, TYPE, EOF )
 
C$ Abstract
C
C     Umbrella routine for MEX2KER low-level readers.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
 
      IMPLICIT NONE
 
      CHARACTER*(*)         FILE
      CHARACTER*(*)         LINE
      INTEGER               LNUM
      CHARACTER*(*)         TYPE
      LOGICAL               EOF

C$ Brief_I/O
C
C     VARIABLE  I/O  Entry points
C     --------  ---  --------------------------------------------------
C     FILE       I   RDLIN
C     LINE       O   RDLIN
C     LNUM       O   RDLIN
C     TYPE       O   RDLIN
C     EOF        O   RDLIN
C
C$ Detailed_Input
C
C     See the entry points for a description of their arguments.
C                     
C$ Detailed_Output
C
C     See the entry points for a description of their arguments.
C                     
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If this routine is called directly, the error 
C        SPICE(BOGUSENTRY) is signaled.
C
C     See the entry points for a description of exceptions specific
C     to those routines.
C
C$ Files
C
C     This suite of routines supports sequential reading of lines
C     of text from an ESOC OASW orbit/attitude file.
C
C$ Particulars
C
C     This routine simplifies the logic required in order for 
C     the program MEX2KER to parse an MEX file.  Lines obtained
C     from this routine are always non-blank, and a line 
C     classification string and line number are returned with
C     each line.
C
C     The entry point PUSHLN enables a caller to "put back" a line
C     so that the line will be returned on the next read operation.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    MEX2KER Version 2.1.0, 28-JUN-2012 (NJB)
C
C        Unused parameter declaration for FILSIZ was deleted. Initial
C        blank string values are now assigned to saved variables SAVTYP
C        and LINBUF. Header format was updated.
C
C-    MEX2KER Version 2.0.0, 03-FEB-2003 (NJB)
C
C-&

C
C     SPICELIB  functions
C     
      INTEGER               RTRIM

C
C     Local parameters
C
      CHARACTER*(*)         ORBMRK
      PARAMETER           ( ORBMRK = 'ESOC_TOS_GFI_ORBIT_FILE_VERSION' )

      CHARACTER*(*)         ATTMRK
      PARAMETER           ( ATTMRK = 
     .                      'ESOC_TOS_GFI_ATTITUDE_FILE_VERSION' )

      CHARACTER*(*)         CREMRK
      PARAMETER           ( CREMRK = 'CREATION_DATE' )
      
      CHARACTER*(*)         COMMRK
      PARAMETER           ( COMMRK = 'COMMENT' )
      
      CHARACTER*(*)         MTBMRK
      PARAMETER           ( MTBMRK = 'META_START' )

      CHARACTER*(*)         MTEMRK
      PARAMETER           ( MTEMRK = 'META_STOP' )

      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 256 )

      INTEGER               NBUFF
      PARAMETER           ( NBUFF  = 1 )

C
C     Local variables
C
      CHARACTER*(LNSIZE)    LINBUF
      CHARACTER*(LNSIZE)    MAPATT
      CHARACTER*(LNSIZE)    MAPCOM
      CHARACTER*(LNSIZE)    MAPCRE
      CHARACTER*(LNSIZE)    MAPMTB
      CHARACTER*(LNSIZE)    MAPMTE
      CHARACTER*(LNSIZE)    MAPORB
      CHARACTER*(LNSIZE)    SAVTYP

      INTEGER               LEVEL
      INTEGER               LNO

      LOGICAL               FIRST
      LOGICAL               INDATA
      LOGICAL               INMETA
      LOGICAL               SAVEOF

C
C     Saved variables
C
    
      SAVE                  FIRST
      SAVE                  INDATA
      SAVE                  INMETA
      SAVE                  LEVEL
      SAVE                  LINBUF
      SAVE                  LNO
      SAVE                  MAPATT
      SAVE                  MAPCOM
      SAVE                  MAPCRE
      SAVE                  MAPMTB
      SAVE                  MAPMTE
      SAVE                  MAPORB
      SAVE                  SAVEOF
      SAVE                  SAVTYP

C
C     Initial values
C
      DATA                  FIRST  / .TRUE.  /
      DATA                  INDATA / .FALSE. /
      DATA                  INMETA / .FALSE. /
      DATA                  LEVEL  /  0      /
      DATA                  LINBUF / ' '     /
      DATA                  LNO    /  0      /
      DATA                  SAVEOF / .FALSE. /
      DATA                  SAVTYP / ' '     /


      CALL CHKIN ( 'RDMEX' )

      CALL SIGERR ( 'SPICE(BOGUSENTRY)' )

      CALL CHKOUT ( 'RDMEX' )
      RETURN





C$Procedure RDLIN ( Read non-blank text line from MEX file )

      ENTRY RDLIN ( FILE, LINE, LNUM, TYPE, EOF ) 
 
C$ Abstract
C
C     Read a non-blank line of text from an MEX file.  Classify
C     the line.  Return the line number. 
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
C 
C     CHARACTER*(*)         FILE
C     CHARACTER*(*)         LINE
C     INTEGER               LNUM
C     CHARACTER*(*)         TYPE
C     LOGICAL               EOF
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     FILE       I   Name of MEX file.
C     LINE       O   Current non-blank text line read from MEX.
C     LNUM       O   Line number.
C     TYPE       O   Line classification.
C     EOF        O   End-of-file flag.
C
C$ Detailed_Input
C
C     FILE           is the name of an MEX file to be read.
C                     
C$ Detailed_Output
C
C     LINE           is the "next" non-blank line in the MEX file.
C                    Lines are read sequentially from the file.
C
C                    LINE contains no more than 255 characters.
C
C     LNUM           is the ordinal position of LINE in the file.
C                    Line numbers start at 1.
C
C     TYPE           is a string indicating the classification of the
C                    output line.  Possible values are:
C
C                       'VERSION_LINE'            
C                       'CREATION_DATE_LINE'     
C                       'COMMENT'                
C                       'METADATA_START_LINE'    
C                       'METADATA_STOP_LINE'      
C                       
C     EOF            is a logical flag indicating the end of the input
C                    file has been reached.  When EOF is .TRUE., the 
C                    other output arguments are undefined.  Subsequent
C                    read attempts will continue to return EOF = .TRUE.
C                     
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If the current line cannot be classified, the error 
C        SPICE(FORMATERROR) is signaled.
C
C     2) If a read error occurs, the problem will be diagnosed by
C        routines called by this routine.
C
C$ Files
C
C     This suite of routines supports sequential reading of lines
C     of text from an ESOC OASW orbit/attitude file.
C
C$ Particulars
C
C     This routine simplifies the parsing logic in the program 
C     MEX2KER by centralizing line classification functions.  
C
C     Note that in some circumstances, after reading a line, it 
C     is convenient to push the line back on the input stream. 
C     The entry point PUSHLN is provided for this purpose.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    MEX2KER Version 1.0.1, 28-JUN-2012 (NJB)
C
C        Header format was updated.
C
C-    MEX2KER Version 1.0.0, 30-JAN-2003 (NJB)
C
C-&
      CALL CHKIN ( 'RDLIN' )

C
C     On the first pass, fetch the marker values to look for.
C     These are obtained from the inverse of the string mapping
C     defined in the kernel pool.
C
      IF ( FIRST ) THEN

         CALL CMAPI ( COMMRK, MAPCOM )
         CALL CMAPI ( ATTMRK, MAPATT )
         CALL CMAPI ( ORBMRK, MAPORB )
         CALL CMAPI ( CREMRK, MAPCRE )
         CALL CMAPI ( MTBMRK, MAPMTB )
         CALL CMAPI ( MTEMRK, MAPMTE )
         
         FIRST = .FALSE.

      END IF

C
C     If we're already at EOF, just return.
C
      IF ( SAVEOF ) THEN
         
         EOF = .TRUE.

         CALL CHKOUT ( 'RDLIN' )
         RETURN

      END IF

C
C     If there's a stacked line, return it.  The saved line
C     number has already been set by the push operation that
C     stacked the line.
C
      IF ( LEVEL .GE. 1 ) THEN

         LINE  = LINBUF
         LEVEL = LEVEL - 1
         TYPE  = SAVTYP
         EOF   = SAVEOF

         CALL CHKOUT ( 'RDLIN' )
         RETURN

      ELSE
C
C        Read a line from the file.  Keep going until we get a 
C        non-blank line.
C
         CALL RDTEXT ( FILE, LINE, EOF )

         SAVEOF = EOF
         LNO    = LNO + 1

         DO WHILE (  ( .NOT. EOF ) .AND. ( LINE .EQ. ' ' )  )

            CALL RDTEXT ( FILE, LINE, EOF )

            SAVEOF = EOF
            LNO    = LNO + 1

         END DO

         IF ( EOF ) THEN
            CALL CHKOUT ( 'RDLIN' )
            RETURN
         END IF

      END IF

C
C     We found a non-blank line.  Classify it.
C
C     Note that we order our checks so that we don't look 
C     for special flags, other than comment markers and
C     the metadata end flag, within a metadata block.
C
      CALL LJUST ( LINE, LINE )


      IF ( LINE(:RTRIM(MAPCOM)) .EQ. MAPCOM ) THEN

         TYPE = 'COMMENT'


      ELSE IF ( LINE .EQ. MAPMTE ) THEN

         TYPE = 'METADATA_STOP_LINE'

C
C        Change our state to indicate we're no longer reading 
C        a metadata block and that we are reading data lines.
C
         INMETA = .FALSE.
         INDATA = .TRUE.


      ELSE IF ( INMETA ) THEN

         TYPE = 'METADATA_LINE'


      ELSE IF (     ( LINE(:RTRIM(MAPATT)) .EQ. MAPATT )  
     .         .OR. ( LINE(:RTRIM(MAPORB)) .EQ. MAPORB )  ) THEN

         TYPE = 'VERSION_LINE'


      ELSE IF (  LINE(:RTRIM(MAPCRE)) .EQ. MAPCRE ) THEN

         TYPE = 'CREATION_DATE_LINE'


      ELSE IF ( LINE .EQ. MAPMTB ) THEN

         TYPE = 'METADATA_START_LINE'

C
C        Change our state to indicate we're reading a metadata 
C        block.  If we were reading data previously, we're no
C        longer doing so.
C
         INMETA = .TRUE.
         INDATA = .FALSE.


      ELSE IF ( INDATA ) THEN

         TYPE = 'DATA_LINE'

      ELSE
C
C        We really oughtn't get to this point.
C
         CALL SETMSG ( 'Line #* is unclassified: *' )
         CALL ERRINT ( '*',  LNO                    )
         CALL ERRCH  ( '*',  LINE                   )
         CALL SIGERR ( 'SPICE(FORMATERROR)'         )
         CALL CHKOUT ( 'RDLIN'                      )
         RETURN

      END IF

      
      LINBUF = LINE
      LNUM   = LNO
      SAVTYP = TYPE
      
      CALL CHKOUT ( 'RDLIN' )
      RETURN




C$Procedure PUSHLN ( Push MEX line )

      ENTRY PUSHLN
 
C$ Abstract
C
C     Push a line returned from RDLIN back on the virtual MEX 
C     input stream.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
C 
C     None.
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     None.
C
C$ Detailed_Input
C
C     None.
C                     
C$ Detailed_Output
C
C     None.
C                     
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If two push operations are requested with no intervening
C        read operation, the error SPICE(BUFFERFULL) is signaled.
C
C$ Files
C
C     This suite of routines supports sequential reading of lines
C     of text from an ESOC OASW orbit/attitude file.
C
C$ Particulars
C
C     Note that in some circumstances, after reading a line, it 
C     is convenient to push the line back on the input stream. 
C     This entry point is provided for this purpose.  After this
C     routine is called, the next call to RDLIN will return the
C     same outputs as the previous call to RDLIN.  Subsequent
C     calls to RDLIN will return new lines from the input MEX file.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     1) Multiple lines cannot be pushed without intervening RDLIN
C        calls.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    MEX2KER Version 1.0.1, 28-JUN-2012 (NJB)
C
C        Header format was updated.
C
C-    MEX2KER Version 1.0.0, 30-JAN-2003 (NJB)
C
C-&
      CALL CHKIN ( 'PUSHLN' )

      IF ( LEVEL .LT. NBUFF ) THEN

         LEVEL  = 1

C
C        The line number remains unchanged.
C
      ELSE

         CALL SETMSG ( 'Buffer full; level = 1.' )
         CALL SIGERR ( 'SPICE(BUFFERFULL)'       )
         CALL CHKOUT ( 'PUSHLN'                  )
         RETURN

      END IF


      CALL CHKOUT ( 'PUSHLN' )
      RETURN
      END



 
      
