C$Procedure PRSMET ( Parse metadata block )
 
      SUBROUTINE PRSMET ( INPUT, FILTYP, SUBTYP, OBJECT, CENTER, 
     .                    FRAME, TBEG,   TEND,   TIMSYS, CRTTIM )
 
C$ Abstract
C
C     Parse an MEX ASCII format orbit or attitude file metadata block.
C     Return file type, segment descriptor information, the name of 
C     the time system associated with the metadata block, and 
C     parameters indicating the SPICE data type to use.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'ck05.inc'
      INCLUDE 'spk18.inc'

      CHARACTER*(*)         INPUT
      CHARACTER*(*)         FILTYP
      INTEGER               SUBTYP
      INTEGER               OBJECT
      INTEGER               CENTER
      CHARACTER*(*)         FRAME
      DOUBLE PRECISION      TBEG
      DOUBLE PRECISION      TEND
      CHARACTER*(*)         TIMSYS
      CHARACTER*(*)         CRTTIM
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     INPUT      I   Name of OASW orbit/attitude file.
C     FILTYP     O   File type.
C     SUBTYP     O   Subtype of SPK or CK data type.
C     OBJECT     O   Ephemeris object.
C     CENTER     O   Center of motion of object.
C     FRAME      O   Reference frame name.
C     TBEG       O   Segment begin time.
C     TEND       O   Segment end time.
C     TIMSYS     O   Time system name.
C     NCOMP      O   Number of numeric components in datum.
C     CRTTIM     O   Input file creation time.
C
C$ Detailed_Input
C           
C     INPUT          is the name of an OASW orbit/attitude file from
C                    which a metadata block to be read.
C          
C$ Detailed_Output
C
C     FILTYP         is the type of the input file:  'ORBIT' or
C                    'ATTITUDE'.
C
C     SUBTYP         is the subtype of the SPK or CK data type 
C                    corresponding to the input data representation.
C                    See the include file spk18.inc for SPK type 18
C                    subtype codes.
C
C     OBJECT         is the NAIF integer ID code of the ephemeris
C                    object to which the data following the 
C                    current metadata block correspond.  
C
C     CENTER         is the NAIF integer ID code of the center of
C                    motion of the ephemeris object.  CENTER is 
C                    applicable only when the file type is 'ORBIT'.
C
C     FRAME          is the name of the reference frame of the data.
C
C     TBEG           is the ephemeris time, expressed as seconds
C                    past J2000 TDB, of the start of coverage.
C
C                    TBEG is greater than or equal to the epoch 
C                    of the first state of the contiguous set of 
C                    states following the metadata block.
C
C     TEND           is the ephemeris time, expressed as seconds
C                    past J2000 TDB, of the end of coverage.
C
C                    TEND is less than or equal to the epoch 
C                    of the last state of the contiguous set of
C                    states following the metadata block.
C
C     TIMSYS         is the name of the time system associated with
C                    the metadata block.
C
C     CRTTIM         is the creation time of the input orbit or
C                    attitude file. 
C                
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If a required metadata keyword is not found in the first
C        metadata block seen, the error SPICE(MISSINGITEM) is 
C        signaled.  The last metadata value seen will be used as 
C        a default for any item not supplied in a subsequent metadata
C        block.
C
C     2) If either the object or center names cannot be mapped to
C        a NAIF integer ID code, the error SPICE(IDCODENOTFOUND)
C        is signaled.
C
C     3) If any metadata line does not follow the expected 
C        "keyword = value" syntax, the error will be diagnosed
C        by routines called by this routine.
C
C     4) If any time bounds given by the metadata block (either
C        start and stop times, or usable start and stop times)
C        cannot be parsed and converted to TDB, the error will
C        be diagnosed by routines called by this routine.
C
C     5) If the reference frame name cannot be mapped to a NAIF
C        integer code, the error SPICE(INVALIDREFFRAME) will be 
C        signaled.
C
C$ Files
C
C     See the description of INPUT.
C
C$ Particulars
C
C     This routine assumes that the first non-blank, non-comment
C     line it reads belongs to a metadata block.  The last line
C     this routine will normally read is a metadata block end 
C     marker.
C
C     If both the USABLE_START_TIME and USABLE_STOP_TIME keywords
C     are found in the metadata block, the corresponding epochs
C     will be used as the output segment descriptor's time bounds.
C     If these keywords are not present, the time bounds will be
C     derived from the values associated with the START_TIME and
C     STOP_TIME keywords.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     1)  This routine assumes that any time tag used in an OASW
C         orbit/attitude file can be parsed by the SPICE routine
C         TPARTV.
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Red Book, revision Acton 502-H.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C     J. Diaz del Rio (ESA)
C
C$ Version
C
C-    MEX2KER Version 2.0.0, 15-JUL-2014 (NJB) (JDR) (BVS)
C
C        Integrated updates by Jorge Diaz del Rio. These updates
C        enable inclusion of the input file creation time in the
C        comments created by this routine.
C
C-    MEX2KER Version 1.10, 26-JUN-2012 (NJB)
C
C        Removed declaration of unused variable
C        TVSIZ. Updated header format.
C
C-    MEX2KER Version 1.0.0, 13-MAY-2002 (NJB)
C
C-&


C
C     SPICELIB functions
C
      INTEGER               CARDC
      INTEGER               ISRCHC

      LOGICAL               ELEMC
      LOGICAL               FAILED
      
C
C     Local parameters
C
      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 80 )

      INTEGER               NREQA
      PARAMETER           ( NREQA  =  8 )

      INTEGER               NREQO
      PARAMETER           ( NREQO  =  10 )

      INTEGER               NREQMX
      PARAMETER           ( NREQMX = NREQO )

      INTEGER               NKEYS
      PARAMETER           ( NKEYS  = 12 )

      INTEGER               LBCELL
      PARAMETER           ( LBCELL = -5 )

C
C     Indices of metadata items
C
      INTEGER               CTRIDX 
      PARAMETER           ( CTRIDX = 1 )

      INTEGER               DERIDX
      PARAMETER           ( DERIDX = CTRIDX + 1 )

      INTEGER               FTPIDX
      PARAMETER           ( FTPIDX = DERIDX  + 1 )

      INTEGER               OBJIDX
      PARAMETER           ( OBJIDX = FTPIDX + 1 )

      INTEGER               REFIDX
      PARAMETER           ( REFIDX = OBJIDX + 1 )
      
      INTEGER               TBIDX
      PARAMETER           ( TBIDX  = REFIDX + 1 )

      INTEGER               TEIDX
      PARAMETER           ( TEIDX  = TBIDX  + 1 )

      INTEGER               SYSIDX
      PARAMETER           ( SYSIDX = TEIDX + 1 )
      
      INTEGER               UBIDX
      PARAMETER           ( UBIDX  = SYSIDX + 1 )

      INTEGER               UEIDX
      PARAMETER           ( UEIDX  = UBIDX  + 1 )

      INTEGER               NVRIDX
      PARAMETER           ( NVRIDX = UEIDX + 1 )

      INTEGER               CTIDX
      PARAMETER           ( CTIDX  = NVRIDX + 1 )

C
C     Local variables
C
      CHARACTER*(LNSIZE)    ITEMS   ( LBCELL : NKEYS )
      CHARACTER*(LNSIZE)    KEYWRD
      CHARACTER*(LNSIZE)    KEYS    ( NKEYS )
      CHARACTER*(LNSIZE)    KLINES  ( NKEYS )
      CHARACTER*(LNSIZE)    LINE
      CHARACTER*(LNSIZE)    LOCVAL
      CHARACTER*(LNSIZE)    RITEMS  ( LBCELL : NREQMX )
      CHARACTER*(LNSIZE)    RAKEYS  ( NREQA )
      CHARACTER*(LNSIZE)    ROKEYS  ( NREQO )
      CHARACTER*(LNSIZE)    TYPE
      CHARACTER*(LNSIZE)    VALUE
      CHARACTER*(LNSIZE)    VSTRS   ( NKEYS )
      
      INTEGER               BIDX
      INTEGER               EIDX
      INTEGER               KNUMS   ( NKEYS )
      INTEGER               LOC
      INTEGER               LOC2
      INTEGER               LNUM
      INTEGER               REFCOD      

      LOGICAL               EOF
      LOGICAL               FIRST
      LOGICAL               FOUND
      LOGICAL               HAVEAV

C
C     Saved variables
C
      SAVE                  ITEMS
      SAVE                  RITEMS
      SAVE                  KEYS
      SAVE                  KLINES
      SAVE                  KNUMS
      SAVE                  RAKEYS
      SAVE                  ROKEYS
      SAVE                  VSTRS

C
C     Initial values
C
      DATA                  KEYS     /  'CENTER_NAME',
     .                                  'DERIVATIVES_FLAG',
     .                                  'FILE_TYPE',
     .                                  'OBJECT_NAME',
     .                                  'REF_FRAME', 
     .                                  'START_TIME',
     .                                  'STOP_TIME',
     .                                  'TIME_SYSTEM',
     .                                  'USABLE_START_TIME',
     .                                  'USABLE_STOP_TIME',
     .                                  'VARIABLES_NUMBER',
     .                                  'CREATION_DATE'      /

      DATA                  RAKEYS   /  'DERIVATIVES_FLAG',
     .                                  'FILE_TYPE',
     .                                  'OBJECT_NAME',
     .                                  'REF_FRAME', 
     .                                  'START_TIME',
     .                                  'STOP_TIME',
     .                                  'TIME_SYSTEM',
     .                                  'VARIABLES_NUMBER'   /
     

      DATA                  ROKEYS   /  'CENTER_NAME',
     .                                  'DERIVATIVES_FLAG',
     .                                  'FILE_TYPE',
     .                                  'OBJECT_NAME',
     .                                  'REF_FRAME', 
     .                                  'START_TIME',
     .                                  'STOP_TIME',
     .                                  'TIME_SYSTEM',
     .                                  'VARIABLES_NUMBER',
     .                                  'CREATION_DATE'      /
     
      DATA                  FIRST    /  .TRUE.           /



      CALL CHKIN ( 'PRSMET' )

      IF ( FIRST ) THEN
C
C        Initialize the sets of metadata items seen so far in
C        the current metadata block.  The set ITEMS contains all
C        keywords seen; the set RITEMS contains the required 
C        keywords seen.
C
         CALL SSIZEC ( NKEYS,  ITEMS  )
         CALL SSIZEC ( NREQMX, RITEMS )

         FIRST = .FALSE.

      END IF

C
C     Read non-comment lines until we hit the end
C     of file or the end of the metadata block.
C
      CALL GETNCL ( INPUT, LINE, LNUM, TYPE, EOF )

      DO WHILE ( ( .NOT. EOF ) .AND. ( TYPE .EQ. 'METADATA_LINE' ) )
C
C        We expect the current line to conform to the "keyword = value"
C        syntax.
C
         CALL KEYVLC ( INPUT, LINE, LNUM, KEYWRD, VALUE )
         
         IF ( FAILED() ) THEN

            CALL CHKOUT ( 'PRSMET' )
            RETURN

         END IF

C
C        Map both keyword and value to their standardized forms.
C
         CALL CMAP  ( KEYWRD, KEYWRD )
         CALL LJUST ( KEYWRD, KEYWRD )
         CALL UCASE ( KEYWRD, KEYWRD )

         CALL CMAP  ( VALUE,  LOCVAL )
         CALL LJUST ( LOCVAL, LOCVAL )
         CALL UCASE ( LOCVAL, LOCVAL )

C
C        Replace with blanks any underscores found in the value string.
C
         CALL REPLCH ( LOCVAL, '_', ' ', LOCVAL )

C
C        Save the value, line, and line number associated with 
C        the current keyword, if the keyword is recognized.  
C
         LOC = ISRCHC ( KEYWRD, NKEYS, KEYS )

         IF ( LOC .GT. 0 ) THEN
C
C           Store the RHS value.
C
            VSTRS(LOC) = LOCVAL

C
C           Mark the current keyword as seen.           
C
            CALL INSRTC ( KEYWRD, ITEMS )

C
C           If the keyword is required, add it to the set
C           of required keywords seen.
C
            LOC2 = ISRCHC ( KEYWRD, NREQO, ROKEYS )

            IF (  LOC2 .GT. 0 ) THEN

               CALL INSRTC ( KEYWRD, RITEMS )

            END IF

            LOC2 = ISRCHC ( KEYWRD, NREQA, RAKEYS )

            IF (  LOC2 .GT. 0 ) THEN

               CALL INSRTC ( KEYWRD, RITEMS )

            END IF

C
C           Save the source line and line number for this keyword.
C           These will be needed if a semantic error is detected
C           later.
C
            KLINES(LOC) = LINE
            KNUMS(LOC)  = LNUM
    
         END IF

         CALL GETNCL ( INPUT, LINE, LNUM, TYPE, EOF )

      END DO


C
C     Make sure we recognize the file type.  Set the output argument
C     to a standard form.
C
      IF (      ( VSTRS(FTPIDX) .NE. 'ORBIT FILE'    ) 
     .    .AND. ( VSTRS(FTPIDX) .NE. 'ATTITUDE FILE' )  ) THEN

         CALL SETMSG ( 'Unrecognized file type <*> seen '           //
     .                 'in the metadata block ending at line #* in '//
     .                 'file <*>. '                                 )
         CALL ERRCH  ( '*', VSTRS(FTPIDX)                           )
         CALL ERRINT ( '*', LNUM                                    )
         CALL ERRCH  ( '*', INPUT                                   ) 
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                        )
         CALL CHKOUT ( 'PRSMET'                                     )
         RETURN

      END IF


C
C     Set the file type.
C
      FILTYP = VSTRS(FTPIDX)

C
C     Perform keyword processing specific to orbit files.
C
      IF ( FILTYP .EQ. 'ORBIT FILE' ) THEN
C
C        Is the set of required items full?
C
         IF ( CARDC(RITEMS) .LT. NREQO ) THEN

            CALL SETMSG ( 'Not all required metadata items were seen '//
     .                    'in the metadata block ending at line #*  ' //
     .                    'in file <*>.  Required items are:  '       //
     .                    'CENTER_NAME,DERIVATIVES_FLAG, FILE_TYPE, ' //
     .                    'OBJECT_NAME, REF_FRAME, START_TIME, '      //
     .                    'STOP_TIME, TIME_SYSTEM, VARIABLES_NUMBER.'  )
            CALL ERRINT ( '*', LNUM                                    )
            CALL ERRCH  ( '*', INPUT                                   )
            CALL SIGERR ( 'SPICE(MISSINGITEM)'                         )
            CALL CHKOUT ( 'PRSMET'                                     )
            RETURN

         END IF

C
C        Make sure whole state vectors are present:  we can't
C        make an SPK file from position-only data.
C
         IF ( VSTRS(NVRIDX) .NE. '6' ) THEN

            CALL SETMSG ( 'Unsupported "VARIABLES_NUMBER <*> seen '//
     .                    'in the metadata block ending at line #*'//
     .                    ' in file <*>. '                         )
            CALL ERRCH  ( '*', VSTRS(NVRIDX)                       )
            CALL ERRINT ( '*', LNUM                                )
            CALL ERRCH  ( '*', INPUT                               ) 
            CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                    )
            CALL CHKOUT ( 'PRSMET'                                 )
            RETURN

         END IF


         IF ( VSTRS(DERIDX) .EQ. '1' ) THEN
C
C           This is a Hermite style file.
C
            SUBTYP = S18TP0

         ELSE IF ( VSTRS(DERIDX) .EQ. '0' ) THEN
C
C           This is a Lagrange style file.
C
            SUBTYP = S18TP1
             
         ELSE

            CALL SETMSG ( 'Unsupported derivative flag <*> seen '  //
     .                    'in the metadata block ending at line #*'//
     .                    ' in file <*>. '                         )
            CALL ERRCH  ( '*', VSTRS(DERIDX)                       )
            CALL ERRINT ( '*', LNUM                                )
            CALL ERRCH  ( '*', INPUT                               ) 
            CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                    )
            CALL CHKOUT ( 'PRSMET'                                 )
            RETURN

         END IF
         
C
C        Map center name to NAIF ID code.
C
         CALL BODN2C ( VSTRS(CTRIDX), CENTER, FOUND )

         IF ( .NOT. FOUND ) THEN

            CALL SETMSG ( 'Center * specified in metadata block '     //
     .                    'at line #* in file <*> could not be '      //
     .                    'mapped to a NAIF ID code.  Line was *'      )
            CALL ERRCH  ( '*',  VSTRS(CTRIDX)                          )
            CALL ERRINT ( '*',  KNUMS(OBJIDX)                          )
            CALL ERRCH  ( '*',  INPUT                                  )
            CALL ERRCH  ( '*',  KLINES(CTRIDX)                         )
            CALL SIGERR ( 'SPICE(IDCODENOTFOUND)'                      )
            CALL CHKOUT ( 'PRSMET'                                     )
            RETURN

         END IF 

C
C        Set the creation date output argument.
C
         CRTTIM = VSTRS ( CTIDX )


      ELSE
C
C        We have an attitude file.
C
C        Is the set of required items full?
C
         IF ( CARDC(RITEMS) .LT. NREQA ) THEN

            CALL SETMSG ( 'Not all required metadata items were seen '//
     .                    'in the metadata block ending at line #*  ' //
     .                    'in file <*>.  Required items are:  '       //
     .                    'DERIVATIVES_FLAG, FILE_TYPE, '             //
     .                    'OBJECT_NAME, REF_FRAME, START_TIME, '      //
     .                    'STOP_TIME, TIME_SYSTEM, VARIABLES_NUMBER.'  )
            CALL ERRINT ( '*', LNUM                                    )
            CALL ERRCH  ( '*', INPUT                                   )
            CALL SIGERR ( 'SPICE(MISSINGITEM)'                         )
            CALL CHKOUT ( 'PRSMET'                                     )
            RETURN

         END IF

C
C        Make sure we have a valid variable count.
C
         IF (       ( VSTRS(NVRIDX) .NE. '4' ) 
     .        .AND. ( VSTRS(NVRIDX) .NE. '7' ) ) THEN

            CALL SETMSG ( 'Unsupported "VARIABLES_NUMBER <*> seen '//
     .                    'in the metadata block ending at line #*'//
     .                    ' in file <*>. '                         )
            CALL ERRCH  ( '*', VSTRS(NVRIDX)                       )
            CALL ERRINT ( '*', LNUM                                )
            CALL ERRCH  ( '*', INPUT                               ) 
            CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                    )
            CALL CHKOUT ( 'PRSMET'                                 )
            RETURN

         END IF

C
C        Set our angular velocity flag.
C
         HAVEAV =  VSTRS(NVRIDX) .EQ. '7'


         IF ( VSTRS(DERIDX) .EQ. '1' ) THEN
C
C           This is a Hermite style file.
C
            IF ( HAVEAV ) THEN
               SUBTYP = C05TP2
            ELSE
               SUBTYP = C05TP0
            END IF

         ELSE IF ( VSTRS(DERIDX) .EQ. '0' ) THEN
C
C           This is a Lagrange style file.
C
            IF ( HAVEAV ) THEN
               SUBTYP = C05TP3
            ELSE
               SUBTYP = C05TP1
            END IF
             
         ELSE

            CALL SETMSG ( 'Unsupported derivative flag <*> seen '  //
     .                    'in the metadata block ending at line #*'//
     .                    ' in file <*>. '                         )
            CALL ERRCH  ( '*', VSTRS(DERIDX)                       )
            CALL ERRINT ( '*', LNUM                                )
            CALL ERRCH  ( '*', INPUT                               ) 
            CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                    )
            CALL CHKOUT ( 'PRSMET'                                 )
            RETURN

         END IF

C
C        Set the creation date output argument. Not used
C        for attitude files. 
C        
         CRTTIM = ' '

      END IF


C
C     Below, we process file-type-independent keywords.
C
C     Map object name to NAIF ID code.
C
      CALL BODN2C ( VSTRS(OBJIDX), OBJECT, FOUND )

      IF ( .NOT. FOUND ) THEN

         CALL SETMSG ( 'Object * specified in metadata block '     //
     .                 'at line #* in file <*> could not be '      //
     .                 'mapped to a NAIF ID code.  Line was *'      )
         CALL ERRCH  ( '*',  VSTRS(OBJIDX)                          ) 
         CALL ERRINT ( '*',  KNUMS(OBJIDX)                          )
         CALL ERRCH  ( '*',  INPUT                                  ) 
         CALL ERRCH  ( '*',  KLINES(OBJIDX)                         ) 
         CALL SIGERR ( 'SPICE(IDCODENOTFOUND)'                      )
         CALL CHKOUT ( 'PRSMET'                                     )
         RETURN

      END IF

C
C     Convert time bounds to ET.   If "usable" bounds have been 
C     supplied, use these bounds.  Otherwise use the default
C     start and stop times.
C     
      IF (        ELEMC( KEYS(UBIDX), ITEMS )
     .     .AND.  ELEMC( KEYS(UEIDX), ITEMS )  ) THEN

         BIDX = UBIDX
         EIDX = UEIDX

      ELSE

         BIDX = TBIDX
         EIDX = TEIDX

      END IF


      TIMSYS  =  VSTRS(SYSIDX)

      CALL TIMCVT ( INPUT,   KLINES(BIDX),  KNUMS(BIDX), 
     .              TIMSYS,  VSTRS(BIDX),   TBEG          )


      CALL TIMCVT ( INPUT,   KLINES(EIDX),  KNUMS(EIDX), 
     .              TIMSYS,  VSTRS(EIDX),   TEND          )

C
C     Set the frame output argument.  Check the frame.
C
      FRAME = VSTRS(REFIDX) 

      CALL NAMFRM ( FRAME, REFCOD )
 
      IF ( REFCOD .EQ. 0 ) THEN
 
         CALL SETMSG ( 'The reference frame # is not supported.'   )
         CALL ERRCH  ( '#', FRAME                                  )
         CALL SIGERR ( 'SPICE(INVALIDREFFRAME)'                    )
         CALL CHKOUT ( 'PRSMET'                                    )
         RETURN
 
      END IF

C
C     All of the outputs are set.
C
      CALL CHKOUT ( 'PRSMET' )
      RETURN
      END
