C$Procedure MKSKSG ( Make SPK segments )
 
      SUBROUTINE MKSKSG ( INPUT,   HANDLE,  SUBTYP,  
     .                    DEGREE,  OBJECT,  CENTER,  FRAME, 
     .                    KBEG,    KEND,    TIMSYS         )
 
C$ Abstract
C
C     Write out one or more SPK segments representing the ephemeris
C     defined by a metadata block and the contiguous set of states
C     following the block.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading 
C
C     FRAMES
C     NAIF_IDS
C     SPK
C     TIME
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'ck05.inc'
      INCLUDE 'spk18.inc'

      CHARACTER*(*)         INPUT
      INTEGER               HANDLE
      INTEGER               SUBTYP
      INTEGER               DEGREE
      INTEGER               OBJECT
      INTEGER               CENTER
      CHARACTER*(*)         FRAME
      DOUBLE PRECISION      KBEG
      DOUBLE PRECISION      KEND
      CHARACTER*(*)         TIMSYS
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     INPUT      I   EPM file to be read.
C     HANDLE     I   Handle of SPK file.
C     SUBTYP     I   SPK subtype.
C     DEGREE     I   Interpolation degree.
C     OBJECT     I   Ephemeris object or s/c component.
C     CENTER     I   Center of motion.
C     FRAME      I   Name of reference frame.
C     KBEG       I   Coverage start time of file.
C     KEND       I   Coverage stop time of file.
C     TIMSYS     I   Time system.
C
C$ Detailed_Input
C
C     INPUT          is the name of an EPM file to be read.
C
C     HANDLE         is the handle of the SPK file to write.  The
C                    file may be new or may be open for appending.
C
C     SUBTYP         is the subtype of the SPK data type corresponding
C                    to the input data representation. See the include
C                    file spk18.inc for SPK type 18 subtype codes.
C
C     DEGREE         is the polynomial degree associated with the
C                    interpolation method.
C
C     OBJECT         is the NAIF integer ID of the ephemeris object or
C                    spacecraft component for which segments are to be
C                    written.
C
C     CENTER         is the center of motion of the ephemeris object.
C                    CENTER is applicable only when an orbit file is
C                    being processed.
C
C     FRAME          is the name of the reference frame relative to
C                    which the output states are given.
C
C     KBEG           is the start time of the output file's coverage.
C                    By default KBEG is the start time of the input
C                    file, but the user may request a later start time.
C                    KBEG is expressed as seconds past J2000, TDB.
C
C     KEND           is the end time of the output file's coverage.
C                    By default KEND is the end time of the input
C                    file, but the user may request an earlier end time.
C                    KEND is expressed as seconds past J2000, TDB.
C
C     TIMSYS         is the name of the time system associated
C                    with time tags in the current metadata block
C                    or the states associated with this block.
C                     
C$ Detailed_Output
C
C     None.  See Particulars for a description of the action taken
C     by this routine.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If an unrecognized SPK subtype is specified, the 
C        error SPICE(NOTSUPPORTED) will be signaled.
C
C     2) If an unsupported SPK data type is specified, the error
C        SPICE(NOTSUPPORTED) will be signaled.
C
C     3) If an error occurs while writing an SPK segment, the problem
C        will be diagnosed by routines called by this routine.
C
C     4) If an error occurs while parsing packet data from the input
C        EPM file, the problem will be diagnosed by routines called
C        by this routine.
C
C     5) If TIMSYS does not specify a supported time system, the 
C        error will be diagnosed by routines called by this routine.
C
C     6) If the input 
C 
C$ Files
C
C     See the descriptions of the arguments INPUT and OUTPUT.
C
C$ Particulars
C
C     None.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    MEX2KER Version 3.1.0, 26-JUN-2012 (NJB)
C
C        Removed argument RATE from argument list.
C        Made various header format updates.
C
C-    MEX2KER Version 3.0.0, 20-FEB-2004 (NJB)
C     
C        Added support for user-supplied SPK time bounds.
C
C-    MEX2KER Version 2.0.0, 12-FEB-2003 (NJB)
C
C
C-    MEX2KER Version 1.0.0, 07-JUN-2002 (NJB)
C
C-&


C
C     SPICELIB functions
C     
      INTEGER               LSTLED
      INTEGER               LSTLTD
      
C
C     Local parameters
C
      CHARACTER*(*)         ORBTYP
      PARAMETER           ( ORBTYP = 'ORBIT FILE' )


      INTEGER               MAXPK
      PARAMETER           ( MAXPK  = 10000 )

      INTEGER               SIDLEN
      PARAMETER           ( SIDLEN = 40 )

      INTEGER               MAXPS
      PARAMETER           ( MAXPS  = C05PS2 )
C
C     Local variables
C
      CHARACTER*(SIDLEN)    SEGID

      DOUBLE PRECISION      ET
      DOUBLE PRECISION      FIRST
      DOUBLE PRECISION      LAST
      DOUBLE PRECISION      PACKET ( MAXPS )
      DOUBLE PRECISION      PKBUFF ( MAXPS *  MAXPK )
      DOUBLE PRECISION      TBUFF  (          MAXPK )

      INTEGER               B
      INTEGER               E
      INTEGER               FROM
      INTEGER               I
      INTEGER               LIDX
      INTEGER               N
      INTEGER               NSEG
      INTEGER               NWR
      INTEGER               PACKSZ
      INTEGER               PAD
      INTEGER               PKB
      INTEGER               SHIFT
      INTEGER               STIDX
      INTEGER               TO
      INTEGER               WINSIZ

      LOGICAL               FOUND


      CALL CHKIN ( 'MKSKSG' )

C
C     Use the first SIDLEN characters of the input file name as the
C     segment ID.
C
      SEGID = INPUT(:SIDLEN)

C
C     Set the window and packet sizes.
C
      IF (  SUBTYP .EQ. S18TP0 ) THEN

         WINSIZ = ( DEGREE + 1 ) / 2
         PACKSZ = S18PS0

      ELSE IF ( SUBTYP .EQ. S18TP1  ) THEN

         WINSIZ = DEGREE + 1
         PACKSZ = S18PS1

      ELSE

         CALL SETMSG ( 'SPK type 18 SUBTYP <#> is not supported.' )
         CALL ERRINT ( '#', SUBTYP                                )
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                      )
         CALL CHKOUT ( 'MKSKSG'                                   )
         RETURN

      END IF

C
C     If the window size is even, the pad is one less than half the 
C     window size.  If the window size is odd, the pad is half of
C     one less than the window size.  In either case, the pad
C     can be computed using the same integer arithetic expression:
C           
      PAD = ( WINSIZ - 1 ) / 2

C
C     Initialize the index of the first packet in the segment's coverage
C     interval.  If the segment has padding at the beginning, this
C     index will be equal to PAD + 1.
C
      STIDX = 1

C
C     Initialize the packet and segment counts.
C      
      N     = 0
      NSEG  = 0

      CALL PRSDR ( INPUT, ORBTYP, SUBTYP, TIMSYS, ET, PACKET, FOUND )

      DO WHILE ( FOUND ) 

         N  =  N + 1

         TO = (N-1)*PACKSZ + 1

         CALL MOVED ( PACKET, PACKSZ, PKBUFF(TO) )

         TBUFF(N) = ET


         IF ( N .EQ. MAXPK ) THEN
C
C           It's time to emit a segment.  We'll use the last PAD
C           packets as padding, so the last packet in the coverage
C           interval is at index N-PAD.
C
            LIDX  = N - PAD

            FIRST = MAX ( KBEG, TBUFF(STIDX) )
            LAST  = MIN ( KEND, TBUFF(LIDX)  )

C
C           Presuming we have non-empty coverage, write out 
C           packets 1 : MAXPK.
C
            IF ( FIRST .LT. LAST ) THEN

C              Determine the range of packets that actually need to be
C              written. B and E are the indices of these packets. The
C              range allows for padding on each side of the segment
C              bounds.  We constrain B and E to lie in the range 1:N.
C
               B   = MAX ( 1,   LSTLED ( FIRST, N, TBUFF ) - PAD      )
               E   = MIN ( N,   LSTLTD ( LAST,  N, TBUFF ) + PAD + 1  )

C  
C              NWR is the number of packets to write.
C
               NWR = E - B + 1

C
C              PKB is the packet buffer start index.
C
               PKB = (B-1)*PACKSZ + 1

C
C              Write out packets B : E.
C
               CALL SPKW18 ( HANDLE,  SUBTYP,       OBJECT,  
     .                       CENTER,  FRAME,        FIRST,  
     .                       LAST,    SEGID,        DEGREE,     
     .                       NWR,     PKBUFF(PKB),  TBUFF(B) )
C
C              Shift the last (2*PAD+1) packets into the beginning 
C              of the buffer.  Shift the corresponding epochs as well.
C           
C              Increment the count of segments written.
C
               NSEG = NSEG + 1

            END IF
C
C           The following actions are necessary whether or not we
C           actually had to write a segment.

            SHIFT = 2 * PAD  +  1

            DO I = 1, SHIFT

               FROM     =    ( N - SHIFT - 1 + I )*PACKSZ  +  1
               TO       =    ( I - 1             )*PACKSZ  +  1

               CALL MOVED( PKBUFF(FROM), PACKSZ, PKBUFF(TO) )

               TBUFF(I) = TBUFF(N-SHIFT+I)

            END DO

            N = SHIFT

C
C           Set the segment coverage start index.
C
            STIDX = PAD + 1

         END IF

C
C        Get the next packet and epoch.
C         
         CALL PRSDR ( INPUT, ORBTYP, SUBTYP, TIMSYS, ET, PACKET, FOUND )

      END DO

C
C     If we've already written any segments, the last packet of
C     the coverage interval of the last segment will always be 
C     buffered.  There is no need to create a new segment if 
C     that's the only packet in the buffer.
C
      IF (  ( NSEG .GT. 0 )  .AND. ( N .LE. 1 )  ) THEN
         CALL CHKOUT ( 'MKSKSG' )
         RETURN
      END IF

C
C     If we haven't written any segments, return if there are no
C     buffered packets.
C
      IF (  ( NSEG .EQ. 0 )  .AND. ( N .EQ. 0 )  ) THEN
         CALL CHKOUT ( 'MKSKSG' )
         RETURN
      END IF

C
C     At this point, we're going to prepare to write a segment.
C
C     Set the segment coverage bounds.
C
      FIRST = MAX ( KBEG, TBUFF(STIDX) )
      LAST  = MIN ( KEND, TBUFF(N)     )

C
C     Write a segment of the requested type.
C
      IF ( FIRST .LE. LAST ) THEN
C
C        Determine the range of packets that actually need to be
C        written. B and E are the indices of these packets. The range
C        allows for padding on each side of the segment bounds.  We
C        constrain B and E to lie in the range 1:N.
C
         B = MAX ( 1,   LSTLED ( FIRST, N, TBUFF ) - PAD      )
         E = MIN ( N,   LSTLTD ( LAST,  N, TBUFF ) + PAD + 1  )

C  
C        NWR is the number of records to write.
C
         NWR = E - B + 1

C
C        PKB is the packet buffer start index.
C
         PKB = (B-1)*PACKSZ + 1

         CALL SPKW18 (  HANDLE,  SUBTYP,  OBJECT,       CENTER,  
     .                  FRAME,   FIRST,   LAST,         SEGID,      
     .                  DEGREE,  NWR,     PKBUFF(PKB),  TBUFF(B)  )

      END IF

C
C     PRSDR already pushed back the line we read that terminated 
C     our segment.
C

      CALL CHKOUT ( 'MKSKSG' )
      RETURN 
      END


