C$Procedure CMAPI ( Character string map, inverse )
 
      SUBROUTINE CMAPI ( IN, OUT )
 
C$ Abstract
C
C     Invert the mapping performed by CMAP.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     OEM2SPK
C
C$ Declarations
 
      IMPLICIT NONE
 
      CHARACTER*(*)         IN
      CHARACTER*(*)         OUT
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     IN         I   A string in the range of OEM2SPK's string map.
C     OUT        O   Inverse image of IN under the string map.
C
C$ Detailed_Input
C
C     IN             is a string in the range of the mapping defined in
C                    the kernel pool by the kernel variable
C                    STRING_MAPPING.
C
C                    If IN is an even-indexed value in the array
C                    associated with STRING_MAPPING, the preceding
C                    value in the array is the value to which IN is
C                    mapped.
C                     
C$ Detailed_Output
C
C     OUT            is the inverse image of IN under the mapping
C                    defined in the kernel pool by the kernel variable
C                    STRING_MAPPING.  If IN is an even-indexed value in
C                    the array associated with the kernel variable
C                    STRING_MAPPING, OUT is the preceding value in the
C                    array.
C
C                    If the kernel variable STRING_MAPPING does not 
C                    exist in the kernel pool, or if IN is not an 
C                    even-indexed value in the array associated with 
C                    STRING_MAPPING, OUT is set equal to IN.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If the number of values associated with the kernel variable
C        STRING_MAPPING is odd, the error SPICE(INVALIDSIZE) is 
C        signaled.
C
C     2) If no map exists, or if IN is not in the range of the
C        mapping, OUT is set equal to IN.  This case is not an error.
C
C$ Files
C
C     Normally, the mapping implemented by this routine is defined
C     by a kernel variable introduced into the kernel pool by loading
C     a SPICE text kernel.  
C
C$ Particulars
C
C     This routine provides the inverse of the mapping implemented
C     by the OEM2SPK routine CMAP.
C
C$ Examples
C
C     1)  Find the string mapped to the string J2000 by CMAP:
C
C            If the OEM2SPK setup file contains the assignment
C
C               STRING_MAPPING = ( ...
C                                  'EME 2000', 'J2000'
C                                  ...                 )
C
C            The code fragment
C  
C               IN = 'J2000'
C               CALL CMAPI ( IN, OUT )
C
C            will set OUT to 
C
C               'EME 2000'
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-1,
C          September, 2004.
C     
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    OEM2SPK Version 1.2.0, 29-AUG-2010 (NJB)
C
C        Bug fix: MAP array subscript order in LJUST and UCASE 
C        calls has been reversed.
C
C-    OEM2SPK Version 1.1.0, 19-SEP-2006 (NJB)
C
C        Bug fix:  FOUND flag check has been added after GCPOOL call
C        in initialization IF block.
C
C-    OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&


      
C
C     SPICELIB functions
C
      INTEGER               ISRCHC

      LOGICAL               ODD

C
C     Local parameters
C     
      CHARACTER*(*)         MAPNAM
      PARAMETER           ( MAPNAM = 'STRING_MAPPING' )

      INTEGER               MAXMAP
      PARAMETER           ( MAXMAP = 100 )

      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 80 )

C
C     Local variables
C
      CHARACTER*(LNSIZE)    LOCIN
      CHARACTER*(LNSIZE)    MAP  ( MAXMAP, 2      )
      CHARACTER*(LNSIZE)    XMAP ( 2,      MAXMAP )

      INTEGER               I
      INTEGER               J
      INTEGER               LOC
      INTEGER               MAPSIZ
      INTEGER               N

      LOGICAL               FIRST
      LOGICAL               FOUND

C
C     Saved variables
C
      SAVE                  FIRST
      SAVE                  FOUND
      SAVE                  MAP
      SAVE                  MAPSIZ

C
C     Initial values
C
      DATA                  FIRST / .TRUE. /



      CALL CHKIN ( 'CMAPI' )

C
C     By default, the output is equal to the input.
C
      OUT = IN

      IF ( FIRST ) THEN
C
C        Get the transposed character map from the kernel pool.
C        
         CALL GCPOOL ( MAPNAM, 1, 2*MAXMAP, N, XMAP, FOUND )

         IF ( .NOT. FOUND ) THEN
C
C           No map is available.  This is not an error per se, 
C           though an error may occur later if a map was actually
C           needed.
C
            CALL CHKOUT ( 'CMAPI' )
            RETURN

         END IF


         IF ( ODD(N) ) THEN

            CALL SETMSG ( 'Array associated with kernel variable ' //
     .                    '# has size #.  Size must be even.'       )
            CALL ERRCH  ( '#',  MAPNAM                              )
            CALL ERRINT ( '#',  N                                   )
            CALL SIGERR ( 'SPICE(INVALIDSIZE)'                      )
            CALL CHKOUT ( 'CMAPI'                                   )
            RETURN

         END IF

         MAPSIZ = N / 2

C
C        Transpose the map so that the target values are in the
C        second column.
C
         DO I = 1, MAPSIZ
            
            DO J = 1, 2
               MAP(I,J) = XMAP(J,I)
            END DO

         END DO

C
C        Convert all strings to upper case, and left-justify them.
C
         DO I = 1, MAPSIZ
            
            DO J = 1, 2
               
               CALL LJUST ( MAP(I,J), MAP(I,J) )
               CALL UCASE ( MAP(I,J), MAP(I,J) )

            END DO

         END DO

         FIRST = .FALSE.

      END IF

C
C     Attempt a translation only if the character mapping was present.
C     Otherwise, just return.
C
      IF ( .NOT. FOUND ) THEN

         CALL CHKOUT ( 'CMAPI' )
         RETURN

      END IF


      CALL LJUST ( IN,    LOCIN )
      CALL UCASE ( LOCIN, LOCIN )

      LOC = ISRCHC ( LOCIN, MAPSIZ, MAP(1,2) )

      IF ( LOC .GT. 0 ) THEN

         OUT = MAP(LOC,1)

      END IF

      CALL CHKOUT ( 'CMAPI' )
      RETURN
      END

