C$Procedure CHKFNM ( Check file names )

      SUBROUTINE CHKFNM ( SETUP, INPUT, OUTPUT, APPEND )
  
C$ Abstract
C
C     Check names of MEX2KER setup, input, and output files
C     specified as command-line arguments to MEX2KER.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     MEX2KER
C
C$ Declarations
 
      IMPLICIT NONE
 
      CHARACTER*(*)         SETUP
      CHARACTER*(*)         INPUT
      CHARACTER*(*)         OUTPUT
      LOGICAL               APPEND
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     SETUP      I   Name of MEX2KER setup file.
C     INPUT      I   Name of input MEX file.
C     OUTPUT     I   Name of output SPK or CK file.
C     APPEND     I   Append flag.
C
C$ Detailed_Input
C
C     SETUP          is the name of an MEX2KER setup file.
C
C     INPUT          is the name of an MEX orbit or attitude file 
C                    to be converted to SPK format.  
C
C                    Input may be blank.
C
C     OUTPUT         is the name of an SPK or CK file to be written.
C                    OUTPUT may be a new or existing file.  Output
C                    SPK files will contain type 18 segments.  Output
C                    CK files will contain type 5 segments.
C
C                    Output may be blank.
C
C     APPEND         is a logical flag indicating whether to 
C                    append data to an existing output file.
C                    If APPEND is .TRUE., the file designated 
C                    by OUTPUT must exist.  If APPEND is .FALSE.,
C                    the file designated by OUTPUT must not exist.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If the setup file name is blank, the error 
C        SPICE(INVALIDFILENAME) is signaled.
C
C     2) If the setup file is not found, the error SPICE(FILENOTFOUND)
C        is signaled.
C
C     3) If the input MEX file is specified but is not found, the error 
C        SPICE(FILENOTFOUND) is signaled.
C
C     4) If the APPEND is .FALSE. and the output file already
C        exists, the error SPICE(FILEEXISTS) is signaled.
C
C$ Files
C
C     See the descriptions of SETUP, INPUT and OUTPUT above.
C
C$ Particulars
C
C     This routine centralizes file name error checks performed by
C     the utility MEX2KER.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    MEX2KER Version 1.0.1, 21-JUN-2012 (NJB)
C
C        Updated header format.
C
C-    MEX2KER Version 1.0.0, 16-FEB-2003 (NJB)
C
C-&


C
C     SPICELIB functions
C
      LOGICAL               EXISTS


      CALL CHKIN ( 'CHKFNM' )
C
C     Check the command line arguments.  The setup file must exist.
C     The input file must exist if specified.  The output file must
C     not exist if specified.
C
      IF ( SETUP .EQ. ' ' ) THEN

         CALL SETMSG ( 'Setup file name may not be blank.' )
         CALL SIGERR ( 'SPICE(INVALIDFILENAME)'            )
         CALL CHKOUT ( 'CHKFNM'                            )
         RETURN

      END IF


      IF ( .NOT. EXISTS(SETUP) ) THEN

         CALL SETMSG ( 'Setup file <#> was not found.' )
         CALL ERRCH  ( '#',  SETUP                     )
         CALL SIGERR ( 'SPICE(FILENOTFOUND)'           )
         CALL CHKOUT ( 'CHKFNM'                        )
         RETURN

      END IF


      IF ( INPUT .NE. ' ' ) THEN

         IF ( .NOT. EXISTS(INPUT) ) THEN

            CALL SETMSG ( 'Input MEX file <#> was not found.' )
            CALL ERRCH  ( '#',  INPUT                         )
            CALL SIGERR ( 'SPICE(FILENOTFOUND)'               )
            CALL CHKOUT ( 'CHKFNM'                            )
            RETURN

         END IF

      END IF

      IF ( OUTPUT .NE. ' ' ) THEN

         IF ( .NOT. APPEND ) THEN

            IF ( EXISTS(OUTPUT)  ) THEN

               CALL SETMSG ( 'Output file <#> already exists.  File ' //
     .                       'must not exist unless appending '       //
     .                       'is requested.'                           )
               CALL ERRCH  ( '#', OUTPUT                               )
               CALL SIGERR ( 'SPICE(FILEEXISTS)'                       )
               CALL CHKOUT ( 'CHKFNM'                                  )
               RETURN

            END IF

         END IF

      END IF


      CALL CHKOUT ( 'CHKFNM' )
      RETURN
      END




