C$Procedure      SC01S2D ( Type 1 SCLK String to Decimal Form )

      SUBROUTINE SC01S2D ( SC, SCLKCH, SCLKDP )

C$ Abstract
C
C     Convert a type 1 SCLK string to decimal form.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     CHARACTER
C     CONVERSION
C     PARSING
C
C$ Declarations

      IMPLICIT              NONE

      INCLUDE               'sclk.inc'

      INTEGER               SC
      CHARACTER*(*)         SCLKCH
      DOUBLE PRECISION      SCLKDP

C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     SC         I   NAIF integer code for a spacecraft.
C     SCLKCH     I   An SCLK string.
C     SCLKDP     O   A decimal form of SCLK string.
C
C$ Detailed_Input
C
C     SC             is NAIF integer code for a spacecraft.
C
C     SCLKCH         is an input Type 1 SCLK string, with or without
C                    partition, unabridged (all fields) or abridged
C                    (any number of fields on the right may be
C                    missing), with fields separated by any of the
C                    allowed Type 1 delimiters (see sclk.inc).
C
C                    Any Type 1 SCLK string generated by SPICELIB
C                    SCLK routines will be successfully processed by
C                    routine.
C
C                    Type 1 SCLK string produced by other means
C                    will be processed as well if they contain
C                    at least one SCLK field, do not start with a
C                    delimiter, and do not have more than one
C                    delimiter in a row (this restriction extends
C                    to multiple spaces).
C
C
C$ Detailed_Output
C
C     SCLKDP         is a DP number providing a decimal form the input
C                    SCLK with integer part equal to the left-most
C                    field of the clock and the fractional decimal part
C                    computed from the rest of the clock fields as
C                    described in Particulars.
C
C$ Parameters
C
C     See include file.
C
C$ Exceptions
C
C     1) If clock type associated with SC is not 1,
C        'SPICE(NOTSUPPORTED)' is signaled.
C
C     2) If input SCLK string is blank, 'SPICE(BLANKSCLKSTRING)' is
C        signaled.
C
C     3) If number of SCLK fields provided in SCLK01_N_FIELDS is
C        not equal to count of values provided in SCLK01_OFFSETS,
C        'SPICE(BADSCLKDATA1)' is signaled.
C
C     4) If number of SCLK fields provided in SCLK01_N_FIELDS is
C        not equal to count of values provided in SCLK01_MODULI,
C        'SPICE(BADSCLKDATA2)' is signaled.
C
C     5) If any of the MODULI is zero, 'SPICE(BADSCLKDATA3)' is
C        signaled.
C
C     6) If partition separator in the last character,
C        'SPICE(INVALIDSCLKSTRING1)' is signaled.
C
C     7) If delimiter appears at the beginning of the string, of
C        immediately following partition separator or another
C        delimiter, 'SPICE(OUTOFPLACEDELIMITER)' is signaled.
C
C     8) If a field does not represent an integer,
C        'SPICE(NONINTEGERFIELD)' is signaled.
C
C     9) If SCLK string contains more fields that specified in
C        SCLK01_N_FIELDS, 'SPICE(TOOMANYFIELDS)' is signaled.
C
C     10) If no fields were extracted from the string,
C        'SPICE(INVALIDSCLKSTRING2)' is signaled.
C
C     11) Routines called by this routine signal errors if they don't
C        fetch required SCLK parameters.
C
C$ Files
C
C     SCLK kernel for the clock of interest must be loaded before
C     calling this routine. This routine fetches SCLK01_N_FIELDS,
C     SCLK01_OFFSETS, and SCLK01_MODULI for this clock from the POOL.
C
C$ Particulars
C
C     A Type 1 SCLK string is converted decimal form as follows:
C
C        -  partition number, if present, is dropped
C
C        -  the string is split into fields
C
C        -  left-most field is used as the integer part of the output
C           DP number
C
C        -  the fractional decimal part is computed using all clock
C           fields to the right of the left-most field by starting from
C           the right-most field and 1) applying offset to the field
C           value, 2) dividing adjusted value by moduli, 3) adding the
C           resultant fractional part to unadjusted value of the next
C           field to the left until all fields up to the left-most
C           field are processed.
C
C$ Examples
C
C     For an SCLK with the following parameters:
C
C        SCLK01_N_FIELDS_84            = (                 2 )
C        SCLK01_MODULI_84              = ( 10000000000   256 )
C        SCLK01_OFFSETS_84             = (           0     0 )
C
C     this table illustrates the conversions this routine will perform:
C
C            SCLKCH               SCLKDP
C        ----------------     ----------------
C        '0.0'                        0.000000
C        '0.1'                        0.003906
C        '0.128'                      0.500000
C        '0.256'                      1.000000
C        '883612839'          883612839.000000
C        '883612839.0'        883612839.000000
C        '883612839 0'        883612839.000000
C        '883612839-128'      883612839.500000
C        '1/883612839-128'    883612839.500000
C        '1/883612839-255'    883612839.996094
C
C        '1/883612839--255'   error: double delimiter
C        '1/.255'             error: leading delimiter
C        '.255'               error: leading delimiter
C        '0.255.0'            error: too many fields
C
C$ Restrictions
C
C     This routine works only for Type 1 SCLKs.
C
C     SCLK kernel for the clock of interest must be loaded before
C     calling this routine.
C
C     Input SCLK field must not start with a delimiter and must not
C     have more than one delimiter of the same kind (including spaces)
C     or of different kinds (e.g. ':' followed by space) in a row.
C
C     Number of items in OFFSETS and MODULI fetched from the POOL
C     must be equal to the value of N_FIELDS.
C
C     No value of MODULI can be zero.
C
C$ Literature_References
C
C     None.
C
C$ Author_and_Institution
C
C     B.V. Semenov    (JPL)
C
C$ Version
C
C-    FRMDIFF Version 1.0.0, 12-SEP-2008 (BVS)
C
C        Initial version. All parts related to managing SCLK parameters
C        were shamelessly stolen from SC01.
C
C-&

C$ Index_Entries
C
C     Type 1 string SCLK to decimal form
C
C-&

C
C     Local parameters.
C
      INTEGER               NAMLEN
      PARAMETER           ( NAMLEN = 32 )

      INTEGER               NKITEM
      PARAMETER           ( NKITEM = 3 )

      INTEGER               NFLIDX
      PARAMETER           ( NFLIDX =          1 )

      INTEGER               OFFIDX
      PARAMETER           ( OFFIDX = NFLIDX + 1 )

      INTEGER               MODIDX
      PARAMETER           ( MODIDX = OFFIDX + 1 )

C
C     SPICELIB functions
C
      LOGICAL               RETURN
      LOGICAL               FAILED

      INTEGER               SCTYPE
      INTEGER               CPOS
      INTEGER               RTRIM

C
C     Variables
C
      CHARACTER*(NAMLEN)    NAMLST ( NKITEM )
      CHARACTER*(NAMLEN)    KVNAME ( NKITEM )
      CHARACTER*(NAMLEN*10) ERROR

      DOUBLE PRECISION      FVALUE ( MXNFLD )
      DOUBLE PRECISION      OFFSET ( MXNFLD )
      DOUBLE PRECISION      MODULI ( MXNFLD )
      DOUBLE PRECISION      FRACTN

      INTEGER               OLDSC
      INTEGER               I
      INTEGER               N
      INTEGER               NOFF
      INTEGER               NMOD
      INTEGER               NFIELD
      INTEGER               START
      INTEGER               LAST
      INTEGER               PTR
      INTEGER               FINDEX
      INTEGER               NEXT

      LOGICAL               FIRST
      LOGICAL               UPDATE

C
C     Save everything since we will be using many saved values.
C
      SAVE

C
C     Initial values.
C
      DATA NAMLST /
     .              'SCLK01_N_FIELDS',
     .              'SCLK01_OFFSETS',
     .              'SCLK01_MODULI'
     .            /

C
C     Standard SPICE error handling.
C
      IF ( RETURN () ) THEN
         RETURN
      ELSE
         CALL CHKIN ( 'SC01S2D' )
      END IF

C
C     Report an error if this is not a type 1 SCLK.
C
      IF (  SCTYPE ( SC )  .NE.  1  ) THEN

         CALL SETMSG ( 'The clock type associated with the '       //
     .                 'ID # was #. The SCLK string ''#'' '        //
     .                 'cannot be converted to decimal '           //
     .                 'form because such conversion '             //
     .                 'is supported only for type 1 clocks.'      )
         CALL ERRINT ( '#', SC                                     )
         CALL ERRINT ( '#', SCTYPE ( SC )                          )
         CALL ERRCH  ( '#', SCLKCH                                 )
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                       )
         CALL CHKOUT ( 'SC01S2D'                                   )
         RETURN

      END IF

C
C     Report an error if input string is blank.
C
      IF ( SCLKCH .EQ. ' ' ) THEN
         CALL SETMSG ( 'Input SCLK string is blank.'               )
         CALL SIGERR ( 'SPICE(BLANKSCLKSTRING)'                    )
         CALL CHKOUT ( 'SC01S2D'                                   )
         RETURN
      END IF

C
C     On the first pass through the subroutine, or if the spacecraft
C     clock ID changes, we will set watches on the SCLK kernel
C     variables for the current clock.
C
      IF (  FIRST  .OR.  ( SC .NE. OLDSC )  ) THEN

         FIRST = .FALSE.

C
C        Make up a list of names of kernel variables that we'll use.
C
         CALL MOVEC  ( NAMLST,  NKITEM,  KVNAME )

         DO I = 1, NKITEM
            CALL SUFFIX ( '_#',        0,         KVNAME(I) )
            CALL REPMI  ( KVNAME(I),  '#',  -SC,  KVNAME(I) )
         END DO

C
C        Set a watch on all of the kernel variables we use.
C
         CALL SWPOOL ( 'SC01S2D',  NKITEM,  KVNAME )

C
C        Keep track of the last spacecraft clock ID encountered.
C
         OLDSC =  SC

      END IF

C
C     Find out whether we need to look up new format descriptors from
C     the kernel pool. If any relevant kernel variables were updated,
C     we have to do a look-up. Note that changing the s/c clock ID
C     causes a new watch to be set, so a look-up is required.
C
      CALL CVPOOL ( 'SC01S2D',  UPDATE )

      IF ( UPDATE ) THEN
C
C        Our first piece of business is to look up all of the data we
C        require from the kernel pool. We must form the names of the
C        items we want using the input S/C ID code.  The items we need
C        are:
C
C           -  The number of fields in an (unabridged) SCLK string
C           -  The moduli of the fields of an SCLK string
C           -  The offsets for each clock field.
C

         CALL SCLI01 ( NAMLST(NFLIDX), SC, 1,        N,      NFIELD )
         CALL SCLD01 ( NAMLST(OFFIDX), SC, MXNFLD,   NOFF,   OFFSET )
         CALL SCLD01 ( NAMLST(MODIDX), SC, MXNFLD,   NMOD,   MODULI )

C
C        Don't try to continue if we had a lookup error.
C
         IF ( FAILED() ) THEN
            CALL CHKOUT( 'SC01S2D' )
            RETURN
         END IF

C
C        Verify that SCLK items that we got are consistent and that
C        none of the moduli is zero.
C
         IF ( NFIELD .NE. NOFF ) THEN
            CALL SETMSG ( 'Invalid SCLK parameters for ID #. '     //
     .                    'The number of fields (#) provided by '  //
     .                    'keyword ''#'' does not match the count '//
     .                    'of field offsets (#) provided in the '  //
     .                    'keyword ''#''.'                         )
            CALL ERRINT ( '#', SC                                  )
            CALL ERRINT ( '#', NFIELD                              )
            CALL ERRCH  ( '#', KVNAME(NFLIDX)                      )
            CALL ERRINT ( '#', NOFF                                )
            CALL ERRCH  ( '#', KVNAME(OFFIDX)                      )
            CALL SIGERR ( 'SPICE(BADSCLKDATA1)'                    )
            CALL CHKOUT ( 'SC01S2D'                                )
            RETURN
         END IF

         IF ( NFIELD .NE. NMOD ) THEN
            CALL SETMSG ( 'Invalid SCLK parameters for ID #. '     //
     .                    'The number of fields (#) provided by '  //
     .                    'keyword ''#'' does not match the count '//
     .                    'of field moduli (#) provided in the '   //
     .                    'keyword ''#''.'                         )
            CALL ERRINT ( '#', SC                                  )
            CALL ERRINT ( '#', NFIELD                              )
            CALL ERRCH  ( '#', KVNAME(NFLIDX)                      )
            CALL ERRINT ( '#', NMOD                                )
            CALL ERRCH  ( '#', KVNAME(MODIDX)                      )
            CALL SIGERR ( 'SPICE(BADSCLKDATA2)'                    )
            CALL CHKOUT ( 'SC01S2D'                                )
            RETURN
         END IF

         DO I = 1, NMOD
            IF ( MODULI(I) .EQ. 0.D0 ) THEN
               CALL SETMSG ( 'Invalid SCLK parameters for ID #. '  //
     .                       'Modulus for the field number # '     //
     .                       'provided in the keyword ''#'' is '   //
     .                       'zero.'                               )
               CALL ERRINT ( '#', SC                               )
               CALL ERRINT ( '#', I                                )
               CALL ERRCH  ( '#', KVNAME(MODIDX)                   )
               CALL SIGERR ( 'SPICE(BADSCLKDATA3)'                 )
               CALL CHKOUT ( 'SC01S2D'                             )
            RETURN

            END IF
         END DO

      END IF

C
C     Parse input SCLK string. Start either at the beginning or
C     immediately after partition separator (if there is one).
C
      FINDEX = 0

      LAST  = RTRIM(SCLKCH)

      START = MAX( 1, CPOS( SCLKCH, '/', 1 ) + 1 )

      IF ( START .GT. LAST ) THEN
         CALL SETMSG ( 'SCLK string ''#'' is invalid. It has '     //
     .                 'partition separator ''/'' as the last '    //
     .                 'character.'                                )
         CALL ERRCH  ( '#', SCLKCH                                 )
         CALL SIGERR ( 'SPICE(INVALIDSCLKSTRING1)'                 )
         CALL CHKOUT ( 'SC01S2D'                                   )
         RETURN
      END IF

      DO WHILE ( START .LE. LAST  )

C
C        If the starting character is a delimiter, signal an error
C        as we will not accept delimiter following '/', delimiters
C        at the beginning of the SCLK without partition, and
C        multiple delimiters in a row any place in the string.
C
         IF ( CPOS( DELIMS, SCLKCH(START:START), 1 ) .NE. 0 ) THEN
            CALL SETMSG ( 'Delimiter ''#'' at position # in SCLK ' //
     .                    'string ''#'' is out of place.'          )
            CALL ERRCH  ( '#', SCLKCH(START:START)                 )
            CALL ERRINT ( '#', START                               )
            CALL ERRCH  ( '#', SCLKCH                              )
            CALL SIGERR ( 'SPICE(OUTOFPLACEDELIMITER)'             )
            CALL CHKOUT ( 'SC01S2D'                                )
            RETURN
         END IF

C
C        Locate next delimiter.
C
         NEXT = CPOS( SCLKCH, DELIMS, START )
         IF ( NEXT .EQ. 0 .OR. NEXT .GT. LAST ) THEN
            NEXT = LAST
         ELSE
            NEXT = NEXT - 1
         END IF

C
C        Parse field as integer.
C
         CALL NPARSI ( SCLKCH(START:NEXT), N, ERROR, PTR )
         IF ( PTR .NE. 0 ) THEN
            CALL SETMSG ( 'Field ''#'' in SCLK '                   //
     .                    'string ''#'' is not an integer number.' )
            CALL ERRCH  ( '#', SCLKCH(START:NEXT)                  )
            CALL ERRCH  ( '#', SCLKCH                              )
            CALL SIGERR ( 'SPICE(NONINTEGERFIELD)'                 )
            CALL CHKOUT ( 'SC01S2D'                                )
            RETURN
         END IF

C
C        Buffer the field value if we aren't over the maximum number
C        fields.
C
         IF ( FINDEX + 1 .LE. NFIELD ) THEN

            FINDEX = FINDEX + 1
            FVALUE(FINDEX) = N

         ELSE
            CALL SETMSG ( 'SCLK string ''#'' has more fields '     //
     .                    'than the number of fields (#) defined ' //
     .                    'by loaded SCLK parameters.'             )
            CALL ERRCH  ( '#', SCLKCH                              )
            CALL ERRINT ( '#', NFIELD                              )
            CALL SIGERR ( 'SPICE(TOOMANYFIELDS)'                   )
            CALL CHKOUT ( 'SC01S2D'                                )
            RETURN
         END IF

C
C        Reset start to move to the next field.
C
         START = NEXT + 2

      END DO

C
C     Report an error if we have not buffered any field values.
C
      IF ( FINDEX .EQ. 0 ) THEN
         CALL SETMSG ( 'SCLK string ''#'' does not contain any '   //
     .                 'fields.'                                   )
         CALL ERRCH  ( '#', SCLKCH                                 )
         CALL SIGERR ( 'SPICE(INVALIDSCLKSTRING2)'                 )
         CALL CHKOUT ( 'SC01S2D'                                   )
         RETURN
      END IF

C
C     Convert fields to the right of the first field to decimal
C     fraction.
C
      FRACTN = 0.D0

      DO I = FINDEX, 2, -1
         FRACTN = ( FVALUE(I) - OFFSET(I) + FRACTN ) / MODULI(I)
      END DO

C
C     Add fraction to the first field value to get output decimal SCLK.
C
      SCLKDP = FVALUE(1) + FRACTN

C
C     That's all folks.
C
      CALL CHKOUT ( 'SC01S2D' )

      RETURN

      END
