C$Procedure      CHWCML ( Extract arguments from FRMDIFF command line )

      SUBROUTINE CHWCML ( LINE, KERNAM, FFRNAM, FFRID,
     .                    TFRNAM, TFRID, AVFLG, AVFFLG,
     .                    CMPWIN, NITR, STEP, DIFTYP, TIMFMT,
     .                    KERNLS, SCLKID, AXES, AUNITS, SIGDIG )


C$ Abstract
C
C     Extract arguments from FRMDIFF command line and return them
C     and/or default values via individual variables. If input command
C     line is incomplete or if specific command line keys requesting
C     help are specified, this routine displays usage and stops the
C     program.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     FRMDIFF User's Guide.
C
C$ Keywords
C
C     TBD.
C
C$ Declarations

      IMPLICIT NONE

      INCLUDE              'frmdiff.inc'

      CHARACTER*(*)         LINE
      CHARACTER*(*)         KERNAM ( 2 )
      CHARACTER*(*)         FFRNAM ( 2 )
      INTEGER               FFRID  ( 2 )
      CHARACTER*(*)         TFRNAM ( 2 )
      INTEGER               TFRID  ( 2 )
      LOGICAL               AVFLG
      LOGICAL               AVFFLG
      DOUBLE PRECISION      CMPWIN ( LBCELL : * )
      INTEGER               NITR
      DOUBLE PRECISION      STEP
      CHARACTER*(*)         DIFTYP
      CHARACTER*(*)         TIMFMT
      CHARACTER*(*)         KERNLS ( 3 )
      INTEGER               SCLKID ( 2 )
      INTEGER               AXES   ( 3 )
      CHARACTER*(*)         AUNITS
      INTEGER               SIGDIG

C$ Brief_I/O
C
C     Variable  I/O  Description
C     --------  ---  --------------------------------------------------
C     LINE       I   FRMDIFF command line
C     KERNAM     O   1st and 2nd kernel file names
C     FFRNAM     O   1st and 2nd ``from'' frame names
C     FFRID      O   1st and 2nd ``from'' frame IDs
C     TFRNAM     O   1st and 2nd ``to'' frame names
C     TFRID      O   1st and 2nd ``to'' frame IDs
C     AVFLG      O   Angular velocity comparison flag
C     AVFFLG     O   Angular velocity frame flag
C     CMPWIN     O   Time window for comparison
C     NITR       O   Number of points to be used in comparison.
C     STEP       O   Time step in seconds.
C     DIFTYP     O   Type of summary to be generated by FRMDIFF.
C     TIMFMT     O   Output time format string "dump" summaries.
C     KERNLS     O   Lists of additional kernels.
C     SCLKID     O   IDs for ET to SCLK conversions.
C     AXES       O   Rotation axes for output Euler angles.
C     AUNITS     O   Units for output Euler angles.
C     SIGDIG     O   Number of significant digits
C
C$ Detailed_Input
C
C     LINE        is the command line provided to FRMDIFF. See FRMDIFF
C                 User's Guide for the command line syntax and detailed
C                 specification of allowed arguments.
C
C$ Detailed_Output
C
C     KERNAM      is a two element array containing the names of the
C                 first and second kernel files.
C
C     FFRNAM      is a two element array containing the names of the
C                 first and second ``from'' frames.
C
C     FFRID       is a two element array containing the ID of the first
C                 and second ``from'' frames.
C
C     TFRNAM      is a two element array containing the names of the
C                 first and second ``to'' frames.
C
C     TFRID       is a two element array containing the IDs of the first
C                 and second ``to'' frames.
C
C     AVFLG       is a flag indicating whether angular velocities
C                 should be compared: .FALSE. for no, .TRUE. for yes.
C
C     AVFFLG      is a flag indicating whether angular velocities
C                 should be computed with respect to ``from'' frame
C                 (.FALSE.) or ``to'' frame (.TRUE.).
C
C     CMPWIN      is a time window for comparison.
C
C     NITR        is the number of points to be used in comparison.
C
C     STEP        is the time step in seconds. STEP is returned as zero
C                 if it was not provided on the command line.
C
C     DIFTYP      is a string indicating the type of summary to be
C                 generated by FRMDIFF.
C
C     TIMFMT      is a string containing output time format picture for
C                 "dump"-type summaries.
C
C     KERNLS      is an array of three strings containing the lists of
C                 additional kernels provided on the command line.
C                 KERNLS(1) contains the list applicable to the first
C                 kernel to be examined. KERNLS(2) contains the list
C                 applicable to the second kernel to be examined.
C                 KERNLS(3) contains the list applicable both kernels
C                 to be examined.
C
C     SCLKID      is a two element array containing ID to be used in
C                 SCLK routine calls for converting ET to SCLKs for
C                 output. Both elements are returned as zero if SCLK
C                 output was not requested using corresponding command
C                 line switch. If SCLK output was requested, only one
C                 of the elements will be set to SCLK ID, depending on
C                 which of the ``to''/coverage frames were CK frames.
C                 Which element is set will indicate which kernels
C                 should be loaded for SCLK conversions: if SCLK(1) is
C                 not zero, kernels applicable to first attitude set,
C                 if SCLK(2) is non zero, kernels applicable to second
C                 attitude set.
C
C     AXES        is a three element array specifying rotation axes for
C                 output Euler angles.
C
C     AUNITS      is a string specifying units for output Euler angles.
C
C     SIGDIG      is the number of significant digits in the output
C                 numbers in scientific format included in the dump
C                 reports.
C
C$ Parameters
C
C     See include file.
C
C$ Exceptions
C
C     TBD.
C
C$ Files
C
C     TBD.
C
C$ Particulars
C
C     TBD.
C
C$ Examples
C
C     TBD.
C
C$ Restrictions
C
C     TBD.
C
C$ Literature_References
C
C     TBD.
C
C$ Author_and_Institution
C
C     B.V. Semenov   (JPL)
C
C$ Version
C
C-    Version 2.1.0, 29-SEP-2016 (BVS).
C
C        Added initialization to blank of the first kernel name and
C        type for cases when the sole command line argument is a file
C        name. This change prevents rather bogus looking errors being
C        signaled later by LDKLST.
C
C-    Version 2.0.0, 29-MAR-2012 (BVS).
C
C        Changed calling sequence to include additional SIGDIG output.
C
C        Changed fetch the number of significant digits specified with
C        DIGKEY.
C
C        Updated to fetch coverage from non-primary kernels when no
C        primary kernels were specified.
C
C-    Version 1.1.0, 28-OCT-2011 (BVS).
C
C        Moved PARCML to support. Updated its calling sequence.
C
C-    Version 1.0.0, 18-APR-2006 (BVS).
C
C-&

C
C     SPICELIB functions
C
      DOUBLE PRECISION      DPMAX
      DOUBLE PRECISION      DPMIN

      INTEGER               ISRCHC
      INTEGER               RTRIM
      INTEGER               WDCNT
      INTEGER               WNCARD

      LOGICAL               ELEMI
      LOGICAL               EQSTR
      LOGICAL               EXISTS

C
C     Local variables.
C
      CHARACTER*(3)         ARCH   ( 2 )
      CHARACTER*(3)         TYPE   ( 2 )
      CHARACTER*(7)         ARCTYP ( 2 )
      CHARACTER*(7)         HARTYP
      CHARACTER*(DSPSIZ)    HLPMSG ( HLPCNT )
      CHARACTER*(DSPSIZ)    USGMSG ( USGCNT )
      CHARACTER*(DSPSIZ)    VERMSG ( VERCNT )
      CHARACTER*(LINSIZ)    CLVALS ( MAXKEY )
      CHARACTER*(LINSIZ)    ERROR
      CHARACTER*(LINSIZ)    HLINE
      CHARACTER*(LINSIZ)    UNPRSD
      CHARACTER*(LINSIZ)    HLINE2
      CHARACTER*(5*LINSIZ)  HLLINE
      CHARACTER*(LINSIZ)    HKRNAM
      CHARACTER*(WRDSIZ)    CLKEYS ( MAXKEY )
      CHARACTER*(WRDSIZ)    CLKEYU ( MAXKEY )
      CHARACTER*(WRDSIZ)    HWORD
      CHARACTER*(WRDSIZ)    CFRNAM ( 2 )
      CHARACTER*(LINSIZ)    TIME   ( 2 )
      CHARACTER*(LINSIZ)    COVDSC ( 2 )
      CHARACTER*(LINSIZ)    TIMDSC

      DOUBLE PRECISION      COVER1 ( LBCELL : WINSIZ )
      DOUBLE PRECISION      COVER2 ( LBCELL : WINSIZ )
      DOUBLE PRECISION      COVERA ( LBCELL : WINSIZ )
      DOUBLE PRECISION      COVERB ( LBCELL : WINSIZ )
      DOUBLE PRECISION      COVERC ( LBCELL : 2 )
      DOUBLE PRECISION      DC     ( ND )
      DOUBLE PRECISION      DESCR  ( ND + NICK/2 )
      DOUBLE PRECISION      HDP1
      DOUBLE PRECISION      HDP2
      DOUBLE PRECISION      ET     ( 2 )


      INTEGER               FLIDS1 ( LBCELL : MAXIDS )
      INTEGER               FLIDS2 ( LBCELL : MAXIDS )
      INTEGER               HANDLE
      INTEGER               I
      INTEGER               IC     ( NICK )
      INTEGER               NARGS
      INTEGER               PTR
      INTEGER               FRCODE
      INTEGER               TFRCID ( 2 )
      INTEGER               TFRCLS ( 2 )
      INTEGER               CFRCID ( 2 )
      INTEGER               CFRCLS ( 2 )
      INTEGER               CFRID  ( 2 )
      INTEGER               CLSSID
      INTEGER               HINT
      INTEGER               COUNT

      LOGICAL               CLFLAG ( MAXKEY )
      LOGICAL               FOUND
      LOGICAL               CFRFND ( 2 )
      LOGICAL               ISCPCK ( 2 )

C
C     Save everything to prevent potential memory problems in f2c'ed
C     version.
C
      SAVE

C
C     Initialize command line keys.
C
      DATA ( CLKEYS(I), I =  1, MAXKEY )
     .     /
     .      KERKEY,
     .      TF1KEY,
     .      FF1KEY,
     .      CF1KEY,
     .      KE1KEY,
     .      TF2KEY,
     .      FF2KEY,
     .      CF2KEY,
     .      KE2KEY,
     .      AVKEY,
     .      AVFKEY,
     .      BEGKEY,
     .      ENDKEY,
     .      NSTKEY,
     .      STPKEY,
     .      FMTKEY,
     .      TYPKEY,
     .      ANOKEY,
     .      ANUKEY,
     .      DIGKEY,
     .      USGKEY,
     .      UKEY,
     .      HLPKEY,
     .      HKEY,
     .      VKEY
     .     /

C
C     Check in.
C
      CALL CHKIN ( 'CHWCML' )

C
C     Generate uppercase version of command lines keys.
C
      DO I = 1, MAXKEY
         CALL UCASE( CLKEYS(I), CLKEYU(I) )
      END DO

C
C     Initialize version display.
C
      CALL TKVRSN( 'TOOLKIT', HWORD )
      VERMSG(  1 ) = ' '
      VERMSG(  2 ) = PGMNAM // ' -- Version ' // PGMVER //
     .               ' -- Toolkit Version ' // HWORD(:RTRIM(HWORD))
      VERMSG(  3 ) = ' '

C
C     Initialize help display.
C

      HLPMSG(  1 ) = '   # provides means for sampling orient'
     .//             'ation of a single reference'
      HLPMSG(  2 ) = '   frame, or for comparing orientations'
     .//             ' of two reference frames known to'
      HLPMSG(  3 ) = '   SPICE and supported by data from SPI'
     .//             'CE kernels.'
      HLPMSG(  4 ) = ' '
      HLPMSG(  5 ) = '   To sample orientation of a single fr'
     .//             'ame the program computes a set of'
      HLPMSG(  6 ) = '   transformations from one frame (the '
     .//             '``from'''' frame) to another frame'
      HLPMSG(  7 ) = '   (the ``to'''' frame) over an interva'
     .//             'l of time with fixed or variable time'
      HLPMSG(  8 ) = '   step, using a given set of kernel fi'
     .//             'les. Then, depending on the'
      HLPMSG(  9 ) = '   requested type of output report, it '
     .//             'prints to the screen a table'
      HLPMSG( 10 ) = '   containing these transformations exp'
     .//             'ressed as total rotation angles and'
      HLPMSG( 11 ) = '   axes, quaternions, matrices or Euler'
     .//             ' angles and angular velocities, or'
      HLPMSG( 12 ) = '   only magnitude of the maximum rotati'
     .//             'on and angular velocity, or results'
      HLPMSG( 13 ) = '   of a simple statistical analysis of '
     .//             'rotations.'
      HLPMSG( 14 ) = ' '
      HLPMSG( 15 ) = '   To compare orientations of two frame'
     .//             's the program computes a set of'
      HLPMSG( 16 ) = '   transformations from one frame (the '
     .//             '``from'''' frame) to another frame'
      HLPMSG( 17 ) = '   (the ``to'''' frame) over an interva'
     .//             'l of time with a fixed or variable'
      HLPMSG( 18 ) = '   time step using one set of kernels, '
     .//             'computes a set of transformations'
      HLPMSG( 19 ) = '   for the same or a different ``from'''
     .//             '''-``to'''' frame combination at the'
      HLPMSG( 20 ) = '   same times using the same or a diffe'
     .//             'rent set of kernels, and then'
      HLPMSG( 21 ) = '   computes the difference in rotation '
     .//             'and angular velocity between the'
      HLPMSG( 22 ) = '   corresponding transformations. Depen'
     .//             'ding on the requested type of'
      HLPMSG( 23 ) = '   output report the program prints to '
     .//             'the screen only maximum differences'
      HLPMSG( 24 ) = '   in rotation and angular velocity, or'
     .//             ' a complete table of rotation and'
      HLPMSG( 25 ) = '   angular velocity differences express'
     .//             'ed as total rotation angles and'
      HLPMSG( 26 ) = '   axes, matrices, quaternions, or Eule'
     .//             'r angles, or results of a simple'
      HLPMSG( 27 ) = '   statistical analysis of the rotation'
     .//             ' differences.'
      HLPMSG( 28 ) = ' '


      CALL REPMC ( HLPMSG(  1 ), '#', PGMNAM, HLPMSG(  1 ) )

C
C     Initialize usage display.
C
      USGMSG(  1 ) = '   # provides a simple way of sam'
     .//             'pling orientation of a frame or'
      USGMSG(  2 ) = '   comparing orientations of two frames'
     .//             ' known to SPICE and supported by'
      USGMSG(  3 ) = '   data from SPICE kernels. The program'
     .//             ' usage is:'
      USGMSG(  4 ) = ' '
      USGMSG(  5 ) = '      % # [options] <first kernel'
     .//             ' name> <second kernel name>'
      USGMSG(  6 ) = '      % # [options] <kernel name>'
      USGMSG(  7 ) = '      % # [options]'
      USGMSG(  8 ) = ' '
      USGMSG(  9 ) = '   where kernel can be a CK, an FK, or '
     .//             'a PCK. Options are shown below.'
      USGMSG( 10 ) = '   Order and case of keys are not signi'
     .//             'ficant. Values must be'
      USGMSG( 11 ) = '   space-separated from keys, i.e. ''-n'
     .//             ' 10'', not ''-n10''.'
      USGMSG( 12 ) = ' '
      USGMSG( 13 ) = '      #  <supporting kernel(s) name(s)>'
      USGMSG( 14 ) = '      # <first ``from'''' frame, name or ID>'
      USGMSG( 15 ) = '      # <first ``to'''' frame,'
     .//             ' name or ID>'
      USGMSG( 16 ) = '      # <first frame for coverage look '
     .//             'up, name or ID>'
      USGMSG( 17 ) = '      # <additional supporting kernel(s'
     .//             ') for first file>'
      USGMSG( 18 ) = '      # <second ``from'''' frame, name or ID>'
      USGMSG( 19 ) = '      # <second ``to'''' frame'
     .//             ', name or ID>'
      USGMSG( 20 ) = '      # <second frame for coverage look'
     .//             ' up, name or ID>'
      USGMSG( 21 ) = '      # <additional supporting kernel(s'
     .//             ') for second file>'
      USGMSG( 22 ) = '      #  <compare angular velocities: #'
     .//             '|# (default: #)>'
      USGMSG( 23 ) = '      #  <frame for angular velocities:'
     .//             ' #|# (default: #)>'
      USGMSG( 24 ) = '      #  <interval start time>'
      USGMSG( 25 ) = '      #  <interval stop time>'
      USGMSG( 26 ) = '      #  <number of points: # to #'
     .//             ' (default: #)>'
      USGMSG( 27 ) = '      #  <time step in seconds>'
      USGMSG( 28 ) = '      #  <time format: #|#|#|#|picture_'
     .//             'for_TIMOUT (default: #)>'
      USGMSG( 29 ) = '      #  <report: #|#|#|#|#|#|#|#|#>'
      USGMSG( 30 ) = '      #  <rotation axes order (default:'
     .//             ' # # #)>'
      USGMSG( 31 ) = '      #  <units for output angles> '
     .//             '(only for # # and # #)'
      USGMSG( 32 ) = '      #  <number of significant digits: '
     .//             '# to # (default: #)>'
      USGMSG( 33 ) = ' '

      CALL REPMC ( USGMSG(  1 ), '#', PGMNAM, USGMSG(  1 ) )
      CALL REPMC ( USGMSG(  5 ), '#', PGMNAM, USGMSG(  5 ) )
      CALL REPMC ( USGMSG(  6 ), '#', PGMNAM, USGMSG(  6 ) )
      CALL REPMC ( USGMSG(  7 ), '#', PGMNAM, USGMSG(  7 ) )

      CALL REPMC ( USGMSG( 13 ), '#', KERKEY, USGMSG( 13 ) )
      CALL REPMC ( USGMSG( 14 ), '#', FF1KEY, USGMSG( 14 ) )
      CALL REPMC ( USGMSG( 15 ), '#', TF1KEY, USGMSG( 15 ) )
      CALL REPMC ( USGMSG( 16 ), '#', CF1KEY, USGMSG( 16 ) )
      CALL REPMC ( USGMSG( 17 ), '#', KE1KEY, USGMSG( 17 ) )
      CALL REPMC ( USGMSG( 18 ), '#', FF2KEY, USGMSG( 18 ) )
      CALL REPMC ( USGMSG( 19 ), '#', TF2KEY, USGMSG( 19 ) )
      CALL REPMC ( USGMSG( 20 ), '#', CF2KEY, USGMSG( 20 ) )
      CALL REPMC ( USGMSG( 21 ), '#', KE2KEY, USGMSG( 21 ) )
      CALL REPMC ( USGMSG( 22 ), '#', AVKEY,  USGMSG( 22 ) )
      CALL REPMC ( USGMSG( 23 ), '#', AVFKEY, USGMSG( 23 ) )
      CALL REPMC ( USGMSG( 24 ), '#', BEGKEY, USGMSG( 24 ) )
      CALL REPMC ( USGMSG( 25 ), '#', ENDKEY, USGMSG( 25 ) )
      CALL REPMC ( USGMSG( 26 ), '#', NSTKEY, USGMSG( 26 ) )
      CALL REPMI ( USGMSG( 26 ), '#', MINITR, USGMSG( 26 ) )
      CALL REPMI ( USGMSG( 26 ), '#', MAXITR, USGMSG( 26 ) )
      CALL REPMI ( USGMSG( 26 ), '#', DEFITR, USGMSG( 26 ) )
      CALL REPMC ( USGMSG( 27 ), '#', STPKEY, USGMSG( 27 ) )
      CALL REPMC ( USGMSG( 28 ), '#', FMTKEY, USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 29 ), '#', TYPKEY, USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 30 ), '#', ANOKEY, USGMSG( 30 ) )
      CALL REPMC ( USGMSG( 31 ), '#', ANUKEY, USGMSG( 31 ) )

      CALL REPMC ( USGMSG( 22 ), '#', YESVAL, USGMSG( 22 ) )
      CALL REPMC ( USGMSG( 22 ), '#', NOVAL,  USGMSG( 22 ) )
      CALL REPMC ( USGMSG( 22 ), '#', NOVAL,  USGMSG( 22 ) )
      CALL REPMC ( USGMSG( 23 ), '#', FRMVAL, USGMSG( 23 ) )
      CALL REPMC ( USGMSG( 23 ), '#', TOVAL,  USGMSG( 23 ) )
      CALL REPMC ( USGMSG( 23 ), '#', FRMVAL, USGMSG( 23 ) )
      CALL REPMC ( USGMSG( 28 ), '#', ETVAL,  USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', SCSVAL, USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', SCDVAL, USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', SCTVAL, USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', ETVAL,  USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 28 ), '#', ETVAL,  USGMSG( 28 ) )
      CALL REPMC ( USGMSG( 29 ), '#', BASVAL, USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 29 ), '#', STSVAL, USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 29 ), '#', DAAVAL, USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 29 ), '#', DMVAL,  USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 29 ), '#', DQSVAL, USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 29 ), '#', DQOVAL, USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 29 ), '#', DEAVAL, USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 29 ), '#', DCVAL,  USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 29 ), '#', DGVAL,  USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 29 ), '#', BASVAL, USGMSG( 29 ) )
      CALL REPMC ( USGMSG( 30 ), '#', ZVAL,   USGMSG( 30 ) )
      CALL REPMC ( USGMSG( 30 ), '#', YVAL,   USGMSG( 30 ) )
      CALL REPMC ( USGMSG( 30 ), '#', XVAL,   USGMSG( 30 ) )
      CALL REPMC ( USGMSG( 31 ), '#', TYPKEY, USGMSG( 31 ) )
      CALL REPMC ( USGMSG( 31 ), '#', DAAVAL, USGMSG( 31 ) )
      CALL REPMC ( USGMSG( 31 ), '#', TYPKEY, USGMSG( 31 ) )
      CALL REPMC ( USGMSG( 31 ), '#', DEAVAL, USGMSG( 31 ) )
      CALL REPMC ( USGMSG( 32 ), '#', DIGKEY, USGMSG( 32 ) )
      CALL REPMI ( USGMSG( 32 ), '#', MINSDG, USGMSG( 32 ) )
      CALL REPMI ( USGMSG( 32 ), '#', MAXSDG, USGMSG( 32 ) )
      CALL REPMI ( USGMSG( 32 ), '#', DEFSDG, USGMSG( 32 ) )

C
C     Get command line and do first attempt at parsing. All we need to
C     find out in this try is if one of the help/usage/version key
C     variations was present.
C
      HLINE = LINE
      CALL PARCML( HLINE, MAXKEY, CLKEYU, CLFLAG, CLVALS, FOUND,
     .             UNPRSD )

C
C     Was command line blank? Is one of the version, usage, or help
C     keys present? Display usage, help, or version and stop.
C
      NARGS = WDCNT( LINE )
      IF ( NARGS .EQ. 0 .OR.
     .     CLFLAG( ISRCHC( VKEY,   MAXKEY, CLKEYS ) ) .OR.
     .     CLFLAG( ISRCHC( HLPKEY, MAXKEY, CLKEYS ) ) .OR.
     .     CLFLAG( ISRCHC( HKEY,   MAXKEY, CLKEYS ) ) .OR.
     .     CLFLAG( ISRCHC( USGKEY, MAXKEY, CLKEYS ) ) .OR.
     .     CLFLAG( ISRCHC( UKEY,   MAXKEY, CLKEYS ) )       ) THEN

C
C        Display version in all cases.
C
         DO I = 1, VERCNT
            CALL TOSTDO( VERMSG(I) )
         END DO

C
C        Depending on what other information was requested, display
C        usage, help, or nothing and stop.
C
         IF ( NARGS .EQ. 0 .OR.
     .        CLFLAG( ISRCHC( USGKEY, MAXKEY, CLKEYS ) ) .OR.
     .        CLFLAG( ISRCHC( UKEY,   MAXKEY, CLKEYS ) )       ) THEN

            DO I = 1, USGCNT
               CALL TOSTDO ( USGMSG(I) )
            END DO

         ELSE IF (
     .        CLFLAG( ISRCHC( HLPKEY, MAXKEY, CLKEYS ) ) .OR.
     .        CLFLAG( ISRCHC( HKEY,   MAXKEY, CLKEYS ) )       ) THEN

            DO I = 1, HLPCNT
               CALL TOSTDO ( HLPMSG(I) )
            END DO

         END IF

         STOP

      END IF

C
C     Pull the last word from the command line and see if it's the name
C     of a file. If it is, check if it's one of the kernel files that
C     may be pulled in by the frames subsystem (Hint: any DAF or KPL
C     file can be while DAS and all others can't.)
C
      CALL NTHWD ( LINE, NARGS, KERNAM(2), PTR )

      IF ( EXISTS( KERNAM(2) ) ) THEN

         CALL GETFAT ( KERNAM(2), ARCH(2), TYPE(2) )

         IF ( ARCH(2) .EQ. '?' ) THEN
            CALL SETMSG ( 'File ''#'' specified as the last '        //
     .                    'argument on the command line is not '     //
     .                    'a SPICE kernel file.'                     )
            CALL ERRCH  ( '#', KERNAM(2)                             )
            CALL SIGERR ( 'SPICE(NOTAKERNELFILE2)'                   )
         ELSE IF ( .NOT. ( ARCH(2) .EQ. 'DAF' .OR.
     .                     ARCH(2) .EQ. 'KPL'      ) ) THEN
            CALL SETMSG ( 'File ''#'' specified as the last '        //
     .                    'argument on the command line is not '     //
     .                    'one of the kernel types that can '        //
     .                    'be given as the last or second to last '  //
     .                    'argument to this program. This file''s '  //
     .                    'architecture/type were ''#/#''.'          )
            CALL ERRCH  ( '#', KERNAM(2)                             )
            CALL ERRCH  ( '#', ARCH(2)                               )
            CALL ERRCH  ( '#', TYPE(2)                               )
            CALL SIGERR ( 'SPICE(NONAPPLICABLETYPE2)'                )
         END IF

         ARCTYP(2) = ARCH(2) // '/' // TYPE(2)

C
C        OK, it looks like the last item on the command line was a
C        file that we can process. Let's chop it off and check the
C        second to last item.
C
         LINE(PTR:) = ' '
         NARGS = WDCNT( LINE )
         IF ( NARGS .NE. 0 ) THEN

            CALL NTHWD ( LINE, NARGS, KERNAM(1), PTR )

            IF ( EXISTS( KERNAM(1) ) ) THEN

               CALL GETFAT ( KERNAM(1), ARCH(1), TYPE(1) )

               IF ( ARCH(1) .EQ. '?' ) THEN
                  CALL SETMSG ( 'File ''#'' specified as the second '//
     .                          'to last argument on the command '   //
     .                          'line is not a SPICE kernel file.'   )
                  CALL ERRCH  ( '#', KERNAM(1)                       )
                  CALL SIGERR ( 'SPICE(NOTAKERNELFILE1)'             )
               ELSE IF ( .NOT. ( ARCH(1) .EQ. 'DAF' .OR.
     .                           ARCH(1) .EQ. 'KPL'      ) ) THEN
                  CALL SETMSG ( 'File ''#'' specified as the second '//
     .                          'to last  argument on the command '  //
     .                          'line is not one of the kernel '     //
     .                          'types that can be given as the '    //
     .                          'last or second to last argument to '//
     .                          'this program. This file''s '        //
     .                          'architecture/type were ''#/#''.'    )
                  CALL ERRCH  ( '#', KERNAM(1)                       )
                  CALL ERRCH  ( '#', ARCH(1)                         )
                  CALL ERRCH  ( '#', TYPE(1)                         )
                  CALL SIGERR ( 'SPICE(NONAPPLICABLETYPE1)'          )
               END IF

               ARCTYP(1) = ARCH(1) // '/' // TYPE(1)

C
C              Aren't we lucky?! The second to last item on the command
C              line is also a kernel that we can process. Wipe it off
C              the tail of the line before moving on.
C
               LINE(PTR:) = ' '
               NARGS = WDCNT( LINE )

            ELSE

               KERNAM(1) = ' '
               ARCTYP(1) = ' '

            END IF

         ELSE

            KERNAM(1) = ' '
            ARCTYP(1) = ' '

         END IF

      ELSE

         KERNAM(1) = ' '
         ARCTYP(1) = ' '
         KERNAM(2) = ' '
         ARCTYP(2) = ' '

      END IF


C
C     If the files are CKs or PCKs, fetch IDs for later use.
C
      ISCPCK(1) = .FALSE.
      ISCPCK(2) = .FALSE.

      CALL SSIZEI ( MAXIDS, FLIDS1 )
      CALL SSIZEI ( MAXIDS, FLIDS2 )

      IF ( ARCTYP(1) .EQ. 'DAF/CK' ) THEN
         CALL CKOBJ  ( KERNAM(1), FLIDS1 )
         ISCPCK(1) = .TRUE.
      END IF

      IF ( ARCTYP(2) .EQ. 'DAF/CK' ) THEN
         CALL CKOBJ  ( KERNAM(2), FLIDS2 )
         ISCPCK(2) = .TRUE.
      END IF

      IF ( ARCTYP(1) .EQ. 'DAF/PCK' ) THEN
         CALL PCKFRM ( KERNAM(1), FLIDS1 )
         ISCPCK(1) = .TRUE.
      END IF

      IF ( ARCTYP(2) .EQ. 'DAF/PCK' ) THEN
         CALL PCKFRM ( KERNAM(2), FLIDS2 )
         ISCPCK(2) = .TRUE.
      END IF

C
C     Parse the command line again because one or two last items may
C     have been removed from it.
C
      HLINE = LINE
      CALL PARCML( HLINE, MAXKEY, CLKEYU, CLFLAG, CLVALS, FOUND,
     .             UNPRSD )

C
C     Go on processing the rest of the command line. All other
C     arguments are optional and, if not present, will have to be set
C     to some default values.
C

C
C     First, get additional kernels provided on the command line
C     applicable to both files and/or each specific file. We need to
C     deal with them first because some of them may define frames that
C     are needed to process the rest of command line arguments. Start
C     with looking for kernels applicable to the first attitude set.
C
      I = ISRCHC( KE1KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         IF ( CLVALS(I) .NE. ' ' ) THEN

            KERNLS(1) = CLVALS(I)

            HLINE = KERNLS(1)

            DO WHILE ( HLINE .NE. ' ' )

               CALL NEXTWD( HLINE, HLINE2, HLINE )

               IF ( .NOT. EXISTS( HLINE2 ) ) THEN
                  CALL SETMSG ( 'File ''#'' listed after ''#'' key ' //
     .                          'does not exist.'                    )
                  CALL ERRCH  ( '#', HLINE2                          )
                  CALL ERRCH  ( '#', KE1KEY                          )
                  CALL SIGERR ( 'SPICE(FILEDOESNTEXIST1)'            )
               END IF

            END DO

         ELSE
            CALL SETMSG ( 'Although ''#'' key was provided on the '  //
     .                    'command line no kernel file names were '  //
     .                    'following it.'                            )
            CALL ERRCH  ( '#', KE1KEY                                )
            CALL SIGERR ( 'SPICE(MISSINGFILENAMES1)'                 )
         END IF

      ELSE

         KERNLS(1) = ' '

      END IF

C
C     Second, look for kernels applicable to the second attitude set.
C
      I = ISRCHC( KE2KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         IF ( CLVALS(I) .NE. ' ' ) THEN

            KERNLS(2) = CLVALS(I)

            HLINE = KERNLS(2)

            DO WHILE ( HLINE .NE. ' ' )

               CALL NEXTWD( HLINE, HLINE2, HLINE )

               IF ( .NOT. EXISTS( HLINE2 ) ) THEN
                  CALL SETMSG ( 'File ''#'' listed after ''#'' key ' //
     .                          'does not exist.'                    )
                  CALL ERRCH  ( '#', HLINE2                          )
                  CALL ERRCH  ( '#', KE2KEY                          )
                  CALL SIGERR ( 'SPICE(FILEDOESNTEXIST2)'            )
               END IF

            END DO

         ELSE
            CALL SETMSG ( 'Although ''#'' key was provided on the '  //
     .                    'command line no kernel file names were '  //
     .                    'following it.'                            )
            CALL ERRCH  ( '#', KE2KEY                                )
            CALL SIGERR ( 'SPICE(MISSINGFILENAMES2)'                 )
         END IF

      ELSE

         KERNLS(2) = ' '

      END IF

C
C     Last, look for kernels applicable to both attitude sets.
C
      I = ISRCHC( KERKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         IF ( CLVALS(I) .NE. ' ' ) THEN

            KERNLS(3) = CLVALS(I)

            HLINE = KERNLS(3)

            DO WHILE ( HLINE .NE. ' ' )

               CALL NEXTWD( HLINE, HLINE2, HLINE )

               IF ( .NOT. EXISTS( HLINE2 ) ) THEN
                  CALL SETMSG ( 'File ''#'' listed after ''#'' key ' //
     .                          'does not exist.'                    )
                  CALL ERRCH  ( '#', HLINE2                          )
                  CALL ERRCH  ( '#', KERKEY                          )
                  CALL SIGERR ( 'SPICE(FILEDOESNTEXIST3)'            )
               END IF

            END DO

         ELSE
            CALL SETMSG ( 'Although ''#'' key was provided on the '  //
     .                    'command line no kernel file names were '  //
     .                    'following it.'                            )
            CALL ERRCH  ( '#', KERKEY                                )
            CALL SIGERR ( 'SPICE(MISSINGFILENAMES3)'                 )
         END IF

      ELSE

         KERNLS(3) = ' '

      END IF

C
C     Do a sanity check to verify that we have enough kernels to do
C     anything. We will attempt to do something only if at least one of
C     the kernels or kernel lists is non-blank.
C
C     This check is currently inactive to allow comparisons with
C     only built-in frames.
C
C      IF ( .NOT. ( KERNAM(1) .NE. ' ' .OR.
C     .             KERNAM(2) .NE. ' ' .OR.
C     .             KERNLS(1) .NE. ' ' .OR.
C     .             KERNLS(2) .NE. ' ' .OR.
C     .             KERNLS(3) .NE. ' '       )  ) THEN
C
C            CALL SETMSG ( 'No kernels were provided on the '         //
C     .                    'line.'                                    )
C            CALL SIGERR ( 'SPICE(NODATA)'                            )
C
C      END IF
C

C
C     If the first kernel is provided, load it. If not but the
C     second kernel is provided, load the second kernel (recall the
C     program can work on one kernel). Also load additional kernels
C     supplied for the first kernel and kernels applicable for both
C     kernels.
C
      IF      ( KERNAM(1) .NE. ' ' ) THEN
         HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//KERNAM(1)
         CALL LDKLST( HLLINE )
      ELSE IF ( KERNAM(2) .NE. ' ' ) THEN
         HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//KERNAM(2)
         CALL LDKLST( HLLINE )
      ELSE
         HLLINE = KERNLS(3)//' '//KERNLS(1)
         CALL LDKLST( HLLINE )
      END IF

C
C     After loading all kernels provided for computing the first
C     attitude set we can move on to getting command line inputs
C     defining it.
C
C     Start with getting the first ``from'' frame.
C
      FFRNAM(1) = ' '
      FFRID(1)  = 0

      I = ISRCHC( FF1KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

C
C        The first ``from'' frame was provided on the command line. Was
C        it given as a name or as an ID? Try to convert it to integer
C        as see if it worked.
C
         FFRNAM(1) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), FFRID(1), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

C
C           It's not an integer. Assume that it's the frame name and
C           try to convert it to ID. If we can't, report an error.
C
            CALL NAMFRM ( CLVALS(I), FFRID(1) )

            IF ( FFRID(1) .EQ. 0 ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', FF1KEY                             )
               CALL SIGERR ( 'SPICE(BADFROMFRAME1SP1)'               )
            END IF

         ELSE

C
C           It's an integer. Assume that it's the frame ID and try to
C           convert it to name. If we can't, report an error.
C
            CALL FRMNAM ( FFRID(1), FFRNAM(1) )

            IF ( FFRNAM(1) .EQ. ' ' ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', FF1KEY                             )
               CALL SIGERR ( 'SPICE(BADFROMFRAME1SP2)'               )
            END IF

         END IF

      END IF

C
C     Next, get the first ``to'' frame.
C
      TFRNAM(1) = ' '
      TFRID(1)  = 0
      TFRCID(1) = 0
      TFRCLS(1) = 0

      I = ISRCHC( TF1KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

C
C        The first ``to'' frame was provided on the command line. Was
C        it given as a name or as an ID? Try to convert it to integer
C        and see if it worked.
C
         TFRNAM(1) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), TFRID(1), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

C
C           It's not an integer. Assume that it's the frame name and
C           try to convert it to ID. If we can't, report an error.
C
            CALL NAMFRM ( CLVALS(I), TFRID(1) )

            IF ( TFRID(1) .EQ. 0 ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', TF1KEY                             )
               CALL SIGERR ( 'SPICE(BADTOFRAME1SPEC1)'               )
            END IF

         ELSE

C
C           It's an integer; treat it as a frame ID or a frame class ID
C           depending on context and try to map it to frame name.
C
C           If the first or only file is a CK and this ID is one of the
C           IDs present in it, treat it as a CK class ID and try to map
C           it to frame as such.
C
C           If the first or only file is a binary PCK and this ID is
C           one of the IDs present in it, treat it as a PCK class ID
C           and try to map it to frame as such.
C
C           If the first or only file is not a CK or a binary PCK, or
C           if it is BUT the ID could not be mapped to frame as a class
C           ID, try to map it to frame by treating it as a frame ID. If
C           it can't be mapped, report an error.
C
            IF      ( KERNAM(1) .NE. ' ' ) THEN
               HARTYP = ARCTYP(1)
               FOUND = ELEMI( TFRID(1), FLIDS1 )
            ELSE IF ( KERNAM(2) .NE. ' ' ) THEN
               HARTYP = ARCTYP(2)
               FOUND = ELEMI( TFRID(1), FLIDS2 )
            ELSE
               HARTYP = ' '
               FOUND = .FALSE.
            END IF

            IF      ( HARTYP .EQ. 'DAF/CK'  .AND. FOUND ) THEN

               CALL CCIFRM ( 3, TFRID(1), FRCODE, TFRNAM(1), HINT,FOUND)

            ELSE IF ( HARTYP .EQ. 'DAF/PCK' .AND. FOUND ) THEN

               CALL CCIFRM ( 2, TFRID(1), FRCODE, TFRNAM(1), HINT,FOUND)

            END IF

            IF ( FOUND ) THEN
               TFRCID(1) = TFRID(1)
               TFRID(1)  = FRCODE
            ELSE
               TFRNAM(1) = ' '
            END IF

            IF ( TFRNAM(1) .EQ. ' ' ) THEN

               CALL FRMNAM ( TFRID(1), TFRNAM(1) )

               IF ( TFRNAM(1) .EQ. ' ' ) THEN
                  CALL SETMSG ( '''#'' specified after ''#'' key is '//
     .                          'neither an integer number '         //
     .                          'representing a legitimate frame '   //
     .                          'ID nor a frame name recognized in ' //
     .                          'SPICE.'                             )
                  CALL ERRCH  ( '#', CLVALS(I)                       )
                  CALL ERRCH  ( '#', TF1KEY                          )
                  CALL SIGERR ( 'SPICE(BADTOFRAME1SPEC2)'            )
               END IF

            END IF

         END IF

C
C        If we are here, we were able to resolve this frame. Get and
C        buffer its class and class ID; we may need these later
C        to set the coverage frame.
C
         CALL FRINFO( TFRID(1), HINT, TFRCLS(1), TFRCID(1), FOUND )

      END IF

C
C     Next, get the first ``coverage'' frame name or class ID.
C
      CFRNAM(1) = ' '
      CFRCID(1) = 0
      CFRCLS(1) = 0
      CFRFND(1) = .FALSE.

      I = ISRCHC( CF1KEY, MAXKEY, CLKEYS )

      IF (     CLFLAG(I)                                 .AND. 
     .     ( ( KERNAM(1) .NE. ' ' .AND. ISCPCK(1) ) .OR. 
     .       ( KERNAM(1) .EQ. ' ' .AND. ISCPCK(2) )      )    ) THEN

C
C        This case -- specifying the first coverage frame and providing
C        the first or only primary file that is CK or PCK -- was used
C        in the version 1.0.0 of the program which did not allow
C        specifying coverage frames and fetching coverage without CK or
C        PCK primary kernels.
C
C        The coverage frame applies only to CK and binary PCKs files.
C        If it was specified for any other kernels or if no kernels
C        were specified, we report an error.
C
C        If two files were given, check the first file. If one file was
C        given, check that file. If no files were given, report an
C        error right away.
C
         IF ( KERNAM(1) .NE. ' ' ) THEN
            HKRNAM = KERNAM(1)
            HARTYP = ARCTYP(1)
         ELSE
            HKRNAM = KERNAM(2)
            HARTYP = ARCTYP(2)
         END IF

C
C        The two errors below will never execute in versions 1.2.+
C        because we can get here only with a non-blank file name and a
C        CK or PCK file type. This block is left here for ease of
C        comparison with the previous versions of this routine.
C
         IF ( HKRNAM .NE. ' ' ) THEN

            IF ( .NOT. ( HARTYP .EQ. 'DAF/CK'  .OR.
     .                   HARTYP .EQ. 'DAF/PCK'     ) ) THEN

               CALL SETMSG ( '''#'' option cannot be used in this '  //
     .                       'run because the file ''#'' is not a '  //
     .                       'CK or binary PCK.'                     )
               CALL ERRCH  ( '#', CF1KEY                             )
               CALL ERRCH  ( '#', HKRNAM                             )
               CALL SIGERR ( 'SPICE(BADCOVFRAME1SPEC1)'              )

            END IF

         ELSE

            CALL SETMSG ( '''#'' option cannot be used in this '     //
     .                    'run because no files were specified '     //
     .                    'on the command line.'                     )
            CALL ERRCH  ( '#', CF1KEY                                )
            CALL SIGERR ( 'SPICE(BADCOVFRAME1SPEC2)'                 )

         END IF

C
C        OK, we do deal with a CK or a binary PCK. Let's see if we can
C        map this frame to an ID and if there is data for this ID in
C        this file.
C
         CFRNAM(1) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), CFRCID(1), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

C
C           It's not an integer. Assume that it's a frame name and
C           try to get its ID.
C
            CALL NAMFRM ( CLVALS(I), CFRCID(1) )

            IF ( CFRCID(1) .EQ. 0 ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', CF1KEY                             )
               CALL SIGERR ( 'SPICE(BADCOVFRAME1SPEC4)'              )
            END IF

C
C           Get class and class ID for this frame. Check if class
C           matches the file type and if the class ID is one of the
C           IDs present in the file. Again, we need to check this
C           either for the first or for the second file, depending on
C           how many files were provided.
C
            CALL FRINFO ( CFRCID(1), HINT, CFRCLS(1), CLSSID, FOUND )

            IF ( KERNAM(1) .NE. ' ' ) THEN
               HKRNAM = KERNAM(1)
               HARTYP = ARCTYP(1)
               FOUND = ELEMI( CLSSID, FLIDS1 )
            ELSE
               HKRNAM = KERNAM(2)
               HARTYP = ARCTYP(2)
               FOUND = ELEMI( CLSSID, FLIDS2 )
            END IF

            IF ( ( HARTYP .EQ. 'DAF/PCK' .AND. CFRCLS(1) .EQ. 2 ) .OR.
     .           ( HARTYP .EQ. 'DAF/CK'  .AND. CFRCLS(1) .EQ. 3 ) ) THEN

               IF ( FOUND ) THEN

                  CFRCID(1) = CLSSID
                  CFRFND(1) = FOUND

               ELSE

                  CALL SETMSG ( 'The file ''#'' does not contain '   //
     .                          'any data for the frame ''#'' '      //
     .                          '(frame class ID ''#'') '            //
     .                          'specified after ''#'' key.'         )
                  CALL ERRCH  ( '#', HKRNAM                          )
                  CALL ERRCH  ( '#', CLVALS(I)                       )
                  CALL ERRINT ( '#', CLSSID                          )
                  CALL ERRCH  ( '#', CF1KEY                          )
                  CALL SIGERR ( 'SPICE(COVFRAME1NODATA1)'            )

               END IF

            ELSE

               CALL SETMSG ( 'The class of the frame ''#'' '         //
     .                       'provided after ''#'' key does '        //
     .                       'not match the kernel type of the '     //
     .                       'file ''#'', to which the frame '       //
     .                       'applies. The class was ''#''; '        //
     .                       'the kernel architecture/type '         //
     .                       'were ''#''.'                           )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', CF1KEY                             )
               CALL ERRCH  ( '#', HKRNAM                             )
               CALL ERRINT ( '#', CFRCLS(1)                          )
               CALL ERRCH  ( '#', HARTYP                             )
               CALL SIGERR ( 'SPICE(COVFRAME1MISMATCH)'              )

            END IF

         ELSE

C
C           It's an integer. We just need to check that the file in
C           question contains data for this ID. If not, report an
C           error.
C
            IF ( KERNAM(1) .NE. ' ' ) THEN
               HKRNAM = KERNAM(1)
               HARTYP = ARCTYP(1)
               CFRFND(1) = ELEMI( CFRCID(1), FLIDS1 )
            ELSE
               HKRNAM = KERNAM(2)
               HARTYP = ARCTYP(2)
               CFRFND(1) = ELEMI( CFRCID(1), FLIDS2 )
            END IF

            IF ( .NOT. CFRFND(1) ) THEN
               CALL SETMSG ( 'The file ''#'' does not contain '      //
     .                       'any data for the ID ''#''  '           //
     .                       'specified after ''#'' key.'            )
               CALL ERRCH  ( '#', HKRNAM                             )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', CF1KEY                             )
               CALL SIGERR ( 'SPICE(COVFRAME1NODATA2)'               )
            END IF

C
C           OK, this is a stretch but we have to do it to not require
C           SPICE to know the frame associated with this class ID.
C           We will set coverage frame class simply based on the kernel
C           type.
C
            IF      ( HARTYP .EQ. 'DAF/CK' ) THEN
               CFRCLS(1) = 3
            ELSE IF ( HARTYP .EQ. 'DAF/PCK' ) THEN
               CFRCLS(1) = 2
            END IF

         END IF

      ELSE IF ( CLFLAG(I) ) THEN

C
C        If we are here, the first coverage frame was provided on the
C        command line but either no primary kernels files were given or
C        the primary kernel to which this coverage frame applies is not
C        a CK or a PCK. Was the frame given as a name or as an ID? Try
C        to convert it to integer and see if it worked.
C
         CFRNAM(1) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), CFRID(1), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

C
C           It's not an integer. Assume that it's the frame name and
C           try to convert it to ID. If we can't, report an error.
C
            CALL NAMFRM ( CLVALS(I), CFRID(1) )

            IF ( CFRID(1) .EQ. 0 ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', CF1KEY                             )
               CALL SIGERR ( 'SPICE(BADCOVFRAME1SPEC5)'              )
            END IF

         ELSE

C
C           It's an integer. Assume that it's the frame ID and try to
C           convert it to name. If we can't, report an error.
C
            CALL FRMNAM ( CFRID(1), CFRNAM(1) )

            IF ( CFRNAM(1) .EQ. ' ' ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', CF1KEY                             )
               CALL SIGERR ( 'SPICE(BADCOVFRAME1SPEC6)'              )
            END IF

         END IF

C
C        OK, we could recognize this frame. Get its class and class
C        ID. Signal an error if its class is not 2 or 3 as we can
C        get coverage only for PCK- or CK-based frames.
C
         CALL FRINFO( CFRID(1), HINT, CFRCLS(1), CFRCID(1), CFRFND(1) )

         IF ( .NOT. ( CFRCLS(1) .EQ. 2 .OR. CFRCLS(1) .EQ. 3 ) ) THEN
            CALL SETMSG ( 'Frame ''#'' (frame ID #) specified '      //
     .                    'after ''#'' key is neither a PCK '        //
     .                    '(class 2) or a CK (class 3) frame. '      //
     .                    'Its class is #.'                          )
            CALL ERRCH  ( '#', CFRNAM(1)                             )
            CALL ERRINT ( '#', CFRID(1)                              )
            CALL ERRCH  ( '#', CF1KEY                                )
            CALL ERRINT ( '#', CFRCLS(1)                             )
            CALL SIGERR ( 'SPICE(BADCOVFRAME1SPEC5)'                 )
         END IF

C
C        Set data found flag to FALSE because it is applicable only
C        availability of data in the first or only CK or PCK file.
C
         CFRFND(1) = .FALSE.

      END IF

C
C     We are done with command line keys defining the first attitude
C     set. We will unload all kernels, load kernels applicable to the
C     second attitude set and move on to processing keys for it.
C
      CALL KCLEAR

C
C     If the second kernel is provided, load it. Also load
C     additional kernels supplied for the second kernel and kernels
C     applicable for both kernels.
C
      HLLINE = KERNLS(3)//' '//KERNLS(2)//' '//KERNAM(2)
      CALL LDKLST( HLLINE )

C
C     After loading all kernels provided for computing the second
C     attitude set we can move on to getting command line inputs
C     defining it.
C
C     Start with getting the second ``from'' frame.
C
      FFRNAM(2) = ' '
      FFRID(2)  = 0

      I = ISRCHC( FF2KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

C
C        The second ``from'' frame was provided on the command line.
C        Was it given as a name or as an ID? Try to convert it to
C        integer and see if it worked.
C
         FFRNAM(2) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), FFRID(2), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

C
C           It's not an integer. Assume that it's the frame name and
C           try to convert it to ID. If we can't, report an error.
C
            CALL NAMFRM ( CLVALS(I), FFRID(2) )

            IF ( FFRID(2) .EQ. 0 ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', FF2KEY                             )
               CALL SIGERR ( 'SPICE(BADFROMFRAME2SP1)'               )
            END IF

         ELSE

C
C           It's an integer. Assume that it's the frame ID and try to
C           convert it to name. If we can't, report an error.
C
            CALL FRMNAM ( FFRID(2), FFRNAM(2) )

            IF ( FFRNAM(2) .EQ. ' ' ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', FF2KEY                             )
               CALL SIGERR ( 'SPICE(BADFROMFRAME2SP2)'               )
            END IF

         END IF

      END IF

C
C     Next, get the second ``to'' frame.
C
      TFRNAM(2) = ' '
      TFRID(2)  = 0
      TFRCID(2) = 0
      TFRCLS(2) = 0

      I = ISRCHC( TF2KEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

C
C        The second ``to'' frame was provided on the command line. Was
C        it given as a name or as an ID? Try to convert it to integer
C        and see if it worked.
C
         TFRNAM(2) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), TFRID(2), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

C
C           It's not an integer. Assume that it's the frame name and
C           try to convert it to ID. If we can't, report an error.
C
            CALL NAMFRM ( CLVALS(I), TFRID(2) )

            IF ( TFRID(2) .EQ. 0 ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', TF2KEY                             )
               CALL SIGERR ( 'SPICE(BADTOFRAME2SPEC1)'               )
            END IF

         ELSE

C
C           It's an integer; treat it as a frame ID or a frame class ID
C           depending on context and try to map it to frame name.
C
C           If the second or only file is a CK and this ID is one of
C           the IDs present in it, treat it as a CK class ID and try to
C           map it to frame as such.
C
C           If the second or only file is a binary PCK and this ID is
C           one of the IDs present in it, treat it as a PCK class ID
C           and try to map it to frame as such.
C
C           If the second or only file is not a CK or a binary PCK, or
C           if it is BUT the ID could not be mapped to frame as a class
C           ID, try to map it to frame by treating it as a frame ID. If
C           it can't be mapped, report an error.
C
            IF      ( KERNAM(2) .NE. ' ' ) THEN
               HARTYP = ARCTYP(2)
               FOUND = ELEMI( TFRID(2), FLIDS2 )
            ELSE
               HARTYP = ' '
               FOUND = .FALSE.
            END IF

            IF      ( HARTYP .EQ. 'DAF/CK'  .AND. FOUND ) THEN

               CALL CCIFRM ( 3, TFRID(2), FRCODE, TFRNAM(2), HINT,FOUND)

            ELSE IF ( HARTYP .EQ. 'DAF/PCK' .AND. FOUND ) THEN

               CALL CCIFRM ( 2, TFRID(2), FRCODE, TFRNAM(2), HINT,FOUND)

            END IF

            IF ( FOUND ) THEN
               TFRCID(2) = TFRID(2)
               TFRID(2)  = FRCODE
            ELSE
               TFRNAM(2) = ' '
            END IF

            IF ( TFRNAM(2) .EQ. ' ' ) THEN

               CALL FRMNAM ( TFRID(2), TFRNAM(2) )

               IF ( TFRNAM(2) .EQ. ' ' ) THEN
                  CALL SETMSG ( '''#'' specified after ''#'' key is '//
     .                          'neither an integer number '         //
     .                          'representing a legitimate frame '   //
     .                          'ID nor a frame name recognized in ' //
     .                          'SPICE.'                             )
                  CALL ERRCH  ( '#', CLVALS(I)                       )
                  CALL ERRCH  ( '#', TF2KEY                          )
                  CALL SIGERR ( 'SPICE(BADTOFRAME2SPEC2)'            )
               END IF

            END IF

         END IF

C
C        If we are here, we were able to resolve this frame. Get and
C        buffer its class and class ID; we may need these later
C        to set the coverage frame.
C
         CALL FRINFO( TFRID(2), HINT, TFRCLS(2), TFRCID(2), FOUND )

      END IF

C
C     Next, get the second ``coverage'' frame name or class ID.
C
      CFRNAM(2) = ' '
      CFRCID(2) = 0
      CFRCLS(2) = 0
      CFRFND(2) = .FALSE.

      I = ISRCHC( CF2KEY, MAXKEY, CLKEYS )

      IF (     CLFLAG(I)                                 .AND. 
     .       ( KERNAM(2) .NE. ' ' .AND. ISCPCK(2) )           ) THEN

C
C        This case -- specifying the second coverage frame and
C        providing the second or only primary file that is CK or PCK --
C        was used in the version 1.0.0 of the program which did not
C        allow specifying coverage frames and fetching coverage without
C        CK or PCK primary kernels.
C
C        The coverage frame applies only to CK and binary PCKs files.
C        If it was specified for any other kernels or if no kernels
C        were specified, we report an error.
C
C        If the file was given, check it. If no file was given, report
C        an error right away.
C
         HKRNAM = KERNAM(2)
         HARTYP = ARCTYP(2)

C
C        The two errors below will never execute in versions 1.2.+
C        because we can get here only with a non-blank file name and a
C        CK or PCK file type. This block is left here for ease of
C        comparison with the previous versions of this routine.
C
         IF ( HKRNAM .NE. ' ' ) THEN

            IF ( .NOT. ( HARTYP .EQ. 'DAF/CK'  .OR.
     .                   HARTYP .EQ. 'DAF/PCK'     ) ) THEN

               CALL SETMSG ( '''#'' option cannot be used in this '  //
     .                       'run because the file ''#'' is not a '  //
     .                       'CK or binary PCK.'                     )
               CALL ERRCH  ( '#', CF2KEY                             )
               CALL ERRCH  ( '#', HKRNAM                             )
               CALL SIGERR ( 'SPICE(BADCOVFRAME2SPEC1)'              )

            END IF

         ELSE

            CALL SETMSG ( '''#'' option cannot be used in this '     //
     .                    'run because no files were specified '     //
     .                    'on the command line.'                     )
            CALL ERRCH  ( '#', CF2KEY                                )
            CALL SIGERR ( 'SPICE(BADCOVFRAME2SPEC2)'                 )

         END IF

C
C        OK, we do deal with a CK or a binary PCK. Let's see if we can
C        map this frame to an ID and if there is data for this ID in
C        this file.
C
         CFRNAM(2) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), CFRCID(2), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

C
C           It's not an integer. Assume that it's a frame name and
C           try to get its ID.
C
            CALL NAMFRM ( CLVALS(I), CFRCID(2) )

            IF ( CFRCID(2) .EQ. 0 ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', CF2KEY                             )
               CALL SIGERR ( 'SPICE(BADCOVFRAME2SPEC4)'              )
            END IF

C
C           Get class and class ID for this frame. Check if class
C           matches the file type and if the class ID is one of the
C           IDs present in the file.
C
            CALL FRINFO ( CFRCID(2), HINT, CFRCLS(2), CLSSID, FOUND )

            HKRNAM = KERNAM(2)
            HARTYP = ARCTYP(2)
            FOUND = ELEMI( CLSSID, FLIDS2 )

            IF ( ( HARTYP .EQ. 'DAF/PCK' .AND. CFRCLS(2) .EQ. 2 ) .OR.
     .           ( HARTYP .EQ. 'DAF/CK'  .AND. CFRCLS(2) .EQ. 3 ) ) THEN

               IF ( FOUND ) THEN

                  CFRCID(2) = CLSSID
                  CFRFND(2) = FOUND

               ELSE

                  CALL SETMSG ( 'The file ''#'' does not contain '   //
     .                          'any data for the frame ''#'' '      //
     .                          '(frame class ID ''#'') '            //
     .                          'specified after ''#'' key.'         )
                  CALL ERRCH  ( '#', HKRNAM                          )
                  CALL ERRCH  ( '#', CLVALS(I)                       )
                  CALL ERRINT ( '#', CLSSID                          )
                  CALL ERRCH  ( '#', CF2KEY                          )
                  CALL SIGERR ( 'SPICE(COVFRAME2NODATA1)'            )

               END IF

            ELSE

               CALL SETMSG ( 'The class of the frame ''#'' '         //
     .                       'provided after ''#'' key does '        //
     .                       'not match the kernel type of the '     //
     .                       'file ''#'', to which the frame '       //
     .                       'applies. The class was ''#''; '        //
     .                       'the kernel architecture/type '         //
     .                       'were ''#''.'                           )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', CF2KEY                             )
               CALL ERRCH  ( '#', HKRNAM                             )
               CALL ERRINT ( '#', CFRCLS(2)                          )
               CALL ERRCH  ( '#', HARTYP                             )
               CALL SIGERR ( 'SPICE(COVFRAME2MISMATCH)'              )

            END IF

         ELSE

C
C           It's an integer. We just need to check that the file in
C           question contains data for this ID. If not, report an
C           error.
C
            HKRNAM = KERNAM(2)
            HARTYP = ARCTYP(2)
            CFRFND(2) = ELEMI( CFRCID(2), FLIDS2 )

            IF ( .NOT. CFRFND(2) ) THEN
               CALL SETMSG ( 'The file ''#'' does not contain '      //
     .                       'any data for the ID ''#''  '           //
     .                       'specified after ''#'' key.'            )
               CALL ERRCH  ( '#', HKRNAM                             )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', CF2KEY                             )
               CALL SIGERR ( 'SPICE(COVFRAME2NODATA2)'               )
            END IF

C
C           OK, this is a stretch but we have to do it to not require
C           SPICE to know the frame associated with this class ID.
C           We will set coverage frame class simply based on the kernel
C           type.
C
            IF      ( HARTYP .EQ. 'DAF/CK' ) THEN
               CFRCLS(2) = 3
            ELSE IF ( HARTYP .EQ. 'DAF/PCK' ) THEN
               CFRCLS(2) = 2
            END IF

         END IF

      ELSE IF ( CLFLAG(I) ) THEN

C
C        If we are here, the second coverage frame was provided on the
C        command line but either no primary kernels files were given or
C        the second or only primary kernel is not a CK or a PCK. Was
C        the frame given as a name or as an ID? Try to convert it to
C        integer and see if it worked.
C
         CFRNAM(2) = CLVALS(I)
         CALL NPARSI ( CLVALS(I), CFRID(2), ERROR, PTR )

         IF ( PTR .NE. 0 ) THEN

C
C           It's not an integer. Assume that it's the frame name and
C           try to convert it to ID. If we can't, report an error.
C
            CALL NAMFRM ( CLVALS(I), CFRID(2) )

            IF ( CFRID(2) .EQ. 0 ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', CF2KEY                             )
               CALL SIGERR ( 'SPICE(BADCOVFRAME2SPEC5)'              )
            END IF

         ELSE

C
C           It's an integer. Assume that it's the frame ID and try to
C           convert it to name. If we can't, report an error.
C
            CALL FRMNAM ( CFRID(2), CFRNAM(2) )

            IF ( CFRNAM(2) .EQ. ' ' ) THEN
               CALL SETMSG ( '''#'' specified after ''#'' key is '   //
     .                       'neither an integer number '            //
     .                       'representing a legitimate frame '      //
     .                       'ID nor a frame name recognized in '    //
     .                       'SPICE.'                                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', CF2KEY                             )
               CALL SIGERR ( 'SPICE(BADCOVFRAME2SPEC6)'              )
            END IF

         END IF

C
C        OK, we could recognize this frame. Get its class and class
C        ID. Signal an error if its class is not 2 or 3 as we can
C        get coverage only for PCK- or CK-based frames.
C
         CALL FRINFO( CFRID(2), HINT, CFRCLS(2), CFRCID(2), CFRFND(2) )

         IF ( .NOT. ( CFRCLS(2) .EQ. 2 .OR. CFRCLS(2) .EQ. 3 ) ) THEN
            CALL SETMSG ( 'Frame ''#'' (frame ID #) specified '      //
     .                    'after ''#'' key is neither a PCK '        //
     .                    '(class 2) or a CK (class 3) frame. '      //
     .                    'Its class is #.'                          )
            CALL ERRCH  ( '#', CFRNAM(2)                             )
            CALL ERRINT ( '#', CFRID(2)                              )
            CALL ERRCH  ( '#', CF2KEY                                )
            CALL ERRINT ( '#', CFRCLS(2)                             )
            CALL SIGERR ( 'SPICE(BADCOVFRAME2SPEC5)'                 )
         END IF

C
C        Set data found flag to FALSE because it is applicable only
C        availability of data in the second or only CK or PCK file.
C
         CFRFND(2) = .FALSE.
         
      END IF

C
C     Process angular velocity comparison command line key and set
C     the output angular velocity flag (AVFLG) based on its value.
C
      I = ISRCHC( AVKEY, MAXKEY, CLKEYS )

      AVFLG = .FALSE.

      IF ( CLFLAG(I) ) THEN

         IF      ( EQSTR( CLVALS(I), YESVAL ) ) THEN

            AVFLG = .TRUE.

         ELSE IF ( EQSTR( CLVALS(I), NOVAL  ) ) THEN

            AVFLG = .FALSE.

         ELSE

            CALL SETMSG ( 'Angular velocity comparison flag ''#'' '  //
     .                    'specified after ''#'' key is not '        //
     .                    'recognized. Recognized values are ''#'' ' //
     .                    'and ''#''.'                               )
            CALL ERRCH  ( '#', CLVALS(I)                             )
            CALL ERRCH  ( '#', AVKEY                                 )
            CALL ERRCH  ( '#', YESVAL                                )
            CALL ERRCH  ( '#', NOVAL                                 )
            CALL SIGERR ( 'SPICE(BADAVFLAG)'                         )

         END IF

      END IF

C
C     Process angular velocity frame command line key and set the
C     output angular velocity frame flag (AVFFLG) based on its value.
C
      I = ISRCHC( AVFKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         IF      ( EQSTR( CLVALS(I), FRMVAL ) ) THEN

            AVFFLG = .FALSE.

         ELSE IF ( EQSTR( CLVALS(I), TOVAL  ) ) THEN

            AVFFLG = .TRUE.

         ELSE

            CALL SETMSG ( 'Angular velocity frame flag ''#'' '       //
     .                    'specified after ''#'' key is not '        //
     .                    'recognized. Recognized values are ''#'' ' //
     .                    'and ''#''.'                               )
            CALL ERRCH  ( '#', CLVALS(I)                             )
            CALL ERRCH  ( '#', AVFKEY                                )
            CALL ERRCH  ( '#', FRMVAL                                )
            CALL ERRCH  ( '#', TOVAL                                 )
            CALL SIGERR ( 'SPICE(BADAVFRAMEFLAG)'                    )

         END IF

      ELSE

         AVFFLG = .FALSE.

      END IF

C
C     Process begin and end time arguments. As these need to be
C     converted to ET, we will need LSK data, which can come in any of
C     the five file sets on the command line -- first or second file,
C     kernels specific for first of second file or kernels applicable
C     to both file. Note that second file, kernels applicable for it
C     and kernels applicable for both file are already loaded. We will
C     load first file and kernels specific to it to make sure that we
C     have loaded all kernels that we could.
C
      HLLINE = KERNLS(1)//' '//KERNAM(1)
      CALL LDKLST( HLLINE )

      TIME(1) = ' '
      ET(1)   = DPMIN()

      I = ISRCHC( BEGKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN
         TIME(1) = CLVALS(I)
         CALL STR2ET( CLVALS(I), ET(1) )
      END IF

      TIME(2) = ' '
      ET(2)   = DPMAX()

      I = ISRCHC( ENDKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN
         TIME(2) = CLVALS(I)
         CALL STR2ET( CLVALS(I), ET(2) )
      END IF

C
C     Check that begin time is less than the end time.
C
      IF ( TIME(1) .NE. ' ' .AND. TIME(2) .NE. ' ' ) THEN

         IF ( ET(2) .LT. ET(1) ) THEN
            CALL SETMSG ( 'Specified start time ''#'' is greater '   //
     .                    'than specified stop time ''#''.'          )
            CALL ERRCH  ( '#', TIME(1)                               )
            CALL ERRCH  ( '#', TIME(2)                               )
            CALL SIGERR ( 'SPICE(INCONSISTENTTIMES)'                 )
         END IF

      END IF

C
C     Make a string describing time bounds provided on the command
C     line to be used later in error messages.
C
      IF      ( TIME(1) .NE. ' ' .AND. TIME(2) .NE. ' ' ) THEN

         TIMDSC = 'Comparison window specified on the command line ' //
     .            'was from ''#'' to ''#''.'
         CALL REPMC ( TIMDSC, '#', TIME(1), TIMDSC )
         CALL REPMC ( TIMDSC, '#', TIME(2), TIMDSC )

      ELSE IF ( TIME(1) .NE. ' ' ) THEN

         TIMDSC = 'Comparison window specified on the command line ' //
     .            'was all times after ''#''.'
         CALL REPMC ( TIMDSC, '#', TIME(1), TIMDSC )

      ELSE IF ( TIME(2) .NE. ' ' ) THEN

         TIMDSC = 'Comparison window specified on the command line ' //
     .            'was all times before ''#''.'
         CALL REPMC ( TIMDSC, '#', TIME(2), TIMDSC )

      ELSE

         TIMDSC = ' '

      END IF

C
C     At this point we processed all command line keys defining
C     attributes of the two attitude sets as well as those restricting
C     time boundaries. It is likely that some of the items we need were
C     not provided (``from'', ``to'', ``cov'' frame) or could not be
C     provided (the final comparison window). Let go ahead and fill in
C     these ``blanks''.
C

C
C     If only one ``from'' name/ID was provided, set the other one to
C     be the same.
C
      IF      ( FFRNAM(1) .EQ. ' ' .AND. FFRNAM(2) .NE. ' ' ) THEN

         FFRID(1)  = FFRID(2)
         FFRNAM(1) = FFRNAM(2)

      ELSE IF ( FFRNAM(2) .EQ. ' ' .AND. FFRNAM(1) .NE. ' ' ) THEN

         FFRID(2)  = FFRID(1)
         FFRNAM(2) = FFRNAM(1)

      END IF

C
C     If only one ``to'' name/ID was provided, set the other one to be
C     the same.
C
      IF      ( TFRNAM(1) .EQ. ' ' .AND. TFRNAM(2) .NE. ' ' ) THEN

         TFRID(1)  = TFRID(2)
         TFRCID(1) = TFRCID(2)
         TFRNAM(1) = TFRNAM(2)
         TFRCLS(1) = TFRCLS(2)

      ELSE IF ( TFRNAM(2) .EQ. ' ' .AND. TFRNAM(1) .NE. ' ' ) THEN

         TFRID(2)  = TFRID(1)
         TFRCID(2) = TFRCID(1)
         TFRNAM(2) = TFRNAM(1)
         TFRCLS(2) = TFRCLS(1)

      END IF

C
C     If after doing this cross-fill either ``from'' or ``to'' frames
C     are still undefined, we need to dig trough the files to get
C     defaults for them.
C
      IF ( ( FFRNAM(1) .EQ. ' ' .AND. FFRNAM(2) .EQ. ' ' ) .OR.
     .     ( TFRNAM(1) .EQ. ' ' .AND. TFRNAM(2) .EQ. ' ' )      ) THEN

C
C        First, check if we got at least one file on the command line.
C        If not, report an error.
C
         IF ( KERNAM(1) .EQ. ' ' .AND. KERNAM(2) .EQ. ' ' ) THEN
            CALL SETMSG ( 'Cannot pick default values for ''#'' '    //
     .                    'frames because no kernels were provided ' //
     .                    'as the last and/or second to last item '  //
     .                    'on the command line.'                     )
            IF ( FFRNAM(1) .EQ. ' ' ) THEN
               CALL ERRCH  ( '#', 'from'                             )
            ELSE
               CALL ERRCH  ( '#', 'to'                               )
            END IF
            CALL SIGERR ( 'SPICE(CANTPICKDEFAULTS1)'                 )
         END IF

C
C        Next, check if the file that we are supposed to examine to get
C        defaults is a CK or binary PCK file. If not, report an error.
C
         IF ( KERNAM(1) .NE. ' ' ) THEN
            HKRNAM = KERNAM(1)
            HARTYP = ARCTYP(1)
         ELSE
            HKRNAM = KERNAM(2)
            HARTYP = ARCTYP(2)
         END IF

         IF ( .NOT. ( HARTYP .EQ. 'DAF/CK'  .OR.
     .                HARTYP .EQ. 'DAF/PCK'      ) ) THEN
            CALL SETMSG ( 'Cannot pick default values for ''#'' '    //
     .                    'frames from the ''#'' file because it '   //
     .                    'is not a CK or binary PCK file.'          )
            IF ( FFRNAM(1) .EQ. ' ' ) THEN
               CALL ERRCH  ( '#', 'from'                             )
            ELSE
               CALL ERRCH  ( '#', 'to'                               )
            END IF
            CALL ERRCH  ( '#', HKRNAM                                )
            CALL SIGERR ( 'SPICE(CANTPICKDEFAULTS2)'                 )
         END IF

C
C        OK, it looks like we can go in and try to get defaults from
C        this file. Open it as a DAF file and fetch descriptor of the
C        last segment. Depending on whether this file is CK or PCK,
C        unpack descriptor using appropriate ND/NI.
C
         CALL DAFOPR ( HKRNAM, HANDLE )
         CALL DAFBBS ( HANDLE )
         CALL DAFFPA ( FOUND )

         IF ( .NOT. FOUND ) THEN
            CALL SETMSG ( 'The ' // HARTYP(5:) // ' file ''#'' '     //
     .                    'does not contain any segments. Cannot '   //
     .                    'pick default ''#'' frame from this file.' )
            CALL ERRCH  ( '#', HKRNAM                                )
            IF ( FFRNAM(1) .EQ. ' ' ) THEN
               CALL ERRCH  ( '#', 'from'                             )
            ELSE
               CALL ERRCH  ( '#', 'to'                               )
            END IF
            CALL SIGERR ( 'SPICE(NOSEGMENTSFOUND)'                   )
         END IF

         CALL DAFGS ( DESCR )

         CALL DAFCLS( HANDLE )

         IF      ( HARTYP .EQ. 'DAF/CK' ) THEN
            CALL DAFUS ( DESCR, ND, NICK,  DC, IC )
         ELSE
            CALL DAFUS ( DESCR, ND, NIPCK, DC, IC )
         END IF

C
C        Since we will need to do the frame name-ID mapping, we need to
C        reload kernels. Currently we have all kernels that have been
C        provided loaded into the program; what we want is only kernels
C        applicable to the first attitude set.
C
         CALL KCLEAR
         HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//HKRNAM
         CALL LDKLST( HLLINE )

C
C        If we need to pick default for ``from'' frames, try to map
C        IC(2) frame ID to frame name.
C
         IF ( FFRNAM(1) .EQ. ' ' ) THEN

            CALL FRMNAM ( IC(2), FFRNAM(1) )

            IF ( FFRNAM(1) .EQ. ' ' ) THEN
               CALL SETMSG ( 'Cannot pick default ''from'' frame '   //
     .                       'because the frame ID ''#'' from '      //
     .                       'the last segment of the '              //
     .                       HARTYP(5:) // ' file '                  //
     .                       '''#'' cannot be mapped to a frame '    //
     .                       'name.'                                 )
               CALL ERRINT ( '#', IC(2)                              )
               CALL ERRCH  ( '#', HKRNAM                             )
               CALL SIGERR ( 'SPICE(CANTPICKDEFAULTS3)'              )
            END IF

            FFRNAM(2) = FFRNAM(1)
            FFRID (1) = IC(2)
            FFRID (2) = IC(2)

         END IF

C
C        If we need to pick default for ``to'' frames, try to map IC(1)
C        frame class ID to frame name.
C
         IF ( TFRNAM(1) .EQ. ' ' ) THEN

            IF ( HARTYP .EQ. 'DAF/CK' ) THEN
               TFRCLS(1) = 3
               CALL CCIFRM ( 3, IC(1), TFRID(1), TFRNAM(1), HINT,FOUND)
            ELSE
               TFRCLS(1) = 2
               CALL CCIFRM ( 2, IC(1), TFRID(1), TFRNAM(1), HINT,FOUND)
            END IF

            IF ( .NOT. FOUND ) THEN
               CALL SETMSG ( 'Cannot pick default ''to'' frame '     //
     .                       'because the frame class ID ''#'' '     //
     .                       'from the last segment of the '         //
     .                       HARTYP(5:)                              //
     .                       ' file ''#'' cannot be mapped to a '    //
     .                       'frame name.'                           )
               CALL ERRINT ( '#', IC(1)                              )
               CALL ERRCH  ( '#', HKRNAM                             )
               CALL SIGERR ( 'SPICE(CANTPICKDEFAULTS4)'              )
            END IF

            TFRNAM(2) = TFRNAM(1)
            TFRID (2) = TFRID(1)
            TFRCLS(2) = TFRCLS(1)
            TFRCID(1) = IC(1)
            TFRCID(2) = IC(1)

         END IF

      END IF

C
C     OK, we now have filled all ``from'' and ``to'' frame attributes.
C     If coverage frame attributes were not set yet, set then to those
C     of the ``to'' frames but only if the ``to'' frames are of CK or
C     PCK types.
C
      IF ( CFRNAM(1) .EQ. ' ' ) THEN

         IF      ( ( ARCTYP(1) .EQ. 'DAF/CK'  .AND.
     .               TFRCLS(1) .EQ. 3               ) .OR. 
     .             ( ARCTYP(1) .EQ. 'DAF/PCK' .AND.
     .               TFRCLS(1) .EQ. 2               )      )  THEN

            CFRNAM(1) = TFRNAM(1)
            CFRCID(1) = TFRCID(1)
            CFRCLS(1) = TFRCLS(1)
            CFRFND(1) = ELEMI( CFRCID(1), FLIDS1 )

      ELSE IF    ( ( KERNAM(1) .EQ. ' '       .AND.
     .               ARCTYP(2) .EQ. 'DAF/CK'  .AND.
     .               TFRCLS(1) .EQ. 3               ) .OR. 
     .             ( KERNAM(1) .EQ. ' '       .AND.
     .               ARCTYP(2) .EQ. 'DAF/PCK' .AND.
     .               TFRCLS(1) .EQ. 2               )      )  THEN

            CFRNAM(1) = TFRNAM(1)
            CFRCID(1) = TFRCID(1)
            CFRCLS(1) = TFRCLS(1)
            CFRFND(1) = ELEMI( CFRCID(1), FLIDS2 )

         ELSE IF ( TFRCLS(1) .EQ. 2 .OR. TFRCLS(1) .EQ. 3 ) THEN

            CFRNAM(1) = TFRNAM(1)
            CFRCID(1) = TFRCID(1)
            CFRCLS(1) = TFRCLS(1)
            CFRFND(1) = .FALSE.

         END IF

      END IF

      IF ( CFRNAM(2) .EQ. ' ' ) THEN

         IF ( ( ARCTYP(2) .EQ. 'DAF/CK'  .AND.
     .          TFRCLS(2) .EQ. 3               ) .OR.
     .        ( ARCTYP(2) .EQ. 'DAF/PCK' .AND. 
     .          TFRCLS(2) .EQ. 2               )     ) THEN

            CFRNAM(2) = TFRNAM(2)
            CFRCID(2) = TFRCID(2)
            CFRCLS(2) = TFRCLS(2)
            CFRFND(2) = ELEMI( CFRCID(2), FLIDS2 )

         ELSE IF ( TFRCLS(2) .EQ. 2 .OR. TFRCLS(2) .EQ. 3 ) THEN

            CFRNAM(2) = TFRNAM(2)
            CFRCID(2) = TFRCID(2)
            CFRCLS(2) = TFRCLS(2)
            CFRFND(2) = .FALSE.

         END IF

      END IF

C
C     Now we need to determine the comparison window.
C
C     If we got no coverage frames specified or picked by default from
C     given CK or PCK files, the user must have provided start and stop
C     times on the command line. If not, we stop.
C
      IF (   CFRNAM(1) .EQ. ' '       .AND.
     .       CFRNAM(2) .EQ. ' '       .AND.
     .     ( TIME(1)   .EQ. ' ' .OR.
     .       TIME(2)   .EQ. ' '      )      ) THEN

         CALL SETMSG ( 'Cannot determine time range for comparison. '//
     .                 'Both start time and stop time must be '      //
     .                 'provided using ''#'' and ''#'' keys when no '//
     .                 'coverage look up frames and kernels from '   //
     .                 'which coverage can be obtained '             //
     .                 'automatically are given on the command line.')
         CALL ERRCH  ( '#', BEGKEY                                   )
         CALL ERRCH  ( '#', ENDKEY                                   )
         CALL SIGERR ( 'SPICE(NOTIMEBOUNDS1)'                        )

      END IF

C
C     We are done checking the obvious error cases. Since we know our
C     coverage frames, let's try to get coverage windows for each of
C     them and do more checks after that. 
C
C     For the first coverage frame we may get coverage either from the
C     first or the only file or from the combination of kernels
C     applicable to both orientations and the first orientation.
C
C     Initialize the first coverage window and description string.
C
      CALL SSIZED ( WINSIZ, COVER1 )
      CALL SCARDD ( 0,      COVER1 )

      CALL SSIZED ( WINSIZ, COVERA )
      CALL SCARDD ( 0,      COVERA )

      CALL SSIZED ( WINSIZ, COVERB )
      CALL SCARDD ( 0,      COVERB )

      COVDSC(1) = ' '

      IF      ( ( KERNAM(1) .NE. ' '       .AND.
     .            ARCTYP(1) .EQ. 'DAF/CK'  .AND.
     .            CFRCLS(1) .EQ. 3         .AND.
     .            CFRFND(1)                      ) .OR.
     .          ( KERNAM(1) .EQ. ' '       .AND.
     .            ARCTYP(2) .EQ. 'DAF/CK'  .AND.
     .            CFRCLS(1) .EQ. 3         .AND.
     .            CFRFND(1)                      )      ) THEN

C
C        Try to get coverage for the first coverage frame from the
C        first or only CK if
C
C           the first file is present,
C           the first file is a CK,
C           the frame is a CK frame, and
C           data for frame's class ID is present in the file
C
C        OR if
C
C           the first file is not present,
C           the second file is a CK,
C           the frame is a CK frame, and
C           data for frame's class ID is present in the file.
C
C        Since CKCOVR call will need SCLK and LSK data to do time
C        conversions, clear loaded kernels and reload the file we will
C        look at and all kernels applicable to the first attitude set.
C

C
C        Set the file we will work with.
C
         IF ( KERNAM(1) .NE. ' ' ) THEN
            HKRNAM = KERNAM(1)
         ELSE
            HKRNAM = KERNAM(2)
         END IF

C
C        Load appropriate kernels.
C
         CALL KCLEAR
         HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//HKRNAM
         CALL LDKLST( HLLINE )

C
C        Get ET coverage window adjusted for round off, at interval
C        level, with zero tolerance, and angular rate flag set earlier.
C
         CALL CKCOVR( HKRNAM, CFRCID(1), AVFLG, 'INTERVAL', 0.D0,
     .                COVER1 )

C
C        If we got an empty window back, report an error. If not,
C        make an information string that we may have to use in the
C        error message(s) later.
C
         IF ( WNCARD(COVER1) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because CK file ''#'' does '   //
     .                    'not provide any roundoff-adjusted '       //
     .                    'coverage for the CK frame with class ID ' //
     .                    '''#''.'                                   )
            CALL ERRCH  ( '#', HKRNAM                                )
            CALL ERRINT ( '#', CFRCID(1)                             )
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS4)'                     )

         ELSE

C
C           The frame name not being an integer means that we know this
C           frame. If so, include frame name and class ID in the
C           string. Otherwise, use only frame class ID.
C
            CALL NPARSI ( CFRNAM(1), HINT, ERROR, PTR )
            IF ( PTR .NE. 0 ) THEN
               COVDSC(1) = 'The first coverage window, for the CK '  //
     .                     'frame ''#'' (class '                     //
     .                     'ID ''#''), was determined from the CK '  //
     .                     'file ''#''.'
               CALL REPMC ( COVDSC(1), '#', CFRNAM(1), COVDSC(1) )
               CALL REPMI ( COVDSC(1), '#', CFRCID(1), COVDSC(1) )
               CALL REPMC ( COVDSC(1), '#', HKRNAM,    COVDSC(1) )
            ELSE
               COVDSC(1) = 'The first coverage window, for the CK '  //
     .                     'frame with class ID '                    //
     .                     '''#'', was determined from the CK file ' //
     .                     '''#''.'
               CALL REPMI ( COVDSC(1), '#', CFRCID(1), COVDSC(1) )
               CALL REPMC ( COVDSC(1), '#', HKRNAM,    COVDSC(1) )
            END IF

         END IF


      ELSE IF ( ( KERNAM(1) .NE. ' '       .AND.
     .            ARCTYP(1) .EQ. 'DAF/PCK' .AND.
     .            CFRCLS(1) .EQ. 2         .AND.
     .            CFRFND(1)                      ) .OR.
     .          ( KERNAM(1) .EQ. ' '       .AND.
     .            ARCTYP(2) .EQ. 'DAF/PCK' .AND.
     .            CFRCLS(1) .EQ. 2         .AND.
     .            CFRFND(1)                      )      ) THEN


C
C        Try to get coverage for the first coverage frame from the
C        first or only PCK if
C
C           the first file is present,
C           the first file is a PCK,
C           the frame is a PCK frame, and
C           data for frame's class ID is present in the file
C
C        OR if
C
C           the first file is not present,
C           the second file is a PCK,
C           the frame is a PCK frame, and
C           data for frame's class ID is present in the file.
C
C        No need to reload any kernels for PCK coverage look up.
C

C
C        Set the file we will work with.
C
         IF ( KERNAM(1) .NE. ' ' ) THEN
            HKRNAM = KERNAM(1)
         ELSE
            HKRNAM = KERNAM(2)
         END IF

C
C        Get coverage.
C
         CALL PCKCOV ( HKRNAM, CFRCID(1), COVER1 )

C
C        I can't think of a way to get empty window out of the
C        previous call, but we will check for it anyway.
C
C        If we got an empty window back, report an error. If not,
C        make an information string that we may have to use in the
C        error message(s) later.
C
         IF ( WNCARD(COVER1) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because PCK file ''#'' does '  //
     .                    'not provide any coverage for the PCK '    //
     .                    'frame with class ID ''#''.'               )
            CALL ERRCH  ( '#', HKRNAM                                )
            CALL ERRINT ( '#', CFRCID(1)                             )
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS5)'                     )

         ELSE

C
C           The frame name not being an integer means that we know this
C           frame. If so, include frame name and class ID in the
C           string. Otherwise, use only frame class ID.
C
            CALL NPARSI ( CFRNAM(1), HINT, ERROR, PTR )
            IF ( PTR .NE. 0 ) THEN
               COVDSC(1) = 'The first coverage window, for the PCK ' //
     .                     'frame ''#'' (class '                     //
     .                     'ID ''#''), was determined from PCK file '//
     .                     '''#''.'
               CALL REPMC ( COVDSC(1), '#', CFRNAM(1), COVDSC(1) )
               CALL REPMI ( COVDSC(1), '#', CFRCID(1), COVDSC(1) )
               CALL REPMC ( COVDSC(1), '#', HKRNAM,    COVDSC(1) )
            ELSE
               COVDSC(1) = 'The first coverage window, for the PCK ' //
     .                     'frame with class '                       //
     .                     'ID ''#'', was determined from PCK '      //
     .                     'file ''#''.'
               CALL REPMI ( COVDSC(1), '#', CFRCID(1), COVDSC(1) )
               CALL REPMC ( COVDSC(1), '#', HKRNAM,    COVDSC(1) )
            END IF

         END IF


      ELSE IF ( ( KERNAM(1) .NE. ' '       .AND.
     .            .NOT. ISCPCK(1)          .AND.
     .            CFRCLS(1) .EQ. 3               ) .OR.
     .          ( KERNAM(1) .EQ. ' '       .AND.
     .            KERNAM(2) .NE. ' '       .AND.
     .            .NOT. ISCPCK(2)          .AND.
     .            CFRCLS(1) .EQ. 3               ) .OR.
     .          ( KERNAM(1) .EQ. ' '       .AND.
     .            KERNAM(2) .EQ. ' '       .AND.
     .            CFRCLS(1) .EQ. 3               )      ) THEN

C
C        Try to get coverage for the first coverage frame from all CKs
C        provided for both orientations and the first orientation if
C
C           the first file is present,
C           the first file is not a CK or PCK, and 
C           the frame is a CK frame, 
C
C        OR if
C
C           the first file is not present,
C           the second file is present,
C           the second file is not a CK or PCK, and 
C           the frame is a CK frame
C
C        OR if
C
C           the first file is not present,
C           the second file is not present, and 
C           the frame is a CK frame
C
C        We need to reload all kernels applicable the first orientation
C        to do this look up. First, set the primary file we will work
C        with.
C
         IF ( KERNAM(1) .NE. ' ' ) THEN
            HKRNAM = KERNAM(1)
         ELSE
            HKRNAM = KERNAM(2)
         END IF

         CALL KCLEAR
         HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//HKRNAM
         CALL LDKLST( HLLINE )

C
C        Get the total count of loaded CKs and, if the count is
C        non-zero, proceed to fetch coverage. 
C
C        Getting the zero count is not an error at this point as we may
C        still come up with a comparison window using the other
C        coverage frame and/or begin and end times specified on the
C        command line. Still we need to make an information string that
C        we may have to use in the error message(s) later.
C
         CALL KTOTAL ( 'CK', COUNT )

         IF ( COUNT .EQ. 0 ) THEN

            COVDSC(1) = 'The first coverage window, for #, could ' //
     .                  'not be determined '  //
     .                  'because no CK files were provided using ' //
     .                  'the ''#'' and/or ''#'' keys. ' 
            
         ELSE

C
C           Loop through all CK files and collect coverage information
C           for this class ID.
C
            DO I = 1, COUNT 

C
C              Get the name of the next CK file.
C
               CALL KDATA ( I, 'CK', 
     .                      HKRNAM, HARTYP, HLINE2, HANDLE, FOUND )

               IF ( .NOT. FOUND ) THEN
                  CALL SETMSG ( 'There is a bug in the program. ' //
     .                          'Please, contact NAIF.'           )
                  CALL SIGERR ( 'SPICE(FRMDIFFBUG3)'              )
               END IF

C
C              Reset auxiliary coverage window and get ET coverage
C              window adjusted for round off, at interval level, with
C              zero tolerance, and angular rate flag set earlier for
C              this CK.
C
               CALL SCARDD ( 0, COVERA )

               CALL CKCOVR ( HKRNAM, CFRCID(1), AVFLG, 'INTERVAL', 0.D0,
     .                       COVERA )

C
C              Merge auxiliary coverage with the total coverage for
C              this frame for all CKs.
C
               CALL WNUNID ( COVER1, COVERA, COVERB )
               CALL COPYD  ( COVERB, COVER1 )

            END DO

C
C           Getting an empty window back is still not an error at this
C           point. Whether it is empty or not, we need to make an
C           information string that we may have to use in the error
C           message(s) later.
C
            IF ( WNCARD(COVER1) .EQ. 0 ) THEN

               COVDSC(1) = 'The first coverage window, for #, could ' //
     .                     'not be determined '                       //
     .                     'because the CK files provided using '     //
     .                     'the ''#'' and/or ''#'' keys contain no '  //
     .                     'data for this frame.'

            ELSE

               COVDSC(1) = 'The first coverage window, for #, '       //
     .                     'was determined from the CK '              //
     .                     'files provided using the ''#'' and/or '   //
     .                     '''#'' keys.'

            END IF

         END IF

C
C        Finish by making the information string by substituting either
C        the frame name and class ID and the kernel command line keys.
C
         CALL REPMC ( COVDSC(1), '#', 
     .                'the CK frame ''#'' (class ID ''#'')', 
     .                                           COVDSC(1) )
         CALL REPMC ( COVDSC(1), '#', CFRNAM(1), COVDSC(1) )
         CALL REPMI ( COVDSC(1), '#', CFRCID(1), COVDSC(1) )
         CALL REPMC ( COVDSC(1), '#', KERKEY,    COVDSC(1) )
         CALL REPMC ( COVDSC(1), '#', KE1KEY,    COVDSC(1) )

                  
      ELSE IF ( ( KERNAM(1) .NE. ' '       .AND.
     .            .NOT. ISCPCK(1)          .AND.
     .            CFRCLS(1) .EQ. 2               ) .OR.
     .          ( KERNAM(1) .EQ. ' '       .AND.
     .            KERNAM(2) .NE. ' '       .AND.
     .            .NOT. ISCPCK(2)          .AND.
     .            CFRCLS(1) .EQ. 2               ) .OR.
     .          ( KERNAM(1) .EQ. ' '       .AND.
     .            KERNAM(2) .EQ. ' '       .AND.
     .            CFRCLS(1) .EQ. 2               )      ) THEN

C
C        Try to get coverage for the first coverage frame from all PCKs
C        provided for both orientations and the first orientation if
C
C           the first file is present,
C           the first file is not a CK or PCK, and 
C           the frame is a PCK frame, 
C
C        OR if
C
C           the first file is not present,
C           the second file is present,
C           the second file is not a CK or PCK, and 
C           the frame is a PCK frame
C
C        OR if
C
C           the first file is not present,
C           the second file is not present, and 
C           the frame is a PCK frame
C
C        We need to reload all kernels applicable the first orientation
C        to do this look up.
C
         IF ( KERNAM(1) .NE. ' ' ) THEN
            HKRNAM = KERNAM(1)
         ELSE
            HKRNAM = KERNAM(2)
         END IF

         CALL KCLEAR
         HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//HKRNAM
         CALL LDKLST( HLLINE )

C
C        Get the total count of loaded binary PCKs and, if the count is
C        non-zero, proceed to fetch coverage.
C
C        Getting the zero count is not an error at this point as we may
C        still come up with a comparison window using the other
C        coverage frame and/or begin and end times specified on the
C        command line. Still we need to make an information string that
C        we may have to use in the error message(s) later.
C
         CALL KTOTAL ( 'PCK', COUNT )

         IF ( COUNT .EQ. 0 ) THEN

            COVDSC(1) = 'The first coverage window, for #, could ' //
     .                  'not be determined '                       //
     .                  'because no binary PCK files were '        //
     .                  'provided using the ''#'' and/or ''#'' keys. '
            
         ELSE

C
C           Loop through all binary PCK files and collect coverage
C           information for this class ID.
C
            DO I = 1, COUNT 

C
C              Get the name of the next binary PCK file.
C
               CALL KDATA ( I, 'PCK', 
     .                      HKRNAM, HARTYP, HLINE2, HANDLE, FOUND )

               IF ( .NOT. FOUND ) THEN
                  CALL SETMSG ( 'There is a bug in the program. ' //
     .                          'Please, contact NAIF.'           )
                  CALL SIGERR ( 'SPICE(FRMDIFFBUG4)'              )
               END IF

C
C              Add coverage for this frame from this binary PCK to
C              the total coverage from all binary PCKs
C
               CALL PCKCOV ( HKRNAM, CFRCID(1), COVER1 )

            END DO

C
C           Getting an empty window back is still not an error at this
C           point. Whether it is empty or not, we need to make an
C           information string that we may have to use in the error
C           message(s) later.
C
            IF ( WNCARD(COVER1) .EQ. 0 ) THEN

               COVDSC(1) = 'The first coverage window, for #, could ' //
     .                     'not be determined '                       //
     .                     'because binary PCK files provided using ' //
     .                     'the ''#'' and/or ''#'' keys contain no '  //
     .                     'data for this frame.'

            ELSE

               COVDSC(1) = 'The first coverage window, for #, '       //
     .                     'was determined from '                     //
     .                     'binary PCK files provided using the '     //
     .                     '''#'' and/or ''#'' keys.'

            END IF

         END IF

C
C        Finish by making the information string by substituting either
C        the frame name and class ID and the kernel command line keys.
C
         CALL REPMC ( COVDSC(1), '#', 
     .                'the PCK frame ''#'' (class ID ''#'')', 
     .                                           COVDSC(1) )
         CALL REPMC ( COVDSC(1), '#', CFRNAM(1), COVDSC(1) )
         CALL REPMI ( COVDSC(1), '#', CFRCID(1), COVDSC(1) )
         CALL REPMC ( COVDSC(1), '#', KERKEY,    COVDSC(1) )
         CALL REPMC ( COVDSC(1), '#', KE1KEY,    COVDSC(1) )

      END IF

C
C     For the second coverage frame we will try to get coverage from
C     the second or only file or from the combination of kernels
C     applicable to both orientations and the second orientation.
C
C     Initialize the second coverage window and description string.
C
      CALL SSIZED ( WINSIZ, COVER2 )
      CALL SCARDD ( 0,      COVER2 )

      CALL SCARDD ( 0,      COVERA )
      CALL SCARDD ( 0,      COVERB )

      COVDSC(2) = ' '

      IF      ( ARCTYP(2) .EQ. 'DAF/CK'  .AND.
     .          CFRCLS(2) .EQ. 3         .AND.
     .          CFRFND(2)                      ) THEN

C
C        Get coverage for the second coverage frame from the second CK
C        file if
C
C           the second file is a CK,
C           the frame is a CK frame, and
C           data for frame's class ID is present in the file.
C
C        Since CKCOV call will need SCLK and LSK data to do time
C        conversions, clear loaded kernels and reload the second file
C        and all kernels applicable to the second attitude set.
C
C
C        Load appropriate kernels.
C
         CALL KCLEAR
         HLLINE = KERNLS(3)//' '//KERNLS(2)//' '//KERNAM(2)
         CALL LDKLST( HLLINE )

C
C        Get ET coverage window adjusted for round off, at interval
C        level, with zero tolerance, and angular rate flag set earlier.
C
         CALL CKCOVR( KERNAM(2), CFRCID(2), AVFLG, 'INTERVAL', 0.D0,
     .                COVER2 )

C
C        If we got an empty window back, report an error. If not,
C        make an information string that we may have to use in the
C        error message(s) later.
C
         IF ( WNCARD(COVER2) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because CK file ''#'' does '   //
     .                    'not provide any roundoff-adjusted '       //
     .                    'coverage for the CK frame with class ID ' //
     .                    '''#''.'                                   )
            CALL ERRCH  ( '#', KERNAM(2)                             )
            CALL ERRINT ( '#', CFRCID(2)                             )
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS6)'                     )

         ELSE

C
C           The frame name not being an integer means that we know this
C           frame. If so, include frame name and class ID in the
C           string. Otherwise, use only frame class ID.
C
            CALL NPARSI ( CFRNAM(2), HINT, ERROR, PTR )
            IF ( PTR .NE. 0 ) THEN
               COVDSC(2) = 'The second coverage window, for the CK ' //
     .                     'frame ''#'' (class '                     //
     .                     'ID ''#''), was determined from the CK ' //
     .                     'file ''#''.'
               CALL REPMC ( COVDSC(2), '#', CFRNAM(2), COVDSC(2) )
               CALL REPMI ( COVDSC(2), '#', CFRCID(2), COVDSC(2) )
               CALL REPMC ( COVDSC(2), '#', KERNAM(2), COVDSC(2) )
            ELSE
               COVDSC(2) = 'The second coverage window, for the CK ' //
     .                     'frame with class ID '                    //
     .                     '''#'', was determined from the CK file ' //
     .                     '''#''.'
               CALL REPMI ( COVDSC(2), '#', CFRCID(2), COVDSC(2) )
               CALL REPMC ( COVDSC(2), '#', KERNAM(2), COVDSC(2) )
            END IF

         END IF


      ELSE IF ( ARCTYP(2) .EQ. 'DAF/PCK' .AND.
     .          CFRCLS(2) .EQ. 2         .AND.
     .          CFRFND(2)                      ) THEN

C
C        Get coverage for the second coverage frame from the second PCK
C        file if
C
C           the second file is is a PCK, 
C           the frame is a PCK frame, and 
C           data for frame's class ID is present in the file. 
C
C        No need to reload any kernels for PCK coverage look up.
C
C        Get coverage.
C
         CALL PCKCOV ( KERNAM(2), CFRCID(2), COVER2 )

C
C        I can't think of a way to get empty window out of the
C        previous call, but we will check for it anyway.
C
C        If we got an empty window back, report an error. If not,
C        make an information string that we may have to use in the
C        error message(s) later.
C
         IF ( WNCARD(COVER2) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because PCK file ''#'' does '  //
     .                    'not provide any coverage for the PCK '    //
     .                    'frame with class ID ''#''.'               )
            CALL ERRCH  ( '#', KERNAM(2)                             )
            CALL ERRINT ( '#', CFRCID(2)                             )
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS7)'                     )

         ELSE

C
C           The frame name not being an integer means that we know this
C           frame. If so, include frame name and class ID in the
C           string. Otherwise, use only frame class ID.
C
            CALL NPARSI ( CFRNAM(2), HINT, ERROR, PTR )
            IF ( PTR .NE. 0 ) THEN
               COVDSC(2) = 'The second coverage window, for the '    //
     .                     'PCK frame ''#'' (class '                 //
     .                     'ID ''#''), was determined from the PCK ' //
     .                     'file ''#''.'
               CALL REPMC ( COVDSC(2), '#', CFRNAM(2), COVDSC(2) )
               CALL REPMI ( COVDSC(2), '#', CFRCID(2), COVDSC(2) )
               CALL REPMC ( COVDSC(2), '#', KERNAM(2), COVDSC(2) )
            ELSE
               COVDSC(2) = 'The second coverage window, for the PCK '//
     .                     'frame with class '                       //
     .                     'ID  ''#'', was determined from the PCK ' //
     .                     'file ''#''.'
               CALL REPMI ( COVDSC(2), '#', CFRCID(2), COVDSC(2) )
               CALL REPMC ( COVDSC(2), '#', KERNAM(2), COVDSC(2) )
            END IF

         END IF


      ELSE IF ( ( KERNAM(2) .NE. ' '       .AND.
     .            .NOT. ISCPCK(2)          .AND.
     .            CFRCLS(2) .EQ. 3               ) .OR.
     .          ( KERNAM(2) .EQ. ' '       .AND.
     .            CFRCLS(2) .EQ. 3               )      ) THEN

C
C        Try to get coverage for the second coverage frame from all
C        CKs provided for both orientations and the second orientation
C        if
C
C           the second file is present,
C           the second file is not a CK or PCK, and 
C           the frame is a CK frame, 
C
C        OR if
C
C           the second file is not present,
C           the frame is a CK frame
C
C        We need to reload all kernels applicable the second orientation
C        to do this look up.
C
         CALL KCLEAR
         HLLINE = KERNLS(3)//' '//KERNLS(2)//' '//KERNAM(2)
         CALL LDKLST( HLLINE )

C
C        Get the total count of loaded CKs and, if the count is
C        non-zero, proceed to fetch coverage. 
C
C        Getting the zero count is not an error at this point as we may
C        still come up with a comparison window using the other
C        coverage frame and/or begin and end times specified on the
C        command line. Still we need to make an information string that
C        we may have to use in the error message(s) later.
C
         CALL KTOTAL ( 'CK', COUNT )

         IF ( COUNT .EQ. 0 ) THEN

            COVDSC(2) = 'The second coverage window, for #, could ' //
     .                  'not be determined '                        //
     .                  'because no CK files were provided using '  //
     .                  'the ''#'' and/or ''#'' keys. '
            
         ELSE

C
C           Loop through all CK files and collect coverage information
C           for this class ID.
C
            DO I = 1, COUNT 

C
C              Get the name of the next CK file.
C
               CALL KDATA ( I, 'CK', 
     .                      HKRNAM, HARTYP, HLINE2, HANDLE, FOUND )

               IF ( .NOT. FOUND ) THEN
                  CALL SETMSG ( 'There is a bug in the program. ' //
     .                          'Please, contact NAIF.'           )
                  CALL SIGERR ( 'SPICE(FRMDIFFBUG5)'              )
               END IF

C
C              Reset auxiliary coverage window and get ET coverage
C              window adjusted for round off, at interval level, with
C              zero tolerance, and angular rate flag set earlier for
C              this CK.
C
               CALL SCARDD ( 0, COVERA )

               CALL CKCOVR ( HKRNAM, CFRCID(2), AVFLG, 'INTERVAL', 0.D0,
     .                       COVERA )

C
C              Merge auxiliary coverage with the total coverage for
C              this frame for all CKs.
C
               CALL WNUNID ( COVER2, COVERA, COVERB )
               CALL COPYD  ( COVERB, COVER2 )

            END DO

C
C           Getting an empty window back is still not an error at this
C           point. Whether it is empty or not, we need to make an
C           information string that we may have to use in the error
C           message(s) later.
C
            IF ( WNCARD(COVER2) .EQ. 0 ) THEN

               COVDSC(2) = 'The second coverage window, for #, could '//
     .                     'not be determined '                       //
     .                     'because the CK files provided using '     //
     .                     'the ''#'' and/or ''#'' keys contain no '  //
     .                     'data for this frame.'

            ELSE

               COVDSC(2) = 'The second coverage window, for #, '      //
     .                     'was determined from the CK '              //
     .                     'files provided using the ''#'' and/or '   //
     .                     '''#'' keys.'

            END IF

         END IF

C
C        Finish by making the information string by substituting either
C        the frame name and class ID and the kernel command line keys.
C
         CALL REPMC ( COVDSC(2), '#', 
     .                'the CK frame ''#'' (class ID ''#'')', 
     .                                           COVDSC(2) )
         CALL REPMC ( COVDSC(2), '#', CFRNAM(2), COVDSC(2) )
         CALL REPMI ( COVDSC(2), '#', CFRCID(2), COVDSC(2) )
         CALL REPMC ( COVDSC(2), '#', KERKEY,    COVDSC(2) )
         CALL REPMC ( COVDSC(2), '#', KE2KEY,    COVDSC(2) )


      ELSE IF ( ( KERNAM(2) .NE. ' '       .AND.
     .            .NOT. ISCPCK(2)          .AND.
     .            CFRCLS(2) .EQ. 2               ) .OR.
     .          ( KERNAM(2) .EQ. ' '       .AND.
     .            CFRCLS(2) .EQ. 2               )      ) THEN

C
C        Try to get coverage for the second coverage frame from all
C        PCKs provided for both orientations and the second orientation
C        if
C
C           the second file is present,
C           the second file is not a CK or PCK, and 
C           the frame is a PCK frame, 
C
C        OR if
C
C           the second file is not present,
C           the frame is a PCK frame
C
C        We need to reload all kernels applicable the second orientation
C        to do this look up.
C
         CALL KCLEAR
         HLLINE = KERNLS(3)//' '//KERNLS(2)//' '//KERNAM(2)
         CALL LDKLST( HLLINE )

C
C        Get the total count of loaded binary PCKs and, if the count is
C        non-zero, proceed to fetch coverage.
C
C        Getting the zero count is not an error at this point as we may
C        still come up with a comparison window using the other
C        coverage frame and/or begin and end times specified on the
C        command line. Still we need to make an information string that
C        we may have to use in the error message(s) later.
C
         CALL KTOTAL ( 'PCK', COUNT )

         IF ( COUNT .EQ. 0 ) THEN

            COVDSC(2) = 'The second coverage window, for #, could ' //
     .                  'not be determined '                        //
     .                  'because no binary PCK files were '         //
     .                  'provided using the ''#'' and/or ''#'' keys. '
            
         ELSE

C
C           Loop through all binary PCK files and collect coverage
C           information for this class ID.
C
            DO I = 1, COUNT 

C
C              Get the name of the next binary PCK file.
C
               CALL KDATA ( I, 'PCK', 
     .                      HKRNAM, HARTYP, HLINE2, HANDLE, FOUND )

               IF ( .NOT. FOUND ) THEN
                  CALL SETMSG ( 'There is a bug in the program. ' //
     .                          'Please, contact NAIF.'           )
                  CALL SIGERR ( 'SPICE(FRMDIFFBUG6)'              )
               END IF

C
C              Add coverage for this frame from this binary PCK to
C              the total coverage from all binary PCKs
C
               CALL PCKCOV ( HKRNAM, CFRCID(2), COVER2 )

            END DO

C
C           Getting an empty window back is still not an error at this
C           point. Whether it is empty or not, we need to make an
C           information string that we may have to use in the error
C           message(s) later.
C
            IF ( WNCARD(COVER2) .EQ. 0 ) THEN

               COVDSC(2) = 'The second coverage window, for #, could '//
     .                     'not be determined '                       //
     .                     'because binary PCK files provided using ' //
     .                     'the ''#'' and/or ''#'' keys contain no '  //
     .                     'data for this frame.'

            ELSE

               COVDSC(2) = 'The second coverage window, for #, '      //
     .                     'was determined from '                     //
     .                     'binary PCK files provided using the '     //
     .                     '''#'' and/or ''#'' keys.'

            END IF

         END IF

C
C        Finish by making the information string by substituting either
C        the frame name and class ID and the kernel command line keys.
C
         CALL REPMC ( COVDSC(2), '#', 
     .                'the PCK frame ''#'' (class ID ''#'')', 
     .                                           COVDSC(2) )
         CALL REPMC ( COVDSC(2), '#', CFRNAM(2), COVDSC(2) )
         CALL REPMI ( COVDSC(2), '#', CFRCID(2), COVDSC(2) )
         CALL REPMC ( COVDSC(2), '#', KERKEY,    COVDSC(2) )
         CALL REPMC ( COVDSC(2), '#', KE2KEY,    COVDSC(2) )

      END IF

C
C     Check that either both begin and end time are specified or that
C     at least one of the coverage windows is not empty. If not, report
C     a bug because at this point we should have had enough information
C     to determine final comparison window. (At the moment I can think
C     only these cases when this can happen: either or both ``to''
C     frames were were given and are non-CK/PCK frames or are CK/PCK
C     frames but with no data in the files.)
C
      IF (   WNCARD(COVER1) .EQ. 0           .AND.
     .       WNCARD(COVER2) .EQ. 0           .AND.
     .     ( TIME(1)   .EQ. ' '      .OR.
     .       TIME(2)   .EQ. ' '            )       ) THEN

         CALL SETMSG ( 'Insufficient information was provided '      //
     .                 'on the command line to determine time '      //
     .                 'range for comparison. If the program is '    //
     .                 'run with kernel sets that include CKs '      //
     .                 'and/or binary PCKs, the ''#'' and/or '       //
     .                 '''#'' keys can be used to direct the '       //
     .                 'program to determine default coverage '      //
     .                 'using data from the files. Otherwise, '      //
     .                 'both of the ''#'' and ''#'' keys must '      //
     .                 'be used to specify the start and stop '      //
     .                 'times of the comparison window.'             )
         CALL ERRCH  ( '#', CF1KEY                                   )
         CALL ERRCH  ( '#', CF2KEY                                   )
         CALL ERRCH  ( '#', BEGKEY                                   )
         CALL ERRCH  ( '#', ENDKEY                                   )
         CALL SIGERR ( 'SPICE(NOTIMEBOUNDS8)'                        )

      END IF

C
C     Wrap up determining the final comparison window by intersecting
C     the two coverage windows and the window made using begin and end
C     time given on the command line. We will do it in a rather long
C     "if" block instead of a simple way -- setting undefined windows
C     to (DPMIN,DPMAX), intersecting all three of them, and reporting
C     error if there is no intersection -- because we want report more
C     meaningful errors. First, make a constraint window using start
C     and stop time provided on the command line. Note that ET(1) is
C     either a real time provided on the command line or DPMIN while
C     ET(2) is either a real time provided on the command line or
C     DPMAX.
C
      CALL SSIZED ( 2, COVERC )
      CALL SCARDD ( 0, COVERC )

      CALL WNINSD ( ET(1), ET(2), COVERC )

C
C     Now check if we have determined both, one, or no coverage windows
C     from the file(s).
C
      IF      ( WNCARD(COVER1) .NE. 0  .AND.
     .          WNCARD(COVER2) .NE. 0        ) THEN

C
C        Both coverage windows are not empty. Intersect them and report
C        an error if intersection is empty.
C
         CALL WNINTD( COVER1, COVER2, CMPWIN )

         IF ( WNCARD(CMPWIN) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because coverage windows '     //
     .                    'obtained by examining kernels provided '  //
     .                    'on the command line do not overlap. # #'  )
            CALL ERRCH  ( '#', COVDSC(1)                             )
            IF ( COVDSC(1) .NE. COVDSC(2) ) THEN
               CALL ERRCH  ( '#', COVDSC(2)                          )
            ELSE
               CALL ERRCH  ( '#', ' '                                )
            END IF
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS9)'                     )

         END IF

C
C        Intersect result window with the start/stop window and place
C        it in a temporary window. Report an error if intersection is
C        empty.
C
         CALL WNINTD( CMPWIN, COVERC, COVER1 )

         IF ( WNCARD(COVER1) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because the intersection of '  //
     .                    'the coverage windows obtained by '        //
     .                    'examining kernels provided on the '       //
     .                    'command line does not overlap the time '  //
     .                    'range specified in the command line '     //
     .                    'using ''#'' and/or ''#'' keys. # # # '    )
            CALL ERRCH  ( '#', BEGKEY                                )
            CALL ERRCH  ( '#', ENDKEY                                )
            CALL ERRCH  ( '#', COVDSC(1)                             )
            IF ( COVDSC(1) .NE. COVDSC(2) ) THEN
               CALL ERRCH  ( '#', COVDSC(2)                          )
            ELSE
               CALL ERRCH  ( '#', ' '                                )
            END IF
            CALL ERRCH  ( '#', TIMDSC                                )
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS10)'                    )

         END IF

C
C        Copy temporary window to final window.
C
         CALL COPYD  ( COVER1, CMPWIN )

      ELSE IF ( WNCARD(COVER1) .NE. 0 ) THEN

C
C        Only first coverage window is not empty. Intersect it with the
C        start/stop window. Report an error if intersection is empty.
C
         CALL WNINTD( COVER1, COVERC, CMPWIN )

         IF ( WNCARD(CMPWIN) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because the coverage window '  //
     .                    'obtained by examining the kernels '       //
     .                    'provided on the command line does not '   //
     .                    'overlap the time range specified in '     //
     .                    'the command line using ''#'' and/or '     //
     .                    '''#'' keys. # # #'                        )
            CALL ERRCH  ( '#', BEGKEY                                )
            CALL ERRCH  ( '#', ENDKEY                                )
            CALL ERRCH  ( '#', COVDSC(1)                             )
            CALL ERRCH  ( '#', COVDSC(2)                             )
            CALL ERRCH  ( '#', TIMDSC                                )
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS11)'                    )

         END IF

      ELSE IF ( WNCARD(COVER2) .NE. 0 ) THEN

C
C        Only second coverage window is not empty. Intersect it with
C        the start/stop window. Report an error if intersection is
C        empty.
C
         CALL WNINTD( COVER2, COVERC, CMPWIN )

         IF ( WNCARD(CMPWIN) .EQ. 0 ) THEN

            CALL SETMSG ( 'Cannot determine time range for '         //
     .                    'comparison because the coverage window '  //
     .                    'obtained by examining the kernels '       //
     .                    'provided on the command line does not '   //
     .                    'overlap the time range specified in '     //
     .                    'the command line using ''#'' and/or '     //
     .                    '''#'' keys. # # #'                        )
            CALL ERRCH  ( '#', BEGKEY                                )
            CALL ERRCH  ( '#', ENDKEY                                )
            CALL ERRCH  ( '#', COVDSC(1)                             )
            CALL ERRCH  ( '#', COVDSC(2)                             )
            CALL ERRCH  ( '#', TIMDSC                                )
            CALL SIGERR ( 'SPICE(NOTIMEBOUNDS12)'                    )

         END IF

      ELSE

C
C        Both coverage windows are empty. Copy the window set using
C        start and stop time to final comparison window.
C
         CALL COPYD  ( COVERC, CMPWIN )

C
C        Sanity check. If either start or stop time is blank, something
C        is wrong with the code determining coverages. Report a bug.
C
         IF ( TIME(1) .EQ. ' ' .OR. TIME(2) .EQ. ' ' ) THEN
            CALL SETMSG ( 'There is a bug in the program. Please, '  //
     .                    'contact NAIF.'                            )
            CALL SIGERR ( 'SPICE(FRMDIFFBUG1)'                       )
         END IF

      END IF

C
C     Check command line for the rest of arguments. Unlike the
C     attributes of the first and second attitude sets and comparison
C     window, the rest of argument are truly optional.
C
C     Start with the type of output. Default type is basic.
C
      DIFTYP = BASVAL

      I = ISRCHC( TYPKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

         DIFTYP = CLVALS(I)

         IF ( .NOT. ( EQSTR( DIFTYP, BASVAL ) .OR.
     .                EQSTR( DIFTYP, STSVAL ) .OR.
     .                EQSTR( DIFTYP, DMVAL  ) .OR.
     .                EQSTR( DIFTYP, DQSVAL ) .OR.
     .                EQSTR( DIFTYP, DQOVAL ) .OR.
     .                EQSTR( DIFTYP, DEAVAL ) .OR.
     .                EQSTR( DIFTYP, DAAVAL ) .OR.
     .                EQSTR( DIFTYP, DCVAL  ) .OR.
     .                EQSTR( DIFTYP, DGVAL  )      ) ) THEN

            CALL SETMSG ( 'Output type ''#'' specified after '       //
     .                    '''#'' key is not recognized. Recognized ' //
     .                    'output types are ''#'', ''#'', ''#'', '   //
     .                    '''#'', ''#'', ''#'', ''#'', ''#'', and '  //
     .                    '''#''.'   )
            CALL ERRCH  ( '#', CLVALS(I)                             )
            CALL ERRCH  ( '#', TYPKEY                                )
            CALL ERRCH  ( '#', BASVAL                                )
            CALL ERRCH  ( '#', STSVAL                                )
            CALL ERRCH  ( '#', DAAVAL                                )
            CALL ERRCH  ( '#', DMVAL                                 )
            CALL ERRCH  ( '#', DQSVAL                                )
            CALL ERRCH  ( '#', DQOVAL                                )
            CALL ERRCH  ( '#', DEAVAL                                )
            CALL ERRCH  ( '#', DCVAL                                 )
            CALL ERRCH  ( '#', DGVAL                                 )
            CALL SIGERR ( 'SPICE(BADOUTPUTTYPE)'                     )

         END IF

      END IF

C
C     Next, get time step or number of steps. If both are specified,
C     time step has higher priority and, for this reason, should be
C     processed first. Default step is zero (meaning
C     "not set") and default number of points is a parameter set
C     in the include file.
C
      STEP = 0.D0
      NITR = DEFITR

      I = ISRCHC( STPKEY, MAXKEY, CLKEYS )

      IF ( CLFLAG(I) ) THEN

C
C        Is the step a DP number?
C
         CALL NPARSD ( CLVALS(I), STEP, ERROR, PTR )

         IF ( PTR .EQ. 0 ) THEN

C
C           Check that step is a positive number and is greater that
C           the smallest step we can allow.
C
            IF ( STEP .LT. STPTOL ) THEN
               CALL SETMSG ( 'Time step ''#'' specified after '   //
     .                       '''#'' key is smaller than # '       //
     .                       'seconds.'                           )
               CALL ERRCH  ( '#', CLVALS(I)                       )
               CALL ERRCH  ( '#', STPKEY                          )
               CALL ERRDP  ( '#', STPTOL                          )
               CALL SIGERR ( 'SPICE(STEPTOOSMALL1)'               )
            END IF

C
C           Compute the number of steps that will be required to
C           step over the comparison window with this step. To do
C           that, we loop over the window and increment the total
C           step count by 2 (one interval's start and one for stop)
C           plus whatever number of steps fits within the interval.
C
            NITR = 0
            DO I = 1, WNCARD(CMPWIN)

C
C              Fetch endpoints of the next interval.
C
               CALL WNFETD ( CMPWIN, I, ET(1), ET(2) )

C
C              Add one step for start of the interval.
C
               NITR = NITR + 1

C
C              Add one step for each point between endpoints
C              up to the end of the interval minus padding.
C
               HDP2 = ET(2) - STEP * STPPAD

               COUNT = 1
               HDP1  = ET(1) + STEP * COUNT

               DO WHILE ( HDP1 .LT. HDP2 .AND. NITR .LT. MAXITR )

                  NITR = NITR + 1

                  COUNT = COUNT + 1 
                  HDP1  = ET(1) + STEP * COUNT

               END DO

C
C              If interval begin time is not equal to interval end
C              time add one step for the end of the interval. If we
C              terminated the loop with NITR equal to MAXITR this
C              will take us over the allowed maximum and will
C              trigger the error message that follows.
C
               IF ( ET(1) .NE. ET(2) ) THEN
                  NITR = NITR + 1
               END IF

            END DO

C
C           Check if this number of points fits into the buffer.
C
            IF ( NITR .GT. MAXITR ) THEN
               CALL SETMSG ( 'The number of points within the '   //
     .                       'comparison window determined using '//
     .                       'step of ''#'' seconds specified '   //
     .                       'after the command line key ''#'' '  //
     .                       'is greater than can fit into '      //
     .                       'program''s buffers (# epochs '      //
     .                       'maximum.) Increase the step or use '//
     .                       'command line keys ''#'' and ''#'' ' //
     .                       'to make the time window smaller in '//
     .                       'order to run the program. The '     //
     .                       'comparison window was #. # # #'     )
               CALL ERRDP  ( '#', STEP                            )
               CALL ERRCH  ( '#', STPKEY                          )
               CALL ERRINT ( '#', MAXITR                          )
               CALL ERRCH  ( '#', BEGKEY                          )
               CALL ERRCH  ( '#', ENDKEY                          )
               IF ( TIMDSC .NE. ' ' ) THEN
                  IF      ( COVDSC(1) .NE. ' ' .AND.
     .                      COVDSC(2) .NE. ' '        ) THEN
                     CALL ERRCH  ( '#', 'determined by applying ' //
     .                             'constraints specified on the '//
     .                             'command line to the '         //
     .                             'intersection of coverages '   //
     .                             'obtained from the files'      )
                  ELSE IF ( COVDSC(1) .NE. ' ' .OR.
     .                      COVDSC(2) .NE. ' '        ) THEN
                     CALL ERRCH  ( '#', 'determined by applying ' //
     .                             'constraints specified on the '//
     .                             'command line to the coverage '//
     .                             'obtained from the file'       )
                  ELSE
                     CALL ERRCH  ( '#', 'specified on the '       //
     .                             'command line'                 )
                  END IF
               ELSE
                  IF      ( COVDSC(1) .NE. ' ' .AND.
     .                      COVDSC(2) .NE. ' '        ) THEN
                     CALL ERRCH  ( '#', 'determined by '          //
     .                             'intersecting coverages '      //
     .                             'obtained from the files'      )
                  ELSE IF ( COVDSC(1) .NE. ' ' .OR.
     .                      COVDSC(2) .NE. ' '        ) THEN
                     CALL ERRCH  ( '#', 'determined from the file')
                  ELSE
C
C                    We can never hit this branch. Set replacement
C                    string to indicate a bug.
C
                     CALL ERRCH  ( '#', 'not determined properly '//
     .                             'due to a bug in the program. '//
     .                             'Please, contact NAIF.'        )
                  END IF
               END IF
               CALL ERRCH  ( '#', COVDSC(1)                       )
               IF ( COVDSC(1) .NE.  COVDSC(2) ) THEN
                  CALL ERRCH  ( '#', COVDSC(2)                    )
               ELSE
                  CALL ERRCH  ( '#', ' '                          )
               END IF
               CALL ERRCH  ( '#', TIMDSC                          )
               CALL SIGERR ( 'SPICE(STEPTOOSMALL2)'               )
            END IF

         ELSE
            CALL SETMSG ( 'Time step ''#'' specified after ''#'' '//
     .                    'key is not a DP number.'               )
            CALL ERRCH  ( '#', CLVALS(I)                          )
            CALL ERRCH  ( '#', STPKEY                             )
            CALL SIGERR ( 'SPICE(NOTANDPNUMBER)'                  )
         END IF

      ELSE

C
C        Step was not provided on the command line. What about the
C        number of steps?
C
         I = ISRCHC( NSTKEY, MAXKEY, CLKEYS )

         IF ( CLFLAG(I) ) THEN

C
C           Is the number of step an integer number?
C
            CALL NPARSI ( CLVALS(I), NITR, ERROR, PTR )

            IF ( PTR .EQ. 0 ) THEN

               IF ( NITR .LT. MINITR .OR. NITR .GT. MAXITR ) THEN
                  CALL SETMSG ( 'Number of points must be an '    //
     .                          'integer number between # and '   //
     .                          '#. The number of points '        //
     .                          'specified after the ''#''  key ' //
     .                          'was #.'                          )
                  CALL ERRINT ( '#', MINITR                       )
                  CALL ERRINT ( '#', MAXITR                       )
                  CALL ERRCH  ( '#', NSTKEY                       )
                  CALL ERRINT ( '#', NITR                         )
                  CALL SIGERR ( 'SPICE(BADNUMBEROFPOINTS)'        )
               END IF

            ELSE
               CALL SETMSG ( 'Number of points ''#'' specified '  //
     .                       'after ''#''  key is not an integer '//
     .                       'number.'                            )
               CALL ERRCH  ( '#', CLVALS(I)                       )
               CALL ERRCH  ( '#', NSTKEY                          )
               CALL SIGERR ( 'SPICE(NOTANINTEGERNUMBER)'          )
            END IF

         END IF

      END IF

C
C     If any of the dump reports was requested we need to check whether
C     output time format was provided on the command line. Default
C     format is ET seconds. Default SCLK ID is zero (means "not set").
C
      TIMFMT = ETVAL
      SCLKID(1) = 0
      SCLKID(2) = 0

      IF ( EQSTR( DIFTYP, DMVAL  ) .OR.
     .     EQSTR( DIFTYP, DQSVAL ) .OR.
     .     EQSTR( DIFTYP, DQOVAL ) .OR.
     .     EQSTR( DIFTYP, DEAVAL ) .OR.
     .     EQSTR( DIFTYP, DAAVAL ) .OR.
     .     EQSTR( DIFTYP, DCVAL  ) .OR.
     .     EQSTR( DIFTYP, DGVAL  )      ) THEN

         I = ISRCHC( FMTKEY, MAXKEY, CLKEYS )

         IF ( CLFLAG(I) ) THEN

            TIMFMT = CLVALS(I)

C
C           Time format should be either one of the recognized values
C           or a valid picture for TIMOUT.
C
            IF      ( EQSTR( TIMFMT, ETVAL ) ) THEN

C
C              It's ET seconds. No checks are needed.
C

            ELSE IF ( EQSTR( TIMFMT, SCSVAL ) .OR.
     .                EQSTR( TIMFMT, SCDVAL ) .OR.
     .                EQSTR( TIMFMT, SCTVAL )      ) THEN

C
C              It's one of the SCLK formats. Check if it applies, i.e.
C              if we can figure out the SCLK ID to use to convert to
C              these SCLKs. Set it based on the first CK frame in this
C              list: first ``to'' frame, first ``cov'' frame, second
C              ``to'' frame, second ``cov'' frame. If nether of these
C              frames is a CK frame, report an error. In each case,
C              we should load all applicable kernels to make sure that
C              CKMETA checks in the pool for information applicable to
C              the frame of interest.
C
               IF      ( TFRCLS(1) .EQ. 3 ) THEN

                  CALL KCLEAR
                  IF ( KERNAM(1) .NE. ' ' ) THEN
                     HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//KERNAM(1)
                     CALL LDKLST( HLLINE )
                  ELSE
                     HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//KERNAM(2)
                     CALL LDKLST( HLLINE )
                  END IF

                  CALL CKMETA( TFRCID(1), 'SCLK', SCLKID(1) )

               ELSE IF ( CFRCLS(1) .EQ. 3 ) THEN

                  CALL KCLEAR
                  IF ( KERNAM(1) .NE. ' ' ) THEN
                     HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//KERNAM(1)
                     CALL LDKLST( HLLINE )
                  ELSE
                     HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//KERNAM(2)
                     CALL LDKLST( HLLINE )
                  END IF

                  CALL CKMETA( CFRCID(1), 'SCLK', SCLKID(1) )

               ELSE IF ( TFRCLS(2) .EQ. 3 ) THEN

                  CALL KCLEAR
                  HLLINE = KERNLS(3)//' '//KERNLS(2)//' '//KERNAM(2)
                  CALL LDKLST( HLLINE )

                  CALL CKMETA( TFRCID(2), 'SCLK', SCLKID(2) )

               ELSE IF ( CFRCLS(2) .EQ. 3 ) THEN

                  CALL KCLEAR
                  HLLINE = KERNLS(3)//' '//KERNLS(2)//' '//KERNAM(2)
                  CALL LDKLST( HLLINE )

                  CALL CKMETA( CFRCID(2), 'SCLK', SCLKID(2) )

               ELSE

                  CALL SETMSG ( 'The ''#'' time format specified '   //
     .                          'after the ''#'' key cannot be '     //
     .                          'used when neither of the ''to'' '   //
     .                          'frames (''#'' and ''#'') # is '     //
     .                          'a CK-based (class 3) frame.'        )
                  CALL ERRCH  ( '#', CLVALS(I)                       )
                  CALL ERRCH  ( '#', FMTKEY                          )
                  CALL ERRCH  ( '#', TFRNAM(1)                       )
                  CALL ERRCH  ( '#', TFRNAM(2)                       )
                  IF      ( CFRNAM(1) .NE. ' ' .AND.
     .                      CFRNAM(2) .NE. ' '       ) THEN
                     HLINE2 = 'and coverage look up frames (PCK '    //
     .                        'frames with class IDs # and #)'
                     CALL REPMI ( HLINE2, '#', CFRCID(1), HLINE2 )
                     CALL REPMI ( HLINE2, '#', CFRCID(2), HLINE2 )
                     CALL ERRCH ( '#', HLINE2                        )
                  ELSE IF ( CFRNAM(1) .NE. ' '       ) THEN
                     HLINE2 = 'and coverage look up frame (PCK '     //
     .                        'frame with class ID #)'
                     CALL REPMI ( HLINE2, '#', CFRCID(1), HLINE2 )
                     CALL ERRCH ( '#', HLINE2                        )
                  ELSE IF ( CFRNAM(2) .NE. ' '       ) THEN
                     HLINE2 = 'and coverage look up frame (PCK '     //
     .                        'frame with class ID #)'
                     CALL REPMI ( HLINE2, '#', CFRCID(2), HLINE2 )
                     CALL ERRCH ( '#', HLINE2                        )
                  ELSE
                     CALL ERRCH ( '#', ' '                           )
                  END IF
                  CALL SIGERR ( 'SPICE(FORMATDOESNTAPPLY)'           )

               END IF

C
C              At this point one of the SCLKID elements should be
C              set. If not, there is problem in the logic upstream.
C              Put in a safety check indicating a bug.
C
               IF ( SCLKID(1) .EQ. 0 .AND. SCLKID(2) .EQ. 0 ) THEN
                  CALL SETMSG ( 'There is a bug in the program. '    //
     .                          'Please, contact NAIF.'              )
                  CALL SIGERR ( 'SPICE(FRMDIFFBUG2)'                 )
               END IF

            ELSE

C
C              This must be a TIMOUT format picture. We will check only
C              one thing about, that it is not blank.
C
               IF ( TIMFMT .EQ. ' ' ) THEN
                  CALL SETMSG ( 'The time format specified after '   //
     .                          'the ''#'' key was blank.'           )
                  CALL ERRCH  ( '#', FMTKEY                          )
                  CALL SIGERR ( 'SPICE(BLANKTIMEFORMAT)'             )
               END IF

C
C              The very elaborate check is below turned out to be
C              completely useless because TIMOUT allows pretty much
C              any string as a picture without complaints :(. This is
C              why it's commented out.
C
C              This must be a TIMOUT format picture. We will use the
C              following "sledgehammer" approach to verify that this is
C              a legitimate picture: 1) load all kernels, 2) set error
C              handing to RETURN and messages to NONE, 3) call TIMOUT
C              to get out for zero ET, 4) check FAILED to see if
C              conversion went well, 5a) if yes, clear errors and reset
C              error handing response and messages back to what they
C              were, or 5b) if not, retrieve format description error
C              and report it to the user.
C
C               CALL KCLEAR
C               HLLINE = KERNLS(3)//' '//KERNLS(1)//' '//KERNAM(1)//
C     .                             ' '//KERNLS(2)//' '//KERNAM(2)
C               CALL LDKLST( HLLINE )
C
C               CALL ERRACT ( 'GET', HLINE    )
C               CALL ERRACT ( 'SET', 'RETURN' )
C
C               CALL ERRPRT ( 'GET', HLINE2   )
C               CALL ERRPRT ( 'SET', 'NONE'   )
C
C               CALL TIMOUT ( 0.D0, TIMFMT, ERROR )
C
C               IF ( .NOT. FAILED() ) THEN
C
C                  CALL ERRACT ( 'SET', HLINE  )
C                  CALL ERRPRT ( 'SET', HLINE2 )
C
C               ELSE
C
C                  CALL GETMSG ( 'LONG', ERROR )
C                  CALL RESET  ( )
C                  CALL ERRACT ( 'SET', HLINE  )
C                  CALL ERRPRT ( 'SET', HLINE2 )
C
C                  CALL SETMSG ( 'The ''#'' time format specified '   //
C     .                          'after the ''#'' is neither one of ' //
C     .                          'the preset formats (''#'', ''#'', ' //
C     .                          '''#'', ''#'') nor a valid TIMOUT '  //
C     .                          'format specification string. The '  //
C     .                          'error from parsing this value '     //
C     .                          'as a TIMOUT format specification '  //
C     .                          'string was: #'                      )
C                  CALL ERRCH  ( '#', CLVALS(I)                       )
C                  CALL ERRCH  ( '#', FMTKEY                          )
C                  CALL ERRCH  ( '#', ETVAL                           )
C                  CALL ERRCH  ( '#', SCSVAL                          )
C                  CALL ERRCH  ( '#', SCDVAL                          )
C                  CALL ERRCH  ( '#', SCTVAL                          )
C                  CALL ERRCH  ( '#', ERROR                           )
C                  CALL SIGERR ( 'SPICE(BADFORMATSTRING)'             )
C
C
C               END IF

            END IF

         END IF

      END IF

C
C     Finally, if angle dump was requested, check whether the rotation
C     order and angle units were provided on the command line. Default
C     order is 3, 2, 1. Default units are radians.
C
      AXES(1) = 3
      AXES(2) = 2
      AXES(3) = 1
      AUNITS = 'radians'

      IF ( EQSTR( DIFTYP, DEAVAL ) .OR.
     .     EQSTR( DIFTYP, DAAVAL )      ) THEN

C
C        Check if rotation order was provided.
C
         I = ISRCHC( ANOKEY, MAXKEY, CLKEYS )

         IF ( CLFLAG(I) ) THEN

            HLINE = CLVALS(I)

C
C           Check if the order is a list of three space-delimited
C           items. If not, report an error.
C
            IF ( WDCNT( HLINE ) .EQ. 3 ) THEN

               DO I = 1, 3

C
C                 Check each of the items in the order and set
C                 corresponding element of axes array. If we can't
C                 recognize the token, report an error.
C
                  CALL NTHWD ( HLINE, I, HWORD, HINT )

                  IF      ( EQSTR( HWORD, XVAL ) ) THEN
                     AXES(I) = 1
                  ELSE IF ( EQSTR( HWORD, YVAL ) ) THEN
                     AXES(I) = 2
                  ELSE IF ( EQSTR( HWORD, ZVAL ) ) THEN
                     AXES(I) = 3
                  ELSE
                     CALL SETMSG ( 'Token ''#'' in the rotation '    //
     .                             'order  ''#'' specified after '   //
     .                             'after the ''#'' key is not one ' //
     .                             'of the recognized axis '         //
     .                             'identifiers (#, #, #).'          )
                     CALL ERRCH  ( '#', HWORD                        )
                     CALL ERRCH  ( '#', HLINE                        )
                     CALL ERRCH  ( '#', ANOKEY                       )
                     CALL ERRCH  ( '#', XVAL                         )
                     CALL ERRCH  ( '#', YVAL                         )
                     CALL ERRCH  ( '#', ZVAL                         )
                     CALL SIGERR ( 'SPICE(BADROTATIONORDER1)'        )
                  END IF

               END DO

C
C              Check that neighbor elements in the list are distinct.
C              If not, report an error.
C
               IF ( AXES(1) .EQ. AXES(2) .OR.
     .              AXES(2) .EQ. AXES(3)      ) THEN
                  CALL SETMSG ( 'Middle axis in the rotation order ' //
     .                          '''#'' specified after the ''#'' '   //
     .                          'key matches one of the neighbors.'  )
                  CALL ERRCH  ( '#', HLINE                           )
                  CALL ERRCH  ( '#', ANOKEY                          )
                  CALL SIGERR ( 'SPICE(BADROTATIONORDER2)'           )

               END IF

            ELSE
               CALL SETMSG ( 'The rotation order ''#'' specified '   //
     .                       'after the ''#'' key is not a set '     //
     .                       'of three space-delimited axes '        //
     .                       'identifiers (#, #, #).'                )
               CALL ERRCH  ( '#', HLINE                              )
               CALL ERRCH  ( '#', ANOKEY                             )
               CALL ERRCH  ( '#', XVAL                               )
               CALL ERRCH  ( '#', YVAL                               )
               CALL ERRCH  ( '#', ZVAL                               )
               CALL SIGERR ( 'SPICE(BADROTATIONORDER3)'              )
            END IF

         END IF

C
C        Check if angle units were provided.
C
         I = ISRCHC( ANUKEY, MAXKEY, CLKEYS )

         IF ( CLFLAG(I) ) THEN

            AUNITS =  CLVALS(I)

C
C           Check against the following set of units supported by
C           CONVRT (circa N0062). If input units don't match any of
C           these, report an error.
C
            IF ( .NOT. ( EQSTR( AUNITS, 'RADIANS'     ) .OR.
     .                   EQSTR( AUNITS, 'DEGREES'     ) .OR.
     .                   EQSTR( AUNITS, 'ARCMINUTES'  ) .OR.
     .                   EQSTR( AUNITS, 'ARCSECONDS'  ) .OR.
     .                   EQSTR( AUNITS, 'HOURANGLE'   ) .OR.
     .                   EQSTR( AUNITS, 'MINUTEANGLE' ) .OR.
     .                   EQSTR( AUNITS, 'SECONDANGLE' )      ) ) THEN

               CALL SETMSG ( 'The angle units ''#'' specified '      //
     .                       'after the ''#'' key are not '          //
     .                       'recognized. Refer to the program''s '  //
     .                       'User''s Guide for the list of '        //
     .                       'supported units.'                      )
               CALL ERRCH  ( '#', AUNITS                             )
               CALL ERRCH  ( '#', ANUKEY                             )
               CALL SIGERR ( 'SPICE(BADANGLEUNITS)'                  )

            END IF

         END IF

      END IF

C
C     If any of the dump reports was requested we need to check whether
C     the number significant digits for output was provided on the
C     command line.
C
      SIGDIG = DEFSDG

      IF ( EQSTR( DIFTYP, DMVAL  ) .OR. 
     .     EQSTR( DIFTYP, DQSVAL ) .OR. 
     .     EQSTR( DIFTYP, DQOVAL ) .OR. 
     .     EQSTR( DIFTYP, DEAVAL ) .OR. 
     .     EQSTR( DIFTYP, DAAVAL ) .OR. 
     .     EQSTR( DIFTYP, DCVAL  ) .OR. 
     .     EQSTR( DIFTYP, DGVAL  )      ) THEN

         I = ISRCHC( DIGKEY, MAXKEY, CLKEYS )

         IF ( CLFLAG(I) ) THEN

C
C           Is the number of digits an integer number?
C
            CALL NPARSI ( CLVALS(I), SIGDIG, ERROR, PTR )

            IF ( PTR .EQ. 0 ) THEN

               IF ( SIGDIG .LT. MINSDG .OR. SIGDIG .GT. MAXSDG ) THEN
                  CALL SETMSG ( 'The number of significant digits '  //
     .                          'specified using the ''#'' key '     //
     .                          'must be an integer number between ' //
     .                          '# and #. It was #.'                 )
                  CALL ERRCH  ( '#', DIGKEY                          )
                  CALL ERRINT ( '#', MINSDG                          )
                  CALL ERRINT ( '#', MAXSDG                          )
                  CALL ERRINT ( '#', SIGDIG                          )
                  CALL SIGERR ( 'SPICE(BADNOFDIGITS)'                )
               END IF

            ELSE 
               CALL SETMSG ( 'The number of significant digits '     //
     .                       '''#'' specified after ''#''  key is '  //
     .                       'not an integer number.'                )
               CALL ERRCH  ( '#', CLVALS(I)                          )
               CALL ERRCH  ( '#', DIGKEY                             )
               CALL SIGERR ( 'SPICE(NOTINTEGERNUMBER2)'              )
            END IF

         END IF

      END IF

C
C     Check out.
C
      CALL CHKOUT( 'CHWCML' )

      END
