C$Procedure PRSMET ( Parse metadata block )
 
      SUBROUTINE PRSMET ( INPUT,  OBJECT, FRAMEA, FRAMEB, 
     .                    TBEG,   TEND,   UBEG,   UEND,
     .                    TIMSYS, ATTREP, QLAST,  A2B,   
     .                    METHOD, DEGREE                 )
 
C$ Abstract
C
C     Parse an AEM metadata block. Return segment descriptor
C     information, plus the name of the time system associated
C     with the metadata block. Also return parameters required
C     to interpret data lines.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     AEM2CK
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'aem2ck.inc'

      CHARACTER*(*)         INPUT
      INTEGER               OBJECT
      CHARACTER*(*)         FRAMEA
      CHARACTER*(*)         FRAMEB
      DOUBLE PRECISION      TBEG
      DOUBLE PRECISION      TEND
      DOUBLE PRECISION      UBEG
      DOUBLE PRECISION      UEND
      CHARACTER*(*)         TIMSYS
      CHARACTER*(*)         ATTREP
      LOGICAL               QLAST
      LOGICAL               A2B
      CHARACTER*(*)         METHOD
      INTEGER               DEGREE
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     INPUT      I   Name of AEM file.
C     OBJECT     O   NAIF ID of ephemeris object
C     FRAMEA     O   First reference frame name.
C     FRAMEB     O   Second reference frame name.
C     TBEG       O   Block begin time.
C     TEND       O   Block end time.
C     UBEG       O   Block usable begin time.
C     UEND       O   Block usable end time.
C     TIMSYS     O   Time system name.
C     ATTREP     O   Attitude representation.
C     QLAST      O   Flag indicating whether quaternion scalar is last.
C     A2B        O   Flag indicating whether rotation is from A to B.
C     METHOD     O   Interpolation method.
C     DEGREE     O   Interpolation degree.
C
C$ Detailed_Input
C           
C     INPUT          is the name of an OEM file from which a 
C                    metadata block to be read.
C          
C$ Detailed_Output
C
C     OBJECT         is the NAIF integer ID code of the ephemeris
C                    object to which the data following the current
C                    metadata block correspond. This object is normally
C                    the center of one of the output reference frames.
C
C     FRAMEA,      
C     FRAMEB         are the names of the two reference frames related
C                    by the attitude data. The data define a rotation
C                    that maps vectors from frame A to frame B or 
C                    vice versa depending on the value of the flag
C                    A2B.
C
C     TBEG,
C     TEND           are, respectively, the start and stop times of the
C                    data block. The times are expressed as seconds
C                    past J2000 TDB.
C
C     UBEG,
C     UEND           are, respectively, the usable start and stop times
C                    of the data block. The times are expressed as
C                    seconds past J2000 TDB.
C
C                    UBEG is greater than or equal to the epoch of the
C                    first data line of the contiguous set of data
C                    lines following the metadata block.
C
C                    UEND is less than or equal to the epoch of the
C                    last data line of the contiguous set of data lines
C                    following the metadata block.
C
C     TIMSYS         is the name of the time system associated with
C                    the metadata block.
C
C     ATTREP         is the attitude representation. Possible values
C                    are
C
C                       QUATERNION
C                       QUATERNION/DERIVATIVE
C                       QUATERNION/RATE
C                       EULER_ANGLE
C                       EULER_ANGLE/RATE
C                       SPIN
C                       SPIN/NUTATION
C
C                    NOTE: the current implementation of this routine
C                    supports ONLY the QUATERNION representation.
C
C
C     QLAST          is a flag that is applicable only if the attitude
C                    representation is one of the QUATERNION types.
C                    Given that it is, QLAST is true if and only if 
C                    the quaternion scalar component is the last element
C                    of the quaternion.
C
C     A2B            is a flag that is true if and only if the mapping
C                    is from frame A to frame B.
C
C     METHOD         is the name of the interpolation method, if
C                    one was supplied in the metadata block.  
C                    Otherwise METHOD is returned blank.
C
C     DEGREE         is the degree of the interpolation method, if
C                    a method was supplied in the metadata block.  
C                    Otherwise DEGREE is set to zero.
C             
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) If a required metadata keyword is not found, the error
C        SPICE(MISSINGITEM) is signaled.
C
C     2) If either the object or center names cannot be mapped to
C        a NAIF integer ID code, the error SPICE(IDCODENOTFOUND)
C        is signaled.
C
C     3) If any metadata line does not follow the expected 
C        "keyword = value" syntax, the error will be diagnosed
C        by routines called by this routine.
C
C     4) If any time bounds given by the metadata block (either
C        start and stop times, or usable start and stop times)
C        cannot be parsed and converted to TDB, the error will
C        be diagnosed by routines called by this routine.
C
C     5) If either reference frame name cannot be mapped to a NAIF
C        integer code, the error SPICE(INVALIDREFFRAME) will be 
C        signaled.
C
C     6) If the usable time bounds are not contained in the
C        range defined by the metadata block time bounds, the 
C        error SPICE(TIMESOUTOFRANGE) will be signaled.
C
C     7) If the attitude representation is not 'QUATERNION', the
C        error SPICE(NOTSUPPORTED) will be signaled.
C
C     8) If the attitude direction is not either 'A2B' or 'B2A',
C        the error SPICE(NOTSUPPORTED) will be signaled.
C
C     9) If the quaternion type is not either 'FIRST' or 'LAST',
C        the error SPICE(NOTSUPPORTED) will be signaled.
C
C
C$ Files
C
C     See the description of INPUT.
C
C$ Particulars
C
C     This routine assumes that the first non-blank, non-comment
C     line it reads belongs to a metadata block.  The last line
C     this routine will normally read is a metadata block end 
C     marker.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     1) This routine assumes that any time tag used in an AEM file
C        can be parsed by the SPICE routine TPARTV.
C
C$ Literature_References
C
C     [1]  CCSDS Attitude Data Messages Blue Book, version CCSDS 
C          504.0-B-1, May, 2008.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    AEM2CK Version 1.0.0, 19-AUG-2015 (NJB)
C
C-&


C
C     SPICELIB functions
C
      INTEGER               CARDC
      INTEGER               ISRCHC

      LOGICAL               ELEMC
      LOGICAL               FAILED
      LOGICAL               MATCHI
      LOGICAL               RETURN
      
C
C     Local parameters
C
      INTEGER               NREQ
      PARAMETER           ( NREQ   = 8 )

      INTEGER               NKEYS
      PARAMETER           ( NKEYS  = 13 )

C
C     Indices of saved items
C
      INTEGER               ADRIDX
      PARAMETER           ( ADRIDX = 1 )

      INTEGER               ATPIDX
      PARAMETER           ( ATPIDX = ADRIDX + 1 )

      INTEGER               DEGIDX
      PARAMETER           ( DEGIDX = ATPIDX + 1 )

      INTEGER               METIDX
      PARAMETER           ( METIDX = DEGIDX + 1 )
      
      INTEGER               OBJIDX
      PARAMETER           ( OBJIDX = METIDX + 1 )
      
      INTEGER               QTPIDX
      PARAMETER           ( QTPIDX = OBJIDX + 1 )

      INTEGER               RFAIDX
      PARAMETER           ( RFAIDX = QTPIDX  + 1 )

      INTEGER               RFBIDX
      PARAMETER           ( RFBIDX = RFAIDX + 1 )
      
      INTEGER               TBIDX
      PARAMETER           ( TBIDX  = RFBIDX + 1 )

      INTEGER               TEIDX
      PARAMETER           ( TEIDX  = TBIDX + 1 )

      INTEGER               TSYIDX
      PARAMETER           ( TSYIDX = TEIDX + 1 )

      INTEGER               UBIDX
      PARAMETER           ( UBIDX  = TSYIDX  + 1 )

      INTEGER               UEIDX
      PARAMETER           ( UEIDX  = UBIDX  + 1 )

C
C     Other parameters
C

C
C     Local variables
C
      CHARACTER*(FRNMLN)    FRAME
      CHARACTER*(LNSIZE)    ITEMS   ( LBCELL : NKEYS )
      CHARACTER*(LNSIZE)    KEYWRD
      CHARACTER*(LNSIZE)    KEYS    ( NKEYS )
      CHARACTER*(LNSIZE)    KLINES  ( NKEYS )
      CHARACTER*(LNSIZE)    LINE
      CHARACTER*(LNSIZE)    RITEMS  ( LBCELL : NREQ )
      CHARACTER*(LNSIZE)    RKEYS   ( NREQ )
      CHARACTER*(LNSIZE)    TYPE
      CHARACTER*(LNSIZE)    VALUE
      CHARACTER*(LNSIZE)    VSTRS   ( NKEYS )
      
      INTEGER               I
      INTEGER               KNUMS   ( NKEYS )
      INTEGER               LOC
      INTEGER               LOC2
      INTEGER               LNUM
      INTEGER               REFCOD
      

      LOGICAL               EOF
      LOGICAL               FOUND

C
C     Saved variables
C
      SAVE                  KEYS
      SAVE                  KLINES
      SAVE                  KNUMS
      SAVE                  RKEYS
      SAVE                  VSTRS

C
C     Initial values
C
      DATA                  KEYS     /  'ATTITUDE_DIR',
     .                                  'ATTITUDE_TYPE',
     .                                  'INTERPOLATION_DEGREE',
     .                                  'INTERPOLATION_METHOD',
     .                                  'OBJECT_NAME',
     .                                  'QUATERNION_TYPE',
     .                                  'REF_FRAME_A',
     .                                  'REF_FRAME_B',
     .                                  'START_TIME',
     .                                  'STOP_TIME',
     .                                  'TIME_SYSTEM',
     .                                  'USEABLE_START_TIME',
     .                                  'USEABLE_STOP_TIME'
     .                               /
      
      DATA                  RKEYS    /  'ATTITUDE_DIR',
     .                                  'ATTITUDE_TYPE',
     .                                  'OBJECT_NAME',
     .                                  'REF_FRAME_A',
     .                                  'REF_FRAME_B',
     .                                  'START_TIME',
     .                                  'STOP_TIME',
     .                                  'TIME_SYSTEM'
     .                               /

 

      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'PRSMET' )

C
C     Initialize the sets of metadata items seen so far in
C     the current metadata block.  The set ITEMS contains all
C     keywords seen; the set RITEMS contains the required 
C     keywords seen.
C
      CALL SSIZEC ( NKEYS, ITEMS  )
      CALL SSIZEC ( NREQ,  RITEMS )

C
C     Read the first line of the metadata block. We expect this
C     line to have type 'METADATA_LINE'.
C
      CALL GETNCL ( INPUT, LINE, LNUM, TYPE, EOF )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'PRSMET' )
         RETURN
      END IF

      
      IF ( EOF ) THEN

         CALL SETMSG ( 'Encountered unexpected end of file. '
     .   //            'A metadata line was expected.'        )
         CALL SIGERR ( 'SPICE(UNEXPECTEDEOF)'                 )
         CALL CHKOUT ( 'PRSMET'                               )
         RETURN

      ELSE IF ( TYPE .NE. METLIN ) THEN

         CALL SETMSG ( 'Encountered unexpected input line type: # '
     .   //            'at line number #. A metadata line was '
     .   //            'expected. Beginning of line was #'         )
         CALL ERRCH  ( '#', TYPE                                   )
         CALL ERRINT ( '#', LNUM                                   )
         CALL ERRCH  ( '#', LINE                                   )
         CALL SIGERR ( 'SPICE(BADAEMFORMAT)'                       )
         CALL CHKOUT ( 'PRSMET'                                    )
         RETURN

      END IF


      DO WHILE (       ( .NOT. EOF        )
     .           .AND. ( .NOT. FAILED()   ) 
     .           .AND. ( TYPE .EQ. METLIN ) )
C
C        We expect the current line to conform to the "keyword = value"
C        syntax.
C
         CALL KEYVLC ( INPUT, LINE, LNUM, KEYWRD, VALUE )
         
         IF ( FAILED() ) THEN
            CALL CHKOUT ( 'PRSMET' )
            RETURN
         END IF

C
C        Map both keyword and value to their standardized forms.
C
         CALL CMAP  ( KEYWRD, KEYWRD )
         CALL LJUST ( KEYWRD, KEYWRD )
         CALL UCASE ( KEYWRD, KEYWRD )

         CALL CMAP  ( VALUE,  VALUE  )
         CALL LJUST ( VALUE,  VALUE  )
         CALL UCASE ( VALUE,  VALUE  )

C
C        Replace with blanks any underscores found in the value string.
C
C
C        DANGEROUS! Use string mapping to accomplish this.
C
C         CALL REPLCH ( VALUE, '_', ' ', VALUE )

C
C        Save the value, line, and line number associated with 
C        the current keyword, if the keyword is recognized.  
C
         LOC = ISRCHC ( KEYWRD, NKEYS, KEYS )

         IF ( LOC .GT. 0 ) THEN
C
C           Store the RHS value.
C
            VSTRS(LOC) = VALUE

C
C           Mark the current keyword as seen.           
C
            CALL INSRTC ( KEYWRD, ITEMS )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'PRSMET' )
               RETURN
            END IF

C
C           If the keyword is required, add it to the set
C           of required keywords seen.
C
            LOC2 = ISRCHC ( KEYWRD, NREQ, RKEYS )

            IF ( LOC2 .GT. 0 ) THEN

               CALL INSRTC ( KEYWRD, RITEMS )

               IF ( FAILED() ) THEN
                  CALL CHKOUT ( 'PRSMET' )
                  RETURN
               END IF

            END IF

C
C           Save the source line and line number for this keyword.
C           These will be needed if a semantic error is detected
C           later.
C
            KLINES(LOC) = LINE
            KNUMS(LOC)  = LNUM
    
         END IF

         CALL GETNCL ( INPUT, LINE, LNUM, TYPE, EOF )

      END DO

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'PRSMET' )
         RETURN
      END IF

C
C     Is the set of required items full?
C
      IF ( CARDC(RITEMS) .LT. NREQ ) THEN

         CALL SETMSG ( 'Not all required metadata items were seen ' 
     .   //            'in the metadata block ending at line #* in '
     .   //            'file <*>.  Required items are:  '           
     .   //            'ATTITUDE_DIR, ATTITUDE_TYPE, '      
     .   //            'OBJECT_NAME, REF_FRAME_A, REF_FRAME_B, '
     .   //            'START_TIME, STOP_TIME, TIME_SYSTEM.'        )
         CALL ERRINT ( '*', LNUM                                    )
         CALL ERRCH  ( '*', INPUT                                   ) 
         CALL SIGERR ( 'SPICE(MISSINGITEM)'                         )
         CALL CHKOUT ( 'PRSMET'                                     )
         RETURN

      END IF

C
C     The interpolation degree keyword is required only if the
C     interpolation keyword was present.
C
      IF (  ELEMC( KEYS(METIDX), ITEMS )  ) THEN
C
C        Set the interpolation method and check for presence of
C        the degree.
C
         METHOD = VSTRS(METIDX)

         IF (  ELEMC( KEYS(DEGIDX), ITEMS )  ) THEN
C
C           Parse the interpolation degree.
C
            CALL PRSINT (  VSTRS(DEGIDX),  DEGREE  )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'PRSMET' )
               RETURN
            END IF

         ELSE
C
C           The required degree is not present.
C
            CALL SETMSG ( 'No interpolation degree was seen '   //
     .                    'in the metadata block ending at '    //
     .                    'line #* in file <*>.  The degree '   //
     .                    'is required when the interpolation ' //
     .                    'method is specified, as it is here.' )
            CALL ERRINT ( '*', LNUM                             )
            CALL ERRCH  ( '*', INPUT                            ) 
            CALL SIGERR ( 'SPICE(MISSINGITEM)'                  )
            CALL CHKOUT ( 'PRSMET'                              )
            RETURN

         END IF

      ELSE
C
C        The interpolation method was not specified.
C
         METHOD = ' '
         DEGREE = 0

      END IF

C
C     Map object name to NAIF ID code.
C
      CALL BODN2C ( VSTRS(OBJIDX), OBJECT, FOUND )

      IF ( .NOT. FOUND ) THEN

         CALL SETMSG ( 'Object * specified in metadata block '     //
     .                 'at line #* in file <*> could not be '      //
     .                 'mapped to a NAIF ID code.  Line was *'      )
         CALL ERRCH  ( '*',  VSTRS(OBJIDX)                          ) 
         CALL ERRINT ( '*',  KNUMS(OBJIDX)                          )
         CALL ERRCH  ( '*',  INPUT                                  ) 
         CALL ERRCH  ( '*',  KLINES(OBJIDX)                         ) 
         CALL SIGERR ( 'SPICE(IDCODENOTFOUND)'                      )
         CALL CHKOUT ( 'PRSMET'                                     )
         RETURN

      END IF

C
C     Convert time bounds to ET.  
C
      TIMSYS  =  VSTRS(TSYIDX)

      CALL TIMCVT ( INPUT,   KLINES(TBIDX),  KNUMS(TBIDX), 
     .              TIMSYS,  VSTRS(TBIDX),   TBEG          )


      CALL TIMCVT ( INPUT,   KLINES(TEIDX),  KNUMS(TEIDX), 
     .              TIMSYS,  VSTRS(TEIDX),   TEND          )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'PRSMET' )
         RETURN
      END IF

C
C     If "usable" bounds have been supplied, use these bounds.
C     Otherwise use the default start and stop times.
C     
      IF (  ELEMC( KEYS(UBIDX), ITEMS )  ) THEN

         CALL TIMCVT ( INPUT,   KLINES(UBIDX),  KNUMS(UBIDX), 
     .                 TIMSYS,  VSTRS(UBIDX),   UBEG          )

         IF ( FAILED() ) THEN
            CALL CHKOUT ( 'PRSMET' )
            RETURN
         END IF

      ELSE

         UBEG = TBEG
         
      END IF

      IF (  ELEMC( KEYS(UEIDX), ITEMS )  ) THEN

         CALL TIMCVT ( INPUT,   KLINES(UEIDX),  KNUMS(UEIDX), 
     .                 TIMSYS,  VSTRS(UEIDX),   UEND          )

         IF ( FAILED() ) THEN
            CALL CHKOUT ( 'PRSMET' )
            RETURN
         END IF

      ELSE

         UEND = TEND
         
      END IF

C
C     Make sure usable times are not outside of the block's
C     time range.
C
      IF (  ( UBEG .LT. TBEG ) .OR. ( UEND .GT. TEND )  ) THEN

         CALL SETMSG ( 'Usable time bounds must lie within range '
     .   //            'defined by the start time stop times '
     .   //            'given in the metadata block. Start time '
     .   //            'is #; usable start time is #; usable end time '
     .   //            'is #.'                                         )
         CALL ERRCH  ( '#', VSTRS(TBIDX)                               )
         CALL ERRCH  ( '#', VSTRS(UBIDX)                               )
         CALL ERRCH  ( '#', VSTRS(UEIDX)                               )
         CALL ERRCH  ( '#', VSTRS(TEIDX)                               )
         CALL SIGERR ( 'SPICE(TIMESOUTOFRANGE)'                        )
         CALL CHKOUT ( 'PRSMET' )
         RETURN

      END IF

C
C     Set the frame output arguments. Check the frames.
C
      FRAMEA = ' '
      FRAMEB = ' '

      DO I = 1, 2

         IF ( I .EQ. 1 ) THEN
            
            FRAMEA = VSTRS(RFAIDX) (:FRNMLN)
            FRAME  = FRAMEA
         ELSE
            FRAMEB = VSTRS(RFBIDX) (:FRNMLN)
            FRAME  = FRAMEB
         END IF

         CALL NAMFRM ( FRAME, REFCOD )
 
         IF ( REFCOD .EQ. 0 ) THEN
 
            CALL SETMSG ( 'The reference frame # is not supported. '  
     .      //            'This problem may be due to the lack of a ' 
     .      //            'STRING_MAPPING assignment in your setup '  
     .      //            'file; such an assignment may be used to '  
     .      //            'map a frame name not recognized by SPICE ' 
     .      //            'to an equivalent name that is recognized. '
     .      //            'For example, the frame name EME2000 '      
     .      //            'may be mapped to the frame name J2000. '   
     .      //            'See the AEM2C User''s Guide '            
     .      //            'for more information.  Also note that '    
     .      //            'the setup file must end with a newline '   
     .      //            'character in order for the final line '    
     .      //            'of the file to be readable.'               )
            CALL ERRCH  ( '#', FRAME                                  )
            CALL SIGERR ( 'SPICE(INVALIDREFFRAME)'                    )
            CALL CHKOUT ( 'PRSMET'                                    )
            RETURN
 
         END IF

      END DO

C
C     Set the attitude type.
C
      ATTREP = VSTRS( ATPIDX )

      IF ( ATTREP .NE. 'QUATERNION' ) THEN

         CALL SETMSG ( 'Attitude type was #; only supported '
     .   //            'value is QUATERNION.'                )
         CALL ERRCH  ( '#', ATTREP                           )
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                 )
         CALL CHKOUT ( 'PRSMET'                              )
         RETURN

      END IF

C
C     Set the attitude direction.
C
      IF ( VSTRS(ADRIDX) .EQ. 'A2B' ) THEN

         A2B = .TRUE.

      ELSE IF ( VSTRS(ADRIDX) .EQ. 'B2A' ) THEN

         A2B = .FALSE.

      ELSE

         CALL SETMSG ( 'Attitude direction was #; only supported '
     .   //            'values are A2B and B2A.'                  )
         CALL ERRCH  ( '#', VSTRS(ADRIDX)                         )
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                      )
         CALL CHKOUT ( 'PRSMET'                                   )
         RETURN

      END IF

C
C     Set the "quaternion last" flag.
C
      QLAST = .FALSE.

      IF (  MATCHI(ATTREP, 'QUATERNION*', '*', '?' )  ) THEN
C
C        We have a quaternion attitude representation. The
C        quaternion order must be specified.
C
         IF (  .NOT. ELEMC( KEYS(QTPIDX), ITEMS )  ) THEN

            CALL SETMSG ( 'Attitude representation uses '
     .      //            'quaternions but quaternion type '
     .      //            'was not specified.'               )
            CALL SIGERR ( 'SPICE(MISSINGITEM)'               )
            CALL CHKOUT ( 'PRSMET'                           )
            RETURN

         END IF

         IF ( VSTRS(QTPIDX) .EQ. 'LAST' ) THEN

            QLAST = .TRUE.

         ELSE IF ( VSTRS(QTPIDX) .EQ. 'FIRST' ) THEN

            QLAST = .FALSE.

         ELSE

            CALL SETMSG ( 'Quaternion type was #; only supported '
     .      //            'values are FIRST and LAST.'             )
            CALL ERRCH  ( '#', VSTRS(QTPIDX)                       )
            CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                    )
            CALL CHKOUT ( 'PRSMET'                                 )
            RETURN

         END IF

      END IF

C
C     All of the outputs are set.
C
      CALL CHKOUT ( 'PRSMET' )
      RETURN
      END
