C$Procedure PRCSET ( Process setup file for AEM2CK---umbrella routine )

      SUBROUTINE PRCSET ( SETUP,   INPUT,   OUTPUT,  APPEND, 
     .                    CMTFIL,  METHOD,  DEGREE,  SPCLIM,
     .                    QUOLIM,  CKBEG,   CKEND,   RATE,
     .                    SELLST                             )
  
C$ Abstract
C
C     Umbrella routine for AEM2CK setup file processing.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     AEM2CK
C
C$ Declarations
 
      IMPLICIT NONE

      INCLUDE 'aem2ck.inc'

      CHARACTER*(*)         SETUP
      CHARACTER*(*)         INPUT
      CHARACTER*(*)         OUTPUT
      LOGICAL               APPEND
      CHARACTER*(*)         CMTFIL
      CHARACTER*(*)         METHOD
      INTEGER               DEGREE
      DOUBLE PRECISION      SPCLIM
      DOUBLE PRECISION      QUOLIM
      DOUBLE PRECISION      CKBEG
      DOUBLE PRECISION      CKEND
      DOUBLE PRECISION      RATE
      LOGICAL               SELLST

C$ Brief_I/O
C
C     VARIABLE  I/O  Entry points
C     --------  ---  --------------------------------------------------
C     SETUP      I   GETSET
C     INPUT     I/O  GETSET
C     OUTPUT    I/O  GETSET
C     APPEND    I/O  GETSET
C     CMTFIL     O   GETPAR
C     METHOD     O   GETPAR
C     DEGREE     O   GETPAR
C     SPCLIM     O   GETPAR
C     QUOLIM     O   GETPAR
C     CKBEG      O   GETPAR
C     CKEND      O   GETPAR
C     RATE       O   GETPAR
C     SELLST     O   GETPAR
C
C$ Detailed_Input
C
C     See the entry points for a description of their inputs.
C
C$ Detailed_Output
C
C     See the entry points for a description of their outputs.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     If this routine is called directly, the error SPICE(BOGUSENTRY)
C     is signaled.  See the entry points for descriptions of 
C     exceptions specific to those routines.
C
C$ Files
C
C     This suite of routines reads and returns information from
C     an AEM2CK setup file.  This file is a SPICE text kernel
C     that specifies 
C
C        - The names of SPICE kernels to load. This list must include
C
C              > A leapseconds kernel
C              > An SCLK kernel
C              > A frame kernel for the instrument/spacecraft
C                structure frame
C
C          The keyword for this list of names is that used by 
C          the SPICE KEEPER subsystem:
C
C             KERNELS_TO_LOAD
C
C        - The nominal spacecraft clock rate in units of seconds/tick.
C          The keyword for this value is
C           
C             NOMINAL_SCLK_RATE
C
C        - The policy for mini-segment selection when a request time
C          lies on AEM block time boundaries. A value of 'YES'
C          indicates that data from the later block are to be used; a
C          value of 'NO' indicates that the data from the earlier block
C          are to be used. The keyword for the selection is
C
C             SELECT_LAST
C
C          Values associated with this keyword are not case-sensitive.
C          This keyword is optional; the default policy is to select
C          the later mini-segment.
C
C        - Optionally, the name of an interpolation method to use.  The 
C          keyword is 
C
C             INTERPOLATION_METHOD
C
C          Currently the supported values are 
C
C             'HERMITE'
C             'LAGRANGE'
C
C          If specified, the interpolation method and degree (see below)
C          supersede values that may be provided in AEM metadata
C          blocks.
C
C        - If an interpolation method is specified, the polynomial
C          degree to use with the interpolation method is required.
C          (This item may become optional if other interpolation
C          methods are added.) The associated keyword is
C
C             INTERPOLATION_DEGREE
C        
C        - Optionally, the name of a text file whose contents are
C          to be placed in the comment area of the output CK file.
C          The associated keyword is
C      
C             COMMENT_FILE
C
C        - Optionally, the names of the input AEM file and the 
C          output CK file may be specified in the setup file.
C         
C          The associated keywords are
C
C             INPUT_AEM_FILE
C             OUTPUT_CK_FILE
C
C
C        - Optionally, time bounds for the output CK file may 
C          be specified in the setup file.  The coverage interval
C          of the CK file will be the intersection of the coverage
C          interval of the AEM file and of the interval specified
C          by the time bounds in the setup file
C         
C          The associated keywords are
C
C             START_TIME
C             STOP_TIME
C
C          Permissible values are any time strings recognized by 
C          STR2ET.  The time strings must be enclosed in single
C          quotes.
C
C        - Optionally, the minimum allowed spacing of states in the
C          CK file may be specified.  An error will be signaled
C          if adjacent states in a data block have spacing which
C          is less than the specified limit.
C         
C          The associated keyword is
C
C             MINIMUM_SPACING
C
C          The associated value is interpreted to have units of 
C          seconds.
C
C        - Optionally, the maximum allowed allowed value of 
C
C             (t3-t2)/(t2-t1) and (t2-t1)/(t3-t2)
C
C          for consecutive time tags t1, t2, t3 in a data block may be
C          specified.  An error will be signaled if adjacent states in
C          a data block have spacing which is exceeds the specified
C          limit.
C         
C          The associated keyword is
C
C             MAXIMUM_SEP_QUOTIENT
C
C        - Optionally, a mapping may be defined from values appearing
C          in the AEM metadata to specified values.  
C          
C          The associated keyword is
C
C             STRING_MAPPING
C
C          The associated values are an array of strings.  The format
C          of the array is
C         
C             ( 'first input value',   'first mapped value',
C               'second input value',  'second mapped value',
C                       ...                   ...             )
C                      
C$ Particulars
C
C     The entry points in this package are
C
C        GETSET  {Get setup file information for AEM2CK}
C        GETPAR  {Get CK parameters for AEM2CK}
C
C     GETSET must be called before GETPAR may be called.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     This routine is intended for use only within the program 
C     AEM2CK.
C
C$ Literature_References
C
C     [1]  CCSDS Attitude Data Messages Blue Book, version CCSDS 
C          504.0-B-1, May, 2008.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    AEM2CK Version 2.0.0, 03-MAR-2017 (NJB)
C
C        No longer looks up nominal SCLK rate.
C
C-    AEM2CK Version 1.0.0, 19-AUG-2015 (NJB)
C      
C        Based on OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&

C
C     SPICELIB functions
C
      DOUBLE PRECISION      DPMAX
      DOUBLE PRECISION      DPMIN

      LOGICAL               EQCHR
      LOGICAL               EQSTR
      LOGICAL               EXISTS
      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C
      CHARACTER*(*)         APPNAM
      PARAMETER           ( APPNAM = 'APPEND_TO_OUTPUT'     )

      CHARACTER*(*)         BEGNAM
      PARAMETER           ( BEGNAM = 'START_TIME'           )

      CHARACTER*(*)         DEGNAM
      PARAMETER           ( DEGNAM = 'INTERPOLATION_DEGREE' )

      CHARACTER*(*)         INPNAM
      PARAMETER           ( INPNAM = 'INPUT_AEM_FILE'       )

      CHARACTER*(*)         INTNAM
      PARAMETER           ( INTNAM = 'INTERPOLATION_METHOD' )

      CHARACTER*(*)         OUTNAM
      PARAMETER           ( OUTNAM = 'OUTPUT_CK_FILE'      )

      CHARACTER*(*)         CMTNAM
      PARAMETER           ( CMTNAM = 'COMMENT_FILE'         )

      CHARACTER*(*)         QUONAM
      PARAMETER           ( QUONAM = 'MAXIMUM_SEP_QUOTIENT' )
      
      CHARACTER*(*)         SELNAM
      PARAMETER           ( SELNAM = 'SELECT_LAST'          )

      CHARACTER*(*)         SPCNAM
      PARAMETER           ( SPCNAM = 'MINIMUM_SPACING'      )

      CHARACTER*(*)         STPNAM
      PARAMETER           ( STPNAM = 'STOP_TIME'            )

      INTEGER               SHORT
      PARAMETER           ( SHORT  = 15 )

      INTEGER               TIMLEN
      PARAMETER           ( TIMLEN = 50 )
C
C     Local variables
C
      CHARACTER*(SHORT)     APPVAL
      CHARACTER*(1)         DTYPE
      CHARACTER*(LNSIZE)    SAVMTH
      CHARACTER*(SHORT)     SELSTR
      CHARACTER*(TIMLEN)    TIMSTR

      DOUBLE PRECISION      SAVBEG
      DOUBLE PRECISION      SAVEND
      DOUBLE PRECISION      SAVQUO
      DOUBLE PRECISION      SAVSPC

      INTEGER               SAVDEG
      INTEGER               N

      LOGICAL               INIT
      LOGICAL               FOUND
      LOGICAL               SAVSEL

C
C     Saved variables
C     
      SAVE                 INIT
      SAVE                 SAVBEG
      SAVE                 SAVEND
      SAVE                 SAVMTH
      SAVE                 SAVDEG
      SAVE                 SAVQUO
      SAVE                 SAVSEL
      SAVE                 SAVSPC

C
C     Initial values
C
      DATA                 INIT  / .FALSE. /


      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'PRCSET' )

      CALL SIGERR ( 'SPICE(BOGUSENTRY)' )

      CALL CHKOUT ( 'PRCSET' )
      RETURN

       





C$Procedure GETSET ( Get setup file information for AEM2CK )
 
      ENTRY GETSET ( SETUP, INPUT, OUTPUT, APPEND, CMTFIL )
 
C$ Abstract
C
C     Get the names of the input AEM file, the output CK file, and
C     the append flag from an AEM2CK setup file.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     AEM2CK
C
C$ Declarations
C
C     CHARACTER*(*)         SETUP
C     CHARACTER*(*)         INPUT
C     CHARACTER*(*)         OUTPUT
C     LOGICAL               APPEND
C     CHARACTER*(*)         CMTFIL
C
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     SETUP      I   Name of setup file.
C     INPUT     I/O  Name of input AEM file.
C     OUTPUT    I/O  Name of output CK file.
C     APPEND    I/O  Append flag.
C     CMTFIL     O   Name of comment file.
C
C$ Detailed_Input
C
C     SETUP          is the name of an AEM2CK setup file.
C
C     INPUT          is the name of an AEM file to be converted
C                    to CK format.  This file conforms to the 
C                    format specification given by the 
C                    draft CCSDS Orbit Data Messages Red Book
C                    Acton 502-H.
C
C     OUTPUT         is the name of an CK file to be written.
C                    OUTPUT may be a new or existing file.
C                    OUTPUT may contain type 13 or type 9 segments.
C
C     APPEND         is a logical flag indicating whether to 
C                    append data to an existing output file.
C                    If APPEND is .TRUE., the file designated 
C                    by OUTPUT need not exist.  If APPEND is .FALSE.,
C                    the file designated by OUTPUT must not exist.
C
C$ Detailed_Output
C
C     CMTFIL         is the name of a comment file whose contents
C                    are to be added to the comment area of
C                    the CK file created by this program.  The
C                    comment file contents precede the default
C                    comments added by AEM2CK.
C
C                    If no comment file is specified, CMTFIL is
C                    returned blank.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1)  If the setup file name is blank, the error 
C         SPICE(BLANKFILENAME) is signaled.
C
C     2)  If the setup file doesn't exist, the error 
C         SPICE(FILENOTFOUND) is signaled.
C
C     3)  If the AEM file name is not specified on the command line
C         and doesn't appear in the setup file, the error
C         SPICE(NOFILESPEC) is signaled.
C
C     4)  If the input file name is not specified on the command line
C         on in the setup file, the error SPICE(NOFILESPEC) is signaled.
C
C     5)  If the input AEM file doesn't exist, the error
C         SPICE(FILENOTFOUND) is signaled.
C
C     6)  If the output file name is not specified on the command line
C         on in the setup file, the error SPICE(NOFILESPEC) is signaled.
C
C     7)  If the output file name matches that of an existing file,
C         and if appending is not requested, the error 
C         SPICE(FILEEXISTS) is signaled.
C
C     8)  If a CK mini-segment selection policy is not specified in the
C         setup file, the error SPICE(NOSELECTIONFLAG) is signaled.
C         If the flag is present in the setup file but the value is 
C         neither 'YES' nor 'NO', the error SPICE(BADSELECTIONFLAG) is
C         signaled.
C
C     9)  If the interpolation method is not specified in
C         the setup file, the method is set to 'DEFAULT'. This
C         is not an error.
C
C     10) If the interpolation degree is not specified in
C         the setup file and the interpolation
C         method is not specified either, the degree is set to zero.
C         This is not an error.  However, if the interpolation
C         method is specified without a degree, the error 
C         SPICE(NODEGREE) is signaled.
C
C     11) If a comment file keyword is present in the setup file
C         but the associated value does not parse as a quoted string,
C         the error SPICE(TYPEMISMATCH) is signaled.
C
C     12) If a minimum spacing keyword is present in the setup file
C         but the associated value does not parse as a number, 
C         the error SPICE(TYPEMISMATCH) is signaled.
C
C     13) If a minimum spacing keyword is present in the setup file
C         but the associated value is not greater than zero, 
C         the error SPICE(VALUEOUTOFRANGE) is signaled.
C
C     14) If a maximum separation quotient keyword is present 
C         in the setup file but the associated value does not parse as
C         a number, the error SPICE(TYPEMISMATCH) is signaled.
C
C     15) If a maximum separation quotient keyword is present in the
C         setup file but the associated value is not greater than zero,
C         the error SPICE(VALUEOUTOFRANGE) is signaled.
C
C     16) If an CK start time is present in the setup file
C         but the associated value does not parse as a quoted
C         string, the error SPICE(TYPEMISMATCH) is signaled.
C
C     17) If an CK stop time is present in the setup file
C         but the associated value does not parse as a quoted
C         string, the error SPICE(TYPEMISMATCH) is signaled.
C
C$ Files
C
C     See the descriptions of INPUT and OUTPUT above.
C
C$ Particulars
C
C     None.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     This routine is intended for use only within the program 
C     AEM2CK.
C
C$ Literature_References
C
C     [1]  CCSDS Attitude Data Messages Blue Book, version CCSDS 
C          504.0-B-1, May, 2008.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    AEM2CK Version 1.0.0, 19-AUG-2015 (NJB)
C      
C        Based on OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&

      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'GETSET' )

C
C     Check the setup file name.
C
      IF ( SETUP .EQ. ' ' ) THEN

         CALL SETMSG ( 'Setup file name may not be blank.' )
         CALL SIGERR ( 'SPICE(BLANKFILENAME)'              )
         CALL CHKOUT ( 'GETSET'                            )
         RETURN

      END IF

      IF ( .NOT. EXISTS(SETUP) ) THEN

         CALL SETMSG ( 'Setup file <#> was not found.' )
         CALL ERRCH  ( '#', SETUP                      )
         CALL SIGERR ( 'SPICE(FILENOTFOUND)'           )
         CALL CHKOUT ( 'GETSET'                        )
         RETURN
         
      END IF

C
C     Load the setup file.
C      
      CALL FURNSH ( SETUP )

      IF ( FAILED() ) THEN
      
         CALL CHKOUT ( 'GETSET' )
         RETURN

      END IF

C
C     Check the append flag.  If APPEND is .FALSE. on entry, see
C     whether appending is specified in the setup file.  Otherwise
C     leave APPEND untouched.
C
      IF ( .NOT. APPEND ) THEN

         CALL GCPOOL ( APPNAM, 1, 1, N, APPVAL, FOUND )

         IF ( FOUND ) THEN
C
C           Use the first non-blank letter of the value to 
C           determine the RHS value.
C
            CALL LJUST ( APPVAL, APPVAL )

            APPEND = EQCHR( APPVAL(1:1), 'Y' ) 

         END IF

      END IF
      
C
C     Check the input file name.  If the name is available, it 
C     supersedes an input file name supplied in the setup file.
C     If the name is blank, the setup file must supply the name.
C
      IF ( INPUT .EQ. ' ' ) THEN         
C
C        Extract the input file name from the kernel pool.
C        Set the INPUT argument to the specified file name.
C
         CALL GCPOOL ( INPNAM, 1, 1, N, INPUT, FOUND )

         IF ( .NOT. FOUND ) THEN

            CALL SETMSG ( 'Input file was not specified on the ' //
     .                    'command line or in the setup file.'    )
            CALL SIGERR ( 'SPICE(NOFILESPEC)'                     )
            CALL CHKOUT ( 'GETSET'                                )
            RETURN

         END IF

      END IF

      IF ( .NOT. EXISTS(INPUT) ) THEN

         CALL SETMSG ( 'Input file <#> was not found.' )
         CALL ERRCH  ( '#', INPUT                      )
         CALL SIGERR ( 'SPICE(FILENOTFOUND)'           )
         CALL CHKOUT ( 'GETSET'                        )
         RETURN

      END IF
      
C
C     Check the output file name.  If the name is available, it 
C     supersedes an output file name supplied in the setup file.
C     If the name is blank, the setup file must supply the name.
C
      IF ( OUTPUT .EQ. ' ' ) THEN
C
C        Extract the output file name from the kernel pool.
C        Set the INPUT argument to the specified file name.
C
         CALL GCPOOL ( OUTNAM, 1, 1, N, OUTPUT, FOUND )

         IF ( .NOT. FOUND ) THEN

            CALL SETMSG ( 'Output file was not specified on the ' //
     .                    'command line or in the setup file.'     )
            CALL SIGERR ( 'SPICE(NOFILESPEC)'                      )
            CALL CHKOUT ( 'GETSET'                                 )
            RETURN

         END IF

      END IF

      IF ( .NOT. APPEND ) THEN

         IF ( EXISTS(OUTPUT)  ) THEN

            CALL SETMSG ( 'Output file <#> already exists.  File ' //
     .                    'must not exist unless appending '       //
     .                    'is requested.'                           )
            CALL ERRCH  ( '#', OUTPUT                               )
            CALL SIGERR ( 'SPICE(FILEEXISTS)'                       )
            CALL CHKOUT ( 'GETSET'                                  )
            RETURN

         END IF

      END IF
 

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'GETSET' )
         RETURN
      END IF

C
C     Obtain the mini-segment selection flag. The value of 
C     the flag is 'YES' or 'NO'. The default value is 'YES',
C     meaning the later mini-segment will be selected if a
C     request time lies on the boundary between two 
C     interpolation intervals.
C
      CALL GCPOOL ( SELNAM, 1, 1, N, SELSTR, FOUND )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'GETSET' )
         RETURN
      END IF

      IF ( FOUND ) THEN
         
         IF (  EQSTR( SELSTR, 'YES' )  ) THEN

            SAVSEL = .TRUE.

         ELSE IF ( EQSTR( SELSTR, 'NO' )  ) THEN

            SAVSEL = .FALSE.

         ELSE

            CALL SETMSG ( 'Mini-segment selection policy flag '
     .      //            'must be YES or NO but was #.'        )
            CALL ERRCH  ( '#',  SELSTR                          )
            CALL SIGERR ( 'SPICE(BADSELECTIONFLAG)'             )
            CALL CHKOUT ( 'GETSET'                              )
            RETURN

         END IF

      ELSE

         SAVSEL = .TRUE.

      END IF

C
C     Look up the interpolation method and interpolation degree 
C     (we're assuming a polynomial method is used).
C
      CALL GCPOOL ( INTNAM, 1, 1, N, SAVMTH, FOUND )

      IF ( .NOT. FOUND ) THEN

         SAVMTH = 'DEFAULT'

      END IF


      CALL GIPOOL ( DEGNAM, 1, 1, N, SAVDEG, FOUND )

      IF ( .NOT. FOUND ) THEN

         IF ( SAVMTH .EQ. 'DEFAULT' ) THEN

            SAVDEG = 0

         ELSE

            CALL SETMSG ( 'Interpolation degree was not specified ' //
     .                    'in the setup file.'                      )
            CALL SIGERR ( 'SPICE(NODEGREE)'                         )
            CALL CHKOUT ( 'GETSET'                                  )
            RETURN

         END IF

      END IF

C
C     See whether a comment file specification was given.
C     Capture the file name if so.
C
      CALL DTPOOL ( CMTNAM, FOUND, N, DTYPE )

      IF ( FOUND ) THEN

         IF ( DTYPE .NE. 'C' ) THEN

            CALL SETMSG ( 'Comment file name was not given a '       //
     .                    'character string value in the setup file.' )
            CALL SIGERR ( 'SPICE(TYPEMISMATCH)'                       )
            CALL CHKOUT ( 'GETSET'                                    )
            RETURN

         END IF

         CALL GCPOOL ( CMTNAM, 1, 1, N, CMTFIL, FOUND ) 
         
      END IF

      IF ( .NOT. FOUND ) THEN
         CMTFIL =  ' '
      END IF

C
C     See whether a minimum state spacing was given.
C     Capture the value if so.  Otherwise use the default.
C
      CALL DTPOOL ( SPCNAM, FOUND, N, DTYPE )

      IF ( FOUND ) THEN

         IF ( DTYPE .NE. 'N' ) THEN

            CALL SETMSG ( 'Minimum state spacing was not given a '   //
     .                    'double precision value in the setup file.' )
            CALL SIGERR ( 'SPICE(TYPEMISMATCH)'                       )
            CALL CHKOUT ( 'GETSET'                                    )
            RETURN

         END IF

         CALL GDPOOL ( SPCNAM, 1, 1, N, SAVSPC, FOUND ) 
         
      END IF

      IF ( .NOT. FOUND ) THEN
         SAVSPC = MINSPC
      END IF

C
C     The minimum allowed spacing must be positive.
C
      IF ( SAVSPC .LE. 0.D0 ) THEN

         CALL SETMSG ( 'Minimum state spacing # is non-positive.' )  
         CALL ERRDP  ( '#',  SAVSPC                               )
         CALL SIGERR ( 'SPICE(VALUEOUTOFRANGE)'                   )
         CALL CHKOUT ( 'GETSET'                                   )
         RETURN

      END IF

C
C
C     See whether a maximum state spacing quotient was given.
C     Capture the value if so.  Otherwise use the default.
C
      CALL DTPOOL ( QUONAM, FOUND, N, DTYPE )

      IF ( FOUND ) THEN

         IF ( DTYPE .NE. 'N' ) THEN

            CALL SETMSG ( 'Maximum state separation quotient was '  //
     .                    'not given a double precision value '     //
     .                    'in the setup file.'                      )
            CALL SIGERR ( 'SPICE(TYPEMISMATCH)'                     )
            CALL CHKOUT ( 'GETSET'                                  )
            RETURN

         END IF

         CALL GDPOOL ( QUONAM, 1, 1, N, SAVQUO, FOUND ) 
         
      END IF

      IF ( .NOT. FOUND ) THEN
         SAVQUO = MAXQUO
      END IF

C
C     The maximum allowed separation quotient must be positive.
C
      IF ( SAVQUO .LE. 0.D0 ) THEN

         CALL SETMSG ( 'Minimum state separation quotient # is ' //
     .                 'non-positive.'                            )  
         CALL ERRDP  ( '#',  SAVQUO                               )
         CALL SIGERR ( 'SPICE(VALUEOUTOFRANGE)'                   )
         CALL CHKOUT ( 'GETSET'                                   )
         RETURN

      END IF

C
C     See whether an output CK start time was given.
C     Capture the value if so.  Otherwise use the default.
C
      CALL DTPOOL ( BEGNAM, FOUND, N, DTYPE )

      IF ( FOUND ) THEN

         IF ( DTYPE .NE. 'C' ) THEN

            CALL SETMSG ( 'CK start time was not '                  //
     .                    'given a character string value in the '   //
     .                    'setup file.'                               )
            CALL SIGERR ( 'SPICE(TYPEMISMATCH)'                       )
            CALL CHKOUT ( 'GETSET'                                    )
            RETURN

         END IF

         CALL GCPOOL ( BEGNAM, 1, 1, N, TIMSTR, FOUND ) 
         
         IF ( FOUND ) THEN

            CALL STR2ET ( TIMSTR, SAVBEG )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'GETSET' )
               RETURN
            END IF

         END IF

      END IF

      IF ( .NOT. FOUND ) THEN
         SAVBEG = DPMIN()
      END IF

C
C     See whether an output CK stop time was given.
C     Capture the value if so.  Otherwise use the default.
C
      CALL DTPOOL ( STPNAM, FOUND, N, DTYPE )

      IF ( FOUND ) THEN

         IF ( DTYPE .NE. 'C' ) THEN

            CALL SETMSG ( 'CK stop time was not '                   //
     .                    'given a character string value in the '   //
     .                    'setup file.'                               )
            CALL SIGERR ( 'SPICE(TYPEMISMATCH)'                       )
            CALL CHKOUT ( 'GETSET'                                    )
            RETURN

         END IF

         CALL GCPOOL ( STPNAM, 1, 1, N, TIMSTR, FOUND ) 
         
         IF ( FOUND ) THEN

            CALL STR2ET ( TIMSTR, SAVEND )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'GETSET' )
               RETURN
            END IF

         END IF

      END IF

      IF ( .NOT. FOUND ) THEN
         SAVEND = DPMAX()
      END IF


      INIT = .TRUE.

      CALL CHKOUT ( 'GETSET' )
      RETURN






C$Procedure GETPAR ( Get CK parameters for AEM2CK )
 
      ENTRY GETPAR ( METHOD, DEGREE, SPCLIM, QUOLIM, 
     .               CKBEG,  CKEND,  RATE,   SELLST  )
 
C$ Abstract
C
C     Return CK parameters specified in an AEM2CK setup file.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     AEM2CK
C
C$ Declarations
C
C     CHARACTER*(*)         METHOD
C     INTEGER               DEGREE
C     DOUBLE PRECISION      SPCLIM
C     DOUBLE PRECISION      QUOLIM
C     DOUBLE PRECISION      CKBEG
C     DOUBLE PRECISION      CKEND
C     DOUBLE PRECISION      RATE
C     LOGICAL               SELLST
C
C$ Brief_I/O
C
C     VARIABLE  I/O  Description
C     --------  ---  --------------------------------------------------
C     METHOD     O   Interpolation method.
C     DEGREE     O   Degree of interpolating polynomials.
C     SPCLIM     O   Minimum state spacing.
C     QUOLIM     O   Maximum quotient of successive state separations.
C     CKBEG      O   TDB begin time of output CK coverage. 
C     CKEND      O   TDB end time of output CK coverage.
C     RATE       O   Placeholder SCLK rate; set to zero.
C     SELLST     O   Mini-segment selection policy flag.
C
C$ Detailed_Input
C
C     None.
C
C$ Detailed_Output
C
C     METHOD         is the interpolation method:  'HERMITE' or
C                    'LAGRANGE'.  If no method was specified in the
C                    setup file, the value 'DEFAULT' is returned.
C
C     DEGREE         is the associated polynomial degree. If no
C                    interpolation method was specified in the setup
C                    file, the value 0 is returned.
C
C     SPCLIM         is the minimum spacing, measured in ephemeris
C                    seconds, between successive states in one
C                    input data block (the set of states following
C                    a metadata block and preceding the next
C                    metadata block or EOF).  If not specified in the
C                    setup file, SPCLIM defaults to MINSPC (declared in
C                    aem2ck.inc).
C     
C     QUOLIM         is the maximum quotient of successive separations
C                    between states (considered in either order).  If
C                    not specified in the setup file, QUOLIM defaults to
C                    MAXQUO (declared in aem2ck.inc).
C 
C     CKBEG,
C     CKEND          are time bounds for the output CK file.  These
C                    bounds are optionally specified in the setup 
C                    file by the keywords
C
C                       START_TIME
C                       STOP_TIME
C                   
C
C     RATE           is a placeholder SCLK rate. This value is no longer
C                    used by AEM2CK, but the argument has been left in
C                    place for software compatibility.
C
C
C     SELLST         is a logical flag indicating to the CK type 6
C                    segment reader CKR06 how to select the
C                    mini-segment interval when a request time
C                    coincides with a time boundary shared by two
C                    mini-segment intervals. When SELLST ("select
C                    last") is .TRUE., the later interval is selected;
C                    otherwise the earlier interval is selected.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     If this routine is called before the setup file has been
C     read by GETSET, the error SPICE(NOTINITIALIZED) is signaled.
C
C$ Files
C
C     The parameters returned by this entry point are specified in an
C     AEM2CK setup file. See the header of the umbrella routine PRCSET
C     for further information.
C
C$ Particulars
C
C     None.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     [1]  CCSDS Attitude Data Messages Blue Book, version CCSDS 
C          504.0-B-1, May, 2008.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    AEM2CK Version 2.0.0, 03-MAR-2017 (NJB)
C
C        Sets output SCLK rate to zero.
C
C-    AEM2CK Version 1.0.0, 19-AUG-2015 (NJB)
C      
C        Based on OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&
      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'GETPAR' )

      IF ( .NOT. INIT ) THEN

         CALL SETMSG ( 'GETPAR may not be called before setup ' //
     .                 'initialization has been completed.'      )
         CALL SIGERR ( 'SPICE(NOTINITIALIZED)'                   )
         CALL CHKOUT ( 'GETPAR'                                  )
         RETURN

      END IF

      METHOD = SAVMTH
      DEGREE = SAVDEG
      SPCLIM = SAVSPC
      QUOLIM = SAVQUO
      CKBEG  = SAVBEG
      CKEND  = SAVEND
      RATE   = 0.D0
      SELLST = SAVSEL

      CALL CHKOUT ( 'GETPAR' )
      RETURN
      END

