C$Procedure CVT01 ( Convert type 1 AEM to CK )
 
      SUBROUTINE CVT01 ( SETUP, INPUT, OUTPUT, APPEND, CMTFIL )
 
C$ Abstract
C
C     Convert format version 1 AEM file to CK format.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     AEM2CK
C
C$ Declarations
 
      IMPLICIT NONE
 
      CHARACTER*(*)         SETUP
      CHARACTER*(*)         INPUT
      CHARACTER*(*)         OUTPUT
      LOGICAL               APPEND
      CHARACTER*(*)         CMTFIL
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     SETUP      I   Name of setup file.
C     INPUT      I   Name of input AEM file.
C     OUTPUT     I   Name of output CK file.
C     APPEND     I   Append flag.
C     CMTFIL     I   Name of comment file.
C
C$ Detailed_Input
C
C     SETUP          is the name of an AEM2CK setup file.
C
C     INPUT          is the name of an AEM file to be converted
C                    to CK format.  This file conforms to the 
C                    format specification given by the 
C                    draft CCSDS Orbit Data Messages Red Book
C                    Acton 502-H.
C
C     OUTPUT         is the name of an CK file to be written.
C                    OUTPUT may be a new or existing file.
C                    OUTPUT may contain type 13 or type 9 segments.
C
C     APPEND         is a logical flag indicating whether to 
C                    append data to an existing output file.
C                    If APPEND is .TRUE., the file designated 
C                    by OUTPUT must exist.  If APPEND is .FALSE.,
C                    the file designated by OUTPUT must not exist.
C
C     CMTFIL         is the name of a comment file whose contents
C                    are to be added to the comment area of
C                    the CK file created by this program.  The
C                    comment file contents precede the default
C                    comments added by AEM2CK.
C
C                    If no comment file is specified, CMTFIL is
C                    returned blank.
C
C$ Detailed_Output
C
C     None.
C
C$ Parameters
C
C     None.
C
C$ Exceptions
C
C     1) Errors detected while reading the AEM file will be 
C        diagnosed by routines called by this routine.
C
C     2) Errors detected while parsing the AEM file will be
C        diagnosed by routines called by this routine.
C
C     3) Errors detected while opening the output CK file will be
C        diagnosed by routines called by this routine.
C
C     4) Errors detected while writing the output CK file will be
C        diagnosed by routines called by this routine.
C
C$ Files
C
C     See the descriptions of input arguments INPUT and OUTPUT above.
C
C$ Particulars
C
C     This routine operates by side effects.  The principal actions
C     taken by this routine are:
C
C        1) Open the output file as a new file, or for appending, if
C           the file already exists and appending was requested.
C
C        2) Add comments to the output file.  These include:
C
C             - The contents of a comment file, if any, specified in
C               the setup file.
C
C             - The run time and date, the names of the setup, input,
C               and output files, and an indication of whether the
C               output file was new or appended to.
C
C             - The contents of the setup file.
C
C        3) Read and parse each metadata block and following data
C           block. Write segments to the output CK. Contiguous
C           data blocks are written to mini-segments of a type 6
C           CK segment if the block parameters are compatible.
C
C        4) Close the output file.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Literature_References
C
C     [1]  CCSDS Attitude Data Messages Blue Book, version CCSDS 
C          504.0-B-1, May, 2008.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Version
C
C-    AEM2CK Version 1.0.0, 30-JUL-2015 (NJB)
C-&


C
C     SPICELIB functions
C
      LOGICAL               EXISTS
      LOGICAL               FAILED
      LOGICAL               RETURN

C
C     Local parameters
C
      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 80 )

C
C     Local variables
C
      CHARACTER*(LNSIZE)    DEFMTH

      DOUBLE PRECISION      SPCLIM
      DOUBLE PRECISION      QUOLIM
      DOUBLE PRECISION      CKBEG
      DOUBLE PRECISION      CKEND
      DOUBLE PRECISION      RATE

      INTEGER               DEFDEG
      INTEGER               HANDLE

      LOGICAL               LOCAPP
      LOGICAL               SELLST
      
      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'CVT01' )
C
C     Get the required items from the setup file:
C
C        - Minimum allowed spacing of states within a segment
C        - Maximum quotient of consecutive state separations
C          within a segment
C        - CK time bounds
C     
C     Also get default values, if available:
C
C        - Interpolation method
C        - Polynomial degree
C
C     And get:
C
C        - the nominal SCLK rate in seconds/tick
C        - the mini-segment selection policy flag
C
      CALL GETPAR ( DEFMTH, DEFDEG, SPCLIM, QUOLIM, 
     .              CKBEG,  CKEND,  RATE,   SELLST )
     
C
C     Capture a local copy of the append flag, since we 
C     update the flag if appending to a non-existent file
C     is requested.
C
      LOCAPP = APPEND
C
C
C     Open the output file for appending or as a new file.
C
      IF ( LOCAPP .AND. EXISTS(OUTPUT) ) THEN

         CALL CKOPA ( OUTPUT, HANDLE )

      ELSE

         LOCAPP = .FALSE.
         CALL CKOPN ( OUTPUT, OUTPUT, 0, HANDLE )

      END IF

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'CVT01' )
         RETURN
      END IF

C
C     Add comments to the output file.
C
      CALL ADDCOM ( HANDLE, SETUP, INPUT, OUTPUT, CMTFIL, LOCAPP )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'CVT01' )
         RETURN
      END IF

C
C     Read the input file and create CK type 6 segments.
C
      CALL MKCK06 ( INPUT,  HANDLE, DEFMTH, DEFDEG, SPCLIM, 
     .              QUOLIM, CKBEG,  CKEND,  RATE,   SELLST )

      IF ( FAILED() ) THEN
         CALL CHKOUT ( 'CVT01' )
         RETURN
      END IF
      
C
C     The CK file is complete.
C
      CALL CKCLS ( HANDLE )


      CALL CHKOUT ( 'CVT01' )
      RETURN
      END

      
