import zipfile
import subprocess
import datetime
import logging
import sys
import os
import math  # Used to access "eval" method

from datetime import datetime, timedelta

from .em16 import quaternions_filter
from .bepic import phebus_fov_reader

from adcsng.utils.files import get_date_from_filename


def tcp_reader(kernel, tcp_file, config):

    #
    # We convert the binary files into text files. The binary files
    # are then removed
    #
    if config['input_extension'] == 'zip' in tcp_file:

        zip_ref = zipfile.ZipFile(tcp_file, 'r')
        zip_ref.extractall()

        tcp_file = zip_ref.namelist()[0]

        zip_ref.close()

    lsk_file = kernel.config['kernels_dir']+'/'+kernel.config['lsk']

    # TODO: Remove this dirty workarodund for BepiColombo
    if config['mission_accr'] == 'MPO' or config['mission_accr'] == 'JUICE':
        mission = '_bc'
    elif config['mission_accr'] == 'SOLO':
        mission = '_bc'
    elif config['mission_accr'] == 'RM':
        mission = '_emrsp'
    else:
        mission = ''

    command_line_process = subprocess.Popen(
                    [kernel.directories.executables + '/' +
                     'tcpescan'+mission,
                     tcp_file, lsk_file],
                    stdout=subprocess.PIPE,
                    stderr=subprocess.STDOUT)

    process_output, _ = command_line_process.communicate()
    log = process_output.decode('utf-8')

    tcp_list = log.split('\n')

    #
    # The following lines of the output are generated by TCPESCAN and FORTRAN
    #
    tcp_list_iter = list(tcp_list)
    for element in tcp_list_iter:
        if 'IEEE_DENORMAL' in element:
            tcp_list.remove(element)

        if 'DELTET=' in element:
            tcp_list.remove(element)

        if 'STOP' in element:
            if 'Unexpected end of File' in element:
                sys.exit("TCP2SCLK: Unexpected end of File in {}".format(tcp_file))
            else:
                tcp_list.remove(element)

        if element == '':
            tcp_list.remove(element)

    return tcp_list


def aem_reader(aem_file, config):

    #
    # We init the variables to log the processing
    #
    info_ratio = 0

    with open(aem_file, 'r') as t:

        data_start = False
        aem_list = []
        index = 0
        previous_dt = ''

        for line in t:
            index += 1

            if 'START_TIME' in line:
                try:
                    date = line.split('=')[1].strip()[:-2]
                    segment_start_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S.%f")
                except:
                    date = line.split('=')[1].strip()
                    segment_start_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S")

            if 'STOP_TIME' in line:
                try:
                    date = line.split('=')[1].strip()[:-2]
                    segment_finish_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S.%f")
                except:
                    date = line.split('=')[1].strip()
                    segment_finish_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S")

            # ================================================================================
            # Toolkit version: N0065
            # SPICE(BADSPACINGRATIO) --
            # Consecutive time tags 2017-JAN-11 05:41:09.184225 TDB, 2017-JAN-11
            # 05:46:06.784225 TDB, 2017-JAN-11 05:46:09.184225 TDB, have spacing ratio
            # 1.2400000124176E+02; the limit is 1.0000000000000E+02.
            # A traceback follows.  The name of the highest level module is first.
            # AEM2CK --> CVTAEM --> CVT01 --> MKCK06 --> CHKTAG
            # ================================================================================
            if 'DATA_START' in line:
                data_start = True
                data_start_index = 0
            if 'DATA_STOP' in line:
                data_start = False

                #
                # We need to reset it in between segments otherwise:
                #
                #  ================================================================================
                #  Toolkit version: N0065
                #  SPICE(BOUNDSDISAGREE) --
                #  Mini-segment interval 1 start time 1.7161649282063E+12 precedes mini-segment's
                #  first epoch 1.7162033060708E+12.
                #  A traceback follows.  The name of the highest level module is first.
                #  AEM2CK --> CVTAEM --> CVT01 --> MKCK06 --> CKW06
                #  ================================================================================
                previous_dt = ''

            #
            # We identify the line as data line (after the META_STOP tag and
            # not an empty line)
            #
            if data_start and line.strip() != '\n' and 'DATA_START' not in line:

                date = line.split()[0][:-2]

                if previous_dt:
                    try:
                        current_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S.%f")
                    except:
                        date = line.split()[0]
                        current_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S")

                    deltat = current_dt - previous_dt
                    deltat = abs(deltat.seconds)

                    #
                    # We need to include the data_start_index condition to avoid removing the first entry of
                    # a data block.
                    #
                    if deltat < 5.0:
                        if previous_dt != segment_start_dt:
                            # logging.warning('   AEM Processing: Found spacing ratio less than 5 seconds in {index}:')
                            # logging.warning('   ' + line[:-1])
                            info_ratio += 1

                            #
                            # We need to remove the previous line, not the current line, otherwise we will remove a
                            # segment boundary.
                            #
                            aem_list.remove(previous_line)

                    aem_list.append(line)
                else:
                    aem_list.append(line)

                try:
                    previous_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S.%f")
                except:
                    date = line.split()[0]
                    previous_dt = datetime.strptime(date, "%Y-%m-%dT%H:%M:%S")

                data_start_index += 1
            else:
                aem_list.append(line)

            previous_line = line

    #
    # We report the number of exceptions here
    #
    if info_ratio:
        logging.warning(f'   AEM Processing: Found {info_ratio} intervals with spacing ratio less than 5 seconds')

    return aem_list


def oem_reader(oem_file, config):

    #
    # We init the variables to log the processing
    #
    info_incomplete = 0

    with open(oem_file, 'r') as t:

        data_start = False
        oem_list = []
        index = 0

        for line in t:

            index += 1

            # ==============================================================================
            #
            # Toolkit version: N0066
            #
            # SPICE(TOOFEWTOKENS) --
            #
            # Found only 1 numeric tokens while attempting to read a state vector. OEM file
            # is <OEMS_DA_059_01______00105.EXM>. Last line read was #68269. Data record
            # started on line #68269. Concatenated data line was
            # <2016-06-24T03:25:34.09215033>
            #
            # A traceback follows.  The name of the highest level module is first.
            # OEM2SPK --> CVTOEM --> CVT02 --> MAKSEG --> PRSDR
            #
            # ==============================================================================
            if 'META_START' in line:
                data_start = False

            # We need to remove the end of line character:
            line_splitted = line.split()

            #
            # We identify the line as data line (after the META_STOP tag and
            # not an empty line)
            #

            #
            # Implemented the possibility of having acceleration or not as an
            # input.
            #
            if ',' in config['input_fields']:
                input_fields = config['input_fields'].split(',')
            else:
                input_fields = [config['input_fields']]

            incomplete_line = False

            if data_start and line.strip() != '':

                incomplete_line = True

                for input_field in input_fields:
                    if len(line_splitted) == int(input_field):
                        oem_list.append(line)
                        incomplete_line = False

                if incomplete_line:
                    # logging.info('OEM Processing: Found incomplete data in input line {index}:')
                    # logging.info('   ' + line)
                    info_incomplete += 1
            else:
                oem_list.append(line)

            if 'META_STOP' in line:
                data_start = True

            #  OEM2SPK Program; Ver. 2.2.0, 01-DEC-2016; Toolkit Ver. N0065
            #  ================================================================================
            #  Toolkit version: N0065
            #  SPICE(TIMETAGERROR) --
            #  Consecutive time tags 2022-MAY-24 02:21:45.131521 and 2022-MAY-24
            #  02:21:45.131898 have spacing less than the limit 1.0000000000000E-03 seconds.
            #  A traceback follows.  The name of the highest level module is first.
            #  OEM2SPK --> CVTOEM --> CVT02 --> MAKSEG
            #  ================================================================================

            #  ================================================================================
            #  Toolkit version: N0065
            #  SPICE(TIMETAGERROR) --
            #  Consecutive time tags 2023-JAN-01 02:14:47.324165 2023-JAN-01 03:14:36.702917
            #  2023-JAN-01 03:14:36.846376 have spacing ratio 2.5020268339399E+04; the limit
            #  is 1.0000000000000E+02.
            #  A traceback follows.  The name of the highest level module is first.
            #  OEM2SPK --> CVTOEM --> CVT02 --> MAKSEG
            #  ================================================================================

    #
    # We report the number of exceptions here
    #
    if info_incomplete:
        logging.warning(f'   OEM Processing: Found {info_incomplete} incomplete data line(s)')

    return oem_list


def oasw_orbit_reader(oasw_orbit_file, config):

    #
    # We init the variables to log the processing
    #
    info_duplicate = 0

    with open(oasw_orbit_file, 'r') as t:

        data_start = False
        oasw_orbit_list = []
        index = 0
        previous_time = False

        for line in t:

            index += 1

            # MEX2KER Program; Ver. 2.2.0, 15-JUL-2014; Toolkit Ver. N0066
            # ================================================================================
            # Toolkit version: N0066
            # SPICE(TIMESOUTOFORDER) --
            # EPOCH 8.5707450000000E+08 having index 290 is not greater than its predecessor
            # 8.5707450000000E+08.
            # A traceback follows.  The name of the highest level module is first.
            # MEX2KER --> CVTMEX --> CVT01 --> MKSKSG --> SPKW18
            # ================================================================================

            if 'META_START' in line:
                data_start = False

            #
            # We identify the line as data line (after the META_STOP tag and
            # not an empty line)
            #

            if data_start and line.strip() != '':

                time = line.split()[0]

                if previous_time == time:

                    # logging.info(f'OASW Orbit Processing: Duplicated time in input line {index}:')
                    # logging.info('   ' + line)
                    info_duplicate += 1
                else:
                    oasw_orbit_list.append(line)

                previous_time = time
            else:
                oasw_orbit_list.append(line)

            if 'META_STOP' in line:
                data_start = True

    #
    # We report the number of exceptions here
    #
    if info_duplicate:
        logging.warning(f'   OASW Processing: Found {info_duplicate} duplicate(s)')

    return oasw_orbit_list


def oasw_attitude_reader(oasw_attitude_file, trim=0):

    #
    # We init the variables to log the processing
    #

    t = open(oasw_attitude_file, 'r').readlines()
    stop_date = False
    trimindex = 0
    for line in reversed(t):
        if 'STOP_TIME' in line and not stop_date:
            stop_date = line.split('= ')[1].strip()
            flagdate = datetime.strptime(stop_date.split('.')[0], "%Y-%m-%dT%H:%M:%S") - timedelta(hours=trim)
        if len(line.rstrip()) > 80 and stop_date:
            stop_date = line.split(' ')[0]
            if datetime.strptime(stop_date.split('.')[0], "%Y-%m-%dT%H:%M:%S") < flagdate:
                break
        trimindex += 1

    #
    # trim the source file before the corresponding meta data start
    #
    trimindex = len(t) - trimindex - 1
    with open(oasw_attitude_file, 'w+') as newfile:
        for i in range(0, trimindex, 1):
            newfile.write(t[i])
        newfile.write((flagdate + timedelta(seconds=1)).strftime("%Y-%m-%dT%H:%M:%S") + '.00000000 '
                      + t[trimindex].split(' ', 1)[1])

    return


def hk_quaternions2ck_reader(kernel, tm_file, config):

    #
    # We init the variables to log the processing
    #
    info_duplicate = 0
    info_gt_one = 0
    info_negative = 0
    info_incomplete = 0
    info_quaternions = 0
    #
    # We obtain the time filed number from the configuration file
    # It has to be a list for there can be two columns
    #
    input_time_format = config['tm_file'][0]['time_format']
    header_lines = int(config['tm_file'][0]['header_lines'])

    if input_time_format == 'SCLK':
        input_time_field_number = config['tm_file'][0]['data'][0]['scet']
    else:
        input_time_field_number = int(config['tm_file'][0]['data'][0]['utc'])

    #
    # We obtain the delimiter character
    #
    delimiter = config['tm_file'][0]['delimiter']

    #
    # We obtain the number of data fields and its correspondance
    #
    input_data_field_numbers = [int(config['tm_file'][0]['data'][0]['qx']),
                                int(config['tm_file'][0]['data'][0]['qy']),
                                int(config['tm_file'][0]['data'][0]['qz']),
                                int(config['tm_file'][0]['data'][0]['qs'])]

    tm_list = []
    previous_row_time = ''

    sclk_partition = None  # Just for caching purposes
    sclk_delimiter = None  # Just for caching purposes

    filter_flag = False
    index = 0
    row_prev = []
    sclk_fraction_prev = ''
    sclk_fraction_pprev = ''
    sclk_fraction_ppprev = ''
    sclk_fraction_pppprev = ''
    sclk_initial = ''
    with open(tm_file, 'r', encoding='utf-8') as t:

        for line in t:

            index += 1

            if header_lines == 0 or header_lines < index:

                row_data = []

                # We need to remove the end of line character and the the trailing spaces:
                line = line.split('\n')[0]
                line = ' '.join(line.split())

                try:
                    if ',' in delimiter:

                        if input_time_format == 'SCLK':
                            row_time, sclk_partition, sclk_delimiter = get_sclk_row_time(line,
                                                                                         input_time_field_number,
                                                                                         delimiter,
                                                                                         sclk_partition,
                                                                                         sclk_delimiter,
                                                                                         kernel)
                        elif input_time_format == 'TDB':

                            if 'Z' in line:
                                row_time = str(line.split(delimiter)[input_time_field_number-1]).strip()[0:-1]
                            else:
                                row_time = str(line.split(delimiter)[input_time_field_number-1])
                            row_time = row_time.replace('T', '-') + 'TDB'
                        else:
                            if 'Z' in line:
                                row_time = str(line.split(delimiter)[input_time_field_number-1]).strip()[0:-1]
                            else:
                                row_time = str(line.split(delimiter)[input_time_field_number-1])

                        if ' ' in row_time:
                            if input_time_format == 'SCLK':
                                row_time = row_time.replace(' ', '')
                            else:
                                row_time = row_time.replace(' ', 'T')

                        for data_element_field_number in input_data_field_numbers:

                            #
                            # Solar Orbiter specific processing
                            #
                            if kernel.config['spacecraft'] == 'solo':
                                if 'EVENT' in line:
                                    row_data.append('EVENT')
                                else:
                                    row_data.append(float(line.split(',')[
                                                              data_element_field_number - 1]))
                            else:
                                row_data.append(float(line.split(',')[data_element_field_number-1]))

                    else:

                        line = line.strip()

                        if input_time_format == 'SCLK':
                            row_time, sclk_partition, sclk_delimiter = get_sclk_row_time(line,
                                                                                         input_time_field_number,
                                                                                         delimiter,
                                                                                         sclk_partition,
                                                                                         sclk_delimiter,
                                                                                         kernel)
                        elif input_time_format == 'TDB':
                            if 'Z' in line:
                                row_time = str(line.split()[input_time_field_number-1]).strip()[0:-1]
                            else:
                                row_time = str(line.split()[input_time_field_number-1])
                            row_time = row_time.replace('T', '-') + 'TDB'
                        else:
                            if 'Z' in line:
                                row_time = str(line.split()[input_time_field_number-1]).strip()[0:-1]
                            else:
                                row_time = str(line.split()[input_time_field_number-1])

                        if ' ' in row_time:
                            if input_time_format == 'SCLK':
                                row_time = row_time.replace(' ', '')
                            else:
                                row_time = row_time.replace(' ', 'T')

                        for data_element_field_number in input_data_field_numbers:
                            #
                            # We need to check that
                            #
                            row_data.append(float(line.split()[data_element_field_number-1]))
                except:
                    info_incomplete += 1
                    # logging.info('   HM TM Processing: Found incomplete data line in line {}:'.format(index))
                    # logging.info('   {}'.format(line))
                    continue

                row = row_time + ' '

                # As indicated by Boris Semenov in an e-mail "ROS and MEX "measured" CKs"
                # sometimes the scalar value is negative and the sign of the rest of the
                # components of the quaternions needs to be changed!
                if 'EVENT' not in row_data:
                    if row_data[-1] < 0:
                        neg_data = [-x for x in row_data]
                        info_negative += 1
                        # logging.info(f'   HM TM Processing: Found negative QS on input line {row_data}:')
                        # logging.info('   ' + str(neg_data))
                        row_data = neg_data

                for element in row_data:

                    row += str(element) + ' '

                # We filter out "bad quaternions"

                row += '\n'

                # We remove the latest entry if a time is duplicated
                if row_time == previous_row_time:
                    info_duplicate += 1
                    # logging.info(f'   HM TM Processing: Found duplicate time at {row_time}')

                else:
                    # We do not include the entry if one element equals 1 or gt 1
                    append_bool = True
                    for quaternion in row_data:
                        if 'EVENT' not in row_data:
                            if quaternion > 1.0:
                                append_bool = False
                                info_gt_one += 1
                                # logging.info(f'   HM TM Processing: Found quaternion GT 1 on input line {row_data}:')
                                # logging.info('   ' + str(row))

                    if kernel.input_processing == 'True':

                        if kernel.config['spacecraft'] == 'tgo':
                            (append_bool, line, row, row_prev, index,
                             tm_list, filter_flag, sclk_fraction_prev,
                             sclk_fraction_pprev, sclk_fraction_ppprev,
                             sclk_fraction_pppprev, sclk_initial, info_quaternions) = \
                                quaternions_filter(append_bool, line, row,
                                                   row_prev, index, tm_list,
                                                   filter_flag, sclk_fraction_prev,
                                                   sclk_fraction_pprev, sclk_fraction_ppprev,
                                                   sclk_fraction_pppprev, sclk_initial, info_quaternions)

                    else:
                        append_bool = True

                    if append_bool:
                        tm_list.append(row)

                previous_row_time = row_time

    #
    # We report the number of exceptions here
    #
    if info_duplicate:
        logging.warning(f'   HK TM Processing: Found {info_duplicate} duplicate(s)')
    if info_gt_one:
        logging.warning(f'   HK TM Processing: Found {info_gt_one} quaternion(s) GT 1')
    if info_negative:
        logging.warning(f'   HK TM Processing: Found {info_negative} negative QS')
    if info_incomplete:
        logging.warning(f'   HK TM Processing: Found {info_incomplete} incomplete data line(s)')
    if info_quaternions:
        logging.warning(f'   HK TM Processing: Found {info_quaternions} coarse quaternion(s)')

    # We remove the carriage return from the last line if present
    try:
        last_line = tm_list[-1].split('\n')[0]
        tm_list = tm_list[:-1]
        tm_list.append(last_line)

        return tm_list

    except:

        return tm_list


def hk_tm2ck_reader(kernel, tm_file, config):

    #
    # We init the variables to log the processing
    #
    info_duplicate = 0
    info_incomplete = 0

    # Initialise timestamp variables
    first_row_timestamp = None
    timestamp_seconds_field_number = None
    timestamp_microseconds_field_number = None

    #
    # We obtain the time filed number from the configuration file
    # It has to be a list for there can be two columns
    #
    input_time_format = config['time_format']
    filename_time = ""

    if input_time_format == 'SCLK':
        input_time_field_number = int(config['data'][0]['scet'])

    elif input_time_format == 'UTC':
        #
        # If there is no 'utc' in the 'data' of the TM input, then the time
        # is present in the input FILENAME, and there is only one data element
        # hence we extract the time from there
        #
        if 'utc' in config['data'][0]:
            input_time_field_number = int(config['data'][0]['utc'])
        else:
            input_time_field_number = False

            #
            # For the time being we only need this for CaSSIS, so we only
            # include this option
            #
            filename_time = get_cassis_time_from_filename(tm_file)

    elif input_time_format == 'TIMESTAMP':

        # Check that TIMESTAMP Configuration is valid
        if 'source_date_format' not in config:
            raise Exception("Configuration key 'source_date_format': 'start_index|date_length|strptime_format' "
                            + "is required when using 'time_format: TIMESTAMP', missing for file: " + tm_file)

        if 'timestamp_seconds' not in config['data'][0]:
            raise Exception("Configuration key 'timestamp_seconds' is required when using " +
                            "'time_format: TIMESTAMP', missing for file: " + tm_file)

        if 'timestamp_microseconds' not in config['data'][0]:
            raise Exception("Configuration key 'timestamp_microseconds' is required when using " +
                            "'time_format: TIMESTAMP', missing for file: " + tm_file)

        timestamp_seconds_field_number = int(config['data'][0]['timestamp_seconds'])
        timestamp_microseconds_field_number = int(config['data'][0]['timestamp_microseconds'])

        # Get data start time from filename and convert to et
        filename_time = get_date_from_filename(tm_file, config['source_date_format'])
        filename_time = filename_time.strftime("%Y-%m-%dT%H:%M:%S.%f")
        filename_time = get_et_from_utc(filename_time, kernel)

    else:
        raise Exception('HK TM Processing: Unsupported input_time_format: ' + input_time_format)

    # Check if attached xml to get a more precise start time exists.
    # This will also include the Xml file to the extra sources to be moved to processed directory
    # For the moment only for NOMAD LNO
    if 'xml_date_node' in config:
        filename_time = get_et_from_xml_date_node(tm_file, config['xml_date_node'], kernel, filename_time)

    #
    # Prepare data_factor and rot_axis
    #
    data_factors, rot_axes = get_data_factor_and_rot_axes(config, tm_file)

    #
    # We obtain the delimiter character and header lines
    #
    delimiter = config['delimiter']
    header_lines = int(config['header_lines'])

    #
    # We obtain the number of data fields and its correspondence
    #
    input_data_field_number = int(config['data'][0]['ang'])

    tm_list = []
    previous_row_time = ''

    sclk_partition = None  # Just for caching purposes
    sclk_delimiter = None  # Just for caching purposes

    index = 0
    with open(tm_file, 'r') as t:

        for line in t:

            index += 1

            if header_lines == 0 or header_lines < index:

                row_data = []

                # We need to remove the end of line character:
                line = line.split('\n')[0]

                try:
                    if ',' in delimiter:

                        if input_time_format == 'SCLK':
                            row_time, sclk_partition, sclk_delimiter = get_sclk_row_time(line,
                                                                                         input_time_field_number,
                                                                                         delimiter,
                                                                                         sclk_partition,
                                                                                         sclk_delimiter,
                                                                                         kernel)

                        elif input_time_format == 'UTC':
                            if input_time_field_number:
                                if 'Z' in line:
                                    row_time = str(line.split(delimiter)[input_time_field_number - 1]).strip()[0:-1]
                                else:
                                    row_time = str(line.split(delimiter)[input_time_field_number - 1])

                            else:
                                row_time = filename_time

                        elif input_time_format == 'TIMESTAMP':

                            row_timestamp_seconds = str(line.split(delimiter)[timestamp_seconds_field_number - 1]).strip()
                            row_timestamp_microseconds = str(line.split(delimiter)[timestamp_microseconds_field_number - 1]).strip()
                            row_timestamp = float(row_timestamp_seconds + "." + row_timestamp_microseconds)

                            if not first_row_timestamp:
                                first_row_timestamp = row_timestamp

                            row_time_et = str((row_timestamp - first_row_timestamp) + filename_time)
                            row_time = get_utc_from_et(row_time_et, kernel)

                        if ' ' in row_time:
                            if input_time_field_number:
                                if input_time_format == 'SCLK':
                                    row_time = row_time.replace(' ', '')
                                else:
                                    row_time = row_time.replace(' ', 'T')
                            else:
                                row_time = filename_time

                        row_data.append(float(line.split(',')[input_data_field_number-1]))

                    else:
                        proc_line = line.strip()

                        if input_time_format == 'SCLK':
                            input_time_field_number = int(input_time_field_number)
                            row_time = str(proc_line.split()[input_time_field_number - 1])
                            if ' ' in row_time:
                                row_time = row_time.replace(' ', '')

                        elif input_time_format == 'UTC':
                            row_time = str(proc_line.split()[input_time_field_number-1]).strip()
                            row_time = row_time.replace('Z', '')
                            if ' ' in row_time:
                                row_time = row_time.replace(' ', 'T')
                                row_time = row_time.replace('Z', '')

                        elif input_time_format == 'TIMESTAMP':
                            row_timestamp_seconds = str(line.split()[timestamp_seconds_field_number - 1]).strip()
                            row_timestamp_microseconds = str(line.split()[timestamp_microseconds_field_number - 1]).strip()
                            row_timestamp = float(row_timestamp_seconds + "." + row_timestamp_microseconds)

                            if not first_row_timestamp:
                                first_row_timestamp = row_timestamp

                            row_time_et = str((row_timestamp - first_row_timestamp) + filename_time)
                            row_time = get_utc_from_et(row_time_et, kernel)
                            if ' ' in row_time:
                                row_time = row_time.replace(' ', '').strip()

                        row_data.append(float(proc_line.split()[input_data_field_number-1]))

                except:
                    # logging.info('   fHM TM Processing: Found incomplete data line in line {index}:')
                    # logging.info('   {}'.format(line))
                    info_incomplete += 1
                    continue

                row = row_time + ' '

                #
                # In principle there is only one element for this type of TM
                #
                for element in row_data:

                    #
                    # If we need to factor the data like for example for ACS
                    #
                    ang_rot1 = 0.0
                    ang_rot2 = 0.0
                    ang_rot3 = 0.0

                    for df_idx in range(len(data_factors)):

                        if rot_axes[df_idx] == "1":
                            ang_rot1 = evaluate_data_factor(data_factors[df_idx], element)

                        elif rot_axes[df_idx] == "2":
                            ang_rot2 = evaluate_data_factor(data_factors[df_idx], element)

                        elif rot_axes[df_idx] == "3":
                            ang_rot3 = evaluate_data_factor(data_factors[df_idx], element)

                    row += '{} {} {}'.format(ang_rot1, ang_rot2, ang_rot3)

                #
                # We filter out the input
                #

                # We remove the latest entry if a time is duplicated
                if row_time == previous_row_time:
                    # logging.info(f'   HM TM Processing: Found duplicate time at {row_time}')
                    info_duplicate += 1
                else:
                    append_bool = True

                if append_bool:
                    tm_list.append(row)

                    #
                    # This is the CaSSIS filter, which introduces another element
                    # after 5 seconds only applies to "file_schema":"cas_raw_sc"
                    #
                    if config['file_schema'] == 'cas_raw_sc':
                        utc = row.split(config['delimiter'])[0]
                        et = get_et_from_utc(utc, kernel) + 5.0
                        utc = get_utc_from_et(et, kernel)
                        tm_list.append(utc+' '+' '.join(row.split()[1:]))

                append_bool = False
                previous_row_time = row_time

    #
    # We report the number of exceptions here
    #
    if info_duplicate:
        logging.warning(f'   HK TM Processing: Found {info_duplicate} duplicate(s)')
    if info_incomplete:
        logging.warning(f'   HK TM Processing: Found {info_incomplete} incomplete data line(s)')

    return tm_list


def get_data_factor_and_rot_axes(config, tm_file):
    # Check that data_factor and rot_axis are aligned in case of that data_factor is a list.
    # In this case, each data_factor element will be applied to the given rot_axis element. Eg:
    # "data_factor":["180.0", "45.0"]
    # "rot_axis": ["2", "1"]
    if isinstance(config['data_factor'], list) or isinstance(config['rot_axis'], list):
        if isinstance(config['rot_axis'], list) == isinstance(config['rot_axis'], list):
            if len(config['data_factor']) == len(config['rot_axis']):
                return config['data_factor'], config['rot_axis']
            else:
                raise Exception("Configuration key 'data_factor' and 'rot_axis' are misaligned," +
                                " both keys must have the same number of elements. At file: " + tm_file)
        else:
            raise Exception("Configuration key 'data_factor' and 'rot_axis' are misaligned," +
                            " 'rot_axis' must be a list with same number of elements as 'data_factor'." +
                            " At file: " + tm_file)
    else:
        # If data_factor and rot_axis are strings, convert them to lists
        return [config['data_factor']], [config['rot_axis']]


def evaluate_data_factor(data_factor, element):
    if data_factor != 'False':
        if data_factor.startswith("="):
            # If data factor expression start with "="
            # just assign the returned evaluated value
            return eval(data_factor[1:])
        else:
            # Else multiply by the evaluated value
            return element * eval(data_factor)
    else:
        return element


def get_cassis_time_from_filename(tm_file):
    filename_time = tm_file.split('-')[-4]
    filename_time = filename_time[0:4] + '-' + \
                    filename_time[4:6] + '-' + \
                    filename_time[6:9] + \
                    filename_time[9:11] + ':' + \
                    filename_time[11:13] + ':' + \
                    filename_time[13:15] + \
                    ('.' + filename_time[15:] if filename_time[15:] else '')

    return filename_time


def get_et_from_utc(utc, kernel):
    cmd = kernel.directories.executables + os.sep + 'utc2tdb ' + str(utc) \
          + ' ' + kernel.directories.kernels + os.sep + kernel.config['lsk']
    command_line_process = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output = command_line_process.communicate()[0]
    return float(process_output.decode('utf-8'))


def get_utc_from_et(et, kernel):
    cmd = kernel.directories.executables + os.sep + 'tdb2utc ' + str(et) \
          + ' ' + kernel.directories.kernels + os.sep + kernel.config['lsk']
    command_line_process = subprocess.Popen(cmd, shell=True,
                                            stdout=subprocess.PIPE,
                                            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output = command_line_process.communicate()[0]
    utc = str(process_output.decode('utf-8')).split('\n')[0].strip()
    return utc


def get_et_from_xml_date_node(tm_file, xml_date_node_config, kernel, default_et):
    #
    # Check passed xml_date_node_config
    #
    valid_config = False
    if isinstance(xml_date_node_config, dict):
        if ("xml_node" in xml_date_node_config) and ("date_format" in xml_date_node_config):
            valid_config = True

    if not valid_config:
        raise Exception("Configuration key 'xml_date_node' not valid. Eg: " +
                        "'xml_date_node':{'xml_node':'start_date_time','date_format':'%Y%m%dT%H%M%S.%fZ'}," +
                        " At file: " + tm_file)

    #
    # Read xml file
    #
    tm_name, tm_extension = os.path.splitext(tm_file)
    xml_file = tm_name + ".xml"
    starttime_et = default_et

    if os.path.exists(xml_file):

        # load the xml file
        with open(xml_file) as e:

            xml_node_start = "<{}>".format(xml_date_node_config["xml_node"])
            xml_node_end = "</{}>".format(xml_date_node_config["xml_node"])
            xml_node_found = False

            # Scan the xml file line by line to avoid include a dependency to bs4 module
            for line in e:

                if xml_node_start in line:

                    # Start date node found, extract date
                    line = line.replace(xml_node_start, "")
                    line = line.replace(xml_node_end, "")
                    line = line.strip()

                    starttime = datetime.strptime(line, xml_date_node_config["date_format"])
                    starttime = starttime.strftime("%Y-%m-%dT%H:%M:%S.%f")
                    starttime_et = get_et_from_utc(starttime, kernel)
                    xml_node_found = True
                    break

            if not xml_node_found:
                logging.warning('   HK TM Processing: ' + xml_node_start + ' not found in xml file: ' + xml_file)
    else:
        logging.warning('   HK TM Processing: xml_date_node specified for file: ' + tm_file + ' , ' +
                        'but linked xml not found: ' + xml_file)

    return starttime_et


def hk_tm2ck_dafcat_reader(kernel, tm_file, config, dafcat_element):

    #
    # We init the variables to log the processing
    #
    info_duplicate = 0
    info_incomplete = 0

    #
    # We obtain the time filed number from the configuration file
    # It has to be a list for there can be two columns
    #
    input_time_format = config['tm_file'][0]['time_format']

    if input_time_format == 'SCLK':
        input_time_field_number = config['tm_file'][0]['data'][0]['scet']
    else:
        input_time_field_number = int(config['tm_file'][0]['data'][0]['utc'])

    #
    # We obtain the delimiter character and header lines
    #
    delimiter = config['tm_file'][0]['delimiter']
    header_lines = int(config['tm_file'][0]['header_lines'])

    #
    # We obtain the number of data fields and its correspondence
    #
    if dafcat_element == 1:
        input_data_field_number = int(config['tm_file'][0]['data'][0]['ang1'])
        rot_axis = config['tm_file'][0]['rot_axis1']
    elif dafcat_element == 2:
        input_data_field_number = int(config['tm_file'][0]['data'][0]['ang2'])
        rot_axis = config['tm_file'][0]['rot_axis2']
    else:
        logging.warning(f'   HK TM Processing: Invalid dafcat_element: ' + str(dafcat_element))

    tm_list = []
    previous_row_time = ''
    previous_row_data = ''

    sclk_partition = None  # Just for caching purposes
    sclk_delimiter = None  # Just for caching purposes

    is_phebus = kernel.config["object_name"] == "Phebus"

    index = 0
    with open(tm_file, 'r') as t:

        for line in t:

            index += 1

            if header_lines == 0 or header_lines < index:

                row_data = []

                # We need to remove the end of line character:
                line = line.split('\n')[0]

                try:
                    if ',' in delimiter:

                        if input_time_format == 'SCLK':
                            row_time, sclk_partition, sclk_delimiter = get_sclk_row_time(line,
                                                                                         input_time_field_number,
                                                                                         delimiter,
                                                                                         sclk_partition,
                                                                                         sclk_delimiter,
                                                                                         kernel)
                        else:
                            row_time = str(line.split(delimiter)[input_time_field_number-1]).strip()
                            row_time = row_time.replace('Z', '')

                        if ' ' in row_time:
                            if input_time_format == 'SCLK':
                                row_time = row_time.replace(' ', '')
                            else:
                                row_time = row_time.replace(' ', 'T')
                                row_time = row_time.replace('Z', '')

                        row_data.append(float(line.split(',')[input_data_field_number-1]))

                        if is_phebus:

                            if dafcat_element == 1:
                                angle_direction = float(line.split(',')[int(config['tm_file'][0]['data'][0]['dir1'])-1])
                            elif dafcat_element == 2:
                                angle_direction = float(line.split(',')[int(config['tm_file'][0]['data'][0]['dir2'])-1])

                            if angle_direction == 0:
                                angle_direction = -1
                    else:
                        proc_line = line.strip()

                        if input_time_format == 'SCLK':
                            input_time_field_number = int(input_time_field_number)
                            row_time = str(proc_line.split()[input_time_field_number - 1])
                        else:
                            row_time = str(proc_line.split()[input_time_field_number-1]).strip()
                            row_time = row_time.replace('Z', '')

                        if ' ' in row_time:
                            if input_time_format == 'SCLK':
                                row_time = row_time.replace(' ', '')
                            else:
                                row_time = row_time.replace(' ', 'T')
                                row_time = row_time.replace('Z', '')

                        row_data.append(float(proc_line.split()[input_data_field_number-1]))

                except:
                    # logging.info(f'   HM TM Processing: Found incomplete data line in line {index}:')
                    # logging.info('   {}'.format(line))
                    info_incomplete += 1
                    continue

                row = row_time + ' '

                #
                # In principle there is only one element for this type of TM
                #
                for element in row_data:

                    element = evaluate_data_factor(config['tm_file'][0]['data_factor'], element)

                    #
                    # For BepiColombo Phebus data
                    #
                    if is_phebus and angle_direction != 0:
                        #
                        # interpolation points have to be calculated
                        # to account for the direction of the scanner
                        #
                        if previous_row_time:
                            previous_dt = datetime.strptime(previous_row_time, "%Y-%m-%dT%H:%M:%S.%f")
                            current_dt = datetime.strptime(row_time, "%Y-%m-%dT%H:%M:%S.%f")
                            deltat = abs((current_dt - previous_dt).seconds)
                            if deltat < float(kernel.config['interpolation_interval']) \
                                    and abs(element - previous_row_data) > 45:
                                #
                                # take 1 seconds steps
                                #
                                for i in range(1, deltat, 1):
                                    #
                                    # in the case current angle is higher than previous
                                    #
                                    if (element - previous_row_data) > 0:
                                        #
                                        # clock-wise direction
                                        #
                                        if angle_direction > 0:
                                            element_i = previous_row_data + angle_direction \
                                                        * (element - previous_row_data) / deltat * i
                                        #
                                        # counter clock-wise direction
                                        #
                                        else:
                                            element_i = previous_row_data - \
                                                        (360 + angle_direction * (element - previous_row_data)) \
                                                        / deltat * i
                                    #
                                    # in the case current angle is lower than previous
                                    #
                                    elif (element - previous_row_data) < 0:
                                        #
                                        # clock-wise direction
                                        #
                                        if angle_direction < 0:
                                            element_i = previous_row_data - angle_direction \
                                                        * (element - previous_row_data) / deltat * i
                                        #
                                        # counter clock-wise direction
                                        #
                                        else:
                                            element_i = previous_row_data + \
                                                        (360 + angle_direction * (element - previous_row_data)) \
                                                        / deltat * i
                                    #
                                    # bound angle to 0 <-> 360 range
                                    #
                                    if element_i > 360:
                                        element_i -= 360
                                    elif element_i < 0:
                                        element_i += 360

                                    angles = phebus_fov_reader(element_i, dafcat_element)
                                    time_i = (previous_dt + timedelta(seconds=i)).strftime("%Y-%m-%dT%H:%M:%S.%f")
                                    tm_list.append(time_i + ' ' + f'{angles[2]} {angles[1]} {angles[0]}\n')

                        angles = phebus_fov_reader(element, dafcat_element)
                        row += f'{angles[2]} {angles[1]} {angles[0]}\n'

                    else:

                        ang_rot1 = 0.0
                        ang_rot2 = 0.0
                        ang_rot3 = 0.0

                        if rot_axis == "1":
                            ang_rot1 = element
                        elif rot_axis == "2":
                            ang_rot2 = element
                        elif rot_axis == "3":
                            ang_rot3 = element

                        row += f'{ang_rot1} {ang_rot2} {ang_rot3}\n'

                #
                # We filter out the input
                #

                # We remove the latest entry if a time is duplicated
                if row_time == previous_row_time:
                    # logging.info('   HM TM Processing: Found duplicate time at {row_time}')
                    info_duplicate += 1
                    append_bool = False
                else:
                    append_bool = True

                if append_bool:
                    tm_list.append(row)

                append_bool = False
                previous_row_time = row_time
                previous_row_data = element

    #
    # We report the number of exceptions here
    #
    if info_duplicate:
        logging.warning(f'   HK TM Processing: Found {info_duplicate} duplicate(s)')
    if info_incomplete:
        logging.warning(f'   HK TM Processing: Found {info_incomplete} incomplete data line(s)')

    return tm_list


def event_reader(event_file, config):

    event_flag = False    # Event flag definition
    event_list = []
    event_strg = ''
    with open(event_file, 'r') as e:
        for line in e:

            if event_flag and '/>' in line:
                event_flag = False
                if 'PTEL' in event_file:
                    event_strg += '{} '.format(line.strip())
                event_list.append(event_strg)
                event_strg = ''

            if '<CALIB_ROLL_MAG' in line or '<CALIB_ROLL_RPW' in line or '<ROLL' in line:
                event_flag = True

            if event_flag:
                event_strg += '{} '.format(line.strip())

    return event_list


def sclk_reader(kernel):

    sclk_delimiter = ''
    sclk_partition = ''

    if kernel.k_type != 'sclk':
        k_type = ''
    else:
        k_type = kernel.k_type + os.sep

    try:
        sclk_kernel = open(os.path.join(kernel.directories.kernels + os.sep + k_type + kernel.config['sclk']), 'r')
    except:
        sclk_kernel = open(os.path.join(kernel.directories.output + os.sep + k_type + kernel.config['sclk']), 'r')

    with sclk_kernel:
        for line in sclk_kernel:
            if 'SCLK01_OUTPUT_DELIM' in line:
                if line.split(')')[0].split('(')[-1].strip() == '1':
                    sclk_delimiter = '.'
                else:
                    sclk_delimiter = ':'

            if 'SCLK_PARTITION_START' in line:
                if ')' in line:
                    sclk_partition = '1'
                else:
                    logging.error(f'   {kernel.config["sclk"]} has more than one partition!')
                    raise

    if not sclk_delimiter or not sclk_partition:
        logging.error(f'   {kernel.config["sclk"]} is incorrect!')
        raise

    return sclk_partition, sclk_delimiter


# Extracts the SCLK time from a file row. Also returns the sclk_partition and sclk_delimiter for
# caching purposes only.
def get_sclk_row_time(line, input_time_field_number, delimiter, sclk_partition, sclk_delimiter, kernel):

    fields = line.split(delimiter)
    if ',' in str(input_time_field_number):
        # If line contains the SCLK in divided in two columns: "558057691, 34768" , obtain the partition
        # and return the SCLK in String format

        if not sclk_partition:
            # If sclk_partition and sclk_delimiter not obtained yet, obtain them
            # and return them to avoid this computation again.

            # TODO: The SCLK Kernel is supposed to have only one partition, otherwise
            #       sclk_reader will raise an exception. If more partitions are required,
            #       one workaround could be to use the CHRONOS util to obtain the partition
            #       from a given time.

            sclk_partition, sclk_delimiter = sclk_reader(kernel)

        first_col_idx = int(input_time_field_number[0]) - 1
        second_col_idx = int(input_time_field_number[2]) - 1

        sclk_time = sclk_partition + '/' + str(fields[first_col_idx]) + sclk_delimiter + str(fields[second_col_idx])

    else:
        # If line contains the SCLK time in string format: 1/0630115375.18204, return as is:
        sclk_time = str(fields[int(input_time_field_number) - 1])

    return sclk_time, sclk_partition, sclk_delimiter
