#!/usr/bin/python
#
# $ Script     adcs_orbnumgen (Generate ORBNUM files for ESA Planetary missions)
#
# $ Abstract
#
#     This script is a temporary script to produce ORBNUM files while ADCSng is in
#     development. It should be seen as a prototype of the ORBNUM generation
#     functionality of ADCSng
#
#
# $ Usage
#
#    adcs_orbnumgen.py
#
#
# $ Brief_I/O
#
#     Parameter  I/O  Description
#     ---------  ---  --------------------------------------------------
#     -          -    -
#
# $ Detailed_Input
#
#     -
#
#
# $ Detailed_Output
#
#
#     -
#
# $ Author_and_Institution
#
#     Alfredo Escalante Lopez (ESAC/ESA)
#
# $ Version
#
# -    adcs_orbnumgen Version 0.1.2, 14-MAR-2017 (MCS)
#
#        The BLACKOUT_WINDOW definition has been updated and now includes the
#        possibility of excluding a window covered by the MK but not by the SPK
#        Input file.
#
# -    adcs_orbnumgen Version 0.1.1, 06-MAR-2017 (MCS)
#
#        Thomas Roatsch found a bug: When the last line of the input ORBNUM contains
#        an unknown Apocenter tag, it needs to be discarded and the info of the input
#        SPK file needs to be used.
#
# -    adcs_orbnumgen Version 0.1.0, 24-FEB-2017 (MCS)
#
#        First release.
#
# -&
import subprocess
import glob
import logging
import os
import shutil
import sys


from adcsng.utils.files import fill_template
from adcsng.utils.time import utc2et
from adcsng.utils.time import month_num2str
from adcsng.core.config import get_latest_kernel

def orbnumgen(config, directories, input_spk, mk):

    cwd = os.getcwd()
    os.chdir(input_spk.working_directory)

    #
    # Executes the function to create the kernels
    #
    logging.info(
            '- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -')
    logging.info('+ Executing ORBNUM for SPK {}: '.format(str(input_spk.name)))

    setup = config['orbnum'][0]

    meta_kernel = directories.output + '/mk/' + mk

    if input_spk.orbnum == "merged":

        kernel_type = 'orbnum'
        path = directories.kernels
        pattern = setup['general'][0]['filename_schema']

        input_orbnum = get_latest_kernel(kernel_type, path, pattern)

    else:
        input_orbnum = ''

    #
    # The orbnumgen configuration is loaded
    #
    tol = float(setup['general'][0]['tolerance'])

    #
    # We derive the filename of the output ORBNUM file. The logic is as follows:
    #
    #    (a) if there is an input SPK and no input ORBNUM file the name is the same as the input
    #        SPK but changing the extension from .bsp/.BSP to .orb/.ORB,
    #
    #    (b) If there is an input SPK and an input ORBNUM file the schema given as a parameter in the configuration
    #        file is used,
    #
    if input_orbnum:

        orbnum_filename = setup['general'][0]['filename_schema']

        #
        # We obtain the version for the ORBNUM filename from the input SPK
        #
        # DISCLAIMER: PLEASE NOTE THAT THIS IS VERY MUCH TALYORED FOR MARS-EXPRESS for a SPK file like:
        #
        #    ORMM_T19_170201000000_01318.BSP
        #
        version = str(input_spk.name).split('.')[-2]
        version = version.split('_')[-1]

        orbnum_filename = orbnum_filename.replace('*', version)

    else:
        orbnum_filename = str(input_spk.name)

        if '/' in orbnum_filename:
            orbnum_filename = orbnum_filename.split('/')[-1]

        if orbnum_filename.split('.')[-1] == 'bsp':
            orbnum_filename = orbnum_filename.split('.')[0] + '.orb'

        if orbnum_filename.split('.')[-1] == 'BSP':
            orbnum_filename = orbnum_filename.split('.')[0] + '.ORB'

    #
    # We start with an initial_orbit to 1
    #
    initial_orbit = '1'

    #
    # We parse the meta-kernel (MK) to determine the kernels to load and to determine the exclusion period to run
    # ORBNUM
    #
    with open(meta_kernel, 'r') as mk:

        meta_kernel_list = ''
        #
        # This flag is to indicate when to extract information form the MK. This is information between the
        # usual SPICE keywords:
        #
        #   \begintdata
        #     ...
        #   \begintext
        #
        copy_flag = False

        for line in mk:

            if r'\begintext' in line:
                copy_flag = False

            if copy_flag and line.strip() != '':

                #
                # The path in the variable:
                #
                #    PATH_VALUES = ('..')
                #
                # is substituted by the input variable 'directories.kernels'
                #
                if '..' in line:
                    line = line.replace('..', directories.kernels)
                    pathdir = line.split('\'')[1].split('\\')[0] + '/'
                    pathgen = line.replace('=', '+=').replace('ker_dir', 'ker_gen').split('\'')[1].split('\\')[0] + '/'
                    line = line + line.replace('=', '+=').replace('ker_dir', 'ker_gen')

                if 'PATH_SYMBOLS' in line:
                    line = line + line.replace('=', '+=').replace('KERNELS', 'KER_GEN')

                if '$KERNELS' in line:
                    if not os.path.exists(line.replace('$KERNELS', pathdir).split('\'')[1].split('\\')[0]):
                        line = line.replace('$KERNELS', '$KER_GEN')

                if ('_T19' in line or '_t19' in line) and '$KER_GEN' in line:
                    line = ''

                meta_kernel_list = meta_kernel_list + line

            if r'\begindata' in line:
                copy_flag = True

        #
        # We create the dictionary to fill the 'KERNELS_TO_LOAD' parameter in the ORBNUM Preferences file
        #
        meta_kernel_dict = {'meta_kernel': meta_kernel_list + '     KERNELS_TO_LOAD   +=  ' \
                                               + "'" + str(input_spk.name) + "'"}

    #
    # We call the fill_template function to fill the ORBNUM Preferences file template as specified in the
    # adcs_orbnumgen configuration
    #
    fill_template(template=directories.templates+'/orbnum.pref',
                  file='temporary.pref',
                  replacements=setup['preferences_template'][0])

    #
    # We need to call the function again to add the 'KERNELS_TO_LOAD' files
    # and cleanup BLACKOUT if not present
    #
    fill_template(template='temporary.pref',
                        file='temporary.pref',
                        replacements=meta_kernel_dict,
                        cleanup=True)

    lsk = directories.kernels + '/' + input_spk.config['lsk']

    #
    # If there is an orbnum with an 'unable to determine' the current SPK
    # will *NOT* contain that line and we need to add it by using the
    # coverage in the meta-kernel and by start the computation at the 'unable
    # to determine' start time
    #
    if input_orbnum:
        with open(directories.kernels + '/orbnum/' + input_orbnum, 'r') as orb:
            for line in reversed(list(orb)):
                if line.strip() != '':
                    utc_start_orbnum = line.split()[1] + ' ' + line.split()[2] + ' ' + \
                        line.split()[3] + ' ' +line.split()[4]
                    break
        try:
            utc_start_spk = input_spk.utc_start
        except:
            input_spk.set_times_in_name()
            utc_start_spk = input_spk.utc_start

        #
        # We check if we are using an SPK that finishes before the ORBNUM file.
        #
        cmd = directories.executables + os.sep + 'utc2tdb "' + \
              utc_start_orbnum + '" ' + lsk
        command_line_process = subprocess.Popen(cmd, shell=True,
                                                stdout=subprocess.PIPE,
                                                stderr=subprocess.STDOUT)

        process_output = command_line_process.communicate()[0]
        et_start_orbnum = float(process_output.decode('utf-8'))

        #
        # We need to substract a margin to the utc_start_orbnum to account
        # for month change
        #
        orbnum_margin = 360
        et_start_orbnum -= orbnum_margin

        cmd = directories.executables + os.sep + 'tdb2utc "' + \
              str(et_start_orbnum) + '" ' + lsk
        command_line_process = subprocess.Popen(cmd, shell=True,
                                                stdout=subprocess.PIPE,
                                                stderr=subprocess.STDOUT)

        process_output = command_line_process.communicate()[0]
        utc_start_orbnum = process_output.decode('utf-8')
        utc_start_orbnum = utc_start_orbnum.split('-')[0] + ' ' + \
                           month_num2str(utc_start_orbnum.split('-')[1]) + ' ' + \
                           utc_start_orbnum.split('-')[2].split('T')[0] + ' ' + \
                           utc_start_orbnum.split('T')[-1].strip()

        cmd = directories.executables + os.sep + 'utc2tdb "' + \
              utc_start_spk + '" ' + lsk
        command_line_process = subprocess.Popen(cmd, shell=True,
                                                stdout=subprocess.PIPE,
                                                stderr=subprocess.STDOUT)

        process_output = command_line_process.communicate()[0]
        et_start_spk = float(process_output.decode('utf-8'))

        if et_start_spk > et_start_orbnum:
            utc_start = utc_start_orbnum
        else:
            utc_start = utc_start_spk

    else:
        try:
            utc_start = input_spk.utc_start
        except:
            input_spk.set_times_in_name()
            utc_start = input_spk.utc_start

    #
    # For the time being ORBNUMGEN execution will stop and the end of the SPK
    # file considered
    #
    utc_finish = input_spk.utc_finish

    utc_start = utc_start+'\n'
    utc_finish = utc_finish+'\n'

    #
    # ORBNUM is run. For more details on the utility description please see:
    #    doc/orbnum.ug
    # or visit:
    #    https://naif.jpl.nasa.gov/pub/naif/utilities/MacIntel_OSX_64bit/orbnum.ug
    #
    command_line_process = subprocess.Popen([directories.executables + '/orbnum',
                                             '-pref', 'temporary.pref',
                                             '-num', initial_orbit,
                                             '-file', 'temporary.orb'],
                                            stdout=subprocess.PIPE,
                                            stdin=subprocess.PIPE,
                                            stderr=subprocess.STDOUT)

    command_line_process.stdin.write(utc_start.encode())
    command_line_process.stdin.write(utc_finish.encode())

    process_output, _ = command_line_process.communicate()
    log = process_output.decode('utf-8')

    log = log.split('\n')
    for line in log:
        if line.strip() != '':
            logging.info('   ' + line)

    #
    # If we have provided an input ORBNUM file then we need to join the input ORBNUM file with the one that we
    # have just generated. In order to do so we need to sort out:
    #
    #    (a) The first line in the input ORBNUM file which overlaps with the new ORBNUM temporary file
    #        (a.1) If the line that overlaps contains an unresolved APO/PER event, the previous line is used.
    #    (b) Once we have (a) we need to add all the entries of the new ORBNUM file to the input ORBNUM file
    #        to generate the final ORBNUM file.
    #    (c) In order to do so we also need to update the column No. (for orbit number) in the new file
    #
    if input_orbnum:

        with open(orbnum_filename, 'w+') as new_orb:
            with open('temporary.orb', 'r') as temp_orb:
                for line in temp_orb:

                    #
                    # We get the first orbit number entry of the temporary ORBNUM file in order to obtain the
                    # ephemeris time (ET/TDB) of the first orbit
                    #
                    if '    1  ' in line:

                        event_utc = line.lstrip().split('  ')[1]
                        event_et = utc2et(directories.executables, event_utc, lsk)


                    #
                    # We get the second orbit number entry of the temporary ORBNUM file in order to obtain the
                    # ephemeris time (ET/TDB) of the second orbit and to determine the orbital period for this
                    # pair or orbits
                    #
                    if '    2' in line:
                        next_event_utc = line.lstrip().split('  ')[1]
                        next_event_et = utc2et(directories.executables, next_event_utc, lsk)
                        orbital_period = next_event_et - event_et

                        break
            #
            # Now we open the input ORBNUM file to start appending the output ORBNUM file
            #
            with open(directories.kernels + '/orbnum/' + input_orbnum, 'r') as orb:

                #
                # We use the count to skip the header of the file
                #
                count = 0
                for line in orb:

                    #
                    # We start checking the orbit ET times after the second line (the first two lines are headers
                    # of the ORBNUM file)
                    #
                    if count >= 2:
                        old_event_utc = line.lstrip().split('  ')[1]
                        old_event_et = utc2et(directories.executables, old_event_utc, lsk)

                        #
                        # If the times difference between the temporary ORBNUM file first entry and the first overlap
                        # entry of the input ORBNUM file is lower than a 'tol'n-th of the computed Orbital Period then
                        # we exit the loop.
                        #
                        if abs(event_et - old_event_et) <= orbital_period / tol:
                            break

                        #
                        # If the line contains an undetermined complementary event (APO/PER) then we exit the loop in
                        # order to avoid it being present in the output ORBNUM file. Note that if such event exists it
                        # also indicates the end of the input ORBNUM file.
                        #
                        if 'Unable to determine' in line:
                            break

                        # We update the initial orbit variable at every loop until we reach the ET in which the
                        # temporary ORBNUM file starts. Once this happens we have the last orbit number of the input
                        # ORBNUM file
                        #
                        initial_orbit = int(line.lstrip().split('  ')[0])

                    new_orb.write(line)
                    count += 1

            with open('temporary.orb', 'r') as temp_orb:

                #
                # We use the count to skip the header of the file
                #
                count = 0
                for line in temp_orb:

                    #
                    # We update the orbit number (No. column) in the temporary ORBNUM file and then we append it to
                    # the output ORBNUM file
                    #
                    if count >= 2:
                        orbit_number = int(line[0:5].lstrip()) + initial_orbit
                        line = '{0:05}'.format(orbit_number) + line[5:]
                        new_orb.write(line)
                    count += 1

    else:
        os.rename('temporary.orb', orbnum_filename)


    orbnum_path = os.path.join(directories.output, 'orbnum')

    #
    # Release the orbnum file to the output area.
    #
    try:
        shutil.copy(orbnum_filename, os.path.join(orbnum_path,orbnum_filename))
    except:
        logging.error('')
        logging.error('{} directory does not exist'.format(orbnum_path))
        logging.error('ORBNUM file not generated.'.format(orbnum_path))
        logging.error('')
        raise

    if input_orbnum:
        moved_orbnum = []
        orbnum_path = os.path.join(directories.output, 'orbnum')
        os.chdir(orbnum_path)
        orbnums_in_dir = glob.glob(setup['general'][0]['filename_schema'])

        for orbnum_in_dir in orbnums_in_dir:
            if orbnum_in_dir != orbnum_filename:
                moved_orbnum.append(orbnum_in_dir)
                try:

                    shutil.move(orbnum_in_dir, os.path.join(directories.output,
                                                        'orbnum/former_versions'))

                except:
                    logging.info(
                        'ORBNUM {} already present in former_versions directory'.format(
                            orbnum_in_dir))

    os.chdir(cwd)

    return orbnum_filename