import os
import shutil
import subprocess
import collections
import logging
import glob
import fnmatch
import re
import copy
import math

import pandas as pd

from pathlib import Path
from datetime import datetime, timedelta

from adcsng.core.setup import setup_production, make_setup_file
from adcsng.core.config import get_version_number, get_latest_kernel, \
                               cks_are_in_different_rotation_period, get_ck_kernel_coverage_from_path, \
                               get_ck_kernel_coverage_with_paths
from adcsng.core.input import process_input
from adcsng.utils.files import fill_template, kernel_type2extension, commnt_read, commnt_add
from adcsng.core.config import get_description

import adcsng.classes.kernel as Kernels
from adcsng.classes.comments import get_pedigree_files, set_pedigree_files, \
                                    get_dafcat_comments_header, \
                                    set_aprox_time_coverage


def kernel_generation(kernels, directories, debug, kernel_names=[], setup=''):

    temp_dirs = []

    #
    # We need to create another list for we will remove items from the list
    #
    for kernel in kernels:

        #
        # Set the temporary execution directory
        #
        cwd, working_directory = setup_production(kernel,
                                                  directories,
                                                  directories.source)


        #
        # We add the temporary directory to the list of temporary directories
        #
        temp_dirs.append(working_directory)

        #
        # Create the binary file and update the kernel name with the
        # start/end dates.
        #
        logging.info('+ Processing input:')
        if isinstance(kernel.source, list):
            for source in kernel.source:
                #
                # The CK generated from SPKs with default attitude don't have
                # sources.
                #
                if kernel.source:
                    logging.info('   {}'.format(source))
        else:
            if kernel.source:
                logging.info('   {}'.format(kernel.source))

        #
        # We do pre-processing of the input when needed.
        #
        process_input(kernel, setup, directories, kernels)

        if kernel.input_type == 'OEM':
            naif_util = 'oem2spk'
        elif kernel.input_type == 'OEM_MERGED':
            naif_util = 'oem2spk'
        elif kernel.input_type == 'OEM_TO_MERGE':
            naif_util = 'oem2spk'
        elif kernel.input_type == 'AEM':
            naif_util = 'aem2ck'
        elif kernel.input_type == 'OASW_ORBIT':
            naif_util = 'mex2ker_orbit'
        elif kernel.input_type == 'OASW_ATTITUDE':
            naif_util = 'mex2ker_attitude'
        elif kernel.input_type == 'TM_QUATERNIONS':
            naif_util = 'msopck_quat'
        elif kernel.input_type == 'MAPPS_QUATERNIONS':
            naif_util = 'msopck_quat'
        elif kernel.input_type == 'SPECS':
            naif_util = 'prediCkt'
        elif kernel.input_type == 'TM':
            naif_util = 'msopck'
        elif kernel.input_type == 'TM_DAFCAT':
            naif_util = 'msopck'
        elif kernel.input_type == 'EVENTS':
            naif_util = 'prediCkt'
        elif kernel.input_type == 'COG':
            naif_util = 'mkspk'
        elif kernel.input_type == 'SEP':
            naif_util = 'msopck'
        elif kernel.input_type == 'CK5':
            naif_util = 'ck5to6'
        elif kernel.input_type == 'SPK13':
            naif_util = 'spk13to19'
        elif kernel.input_type == 'SPK18':
            naif_util = 'spk18to19'
        elif 'AEM_QUAT' in kernel.input_type:
            naif_util = 'msopck_quat'
        elif 'BSUP' in kernel.input_type:
            naif_util = 'msopck_bsup'
        elif kernel.input_type == 'TCP_MEX':
            naif_util = 'tcp2scet'
        elif kernel.input_type == 'TCP':
            naif_util = 'None'
        else:
            naif_util = ''

        #
        # Create the configuration file in the temporary directory
        #
        if naif_util and naif_util != 'ck5to6' and naif_util != 'spk18to19' and naif_util != 'spk13to19' and naif_util != 'None':
            make_setup_file(directories, kernel, naif_util)

            #
            # for MEX TCP two NAIF utils are needed tcp2scet and makclk
            #
            if naif_util == 'tcp2scet':
                kernel.config['kernel_full_name'] = str(kernel.name)
                make_setup_file(directories, kernel, 'makclk')

            #
            # Executes the function to create the kernels
            #
            logging.info('+ Executing {} for {}: '.format(naif_util.upper(), kernel.source))

            log = create_kernel(directories, kernel, naif_util)

            log = log.split('\n')
            for line in log:
                if line.strip() != '':
                    if '******************' not in line:
                        logging.info('   ' + line)
            logging.info('')

            #
            # Completes the name of the kernel according to its coverage
            #
            kernel_temporary_name = str(kernel.name)

            if 'date}' in str(kernel.name):
                kernel.set_times_in_name()

            #
            # Update the version number of the kernel to be created.
            # With this the name of the kernel is complete.
            #
            version = get_version_number(kernel.k_type,
                                                directories.kernels,
                                                str(kernel.name))

            try:
                kernel.name.set('version', version)
            except:
                pass

            #
            # Renames the kernel after getting version in name
            #
            shutil.move(str(kernel_temporary_name), str(kernel.name))

            #
            # Comments the kernel (not required for SPECS kernels)
            #
            if kernel.input_type != 'SPECS' and kernel.input_type != 'EVENTS' and kernel.input_type != 'TCP_MEX':
                #
                # Compute coverage for kernel without date in the name
                #
                if kernel.k_type == 'spk' or kernel.k_type == 'ck':
                    kernel.coverage()
                kernel.comment_kernel()
                #
                # Specs kernel inherit the description field of the name from
                # the original SPK.
                #
            elif kernel.input_type == 'SPECS':
                kernel.name.set('description', kernel.config['description'])

            #
            # If the CK Kernel is for Planning, we add the MSOPCK segment to it
            #
            if kernel.input_type == 'AEM' and '_fict_' in kernel.config['sclk'] or \
                kernel.input_type == 'OASW_ATTITUDE' and '_fict_' in kernel.config['sclk'] or \
                  kernel.input_type == 'SPECS' and '_fict_' in kernel.config['sclk']:

                with open('msopck_plan.input', 'w', newline='\n') as i:
                    # kernel.coverage(utc=True) -> already done by kernel.set_times_in_name()
                    coverage = kernel.config['coverage'].strip().split('\n')

                    start_date = \
                    coverage[0].split('Begin UTC: ')[-1].split('  End')[0]
                    finish_date = coverage[-1].split('End UTC: ')[-1]


                    #
                    # We need to substract 1 minute from the finish time
                    # and add one to the start time.
                    #
                    dt = datetime.strptime(finish_date,
                                           '%Y-%b-%d %H:%M:%S.%f')
                    difference = timedelta(minutes=-1)
                    finish_dt = dt + difference
                    finish_date = finish_dt.strftime(
                        "%Y-%m-%d %H:%M:%S.%f").replace(' ','T')

                    dt = datetime.strptime(start_date,
                                           '%Y-%b-%d %H:%M:%S.%f')
                    difference = timedelta(minutes=+1)
                    start_dt = dt + difference
                    start_date = start_dt.strftime(
                        "%Y-%m-%d %H:%M:%S.%f").replace(' ','T')

                    i.write(start_date +  ' 0 0 0 1 {}'.format(os.linesep))
                    i.write(finish_date + ' 0 0 0 1 {}'.format(os.linesep))

                logging.info('')
                logging.info('+ Executing MSOPCK for Planning CK {}: '.format(naif_util.upper(),
                                                                       kernel.source))

                #
                # For MSOPCK we need to load the step SCLK
                #
                kernel.config['sclk'] =  'sclk/' + \
                                         get_latest_kernel('sclk', [directories.kernels, directories.output],
                                                                  setup['generic'][0][kernel.config['spacecraft']][0]['sclk_step_schema'])
                #
                # this is not done via kernel configuration for SPECS
                #
                try:
                    shutil.copy(os.path.join(directories.kernels, kernel.config['sclk']),
                        os.path.join(kernel.working_directory, kernel.config['sclk']))
                except:
                    shutil.copy(os.path.join(directories.output, kernel.config['sclk']),
                                os.path.join(kernel.working_directory, kernel.config['sclk']))

                make_setup_file(directories, kernel, 'msopck_plan')
                log = create_kernel(directories, kernel, 'msopck_plan')

                log = log.split('\n')
                for line in log:
                    if line.strip() != '':
                        logging.info('   ' + line)

        elif naif_util == 'None':

            #
            # This basically is indicating that its a SCLK because we have
            # no NAIF util associated with
            if kernel.config['previous_sclk_separator'] != kernel.config['sclk_separator']:
                logging.error('')
                logging.error(
                        f'   Previous SCLK and SCLK configuration have different delimiter specifications')
                logging.error(
                        f'   The one from the pervious SCLK -> {kernel.config["previous_sclk_separator"]} will be used.')


            # Only for ExoMarsRSP
            # Update the version number of the kernel to be created.
            # With this the name of the kernel is complete.
            #
            if 'rm' in kernel.config['spacecraft']:

                version = get_version_number(kernel.k_type,
                                                    directories.kernels,
                                                    str(kernel.name))

                kernel.name.set('version',version)

            #
            #
            # Executes the function to create the kernels
            #
            logging.info('')
            logging.info('+ Generating SCLK for {}: '.format(kernel.source))

            (generation_flag) = create_sclk(directories, kernel, debug)

            if not generation_flag:
                return [], kernels, temp_dirs

        elif not kernel.input_type == 'C-SMITHED':
            #
            # This basically is indicating that its a CK, CK5, SPK13 or SPK18
            # and no setup is needed
            #
            logging.info('')
            logging.info('+ Kernel input {}'.format(kernel.source))

    #
    # Generate C-SMITHED CK
    #
    kernels_iter = kernels.copy()
    for kernel in kernels_iter:
        if kernel.input_type == 'C-SMITHED':

            cwd = os.getcwd()
            os.chdir(kernel.working_directory)

            log = merge_cks(kernel, kernel.working_directory)

            os.chmod(str(kernel.name), 0o644)

            #
            # Executes the function to convert the kernel
            #
            logging.info('')
            logging.info('+ Merging C-Smithed CKs:')
            for line in log.split('\n'):
                logging.info('    {}'.format(line))

            os.chdir(cwd)

    #
    # We repeat the loop for DAFCAT kernels, kernels that need merging and
    # kernels that need conversion (e.g.: SPK type13 to type19)
    #
    dafcated_kernel = ''
    kernels_iter = kernels.copy()

    #
    # 1. HK_TM DAFCAT kernels for HGA and Solar Arrays
    #
    for kernel in kernels_iter:
        #
        # We merge the kernels that need to be merged
        #
        if kernel.dafcat:
            for kernel_dafcat in kernels:
                if kernel_dafcat != kernel:
                    merged_kernel_name = str(kernel.name).split('_{}.'.format(str(kernel.dafcat)))[0]
                    merged_kernel_dafcat_name = str(kernel_dafcat.name).split('_{}.'.format(str(kernel_dafcat.dafcat)))[0]
                    if  merged_kernel_name ==  merged_kernel_dafcat_name:

                        #
                        # Executes the function to create the kernels
                        #
                        logging.info('')
                        logging.info('+ Executing DAFCAT for HK_TM kernel:')
                        logging.info('     {}'.format(str(kernel_dafcat.name)))
                        logging.info('     {}'.format(str(kernel.name)))

                        (merged_kernel, log) = merge_tm2ck(kernel_dafcat, kernel)

                        log = log.split('\n')
                        for line in log:
                            if line.strip() != '':
                                logging.info('   ' + line)

                        kernels.remove(kernel_dafcat)
                        kernel.name = merged_kernel

                        if not debug:
                            if not isinstance(kernel.source_filename, list):
                                kernel.source_filename = [kernel.source_filename]

                            sources_to_remove = kernel.source_filename

                            # Include possible files attached to source ones,
                            # like an .xml file with .tab for NOMAD LNO
                            sources_to_remove.extend(kernel.extra_sources)

                            for source in sources_to_remove:
                                if os.path.exists(os.path.join(directories.source, str(source))):
                                    os.remove(os.path.join(directories.source, str(source)))

                        break


    #
    # 2. AEM DAFCAT kernels, in principle for ExoMarsRSP
    #
    kernels_iter = kernels.copy()
    kernels_aem_dafcat = []
    kernels_aem_merged_dafcat = []
    for kernel in kernels_iter:
        #
        # We merge the kernels that need to be merged
        #
        if 'input_type' in kernel.config.keys():
            if kernel.config['input_type'] == 'AEM_QUAT':
                #
                # We only consider one kernel type to merge and the filename
                #
                pattern = re.compile('.+[_][t][e][m][p].+')
                if pattern.match(str(kernel.name)):
                    kernels_aem_dafcat.append(kernel)

            elif kernel.config['input_type'] == 'AEM_QUAT_MERGED':
                kernels_aem_merged_dafcat.append(kernel)

    #
    # 2.1 AEM DAFCAT kernels: Merging all CKs for mechanisms in a single CK
    #
    if kernels_aem_dafcat:

        #
        # If there is only one kernel, nothing needs to be done except
        # to update the name.
        #
        if len(kernels_aem_dafcat) == 1:
            kernel_name = f"{kernels_aem_dafcat[0].name.tokens['kernel_name']}_{kernels_aem_dafcat[0].name.tokens['description']}.bc"
            shutil.move(os.path.join(kernels_aem_dafcat[0].working_directory,str(kernels_aem_dafcat[0].name)),
                        os.path.join(kernels_aem_dafcat[0].working_directory,kernel_name))
            kernels_aem_dafcat[0].name.xkma = kernel_name
        else:
            #
            # Executes the function to create the kernels
            #
            logging.info('')
            logging.info('+ Executing DAFCAT for AEM_QUAT kernels:')
            for kernel_aem_dafcat in kernels_aem_dafcat:
                logging.info('+    {}'.format(str(kernel_aem_dafcat.name)))

            (merged_kernel, log) = merge_aemquat(kernels_aem_dafcat)

            log = log.split('\n')
            for line in log:
                if line.strip() != '':
                    logging.info('   ' + line)

            kernels.append(merged_kernel)

            for kernel_aem_dafcat in kernels_aem_dafcat:
                kernels.remove(kernel_aem_dafcat)


    #
    # 2.2 AEM_MERGED DAFCAT kernels: Merging all the interpolated and extended telemetry
    #
    if kernels_aem_merged_dafcat:

        #
        # If there is only one kernel, nothing needs to be done, except
        # to update the name.
        #
        if len(kernels_aem_merged_dafcat) == 1:
            kernel_name = f"{kernels_aem_merged_dafcat[0].interp_config[0]['kernel_name']}_{kernels_aem_merged_dafcat[0].name.tokens['description']}.bc"
            shutil.move(os.path.join(kernels_aem_merged_dafcat[0].working_directory, str(kernels_aem_merged_dafcat[0].name)),
                        os.path.join(kernels_aem_merged_dafcat[0].working_directory, kernel_name))
            kernels_aem_merged_dafcat[0].name.xkma = kernel_name
        else:
            #
            # Executes the function to create the kernels
            #
            logging.info('')
            logging.info('+ Executing DAFCAT for AEM_QUAT_MERGED kernels:')
            for kernel_aem_merged_dafcat in kernels_aem_merged_dafcat:
                logging.info('+    {}'.format(str(kernel_aem_merged_dafcat.name)))

            (merged_kernel, log) = merge_aemquat(kernels_aem_merged_dafcat)

            log = log.split('\n')
            for line in log:
                if line.strip() != '':
                    logging.info('   ' + line)

            kernels.append(merged_kernel)

            for kernel_aem_merged_dafcat in kernels_aem_merged_dafcat:
                kernels.remove(kernel_aem_merged_dafcat)


    #
    # 3. HK_TM QUAT DAFCAT kernels, in principle for Solar Orbiter
    #
    kernels_iter = kernels.copy()
    kernels_intervals_dafcat = {}
    for kernel in kernels_iter:
        #
        # We merge the kernels that need to be merged
        #
        if 'merge_intervals' in kernel.config:
            if kernel.config['merge_intervals'] == "True":

                #
                # We only consider one kernel type to merge and the filename
                #
                pattern = re.compile('.+[_][a][d][c][s][n][g].+')
                if pattern.match(str(kernel.source)):
                    kernel_intervals_pattern = re.search('[_][0-9]{2,}[_][_][a][d][c][s][n][g]', str(kernel.source)).group()
                    k_i_p_key = str(kernel.source).split(kernel_intervals_pattern)[0]
                    if k_i_p_key in kernels_intervals_dafcat.keys():
                        kernels_intervals_dafcat[k_i_p_key].append(kernel)
                    else:
                        kernels_intervals_dafcat[k_i_p_key] = [kernel]

    if kernels_intervals_dafcat:

        kernels_intervals_dafcat = collections.OrderedDict(sorted(kernels_intervals_dafcat.items()))
        for kernel_intervals_dafcat in kernels_intervals_dafcat.items():
                #
                # Executes the function to create the kernels
                #
                logging.info('')
                logging.info('+ Executing DAFCAT for TM_QUATERNIONS kernels:')
                for kernel_interval_dafcat in kernel_intervals_dafcat[1]:
                    logging.info('+    {}'.format(str(kernel_interval_dafcat.name)))

                (merged_kernel, temp_dirs, log) = merge_tmquat(kernel_intervals_dafcat[1], temp_dirs)

                log = log.split('\n')
                for line in log:
                    if line.strip() != '':
                        logging.info('   ' + line)

        #
        # We need to remove from the loop the temporary directories that we do not need anymore
        #
        for kernel_intervals_dafcat in kernels_intervals_dafcat.items():
            for kernel_interval_dafcat in kernel_intervals_dafcat[1]:
                pattern = re.compile('.+[_][0-9]{2,}[.][b][c]')
                if pattern.match(str(kernel_interval_dafcat.name)):
                    kernels.remove(kernel_interval_dafcat)


    #
    # 3.1. BSUP DAFCAT kernels, in principle for Solar Orbiter
    #
    kernels_iter = kernels.copy()
    kernels_boresights_dafcat = {}
    for kernel in kernels_iter:
        #
        # We merge the kernels that need to be merged
        #
        if 'input_type' in kernel.config and kernel.config['input_type'] == 'BSUP':
            pattern = re.compile('.+[_][t][e][m][p].+')
            if pattern.match(str(kernel.name)):
                detector = '_adcsng_temp'
                kernel_boresights_pattern = re.search("%s" % detector, str(kernel.source)).group()
                k_i_p_key = str(kernel.source).split(kernel_boresights_pattern)[0]
                if k_i_p_key in kernels_boresights_dafcat.keys():
                    kernels_boresights_dafcat[k_i_p_key].append(kernel)
                else:
                    kernels_boresights_dafcat[k_i_p_key] = [kernel]

    if kernels_boresights_dafcat:

        kernels_boresights_dafcat = collections.OrderedDict(sorted(kernels_boresights_dafcat.items()))
        for kernel_boresights_dafcat in kernels_boresights_dafcat.items():
            #
            # Executes the function to create the kernels
            #
            logging.info('')
            logging.info('+ Executing DAFCAT for BSUP kernels:')
            for kernel_interval_dafcat in kernel_boresights_dafcat[1]:
                logging.info('+    {}'.format(str(kernel_interval_dafcat.name)))

            (merged_kernel, temp_dirs, log) = merge_bsupang(kernel_boresights_dafcat[1], temp_dirs)

            log = log.split('\n')
            for line in log:
                if line.strip() != '':
                    logging.info('   ' + line)

        #
        # We need to remove from the loop the temporary directories that we do not need anymore
        #
        for kernel_boresights_dafcat in kernels_boresights_dafcat.items():
            for kernel_interval_dafcat in kernel_boresights_dafcat[1]:
                detector = kernel_interval_dafcat.config['detector_name'].upper()
                if re.search("%s" % detector, str(kernel_interval_dafcat.name)) and not 'merged' in str(kernel_interval_dafcat.name):
                    kernels.remove(kernel_interval_dafcat)
                elif kernel_interval_dafcat.config['detector_name'] in str(kernel_interval_dafcat.name):
                    kernels.append(kernel_interval_dafcat)


    #
    # 4. Append kernels with other kernels generated in this run if need be.
    #
    kernels_iter = kernels.copy()
    for kernel in kernels_iter:

        dafcat_bool = True
        if 'merge_kernel' in kernel.config:
            if kernel.config['merge_kernel'] == 'True':

                #
                # Since there are many kernels that only need to be DAFCATed once
                # due to the merge_kernel type, we need to pass for the next kernels
                # in the loop and remove it from the kernels list.

                #
                # This part is for SOLO
                #
                if kernel.config['kernel_name'] == dafcated_kernel:
                    kernels.remove(kernel)
                    continue


                if dafcat_bool:
                    log = merge_kernel(kernel)
                    log = log.split('\n')
                    logging.info('')
                    logging.info(f'+ Merging {kernel.k_type.upper()} kernels')
                    for line in log:

                        if line.strip() != '':
                            logging.info('   ' + line.split(os.sep)[-1])

                    dafcated_kernel = kernel.config['kernel_name']


    #
    # 5. Append kernels with former versions
    #
    kernels_iter = kernels.copy()
    for kernel in kernels_iter:

        if 'append_kernel' in kernel.config:
            if kernel.config['append_kernel'] == 'True':
                log = append_kernel(kernel, setup)
                log = log.split('\n')
                logging.info('')
                logging.info(f'+ Appending {kernel.k_type.upper()} with former version')
                for line in log:
                    if line.strip() != '':
                        logging.info('   ' + line)


    #
    # 6. Generate Type 19 SPKs out of Type 13
    #
    kernels_iter = kernels.copy()
    for kernel in kernels_iter:
        if (kernel.input_type == 'OEM' or kernel.input_type == 'SPK13') and kernel.source_config[1]['type19conversion'] == 'True':
            #
            # Executes the function to convert the kernel
            #
            logging.info('')
            logging.info('+ Executing SPK13TO19 for:')
            logging.info('    {}'.format(str(kernel.name)))

            (log) = convert_spk(kernel)

            log = log.split('\n')
            for line in log:
                if line.strip() != '':
                    logging.info('   ' + line)

    #
    # 7. Generate Type 19 SPKs out of Type 18
    #
    kernels_iter = kernels.copy()
    for kernel in kernels_iter:
        if (kernel.input_type == 'OASW_ORBIT' or kernel.input_type == 'SPK18') and kernel.source_config[1]['type19conversion'] == 'True':
            #
            # Executes the function to convert the kernel
            #
            logging.info('')
            logging.info('+ Executing SPK18TO19 for:')
            logging.info('    {}'.format(str(kernel.name)))

            if str(kernel.name) == str(kernel.name).lower():
                typetoken = 't19'
            else:
                typetoken = 'T19'
            if kernel.source_config[1]['type18_keep'] == 'True':
                (log) = convert_spk(kernel, keepboth=True)

                t19kernel = copy.deepcopy(kernel)
                t19kernel.name = str(kernel.name)[:int(kernel.source_config[1]['type19_token_index'])] + typetoken + \
                                str(kernel.name)[int(kernel.source_config[1]['type19_token_index']):]
                t19kernel.orbnum = 'False'
                kernels.append(t19kernel)
            else:
                (log) = convert_spk(kernel, keepboth=False)
                kernel.name = str(kernel.name)[:int(kernel.source_config[1]['type19_token_index'])] + typetoken + \
                              str(kernel.name)[int(kernel.source_config[1]['type19_token_index']):]

            log = log.split('\n')
            for line in log:
                if line.strip() != '':
                    logging.info('   ' + line)


    #
    # 8. Generate Type 6 CKs out of Type 5
    #
    kernels_iter = kernels.copy()
    for kernel in kernels_iter:
        if (kernel.input_type == 'OASW_ATTITUDE' or kernel.input_type == 'CK5') and kernel.source_config[1]['type6_conversion'] == 'True':
            #
            # Executes the function to convert the kernel
            #
            logging.info('')
            logging.info('+ Executing CK5TO6 for:')
            logging.info('    {}'.format(str(kernel.name)))

            typetoken = kernel.source_config[1]['type6_token']
            if kernel.source_config[1]['type5_keep'] == 'True':
                (log) = convert_ck(kernel, keepboth=True)

                t6kernel = copy.deepcopy(kernel)
                t6kernel.name = str(kernel.name)[:int(kernel.source_config[1]['type6_token_index'])] + typetoken + \
                                str(kernel.name)[int(kernel.source_config[1]['type6_token_index']):]
                kernels.append(t6kernel)
            else:
                (log) = convert_ck(kernel, keepboth=False)
                kernel.name = str(kernel.name)[:int(kernel.source_config[1]['type6_token_index'])] + typetoken + \
                              str(kernel.name)[int(kernel.source_config[1]['type6_token_index']):]

            log = log.split('\n')
            for line in log:
                if line.strip() != '':
                    logging.info('    ' + line)


    #
    # 8. Generate CKs out of CK (we are renaming a CK kernel)
    #
    kernels_iter = kernels.copy()
    for kernel in kernels_iter:
        if kernel.input_type == 'CK':

            cwd = os.getcwd()
            os.chdir(kernel.working_directory)

            shutil.copy(kernel.source, str(kernel.name))

            kernel.set_times_in_name()

            #
            # Update the version number of the kernel to be created.
            # With this the name of the kernel is complete.
            #
            version = get_version_number(kernel.k_type,
                                         kernel.directories.kernels,
                                         str(kernel.name))

            kernel.name.set('version', version)
            shutil.copy(kernel.source, str(kernel.name))
            os.chmod(str(kernel.name), 0o644)

            #
            # Executes the function to convert the kernel
            #
            logging.info('')
            logging.info('+ Renaming CK to:')
            logging.info('    {}'.format(str(kernel.name)))

            os.chdir(cwd)


    #
    # We repeat the loop to move the kernels to the release area
    # This could be moved to a different function. Also with this
    # if there is an interruption in the pipeline partial kernels
    # are not moved to the output area.
    #
    kernels_moved = False
    for kernel in kernels:

        os.chdir(kernel.working_directory)
        #
        # Release the kernels to the output area
        #
        # We also need to move the older version of the kernel to the
        # former_versions directory
        #
        if os.path.exists(os.path.join(directories.output,
                                                        kernel.k_type.lower())):
            shutil.copy(str(kernel.name), os.path.join(directories.output,
                                                        kernel.k_type.lower()))
        else:
            if not kernels_moved:
                logging.info("")
                logging.info("- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -")
                kernels_moved = True
            logging.warning('Output dir: {} has been created.'.format(
                    os.path.join(directories.output,
                                     kernel.k_type.lower())))

            os.mkdir(os.path.join(directories.output,kernel.k_type.lower()))
            shutil.copy(str(kernel.name), os.path.join(directories.output,
                                                       kernel.k_type.lower()))

        #
        # Move the previous version of the kernel to former versions.
        #
        kernel_path = os.path.join(directories.output, kernel.k_type.lower())
        if os.path.exists(os.path.join(kernel_path,'former_versions')) or 'former_versions' in kernel.source_config[1].keys():

            if not os.path.exists(f'{kernel_path}/former_versions'):
                try:
                    os.mkdir(f'{kernel_path}/former_versions')
                    if not kernels_moved:
                        logging.info("")
                        logging.info(
                            "- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -")
                        kernels_moved = True
                    logging.error(
                        'Directory {} did not exist but has been generated.')
                except:
                    if not kernels_moved:
                        logging.info("")
                        logging.info(
                            "- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -")
                        kernels_moved = True
                    logging.fatal(
                            'Directory {} did not exist and cannot be.')

            cwd = os.getcwd()

            os.chdir(os.path.join(directories.output, kernel.k_type.lower()))

            ker_name = str(kernel.name)
            ker_pattern = ker_name[::-1]
            numbers     = re.search(r'\d+', ker_pattern).span()
            ker_pattern = ker_pattern[:numbers[0]] + \
                          '*' + ker_pattern[numbers[1]:]
            ker_pattern = ker_pattern[::-1]

            kers_in_dir = glob.glob(ker_pattern)

            #
            # We add the kernels that regardless of the version number need to be moved to
            # former_versions (implemented for Mars Express)
            #
            ker_pattern = []
            if 'former_versions' in kernel.source_config[1].keys():
                if ',' in kernel.source_config[1]['former_versions']:
                    for pattern in  kernel.source_config[1]['former_versions'].split(','):
                        if fnmatch.fnmatch(ker_name, pattern):
                            ker_pattern = pattern
                else:
                    ker_pattern = kernel.source_config[1]['former_versions']
                kers_in_dir += glob.glob(ker_pattern)

            #
            # We remove the duplicates from the list
            #
            kers_in_dir = list(dict.fromkeys(kers_in_dir))


            for ker_in_dir in kers_in_dir:
                if ker_in_dir != ker_name:

                    try:

                        shutil.move(ker_in_dir, os.path.join(directories.output,
                                    kernel.k_type.lower(),'former_versions'))
                        if not kernels_moved:
                            logging.info("")
                            logging.info(
                                "- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -")
                            kernels_moved = True
                        logging.info(
                            '{} moved into former_versions directory'.format(
                                    ker_in_dir))

                    except:
                        if not kernels_moved:
                            logging.info("")
                            logging.info(
                                "- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -")
                            kernels_moved = True
                        logging.info('{} already present in former_versions directory'.format(ker_in_dir))

            os.chdir(cwd)

        #
        # Move the source file to the output area (not required for SPECS kernels)
        # or if the source file output area is not specified
        #
        if kernel.input_type != 'SPECS' and kernel.input_type != 'CK5' and directories.processed:

            if not isinstance(kernel.source_filename, list):
                kernel.source_filename = [kernel.source_filename]

            sources_to_copy = kernel.source_filename

            # Include possible files attached to source ones,
            # like an .xml file with .tab for NOMAD LNO
            sources_to_copy.extend(kernel.extra_sources)

            for source in sources_to_copy:
                if not isinstance(source, list):
                    if 'adcsng_sources' in kernel.config.keys():
                        try:
                            source = kernel.source_config[0]
                            if not os.path.isdir(os.path.join(directories.processed, kernel.originator)):
                                os.mkdir(os.path.join(directories.processed, kernel.originator))
                                logging.info('')
                                logging.error(f'   The processed directory: {str(source)}')
                                logging.error(f'   Did not exist and has been generated')

                            shutil.copy(os.path.join(directories.source, source), os.path.join(directories.processed, kernel.originator))
                        except:
                            logging.info('')
                            logging.error(f'   The input file is not present:')
                            logging.error(f'      {str(source)}')
                    else:
                        if not os.path.isdir(os.path.join(directories.processed, kernel.originator)):
                            os.mkdir(os.path.join(directories.processed, kernel.originator))
                            logging.info('')
                            logging.error(f'   The processed directory: {str(source)}')
                            logging.error(f'   Did not exist and has been generated')

                        if kernel.input_type != 'BSUP':
                            shutil.copy(str(source), os.path.join(directories.processed, kernel.originator))
                        else:
                            shutil.copy(directories.source + '/' + str(source),
                                        os.path.join(directories.processed, kernel.originator))

                else:
                    for file in source:
                        try:
                            shutil.copy(str(file), os.path.join(directories.processed, kernel.originator))
                        except:
                            logging.info('')
                            logging.error(f'   The input file is not present:')
                            logging.error(f'      {str(source)}')

        #
        # we remove the source from the source directory
        # please note that before this happened at the end of the ADCSng
        # execution but it has been moved here in order to avoid unsuccessful
        # consecutive runs after adcsng crashes.
        #
        if not debug and kernel.input_type != 'SPECS' and kernel.input_type != 'TM_DAFCAT':

            if not isinstance(kernel.source_filename, list):
                kernel.source_filename = [kernel.source_filename]

            sources_to_remove = kernel.source_filename

            # Include possible files attached to source ones,
            # like an .xml file with .tab for NOMAD LNO
            sources_to_remove.extend(kernel.extra_sources)

            for source in sources_to_remove:

                #
                # TCPs end up being lists and therefore the list needs to be checked.
                #
                if isinstance(source, list):
                    for src in source:
                        if os.path.exists(os.path.join(directories.source, str(src))):
                            os.remove(os.path.join(directories.source, str(src)))

                if os.path.exists(os.path.join(directories.source, str(source))):
                    os.remove(os.path.join(directories.source, str(source)))
                #
                # We need to account for the merge_kernels whose source
                # filename is modified, we choose the first instance
                #
                if 'merge_kernel' in kernel.config.keys():
                    if kernel.config['merge_kernel'] == 'True':
                        try:
                            original_source = kernel.source_config[0]
                        except:
                            original_source = '.'.join(str(source).split('_0.'))

                        if os.path.exists(os.path.join(directories.source, original_source)):
                            os.remove(os.path.join(directories.source, original_source))
                    #
                    # We include this else for the AEM QUAT kernels that
                    # are not merged but still they modify their original
                    # input
                    #
                    else:
                        try:
                            original_source = kernel.source_config[0]
                            if os.path.exists(os.path.join(directories.source, original_source)):
                                os.remove(os.path.join(directories.source, original_source))
                        except:
                            pass

        #
        # We keep record of the kernel name for the meta-kernel and for
        # the email body and log
        #
        kernel_names.append(str(kernel.name))

    for kernel in kernels:
        if 'TCP' in kernel.input_type:
            logging.info("- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -")
            break

    return kernel_names, kernels, temp_dirs


def append_kernel(kernel, setup):

    #
    # We need to chose the working directory
    #
    cwd = os.getcwd()
    os.chdir(kernel.working_directory)
    log = ''

    if hasattr(kernel.name, 'xkma'):
        original_name = kernel.name.xkma
    kernel_path = kernel.directories.kernels + os.sep + kernel.k_type + os.sep
    #
    # Here we look for the kernel that we want to append, in order to
    # do so, we need to use the MK grammar
    #
    kernel_grammar = glob.glob(kernel.directories.root_dir + '/config/' + setup['generic'][0]['mission'] + '*.grammar')[0]

    with open(kernel_grammar, 'r') as kg:
        excluded_kernels = []
        for line in kg:
            if 'exclude:' in line:
                excluded_kernels.append(line.split('exclude:')[-1].split('\n')[0])

    extension = kernel.config['kernel_schema'].split('.')[-1]

    if kernel.config['input_type'] == 'BSUP':
        #
        # check detector name for Solar Orbiter BSUP
        #
        kernel_pattern = kernel.config['kernel_name'] + '-' + kernel.config['detector_name'] + '*' + '.' + extension
    else:
        kernel_pattern = kernel.config['kernel_name'] + '*' + '.' + extension

    kernels_to_append = glob.glob(kernel_path + kernel_pattern)

    #
    # First we exclude the 'excluded' kernels and CK kernels for ZERO rotation
    #
    kernels_to_iter = kernels_to_append.copy()
    for file in kernels_to_iter:
        if file.split(os.sep)[-1] in excluded_kernels:
            kernels_to_append.remove(file)
        elif '_zero_' in file:
            kernels_to_append.remove(file)

    # Extract kernel name and kernel name without the version
    if not isinstance(kernel.name, str):
        kernel_name = str(kernel.name)
    else:
        kernel_name = kernel.name

    try:
        kernel_name_no_version = kernel_name.split('_v')[0]
    except:
        kernel_name_no_version = kernel_name.split('_V')[0]

    #
    # Second we exclude the kernel that has the same name
    #
    kernels_to_append_copy = kernels_to_append.copy()
    try:
        for file in kernels_to_append_copy:
            if kernel_name_no_version in file:
                kernels_to_append.remove(file)
                log += 'Former version of the kernel discarded: {}\n'.format(file.split(os.sep)[-1])
    except:
        pass

    # If kernel rotation enabled, exclude all kernels that are not in the same
    # rotation period.
    if "kernel_rotation" in kernel.config:

        if kernel.k_type.upper() == "CK":
            # Obtain the rotation period
            rotation_period = kernel.config["kernel_rotation"].lower()
            if rotation_period not in ["yearly", "monthly", "weekly", "daily"]:
                raise Exception("Invalid value '" + rotation_period + "' for key 'kernel_rotation', supported values "
                                                                      "(Yearly, Monthly, Weekly, Daily) for file: " +
                                kernel.config["kernel_rotation"])

            for file in kernels_to_append_copy:
                if cks_are_in_different_rotation_period(kernel_name, file, rotation_period, kernel.directories):
                    # Remove CK kernel if is in different rotation period
                    kernels_to_append.remove(file)

            # Is possible that the kernel needs to be appended to a previous CK before the setup of the CK rotation,
            # in this case all kernels have been removed and we need to find the latest existing kernel which
            # startdate before the kernel startdate.
            if not len(kernels_to_append) and ("kernel_rotation_since_date" in kernel.config):

                kernel_cov = get_ck_kernel_coverage_from_path(kernel_name, kernel.directories)
                if len(kernel_cov):

                    # Check if kernel to be appended is from before the date when the CK rotation
                    # was setup
                    rotation_since_date = datetime.strptime(kernel.config["kernel_rotation_since_date"],
                                                            '%Y-%b-%d %H:%M:%S.%f')
                    if kernel_cov[0][0] < rotation_since_date:
                        log += 'Looking for suitable previous kernels for {}\n'.format(kernel_name)

                        kernels_to_append_copy.sort()
                        latest_valid_kernel = ''
                        for file in kernels_to_append_copy:

                            file_cov = get_ck_kernel_coverage_from_path(file, kernel.directories)
                            if len(file_cov):
                                if file_cov[0][0] <= kernel_cov[0][0]:
                                    latest_valid_kernel = file
                            else:
                                log += 'Cannot obtain CK coverage for {}\n'.format(file)

                        if len(latest_valid_kernel):
                            log += 'Found suitable previous kernel {} for {}\n'.format(latest_valid_kernel, kernel_name)
                            kernels_to_append.append(latest_valid_kernel)
                        else:
                            log += 'No suitable previous kernel found, creating new kernel.\n'
                else:
                    log += 'Cannot obtain CK coverage for {}\n'.format(kernel_name)

    #
    # Various kernels will be chosen, we now merge with the 'last' in the list
    #
    if kernels_to_append and len(kernels_to_append) > 1:
        kernels_to_append.sort()
        kernels_to_append = [kernels_to_append[-1]]
    elif kernels_to_append:
        kernels_to_append = [kernels_to_append[0]]
    else:
        log += 'No kernel to be appended with {}'.format(kernel.name)
        return log

    #
    # We now update the name of the kernel by generating a new KernelName object
    # We also need to keep the name of the original kernel
    #
    original_kernel = str(kernel.name)

    #
    # if the kernel is a TM_DAFCAT kernel, we need to remove the {dafcat}
    # field from the original name.
    #
    if '_{dafcat}' in kernel.config['kernel_schema']:
        updated_kernel_schema = ''.join(kernel.config['kernel_schema'].split('_{dafcat}'))
    else:
        updated_kernel_schema = kernel.config['kernel_schema']

    kernel.name = Kernels.KernelName(updated_kernel_schema)

    try:
        description = get_description(kernel.source_config[0],
                                      kernel.source_config[1]['input_description_index'])
        kernel.name.set('description', description)
    except:
        pass

    kernel.name.set('kernel_name', kernel.config['kernel_name'])
    if hasattr(kernel.config, 'detector_name') or kernel.config['input_type'] == 'BSUP':
        kernel.name.set('detector_name', kernel.config['detector_name'].lower())
        commnt_name = original_name.split('.')[0]
        cwd = kernel.directories.temp + '/' + kernel.source_filename
    else:
        commnt_name = str(original_kernel)

    if kernel.k_type.lower() == 'ck':
        kernel.name.set('sclkid', kernel.config['sclkid'])

    #
    # We provide the kernel with a temporary name
    #
    appended_kernel = str(kernel.name)

    #
    # Careful: if the file ends up with the same name we need to bypass it
    # with a temporary name for the DAFCAT process.
    #
    if appended_kernel == original_kernel:
        appended_kernel += '_adcsng_temp'

    with open('grouped.daf', 'w') as i:
        for kernel_to_append in kernels_to_append:
            i.write('{}\n'.format(kernel_to_append))
        i.write('{}\n'.format(original_kernel))

    cmd = kernel.directories.executables + os.sep + 'dafcat < grouped.daf ' + appended_kernel
    command_line_process = subprocess.Popen(cmd,
                                            shell=True,
                                            stdout=subprocess.PIPE,
                                            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output = command_line_process.communicate()[0]
    log += process_output.decode('utf-8')

    #
    # Now we need to return the file to the original name
    #
    if '_adcsng_temp' in appended_kernel:
        appended_kernel = appended_kernel.split('_adcsng_temp')[0]
        shutil.move(appended_kernel + '_adcsng_temp', appended_kernel)

    #
    # Here we need to prepare the comments of the appended kernel,
    #
    comment_text = get_dafcat_comments_header(kernels_to_append)
    comment_text += Path(commnt_name.split('.')[0] + '.commnt').read_text()

    # We need to merge the pedigree section on comments of the appended kernels,
    filename_pattern = "*." + str(kernel.config["input_extension"]).lower()
    pedigree_files = []
    for kernel_to_append in kernels_to_append:
        pedigree_files += get_pedigree_files(kernel_to_append, filename_pattern, kernel.directories)
    for src_file in kernel.source_filename:
        if src_file not in pedigree_files:
            pedigree_files.append(src_file)

    comment_text = set_pedigree_files(comment_text, pedigree_files, filename_pattern)

    # And also update the approximate time coverage to set the full one.
    if kernel.k_type.upper() == "CK":
        coverage = get_ck_kernel_coverage_with_paths(appended_kernel,
                                                     kernel.config["sclk"],
                                                     kernel.config["lsk"],
                                                     kernel.config["fk"],
                                                     kernel.directories)
        if len(coverage):
            comment_text = set_aprox_time_coverage(comment_text, coverage[0][0], coverage[-1][1])

    # Write updated comments on kernel file
    Path('dafcat_append.commnt').write_text(comment_text)
    fill_template('dafcat_append.commnt', 'dafcat_append.commnt', kernel.config)
    commnt_add(appended_kernel, 'dafcat_append.commnt', kernel.directories)

    #
    # Now we update the dates for the filename
    #
    kernel.set_times_in_name()

    #
    # Update the version number of the kernel to be created.
    # With this the name of the kernel is complete.
    #
    version = get_version_number(kernel.k_type,
                                 kernel.directories.kernels,
                                 str(kernel.name))

    kernel.name.set('version', version)

    #
    # We want the final name of the DAFCATed file in the log
    #
    log = '\n'.join(log.split('\n')[:-2]) + '\n   ' + str(kernel.name)

    #
    # Renames the kernel after getting version in name
    #
    shutil.move(str(appended_kernel), str(kernel.name))

    os.chdir(cwd)

    return log


def merge_kernel(kernel):

    #
    # We need to chose the working directory
    #
    cwd = os.getcwd()
    os.chdir(kernel.working_directory)

    #
    # We might be appending published kernels or kernels of this ADCSng run
    # (Solar Orbiter measured attitude for instance)
    #
    # Be careful, with SOLO 'merge_kernel' is enough, but not for EMRSP, when
    # we start dealing with this.
    #
    if kernel.config['spacecraft'] == 'solo':
        kernel_path = '../'
    else:
        kernel_path = kernel.directories.kernels + os.sep + kernel.k_type + os.sep

    #
    # Here we look for the kernel that we want to append
    #
    extension = kernel.config['kernel_schema'].split('.')[-1]
    kernel_pattern = kernel.config['kernel_name'] + '*' + '.' + extension

    #
    # Look for uppercase kernels
    #
    if not kernel_pattern:
        extension = kernel_type2extension(kernel.config['kernel_type'],case='upper')
        kernel_pattern = kernel.config['kernel_name'] + '*' + '.' + extension[0]

    kernels_to_append = []
    for path in Path(kernel_path).rglob(kernel_pattern):
        kernels_to_append.append(str(path))


    #
    # Various kernels will be chosen, we merge with all the kernels
    # (so far this only applies to Solar Orbiter)
    #
    if kernels_to_append and len(kernels_to_append) > 1:

        kernel_dict = {}
        for file in kernels_to_append:
            #
            # This one choses the end part of the kernel: {finish_date}_s{sclkid}_v{version}.bc
            #
            kernel_dict['_'.join(file.split('_')[-3:])] = file

        kernels_to_append = list(sorted(list(kernel_dict.values())))

        for file in kernels_to_append:
            pattern = re.compile('.+[_][0-9]{1,}[.][b][c]')
            if not pattern.match(file):
                kernels_to_append.remove(file)

        original_kernel = str(kernel.name)

    elif kernels_to_append:
        kernels_to_append = [kernels_to_append[0]]
        original_kernel = str(kernel.name)

    else:
        log = 'No kernel to be appended with {}'.format(kernel.name)
        return log


    #
    # if the kernel is a TM_DAFCAT kernel, we need to remove the {dafcat}
    # field from the original name.
    #
    if '_{dafcat}' in kernel.config['kernel_schema']:
        updated_kernel_schema = ''.join(kernel.config['kernel_schema'].split('_{dafcat}'))
    else:
        updated_kernel_schema = kernel.config['kernel_schema']


    kernel.name = Kernels.KernelName(updated_kernel_schema)

    try:
        description = get_description(kernel.source_config[0],
                                      kernel.source_config[1]['input_description_index'])
        kernel.name.set('description', description)
    except:
        pass

    kernel.name.set('kernel_name', kernel.config['kernel_name'])

    if kernel.k_type.lower() == 'ck':
        kernel.name.set('sclkid', kernel.config['sclkid'])


    #
    # We provide the kernel with a temporary name
    #
    appended_kernel = str(kernel.name)


    with open('grouped.daf', 'w') as i:
        for kernel_to_append in kernels_to_append:
            i.write('{}\n'.format(kernel_to_append))

    cmd = kernel.directories.executables + os.sep + 'dafcat < grouped.daf ' + str(kernel.name)
    command_line_process = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE,
                          stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log = process_output.decode('utf-8')
    log_list = log.split('\n')
    log = '\n'.join(log_list[:-2])

    with open('dafcat_append.commnt', 'w') as f:

        f.write('\n   This file contains a series of files concatenated with each other.\n ' +
                '  Complete information of each original file is provided below.\n\n\n')


        #
        # We use the comment of the kernel that has been generated
        #
        with open(str(original_kernel).split('.')[0]+'.commnt', 'r') as i:
            for line in i:
                f.write(line)

        for kernel_to_append in kernels_to_append:
            # We extract the comment from the appending kernel
            comments = commnt_read(kernel_to_append, kernel.directories)
            f.write(comments)

    command_line_process = subprocess.Popen(
            [kernel.directories.executables + '/' +
             'commnt', '-a', appended_kernel, 'dafcat_append.commnt'],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output = command_line_process.communicate()[0]

    #
    # Now we update the dates for the filename
    #
    kernel.set_times_in_name()

    #
    # Update the version number of the kernel to be created.
    # With this the name of the kernel is complete.
    #
    version = get_version_number(kernel.k_type,
                                        kernel.directories.kernels,
                                        str(kernel.name))

    kernel.name.set('version', version)
    log += '\n   ' + str(kernel.name)

    #
    # Renames the kernel after getting version in name
    #
    shutil.move(str(appended_kernel), str(kernel.name))

    os.chdir(cwd)

    return log


def merge_tm2ck(kernel_1, kernel_2):

    os.chdir(kernel_1.working_directory)

    merged_kernel = str(kernel_2.name).replace('_{}.'.format(str(kernel_2.dafcat)), '.')

    if os.path.isfile(merged_kernel):
        os.remove(merged_kernel)

    with open('grouped.daf', 'w') as i:
        inputs = glob.glob('*.bc')
        pattern = re.compile('.+[_][0-9]{1,}[.][b][c]')
        if not inputs:
            inputs = glob.glob('*.BC')
            pattern = re.compile('.+[_][0-9]{1,}[.][B][C]')
        for input in inputs:
            if pattern.match(input): i.write(input+'\n')

    cmd = kernel_1.directories.executables + '/' + 'dafcat < grouped.daf ' + merged_kernel
    command_line_process = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE,
                          stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log = process_output.decode('utf-8')


    # We call the adcs_fill_template function to fill the template file as specified in the
    # adcs_mkgen configuration file 'adcs_mkgen.json'
    #
    fill_template(template=kernel_1.directories.templates+'/dafcat.commnt',
                        file='dafcat.commnt',
                        replacements=kernel_1.config)

    with open('dafcat.commnt', 'a') as f:
        with open(str(kernel_1.name).split('.')[0]+'.commnt', 'r') as i:
            for line in i:
                f.write(line)
        with open(str(kernel_2.name).split('.')[0] + '.commnt', 'r') as i:
            for line in i:
                f.write(line)

    command_line_process = subprocess.Popen(
            [kernel_1.directories.executables + '/' +
             'commnt', '-a', merged_kernel, 'dafcat.commnt'],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log += process_output.decode('utf-8')

    #
    # We need to add this because if the kernel is appended the name of the
    # comments file has to be the same as the resulting kenrel.
    #
    shutil.move('dafcat.commnt', merged_kernel.split('.')[0] + '.commnt')

    return merged_kernel, log


def merge_aemquat(kernels):

    #
    # We chose the working dir based on the first kernel
    #
    os.chdir(kernels[0].working_directory)

    if 'MERGED' in kernels[0].config['input_type']:
        merged_kernel = f"{kernels[0].interp_config[0]['kernel_name']}_{kernels[0].name.tokens['description']}.bc"
    else:
        merged_kernel = f"{kernels[0].name.tokens['kernel_name']}_{kernels[0].name.tokens['description']}.bc"

    if os.path.isfile(merged_kernel):
        os.remove(merged_kernel)

    with open('grouped.daf', 'w') as i:
        with open('dafcat.commnt', 'a') as c:
            for kernel in kernels:
                i.write(os.path.join(kernel.working_directory,str(kernel.name) + '\n'))
                #
                # We also take the chance to join the comments
                #
                with open(os.path.join(kernel.working_directory,str(kernel.name).split('.')[0] + '.commnt'),'r') as d:
                    for line in d:
                        c.write(line)


    cmd = kernels[0].directories.executables + '/' + 'dafcat < grouped.daf ' + merged_kernel
    command_line_process = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE,
                          stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log = process_output.decode('utf-8')

    command_line_process = subprocess.Popen(
            [kernels[0].directories.executables + '/' +
             'commnt', '-a', merged_kernel, 'dafcat.commnt'],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log += process_output.decode('utf-8')

    #
    # We need to add this because if the kernel is appended the name of the
    # comments file has to be the same as the resulting kernel.
    #
    shutil.move('dafcat.commnt', merged_kernel.split('.')[0] + '.commnt')

    #
    # We generate a new temp dir for the merged kernel
    #
    merged_kernel_dir = os.sep.join(kernels[0].working_directory.split(os.sep)[0:-1]) + os.sep + merged_kernel
    try:
        shutil.copytree(kernels[0].working_directory, merged_kernel_dir)
    except:
        log += f'\n\n By the way {merged_kernel_dir} already exists (DEBUG mode)'

    merged_kernel_name = merged_kernel
    merged_kernel = kernels[0]
    merged_kernel.name.xkma = merged_kernel_name
    merged_kernel.working_directory = merged_kernel_dir
    merged_kernel.source = kernel.source_config[0]
    merged_kernel.source_filename = kernel.source_config[0]

    return merged_kernel, log


def merge_tmquat(kernels, temp_dirs):

    #
    # We chose the working dir based on the first kernel
    #
    if not isinstance(kernels, list):
        kernels = [kernels]

    os.chdir(kernels[0].working_directory)

    merged_kernel = '_'.join(str(kernels[0].name).split('_')[:-1])+'.bc'

    if os.path.isfile(merged_kernel):
        os.remove(merged_kernel)

    with open('grouped.daf', 'w') as i:
        with open('dafcat.commnt', 'a') as c:
            for kernel in kernels:
                i.write(os.path.join(kernel.working_directory,str(kernel.name) + '\n'))
                #
                # We also take the chance to join the comments
                #
                with open(os.path.join(kernel.working_directory,str(kernel.name).split('.')[0] + '.commnt'),'r') as d:
                    for line in d:
                        c.write(line)


    cmd = kernels[0].directories.executables + '/' + 'dafcat < grouped.daf ' + merged_kernel
    command_line_process = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE,
                          stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log = process_output.decode('utf-8')

    command_line_process = subprocess.Popen(
            [kernels[0].directories.executables + '/' +
             'commnt', '-a', merged_kernel, 'dafcat.commnt'],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log += process_output.decode('utf-8')

    #
    # We need to add this because if the kernel is appended the name of the
    # comments file has to be the same as the resulting kernel.
    #
    shutil.move('dafcat.commnt', merged_kernel.split('.')[0] + '.commnt')

    #
    # We generate a new temp dir for the merged kernel
    #
    merged_kernel_dir = os.sep.join(kernels[0].working_directory.split(os.sep)[0:-1]) + os.sep + kernel.source_config[0]
    try:
        shutil.copytree(kernels[0].working_directory, merged_kernel_dir)
        temp_dirs.append(merged_kernel_dir)
    except:
        log += f'\n\n By the way {merged_kernel_dir} already exists (DEBUG mode)'

    merged_kernel_name = merged_kernel
    merged_kernel = kernel

    merged_kernel.name.xkma = '_'.join(merged_kernel.name.xkma.split('_')[0:-1]) + '.' + merged_kernel.name.xkma.split('.')[-1]
    merged_kernel.working_directory = merged_kernel_dir
    merged_kernel.source = kernel.source_config[0]
    merged_kernel.source_filename = kernel.source_config[0]
    os.chdir(merged_kernel_dir)

    #
    # We need to rename the kernel otherwise we cannot check the coverage (kernel not found)
    #
    temp_name = str(merged_kernel.name)
    shutil.move(merged_kernel_dir + os.sep + merged_kernel_name, merged_kernel_dir + os.sep + temp_name)

    merged_kernel.set_times_in_name()

    version = get_version_number(merged_kernel.k_type,merged_kernel.directories.kernels, str(merged_kernel.name))
    merged_kernel.name.set('version', version)


    shutil.move(merged_kernel_dir + os.sep + temp_name, merged_kernel_dir + os.sep + str(merged_kernel.name))

    kernels.append(merged_kernel)

    #
    # We want the final name of the DAFCATed file in the log
    #
    log = '\n'.join(log.split('\n')[:-4]) + '\n   ' + str(merged_kernel.name)

    return merged_kernel, temp_dirs, log


def merge_bsupang(kernels, temp_dirs):

    #
    # We chose the working dir based on the first kernel
    #
    if not isinstance(kernels, list):
        kernels = [kernels]

    os.chdir(kernels[0].working_directory)

    merged_kernel = kernels[0].config['kernel_schema']

    if os.path.isfile(merged_kernel):
        os.remove(merged_kernel)

    with open('grouped.daf', 'w') as i:
        with open('dafcat.commnt', 'a') as c:
            for kernel in kernels:
                i.write(os.path.join(kernel.working_directory,str(kernel.name) + '\n'))
                #
                # We also take the chance to join the comments
                #
                with open(os.path.join(kernel.working_directory,str(kernel.name).split('.')[0] + '.commnt'),'r') as d:
                    for line in d:
                        c.write(line)


    cmd = kernels[0].directories.executables + '/' + 'dafcat < grouped.daf ' + merged_kernel
    command_line_process = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE,
                          stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log = process_output.decode('utf-8')

    command_line_process = subprocess.Popen(
            [kernels[0].directories.executables + '/' +
             'commnt', '-a', merged_kernel, 'dafcat.commnt'],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log += process_output.decode('utf-8')

    #
    # We need to add this because if the kernel is appended the name of the
    # comments file has to be the same as the resulting kernel.
    #
    shutil.move('dafcat.commnt', merged_kernel.split('.')[0] + '.commnt')

    #
    # We generate a new temp dir for the merged kernel
    #
    merged_kernel_dir = os.sep.join(kernels[0].working_directory.split(os.sep)[0:-1]) + os.sep + kernel.source_config[0]
    try:
        shutil.copytree(kernels[0].working_directory, merged_kernel_dir)
        temp_dirs.append(merged_kernel_dir)
    except:
        log += f'\n\n By the way {merged_kernel_dir} already exists (DEBUG mode)'

    merged_kernel_name = merged_kernel
    merged_kernel = copy.deepcopy(kernel)

    merged_kernel.name.xkma = merged_kernel_name
    merged_kernel.working_directory = merged_kernel_dir
    merged_kernel.source = kernel.source_config[0]
    merged_kernel.source_filename = kernel.source_config[0]
    os.chdir(merged_kernel_dir)

    #
    # We need to rename the kernel otherwise we cannot check the coverage (kernel not found)
    #
    temp_name = str(merged_kernel.name)
    shutil.move(merged_kernel_dir + os.sep + merged_kernel_name, merged_kernel_dir + os.sep + temp_name)

    merged_kernel.set_times_in_name()

    version = get_version_number(merged_kernel.k_type,merged_kernel.directories.kernels, str(merged_kernel.name))
    merged_kernel.name.set('version', version)


    shutil.move(merged_kernel_dir + os.sep + temp_name, merged_kernel_dir + os.sep + str(merged_kernel.name))

    kernels.append(merged_kernel)

    #
    # We want the final name of the DAFCATed file in the log
    #
    log = '\n'.join(log.split('\n')[:-4]) + '\n   ' + str(merged_kernel.name)

    return merged_kernel, temp_dirs, log



def merge_cks(kernel, temp_dirs):

    #
    # We chose the working dir based on the first kernel
    #

    os.chdir(kernel.working_directory)

    merged_kernel = str(kernel.name).split('.')[0]+'.bc'

    if os.path.isfile(merged_kernel):
        os.remove(merged_kernel)

    with open('grouped.daf', 'w') as i:
        with open('dafcat.commnt', 'w') as c:
            for source in kernel.source:
                i.write(os.path.join(kernel.working_directory,str(source) + '\n'))

                #
                # We also take the chance to join the comments
                #
                command_line_process = subprocess.Popen(
                        [kernel.directories.executables + '/' +
                         'commnt', '-e', source, source+'.commnt'],
                        stdout=subprocess.PIPE,
                        stderr=subprocess.STDOUT)

                command_line_process.wait()

                with open(source+'.commnt', 'r') as com:
                    comment = com.readlines()

                c.write(''.join(comment))

    cmd = kernel.directories.executables + '/' + 'dafcat < grouped.daf ' + merged_kernel
    command_line_process = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE,
                          stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log = process_output.decode('utf-8')

    command_line_process = subprocess.Popen(
            [kernel.directories.executables + '/' +
             'commnt', '-a', merged_kernel, 'dafcat.commnt'],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)
    command_line_process.wait()
    process_output= command_line_process.communicate()[0]
    log += process_output.decode('utf-8')

    #
    # We need to add this because if the kernel is appended the name of the
    # comments file has to be the same as the resulting kernel.
    #
    merged_kernel = kernel

    merged_kernel.name.xkma = merged_kernel.name.xkma.split('.')[0] + '.' + merged_kernel.name.xkma.split('.')[-1]
    merged_kernel.source = kernel.source_config[0]
    merged_kernel.source_filename = kernel.source_filename

    #
    # We need to rename the kernel otherwise we cannot check the coverage (kernel not found)
    #
    temp_name = str(merged_kernel.name)

    merged_kernel.set_times_in_name()

    version = get_version_number(merged_kernel.k_type,merged_kernel.directories.kernels, str(merged_kernel.name))
    merged_kernel.name.set('version', version)

    shutil.move('dafcat.commnt', str(merged_kernel.name).split('.')[0] + '.commnt')
    shutil.move(kernel.working_directory + os.sep + temp_name, kernel.working_directory + os.sep + str(merged_kernel.name))

    #
    # We want the final name of the DAFCATed file in the log
    #
    log = '\n'.join(log.split('\n')[:-4]) + '\n   ' + str(merged_kernel.name)

    return log


def create_sclk(directories, kernel, debug):

    # Now we build the telemetry table
    telemetry_headers = ('N', 'DDS_Time', 'UTCSTR', 'Version', 'Algorithm', 'SynchroCheck', 'Validity', 'Accuracy',
                         'Synchro', 'Gradient', 'Offset', 'OBT_N', 'UTC_N', 'Display_Offset','StdDev', 'StdErr')

    telemetry_table = pd.read_csv(kernel.source,  delim_whitespace=True, names = telemetry_headers)

    input_tcps = telemetry_table.to_string()

    # This are the elements to be dropped
    dds_time_col = telemetry_table['DDS_Time']
    validity_col = telemetry_table['Validity']
    accuracy_col = telemetry_table['Accuracy']

    #
    # Accuracy and Validity filter is applied
    #
    count = 0
    drop_lines = []


    for element in dds_time_col:
        try:
            if int(validity_col[count]) == 0 or int(accuracy_col[count]) == 0:
                drop_lines.append(count)
        #
        # This is introduced for in Linux the following error appears:
        #    ValueError: cannot convert float NaN to integer
        #
        # We simply remove them after the exception.
        #
        except:
            print(drop_lines)
            drop_lines.append(count)
        count += 1

    if drop_lines:
        for id in drop_lines:
            #
            # Alternate version to simple drop.
            #
            telemetry_table = telemetry_table.drop(telemetry_table.loc[telemetry_table.index == id].index)

    #
    # We now sort the values per OBT_N
    #
    telemetry_table = telemetry_table.sort_values('DDS_Time')

    #
    # We now drop the repeated ones.
    #
    telemetry_table = telemetry_table.drop_duplicates('OBT_N', keep='first')

    used_tcps = telemetry_table.to_string()

    telemetry_elements = telemetry_table['N'].count()


    if telemetry_elements > 0:

        lsk = directories.kernels + '/' + kernel.config['lsk']

        sclkcet_list = []
        sclk = 0
        parsys = 0
        offset = 0
        gradient = 0
        ticks = float(kernel.config['ticks_per_second'])
        et_reference_epoch = float(kernel.config['et_reference_epoch'])

        for row in telemetry_table.itertuples():

            gradient_piter = gradient
            offset_piter = offset
            sclk_piter = sclk
            parsys_piter= parsys

            obt_n =    row[12]
            utc_n =    row[13]
            gradient = row[10]
            offset =   row[11]

            if kernel.config['spacecraft'] == 'rm':
                dds_time = row[2]
                obt_n = dds_time
                utc_n = dds_time

            sclk = (obt_n - offset / gradient) / (1.0 / ticks)

            #
            # utc = utc_n + cspice.str2et('1970 January 1, 00:00:00 TDT')
            #
            utc = utc_n + et_reference_epoch


            #
            # parsys = utc + cspice.deltet(utc, 'UTC')
            #
            cmd = directories.executables + '/' + 'delutc ' + str(utc) + ' ' + lsk
            command_line_process = subprocess.Popen(cmd, shell=True,
                                                    stdout=subprocess.PIPE,
                                                    stderr=subprocess.STDOUT)
            command_line_process.wait()
            process_output = command_line_process.communicate()[0]
            deltet = float(process_output.decode('utf-8'))


            parsys = utc + deltet

            rate = gradient

            try:
                #
                # utc_ins = utc_n - 60.0 + cspice.str2et('1970 January 1, 00:00:00 TDT')
                #
                utc_ins = utc_n - 60    + et_reference_epoch


                #
                # parsys_ins = utc_ins + cspice.deltet(utc_ins,'UTC')
                #
                cmd = directories.executables + '/' + 'delutc ' + str(
                    utc_ins) + ' ' + lsk
                command_line_process = subprocess.Popen(cmd, shell=True,
                                                        stdout=subprocess.PIPE,
                                                        stderr=subprocess.STDOUT)

                process_output = command_line_process.communicate()[0]
                deltet = float(process_output.decode('utf-8'))

                parsys_ins = utc_ins +  deltet

                sclk_ins = sclk_piter + ((parsys_ins - parsys_piter - offset_piter) / (gradient_piter) / (1.0 / ticks))

                rate_ins = 60. / (sclk - sclk_ins) * ticks

                #
                # For the first TCP or if there is only one telemetry element
                # we need to build up the step with the last entry of the
                # former SCLK
                #
            except:
                try:
                    current_sclk = open(kernel.directories.kernels + '/sclk/' +
                          kernel.config['sclk'], 'r')
                except:
                    current_sclk = open(kernel.directories.output + '/sclk/' +
                                        kernel.config['sclk'], 'r')
                with current_sclk:
                    coefficients_flag = False
                    for line in current_sclk:
                        if ')' in line:
                            coefficients_flag = False
                        if coefficients_flag and line.split():
                            last_sclk_entry = line.split()
                        if 'SCLK01_COEFFICIENTS' in line:
                            coefficients_flag = True

                sclk_psclk = float(last_sclk_entry[0])
                parsys_psclk = float(last_sclk_entry[1])
                rate_psclk = float(last_sclk_entry[2])

                sclk_ins = sclk_psclk + ((parsys_ins - parsys_psclk) / (rate_psclk) / (1.0 / ticks))
                rate_ins = 60. / (sclk - sclk_ins) * ticks


            sclkcet_list.append(['{:.13E}'.format(round(sclk_ins)),
                                 '{:.13E}'.format(parsys_ins),
                                 '{:.13E}'.format(rate_ins)])


            sclkcet_list.append(['{:.13E}'.format(round(sclk)),
                                 '{:.13E}'.format(parsys),
                                 '{:.13E}'.format(rate)])


        #
        # We duplicate the list to make it iterable
        #
        sclkcet_new = []

        #
        # Now we determine whether if the kernel needs to be created or not
        # based in whether if the TCP is present in the SCLK or not.
        #
        if not kernel.config['mission_accr'] == 'TEST':

            try:
                current_sclk = open(kernel.directories.kernels+'/sclk/'+kernel.config['sclk'], 'r')
            except:
                current_sclk = open(kernel.directories.output + '/sclk/' + kernel.config['sclk'], 'r')

            sclkcet_current = []
            with current_sclk:
                coefficients_flag = False

                #
                # We generate the list of sclkcet from the current SCLK
                #
                for line in current_sclk:

                    if ')' in line:
                        coefficients_flag = False

                    if coefficients_flag and line.split():
                        sclkcet_current.append(line.split())

                    if 'SCLK01_COEFFICIENTS' in line:
                        coefficients_flag = True

                #
                # We add the new TCPs including the ones prior to the last TCP
                # of the current SCLK.
                #
                step_tcp = False
                for sclkcet_in_current in sclkcet_current:
                    sclkcet_list_iter = list(sclkcet_list)
                    for sclkcet in sclkcet_list_iter:
                        if 'INF' in sclkcet and sclkcet in sclkcet_list:
                            sclkcet_list.remove(sclkcet)
                            logging.error(f'   Spurious TCP: {sclkcet}')
                        else:
                            if math.isclose(float(sclkcet[0]), float(sclkcet_in_current[0]), rel_tol=1e-5):
                                sclkcet_list.remove(sclkcet)
                                if step_tcp:
                                    logging.warning(f'   TCP already present in SCLK: {sclkcet}')
                                    step_tcp = False
                                    continue
                                step_tcp = True

                            elif float(sclkcet[0]) < float(sclkcet_in_current[0]):
                                sclkcet_new.append(sclkcet)
                                sclkcet_list.remove(sclkcet)
                                if step_tcp:
                                    logging.warning(f'   Added TCP prior to latest TCP in current SCLK: {sclkcet}')
                                    step_tcp = False
                                    continue
                                step_tcp = True

                    sclkcet_new.append(sclkcet_in_current)

                sclkcet_new += sclkcet_list

        #
        # We record the last_tcp
        #
        cmd = directories.executables + '/' + 'tdb2utc ' + str(
            sclkcet_new[-1][1]) + ' ' + lsk
        command_line_process = subprocess.Popen(cmd, shell=True,
                                                stdout=subprocess.PIPE,
                                                stderr=subprocess.STDOUT)

        process_output = command_line_process.communicate()[0]
        last_tcp = process_output.decode('utf-8').strip()
        kernel.last_tcp = last_tcp

        #
        # We build a list with the SCLKCET elements for the new SCLK
        #
        if len(sclkcet_current) == len(sclkcet_new):
            logging.info('')
            logging.info('TCP2SCLK Module successfully Run without SCLK generation.')
            logging.info('- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -')
            return False


        sclk_coefficients = ''
        for line in sclkcet_new:
            if line:
                if not line[0][0] == '-':
                    sclk_w = ' ' + line[0]
                else:
                    sclk_w = line[0]
                if not line[1][0] == '-':
                    parsys_w = ' ' + line[1]
                else:
                    parsys_w = line[1]
                if not line[2][0] == '-':
                    gradient_w = ' ' + line[2]
                else:
                    gradient_w = line[2]

                sclk_coefficients += f'         {sclk_w}    {parsys_w}    {gradient_w}\n'

        kernel.sclk_coefficients = sclk_coefficients

        kernel.generate_sclk(os.getcwd())

    else:
        logging.error('New TCPs do not comply the Accuracy and Validity check')
        logging.error('')
        logging.error('TCP2SCLK Module did not Run.')
        logging.info('- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -')

        return False

    logging.info('')
    logging.info('Input TCPs:')
    logging.info(input_tcps)
    logging.info('Used TCPs:')
    logging.info(used_tcps)
    logging.info('')
    logging.info('TCP2SCLK Module successfully Run.')

    return True


def create_kernel(directories, kernel, naif_util):
    """
    Function that generates the required NAIF utility setup file and
    calls the given NAIF utility to generate the kernel.

    :param templates_path: Directory where the template for the NAIF utility is
    :type templates_path: str
    :param kernel: Kernel Object to be converted
    :type kernel: pbject
    :param naif_util: NAIF utility to be used: 'oem2pk', 'aem2spk' or 'msopck'
    :type naif_util:str
    :param kernels_path: Directory where the support kernels to be loaded are
    :type kernels_path: str
    :return: Returns a log text of the execution of the NAIF utility
    :rtype: stradcsng.core.core.process_input
    """

    if naif_util == 'msopck_quat':

        command_line_process = subprocess.Popen([directories.executables + '/' +
             'msopck', naif_util + '.setup',
             kernel.source,
             str(kernel.name)],
             stdout=subprocess.PIPE,
             stderr=subprocess.STDOUT)

    elif naif_util == 'msopck':

        command_line_process = subprocess.Popen([directories.executables + '/' +
             naif_util, naif_util + '.setup',
             kernel.source,
             str(kernel.name)],
             stdout=subprocess.PIPE,
             stderr=subprocess.STDOUT)

    elif naif_util == 'msopck_plan':

        command_line_process = subprocess.Popen([directories.executables + '/' +
             'msopck', naif_util + '.setup',
             'msopck_plan.input',
             str(kernel.name)],
             stdout=subprocess.PIPE,
             stderr=subprocess.STDOUT)

    elif naif_util == 'msopck_bsup':

        command_line_process = subprocess.Popen([directories.executables + '/' +
             'msopck', naif_util + '.setup',
             kernel.source,
             str(kernel.name)],
             stdout=subprocess.PIPE,
             stderr=subprocess.STDOUT)

    elif naif_util == 'prediCkt':

        command_line_process = subprocess.Popen([directories.executables + '/' +
                 naif_util,
                 '-spec', naif_util + '.setup',
                 '-furnish', 'prediCkt.tm',
                 '-ck', str(kernel.name),
                 '-tol', kernel.config['tolerance'],
                 '-sclk', directories.kernels + '/' + kernel.config['sclk']],
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT)

    elif naif_util == 'tcp2scet':
        naif_util_setup = naif_util

        test_setup = []
        with open('tcp2scet.setup', 'r+') as fp:
            for line in fp.readlines():
                if 'DISCONTINUITY_THRESHOLD' in line:
                    line = '   DISCONTINUITY_THRESHOLD =  0.5 \n'
                test_setup.append(line)
        with open('tcp2scet_test.setup', 'w+') as fp:
            for line in test_setup: fp.write(line)

        # generate scet file
        command_line_process = subprocess.Popen([directories.executables + '/' +
                                                 naif_util,
                                                 '-SETUP', 'tcp2scet_test.setup',
                                                 '-MODE', 'STEP',
                                                 '-TCP', kernel.source,
                                                 '-SCET', str(kernel.name) + '.SCET'],
                                                stdout=subprocess.PIPE,
                                                stderr=subprocess.STDOUT)
        process_output, _ = command_line_process.communicate()
        log = process_output.decode('utf-8')
        check_flag = False
        bad_tcps = False
        for line in log.splitlines():
            if check_flag:
                if 'TOO BIG' in line:
                    bad_tcps = True
                    print('WARNING: Bad TCP encountered after gyroless event, please update bad TCP list...')
            if '-21.335 TOO BIG' in line: check_flag = True

        if not bad_tcps:
            # generate scet file
            command_line_process = subprocess.Popen([directories.executables + '/' +
                                                     naif_util,
                                                     '-SETUP', naif_util_setup + '.setup',
                                                     '-MODE' , 'STEP',
                                                     '-TCP', kernel.source,
                                                     '-SCET', str(kernel.name) + '.SCET'],
                                                    stdout=subprocess.PIPE,
                                                    stderr=subprocess.STDOUT)
            process_output, _ = command_line_process.communicate()
            log = process_output.decode('utf-8')

            # generate sclk kernel
            command_line_process = subprocess.Popen('echo makclk.setup | ' + directories.executables + '/makclk',
                                                    shell=True,
                                                    stdout=subprocess.PIPE,
                                                    stderr=subprocess.STDOUT)
            process_output, _ = command_line_process.communicate()
            log += process_output.decode('utf-8')

            # rename SCLK with last TCP date
            file = open(str(kernel.name) + '.SCET').readlines()
            utcf = file[-3].strip().split(' ')[1]
            utcf = datetime.strptime(utcf, "%Y-%jT%H:%M:%S.%f").strftime("%Y%m%d %H:%M:%S").split(' ')[0]
            newname = str(kernel.name).split('_')[0] + '_' + utcf[2:] + '_' + str(kernel.name).split('_')[2]
            os.rename(str(kernel.name), newname)

            # add comments at the top of the kernel
            with open(newname, 'r+') as fin:
                data = fin.read().splitlines(True)
            with open(newname, 'w') as fout:
                fout.write(str(kernel.comments).replace(str(kernel.name) + "\n", newname + "\n") + '\n')
                fout.writelines(data[6:])
            kernel.name = newname

        return log


    else:

        naif_util_setup = naif_util

        if '_' in naif_util:
            naif_util = naif_util.split('_')[0]
        if 'sclk' in kernel.config and 'fict' in kernel.config['sclk']:
            naif_util_setup = naif_util_setup + '_plan'

        command_line_process = subprocess.Popen([directories.executables + '/' +
                                                 naif_util,
                                                 '-setup', naif_util_setup + '.setup',
                                                 '-input', kernel.source,
                                                 '-output', str(kernel.name)],
                                                 stdout=subprocess.PIPE,
                                                 stderr=subprocess.STDOUT)

    process_output, _ = command_line_process.communicate()
    log = process_output.decode('utf-8')

    return log


def convert_spk(kernel, keepboth=False):

    kernel_name = str(kernel.name)
    os.chdir(kernel.working_directory)

    if kernel_name == kernel_name.lower(): typetoken = 't19'
    else: typetoken = 'T19'

    if kernel.input_type == 'OEM':
        naif_util = 'spk13to19'
        kernel_t19_name = typetoken.join(kernel_name.split('.'))

    if kernel.input_type == 'OASW_ORBIT' or kernel.input_type == 'SPK18':
        naif_util = 'spk18to19'
        kernel_t19_name = str(kernel.name)[:int(kernel.source_config[1]['type19_token_index'])] + typetoken + \
                      str(kernel.name)[int(kernel.source_config[1]['type19_token_index']):]

    command_line_process = subprocess.Popen(
            [kernel.directories.executables + '/' +
             naif_util, kernel_name, kernel_t19_name],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)

    process_output= command_line_process.communicate()[0]
    log = process_output.decode('utf-8')

    #
    # Could eventually incorporate chosing duplicating the kernel (as for MEX)
    #
    if keepboth == False and kernel.input_type == 'OEM':
        shutil.move(kernel_t19_name, kernel_name)
    elif keepboth == False and kernel.input_type == 'OASW_ORBIT':
        os.remove(kernel_name)

    return log


def convert_ck(kernel, keepboth=False):

    kernel_name = str(kernel.name)
    os.chdir(kernel.working_directory)

    typetoken = kernel.source_config[1]['type6_token']
    if typetoken == '':
        typetoken = '_T6'

    kernel_t6_name = str(kernel.name)[:int(kernel.source_config[1]['type6_token_index'])] + typetoken + \
                     str(kernel.name)[int(kernel.source_config[1]['type6_token_index']):]

    command_line_process = subprocess.Popen(
            [kernel.directories.executables + '/' +
             'ck5to6', kernel_name, kernel_t6_name],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)

    process_output= command_line_process.communicate()[0]
    log = process_output.decode('utf-8')

    #
    # account for having null type token
    #
    if kernel.source_config[1]['type6_token'] == '':
        os.remove(kernel_name)
        os.rename(kernel_t6_name, kernel_name)

    #
    # Could eventually incorporate chosing duplicating the kernel (as for MEX)
    #
    if keepboth == False and kernel.source_config[1]['type6_token'] != '':
        os.remove(kernel_name)

    return log