#
# import the required libraries and modules
#
import subprocess
import os
import shutil

from adcsng.utils.files import format_str, commnt_read, commnt_add, commnt_delete, run_ckbrief
from adcsng.utils.time import month_str2num
from adcsng.classes.comments import Comments


class KernelName(object):
    """
    Object to construct the Name of the Output Kernel
    """
    def __init__(self, xkma):
        """
        Initialization method for the KernelName Class.

        :param xkma: Reads the Schema for a given kernel. This schema is
           provided in the mission specific JSON configuration file an
           example:
           {kernel_name}_{description}_{start_date}_{finish_date}_v{version}.bsp

        :type xkma: str
        """
        self.xkma = xkma
        self.tokens = {}

    def __str__(self):
        return format_str(self.xkma, self.tokens)

    def set(self, key, value):
        """
        This method assigns a value to a key of the provided schema

        """
        self.tokens.update({key: value})


class Kernel(object):
    """
    The Kernel class is designed to support to generation of kernels within
    adcsng. The kernel object does not contain the kernel itself but rather
    carries different properties needed for the generation across the adcsng
    exectution.
    """

    def __init__(self, k_type, directories, source, name=KernelName('')):
        """
        Initialization method for the Kernel class.

        :param k_type: kernel type, it is determined with the child class.
           It can be either 'ck' or 'spk'
        :type k_type: str
        :param source: Source file or files from which the kernel will be
           created from
        :type source: Union[str, list]
        :param name: Object that determines the name of the kernel. Since the
           name will only be complete when the kernel is generated we cannot
           provide a kernel name from the beginning
        :type name: object
        """

        self.directories = directories
        self.k_type = k_type
        self.source = source
        self.name = name
        self.dafcat = 0
        self.config = {'kernels_dir': directories.kernels}
        self.ker2load = []
        self.mapping = []
        self.comments = Comments
        self.extra_sources = []

    def set(self, key, value):
        if key == 'ker2load' and value:
            self.ker2load.append(value)

            # If we do not want to extract the full kernels to load list
            # we need to be able to access the kernels individually
            if value.split('.')[-1].lower() == 'tsc':
                self.config['sclk'] = value
            if value.split('.')[-1].lower() == 'tf':
                self.config['fk'] = value
            if value.split('.')[-1].lower() == 'tls':
                self.config['lsk'] = value
            if value.split('.')[-1].lower() == 'tm':
                self.config['mk'] = value

        elif key == 'sclk_coefficients' and value:
            self.sclk_coefficients(value)
        elif key == 'last_tcp' and value:
            self.last_tcp(value)

        elif key == 'config' and value:
            self.config.update(value)

        elif value:
            setattr(self, key, value)


    def get_kernels2load(self):
        path = self.config['kernels_dir']

        k2l = ""
        if path:
            k2l += ("      PATH_VALUES  = (\n"
                    "                 '" +
                    path + "'\n"
                    "                            )\n")
            k2l += "      PATH_SYMBOLS = ('KERNELS')\n\n"
            symbol = '$KERNELS/'
        else:
            symbol = ''

        k2l += "      KERNELS_TO_LOAD = (\n"

        for kernel in self.ker2load:
            k2l += ("                 " +
                    "'" + symbol + kernel + "',\n")
        #
        # In the previous loop we have added a comma that we
        # should have not. Remove it.
        #
        k2l = k2l[:-2] + '\n'
        k2l += "                            )"
        k2l += "\n"

        #
        # In order to avoid path lenght issues, we copy the kernels to
        # load to the working_directory
        #
        for ker in self.ker2load:
            if not '.tm' in ker.lower():
                os.makedirs(os.path.dirname(os.path.join(self.working_directory, ker)), exist_ok=True)
                try:
                    shutil.copy(os.path.join(self.directories.kernels, ker),
                        os.path.join(self.working_directory, ker))
                except:
                    shutil.copy(os.path.join(self.directories.output, ker),
                                os.path.join(self.working_directory, ker))

        return k2l


    def get_str_map(self):
        str_map = "     STRING_MAPPING       = (\n"
        for i in range(0, len(self.mapping), 2):
            str_map += (("                  " +
                         "'" + self.mapping[i] + "',").ljust(50) +
                        "'" + self.mapping[i+1] + "',\n")

        #
        # In the previous loop we have added a comma that we
        # should have not. Remove it.
        #
        str_map = str_map[:-2] + '\n'
        str_map += "                            )"
        str_map += "\n"

        return str_map


    def __str__(self):
        return self.name


    def comment_kernel(self):
        """
        Function to comment the kernel properly using the NAIF utility COMMNT

        :param kernel: Kernel Object that has just been converted
        :type kernel: object
        """

        kernel = str(self.name)

        #
        # Get the NAIF name and ID, and the coverage of the kernel.
        #
        if 'coverage' in self.config:
            coverage = self.config['coverage']
        else:
            coverage = ''

        #
        # Get the source_data if present
        #
        if 'input_data' in self.config:
            input_data = self.config['input_data']
        else:
            input_data = ''

        self.id()

        self.comments.obj_id = self.config['obj_id']
        self.comments.adcsng_version = self.config['adcsng_version']
        self.comments.load_source(self.source_filename)
        self.comments.coverage = coverage
        self.comments.input_data = input_data

        #
        # Extract the existing contents from the kernel comment area, and
        # clean it.
        #
        comments = commnt_read(kernel, self.directories)
        commnt_delete(kernel, self.directories)

        #
        # Create the comment file and add it to the kernel.
        #
        comment_file = kernel.split('.')[0]+'.commnt'

        with open(comment_file, 'wt') as f:
            f.write(str(self.comments))
            f.write(comments)

        commnt_add(kernel, comment_file, self.directories)

        return


class SPK(Kernel):

    def __init__(self, source, directories, name=KernelName('')):
        #
        # Create the kernel instance with the output file name being the SPK
        # schema.
        #
        super(SPK, self).__init__('spk', directories, source, name)

    def coverage(self, utc=False):

        cwd = os.getcwd()
        kernel = cwd + '/' + str(self.name)

        utility = 'brief'
        option = '-n '
        reference = 'Body'

        if utc:
            option += '-utc '
            kernel += ' ' + self.directories.kernels + '/' + self.config['lsk']

        command_line_process = subprocess.Popen([self.directories.executables + '/' +
            utility, option, kernel],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT)
        command_line_process.wait()
        process_output, _ = command_line_process.communicate()
        coverage = process_output.decode("utf-8")
        cov_list = coverage.split('\n')

        tab_spaces = '   '
        flag = False
        coverage = ''

        for element in cov_list:

            if flag is True:
                coverage += '\n' + tab_spaces + element
                if element.strip(' ') == '':
                    flag = False

            if reference in element:

                    coverage += '\n' + tab_spaces + element
                    flag = True

        self.set('config', {'coverage':coverage})

        return


    def set_times_in_name(self):

        self.coverage()

        coverage = self.config['coverage'].strip().split('\n')[-1]

        #      2017 APR 06 04:02:53.439            2017 APR 16 05:52:54.693    "

        coverage = coverage.strip()
        # 2017 APR 06 04:02:53.439            2017 APR 16 05:52:54.693"

        utc_start = coverage.split('  ')[0].strip()
        utc_finish = coverage.split('  ')[-1].strip()

        self.set('utc_start', utc_start )
        self.set('utc_finish', utc_finish)

        start_date = coverage.split(' ')[0] + \
                         month_str2num(coverage.split(' ')[1]) + \
                         coverage.split(' ')[2]

        finish_date = coverage.split(' ')[-4] + \
                          month_str2num(coverage.split(' ')[-3]) + \
                          coverage.split(' ')[-2]

        if self.config['kernel_date_format'] == "YYMMDD":
                start_date = start_date[2:]
                finish_date = finish_date[2:]

        if self.config['kernel_date_format'] == "YYYY":
                start_date = start_date[0:3]
                finish_date = finish_date[0:3]


        self.set('config', {'start_date': start_date})
        self.set('config', {'finish_date': finish_date})

        if ('start_date' in str(self.name)): self.name.set('start_date', start_date)
        if ('finish_date' in str(self.name)): self.name.set('finish_date', finish_date)

        return start_date, finish_date


    def id(self):

        cwd = os.getcwd()
        kernel = cwd + '/' + str(self.name)

        command_line_process = subprocess.Popen([self.directories.executables + '/' +
                'brief', kernel],
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT)
        command_line_process.wait()
        process_output, _ = command_line_process.communicate()
        coverage = process_output.decode("utf-8")
        cov_list = coverage.split('\n')

        id = ''
        for element in cov_list:

            if 'Body:' in element:
                id = element.split('Body:')[1].strip()

        self.set('config', {'obj_id': id})

        return


class CK(Kernel):

    def __init__(self, source, directories, name=KernelName('')):
        super(CK, self).__init__('ck', directories, source, name)

    def coverage(self):

        cwd = os.getcwd()
        kernel = cwd + '/' + str(self.name)

        #
        # We need to sort out were the SCLK is kernels or output dirs.
        #
        sclk = self.directories.kernels + '/' + self.config['sclk']
        if not os.path.isfile(sclk):
            sclk = self.directories.output + '/' + self.config['sclk']

        support_kernels = sclk + ' ' + \
                          self.directories.kernels + '/' + self.config['lsk'] + ' ' + \
                          self.directories.kernels + '/' + self.config['fk']

        coverage_text = run_ckbrief(kernel, support_kernels, self.directories)
        cov_list = coverage_text.split('\n')

        tab_spaces = '   '
        coverage = ''

        for element in cov_list:

            if 'Begin' in element:
                coverage += '\n' + tab_spaces + element

        if not coverage:
            raise Exception(coverage_text)

        self.set('config', {'coverage': coverage})

        return

    def set_times_in_name(self):

        self.coverage()
        coverage = self.config['coverage'].strip().split('\n')

        coverage_start_list = []
        coverage_stop_list = []
        for interval in coverage:

            coverage_start = interval.split('Begin UTC: ')[-1].split('  End')[0]
            start_date = coverage_start.split('-')[0] + \
                         month_str2num(coverage_start.split('-')[1]) + \
                         coverage_start.split('-')[2].split(' ')[0]
            if 'kernel_date_format' in self.config:
                if 'T' in self.config['kernel_date_format']:
                    start_date += 'T'
                if 'hh' in self.config['kernel_date_format']:
                    start_date += coverage_start.split(' ')[1].split(':')[0]
                if 'mm' in self.config['kernel_date_format']:
                    start_date += coverage_start.split(' ')[1].split(':')[1]
                if 'ss' in self.config['kernel_date_format']:
                    start_date += coverage_start.split(' ')[1].split(':')[2].split('.')[0]


            coverage_start_list.append(start_date)
            coverage_stop = interval.split('End UTC: ')[-1]
            finish_date = coverage_stop.split('-')[0] + \
                          month_str2num(coverage_stop.split('-')[1]) + \
                          coverage_stop.split('-')[2].split(' ')[0]
            if 'kernel_date_format' in self.config:
                if 'T' in self.config['kernel_date_format']:
                    finish_date += 'T'
                if 'hh' in self.config['kernel_date_format']:
                    finish_date += coverage_stop.split(' ')[1].split(':')[0]
                if 'mm' in self.config['kernel_date_format']:
                    finish_date += coverage_stop.split(' ')[1].split(':')[1]
                if 'ss' in self.config['kernel_date_format']:
                    finish_date += coverage_stop.split(' ')[1].split(':')[2].split('.')[0]

            coverage_stop_list.append(finish_date)

        coverage_start_list.sort()
        coverage_stop_list.sort()

        start_date = coverage_start_list[0]
        finish_date = coverage_stop_list[-1]

        try:
            if 'kernel_date_format' in self.config:

                if self.config['kernel_date_format'] == "YYMM":
                    start_date = start_date[2:6]
                    finish_date = finish_date[2:6]

                elif self.config['kernel_date_format'].startswith("YYM"):
                    start_date = start_date[2:]
                    finish_date = finish_date[2:]

                if self.config['kernel_date_format'] == "YYYYMM":
                    start_date = start_date[0:6]
                    finish_date = finish_date[0:6]

                if self.config['kernel_date_format'] == "YYYY":
                    start_date = start_date[0:4]
                    finish_date = finish_date[0:4]

            self.set('config', {'start_date': start_date})
            self.set('config', {'finish_date': finish_date})

            # We rename the kernel
            self.name.set('start_date', start_date)
            self.name.set('finish_date', finish_date)
        except:
            self.set('config', {'start_date': start_date})
            self.set('config', {'finish_date': finish_date})

            # We rename the kernel
            self.name.set('start_date', start_date)
            self.name.set('finish_date', finish_date)
            pass

        return

    def id(self):

        cwd = os.getcwd()
        kernel = cwd + '/' + str(self.name)

        support_kernels = cwd + '/' + self.config['sclk'] + ' ' + \
                          cwd + '/' + self.config['lsk']

        coverage_text = run_ckbrief(kernel, support_kernels, self.directories)
        cov_list = coverage_text.split('\n')

        obj_id = ''
        for element in cov_list:
            if 'Body:' in element:
                obj_id = element.split('Body:')[1].strip()

        self.set('config', {'obj_id': obj_id})

        return


class SCLK(Kernel):

    def __init__(self, source, directories, name=KernelName('')):
        super(SCLK, self).__init__('sclk', directories, source, name)

    def generate_sclk(self, path):

        date = self.last_tcp.split('T')[0]
        time = self.last_tcp.split('T')[1]

        self.sclk_kernel_id = '@{0}/{1}'.format(date, time)

        tcp_date = date.split('-')[0] + date.split('-')[1] + date.split('-')[2]

        self.name.set('tcp_date', tcp_date)

        dict = {'sclk_coefficients': self.sclk_coefficients,
                'sclk_kernel_id': self.sclk_kernel_id,
                'sclk_separator_id': self.config['sclk_separator_id']}

        self.comments.load_tokens(dict)

        with open(path + '/' + str(self.name), 'w+') as f:
            f.write(str(self.comments))

        return
