#
# Import the required libraries and modules
#
import datetime
import os
import platform
import fnmatch
from adcsng.utils.files import format_str, commnt_read


class Comments(object):
    """
      The class Comments is a class to be used as an input on a Kernel Object.
      Like the Kernel Object this class exists because the comments of the
      kernel are being completed as the kernel is generated.
      """

    __type2descr__ = {
        'tls': 'a Leapseconds Kernel file is used in\n'
               '{SPC}conversions between ephemeris time and UTC\n'
               '{SPC}(coordinated universal time).\n',
        'tsc': 'a Spacecraft Clock Kernel file is used\n'
               '{SPC}in conversions between the {mission_accr} Clock\n'
               '{SPC}time and ephemeris time.\n',
        'tf': 'a Frame Kernel file required to\n'
              '{SPC}establish relationships between frames\n'
              '{SPC}used in geometry computations.\n',
        'tm': 'a Meta-Kernel file required to\n'
              '{SPC}load a list of required kernels\n'
              '{SPC}to generate the given attitude profile.\n'
    }

    def __init__(self, config='', path='', template=''):
        """
          Initialization method of the Comments Class

          :param config: The generic part of the adcsng JSON Configuration
             file
          :type config: dict
          :param path: The path of the directory where the Comments Templates
             are placed.
          :type path: str
          :param template: The given template to be used for the kernel type.
            Usually it also depends on the NAIF utility that generates the
            kernel.
          :type template: str
          """
        self.path = path
        self.template_source = template

        #
        # This adds the fields which are not present in the adcsng JSON
        # configuration file.
        #
        self.tokens = {
            'gen_month': datetime.datetime.now().strftime("%B"),
            'gen_day': str(datetime.datetime.now().day),
            'gen_year': str(datetime.datetime.now().year),
            'SPC': ' ' * 35  # 5 spaces + a maximum filename of 30 chars.
        }

        self.load_tokens(config)

    def __setattr__(self, key, value):
        if key in ['path', 'template_source', 'tokens']:
            super(Comments, self).__setattr__(key, value)
        else:
            self.tokens.update({key: value})

    def __str__(self):
        comments = ''
        with open(os.path.join(self.path, self.template_source)) as f:
            for line in f.readlines():
                if not line.startswith('%'):
                    comments += format_str(line, self.tokens)
        return comments

    def load_kernels(self, kernel_list):
        """
          Creates the {KERNELS} keyword value from the ``kernel_list`` based
          on the default descriptions.

          :param kernel_list: list of str
          :return: NoneType
          """
        if type(kernel_list) != list:
            kernel_list = [kernel_list]

        kernels = ''
        for kernel_file in kernel_list:
            name = os.path.basename(kernel_file)
            kernels += (' ' * 5) + name.ljust(30)
            if len((' ' * 5) + name.ljust(30)) > 35:
                kernels += '\n\n' + (' ' * 35)
            kernels += self.__type2descr__[name.split('.')[-1].lower()] + '\n'

        #
        # Apply some formatting to the output string and add the value to the
        # list of available tokens.
        #
        kernels = format_str(kernels, self.tokens)
        self.__setattr__('support_kernels', kernels[:-1])

    def load_tokens(self, dictionary):
        for (key, value) in dictionary.items():
            self.tokens.update({key: value})

    def text2value(self, key, text, tab=0):
        value = ''.join(' ' * tab + line + '\n' for line in text)
        self.tokens.update({key: value[:-1]})

    def load_source(self, source):

        src_dict = {}
        sources = ''

        first_bool = True
        if isinstance(source, list):
            for file in source:

                if first_bool:
                    sources = file
                    first_bool = False
                else:
                    sources += '\n' + ' ' * 6 + file

        if sources:
            source_filename = sources
        else:
            source_filename = source

        if not isinstance(source, list):
            source = [source]

        # We only use the date birth of the last file.

        src_dict['input_file'] = source_filename

        if platform.system() == 'Windows':
            src_date = os.path.getctime(source[-1])
        else:
            stat = os.stat(source[-1])
            try:
                src_date = stat.st_birthtime
            except AttributeError:
                # We're probably on Linux. No easy way to get creation dates here,
                # so we'll settle for when its content was last modified.
                src_date = stat.st_mtime

        src_date = datetime.date.fromtimestamp(src_date)

        src_dict['src_year'] = str(src_date.year)
        src_dict['src_month'] = src_date.strftime("%B")
        src_dict['src_day'] = str(src_date.day)

        self.load_tokens(src_dict)


def get_pedigree_files(kernel_path, input_filename_pattern, directories):
    comment_text = commnt_read(kernel_path, directories)
    comment_text = comment_text.splitlines()

    pedigree_files = []
    inside_pedigree_section = False
    previous_line = ""
    for line in comment_text:

        if inside_pedigree_section and line.startswith("----"):
            break

        if previous_line.startswith("Pedigree") and line.startswith("----"):
            inside_pedigree_section = True

        if inside_pedigree_section:

            line = line.strip().lower()

            if fnmatch.fnmatch(line, input_filename_pattern) \
                    and line not in pedigree_files:
                pedigree_files.append(line)

        previous_line = line

    return pedigree_files


def set_pedigree_files(comment_text, pedigree_files, input_filename_pattern):
    comment_text = comment_text.splitlines()

    inside_pedigree_section = False
    previous_line = ""
    new_comment_text = ""
    for line in comment_text:

        if inside_pedigree_section and line.startswith("----"):
            inside_pedigree_section = False

        if previous_line.startswith("Pedigree") and line.startswith("----"):
            inside_pedigree_section = True

        if inside_pedigree_section:

            if fnmatch.fnmatch(line.strip().lower(), input_filename_pattern):

                # This indicates that this line has a pedigree file
                # Add passed pedigree files and ignore next pedigree files occurrences in the comment_text
                for pedigree_file in pedigree_files:
                    new_comment_text += f'      {pedigree_file.split(os.sep)[-1]}\n'
                pedigree_files = []  # Empty list to ignore next occurrences

            else:
                new_comment_text += line + '\n'

        else:
            new_comment_text += line + '\n'

        previous_line = line

    return new_comment_text


def get_dafcat_comments_header(kernels_to_append):
    comments_header = '\n' + \
                      '   This file contains a series of files concatenated with each other.\n' + \
                      '   For complete information please refer to the comments of:\n'

    for kernel_to_append in kernels_to_append:
        comments_header += f'     {kernel_to_append.split(os.sep)[-1]}\n'

    comments_header += '\n\n\n'

    return comments_header


def set_aprox_time_coverage(comment_text, start_time, end_time):
    comment_text = comment_text.splitlines()

    inside_coverage_section = False
    previous_line = ""
    new_comment_text = ""
    for line in comment_text:

        if inside_coverage_section and line.startswith("----"):
            inside_coverage_section = False

        if previous_line.startswith("Approximate Time Coverage") and line.startswith("----"):
            inside_coverage_section = True

        if inside_coverage_section:

            if str(line.strip().lower()).startswith("begin utc:"):

                start_time_str = str(start_time.strftime('%Y-%b-%d %H:%M:%S.%f')).upper()[0:-3]
                end_time_str = str(end_time.strftime('%Y-%b-%d %H:%M:%S.%f')).upper()[0:-3]

                new_comment_text += f'   Begin UTC: {start_time_str}  End UTC: {end_time_str}\n'

            else:
                new_comment_text += line + '\n'

        else:
            new_comment_text += line + '\n'

        previous_line = line

    return new_comment_text
