import os
import json
import shutil
import glob
import datetime
from .utils import comparer
from .utils import skd_validator
from unittest import TestCase
from arcgen.command_line import main
from arcgen.utils.manifest import transfer


class ARCGENTest(TestCase):
    bundle = ''
    config_file = ''
    config = None
    gen_folder = "/gen"
    val_folder = "/val"
    check_extensions = [".txt", ".csv", ".xml", ".html", ".tm", ".lbl", ".tab"]
    remove_str_list = []
    ignore_str_list = ["<creation_date_time>",
                       "Last update:",
                       "<md5_checksum>",
                       "<publication_year>",
                       "<publication_date>",
                       "This file was created on",
                       "PRODUCT_CREATION_TIME"]
    cleanup_when_finish = True
    tests_path = "."

    def run_test(self, bundle, config_file):

        # Init members
        self.bundle = bundle
        self.config_file = config_file
        self.tests_path = os.path.dirname(__file__)
        os.chdir(self.tests_path)

        # Clean up env at start
        self.cleanup()

        # Init environment (prepare config and MK files)
        self.init_environment()

        # Check gen folder
        start_dir = self.get_config()["bundle_directory"]
        gen_dir = os.path.dirname(start_dir)  # parent dir
        self.assertIn(self.gen_folder, gen_dir,
                      msg="bundle_directory shall be under a 'gen' folder")

        # Run ARCGEN
        local_config = os.path.join(self.tests_path, self.get_local_config_file())
        main(local_config)

        # Prepare transfer file
        if self.bundle:
            transfer(self.bundle, start_dir, gen_dir)

        # TODO: Prepare checksum file

        # Prepare a report with the MK info
        report_file = os.path.join(gen_dir, "mk_report.txt")
        skd_validator.write_mk_kernels_report(self.get_local_mk_file(), report_file)

        # Prepare replace strings map
        self.replace_str_map = {}
        self.replace_str_map[self.get_config()["kernels_directory"]] = "___KERNELS___"

        # Check that generated contents match the data for validation
        self.check_equal_contents()

        if self.cleanup_when_finish:
            # Clean up env at finish if all OK!
            self.cleanup()

    def get_config(self):
        if not self.config:
            local_config = os.path.join(self.tests_path, self.get_local_config_file())
            if os.path.exists(local_config):
                with open(local_config, 'r') as f:
                    self.config = json.load(f)
        return self.config

    def init_environment(self):
        # Update config to point local paths
        local_config = os.path.join(self.tests_path, self.get_local_config_file())
        comparer.file_replace(os.path.join(self.tests_path, self.config_file),
                              "$PWD$",
                              self.tests_path,
                              local_config)

        # Prepare local MK
        comparer.file_replace(self.get_config()["metakernel"],
                              "= ( '..' )",
                              "= ( '" + self.get_config()["kernels_directory"] + "' )",
                              self.get_local_mk_file())

        # Update config to point local MK and force refresh
        tmp_local_config = os.path.join(self.tests_path, self.get_local_config_file().replace(".json", "_tmp.json"))
        comparer.file_replace(local_config,
                              self.get_config()["metakernel"],
                              self.get_local_mk_file(),
                              tmp_local_config)
        os.remove(local_config)
        os.rename(tmp_local_config, local_config)
        self.config = None

    def cleanup(self):
        if os.path.isfile(self.get_local_config_file()):
            # Remove local MK file if exists
            if os.path.isfile(self.get_local_mk_file()):
                os.remove(self.get_local_mk_file())

            # Remove gen dir if exist, and create it again
            gen_dir = os.path.dirname(self.get_config()["bundle_directory"])  # parent dir

            self.assertIn(self.gen_folder, gen_dir,
                          msg="bundle_directory shall be under a 'gen' folder")

            if os.path.isdir(gen_dir):
                shutil.rmtree(gen_dir)
            os.mkdir(gen_dir)

            # Remove local config file
            os.remove(self.get_local_config_file())
            self.config = None

    def get_local_config_file(self):
        return self.config_file.replace(".json", "_local.json")

    def get_local_mk_file(self):
        # We don't know if config was loaded for the .json or from the _local_json
        # so lets check if metakernel is already pointing to _local.tm
        if not str(self.get_config()["metakernel"]).endswith("_local.tm"):
            if str(self.get_config()["metakernel"]).endswith(".tm"):
                return self.get_config()["metakernel"].replace(".tm", "_local.tm")
            elif str(self.get_config()["metakernel"]).endswith(".TM"):
                return self.get_config()["metakernel"].replace(".TM", "_local.tm")
            else:
                print("Invalid METAKERNEL filename!!")
                exit(-1)
        else:
            return self.get_config()["metakernel"]

    def check_equal_contents(self):
        # Prepared paths for gen and val dirs
        gen_dir = os.path.dirname(self.get_config()["bundle_directory"])  # parent dir
        val_dir = gen_dir.replace(self.gen_folder, self.val_folder)

        # List all files
        gen_dir_files = list(glob.iglob(gen_dir + '/**/*', recursive=True))
        val_dir_files = list(glob.iglob(val_dir + '/**/*', recursive=True))

        self.assertEqual(len(gen_dir_files), len(val_dir_files),
                         msg="gen dir and val dir directories have different number of files!")

        # Check contents file by file
        for val_filename in val_dir_files:
            extension = str(os.path.splitext(val_filename)[1]).lower()
            if extension in self.check_extensions:
                gen_filename = val_filename.replace("/val/", "/gen/")

                self.assertTrue(os.path.isfile(gen_filename),
                                msg="File not exists! File: " + val_filename)

                if gen_filename.endswith("INDEX/INDEX.TAB"):
                    # For INDEX/INDEX.TAB ignore the date from third column
                    self.assertTrue(comparer.equal_csv_contents(val_filename,
                                                                gen_filename,
                                                                separator=",",
                                                                ignore_columns=[2],
                                                                remove_str_list=self.remove_str_list,
                                                                ignore_str_list=self.ignore_str_list,
                                                                replace_str_map=self.replace_str_map),
                                    msg="CSV Files are not equal! (Dates were ignored) File: " + val_filename)
                else:
                    self.assertTrue(comparer.equal_file_contents(val_filename,
                                                                 gen_filename,
                                                                 remove_str_list=self.remove_str_list,
                                                                 ignore_str_list=self.ignore_str_list,
                                                                 replace_str_map=self.replace_str_map),
                                    msg="Files are not equal! File: " + val_filename)
