#!/usr/bin/env python3
import logging
import datetime
import socket

from textwrap import dedent
from os import listdir
from os import sep
from os import getcwd
from os import system

from os.path import isfile, join, dirname
from argparse import ArgumentParser, RawDescriptionHelpFormatter

from spiceypy import kclear

from arcgen.utils.manifest import transfer

from .classes.bundle import Bundle
from .classes.product import KernelProduct
from .classes.product import MetaKernelProduct
from .classes.mission import Mission
from .classes.collection import SpiceKernelsCollection
from .classes.collection import DocumentCollection
from .classes.product import InventoryProduct
from .classes.product import SpicedsProduct
from .utils.files import get_increment_kernels


def main(config=False, labels=False, transfer_manifest=False, log=False):


    with open(dirname(__file__) + '/config/version',
              'r') as f:
        for line in f:
            version = line

    if not config:
        parser = ArgumentParser(formatter_class=RawDescriptionHelpFormatter,
                                description=dedent('''\

         ARCGEN -- Version {}, SPICE PDS3/PDS4 Archive Generation Pipeline 

           ARCGEN is the command-line utility program that generates PDS4 
           Bundles and PDS3 Data Sets for SPICE Kernel Datasets. Source and 
           documentation are available here: 

              https://repos.cosmos.esa.int/socci/projects/SPICE/repos/arcgen


        '''.format(version)),
                                epilog='''
            __   __   __      __   __     __   ___     __   ___  __          __   ___
           /__\ /__` '__\    /__` |__) | /  ` |__     /__` |__  |__) \  / | /  ` |__
           \__, .__/ \__/    .__/ |    | \__, |___    .__/ |___ |  \  \/  | \__, |___

         esa_spice@sciops.esa.int
         http://spice.esac.esa.int

        ''')
        parser.add_argument('config', metavar='CONFIG', type=str, nargs='+',
                            help='Mission specific JSON configuration file')
        parser.add_argument('-e', '--labels',
                        help='Execute ARCGEN to generate labels only',
                        action='store_true')
        parser.add_argument('-l', '--log',
                        help='Prompt log during execution',
                        action='store_true')
        parser.add_argument('-t', '--transfer',
                        help='Execute ARCGEN to generate PSA transfer manifest',
                        action='store_true')

        args = parser.parse_args()
        config = args.config[0]
        labels = args.labels
        transfer_manifest = args.transfer
        log = args.log
    else:
        config = config


    #
    # Setup the log
    #
    logger = logging.getLogger()
    logger.setLevel(logging.INFO)

    if log:
        log_format = '%(asctime)s.%(msecs)03d %(levelname)-8s %(module)-16s %(message)s'
        date_format = '%Y-%m-%dT%H:%M:%S'
        ch = logging.StreamHandler()
        ch.setLevel(logging.INFO)
        formatter = logging.Formatter(log_format,date_format)
        ch.setFormatter(formatter)
        logger.addHandler(ch)
    else:
        logger.setLevel(logging.ERROR)


    #
    # The configuration is setup with the Mission Class
    #
    mission = Mission(config, getcwd())


    #
    # Cleanup the staging area for MAC OS
    #
    try:
        system(f"find {mission.kernels_directory} -name '.DS_Store' -type f -delete")
    except:
        pass


    logging.info('ARCGEN v{} for {} run on {} at {}'.format(
            version, mission.name, socket.gethostname(),
            str(datetime.datetime.now())[:-5]))

    #
    # Generate the PSA transfer manifest
    #
    if transfer_manifest:
        transfer(mission.bundle_directory)

    #
    # First we generate only labels if so
    #
    if labels:

        kernel_directory_list = listdir(mission.kernels_directory)
        for directory in kernel_directory_list:
            try:
                dir = mission.kernels_directory + sep + directory
                files = [join(dir, f) for f in listdir(dir) if isfile(join(dir, f))]

                for file in files:
                    if file.split('.')[-1].upper() != 'TXT' and \
                       file.split('.')[-1].upper() != 'OBJ' and \
                       file.split('.')[-1].upper() != 'PNG' and \
                       file.split('.')[-1].upper() != 'HTML' and \
                       file.split('.')[-1].upper() != 'XML' and \
                       file.split('.')[-1].upper() != 'TM':

                        logger.info(file)
                        kernel = KernelProduct(file, mission, label_only=True)

                    if file.split('.')[-1].upper() == 'TM':
                        mk = MetaKernelProduct(mission, product=file)
                        logger.info(file)

            except Exception as e:
                logger.error(e)
                pass


        return


    #
    # SPICE Kernels Bundle generation
    # -------------------------------
    #
    bundle = Bundle(mission)

    kernel_directory_list = listdir( mission.kernels_directory )
    try:
        kernel_directory_list.remove('aareadme.txt')
    except:
        pass


    #
    # SPICE Kernels Collection
    # ------------------------
    #
    spice_kernels_collection = SpiceKernelsCollection(mission)

    # File to check pattern matches used for choosing the description:
    if mission.pds == '3':
        check = open( 'desc_pattern_check.txt', 'w' )
        check.write ( 'Kernel file name vs. name pattern in ' + \
                      'aareadme.txt\n\n' )

    #
    #    Generation of individual kernels
    #
    files_list = []
    for directory in kernel_directory_list:

        dir = mission.kernels_directory + sep + directory
        files = [join(dir, f) for f in listdir(dir) if isfile(join(dir, f))]

        for file in files:
            if file.split('.')[-1].upper() != 'TXT' and \
               file.split('.')[-1].upper() != 'OBJ' and \
               file.split('.')[-1].upper() != 'PNG' and \
               file.split('.')[-1].upper() != 'HTML' and \
               file.split('.')[-1].upper() != 'XML' and \
               file.split('.')[-1].upper() != 'TM' and \
               '.DS_Store' not in file:

                logger.info(file)
                files_list.append(file.split('/')[-1])
                kernel = KernelProduct(file, mission)
                spice_kernels_collection.add(kernel)

    #
    #   Add last increment kernels
    #
    get_increment_kernels(mission, files_list)

    #
    #   Generation of the meta-kernel
    #
    mk = MetaKernelProduct(mission,
                           spice_kernels_collection=spice_kernels_collection)

    #
    #   The metakernel is added to the spice_products_collection
    #
    spice_kernels_collection.add(mk)

    #
    #    Generation of the kernels inventory
    #
    InventoryProduct(mission, spice_kernels_collection)

    #
    # Document Collection
    # -------------------
    #
    document_collection = DocumentCollection(mission)

    #
    #    Generation of SPICEDS documnet
    #
    if mission.pds == '4':
        spiceds = SpicedsProduct(mission, document_collection)
        if spiceds.generated:
            document_collection.add(spiceds)

            #
            #    Generation of the documents inventory
            #
            InventoryProduct(mission, document_collection)

        #
        # Adding Collections to the Bundle
        # --------------------------------
        #
        bundle.add(spice_kernels_collection)
        if spiceds.generated:
            bundle.add(document_collection)

        #
        # SPICE Kernels Bundle Readme
        # ---------------------------
        #
        bundle.write_readme()

        #
        # Clean-up execution
        #
        kclear()

    return

