import os
import re
import glob
import shutil
import tarfile
import subprocess
import numpy.testing as npt

from adcsng.utils.files import print_html
from adcsng.utils.files import get_exe_dir
from adcsng.command_line import main as adcsng


def runADCSng(version, debug=False, log=False, untar=True):

    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    if untar:
        tar = tarfile.open('incoming.tar.gz')
        tar.extractall()
        tar.close()

    print_html(f"Executing ADCSng v{version} for BepiColombo")
    adcsng('bepicolombo.json', debug=debug, log=log)


def cleanup():
    cwd = os.path.dirname(__file__)
    os.chdir(cwd)

    try:
        os.remove('adcsng_bepicolombo.lock')
    except:
        pass

    try:
        shutil.rmtree('incoming')
        os.remove('._incoming')
    except:
        pass

    try:
        shutil.rmtree('ker_gen/reports')
    except:
        pass

    for CleanUp in glob.glob('ker_gen/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('ancdr/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('log/**/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt') and \
                not CleanUp.endswith('adcsng.eop') and \
                not CleanUp.endswith('adcsng.log'):
            os.remove(CleanUp)

    for CleanUp in glob.glob('temp/*.*', recursive=True):
        if not CleanUp.endswith('aareadme.txt') or not CleanUp.endswith('temp.log'):
            try:
                shutil.rmtree(CleanUp)
            except:
                if 'aareadme.txt' not in CleanUp:
                    os.remove(CleanUp)


def SPKcompare():

    spks = glob.glob('ker_gen/spk/*.bsp')
    for spk in spks:

        exe_dir = get_exe_dir()
        spk_1 = spk
        spk_2 = f'ker_val/spk/{spk.split(os.sep)[-1]}'
        fk = 'ker_dir/fk/bc_mpo_v24.tf'

        if not os.path.exists(spk_1): raise NameError(f"SPK Kernel {spk_1} does not exist")
        if not os.path.exists(spk_2): raise NameError(f"SPK Kernel {spk_2} does not exist")

        command = f'../..{exe_dir}/spkdiff -k {fk} {spk_1}  {spk_2}'
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                           stdout=subprocess.PIPE,stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)
        if 'Toolkit version:' in text:
            raise NameError(f"SPKDIFF did not work with SPK Kernels {spk}")

        try:
            text = text.split('\n')
            for line in text:
                if 'Position:' in line or 'Velocity:' in line:
                    npt.assert_almost_equal(float(line.split()[1]), 0.0, decimal=6)
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=6)
                if 'Position (km)' in line or 'Velocity (km/s)' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=6)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=6)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"SPK Kernels {spk} are different")


def CKcompare():

    kernels =  'ker_dir/fk/bc_mpo_v24.tf ' + \
               'ker_gen/sclk/bc_mpo_step_20200109.tsc ' + \
               'ker_dir/lsk/naif0012.tls ' + \
               'ker_dir/sclk/bc_mpo_fict_20181127.tsc ' + \
               'ker_dir/fk/bc_mtm_v04.tf '

    cks = {'1':{'ck':     'bc_mpo_sc_fsp_00042_20190110_20190124_f20181127_v04.bc',
                'fframe': 'MPO_SPACECRAFT',
                'tframe': 'J2000',
                'start':  '2019-JAN-10 03:26:06 TDB',
                'finish': '2019-JAN-24 08:05:09 TDB'},
           '2': {
                'ck':    'bc_mpo_sc_scc_20190809_20190810_s20200109_v01.bc',
               'fframe': 'MPO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2019-AUG-09 23:59:07 TDB',
               'finish': '2019-AUG-10 00:01:05 TDB'},
           '3': {
               'ck':     'bc_mpo_sc_scm_20190806_20190810_s20200109_v01.bc',
               'fframe': 'MPO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2019-AUG-10 00:00:30 TDB',
               'finish': '2019-AUG-10 00:01:09 TDB'},
           '4': {
               'ck':     'bc_mpo_sc_scm_20190806_20190810_s20200109_v01.bc',
               'fframe': 'MPO_SPACECRAFT',
               'tframe': 'J2000',
               'start':  '2019-AUG-10 00:00:30 TDB',
               'finish': '2019-AUG-10 00:01:09 TDB'},
           '5': {
               'ck':     'bc_mpo_sa_scm_20190809_20190809_s20200109_v01.bc',
               'fframe': 'MPO_SA_SADM',
               'tframe': 'MPO_SA',
               'start':  '2019-AUG-09 00:01:14 TDB',
               'finish': '2019-AUG-09 00:23:13 TDB'},
           '6': {
               'ck':     'bc_mpo_mga_scm_20190806_20190809_s20200109_v01.bc',
               'fframe': 'MPO_MGA_BOOM',
               'tframe': 'MPO_MGA_BOOM-H',
               'start':  '2019-AUG-09 00:01:14 TDB',
               'finish': '2019-AUG-09 00:01:53 TDB'},
           '7': {
               'ck':     'bc_mpo_mga_scm_20190806_20190809_s20200109_v01.bc',
               'fframe': 'MPO_MGA_BOOM',
               'tframe': 'MPO_MGA_BOOM-H',
               'start':  '2019-AUG-06 00:01:14 TDB',
               'finish': '2019-AUG-06 00:23:13 TDB'},
           '8': {
               'ck':     'bc_mpo_hga_scm_20190809_20190810_s20200109_v01.bc',
               'fframe': 'MPO_HGA_AZ',
               'tframe': 'MPO_HGA_EL',
               'start':  '2019-AUG-09 23:59:54 TDB',
               'finish': '2019-AUG-10 00:02:13 TDB'},
           '9': {
               'ck':     'bc_mtm_sa_scm_20190114_20190116_s20200109_v01.bc',
               'fframe': 'MTM_SA-X_ZERO',
               'tframe': 'MTM_SA-X',
               'start':  '2019-JAN-16 01:16:04 TDB',
               'finish': '2019-JAN-17 00:01:03 TDB'},
           '10': {
               'ck':     'bc_mtm_sa_scm_20190114_20190116_s20200109_v01.bc',
               'fframe': 'MTM_SA-X_ZERO',
               'tframe': 'MTM_SA-X',
               'start':  '2019-JAN-14 01:16:04 TDB',
               'finish': '2019-JAN-14 10:01:03 TDB'},
           '11': {
              'ck':     'bc_mtm_sep_00061_20181118_20211009_f20181020_v01.bc',
              'fframe': 'MTM_SEP',
              'tframe': 'MTM_SPACECRAFT',
              'start':  '2018-NOV-19 01:00:00 TDB',
              'finish': '2021-OCT-08 23:00:00 TDB'},
           '12': {
              'ck':     'bc_mpo_phebus_20190524_20190524_s20200109_v01.bc',
              'fframe': 'MPO_PHEBUS_SCAN_ZERO',
              'tframe': 'MPO_PHEBUS_SCAN',
              'start':  '2019-MAY-24 06:30:00.000 TDB',
              'finish': '2019-MAY-24 13:20:00.000 TDB'},
           '13': {
              'ck':     'bc_mpo_phebus_20190524_20190524_s20200109_v01.bc',
              'fframe': 'MPO_PHEBUS_SM',
              'tframe': 'MPO_PHEBUS',
              'start':  '2019-MAY-24 06:30:00.000 TDB',
              'finish': '2019-MAY-24 13:20:00.000 TDB'}
           }

    if len(cks) < len(os.listdir("ker_gen/ck/")):
        raise NameError("WARNING: some kernels in ker_gen are not being tested")


    for ck in cks.items():
        ck = ck[1]
        exe_dir = get_exe_dir()

        ck_1 = f"ker_gen/ck/{ck['ck']}"
        ck_2 = f"ker_val/ck/{ck['ck']}"

        if not os.path.exists(ck_1): raise NameError(f"CK Kernel {ck_1} does not exist")
        if not os.path.exists(ck_2): raise NameError(f"CK Kernel {ck_2} does not exist")

        command = f"../..{exe_dir}/frmdiff -k {kernels} -f1 {ck['fframe']} " \
                  f"-t1 {ck['tframe']} -b {ck['start']} -e {ck['finish']} " \
                  f" {ck_1} {ck_2}"
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                                stdout=subprocess.PIPE,
                                                stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)
        if 'version: N0066' in text:
            raise NameError(f"FRMDIFF did not work with CK Kernels {ck['ck']}: {text}")
        #
        # Whilst with the SPKs and SCLKs it is enough to compare the DIFF
        # files, for PREDICKT the mantissa and the precision differ from
        # Linux to MacOS:
        #
        #   OS:    Rotation (rad):             +4.8013978590743E-16      +1.4316667159571E-16
        #   LINUX: Rotation (rad):             +0.0000000000000E+00      +0.0000000000000E+00
        #
        # Therefore we will use an assertion instead of a file comparison
        #
        try:
            text = text.split('\n')
            for line in text:
                if 'Rotation (rad):' in line:
                    npt.assert_almost_equal(float(line.split()[2]), 0.0, decimal=3)
                    npt.assert_almost_equal(float(line.split()[3]), 0.0, decimal=3)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"CK Kernels {ck['ck']} are different")



    return


def SCLKcompare():

    begin = '2018-10-30'
    finish = '2020-01-01'
    id =  '-121'

    lsk = 'ker_dir/lsk/naif0012.tls'
    sclks = glob.glob('ker_gen/sclk/*.tsc')
    for sclk in sclks:

        sclk = sclk.split(os.sep)[-1]
        exe_dir = get_exe_dir()
        command = f"../..{exe_dir}/diffsclk {lsk} {begin} {finish} " \
                  f"864000 {id} ker_val/sclk/{sclk} ker_gen/sclk/{sclk}"
        print_html(command)
        command_process = subprocess.Popen(command, shell=True,
                                           stdout=subprocess.PIPE,
                                           stderr=subprocess.STDOUT)

        process_output, _ = command_process.communicate()
        text = process_output.decode('utf-8')
        print_html(text)

        try:
            text = text.split('\n')
            for line in text:
                if line.strip():
                        npt.assert_almost_equal(float(line.split()[1]), 0.0,
                                            decimal=5)
        except AssertionError as error:
            print_html(error)
            raise NameError(f"SCLK Kernels {sclk} are different")

    return


def MKcompare():

    mks = [f for f in os.listdir('ker_gen/mk/') if re.search(r'.*[^0-9].tm', f)]

    if not mks:
        raise NameError("No MK Kernels available")

    for mk in mks:
        data_gen = []
        record = False
        with open(f'ker_gen/mk/{mk}', 'r') as m:
            for line in m:
                if record: data_gen.append(line)
                if '\\begindata' in line: record = True
                if '\\begintext' in line: break

        print_html(''.join(data_gen))

        with open(f'ker_val/mk/{mk}', 'r') as m: data_val = m.readlines()

        if not set(data_gen).issubset(data_val):
            raise NameError("MK Kernels are different")

    return


def LOGcompare():


    log_files = [f for f in os.listdir('log/') if re.search(r'.*.log', f)]
    eop_files = [f for f in os.listdir('log/') if re.search(r'.*.eop', f)]

    log_lines = []
    logs = []
    for log in log_files:
        with open('log/'+ log, 'r') as l:
            for line in l:
                if 'PRODUCT_CREATION_TIME' not in line and \
                        'for BEPICOLOMBO run' not in line:
                    log_lines.append(' '.join(line.split(' ')[1:]))
            logs.append(log_lines)

    if sorted(logs[0]) != sorted(logs[1]):
        raise NameError("LOG files are different")


    eop_lines = []
    eops = []
    for eop in eop_files:
        with open('log/' + eop, 'r') as l:
            for line in l:
                print_html(line)
                if 'PRODUCT_CREATION_TIME' not in line and \
                        'for BEPICOLOMBO run' not in line:
                    eop_lines.append(' '.join(line.split(' ')[1:]))
            eops.append(eop_lines)

    if sorted(eops[0]) != sorted(eops[1]):
        raise NameError("EOP files are different")

    return


def GAPcompare():

    gaps = [f for f in os.listdir('ker_gen/reports/') if
           re.search(r'.*gap', f)]

    if not gaps:
        raise NameError("No GAP files available")

    for gap in gaps:
        with open(f'ker_gen/reports/{gap}', 'r') as m: gap_gen = m.readlines()[2:]
        with open(f'misc/reports/{gap}', 'r') as m: gap_val = m.readlines()[2:]
        for line in gap_gen: print_html(line)

        if not gap_gen == gap_val:
            raise NameError("GAP files are different")

    return


if __name__ == '__main__':
    cleanup()
    runADCSng('1.0.0', debug=True, log=True, untar=True)
    #os.chdir(os.getcwd() + '/..')
    #SPKcompare()
    #CKcompare()

    #LOGcompare()