C$Procedure MAKSEG ( Make SPK segments )
 
      SUBROUTINE MAKSEG ( INPUT,   HANDLE,  METHOD,  DEGREE,  
     .                    SPCLIM,  QUOLIM,  BODY,    CENTER,  
     .                    FRAME,   TBEG,    TEND,    TIMSYS,
     .                    SPKBEG,  SPKEND                    )
 
C$ Abstract
C
C     Write out one or more SPK segments representing the ephemeris
C     defined by a metadata block and the contiguous set of states
C     following the block.
C
C$ Disclaimer
C
C     THIS SOFTWARE AND ANY RELATED MATERIALS WERE CREATED BY THE
C     CALIFORNIA INSTITUTE OF TECHNOLOGY (CALTECH) UNDER A U.S.
C     GOVERNMENT CONTRACT WITH THE NATIONAL AERONAUTICS AND SPACE
C     ADMINISTRATION (NASA). THE SOFTWARE IS TECHNOLOGY AND SOFTWARE
C     PUBLICLY AVAILABLE UNDER U.S. EXPORT LAWS AND IS PROVIDED "AS-IS"
C     TO THE RECIPIENT WITHOUT WARRANTY OF ANY KIND, INCLUDING ANY
C     WARRANTIES OF PERFORMANCE OR MERCHANTABILITY OR FITNESS FOR A
C     PARTICULAR USE OR PURPOSE (AS SET FORTH IN UNITED STATES UCC
C     SECTIONS 2312-2313) OR FOR ANY PURPOSE WHATSOEVER, FOR THE
C     SOFTWARE AND RELATED MATERIALS, HOWEVER USED.
C
C     IN NO EVENT SHALL CALTECH, ITS JET PROPULSION LABORATORY, OR NASA
C     BE LIABLE FOR ANY DAMAGES AND/OR COSTS, INCLUDING, BUT NOT
C     LIMITED TO, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND,
C     INCLUDING ECONOMIC DAMAGE OR INJURY TO PROPERTY AND LOST PROFITS,
C     REGARDLESS OF WHETHER CALTECH, JPL, OR NASA BE ADVISED, HAVE
C     REASON TO KNOW, OR, IN FACT, SHALL KNOW OF THE POSSIBILITY.
C
C     RECIPIENT BEARS ALL RISK RELATING TO QUALITY AND PERFORMANCE OF
C     THE SOFTWARE AND ANY RELATED MATERIALS, AND AGREES TO INDEMNIFY
C     CALTECH AND NASA FOR ALL THIRD-PARTY CLAIMS RESULTING FROM THE
C     ACTIONS OF RECIPIENT IN THE USE OF THE SOFTWARE.
C
C$ Required_Reading
C
C     None.
C
C$ Keywords
C
C     OEM2SPK
C
C$ Declarations
 
      IMPLICIT NONE

      CHARACTER*(*)         INPUT
      INTEGER               HANDLE
      CHARACTER*(*)         METHOD
      INTEGER               DEGREE
      DOUBLE PRECISION      SPCLIM
      DOUBLE PRECISION      QUOLIM
      INTEGER               BODY
      INTEGER               CENTER
      CHARACTER*(*)         FRAME
      DOUBLE PRECISION      TBEG
      DOUBLE PRECISION      TEND
      CHARACTER*(*)         TIMSYS
      DOUBLE PRECISION      SPKBEG
      DOUBLE PRECISION      SPKEND
 
C$ Brief_I/O
C
C     VARIABLE  I/O  DESCRIPTION
C     --------  ---  --------------------------------------------------
C     INPUT      I   OEM file to be read.
C     HANDLE     I   Handle of SPK file.
C     METHOD     I   Interpolation method.
C     DEGREE     I   Interpolation degree.
C     SPCLIM     I   Minimum state spacing.
C     QUOLIM     I   Maximum quotient of successive state separations.
C     BODY       I   Ephemeris object.
C     CENTER     I   Center of motion.
C     FRAME      I   Name of reference frame.
C     TBEG       I   Coverage start time.
C     TEND       I   Coverage stop time.
C     TIMSYS     I   Time system.
C     SPKBEG     I   SPK start time bound.
C     SPKEND     I   SPK end time bound.
C
C$ Detailed_Input
C
C     INPUT          is the name of an OEM file to be read.
C
C     HANDLE         is the handle of the SPK file to write.  The
C                    file may be new or may be open for appending.
C
C     METHOD         is a string specifying the interpolation method
C                    to use.  Allowed values are 
C
C                       'HERMITE'
C                       'LAGRANGE'
C                       
C                    Case and white space are insignificant.
C
C     DEGREE         is the polynomial degree associated with the
C                    interpolation method.
C
C     SPCLIM         is the minimum spacing, measured in ephemeris
C                    seconds, between successive states in one
C                    input data block (the set of states following
C                    a metadata block and preceding the next
C                    metadata block or EOF).  If not specified in the
C                    setup file, SPCLIM defaults to MINSPC (declared in
C                    oem2spk.inc).
C     
C     QUOLIM         is the maximum quotient of successive separations
C                    between states (considered in either order).  If
C                    not specified in the setup file, QUOLIM defaults
C                    to MAXQUO (declared in oem2spk.inc).
C
C     BODY           is the NAIF integer ID of the ephemeris object
C                    for which segments are to be written.
C
C     CENTER         is the center of motion of the ephemeris object.
C
C     FRAME          is the name of the reference frame relative to
C                    which the output states are given.
C
C     TBEG           is the start time of coverage.  This is the 
C                    start time of the first segment written on a
C                    given call to this routine.
C
C     TEND           is the end time of coverage.  This is the 
C                    end time of the last segment written on a
C                    given call to this routine.
C
C     TIMSYS         is the name of the time system associated
C                    with time tags in the current metadata block
C                    or the states associated with this block.
C
C     SPKBEG,
C     SPKEND         are, respectively, output SPK file begin and 
C                    end times specified in the setup file.  The
C                    coverage of the output file is that of the
C                    input OEM file intersected with the interval
C                    [SPKBEG, SPKEND].  The bounds are expressed
C                    as seconds past J2000 TDB.
C                                         
C$ Detailed_Output
C
C     None.  See Particulars for a description of the action taken
C     by this routine.
C
C$ Parameters
C
C     None.
C
C$ Files
C
C     See the descriptions of the arguments INPUT and OUTPUT.
C
C$ Exceptions
C
C     1) If an unrecognized interpolation method is requested, the 
C        error SPICE(NOTSUPPORTED) will be signaled.
C
C     2) If an unsupported SPK data type is specified, the error
C        SPICE(NOTSUPPORTED) will be signaled.
C
C     3) If an error occurs while writing an SPK segment, the problem
C        will be diagnosed by routines called by this routine.
C
C     4) If an error occurs while parsing state data from the input
C        OEM file, the problem will be diagnosed by routines called
C        by this routine.
C
C     5) If TIMSYS does not specify a supported time system, the 
C        error will be diagnosed by routines called by this routine.
C
C     6) If the time tags of any two consecutive states in the OEM data
C         block have separation less than SPCLIM, the error
C         SPICE(TIMETAGERROR) will be signaled.
C
C     7) If the time tags T1, T2, T3 of any three consecutive states in
C        the OEM data block have spacing ratio R greater than QUOLIM,
C        where R is defined by
C
C           R  =  MAX ( (T3-T2)/(T2-T1), (T2-T1)/(T3-T2) )
C
C        the error SPICE(TIMETAGERROR) will be signaled.
C 
C$ Particulars
C
C     This routine creates a type 9 or type 13 segment in the
C     specified output SPK file, using data from the current
C     OEM data block.  The degree associated with the output
C     segment is based on the data type and the number of states
C     available in the input block; for short blocks, the
C     segment degree will be reduced as necessary.
C
C$ Examples
C
C     None.
C
C$ Restrictions
C
C     None.
C
C$ Author_and_Institution
C
C     N.J. Bachman    (JPL)
C
C$ Literature_References
C
C     [1]  CCSDS Orbit Data Messages Blue Book, version CCSDS 502.0-B-1,
C          September, 2004.
C
C$ Version
C
C-    OEM2SPK Version 1.0.0, 25-FEB-2005 (NJB)
C
C-&

C$ Revisions
C
C
C-&


C
C     SPICELIB functions
C      
      INTEGER               LSTLED
      INTEGER               LSTLTD

      LOGICAL               FAILED
      LOGICAL               EQSTR
      LOGICAL               ODD
      LOGICAL               RETURN

C
C     Local parameters
C
      INTEGER               LNSIZE
      PARAMETER           ( LNSIZE = 80 )

      INTEGER               MAXST
      PARAMETER           ( MAXST  = 10000 )

      INTEGER               SIDLEN
      PARAMETER           ( SIDLEN = 40 )

      INTEGER               TIMLEN
      PARAMETER           ( TIMLEN = 50 )

C
C     Local variables
C
      CHARACTER*(SIDLEN)    SEGID
      CHARACTER*(TIMLEN)    TIMSTR ( 2 )

      DOUBLE PRECISION      ET
      DOUBLE PRECISION      FIRST
      DOUBLE PRECISION      LAST
      DOUBLE PRECISION      R
      DOUBLE PRECISION      STATE  ( 6 )
      DOUBLE PRECISION      STBUFF ( 6, MAXST )
      DOUBLE PRECISION      TBUFF  (    MAXST )

      INTEGER               B
      INTEGER               DEG
      INTEGER               E
      INTEGER               I
      INTEGER               J
      INTEGER               LIDX
      INTEGER               N
      INTEGER               NSEG
      INTEGER               NWR
      INTEGER               PAD
      INTEGER               SEGTYP
      INTEGER               SHIFT
      INTEGER               STIDX
      INTEGER               WINSIZ

      LOGICAL               FOUND


      IF ( RETURN() ) THEN
         RETURN
      END IF

      CALL CHKIN ( 'MAKSEG' )

C
C     Use the first SIDLEN characters of the input file name as the
C     segment ID.
C
      SEGID = INPUT(:SIDLEN)

C
C     Compute the pad size for the current interpolation method
C     and degree.
C
      IF (  EQSTR( METHOD, 'LAGRANGE' )  ) THEN
         
         WINSIZ = DEGREE + 1
         SEGTYP = 9

      ELSE IF ( EQSTR( METHOD, 'HERMITE' )  ) THEN

         WINSIZ = ( DEGREE + 1 ) / 2
         SEGTYP = 13

      ELSE

         CALL SETMSG ( 'Interpolation method <#> is not supported.' )
         CALL ERRCH  ( '#', METHOD                                  )
         CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                        )
         CALL CHKOUT ( 'MAKSEG'                                     )
         RETURN

      END IF

C
C     If the window size is even, the pad is one less than half the 
C     window size.  If the window size is odd, the pad is half of
C     one less than the window size.  In either case, the pad
C     can be computed using the same integer arithmetic expression:
C           
      PAD = ( WINSIZ - 1 ) / 2

C
C     Initialize the index of the first state in the segment's coverage
C     interval.  If the segment has padding at the beginning, this
C     index will be equal to PAD + 1.
C
      STIDX = 1

C
C     Initialize the state and segment counts.
C      
      N     = 0
      NSEG  = 0

      CALL PRSDR ( INPUT, TIMSYS, ET, STATE, FOUND )


      DO WHILE (  FOUND  .AND.  ( .NOT. FAILED() )  ) 

         N = N + 1
                  
         CALL MOVED ( STATE, 6, STBUFF(1,N) )

         TBUFF(N) = ET

         IF ( N .EQ. MAXST ) THEN
C
C           It's time to emit a segment.  We'll use the last PAD
C           states as padding, so the last state in the coverage
C           interval is at index N-PAD.
C
            LIDX  = N - PAD

            FIRST = MAX ( TBEG, TBUFF(STIDX) )
            LAST  = MIN ( TEND, TBUFF(LIDX)  )

C
C           Adjust FIRST and LAST so they lie within the interval
C           [SPKBEG, SPKEND].
C
            FIRST = MAX ( FIRST, SPKBEG )
            LAST  = MIN ( LAST,  SPKEND )

C
C           If we have non-empty coverage, write out 
C           states 1 : MAXST.
C
            IF ( FIRST .LT. LAST ) THEN
C
C              Before writing the segment, perform spacing tests.
C
C              First, make sure all consecutive time tags are 
C              separated by at least the required interval SPCLIM.
C
               DO I = 2, N
                  
                  IF (  ( TBUFF(I) - TBUFF(I-1) ) .LT. SPCLIM ) THEN

                     DO J = 1, 2
                        CALL TIMOUT ( TBUFF(I-2+J),
     .                                'YYYY-MON-DD HR:MN:SC.######',
     .                                TIMSTR(J)                       )
                     END DO

                     CALL SETMSG ( 'Consecutive time tags # and # '   //
     .                             'have spacing less than the limit '//
     .                             '# seconds.'                       )
                     DO J = 1, 2
                        CALL ERRCH  ( '#', TIMSTR(J)                  )
                     END DO
                     
                     CALL ERRDP  ( '#',  SPCLIM                       )
                     CALL SIGERR ( 'SPICE(TIMETAGERROR)'              )
                     CALL CHKOUT ( 'MAKSEG'                           )
                     RETURN

                  END IF

               END DO

C
C              Now make sure that for any three consecutive time tags
C              T1, T2, T3 the spacing ratio R defined by 
C
C                 R  =  MAX ( (T3-T2)/(T2-T1), (T2-T1)/(T3-T2) )
C
C              is no greater than the limit QUOLIM.
C
               DO I = 3, N
                  
                  R = MAX ( 
     .                       ( TBUFF(I)   - TBUFF(I-1) ) /
     .                       ( TBUFF(I-1) - TBUFF(I-2) ),
     .                       ( TBUFF(I-1) - TBUFF(I-2) ) /
     .                       ( TBUFF(I)   - TBUFF(I-1) )   )


                  IF ( R .GT. QUOLIM ) THEN

                     DO J = 1, 3
                        CALL TIMOUT ( TBUFF(I-3+J),
     .                                'YYYY-MON-DD HR:MN:SC.######',
     .                                TIMSTR(J)                       )
                     END DO

                     CALL SETMSG ( 'Consecutive time tags #  #  # '   //
     .                             'have spacing ratio #; the limit ' //
     .                             'is #.'                            )
                     DO J = 1, 3
                        CALL ERRCH  ( '#', TIMSTR(J)                  )
                     END DO
                     
                     CALL ERRDP  ( '#',  R                            )
                     CALL ERRDP  ( '#',  QUOLIM                       )
                     CALL SIGERR ( 'SPICE(TIMETAGERROR)'              )
                     CALL CHKOUT ( 'MAKSEG'                           )
                     RETURN

                  END IF

               END DO

C
C              It's possible that the bounds FIRST and LAST exclude
C              some of the states we're going to write.  We determine
C              the set of states that actually need to be written.
C              To begin, find the indices of the last time tag less
C              than FIRST and the first time tag greater than LAST.
C
               B = LSTLTD ( FIRST, N, TBUFF )
               E = LSTLED ( LAST,  N, TBUFF ) + 1

C
C              Extend the indices we just found by the window size.
C              This is a bit more than the optimal number, but it's a
C              simple way to ensure the window has size at least
C              WINSIZ.
C
               B = B - WINSIZ
               E = E + WINSIZ

C
C              Bound the indices.  The resulting numbers give the
C              range of indices of time tags and states that actually
C              must be written.
C
               B   = MAX ( 1, B )
               E   = MIN ( N, E )

               NWR = E - B + 1

               IF ( SEGTYP .EQ. 9 ) THEN

                  CALL SPKW09 ( HANDLE, BODY,        CENTER, FRAME, 
     .                          FIRST,  LAST,        SEGID,  DEGREE,
     .                          NWR,    STBUFF(1,B), TBUFF(B)       )
 

                  IF ( FAILED() ) THEN
                     CALL CHKOUT ( 'MAKSEG' )
                     RETURN
                  END IF

               ELSE IF ( SEGTYP .EQ. 13 ) THEN

                  CALL SPKW13 ( HANDLE, BODY,        CENTER, FRAME, 
     .                          FIRST,  LAST,        SEGID,  DEGREE,
     .                          NWR,    STBUFF(1,B), TBUFF(B)       )

                  IF ( FAILED() ) THEN
                     CALL CHKOUT ( 'MAKSEG' )
                     RETURN
                  END IF

               ELSE

                  CALL SETMSG ( 'Requested segment type <#> is not '//
     .                          'supported.'                         )
                  CALL ERRINT ( '#',  SEGTYP                         )
                  CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                )
                  CALL CHKOUT ( 'MAKSEG'                             )
                  RETURN

               END IF
C
C              Increment the count of segments written.
C
               NSEG = NSEG + 1

            END IF

C
C           Shift the last (2*PAD+1) states into the beginning 
C           of the buffer.  Shift the corresponding epochs as well.
C           
            SHIFT = 2 * PAD  +  1

            DO I = 1, SHIFT

               CALL MOVED ( STBUFF(1, N-SHIFT+I), 6, STBUFF(1,I) )

               TBUFF(I) = TBUFF(N-SHIFT+I)

            END DO

            N = SHIFT

C
C           Set the segment coverage start index.
C
            STIDX = PAD + 1

         END IF

C
C        Get the next state and epoch.
C         
         CALL PRSDR ( INPUT, TIMSYS, ET, STATE, FOUND )

      END DO

C
C     If we've already written any segments, the last state of
C     the coverage interval of the last segment will always be 
C     buffered.  There is no need to create a new segment if 
C     that's the only state in the buffer.
C
      IF (  ( NSEG .GT. 0 )  .AND. ( N .LE. 1 )  ) THEN
         CALL CHKOUT ( 'MAKSEG' )
         RETURN
      END IF

C 
C     If we haven't written any segments, return if there are no
C     buffered states.
C
      IF (  ( NSEG .EQ. 0 )  .AND. ( N .EQ. 0 )  ) THEN
         CALL CHKOUT ( 'MAKSEG' )
         RETURN
      END IF

C
C     At this point, we're going to prepare to write a segment.
C
C     We may need to adjust the interpolation degree if too few
C     states are available.
C
      DEG = DEGREE

      IF ( N .LT. WINSIZ ) THEN
C
C        Pick the largest even window size less than or equal to N. 
C        For Lagrange interpolation, the degree is the window size
C        minus 1. For Hermite interpolation, the degree is twice the 
C        window size minus 1.
C        
         IF ( ODD(N) ) THEN
            WINSIZ = N - 1
         ELSE
            WINSIZ = N
         END IF


         IF ( SEGTYP .EQ. 9 ) THEN
            DEG = WINSIZ - 1
         ELSE IF ( SEGTYP .EQ. 13 ) THEN
            DEG = 2 * WINSIZ  -  1
         ELSE

            CALL SETMSG ( 'Requested segment type <#> is not '//
     .                    'supported.'                         )
            CALL ERRINT ( '#',  SEGTYP                         )
            CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                )
            CALL CHKOUT ( 'MAKSEG'                             )
            RETURN

         END IF

      END IF
      
C
C     Set the segment coverage bounds.
C
      FIRST = MAX ( TBEG, TBUFF(STIDX) )
      LAST  = MIN ( TEND, TBUFF(N)     )

C
C     Adjust FIRST and LAST so they lie within the interval
C     [SPKBEG, SPKEND].
C
      FIRST = MAX ( FIRST, SPKBEG )
      LAST  = MIN ( LAST,  SPKEND )

C
C     Write a segment of the requested type.
C
      IF ( FIRST .LT. LAST ) THEN
C
C        Before writing the segment, perform spacing tests.
C
C        First, make sure all consecutive time tags are 
C        separated by at least the required interval SPCLIM.
C
         DO I = 2, N
                  
            IF (  ( TBUFF(I) - TBUFF(I-1) ) .LT. SPCLIM ) THEN

               DO J = 1, 2
                  CALL TIMOUT ( TBUFF(I-2+J),
     .                          'YYYY-MON-DD HR:MN:SC.######',
     .                           TIMSTR(J)                        )
               END DO

               CALL SETMSG ( 'Consecutive time tags # and # have '//
     .                       'spacing less than the limit #      '//
     .                       'seconds.'                           )
               DO J = 1, 2
                  CALL ERRCH  ( '#', TIMSTR(J)                    )
               END DO

               CALL ERRDP  ( '#',  SPCLIM                         )
               CALL SIGERR ( 'SPICE(TIMETAGERROR)'                )
               CALL CHKOUT ( 'MAKSEG'                             )
               RETURN

            END IF

         END DO

C
C        Now make sure that for any three consecutive time tags
C        T1, T2, T3 the spacing ratio R defined by 
C
C           R  =  MAX ( (T3-T2)/(T2-T1), (T2-T1)/(T3-T2) )
C
C        is no greater than the limit QUOLIM.
C
         DO I = 3, N

            R = MAX ( 
     .                 ( TBUFF(I)   - TBUFF(I-1) ) /
     .                 ( TBUFF(I-1) - TBUFF(I-2) ),
     .                 ( TBUFF(I-1) - TBUFF(I-2) ) /
     .                 ( TBUFF(I)   - TBUFF(I-1) )   )


            IF ( R .GT. QUOLIM ) THEN

               DO J = 1, 3
                  CALL TIMOUT ( TBUFF(I-3+J),
     .                          'YYYY-MON-DD HR:MN:SC.######',
     .                          TIMSTR(J)                       )
               END DO

               CALL SETMSG ( 'Consecutive time tags #  #  # '   //
     .                       'have spacing ratio #; the limit ' //
     .                       'is #.'                            )
               DO J = 1, 3
                  CALL ERRCH  ( '#', TIMSTR(J)                  )
               END DO

               CALL ERRDP  ( '#',  R                            )
               CALL ERRDP  ( '#',  QUOLIM                       )
               CALL SIGERR ( 'SPICE(TIMETAGERROR)'              )
               CALL CHKOUT ( 'MAKSEG'                           )
               RETURN

            END IF

         END DO

C
C        It's possible that the bounds FIRST and LAST exclude
C        some of the states we're going to write.  We determine
C        the set of states that actually need to be written.
C        To begin, find the indices of the last time tag less
C        than FIRST and the first time tag greater than LAST.
C
         B = LSTLTD ( FIRST, N, TBUFF )
         E = LSTLED ( LAST,  N, TBUFF ) + 1

C
C        Extend the indices we just found by the window size.
C        This is a bit more than the optimal number, but it's a
C        simple way to ensure the window has size at least
C        WINSIZ.
C
         B = B - WINSIZ
         E = E + WINSIZ

C
C        Bound the indices.  The resulting numbers give the
C        range of indices of time tags and states that actually
C        must be written.
C
         B   = MAX ( 1, B )
         E   = MIN ( N, E )

         NWR = E - B + 1

         IF ( SEGTYP .EQ. 9 ) THEN

            CALL SPKW09 ( HANDLE, BODY,  CENTER, FRAME, FIRST,
     .                    LAST,   SEGID, DEG,    NWR,   STBUFF(1,B),
     .                    TBUFF(B)                                 )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'MAKSEG' )
               RETURN
            END IF

         ELSE IF ( SEGTYP .EQ. 13 ) THEN

            CALL SPKW13 ( HANDLE, BODY,  CENTER, FRAME, FIRST,
     .                    LAST,   SEGID, DEG,    NWR,   STBUFF(1,B),
     .                    TBUFF(B)                                 )

            IF ( FAILED() ) THEN
               CALL CHKOUT ( 'MAKSEG' )
               RETURN
            END IF

         ELSE

            CALL SETMSG ( 'Requested segment type <#> is not '//
     .                    'supported.'                         )
            CALL ERRINT ( '#',  SEGTYP                         )
            CALL SIGERR ( 'SPICE(NOTSUPPORTED)'                )
            CALL CHKOUT ( 'MAKSEG'                             )
            RETURN

         END IF

      END IF

C
C     PRSDR already pushed back the line we read that terminated 
C     our segment.
C

      CALL CHKOUT ( 'MAKSEG' )
      RETURN 
      END


